/******************************************************************************
 *  FILE:      @(#)c_getresource.c      1.33 04/11/05
 *  DESCRIPTION:
 *      Source code for GetResource
 *
 *  Task level
 *  ----------
 *  2. RES_SCHEDULER:
 *      - Lock scheduler
 *      - Extended Nested resources handling.
 *  3. R (owned by ISR):
 *      - Lock scheduler
 *      - Extended Nested resources handling.
 *      - ISR Ceiling Protocol
 *  4. R (not owned by ISR):
 *      - Extended Nested resources handling.
 *      - Task Ceiling Protocol
 *  Interrupt level
 *  ---------------
 *      - Extended Nested resources handling.
 *      - ISR Ceiling Protocol
 *****************************************************************************/ 
#include "c_common.h"
#include "c_target.h"

#if     (_os_OS_0_POSTMORTEM != _os_OS_NONE)
#include "c_postmortem.h"
#endif

StatusType
#if (_os_SIA>0)
_os_GetResource
#else
GetResource
#endif
(ResourceType sm)
{
        StatusType ret = E_OK;
        _os_RTOSMEM _os_RESOURCE*   resptr;
        uint_fast8_t            prio;
        
        /* first make the service non-preemptable */
        _os_lock++;

        /* system entry actions */
        _os_ServiceEntry(_os_N_GetResource);

        /* and then initialize all locals */
        resptr = &_os_resource_table[sm];
        prio = _os_current_task->PRIORITY;
        
#if (_os_OS_0_STATUS == _os_OS_EXTENDED)
        
        #if (_os_OS_0_USERESSCHEDULER==0)
        if(sm==0)
        {
                /* not permission for scheduler resource */
                ret = E_OS_ID;
        }
        else
        #endif
        if (sm >= _os_NO_RESOURCE)
        {
                /* not right ID */
                ret = E_OS_ID;
        }
        else /* resource cannot be internal */
        if( resptr->RESOURCEPROPERTY == _os_RE_INTERNAL )
        {
                ret = E_OS_ID;
        }
        else
        #if (MAX_RESOURCE_ISR>0)
        /* from ISR */  
        if ( _os_IsInterruptLevel() )
        {
                if (!_os_IsResOwnedByCurrentISR(sm))
                {
                        /* undefined if the ISR does not own
                           the resource, ret = E_OS_ID */
                        ret = E_OS_ID;
                }
        }
        else
        #endif
        if (_os_IsTaskLevel())
        {
                if(!_os_IsResOwnedByCurrentTask(sm))
                {
                        /* undefined if the task does not own
                           the resource, ret = E_OS_ID */
                        ret = E_OS_ID;
                }
                        /* the current task static priority cannot be 
                         * higher than the ceiling priority of
                         * the resource to be accessed
                         */                        
                else if (_os_current_task->staticprio > resptr->CEILING)
                {
                        ret = E_OS_ACCESS;

                }                
        }
        
        if (ret == E_OK)
        {
                _os_DisableInterrupts();
                
                if (resptr->preprio)
                {
                        /* Attempt to get resource which is already 
                         * occupied by a task or ISR.
                         * Only locked resources have a
                         * non-zero value in their preprio field */
                        ret = E_OS_ACCESS;
                }
                
                _os_EnableInterrupts();
        }
        
        
        if ( ret == E_OK )
        {
#endif   // EXTENDED     
        

           #if (_os_OS_0_USERESSCHEDULER>0)
           if (sm==0)
           {
                   /* we just need to lock the scheduler */
                   _os_lock++;
                   resptr->preprio = prio;
                   #if (_os_OS_0_ORTI>0)
                   /* ORTI should display the task as it is
                    * running with maximum priority while 
                    * the scheduler is locked */
                   _os_current_task->PRIORITY = _os_OS_0_NO_PRIORITIES-1;
                   #endif
           }
           else 
           {
           #endif       

                /* does this resource require extension of the 
                 * ceiling priority protocol to ISR level? */
                if (resptr->isrceiling)
                {
                        /* this could be task or interrupt level */
                        #if (MAX_RESOURCE_ISR>0)
                        /********************************************* 
                         * ISR Ceiling Priotity Protocol:
                         * It is assumed that no ISR can
                         * own the RES_SCHEDULER!
                         * needs to update resptr->preprio
                         *********************************************/
                        _os_GetISRCeilingProtocol(sm);
                        #if ((_os_OS_0_STATUS == _os_OS_EXTENDED)||(_os_OS_0_ORTI>0))
                        if ( _os_IsInterruptLevel() )
                        {
                                /* handling nested accesses at
                                 * interrupt level */
                                if (_os_current_isr->res != _os_INVALID_RES)
                                {
                                        /* nested call has happened */
                                        resptr->nestedto = _os_current_isr->res;
                                }
                                /* resource 'sm' is owned by this isr */
                                _os_current_isr->res  = sm;
                                #if  (_os_OS_0_ORTI>0) 
                                /* only for ORTI */
                                resptr->locker = _os_NO_TASK + (int)(_os_current_isr-&_os_isr_table[0]); 
                                #endif
                        }
                        #endif
                        #endif
                }
                else 
                {
                        if (_os_lock==_os_SCHEDULABLE)
                        {
                                /* this is always task level */
                                if ( prio < resptr->CEILING)
                                {
                                        /* TASK Ceiling Priority Protocol */
                                        _os_SuspendOS();
                                        #if     (_os_OS_0_POSTMORTEM != _os_OS_NONE)
                                                _os_LogPlaybackEvent((uint8_t)RAISE_PRIORITY,
                                                (uint8_t)sm,(uint32_t)resptr->CEILING);
                                        #endif
                                        /* no other task accesses resource until release */
                                        _os_SwapPrioReady(resptr->CEILING);
                                        _os_ResumeOS();
                                }
                        }
                        /* indicates sm is locked (with priority of
                         * accessing task) */ 
                        resptr->preprio = prio;

                }
           #if (_os_OS_0_USERESSCHEDULER>0)
           }
           #endif
          
          /* task level: common for Task/ISR ceiling protocol and RES_SCHEDULER */      
          if (_os_IsTaskLevel())
          {
                  #if ((_os_OS_0_STATUS == _os_OS_EXTENDED)||(_os_OS_0_ORTI>0))
                  if (_os_current_task->res != _os_INVALID_RES)
                  {
                          /* handling errors for nested resources */
                          resptr->nestedto = _os_current_task->res;
                   }
                  /* resource 'sm' is owned by this task */
                  _os_current_task->res  = sm;
                  #endif
                 
                  #if  (_os_OS_0_ORTI>0)
                  /* only for ORTI */
                  resptr->locker = _os_current_task->user_id;
                  #endif
           }
                    

#if (_os_OS_0_STATUS == _os_OS_EXTENDED )
        }
#endif        
        #if (_os_OS_0_ERRORHOOK==1)
        if ( ret != E_OK ) 
        {
                _os_LogError(ret,_os_TO_SEM_TAKE,(int32_t)sm,-1,-1);
        }
        #endif

        if (_os_NeedToSwitchTasks())
        {
                /* OS are suspended now */
                _os_TaskSwitch();
                /* never here */
        }
        
        /* _os_lock-- */

        /* system exit actions */
        _os_ServiceExit(_os_N_GetResource);
        
        return ret;
}

#if (_os_OS_0_STATUS == _os_OS_EXTENDED )

#if (_os_NO_ISR>0)
#if (MAX_RESOURCE_ISR>0)
uint_fast8_t _os_IsResOwnedByCurrentISR(ResourceType res)
{
        uint_fast8_t j;
        
        #if (_os_OS_0_USERESSCHEDULER == 1)
        if (res==RES_SCHEDULER)
        {
                /* taking the scheduler as a resource is 
                 * not allowed from interrupt level 
                 */
                        return 0;
        }
        #endif
        
        for (j=0;j<MAX_RESOURCE_ISR &&
            _os_current_isr->RESOURCE[j] != _os_END_SEARCH; j++)
        {
                if ( _os_current_isr->RESOURCE[j] == res )
                {
                        return 1;
                }
        }
        return 0;
}

#endif
#endif
uint_fast8_t _os_IsResOwnedByCurrentTask(ResourceType res)
{
        uint_fast8_t i;
        
        #if (_os_OS_0_USERESSCHEDULER == 1)
        if (res==RES_SCHEDULER)
        {       
                /* taking the scheduler as a resource is 
                 * allowed for all tasks */
                return 1;
        }
        #endif
        
        for (i=0;i<MAX_RESOURCE_TASK && 
            _os_current_task->RESOURCE[i] != _os_END_SEARCH ;i++)
        {   
           /* find the resource is owned by current task */    
           if ( _os_current_task->RESOURCE[i] == res )
           {
                   return 1;
           }   
        }
        return 0;
}

#endif // EXTENDED


