/*****************************************************************************
 *  FILE:      @(#)c_dispatch.c 1.96 04/11/22  
 *  DESCRIPTION:
 *       Source code for scheduler and other dispatching routines.
 *       It does also implement the ready-queue.
 ****************************************************************************/
#include "c_common.h"
#include "c_target.h"
#include "g_conf.h"
#if     (_os_OS_0_POSTMORTEM != _os_OS_NONE)
#include "c_postmortem.h"
#endif


/******************************************************************************
 * _os_current_task: pointing always to running task
 * _os_ready_to_run_index: higher active level in ready
 * _os_lock: locks the scheduler
 * _os_run_level: indicates the current running level
*****************************************************************************/
_os_RTOSMEM _os_TASK*  volatile _os_FASTALLOC  _os_current_task  = (_os_RTOSMEM _os_TASK*) 0;
volatile uint_fast8_t           _os_FASTALLOC  _os_ready_to_run_index  = 0;
volatile int_fast8_t            _os_FASTALLOC  _os_lock         = -1;
volatile uint_fast8_t           _os_FASTALLOC  _os_run_level    = 0;


#if (_os_OS_0_ONE_TASK_PER_PRIO ==1)
/******************************************************************************
 * The TOC tools has found out that there are not two TASK objects with
 * the same priority in the OIL file.
 * Thus we do not need a per-priority-queues if the TOC tool takes
 * care of adding extra levels for ceiling priorities.
 * Now, the physical priority levels can be populated at any given
 * time by a maximum of only one task.
 * This situation is highly recommended when improving context
 * switch time and ISR latency becomes an issue.
*****************************************************************************/
_os_RTOSMEM _os_TASK* _os_RTOSMEM _os_ready_to_run[ _os_OS_0_NO_PRIORITIES];
#else
/******************************************************************************
 * READY_TO_RUN queue:
 * - _os_ready_to_run queue: Every element of the array represents a
 *   priority level where all the 'ready' tasks corresponding
 *   to that priority are queued in a circular linear linked list.
 *   The items of the priority queues are linked via their ready
 *   field.
 * 
 * - _os_AddBegin:          Adds an item at the beginning of the queue
 *                      for a specific priority level.
 * - _os_AddEnd:            Adds an item at the end of the queue for a
 *                      specific priority level.
 * - _os_RemoveCurrent:        Remove first item of the queue.
 *   Access to the queue must be protected.
*****************************************************************************/
_os_RTOSMEM _os_TASK** _os_RTOSMEM _os_ready_to_run[ _os_OS_0_NO_PRIORITIES];

static void             _os_AddBegin(_os_RTOSMEM _os_TASK*,uint_fast8_t);
static void             _os_AddEnd(_os_RTOSMEM _os_TASK*,uint_fast8_t);
#endif

static void             _os_SetReady(_os_RTOSMEM _os_TASK* tcb);

#if (_os_OS_0_ONE_TASK_PER_PRIO !=1)
/******************************************************************************
 * FUNCTION:            _os_RemoveCurrent 
 * DESCRIPTION:
 *      It removes the current running task (only a
 *      running task can set itself to non-ready).
 *      Used in:
 *      - _os_swap_prio_ready: protected
 *      - _os_remove_current_task: protected
 *****************************************************************************/
static void _os_RemoveCurrent(void)
{
        uint_fast8_t pri = _os_current_task->PRIORITY;
        _os_RTOSMEM _os_TASK* task = _os_current_task;
        
        /* Dequeue the current task */
        *_os_ready_to_run[pri] = (*_os_ready_to_run[pri])->ready;

        /* set current task to not queued */
        task->ready = (_os_RTOSMEM _os_TASK*)0;
        
        /* If current task was the only item in the queue,
         * we must indicate that the queue is empty */
        if ( *_os_ready_to_run[pri] == (_os_RTOSMEM _os_TASK*)0 )
        {
                _os_ready_to_run[pri] = (_os_RTOSMEM _os_TASK**)0;
        }     
        
        return;
}


/******************************************************************************
 * FUNCTION:            _os_AddEnd 
 * DESCRIPTION:
 *      If queues, under normal circunstances, when a
 *      task is added to a ready queue it is inserted
 *      at the end, i.e becomes the newest.
 *      Used in:                        
 *      - _os_set_task_ready: protected
 ****************************************************************************/
static void _os_AddEnd(_os_RTOSMEM _os_TASK* T,uint_fast8_t pri)
{

        if (_os_ready_to_run[pri])
        {
                /* not empty queue */
                T->ready = *_os_ready_to_run[pri];
                *_os_ready_to_run[pri] = T;
                _os_ready_to_run[pri] = (_os_RTOSMEM _os_TASK** _os_RTOSMEM )&T->ready;
        }
        else
        { 
                /* T is only element */
                _os_ready_to_run[pri] = (_os_RTOSMEM _os_TASK** _os_RTOSMEM )&T->ready;
                T->ready = T;
        }
        return;

}


/******************************************************************************
 * FUNCTION:            _os_AddBegin   
 * DESCRIPTION:
 *      ChainTask imposes the chained task to execute
 *      immediately thus being inserted at the
 *      beginning of the queue (like the oldest).
 *      Used in:
 *      - _os_SetTaskReadyFirst: protected
 *      - _os_SwapPrioReady: protected
 *****************************************************************************/
static void _os_AddBegin(_os_RTOSMEM _os_TASK* T,uint_fast8_t pri)
{
        if (_os_ready_to_run[pri])
        {
                /* not empty queue */
                T->ready = *_os_ready_to_run[pri];
                *_os_ready_to_run[pri] = T;
        }
        else
        { 
                /* T is only element */
                _os_ready_to_run[pri] =(_os_RTOSMEM _os_TASK** _os_RTOSMEM ) &T->ready;
                T->ready = T;
        }

        return;
}
#endif

/******************************************************************************
 * FUNCTION:            _os_SetReady
 * DESCRIPTION:
 *      Updates some task block fields after the task
 *      has been inserted to a new priority level.
 *      Used in:
 *      - _os_SetTaskReadyFirst: protected
 *      - _os_SetTaskReady: protected
 *
 *      waiting 
 *        or        -----> ready
 *      suspended       
 *****************************************************************************/
static void _os_SetReady(_os_RTOSMEM _os_TASK* tcb)
{
        /* 'tcb->PRIORITY' equals here 'tcb->staticprio'
         * since when the task is on waiting/suspended
         * all resources have been released.
         * A task terminates always with its static priority */

        if ( tcb->PRIORITY > _os_ready_to_run_index )
        {
                /*
                ** If the new priority is higher than the highest until now,
                ** set a new highest priority
                */
                _os_ready_to_run_index = tcb->PRIORITY;
        }
        
        /* ready to run */
        tcb->state      = READY;             
        
        return;
}




/******************************************************************************
 * READY_TO_RUN queue: Interfaces
 *
 * - _os_nextReadyLevel
 * - _os_set_task_ready
 * - _os_set_task_ready_first 
 * - _os_swap_prio_level
 *   
 * REMARKS:
 *      The callers are responsible of disabling the ISR2s interrupts.
 *****************************************************************************/


/******************************************************************************
 * FUNCTION:            _os_NextReadyLevel
 * DESCRIPTION:
 *      It finds the highest non-empty priority level starting
 *      at a given value       
 *****************************************************************************/
uint_fast8_t _os_NextReadyLevel(uint_fast8_t ref)
{
        uint_fast8_t i=ref;
#if (_os_OS_0_ONE_TASK_PER_PRIO ==1)
        while(_os_ready_to_run[i]==(_os_RTOSMEM _os_TASK*)0) i--;
#else
        while(_os_ready_to_run[i]==(_os_RTOSMEM _os_TASK**)0) i--;
#endif
        return i;
}

/******************************************************************************
 *  DESCRIPTION:
 *  Insert the task at the end of ready-to-run queue and make it ready to run.
 *****************************************************************************/
void
_os_SetTaskReady(_os_RTOSMEM _os_TASK* T )
{
        if (T->state != READY)
        {
 #if (_os_OS_0_ONE_TASK_PER_PRIO ==1)
                _os_ready_to_run[T->staticprio] = (_os_RTOSMEM _os_TASK*)T;
 #else
                _os_AddEnd(T,T->staticprio);
 #endif
                _os_SetReady(T);
        }
        else
        {
                /* io request (setevent) completed before
                 * task actually entered wait state
                 */
                T->rc++;
        }

        return;
}

#if (_os_OS_0_ONE_TASK_PER_PRIO !=1)
/******************************************************************************
 *  FUNCTION:           _os_SetTaskReadyFirst 
 *  DESCRIPTION:
 *      Insert the task at the beginning of ready-to-run queue and make
 *      it ready to run.       
 *      This is used from 'sc_chain_task' when the succeding task must be the  
 *      first to be running upon termination of the current task.
 *  REMARKS:
 *      The caller of ChainTask has been already dequeued. 
 *****************************************************************************/
void
_os_SetTaskReadyFirst(_os_RTOSMEM _os_TASK* T )
{
        _os_AddBegin(T,T->staticprio);
        _os_SetReady(T);
        return;
}
#endif

/******************************************************************************
 * FUNCTION:            _os_RemoveCurrentTask
 * DESCRIPTION:
 *      Removes current task from current queue
*****************************************************************************/
void _os_RemoveCurrentTask(void)
{
        _os_RTOSMEM _os_TASK* task = _os_current_task;
        
        if (task->rc >0)
        {
                /* Ignore remove. The task has been set to ready
                 * even again (io) before it managed to be removed
                 * from the ready queue.
                 */
                task->rc = 0;
        }
        else
        {
        #if (_os_OS_0_ONE_TASK_PER_PRIO !=1)
                _os_RemoveCurrent();
        #else
                _os_ready_to_run[task->PRIORITY]  = (_os_RTOSMEM _os_TASK*)0;
        
        #endif
        }
        return;
}

/******************************************************************************
 *  FUNCTION:           _os_SwapPrioReady
 *  DESCRIPTION:
 *      It raises/lowers the priority of the running task. The task is actually
 *      dequeud from current queue and then enqueued in the new level.
 *      CEILING protocol      
 *****************************************************************************/
void _os_SwapPrioReady(uint_fast8_t pri)
{
        _os_RTOSMEM _os_TASK* task = _os_current_task;
        
        if (task->PRIORITY != pri)
        {
        #if (_os_OS_0_ONE_TASK_PER_PRIO !=1)

                /* remove current task */
                _os_RemoveCurrent();
                
                /* add current task to  a new queue */
                _os_AddBegin(task,pri);
        #else
                _os_ready_to_run[task->PRIORITY]  = (_os_RTOSMEM _os_TASK*)0;
        
                _os_ready_to_run[pri]  = (_os_RTOSMEM _os_TASK*)task;
                
        #endif
                /* update task to its new priority */
                task->PRIORITY = pri;

                /* adapt new ready priority */
                _os_ready_to_run_index = ( pri > _os_ready_to_run_index) ? pri : 
                _os_NextReadyLevel(_os_ready_to_run_index);
        }
        return;
}

/******************************************************************************
 *  FUNCTION:           _os_Schedule
 *  DESCRIPTION:
 *      From the set of tasks in the ready/running state the
 *      scheduler determines first the set of tasks with the
 *      highest priority and then schedules the oldest task 
 *      whithin this set.
 *      The system may follow the contract of a FULL PREEMPTIVE
 *      (a task which is presently running may be rescheduled at any     
 *      instruction by the occurrence of trigger conditions pre-set by the
 *      operating system) or NON PREEMPTIVE (task switching   
 *      is only performed via one of a selection of explicitly
 *      defined points of rescheduling).
 *
 *      The previous paragraph applies only to application code, the RTOS locks
 *      the scheduler while running its own code.
 *      This is done by:
 *       - _os_lock == -1 in application code.
 *       - increased by one when entering an ISR2 or a system service.
 *      - '_os_Schedule' executes only if _os_lock==0, i.e. at the end of a
 *      system service or at the falling edge of an ISR2 that has preempted
 *      user code.
 *      Obviously, an ISR2 (or more) can always preempt a system service,
 *      the point is that the consequences of the preemption(s) are not taken
 *      into account until the service has finished.
 *
 *      There is always at least a task to be scheduled: 'idle'
 *****************************************************************************/
        
void
_os_Schedule( void )
{
        _os_RTOSMEM _os_TASK* task;
        
        _os_SuspendOS();
       
        /*********************************************************************
         * Some platforms may need specific actions entering the dispatcher.
         * The code differs per platform:
         * - 8051,M16C: Nothing is done
         * - TriCore:   Remove all stack frames
         ********************************************************************/
        _os_EnterScheduler();

#if (_os_OS_0_STATUS == _os_OS_EXTENDED)
        if (
                _os_lock          != _os_SCHEDULABLE          ||
                _os_current_task  != (_os_RTOSMEM _os_TASK*)0 
           )
        {
            ShutdownOS(E_OS_SYS_ERROR);
        }
#endif

        /* Check if the ready-to-run queue for this priority contains still
         * a queue. If not, search for lower priority tasks */
        _os_ready_to_run_index = _os_NextReadyLevel(_os_ready_to_run_index);

        #if (_os_OS_0_ONE_TASK_PER_PRIO ==1)
        task = _os_ready_to_run[_os_ready_to_run_index];
        #else
        task = (_os_RTOSMEM _os_TASK*)*_os_ready_to_run[_os_ready_to_run_index];
        #endif

        /* 'current_task' contains now the next running task */     
        _os_current_task = task;
       
        #if (_os_NO_ISR>0)
        /* return always to task level */
        _os_isr2_counter        = 0;
        #endif

        _os_lock                = -1;
        _os_run_level           = 0;

        /***************************************************************************** 
         * Special action are required if the task is defined as NON-preemptable
         * or belongs to a group of tasks (internal resources)
         * 
         * 1. NON-preemptable: No other processes preempt a NON-preemptable task. 
         *      The only possible points of 're-scheduling' are:
         *      - Successful termination of a task (system services 'TerminateTask' 
         *      and 'ChainTask').
         *      - Explicit call of scheduler
         *      - Transition into the waiting state (system service 'WaitEvent')
         *
         * 2. Group of Tasks: The internal resource is automatically taken when the 
         *      task enters the running state, except when it has already taken
         *      the resource. As a result, the priority of the task is automatically
         *      changed to the ceiling priority of the resource. 
         *                      
         *****************************************************************************/
        if ( task->SCHEDULE == _os_TA_NON )
        {
                /* scheduler is locked for non-preemptable tasks */
                _os_lock++;
        }
        else if (task->internal != _os_INVALID_RES )
        {
             _os_SwapPrioReady(_os_resource_table[task->internal].CEILING);
             #if ((_os_OS_0_STATUS == _os_OS_EXTENDED)||(_os_OS_0_ORTI>0))
                 task->res = task->internal;
             #endif
        }

        /* Modify current state of the task */
        task->state   = RUNNING;
        /* Reset ready count                */
        task->rc      = 0;          
        
        #if     (_os_OS_0_POSTMORTEM != _os_OS_NONE)
                _os_LogPlaybackEvent((uint8_t)CTX_SWITCH,
                                (uint8_t)task->user_id,
                                (uint32_t)_os_ready_to_run_index);
        #endif
  
        #if (_os_OS_0_PRETASKHOOK==1)
        /* User's Hook routine */
        _os_SETPREHOOK();
        PreTaskHook();
        _os_CLRPREHOOK();
        #endif

        #if (_os_OS_0_ORTI>0)
        _os_current_task->noruns++;     
        #endif

        #if (_os_OS_0_ORTI>0)
        /* no cycles to scheduler code */
        _os_UpdateRTOSCycles();
        #endif
        
        /* Go to the task:  */
        _os_GoToTask( task );       
        /* never */
        return;                         
}











