/******************************************************************************
 * FILE:        @(#)c_common.h  1.41 04/10/18
 * DESCRIPTION:
 *      Interface for the RTOS common part.
 *      The definitions of these routines
 *      are located in common files.
 *
 * NAMING CONVENTIONS (OS internals):
 * Global variables: _os_xxx_yyy
 * Routines:         _os_XxxYyy
 * Types:            _os_XXX
 *****************************************************************************/ 
#ifndef _C_COMMON_H
#define _C_COMMON_H

#include <osek/osek.h>
#include "t_rtos.h"
#include "g_conf_types.h"
#include "g_conf.h"

/* definitions */
#define _os_ISR_CAT1            1
#define _os_ISR_CAT2            2
#define _os_INVALID_ALARM       -1

#define _os_AL_STOPPED          0
#define _os_AL_RUNNING          1
#define _os_AL_BUSY             2

#define _os_SEM_EMPTY           0
#define _os_SEM_FULL            1
#define _os_INVALID_RES         -1

#define _os_SCHEDULABLE         0

#define INCREASE(pos,msg) pos=(pos+1)%_os_message_table[msg].QUEUESIZE

/* masks for _os_run_level */
#define _os_TASKLV              0x0
#define _os_ISR1LV              0x1
#define _os_ISR2LV              0x2
#define _os_HOOKERRLV           0x4
#define _os_HOOKPRELV           0x8
#define _os_HOOKPOSTLV          0x10
#define _os_HOOKSTARTLV         0x20
#define _os_HOOKSHUTLV          0x40
#define _os_CALLBACKLV          0x80

/* macro to set/clear/monitor running level */
#define _os_SETISR1LEVEL()      _os_run_level |= _os_ISR1LV;                    
#define _os_CLRISR1LEVEL()      _os_run_level &= ~_os_ISR1LV;
#define _os_INISR1LEVEL()       (_os_run_level & _os_ISR1LV)

#define _os_SETISR2LEVEL()      _os_run_level |= _os_ISR2LV;                    
#define _os_CLRISR2LEVEL()      _os_run_level &= ~_os_ISR2LV;                   
#define _os_INISR2LEVEL()       (_os_run_level & _os_ISR2LV)

#define _os_SETERRORHOOK()      _os_run_level |= _os_HOOKERRLV;                 
#define _os_CLRERRORHOOK()      _os_run_level &= ~_os_HOOKERRLV;                        
#define _os_INERRORHOOK()       (_os_run_level & _os_HOOKERRLV)

#define _os_SETPREHOOK()        _os_run_level |= _os_HOOKPRELV;                 
#define _os_CLRPREHOOK()        _os_run_level &= ~_os_HOOKPRELV;                        
#define _os_INPREHOOK()         (_os_run_level & _os_HOOKPRELV)

#define _os_SETPOSTHOOK()       _os_run_level |= _os_HOOKPOSTLV;                        
#define _os_CLRPOSTHOOK()       _os_run_level &= ~_os_HOOKPOSTLV;                       
#define _os_INPOSTHOOK()        (_os_run_level & _os_HOOKPOSTLV)

#define _os_SETSTARTHOOK()      _os_run_level |= _os_HOOKSTARTLV;                       
#define _os_CLRSTARTHOOK()      _os_run_level &= ~_os_HOOKSTARTLV;                      
#define _os_INSTARTHOOK()       (_os_run_level & _os_HOOKSTARTLV)

#define _os_SETSHUTHOOK()       _os_run_level |= _os_HOOKSHUTLV;                        
#define _os_CLRSHUTHOOK()       _os_run_level &= ~_os_HOOKSHUTLV;                       
#define _os_INSHUTHOOK()        (_os_run_level & _os_HOOKSHUTLV)

#define _os_SETCALLBACK()       _os_run_level |= _os_CALLBACKLV;                        
#define _os_CLRCALLBACK()       _os_run_level &= ~_os_CALLBACKLV;       
#define _os_INCALLBACK()        (_os_run_level & _os_CALLBACKLV)

#define _os_SETTASKLEVEL()      _os_run_level = 0;
#define _os_INTASKLEVEL()       (_os_run_level==0)

/******************************************************************************
 * Interface to fast-accessed internals 
 *****************************************************************************/ 
#if (_os_NO_ISR>0)
extern volatile uint_fast8_t            _os_FASTALLOC     _os_isr1_counter;
extern volatile uint_fast8_t            _os_FASTALLOC     _os_isr2_counter;
extern _os_RTOSMEM _os_ISR*  volatile   _os_FASTALLOC     _os_current_isr;
#endif
extern volatile int_fast8_t             _os_FASTALLOC     _os_lock;
extern volatile uint_fast8_t            _os_FASTALLOC     _os_ready_to_run_index;
extern volatile int_fast8_t             _os_FASTALLOC     _os_nesting_sr;
extern volatile int_fast8_t             _os_FASTALLOC     _os_nesting_sros;
extern _os_RTOSMEM _os_TASK* volatile   _os_FASTALLOC     _os_current_task;
extern volatile uint_fast8_t            _os_FASTALLOC     _os_run_level;
#if (_os_OS_0_ORTI>0)
extern volatile uint_fast8_t            _os_FASTALLOC     _os_servicetrace;
#endif
extern uint_fast8_t                     _os_shutting;


#if (_os_OS_0_ERRORHOOK==1 || _os_COM_0_COMERRORHOOK==1)
#if ((_os_OS_0_USEGETSERVICEID == 1) || (_os_COM_0_COMERRORGETSERVICEID==1)||(_os_OS_0_ORTI>0) )
/* last failing system service */
extern _os_RTOSMEM uint_fast8_t                 _os_errid;
#endif
#if (_os_OS_0_ORTI>0)
extern _os_RTOSMEM StatusType                   _os_errno;
#endif

#if (_os_OS_0_USEPARAMETERACCESS == 1 || _os_COM_0_COMUSEPARAMETERACCESS==1)
/* Array of last system service's parameters */
extern int_fast32_t                             _os_sys_par[];
#endif
#endif

/* Storage for application modes */
extern _os_RTOSMEM AppModeType                  _os_system_mode;
extern _os_RTOSMEM COMApplicationModeType       _os_com_mode;
/* it counts the number of application resets */
extern _os_RTOSMEM uint_fast8_t                 _os_no_reset; 
/* indicates COM has been started */
extern _os_RTOSMEM uint_fast8_t                 _os_com_usage;

#if (_os_OS_0_ORTI>0)
extern _os_RTOSMEM clock_t                      _os_rtos_cycles;
extern _os_RTOSMEM clock_t                      _os_enter_cycles;
extern _os_RTOSMEM clock_t                      _os_total_cycles;
#endif


#if (_os_OS_0_ONE_TASK_PER_PRIO ==1)
/******************************************************************************
 * The TOC tools has found out that there are not two TASK objects with
 * the same priority in the OIL file.
 * Thus we do not need a per-priority-queues if the TOC tool takes
 * care of adding extra levels for ceiling priorities.
 * Now, the physical priority levels can be populated at any given
 * time by a maximum of only one task.
 * This situation is highly recommended when improving context
 * switch time and ISR latency becomes an issue.
 *****************************************************************************/ 
extern _os_RTOSMEM _os_TASK* _os_RTOSMEM        _os_ready_to_run[];
#else
/******************************************************************************
 * READY_TO_RUN queue:
 * - ready_to_run queue: Every element of the array represents a
 *   priority level where all the 'ready' tasks corresponding
 *   to that priority are queued in a circular linear linked list.
 *   The items of the priority queues are linked via their ready field.
 *****************************************************************************/ 
extern _os_RTOSMEM _os_TASK** _os_RTOSMEM       _os_ready_to_run[];
#endif


/******************************************************************************
 * system actions upon entrying/exiting service routines
 * REMARK:
 *      _os_current_task != 0
 *****************************************************************************/
extern void     _os_ServiceEntry(uint_fast8_t ss);
extern void     _os_ServiceExit(uint_fast8_t ss);
extern void     _os_CheckRunLevel(uint_fast8_t ss);


/******************************************************************************
 * log number of cycles spent in last run
 *****************************************************************************/
#if (_os_OS_0_ORTI>0)
extern void     _os_UpdateTaskCycles(void);
extern void     _os_UpdateRTOSCycles(void);
#if (_os_NO_ISR>0)
extern void     _os_UpdateISRCycles(IsrType);
#endif
extern void     _os_UpdateCycles(void);
#endif

/******************************************************************************
 * Actions on the ready-to-run queue (in 'c_dispatch.c')
 *****************************************************************************/ 
extern void     _os_SetTaskReady( _os_RTOSMEM _os_TASK* );
#if (_os_OS_0_ONE_TASK_PER_PRIO !=1)
extern void     _os_SetTaskReadyFirst(_os_RTOSMEM _os_TASK*);
#else
#define         _os_SetTaskReadyFirst _os_SetTaskReady
#endif
extern void     _os_RemoveCurrentTask(void);
extern void     _os_SwapPrioReady(uint_fast8_t);
uint_fast8_t    _os_NextReadyLevel(uint_fast8_t);

/******************************************************************************
 * scheduling routines (in 'c_dispatch.c' 'c_system.c'
 *****************************************************************************/         
extern void     _os_Schedule( void );
uint_fast8_t    _os_NeedToSwitchTasks(void);
extern void     _os_TaskSwitch(void);
extern void     _os_WaitSwitch(void);

/******************************************************************************
 * disable/enable OS interrupts'. TOC generated in 'g_isrframe.c'.
 *****************************************************************************/         
extern void      _os_SuspendISR2(void);
extern void      _os_ResumeISR2(void);

/******************************************************************************
 * extract isr framework from g_isrframe object
 *****************************************************************************/         
#if (_os_NO_ISR>0)
extern void     _os_ExtractISR(void); 
#endif

/******************************************************************************
 * routines to initialize messages, send-receive data,
 * and notification mechanisms, (in 'c_com30.c')
 *****************************************************************************/         
#if (_os_NO_MESSAGE>0)
uint_fast8_t    _os_SetDataReceiver(SymbolicName, uint8_t *);
#if (_os_COM_0_COMSTATUS == _os_CO_COMEXTENDED)
StatusType      _os_IsMessageOwned(SymbolicName);
#endif
#endif

#if (_os_OS_0_ERRORHOOK==1 || _os_COM_0_COMERRORHOOK==1)
extern void     _os_LogError(StatusType,uint_fast8_t,int32_t,int32_t,int32_t);
#endif

/******************************************************************************
 * Maintenance of counter queues (in 'c_clock.c')
 *****************************************************************************/         
#if (_os_NO_ALARM >0 )
extern void     _os_UpdateCounter(CounterType);
extern void     _os_ActivateAlarm(_os_RTOSMEM _os_ALARM *);
uint_fast8_t    _os_DeleteAlarm(_os_RTOSMEM _os_ALARM *);
StatusType      _os_SetAlarm(_os_RTOSMEM _os_ALARM*,TickType,TickType);
#endif

/******************************************************************************
 * Init/Reset RTOS routines
 *****************************************************************************/         
extern void     _os_InitRTOS(void);
#if (_os_OS_0_MULTISTART>0)
extern void     _os_ResetRTOS(void);
#endif
/******************************************************************************
 * Reset actions for ISR objects
 *****************************************************************************/         
#if (_os_OS_0_MULTISTART>0)
extern void     _os_ResetISR(void);
#endif

/******************************************************************************
 * resources owned by tasks/isr
 *****************************************************************************/         
#if (_os_OS_0_STATUS == _os_OS_EXTENDED )
#if (_os_NO_ISR>0)
#if (MAX_RESOURCE_ISR>0)
uint_fast8_t    _os_IsResOwnedByCurrentISR(ResourceType);
#endif
#endif
uint_fast8_t    _os_IsResOwnedByCurrentTask(ResourceType);
#endif

/******************************************************************************
 * idle task label
 *****************************************************************************/         
extern void     _os_Idle(void);

/******************************************************************************
 * '_os_stack_address' array.
 * This array stores the start addresses of the stack areas for all tasks
 * (including the 'idle' task). The layout of the stack is target dependent,
 * thus it is only used at run time by target software to access stack fields.
 *
 * It is defined in 'g_conf.c' as:
 *      _os_EXTRASTACKINFO = {
 *              _os_STACKIDLEINFO,
 *              _os_STACKTASKINFO( 1 ),
 *              _os_STACKTASKINFO( 2 ), ...
 *              };
 * PRE:
 *      The name of the stack for the idle task is: _os_stack_table
 *****************************************************************************/

#define _os_EXTRASTACKINFO      _os_STACKMEM void*   _os_stack_address[_os_NO_TASK]

#define _os_STACKIDLEINFO       (_os_STACKMEM void *) &_os_idle_stack

#define _os_STACKTASKINFO(var)  (_os_STACKMEM void*) &_os_task_ ## var ## _stack

extern _os_EXTRASTACKINFO;

#if (_os_OS_0_STACKMONITOR >0)

/******************************************************************************
 * STACK OVERFLOW:
 * 
 * The RTOS uses many stacks. When the non-standard attribute STACKMONITOR 
 * is set as TRUE in the application OIL file, we need to test continously
 * that none of them have overflown. The RTOS does this in two ways:
 *
 * 1. Upon entrying RTOS code, i.e. when task calls a service, a test is
 * performed to detect the task's stack and the system stacks are still
 * in range. This is done with '_os_CheckStackRange()'. Should the test 
 * fail, the system directly shuts down.
 *
 * 2. During RTOS startup, the "last" two bytes of all the stacks are 
 * written with values STCK_PATTERN_1 (last byte) and STCK_PATTERN_2
 * (last but one). This is done in '_os_InitStackMonitor'.
 *    At every context switch the RTOS checks that the preempted task
 * has not overwritten these patterns in its last run.This is done with
 * '_os_StackMonitor()'.
 *
 * REMARK: These methods do not detect all overflows, obviously. 
 *         Code like:
 *              f()
 *              {
 *                      int array[1000];
 *                      ...
 *                       no system service calls 
 *                      ...
 *              }
 *         can run undetected should the stack pattern bytes not be 
 *         overwritten.
 *
 *         Extra refinement would require working in parallel with the compiler
 *         to do extra checks while building all function's prologs. 
 * 
 *****************************************************************************/
 
/******************************************************************************
 * ARRAY: _os_stack_monitor
 * This array stores the last addresses of the stacks plus an indication of
 * whether they do grow upwards or downwards.
 * The number of stacks in the RTOS equals to:
 *      _os_TASK_N0 + _os_SYSTEM_STACK_NO
 * where _os_SYSTEM_STACK_NO equals to the number of shared (by all tasks)
 * stack areas (typically interrupt stacks, temporary stacks, ..).
 *
 * The array is indexed as shown below:
 *      0                                       -> idle stack
 *      ..
 *      (_os_NO_TASK-1)                         -> last task stack
 *      (_os_NO_TASK)                           -> first global stack
 *      (_os_NO_TASK + _os_SYSTEM_STACK_NO - 1) -> last global stack
 * and it is initialized in '_os_TargetInitStackMonitor'.
 *
 * REMARK: This array is used by both '_os_StackMonitor' and '_os_CheckStackRange'
 *         routines.
 *****************************************************************************/

#define _os_STCK_PATTERN_1              0x5a
#define _os_STCK_PATTERN_2              0xa5

#define _os_STACK_UPWARDS               -1
#define _os_STACK_DOWNWARDS              1

struct _os_stack_info
{
        _os_STACKMEM uint8_t * address;
        int_fast8_t         growth;     
};

extern _os_RTOSMEM struct _os_stack_info _os_stack_monitor[];

/******************************************************************************
 * FUNCTION:    _os_CheckStackRange
 * DESCRIPTION:
 *      Whenever called, this function detects whether the stack of
 *      the running task and the system stacks are in range.
 *      Shuts down with  E_OS_SYS_UOFLW if task's stack  overflow.
 *                       E_OS_SYS_SOFLW if system stack overflow.
 *                       E_OS_SYS_TOFLW if target specific overflow.
 *      Returns E_OK otherwise.
 *      The RTOS calls this routine upon entrying RTOS code, i.e. at the
 *      beginning of every system service.
 *
 * REMARK:
 *      Only if non-standard attribute STACKMONITOR is set to TRUE.
 *****************************************************************************/
extern void _os_CheckStackRange(void);


/******************************************************************************
 * FUNCTION:    _os_StackMonitor
 * DESCRIPTION:
 *      This function is called after every task's run 
 *      to detect possible reasons of stack overflow.
 *      It basically tests that the initial patterns
 *      written on the edge of stacks have not been
 *      written in the last run.
 *      
 *      When no stack overflow is found, the routine
 *      simply returns. Had stack overflow happened,
 *      this routine would be calling ShutdownOS()
 *      with E_OS_SYS_UOFLW as parameter (indicating 
 *      the stack of the last user task), with 
 *      E_OS_SYS_SOFLW (indicating a global system
 *      stack) or with E_OS_SYS_TOFLW (indicating 
 *      target specific reasons) .
 *
 *      This method leaves almost no chance to the
 *      system to either hang or crush since most
 *      likely (unless this is the first or last tas
 *      k object in the OIL file) the corrupted data belong to
 *      the stack of task that has had no chance to execute.
 *      It basically tests that the initial patterns
 *      written on the edge of stacks have not been
 *      written in the last run.
 * REMARK:
 *      Only if non-standard attribute STACKMONITOR is set to TRUE.
 *****************************************************************************/
extern void _os_StackMonitor(void);


/******************************************************************************
 * FUNCTION:    _os_InitStackMonitor
 * DESCRIPTION:
 *      Write the stack patterns in the last two bytes of the stack
 *      (see previous discussion).
 *****************************************************************************/
extern void _os_InitStackMonitor(void);

#endif /* STACKMONITOR */


#endif



