/******************************************************************************
 *  FILE:       @(#)c_clock.c   1.97 04/11/17
 *  DESCRIPTION:
 *      Source code to maintain the queues of alarms.
 *      attached to all counters. All alarms are allocated
 *      statically at compile time.
 *              Counters are defined in:
 *                      _OS_COUNTER _os_counter_table[]
 *              Alarms are defined in:
 *                      _OS_ALARM _os_alarm_table[]
 *
 *      - SYSTEM TIMER: The RTOS uses a hardware clock to feed this counter.
 *        The non-standard atttibute USERTOSTIMER of the OS OIL object must 
 *        be set as TRUE if alarms are based on this counter.
 *        It is always the zeroth member of the '_os_counter_table' array.
 *
 *      - APPLICATION COUNTERS: Applications may define in the
 *        OIL file other counters. Applications refer to these counters
 *        via their names (indexes in the '_os_counter_table').
 *        The events triggering these counters are application specific.
 *        Applications use the system service 'IncrementCounter' to notify
 *        the RTOS that a new recurring event associated with a counter
 *        has been detected.
 *
 *        The routine '_os_UpdateCounter' is used for the counter maintenance
 *        for both system and application timers.
 *
 *        The only actions allowed for the alarms upon expiration are:
 *        - 1. activate a task
 *        - 2. set an event
 *        - 3. call-back
 *
 *      If no alarms objects are defined in the system (_os_NO_ALARM == 0),
 *      there is nothing to do in this module since (The RTOS clock is never
 *      used to share CPU time between task of equal priority, i.e. in OSEK
 *      systems there is no time-slicing).                                                               
 *****************************************************************************/
#include "g_conf.h"

#if (_os_NO_ALARM >0 )
#include "c_common.h"
#include "c_target.h"

#if     (_os_OS_0_POSTMORTEM != _os_OS_NONE)
#include "c_postmortem.h"
#endif

/* statics */
static _os_RTOSMEM _os_ALARM*       _os_UpdateQueue(_os_RTOSMEM _os_ALARM** times);
static _os_RTOSMEM _os_ALARM*       _os_ProcessAlarm(_os_RTOSMEM _os_ALARM* ptr);
static void                         _os_IncreaseCounterValue(CounterType cid);

/******************************************************************************
 * FUNCTION:            _os_UpdateCounter
 * DESCRIPTION:
 *      This function updates (one tick) the given counter.
 *      Actions associated to every counter are summed up alarms.
 *      All alarms of one counter are queued in one linked list.
 *      As a follow-up of the counter update one (or more) alarms
 *      might have expired. This routines takes care of performing
 *      the necessary actions.
 *
 *      Interrupts are always enabled upon entering the routine.
 *      The routine is called from the system rtos clock interrupt
 *      (id=SYSTEM_TIMER) or from system service 'IncrementCounter'.
 *******************************************************************************/
void _os_UpdateCounter(CounterType id)
{
        _os_RTOSMEM _os_ALARM  *ptr    = (_os_RTOSMEM _os_ALARM*) 0;

        /* protect critical code */
        _os_SuspendOS();
                
        /* counter increases */
        _os_IncreaseCounterValue(id);
        
        /* check counter queue is not empty */ 
        if(_os_counter_table[id].times)
        {
                /****************************************************************
                 * Expired alarms are taken out from the counter queue.         *
                 * The outcoming linked list of expired alarms remains local    *
                 * for this routine (head is 'ptr' tail is 0).                  *
                 * 'times' pointer will have been updated to point at the       *
                 * first of non-expired alarms.                                 *
                 * BUT no action has been taken yet after this routine.         *
                 ***************************************************************/
                ptr = _os_UpdateQueue((_os_RTOSMEM _os_ALARM**)&_os_counter_table[id].times);
                
                /* resume OS interrupts if user's counter */
                _os_ResumeOS();
                
                /************************************************************* 
                 * If 'ptr' is non-zero we have now a dequeued linked list
                 * (at least one element) of expired alarms which need
                 * to be attended.
                 * This routine performs the necessary actions for the alarm
                 * pointed by 'ptr' and then returns (if any) 
                 * the next expired alarm of the given queue.
                 * Since ISRs are enabled while executing this code, an ISR
                 * might preempt this flow and attempt to cancel/set the alarm.
                 * This situation is recorded in the 'status' field and 
                 * evaluated at the end of this routine.
                 ************************************************************/
                while(ptr) 
                {
                        ptr = _os_ProcessAlarm(ptr);
                }

                        
        } /* times */
        else
        {
                        _os_ResumeOS();
        }

        return;
}

/******************************************************************************
 *  FUNCTION:           _os_ActivateAlarm
 *  DESCRIPTION:
 *      The alarm has been prepared but needs to be entering its queue 
 *      at the precise position. This is what this routine does. 
 *      The alarm becomes active.
 *****************************************************************************/
void _os_ActivateAlarm(_os_RTOSMEM _os_ALARM *new)
{
        _os_RTOSMEM _os_ALARM **pptr;
        
        /* query the queue and set the new time block at the rightest position */
        for ( pptr = (_os_RTOSMEM _os_ALARM**)&(_os_counter_table[new->COUNTER].times) ; 
                        *pptr; pptr = &(*pptr)->next )
        {
                if ( new->ALARMTIME < (*pptr)->ALARMTIME )
                {
                        break;
                }
        }
        /* update the queue to this new element */
        new->next = *pptr;
        *pptr     = new;

        return;
}


/******************************************************************************
 *  FUNCTION:          _os_DeleteAlarm
 *  DESCRIPTION:   
 *      It deletes an specific alarm from its queue.
 *      Returns 1 in case the block was removed from the queue, 0 otherwise.
 *  REMARK:
 *      OS interrupts are suspended here.
 *****************************************************************************/
uint_fast8_t _os_DeleteAlarm( _os_RTOSMEM _os_ALARM* del)
{
        _os_RTOSMEM _os_ALARM** ptr;
       uint_fast8_t ret=0;
       
        for (ptr =(_os_RTOSMEM _os_ALARM**)&(_os_counter_table[del->COUNTER].times) ;
                        *ptr;ptr = &(*ptr)->next )
        {
             if ( *ptr == del )
             {  
                   *ptr = del->next;
                   del->ALARMTIME = 0;
                   del->CYCLETIME = 0;
                   del->next = 0;
                   ret = 1;
                   break;
              }  
        }
        
        return ret;
}


/******************************************************************************
 *  FUNCTION:          _os_SetAlarm
 *  DESCRIPTION:        It prepares the alarm and then activates it. 
 *****************************************************************************/
StatusType _os_SetAlarm(_os_RTOSMEM _os_ALARM* alarmb,TickType tstart,TickType tcycle)
{
        StatusType ret = E_OK;
        
        _os_SuspendOS();
        
        /***************************************************************** 
         * An alarm can be set only and only if both the 'ALARMTIME' and
         * 'CYCLETIME' fields are both zero. Otherwise it is whether 
         * running or it will soon run
         ****************************************************************/
        if (alarmb->ALARMTIME || alarmb->CYCLETIME)
        {
                ret = E_OS_STATE;
        }
        else
        {
                alarmb->ALARMTIME       = tstart;
                alarmb->CYCLETIME       = tcycle;

                if (alarmb->status == _os_AL_STOPPED)
                {
                        _os_ActivateAlarm(alarmb);
                        alarmb->status = _os_AL_RUNNING;
                }
                #if (_os_OS_0_STATUS == _os_OS_EXTENDED)
                else if (alarmb->status != _os_AL_BUSY)
                {
                        _os_ResumeOS();
                        /* we should be in 'processExpireTimerBlock' */ 
                        ShutdownOS(E_OS_SYS_ERROR);
                }
                #endif  
        }
        
        _os_ResumeOS();

        return E_OK;
}

/******************************************************************************
 * Increase the counter by one tick
 * ISR2s must have been disabled by the caller to avoid concurrency problems.
 *****************************************************************************/
static void _os_IncreaseCounterValue(CounterType cid)
{
        _os_RTOSMEM _os_COUNTER* cptr = (_os_RTOSMEM _os_COUNTER*)&_os_counter_table[cid];

        if(cptr->ticks == cptr->TICKSPERBASE)
        {
                /* update counter value */
                if ( cptr->value == cptr->MAXALLOWEDVALUE)
                {
                        cptr->value = 0;
                }
                else
                {
                        cptr->value++;
                }
                cptr->ticks=1; 
                
        }
        else
        {       
                cptr->ticks++;

        }
        return;
}


/******************************************************************************
 * Updates the queue and returns  a pointer to the oldest
 * among all just expired alarms or zero if no alarms have expired.
 * POST: Expired alarms will have been dequeued and the
 * 'times' pointer will have been updated to point at the
 * first among of non-expired alarms.
 *
 * ISR2 MUST be disabled by caller. 
 *****************************************************************************/
static _os_RTOSMEM _os_ALARM*       _os_UpdateQueue(_os_RTOSMEM _os_ALARM** times)
{       
      _os_RTOSMEM _os_ALARM * ptr = (_os_RTOSMEM _os_ALARM *) *times;
      _os_RTOSMEM _os_ALARM * aux = (_os_RTOSMEM _os_ALARM *) *times;  
      
       /* update the whole queue */
       while(aux)
       {
              aux->ALARMTIME--;
              aux=aux->next;
       }
        
            
       /* update new _os_counter_table[id].times */
       while ( *times && (*times)->ALARMTIME == 0 )
       {
             /* aux is the new dequeued alarm */
             aux                = *times;
             aux->status        = _os_AL_BUSY;
             *times             = (*times)->next;
       }
             
       if ( aux )
       {
             /* Dequeue expired counters */
             aux->next = (_os_RTOSMEM _os_ALARM*)0;
       }
       else
       {
             /* No timers expired */
             ptr = (_os_RTOSMEM _os_ALARM*)0;
             
       }           
       return ptr;      
}


/******************************************************************************
 * Processes next alarm of the expired queue and returns next available 
 * expired alarm (if any).
 *****************************************************************************/
static _os_RTOSMEM _os_ALARM*       _os_ProcessAlarm(_os_RTOSMEM _os_ALARM* alarm)
{
        _os_RTOSMEM _os_ALARM       *aux     = (_os_RTOSMEM _os_ALARM*)0;  
        _os_RTOSMEM _os_TASK        *task    = (_os_RTOSMEM _os_TASK*)&_os_task_table[alarm->TASK];

#if ( (_os_NO_EVENT>0) && (_os_OS_0_STATUS == _os_OS_EXTENDED) )
        uint_fast8_t    status;
#endif
        _os_SuspendOS();
        
        /* process the alarm */
        if ( alarm->ACTION == _os_AL_ACTIVATETASK )
        {
                /* check process is really in 'SUSPENDED' state. */
                 if (task->state == SUSPENDED )
                 {
                        /* set task to ready queue */
                        _os_SetTaskReady(task);
                 }
      
                 if (task->repeat < task->ACTIVATION)
                 {
                        task->repeat++;
                 }
        
                #if     (_os_OS_0_POSTMORTEM != _os_OS_NONE)
                        _os_LogPlaybackEvent(ALR_ACTIVATE,(uint8_t)alarm->id,
                                        (uint32_t)alarm->TASK);
                #endif
                 
        }
#if ( _os_NO_EVENT>0 )
        else if (alarm->ACTION == _os_AL_SETEVENT)
        {
                if ( 
                     task->state != SUSPENDED 
                     #if (_os_OS_0_STATUS == _os_OS_EXTENDED)
                     && task->events & alarm->EVENT 
                     #endif
                )
                {
                        /* set always the event */
                     task->set |= (task->events & alarm->EVENT);

                     #if        (_os_OS_0_POSTMORTEM != _os_OS_NONE)
                        _os_LogPlaybackEvent(ALR_EVENT,(uint8_t)alarm->id,
                                        (uint32_t)alarm->EVENT);
                     #endif
                     
                     if ( task->wait & alarm->EVENT ) 
                     {
                        /* task was waiting for this event */
                        _os_SetTaskReady(task);
                        task->wait = 0;
                     }                      
                }
        }
#endif
        else if (alarm->ACTION == _os_AL_ALARMCALLBACK) 
        {
                /* Callback routine: The body of this routine
                 * must be extremely short. */
                _os_SETCALLBACK();
                 ((void(*)(void))alarm->CALLBACK)();
                _os_CLRCALLBACK();
        }
        
        _os_ResumeOS();
       
        /* dequeue the alarm from the expired local queue */
        aux = alarm;
        alarm  = alarm->next;
        
        #if (_os_OS_0_STATUS == _os_OS_EXTENDED)
        _os_DisableInterrupts();
        status = aux->status;
        _os_EnableInterrupts();
        if (status != _os_AL_BUSY)
        {
                ShutdownOS(E_OS_SYS_ERROR);
        }
        #endif
        
        _os_SuspendOS();
        
        if ( aux->CYCLETIME==0)
        {
                aux->status = _os_AL_STOPPED;
        }
        else
        {
                aux->ALARMTIME = aux->CYCLETIME;                        
                aux->status = _os_AL_RUNNING;
                _os_ActivateAlarm(aux);
        }
        
        _os_ResumeOS();
        
        
        return alarm;
}


#endif /* NO_ALARM */

















