/**************************************************************************
**                                                                        *
**  FILE        :  log.c                                                  *
**                                                                        *
**  DESCRIPTION :  Source file for log() routine                          *
**                 Computes the 'natural logarithm' for the given value   *
**                                                                        *
**        The double coefficients are #2704 from Hart & Cheney. (16.65D)  *
**        The single coefficients are #2701 from Hart & Cheney. ( 8.48D)  *
**                                                                        *
**  Copyright 1996-2005 Altium BV                                         *
**                                                                        *
**************************************************************************/

#include <errno.h>
#include <math.h>

#ifdef __SINGLE_FP__
#pragma alias   logl    = _log_spf
#pragma alias   log             = _log_spf
#endif
#pragma alias   logf    = _log_spf


#define LOG2     0.693147180559945309e0f
#define SQRTO2   0.707106781186547524e0f
#define P0      -0.331355617479e1f
#define P1       0.89554061525e0f
#define Q0      -0.165677797691e1f
#define POLYNOMIAL(zsq,arg,exp) ( ( ( P1 * zsq + P0 ) / ( zsq + Q0 ) ) * arg + exp * LOG2 )

static
float
_log_spf( float arg )
{
        float   zsq;
        int     exp;

        if( arg <= 0.0f )
        {
                if ( arg == 0.0f )
                {                   
                    errno = ERANGE;
                }
                else
                {
                    errno = EDOM;
                }                                   
                return( -HUGE_VALF );
        }
        arg = frexpf( arg, &exp );
        if( arg < SQRTO2 )
        {
                arg = 2.0f * arg;
                exp = exp - 1;
        }

        arg = ( arg - 1.0f ) / ( arg + 1.0f );
        zsq = arg * arg;

        return POLYNOMIAL(zsq,arg,exp);
}


