/**************************************************************************
**                                                                        *
**  FILE        :  log.c                                                  *
**                                                                        *
**  DESCRIPTION :  Source file for log() routine                          *
**                 Computes the 'natural logarithm' for the given value   *
**                                                                        *
**        The double coefficients are #2704 from Hart & Cheney. (16.65D)  *
**        The single coefficients are #2701 from Hart & Cheney. ( 8.48D)  *
**                                                                        *
**  Copyright 1996-2005 Altium BV                                         *
**                                                                        *
**************************************************************************/

#include <errno.h>
#include <math.h>

#ifndef __SINGLE_FP__
#pragma alias   logl    = _log_dpf
#pragma alias   log             = _log_dpf


#define LOG2     0.693147180559945309e0
#define SQRTO2   0.707106781186547524e0
#define P0      -0.90174691662040536328986e2
#define P1       0.934639006428585382474e2
#define P2      -0.18327870372215593212e2
#define Q0      -0.45087345831020305748486e2
#define Q1       0.61761065598471302843e2
#define Q2      -0.20733487895513939345e2
#define POLYNOMIAL(zsq,arg,exp) ( ( ( ( P2 * zsq + P1 ) * zsq + P0 ) / \
                        ( ( ( zsq + Q2 ) * zsq + Q1 ) * zsq + Q0 ) ) * arg + exp * LOG2 )

static
double
_log_dpf( double arg )
{
        double  zsq;
        int     exp;

        if( arg <= 0.0 )
        {
                if ( arg == 0.0 )
                {                   
                    errno = ERANGE;
                }
                else
                {
                    errno = EDOM;
                }                                   
                return( -HUGE_VAL );
        }
        arg = frexp( arg, &exp );
        if( arg < SQRTO2 )
        {
                arg = 2.0 * arg;
                exp = exp - 1;
        }

        arg = ( arg - 1.0 ) / ( arg + 1.0 );
        zsq = arg * arg;

        return POLYNOMIAL(zsq,arg,exp);
}

#endif

