/**************************************************************************
**                                                                        *
**  FILE        :  _sinus.c                                               *
**                                                                        *
**  DESCRIPTION :  Source file for _sinus() routine                       *
**                 Does the actual computation of the 'sin()' and 'cos()' *
**                 routines                                               *
**                                                                        *
**      Coefficients double are #3368 from Hart & Cheney (16.25D).        *
**      Coefficients single are #3362 from Hart & Cheney ( 8.28D).        *
**                                                                        *
**  Copyright 1996-2005 Altium BV                                         *
**                                                                        *
**************************************************************************/

#include <math.h>

#ifdef __SINGLE_FP__
#pragma alias   _sinusl = __sinus_spf
#pragma alias   _sinus          = __sinus_spf
#endif
#pragma alias   _sinusf = __sinus_spf


#define TWOOPI   0.63661977236758134308f
#define P0       0.52930152776255e3f
#define P1      -0.17389497132272e3f
#define P2       0.1042302058487e2f
#define Q0       0.33696381989527e3f
#define Q1       0.2786575519992e2f
#define POLYNOMIAL(x,y) ( ( ( ( P2 * x + P1 ) * x + P0 ) * y ) / \
                          ( ( x + Q1 ) * x + Q0 ) )

static
float
__sinus_spf( float x, int quad )
{
        float   f;
        float   y;
        int     k;

        if( x < 0.0f)   /* Make sure X gets a positive number */
        {
                x = -x;
                quad = quad + 2;
        }

        x = x * TWOOPI;         /* underflow ? */

        if( x > 32764.0f)
        {
                y = modff( x, &f );
                /* From here, we don't need the original 'x' anymore
                 * So we just use it for other purposes
                 */
                x = f;
                x = x + quad;
                modff( 0.25f * x, &f );
                quad = x - 4.0f * f;
        }
        else
        {
                k = x;
                y = x - k;
                quad = ( quad + k ) & 03;
        }
        if ( quad & 0x01 )      /* Quadrant 1 or 3 */
                y = 1.0f - y;
        if( quad > 1 )
                y = -y;

        x = y * y;      /* y-square */

        return POLYNOMIAL(x,y);
}


