/**************************************************************************
**                                                                        *
**  FILE        :  _getflt.h                                              *
**                                                                        *
**  DESCRIPTION :  Common code for the _getfloat()/_getwflt() functions   *
**                                                                        *
**  Copyright 1996-2005 Altium BV                                         *
**                                                                        *
**************************************************************************/

#include <stdio.h>
#include <io.h>
#include <errno.h>
#include <float.h>

#include "_printflags.h"

static int _gethexfloat( unsigned char sign, unsigned short type,
                         unsigned int length, void* ap, struct _io* fp,
                         unsigned int* n_chars );
static int _getnan( unsigned char sign, unsigned short type, void* ap,
                    struct _io* fp, unsigned int* n_chars );
static int _getinf( unsigned char sign, unsigned short type, void* ap,
                    struct _io* fp, unsigned int* n_chars );

/* Return '0' if the conversion has success */
int GETFLOAT( uchar_t ch, unsigned short type, unsigned int length,
              void * ap, struct _io * fp, unsigned int *n_chars )
{
        int_t           c;
        unsigned char   sign    = 0;
        double          value   = 0.0;
        double          fract   = 0.1;
        int             exp;

        /* No length specified?
         */
        if ( length == 0 )
        {
                length = (unsigned int)(-1);
        }

        /* Read white space
         */
        do
        {
                (*n_chars)++;
                c = FGETC( fp );
        } while ( ISSPACE( c ) );

        /* First we use 'exp' as an 'OK' indicator. Conversion is OK when the
         * string of digits is not empty (not including the sign). When exp
         * equals '0' afterwards, possible conversion can be done.
         */
        exp = 1;

        /* First check for sign
         */
        if ( c == '-' || c == '+' )
        {
                if ( c == '-' )
                {
                        sign = 1;
                }
                (*n_chars)++;
                c = FGETC( fp );
                length--;
        }

        /* Check for NaN and INF
         */
        if ( c == 'n' || c == 'N' )
        {
                return _getnan( sign, type, ap, fp, n_chars );
        }
        else if ( c == 'i' || c == 'I' )
        {
                return _getinf( sign, type, ap, fp, n_chars );
        }

        while ( ISDIGIT( c ) && length-- )
        {
                value = value * 10.0 + (c - '0');
                (*n_chars)++;
                c = FGETC( fp );

                /* Conversion OK
                 */
                exp = 0;
        }

        if ( exp == 0 && value == 0.0 && c == 'x' )
        {
                return _gethexfloat(sign, type, length, ap, fp, n_chars);
        }

        if ( length && c == '.' )
        {
                while ( (*n_chars)++, c = FGETC( fp ), ISDIGIT( c ) )
                {
                        value = fract * (c - '0') + value;
                        fract /= 10.0;

                        /* Conversion OK
                         */
                        exp = 0;

                        if ( --length == 0 )
                        {
                                break;
                        }
                }
        }

        /* Still no valid character found, no conversion can be done
         */
        if ( exp )
        {
                (*n_chars)--;
                UNGETC( c, fp );

                return 1;
        }

        /* Make negative when neccessary
         */
        if ( sign )
        {
                value = -value;
        }

        sign = 0;

        /* Check for 'exponent'
         */
        if ( length && ( (c == 'e') || (c == 'E') ) )
        {
                (*n_chars)++;
                c = FGETC( fp );
                length--;

                /* Check for sign
                 */
                if ( length && (c == '-' || c == '+') )
                {
                        if ( c == '-' )
                        {
                                sign = 1;
                        }

                        (*n_chars)++;
                        c = FGETC( fp );
                        length--;
                }
        
                if ( ! ISDIGIT( c ) )
                {
                        return 1;
                }

                /* Read the exponent value
                 */
                while ( ISDIGIT( c ) && length-- )
                {
                        exp = exp * 10 + (c - '0');
                        (*n_chars)++;
                        c = FGETC( fp );
                }

                /* Cut the exponent at 2*DBL_MAX_10_EXP
                 */
                if ( exp > 2*DBL_MAX_10_EXP )
                {
                        exp = 2*DBL_MAX_10_EXP;
                }

                if ( sign )
                {
                        /* Negative exponent
                         */
#ifndef __SINGLE_FP__
                        while ( exp >= 100 )
                        {
                                value *= 1.0E-100;
                                exp -= 100;
                        }
#endif
                        while ( exp >= 10 )
                        {
                                value /= 1.0E10;
                                exp -= 10;
                        }
                        while ( exp >= 1 )
                        {
                                value /= 10.0;
                                exp -= 1;
                        }
                }
                else
                {
                        /* Positive exponent
                         */
#ifndef __SINGLE_FP__
                        while ( exp >= 100 )
                        {
                                value *= 1.0E100;
                                exp -= 100;
                        }
#endif
                        while ( exp >= 10 )
                        {
                                value *= 1.0E10;
                                exp -= 10;
                        }
                        while ( exp >= 1 )
                        {
                                value *= 10.0;
                                exp -= 1;
                        }
                }
        }

        /* Last read character was not used yet
         */
        (*n_chars)--;
        UNGETC( c, fp );

        if ( ap )
        {
                if ( ch == 'f' )
                {
                        if ( ( type & FLG_LONG_LONG ) == FLG_LONG )
                        {
                                *(double *)ap = value;
                        }
                        else if ( type & FLG_DOUBLE )
                        {
                                *(long double *)ap = value;
                        }
                        else
                        {
                                *(float *)ap = value;
                        }
                }
        }

        /* Success
         */
        return 0;
}

/* Return 0 if "NaN" can be parsed (case insensitive).
 *
 *      sign:           0 (positive), 1 (negative)
 *      type:           length modifiers
 *      ap:             pointer to data storage
 *      fp:             input file pointer
 *      n_chars:        number of characters read so far
 */
static int _getnan( unsigned char sign, unsigned short type, void* ap,
                    struct _io* fp, unsigned int* n_chars )
{
        int             status  = 1;
        int_t           c;

        do 
        {
                (*n_chars)++;
                c = FGETC( fp );
                if ( !( c == 'a' || c == 'A' ) )
                {
                        break;
                }

                (*n_chars)++;
                c = FGETC( fp );
                if ( !( c == 'n' || c == 'N' ) )
                {
                        break;
                }

                status = 0;
        } while ( 0 );

        if ( status == 0 )
        {
                /* For NaN, the fractional part of the floating point value is non-zero,
                 * the exponent is maximal. Sign is irrelevant.
                 */
#ifndef __SINGLE_FP__
                double2longs_t  value;

                value.s.lo = 1UL;
                value.s.hi = PUT_DOUBLE_EXPONENT( 0UL, DOUBLE_BIASED_NaN_EXP );

                if ( ( type & FLG_LONG_LONG ) == FLG_LONG )
                {
                        *(double *)ap = value.d;
                }
                else if ( type & FLG_DOUBLE )
                {
                        *(long double *)ap = value.d;
                }
                else
                {
                        *(float *)ap = (float) value.d;
                }               
#else
                float2long_t    value;

                value.l = PUT_FLOAT_EXPONENT( 1UL, FLOAT_BIASED_NaN_EXP );

                if ( ( type & FLG_LONG_LONG ) == FLG_LONG )
                {
                        *(double *)ap = value.f;
                }
                else if ( type & FLG_DOUBLE )
                {
                        *(long double *)ap = value.f;
                }
                else
                {
                        *(float *)ap = value.f;
                }
#endif
        }
        else
        {
                /* The last character is restored
                 */
                (*n_chars)--;
                UNGETC( c, fp );
        }

        return status;
}

/* Return 0 if "INF" or "INFINITY" can be parsed (case insensitive).
 *
 *      sign:           0 (positive), 1 (negative)
 *      type:           length modifiers
 *      ap:             pointer to data storage
 *      fp:             input file pointer
 *      n_chars:        number of characters read so far
 */
static int _getinf( unsigned char sign, unsigned short type, void* ap,
                    struct _io* fp, unsigned int* n_chars )
{
        int             status  = 1;
        int_t           c;

        do 
        {
                (*n_chars)++;
                c = FGETC( fp );
                if ( !( c == 'n' || c == 'N' ) )
                {
                        break;
                }

                (*n_chars)++;
                c = FGETC( fp );
                if ( !( c == 'f' || c == 'F' ) )
                {
                        break;
                }

                (*n_chars)++;
                c = FGETC( fp );
                if ( !( c == 'i' || c == 'I' ) )
                {
                        /* We have scanned "INF"
                         */
                        status = 0;

                        (*n_chars)--;
                        UNGETC( c, fp );
                        break;
                }

                (*n_chars)++;
                c = FGETC( fp );
                if ( !( c == 'n' || c == 'N' ) )
                {
                        break;
                }

                (*n_chars)++;
                c = FGETC( fp );
                if ( !( c == 'i' || c == 'I' ) )
                {
                        break;
                }

                (*n_chars)++;
                c = FGETC( fp );
                if ( !( c == 't' || c == 'T' ) )
                {
                        break;
                }

                (*n_chars)++;
                c = FGETC( fp );
                if ( !( c == 'y' || c == 'Y' ) )
                {
                        break;
                }

                status = 0;
        } while ( 0 );

        if ( status == 0 )
        {
                /* For INF, the fractional part of the floating point value is zero, the
                 * exponent is maximal. Sign is relevant.
                 */
#ifndef __SINGLE_FP__
                double2longs_t  value;

                value.s.lo = 0UL;
                value.s.hi = PUT_DOUBLE_EXPONENT( PUT_DOUBLE_SIGN(0UL, sign), DOUBLE_BIASED_INF_EXP );

                if ( ( type & FLG_LONG_LONG ) == FLG_LONG )
                {
                        *(double *)ap = value.d;
                }
                else if ( type & FLG_DOUBLE )
                {
                        *(long double *)ap = value.d;
                }
                else
                {
                        *(float *)ap = (float) value.d;
                }               
#else
                float2long_t    value;

                value.l = PUT_FLOAT_EXPONENT( PUT_FLOAT_SIGN(0UL, sign), FLOAT_BIASED_INF_EXP );

                if ( ( type & FLG_LONG_LONG ) == FLG_LONG )
                {
                        *(double *)ap = value.f;
                }
                else if ( type & FLG_DOUBLE )
                {
                        *(long double *)ap = value.f;
                }
                else
                {
                        *(float *)ap = value.f;
                }
#endif
        }
        else
        {
                /* The last character is restored
                 */
                (*n_chars)--;
                UNGETC( c, fp );
        }

        return status;
}

/* Return 0 if a hexadecimal floating point number can be parsed.
 *
 *      sign:           0 (positive), 1 (negative)
 *      type:           length modifiers
 *      length:         maximum number of characters to read
 *      ap:             pointer to data storage
 *      fp:             input file pointer
 *      n_chars:        number of characters read so far
 */
static int _gethexfloat( unsigned char sign, unsigned short type,
                         unsigned int length, void* ap, struct _io* fp,
                         unsigned int* n_chars )
{
        int_t           c;
        int             dotbits         = 0;
        int             numbits         = 0;
        unsigned char   first           = 1;
        unsigned char   dotseen         = 0;
#ifndef __SINGLE_FP__
        int             bitcount        = DOUBLE_FRACTION_SIZE;
        double2longs_t  value;
        unsigned long   lo_fract        = 0;
        unsigned long   hi_fract        = 0;
#else
        int             bitcount        = FLOAT_FRACTION_SIZE;
        float2long_t    value;
        unsigned long   fraction        = 0;
#endif
        int             exponent        = 0;
        unsigned char   signed_exponent = 0;

        /* Skip the 'x' character
         */
        length--;

#ifndef __SINGLE_FP__
        value.d = 0.0;
#else
        value.f = 0.0;
#endif

        /* Scan for the fraction of the floating point number.
         */
        while ( (*n_chars)++, c = FGETC( fp ), ( ISXDIGIT( c ) || ( c == '.' && !dotseen ) ) )
        {
                if ( first && c == '0' )
                {
                        /* Skip leading '0'-characters after the 0x.
                         */
                }
                else if ( c != '.' )
                {
                        c = TOUPPER( c );
                        if ( c > '9' )
                        {
                                c = c + 10 - 'A';
                        }
                        else
                        {
                                c = c - '0';
                        }

                        if ( first )
                        {
                                int_t   tmp_c   = c;

                                first = 0;
                                while ( tmp_c != 1 )
                                {
                                        dotbits++;
                                        numbits++;
                                        tmp_c >>= 1;
                                }

                                /* Mask of the implicit 1
                                 */
                                c &= (( 1 << dotbits ) - 1);
                        }
                        else
                        {
                                if ( !dotseen )
                                {
                                        dotbits += 4;
                                }
                                numbits  = 4;
                        }

                        /* Only add the digits that can be stored in the
                         * fractional part of the floating point number
                         */
                        if ( numbits && bitcount )
                        {
                                if ( numbits > bitcount )
                                {
                                        c     >>= ( numbits - bitcount );
                                        numbits = bitcount;
                                }

#ifndef __SINGLE_FP__
                                hi_fract = ((hi_fract << numbits) | (lo_fract >> (32 - numbits)));
                                lo_fract = ((lo_fract << numbits) | (unsigned long) c);
#else
                                fraction = ((fraction << numbits) | (unsigned long) c);
#endif
                                bitcount -= numbits;
                        }
                }
                else
                {
                        dotseen = 1;
                }

                if ( --length == 0 )
                {
                        break;
                }
        }

        /* Check if we need to shift the fractional part
         */
        if ( bitcount > 0 )
        {
#ifndef __SINGLE_FP__
                hi_fract   = ((hi_fract << bitcount) | (lo_fract >> (32 - bitcount)));
                lo_fract <<= bitcount;
#else
                fraction <<= bitcount;
#endif
        }

        /* Scan for possible exponent.
         */
        if ( length && (c == 'p' || c == 'P') )
        {
                (*n_chars)++;
                c = FGETC( fp );
                length--;

                if ( length && (c == '-' || c == '+') )
                {
                        if ( c == '-' )
                        {
                                signed_exponent = 1;
                        }

                        (*n_chars)++;
                        c = FGETC( fp );
                        length--;
                }
                                
                while ( ISDIGIT( c ) && length-- )
                {
                        exponent = exponent * 10 + (c - '0');

                        (*n_chars)++;
                        c = FGETC( fp );
                }
        }

        /* Last read character was not used yet
         */
        if ( length )
        {
                (*n_chars)--;
                UNGETC( c, fp );
        }

        /* OK, try to assign the value
         */
        if ( ap )
        {
                /* Adjust the exponent to its sign
                 */
                if ( signed_exponent )
                {
                        exponent = -exponent;
                }

                /* Adjust the exponent for the number of bits before the
                 * decimal point.
                 */
                exponent += dotbits;

#ifndef __SINGLE_FP__
                value.s.lo = lo_fract;
                value.s.hi = hi_fract;
                value.s.hi = PUT_DOUBLE_EXPONENT(value.s.hi, exponent + DOUBLE_BIAS);
                value.s.hi = PUT_DOUBLE_SIGN(value.s.hi, sign);

                if ( ( type & FLG_LONG_LONG ) == FLG_LONG )
                {
                        *(double *)ap = value.d;
                }
                else if ( type & FLG_DOUBLE )
                {
                        *(long double *)ap = value.d;
                }
                else
                {
                        *(float *)ap = (float) value.d;
                }
#else
                value.l = fraction;
                value.l = PUT_FLOAT_EXPONENT(value.l, exponent + FLOAT_BIAS);
                value.l = PUT_FLOAT_SIGN(value.l, sign);

                if ( ( type & FLG_LONG_LONG ) == FLG_LONG )
                {
                        *(double *)ap = value.f;
                }
                else if ( type & FLG_DOUBLE )
                {
                        *(long double *)ap = value.f;
                }
                else
                {
                        *(float *)ap = value.f;
                }
#endif
        }

        return 0;
}
