/**************************************************************************
**                                                                        *
**  FILE        :  _flsbuf.c                                              *
**                                                                        *
**  DESCRIPTION :  Generic output fallback routine when putc() et. al.    *
**                 are unable to store character `c' directly. Output to  *
**                 a string is not done here but intercepted by fputc().  *
**                 The return value is `c' or EOF on error.               *
**                                                                        *
**  Copyright 1996-2005 Altium BV                                         *
**                                                                        *
**************************************************************************/

#include <stdio.h>
#include <io.h>

/*
 * The only way we get here is via the putc() family of functions/macros
 * which decrement _cnt unconditionally and call us when the result is
 * negative. So, _cnt is guaranteed to be negative: -1 for a plain buffer
 * overflow, `c' being the culprit) or even less than -1 when the buffer
 * is still being used as input buffer.
 *
 * However, _cnt == -1 is not authoritative about the output buffer
 * contents: In case of full buffering it is either completely full or
 * completely empty. In case of line buffering it may be partially filled:
 * by keeping _cnt zero we can push the '\n' handling usually done in
 * putc() et al. downwards to here, saving code at the cost of speed.
 * Speed isn't expected to be an issue for line buffered output.
 *
 * Put another way, a negative _cnt is just a trigger for putc() et. al.
 * to call us and we still have to figure out what to do.
 */

int _flsbuf(int c, register FILE *fp)

#ifdef WCHAR_SUPPORT_ENABLED
{
        fp->_flag |= _IOCHAR;   /* tell fwide() that fp is byte-oriented */
        return _dofls(c, fp);
}
int _dofls(int c, register FILE *fp)
#endif

{
        int     n;
        char    ch;

        /* Undo unconditional decrement done by putc() family */
        fp->_cnt++;

        /* Check if file is writable */
        if (!(fp->_flag & _IOWR))
        {
                fp->_flag |= _IOERR;
                return EOF;
        }

        /* Unbuffered I/O */
        if (fp->_flag & _IONBF)
        {
                ch = (char)c;
                if (_write(fileno(fp), &ch, 1) != 1)
                {
                        fp->_flag |= _IOERR;
                        c = EOF;
                }
                return c;
        }

        /* Buffered I/O */
        /* First drop buffered input, if any, clearing _cnt in the process */
        if (fp->_cnt < 0)
        {
                _fflush(fp);
        }

        /* Check for the line buffering trick which keeps _cnt at zero */
        if (fp->_flag & _IOLBF)
        {
                *fp->_ptr++ = (char)c;
                if (c != '\n' && fp->_ptr - fp->_base != fp->_bufsiz)
                {
                        return c;       /* at least room for one more char so continue buffering */
                }
        }

        /* Must flush */
        ch = (char)c;
        n = fp->_ptr - fp->_base;
        if (n && _write(fileno(fp), fp->_base, n) != n)
        {
                fp->_flag |= _IOERR;
                c = EOF;
        }
        fp->_ptr = fp->_base;
        if (!(fp->_flag & _IOLBF))
        {
                *fp->_ptr++ = ch;
                fp->_cnt = fp->_bufsiz - 1;
        }

        return c;
}
