/* @(#)uart.h   1.3 */

#ifndef UART_H
#define UART_H

#include <osek/uio.h>
#include <osek/ioreq.h>

/*
 * Notes about uart driver modes
 * =============================
 *
 * Cooked [+ xonxoff] [+ echo]
 * ---------------------------
 * In this mode a read request will complete upon ^C or a newline. It is
 * intended for implementing user interfaces. Features are:
 * -    erase char processing: <BS> and <DEL>. For proper erase char processing
 *      anything outside the printable ASCII character set will be echoed as a
 *      question mark. Characters with the eight bit set are assumed to be printable.
 * -    input \r -> \n translation
 * -    output post processing: \n -> \r\n
 * -    three echoing modes: on, off or in non-exposure mode, e.g. for password dialogs.
 *
 *
 * Raw [+ xonxoff]
 * ---------------
 * This mode of operation is suitable for simple machine-machine communication.
 * Read requests will complete once the buffer is full. Of course it is still
 * possible to abort the current read request in order to obtain what has been
 * received so far.
 *
 *
 * Slip (Serial Line IP)
 * ---------------------
 * In this mode, every read and write will handle exactly one IP datagram. The
 * framing will be taken care of by the driver. The driver will start every
 * packet with an END frame marker to avoid packet invalidation by spurious
 * characters (line noise?). In addition, it will silently discard empty frames
 * at the receiver. See also RFC 1055.
 *
 *
 * Notes about optional features
 * =============================
 *
 * XON/XOFF
 * --------
 * Software handshaking is available but off by default for every uart mode. it
 * operates in both directions (i.e. "tandem" mode). Low and high water marks
 * are configurable, see the SKIDLEN* macro definitions.
 *
 *
 * Echo
 * ----
 * This is only supported in "cooked" mode and on by default in that case.
 * Echoing may require infinite memory after receiving an XOFF. So it can be broken
 * easily. As long as the receiver baudrate is not faster than the transmitter baudrate
 * it is not likely to get congestion. The minimum echo buffer size we need is 2: right
 * after receiving a \n we need to echo \r\n and the transmitter might still be busy.
 * A backspace requires echoing three characters back but is usually only typed when the
 * transmitter is idle. But it is relatively cheap to handle that case correctly as well
 * as long as the keyboard repeat rate is at least three times slower than the transmitter
 * baud-rate. But note that since echoing is merely an aid for manual input and not for
 * machine-machine communication we don't care too much.
 * 
 * The size of the echo transmission queue is specified as a power of two (size: 2 ** TXBUF_ORDER):
 */
#define TXBUF_ORDER     2                       /* 4 character buffer, order must be >0 */
#define TXBUF_SIZE      (1 << TXBUF_ORDER)
#define TXBUF_NORM(i)   ((i) & (TXBUF_SIZE - 1))

enum echo
{
        ECHO_OFF = 0,
        ECHO_ON = 1,
        ECHO_HASH = '#'                         /* >ECHO_ON: echo but don't disclose actual input */
};

typedef struct uart     uart_t;
typedef struct xonxoff  xonxoff_t;
typedef enum echo       echo_t;

struct uart
{
        ResourceType    resource;               /* locking */
        ioqueue_t       rx;
        ioqueue_t       tx;
        bool_t          tx_running;             /* transmitter interrupt pending */
        bool_t          (*rx_data)(uart_t *, ioreq_t *, char);
        int             (*tx_data)(uart_t *);

        /*
         * XON/XOFF state (cooked and raw mode)
         */
        xonxoff_t *     xonxoff;                /* XON/XOFF operations or NULL */
        bool_t          xoff_received;
        char            xonxoff_urgent;         /* pending XON/XOFF to transmit or zero */
        int             xonxoff_rxroom;         /* for triggering XON/XOFF transmissions */

        /*
         * echo and output post processing, integrated with cooked mode
         */
        echo_t          tx_echo;
        char            tx_buf[TXBUF_SIZE];     /* echo and output post processing queue */
        int             tx_bufin;
        int             tx_bufout;

        /*
         * SLIP state
         */
        bool_t          rx_slipesc;             /* ESC received, translate next char */
        int             tx_slipstate;           /* no enum, see uart_tx_slip() */
};

/*
 * Device state, see uartdev.c
 */
extern  uart_t          uart_dev[];

/*
 * Device driver initialization, locking and initial I/O configuration.
 * All functions (except lock related ones) are intended for StartupHook()
 * context: interrupts are off so no locking needed and the device array
 * has been cleared in advance by cstart.
 * uart_config() is the root of all uart initialization. It configures the
 * hardware and informs uart.c (the hardware independent slip/tty driver)
 * about it.
 *
 * First, uart_config() should call uart_init (mandatory).
 * Next, it should call one out of uart_cooked(), uart_raw() and uart_slip().
 * uart_config_xonxoff() is optional (don't try that with slip)
 *
 * This could partially be done with the uart_set/enable/disable* functions
 * but that would pull in more code in cases when this matters most: small
 * applications with straightforward uart use.
 */
extern  void            uart_config(void);
extern  void            uart_init(device_t dev, ResourceType resource);
inline  void            uart_lock(device_t dev)
{
        GetResource(uart_dev[dev].resource);
}
inline  void            uart_unlock(device_t dev)
{
        ReleaseResource(uart_dev[dev].resource);
}
extern  void            uart_config_cooked(device_t dev);
extern  void            uart_config_raw(device_t dev);
extern  void            uart_config_slip(device_t dev);
extern  void            uart_config_xonxoff(device_t dev);

/*
 * Operations necessary for abstracting uartdev.c (hardware interface).
 * The tx_getchar function returns a negative value (-1, the traditional
 * EOF) when there is no more data to transmit.
 */
extern  void            uart_rx_putchar(device_t dev, char c);
extern  int             uart_tx_getchar(device_t dev);
extern  void            uart_transmit(device_t dev);

/*
 * Unified I/O
 */
extern  int_least8_t    uart_open(const char *name, uint_least16_t flags);
extern  void            uart_read(device_t dev, ioreq_t *request);
extern  ioreq_t *       uart_read_abort(device_t dev);
extern  void            uart_write(device_t dev, ioreq_t *request);
extern  ioreq_t *       uart_write_abort(device_t dev);

/*
 * I/O controls.
 * A generic device "control" method for everything is not really convenient
 * due to memory usage: one device control call would pull in all device
 * control code.
 */
extern  void            uart_set_cooked(device_t dev);
extern  void            uart_set_raw(device_t dev);
extern  void            uart_set_slip(device_t dev);
extern  void            uart_enable_xonxoff(device_t dev);
extern  void            uart_disable_xonxoff(device_t dev);
extern  void            uart_echo(device_t dev, echo_t mode);

#endif
