/*************************************************************************
**  VERSION CONTROL:    @(#)osek.h      1.44 04/11/12 
**  DESCRIPTION:        OSEK interface ( brief description )
**                      Functionality corresponds to OSEK/VDX Version2.2
**
**                      Please refer to OSEK documentation for a precise
**                      description of the interface (OS2.2)
**                              http://www.osek-vdx.org
*************************************************************************/
#ifndef _OSEK_H
#define _OSEK_H

#include <osek/types.h>

#if defined(_OS_INCLUDE_STDINT)
#include <stdint.h>
#endif

/* OSEK system services id's  */ 
#define _os_N_GetResource                    1
#define _os_N_ReleaseResource                2
#define _os_N_StartOS                        3
#define _os_N_GetActiveApplicationMode       4
#define _os_N_ShutdownOS                     5
#define _os_N_StartCOM                       6
#define _os_N_StopCOM                        7
#define _os_N_GetCOMApplicationMode          8
#define _os_N_GetTaskID                      9
#define _os_N_ActivateTask                  10
#define _os_N_TerminateTask                 11
#define _os_N_GetTaskState                  12
#define _os_N_Schedule                      13
#define _os_N_ChainTask                     14
#define _os_N_GetAlarmBase                  15
#define _os_N_GetAlarm                      16
#define _os_N_SetRelAlarm                   17
#define _os_N_SetAbsAlarm                   18
#define _os_N_CancelAlarm                   19
#define _os_N_SetEvent                      20
#define _os_N_GetEvent                      21
#define _os_N_WaitEvent                     22
#define _os_N_ClearEvent                    23
#define _os_N_IncrementCounter              24
#define _os_N_InitMessage                   25
#define _os_N_SendMessage                   26
#define _os_N_ReceiveMessage                27
#define _os_N_GetMessageStatus              28
#define _os_N_DisableAllInterrupts          29
#define _os_N_EnableAllInterrupts           30
#define _os_N_SuspendAllInterrupts          31
#define _os_N_ResumeAllInterrupts           32
#define _os_N_SuspendOSInterrupts           33
#define _os_N_ResumeOSInterrupts            34
#define _os_N_IsStackInRange                35

#define _os_N_SYSSER                        35          

/**************************************************************** 
 * MACRO:       _os_USETRAPS
 * 
 * DESCRIPTION: This target macro is defined:
 *      - 'non-zero' in targets where support for system 
 *      traps/interrupts exists (and the mechanism is used to 
 *      implement the OSEK system services).
 *      - 'zero' otherwise
 *      - Defined in target file: types.h
 *
 * REMARK:
 *      You can expect '_os_USETRAPS' to be defined as zero
 *      for all OSEK systems. Main reasons are:
 *      - Run-time efficiency.
 *      - Not always supported by the hardware.
 *      - Size (using system calls means the whole RTOS library
 *      must be linked!).
 *      - OSEK makes no real difference between User and Kernel
 *      modes (actually, interfaces like 'DisableAllInterrupts'
 *      implicitly assume that whether such a distinction in modes
 *      do not exist or that system runs always in Kernel Mode).
 ***************************************************************/

/***************************************************************
 * NAME::       _os_SYSINLINE
 * 
 * DESCRIPTION:
 *      This target macro defines the 'inline-ability'
 *      of the OSEK system services.
 *      - Defined in target file: types.h
 * 
 * REMARK:
 *      The same discussion as before applies here.
 *      OSEK systems shall NOT inline the system services
 *      (no use of system calls).
 **************************************************************/


/***************************************************************
 * Name:        TaskType
 *
 * Description: This data type identifies a task.
 **************************************************************/
typedef int_fast8_t    TaskType;

/***************************************************************
 * Name:        TaskRefType
 *
 * Description: This data type points to a variable of TaskType.
 **************************************************************/
typedef TaskType *      TaskRefType;

/***************************************************************
 * Name:        CounterType
 *
 * Description: This data type identifies a counter object
 **************************************************************/
typedef int_fast8_t     CounterType;


/***************************************************************
 * Name:        AlarmType
 *
 * Description: This data type identifies an alarm object
 **************************************************************/
typedef int_fast8_t     AlarmType;


/***************************************************************
 * Name:        ResourceType
 *
 * Description: This data type identifies a resource.
 **************************************************************/
typedef int_fast8_t     ResourceType;

/***************************************************************
 * Name:        EventMaskType
 *
 * Description: This data type identifies an event mask.
 * Remarks:     Max 32-bits (i.e. Max 32 events in OIL file)
 **************************************************************/
typedef uint32_t  EventMaskType;

/***************************************************************
 * Name:        EventMaskRefType
 *
 * Description: This data type references an event mask.
 **************************************************************/
typedef EventMaskType*  EventMaskRefType;

/***************************************************************
 * Name:        TickType
 *
 * Description: This data type represents count values in ticks.
 * Remarks:     Always 32 bits.
 **************************************************************/
typedef uint32_t  TickType;


/***************************************************************
 * Name:        OSMAXALLOWEDVALUE
 * 
 * Description: Maximum value of counter_x in ticks.
 **************************************************************/
#define         OSMAXALLOWEDVALUE               UINT32_MAX

/***************************************************************
 * Name:        TickRefType
 *
 * Description: This data type points to the data type TickType.
 **************************************************************/
typedef TickType*       TickRefType; 

/***************************************************************
 * Name:        AlarmBaseType
 *
 * Description: This data type represents a structure for storage
 *              of counter characteristics. The individual elements
 *              of the structure are (all are TickType):
 * 
 *              - maxallowedvalue: Maximum possible allowed count value
 *               in ticks.
 *              - ticksperbase: Number of ticks required to reach 
 *               a counter-specific (significant) unit.
 *              - mincycle:     Smallest allowed value for the 
 *              cycle-parameter of SetRelAlarm/SetAbsAlarm 
 **************************************************************/
typedef struct 
{
        TickType        maxallowedvalue;
        TickType        ticksperbase;
        TickType        mincycle;
}AlarmBaseType;


/***************************************************************
 * Name:        AlarmBaseRefType
 *
 * Description: This data type points to the data type AlarmBaseType.
 **************************************************************/
typedef AlarmBaseType*  AlarmBaseRefType; 

/***************************************************************
 * Name:        AppModeType
 *
 * Description: This data type represents the application mode
 **************************************************************/
typedef int_fast8_t     AppModeType;

/***************************************************************
 * Name:        TaskStateType
 *
 * Description: This data type identifies the state of a  task.
 **************************************************************/
typedef enum
{
        RUNNING,
        READY,
        WAITING,
        SUSPENDED       
} TaskStateType;

/***************************************************************
 * Name:        TaskStateRefType
 *
 * Description: This data type points to a variable of the data type TaskStateType.
 **************************************************************/
typedef TaskStateType *TaskStateRefType;

typedef int_fast8_t   IsrType;
typedef int_fast8_t   OsType;

/***************************************************************
 * Name:        OSServiceIdType
 *
 * Description: This data type represents the identification of system services.
 **************************************************************/
typedef uint_fast8_t  OSServiceIdType;

/***************************************************************
 * Name:        SymbolicName
 *
 * Description: A SymbolicName is a type that shall provide a range of possible
 *              values for identification of all application specific OSEK messages.
 **************************************************************/
typedef int_fast8_t  SymbolicName;

typedef SymbolicName MessageType;

/***************************************************************
 * Name:        ApplicationDataRef
 *
 * Description: Pointer to the Data field of an application message.
 **************************************************************/
typedef void * ApplicationDataRef;

/***************************************************************
 * Name:        DataLength
 *
 * Description: Contains the length information of an application message.
 * Remarks:     Max 64 kB
 **************************************************************/
typedef uint16_t DataLength;

/***************************************************************
 * Name:        LengthRef
 *
 * Description: Pointer to data field containing length information
 **************************************************************/
typedef DataLength * LengthRef;


/***************************************************************
 * Name:        COMApplicationModeType
 *
 * Description: Identifier for selected COM application mode
 **************************************************************/
typedef int_fast8_t  COMApplicationModeType;


/***************************************************************
 * Name:        COMShutdownModeType
 *
 * Description: Identifier for selected COM shutdown mode
 **************************************************************/
typedef int_fast8_t  COMShutdownModeType;

#define COM_TRUE        1
#define COM_FALSE       0

/***************************************************************
 * Name:        CalloutReturnType
 *
 * Description: Indicates at exit of a callout whether the IL shall 
 *              continue or abandon further processing of the current
 *              message.
 **************************************************************/
typedef uint_fast8_t CalloutReturnType;    


/***************************************************************
 * Name:        COMServiceIdType
 *
 * Description: Unique Identifier of an OSEK COM service.
 **************************************************************/
typedef OSServiceIdType COMServiceIdType;


typedef int_fast8_t     ComType;



/***************************************************************
 * Name:        StatusType
 *
 * Description: This data type is used for all status information the 
 * API services offer. Naming convention: all errors for API services
 * start with E_. Those reserved for the operating system will begin 
 * with E_OS_.
 **************************************************************/
typedef  uint_fast8_t StatusType;

/* System services return values */
#define          E_OK                   (StatusType)0

/* OSEK OS error codes 1 - 31 */
        /* API errors  1 - 8 */
#define         E_OS_ACCESS             (StatusType)1
#define         E_OS_CALLEVEL           (StatusType)2
#define         E_OS_ID                 (StatusType)3
#define         E_OS_LIMIT              (StatusType)4
#define         E_OS_NOFUNC             (StatusType)5
#define         E_OS_RESOURCE           (StatusType)6
#define         E_OS_STATE              (StatusType)7
#define         E_OS_VALUE              (StatusType)8
        /* implementation errors */
#define         E_OS_SYS_ERROR          (StatusType)16
#define         E_OS_SYS_VALUE          (StatusType)17
#define         E_OS_SYS_UOFLW          (StatusType)18  
#define         E_OS_SYS_SOFLW          (StatusType)19  
#define         E_OS_SYS_TOFLW          (StatusType)20  
#define         E_OS_SYS_RTOFLW         (StatusType)21  
#define         E_OS_SYS_RUNLEVEL       (StatusType)22  

/* OSEK COM error codes 32 - 63 */
#define         E_COM_ID                (StatusType)32
#define         E_COM_LENGTH            (StatusType)33
#define         E_COM_LIMIT             (StatusType)34
#define         E_COM_NOMSG             (StatusType)35

/* Implementation Specific Status Codes */
#define         E_COM_SYS_LEVEL         (StatusType)36
#define         E_COM_SYS_FILTER        (StatusType)37
#define         E_COM_SYS_START         (StatusType)38



/**************************************************************************/
/* Constants      Constants       Constants      Constants      Constants */
/**************************************************************************/

/* Constant of data type CounterType to address the OS system timer */
#define         SYSTEM_TIMER               (CounterType)0

/* Default application mode, always a valid parameter to StartOS */ 
#define         OSDEFAULTAPPMODE        (AppModeType)0

/* Constant of data type TaskType for a not defined task */
#define         INVALID_TASK            (TaskType)(-1)

/***************************************************************
 * Constant of data type ResourceType to lock the scheduler
 * If a task has to protect itself against preemptions by other tasks, 
 * it can lock the scheduler. The scheduler is treated like a resource
 * which is accessible to all tasks. Therefore a resource with a 
 * predefined name RES_SCHEDULER is generated.
 **************************************************************/
#define         RES_SCHEDULER           (ResourceType)0

/* shutdowns occurs immediately without waiting for pending interrupts */
#define COM_SHUTDOWN_IMMEDIATE          (_os_COMShutdownModeType)0 

#if (_os_USETRAPS>0)

/***********************************************************************
 * _os_sysserN macros are defined in 'types.h' to provide  
 * prototypes for OSEK inline system services (software traps).
 *
 * #define _os_sysserN(type,name,type1,arg1,...,typeN,argN)
 * type name(type1,arg1,...argN)        
 * ...                                                          
 *
 * If this is the case, the next inline prototypes would
 * become redundant (but harmless).
 *
 * REMARK:
 *      See previous discussion for regarding OSEK systems
 *      and system calls.
 **********************************************************************/

_os_SYSINLINE _os_sysser1(StatusType,ActivateTask,TaskType,TaskID)
_os_SYSINLINE _os_sysser1(StatusType,ChainTask,TaskType,TaskID)
_os_SYSINLINE _os_sysser0(StatusType,Schedule)
_os_SYSINLINE _os_sysser0(StatusType,TerminateTask)
_os_SYSINLINE _os_sysser2(StatusType,GetTaskState,TaskType,TaskID,
                TaskStateRefType,State)
_os_SYSINLINE _os_sysser1(StatusType,GetTaskID,TaskRefType,TaskID);
_os_SYSINLINE _os_sysser1(StatusType,IncrementCounter,
                CounterType,CounterID)
_os_SYSINLINE _os_sysser2(StatusType,GetAlarm,AlarmType,AlarmID,
                TickRefType,Tick)
_os_SYSINLINE _os_sysser2(StatusType,GetAlarmBase,AlarmType,AlarmID,
                AlarmBaseRefType,Info)
_os_SYSINLINE _os_sysser3(StatusType,SetRelAlarm,AlarmType,AlarmID,
                TickType,increment,TickType,cycle)
_os_SYSINLINE _os_sysser3(StatusType,SetAbsAlarm,AlarmType,AlarmID,
                TickType,start,TickType,cycle)
_os_SYSINLINE _os_sysser1(StatusType,CancelAlarm,AlarmType,AlarmID)
_os_SYSINLINE _os_sysser2(StatusType,SetEvent,TaskType,TaskID,
                EventMaskType,Mask)
_os_SYSINLINE _os_sysser2(StatusType,GetEvent,TaskType,TaskID,
                EventMaskRefType,Event)
_os_SYSINLINE _os_sysser1(StatusType,WaitEvent,EventMaskType,Mask)
_os_SYSINLINE _os_sysser1(StatusType,ClearEvent,EventMaskType,Mask)
_os_SYSINLINE _os_sysser1(StatusType,GetResource,ResourceType,ResID)
_os_SYSINLINE _os_sysser1(StatusType,ReleaseResource,ResourceType,ResID)
_os_SYSINLINE _os_sysser1(StatusType,StartCOM,COMApplicationModeType,Mode)
_os_SYSINLINE _os_sysser1(StatusType,StopCOM,COMShutdownModeType,Mode)
_os_SYSINLINE _os_sysser2(StatusType,InitMessage,SymbolicName,Message,
                ApplicationDataRef,DataRef)
_os_SYSINLINE _os_sysser2(StatusType,SendMessage,SymbolicName,Message,
                ApplicationDataRef,DataRef)
_os_SYSINLINE _os_sysser2(StatusType,ReceiveMessage,SymbolicName,Message,
                ApplicationDataRef,DataRef)
_os_SYSINLINE _os_sysser1(StatusType,GetMessageStatus,SymbolicName,Message)
_os_SYSINLINE _os_sysser0(AppModeType,GetActiveApplicationMode)
_os_SYSINLINE _os_sysser0(COMApplicationModeType,GetCOMApplicationMode)

_os_SYSINLINE _os_sysserv1(void,StartOS,AppModeType,Mode)
_os_SYSINLINE _os_sysserv1(void,ShutdownOS,StatusType,Error)

_os_SYSINLINE _os_sysserv0(void,EnableAllInterrupts)
_os_SYSINLINE _os_sysserv0(void,DisableAllInterrupts)
_os_SYSINLINE _os_sysserv0(void,ResumeAllInterrupts)
_os_SYSINLINE _os_sysserv0(void,SuspendAllInterrupts)
_os_SYSINLINE _os_sysserv0(void,ResumeOSInterrupts)
_os_SYSINLINE _os_sysserv0(void,SuspendOSInterrupts)

_os_SYSINLINE _os_sysser0(StatusType,IsStackInRange)

#endif // USETRAPS


/*******************************************************************
|* 
|* MACRO:       DeclareTask
|*
|* DESCRIPTION: 
|*              It serves as an external declaration of a task.
|*              The function and use of this service are similar
|*              to that of the external declaration of variables.
 ******************************************************************/
#define DeclareTask(var)                extern _os_FASTALLOC const TaskType var

/*******************************************************************
|*
|* MACRO:       TASK  
|*
|* DESCRIPTION: 
|*              It declares the label (start address) of a given task.
 ******************************************************************/
#define TASK(name)      void _os_u_##name(void)

/*******************************************************************
|*
|* FUNCTION:    ActivateTask
|*
|* DESCRIPTION: The task <TaskID> is transferred from the suspended 
|*              state into the ready state. The operating system
|*              ensures that the task code is being executed from 
|*              the first statement. The service may be called from 
|*              interrupt level, from task level and the hook routine
|*              StartupHook.
|*              If called successfully, 'ActivateTask' does not return
|*              to the call level and the status can not be evaluated.
|*              In case of error the service returns to the calling task 
|*              and provides a status which can then be evaluated in the
|*              application.
|*              If the service ChainTask is called successfully, 
|*              this enforces a rescheduling.
|*              Ending a task function without call to TerminateTask or
|*              ChainTask is strictly forbidden and may leave the system 
|*              in an unpredictable status.
 ******************************************************************/
_os_SYSINLINE StatusType ActivateTask(TaskType TaskID); 


/*******************************************************************
|*
|* FUNCTION:    ChainTask
|*
|* DESCRIPTION: 
|*              This service causes the termination of the calling
|*              task. After termination of the calling task a succeeding
|*              task <TaskID> is activated. Using this service, 
|*              it ensures that the succeeding task starts to run 
|*              at the earliest after the calling task has been terminated.
|*              If the succeeding task is identical with the current task,
|*              this does not result in multiple requests and the task
|*              is not transferred to the suspended state.
|*              Resources occupied by the calling task must have been 
|*              released before ChainTask is called. If a resource is still
|*              occupied in standard status the behaviour is undefined.
|*              If called successfully, ChainTask does not return to the call
|*              level and the status can not be evaluated.
|*              In case of error the service returns to the calling task and
|*              provides a status which can then be evaluated in the
|*              application.
|*              If the service ChainTask is called successfully, this enforces
|*              a rescheduling.
|*              Ending a task function without call to TerminateTask or
|*               ChainTask is strictly forbidden and may leave the system in an
|*      
|*
|* RETURN:
|*              Standard:                               No return.
|*              Too many task activations of <TaskID>   E_OS_LIMIT
|*              Task <TaskID> is invalid                E_OS_ID
|*              Calling task still occupies resources   E_OS_RESOURCE
|*              Call at interrupt level                 E_OS_CALLEVEL
 ******************************************************************/
_os_SYSINLINE StatusType ChainTask(TaskType TaskID); 



/*******************************************************************
|*
|* FUNCTION:    Schedule        
|*
|* DESCRIPTION: 
|*              If a higher-priority task is ready, the internal 
|*              resource of the task is released, the current task
|*              is put into the ready state, its context is saved 
|*              and the higher-priority task is executed.
|*              Otherwise the calling task is continued.
|*              Rescheduling can only take place if an internal 
|*              resource is assigned to the calling task during
|*              system generation. For these tasks, Schedule enables
|*              a processor assignment to other tasks with lower priority
|*              than the ceiling priority of the internal resource 
|*              and higher priority than the priority of the calling task
|*              in application-specific locations. When returning from
|*              Schedule, the internal resource has been taken again.
|*              This service has no influence on tasks with no internal
|*              resource assigned (preemptable tasks).
 ******************************************************************/
_os_SYSINLINE StatusType Schedule(void);

/*******************************************************************
|*
|* FUNCTION:    TerminateTask   
|*
|* DESCRIPTION: 
|*              This service causes the termination of the calling 
|*              task. The calling task is transferred from the running 
|*              state into the suspended state.
|*              An internal resource assigned to the calling task is
|*              automatically released. Other resources occupied by the
|*              task must have been released before the call to
|*              TerminateTask. If a resource is still occupied in 
|*              standard status the behaviour is undefined.
|*              If the call was successful, TerminateTask does not return to the
|*              call level and the status can not be evaluated.
|*              If the version with extended status is used, the service returns
|*              in case of error, and provides a status which can be evaluated
|*              in the application.
|*              If the service TerminateTask is called successfully, it enforces a
|*              rescheduling.
|*              Ending a task function without call to TerminateTask or
|*              ChainTask is strictly forbidden and may leave the system in an
|*              undefined state.
|*
|* RETURN:
|*              Standard:                       No return
|*              Task still occupies resources   E_OS_RESOURCE
|*              Call at interrupt level         E_OS_CALLEVEL
 ******************************************************************/
 _os_SYSINLINE StatusType TerminateTask( void );

/*******************************************************************
|*
|* FUNCTION:    GetTaskState    
|*
|* DESCRIPTION: 
|*              
|*              Returns the state of a task (running, ready, waiting, suspended)
|*              at the time of calling GetTaskState.
|*              The service may be called from interrupt service routines, task
|*              level, and some hook routines.
|*              Within a full preemptive system, calling this operating system
|*              service only provides a meaningful result if the task runs in an
|*              interrupt disabling state at the time of calling.
|*              When a call is made from a task in a full preemptive system,
|*              the result may already be incorrect at the time of evaluation.
|*              When the service is called for a task, which is multiply
|*              activated, the state is set to running if any instance of the task
|*              is running.
|*
|* RETURN:
|*              Standard: No error              E_OK
|*              Task <TaskID> is invalid        E_OS_ID
 ******************************************************************/
_os_SYSINLINE StatusType GetTaskState(TaskType TaskID, TaskStateRefType State);

/*******************************************************************
|*
|* FUNCTION:    GetTaskID       
|*
|* DESCRIPTION: 
|*              'GetTaskID' returns the information about the TaskID of the task
|*              which is currently running.
|*              Allowed on task level, ISR level and in several hook routines.
|*              If <TaskID> cant be evaluated (no task currently running), the
|*              service returns INVALID_TASK as TaskType|*              
|* RETURN:
|*              E_OK
 ******************************************************************/
_os_SYSINLINE StatusType GetTaskID(TaskRefType TaskID); 

/*******************************************************************
|*
|* FUNCTION:    GetActiveApplicationMode        
|*
|* DESCRIPTION: 
|*              This service returns the current application mode. 
|*              It may be used to write mode dependent code.
 ******************************************************************/
_os_SYSINLINE AppModeType GetActiveApplicationMode(void);


/*******************************************************************
|*
|* FUNCTION:    StartOS
|*
|* DESCRIPTION: 
|*              The user can call this system service to start the operating
|*              system in a specific mode.
|*              At start up, the user code using no system service must determine the mode
|*              and pass it as a parameter to the API-service 'StartOS'.
|*              The application mode passed to StartOS allows the operating system to autostart 
|*              the correct subset of tasks and alarms. The assignment of autostart tasks and 
|*              alarms to application modes is made statically in the OIL file.
|*
|* RETURN:
 ******************************************************************/
 _os_SYSINLINE void StartOS(AppModeType Mode);


/*******************************************************************
|*
|* FUNCTION:    ShutdownOS
|*
|* DESCRIPTION: 
|*              The user can call this system service to abort the overall
|*              system (e.g. emergency off). The operating system also calls
|*              this function internally, if it has reached an undefined internal
|*              state and is no longer ready to run.
|*              If a ShutdownHook is configured the hook routine
|*              ShutdownHook is always called with 'Error' before shutting down the
|*              operating system.
|*              If ShutdownHook returns, further behaviour of ShutdownOS is
|*              implementation specific.
|*              If the operating system calls ShutdownOS it never uses E_OK
|*              as the passed parameter value.
|*
|* RETURN:
|*
 ******************************************************************/
_os_SYSINLINE void ShutdownOS(StatusType Error);


/*******************************************************************
|*
|* MACRO:        DeclareAlarm
|*
|* DESCRIPTION: 
|*               'DeclareAlarm' serves as an external declaration of an alarm. The
|*               function and use of this service are similar to that of the external
|*               declaration of variables.
|*
|* RETURN:
|*
 ******************************************************************/
#define DeclareAlarm(var)               extern _os_FASTALLOC const AlarmType var

/*************************************************************************************
|*
|* MACRO:       ALARMCALLBACK  
|*
|* DESCRIPTION: 
|*              It declares the label (start address) of an alarmcallback.
 *************************************************************************************/
#define ALARMCALLBACK(name)     void _os_u_##name(void)


/*************************************************************************************
|*
|* MACRO:       DeclareCounter  
|*
|* DESCRIPTION: 
|*              'DeclareCounter' serves as an external declaration of a counter . The
|*              function and use of this service are similar to that of the external
|*              declaration of variables.
|*
|* RETURN:
|*
 *************************************************************************************/
 #define DeclareCounter(var)            extern _os_FASTALLOC const CounterType var


/*******************************************************************************
|*
|* FUNCTION:     IncrementCounter       
|*
|* DESCRIPTION: 
|*               The application needs to increment the counter by 
|*               calling the 'IncrementCounter' system service.  
|*               The queue of alarms associated to this counter is
|*               is updated by the RTOS code.
|*      
|*
|* RETURN:
|*               No error, E_OK
|*               Counter <AlarmID> is invalid, E_OS_ID
|*
|* REMARKS:     
|*               'Incrementcounter(system_timer)' is not valid since
|*               the system timer is owned by the RTOS.
 *******************************************************************************/
_os_SYSINLINE StatusType IncrementCounter(CounterType CounterID);


/*******************************************************************************
|*
|* FUNCTION:     IsStackInRange     
|*
|* DESCRIPTION: 
|*               The application checks whether the stack of the running task
|*               is still in range, i.e. it has not overflown.          
|*
|* RETURN:
|*               No error,                        E_OK
|*               Stack of current task overflown, E_OS_SYS_UOFLW
|*               System stack overflown,          E_OS_SYS_SOFLW
|*               Target specific overflown,       E_OS_SYS_TOFLW
 *******************************************************************************/
_os_SYSINLINE StatusType IsStackInRange(void);

/********************************************************************************
|*
|* FUNCTION:     GetAlarmBase   
|*
|* DESCRIPTION: 
|*               The system service GetAlarmBase reads the alarm base
|*               characteristics. The return value <Info> is a structure in which
|*               the information of data type AlarmBaseType is stored.
|*
|* RETURN:
|*               No error, E_OK
|*               Alarm <AlarmID> is invalid, E_OS_ID
 ********************************************************************************/
_os_SYSINLINE StatusType GetAlarmBase(AlarmType AlarmID, AlarmBaseRefType Info);

/********************************************************************************
|*
|* FUNCTION:    GetAlarm        
|*
|* DESCRIPTION: 
|*              The system service GetAlarm returns the relative value in ticks
|*              before the alarm expires.
|*              It is up to the application to decide whether for example a
|*              CancelAlarm may still be useful.
|*
|* RETURN:
|*              No error, E_OK
|*              Alarm <AlarmID> is not used, E_OS_NOFUNC
|*              Alarm <AlarmID> is invalid, E_OS_ID
|*
 ********************************************************************************/
_os_SYSINLINE StatusType GetAlarm(AlarmType AlarmID, TickRefType Tick);

/*********************************************************************************
|*
|* FUNCTION:    SetRelAlarm             
|*
|* DESCRIPTION: 
|*              After <increment> ticks have elapsed, the task assigned to the
|*               alarm is activated or the assigned event is set or the 
|*               alarm-callback routine is called.
|*               If the relative value <alarmTime> is very small, the alarm may
|*               expire, and the task may become ready or the alarm-callback
|*               may be called before the system service returns to the user.
|*               If <cycle> is unequal zero, the alarm element is logged on
|*               again immediately after expiry with the relative value <cycle>.
|*               The alarm <id> must not already be in use.
|*               To change values of alarms already in use the alarm has to be
|*               cancelled first.
|*
|* RETURN:
|*              No error,                                               E_OK
|*              Alarm <id> is already in use,                           E_OS_STATE
|*              Alarm <id> is invalid,                                  E_OS_ID
|*              Value of <alarmTime> outside of the admissible limits 
|*              (lower  than zero or greater than maxallowedvalue),     E_OS_VALUE
|*              Value of <cycle> unequal to 0 and outside of the admissible
|*                      counter limits (less than mincycle or greater than
|*                      maxallowedvalue),                               E_OS_VALUE
|*
 *********************************************************************************/
_os_SYSINLINE StatusType SetRelAlarm(AlarmType AlarmID, TickType increment, TickType cycle);

/**********************************************************************************
|*
|* FUNCTION:    SetAbsAlarm
|*
|* DESCRIPTION: 
|*              When <start> ticks are reached, the task assigned to the alarm
|*              <id> is activated or the assigned event is set or the 
|*              alarm-callback routine is called.
|*              If the absolute value <start> is very close to the current counter
|*              value, the alarm may expire, and the task may become ready or
|*              the alarm-callback may be called before the system service
|*              returns to the user.
|*              If the absolute value <start> already was reached before the
|*              system call, the alarm will only expire when the absolute value
|*              <start> will be reached again, i.e. after the next overrun of the
|*              counter.
|*              If <cycle> is unequal zero, the alarm element is logged on
|*              again immediately after expiry with the relative value <cycle>.
|*              The alarm <id> must not already be in use.
|*              To change values of alarms already in use the alarm has to be
|*              cancelled first.
|*              If the alarm is already in use, this call will be ignored and the
|*              error E_OS_STATE is returned.
|*
|* RETURN:
|*              No error,                                                E_OK
|*              Alarm <id> is already in use,                            E_OS_STATE
|*              Alarm <id> is invalid,                                   E_OS_ID
|*              Value of <start> outside of the admissible counter limit (less
|*                      than zero or greater than maxallowedvalue),      E_OS_VALUE
|*              Value of <cycle> unequal to 0 and outside of the admissible
|*                      counter limits (less than mincycle or greater than
|*                      maxallowedvalue),                                E_OS_VALUE
|*
 **********************************************************************************/
_os_SYSINLINE StatusType SetAbsAlarm(AlarmType AlarmID, TickType start, TickType cycle);

/*******************************************************************
|*
|* FUNCTION:    CancelAlarm     
|*
|* DESCRIPTION: 
|*              The system service cancels the alarm <ID>.      
|*
|* RETURN:
|*              No error, E_OK
|*              Alarm <AlarmID> not in use, E_OS_NOFUNC
|*              Alarm <AlarmID> is invalid, E_OS_ID
 ******************************************************************/
_os_SYSINLINE StatusType CancelAlarm(AlarmType AlarmID);

/*******************************************************************
|*
|* MACRO:       ISR             
|*
|* DESCRIPTION: 
|*              It defines the label adress for an ISR user routine.
 ******************************************************************/
#define ISR(name)       void _os_u_##name(void)

/*******************************************************************
|*
|* FUNCTION:    EnableAllInterrupts             
|*
|* DESCRIPTION: Restores the state saved by 'DisableAllInterrupts'.     
 ******************************************************************/
_os_SYSINLINE void EnableAllInterrupts        ( void );

/*******************************************************************
|*
|* FUNCTION:    DisableAllInterrupts            
|*
|* DESCRIPTION: Disables all interrupts supported by the hardware
|*              saving the current status.
 ******************************************************************/
_os_SYSINLINE void DisableAllInterrupts       ( void );

/*******************************************************************
|*
|* FUNCTION:    ResumeAllInterrupts             
|*
|* DESCRIPTION: Restores the state saved by 'SuspendAllInterrupts'.
|*              Resume/Suspend calls can be nested.
 ******************************************************************/
_os_SYSINLINE void ResumeAllInterrupts        ( void );

/*******************************************************************
|*
|* FUNCTION:    SuspendAllInterrupts            
|*
|* DESCRIPTION: Disables all interrupts supported by the hardware
|*              saving the current status.
|*              Resume/Suspend calls can be nested.
 ******************************************************************/
_os_SYSINLINE void SuspendAllInterrupts       ( void );

/*******************************************************************
|*
|* FUNCTION:    ResumeOSInterrupts              
|*
|* DESCRIPTION: Like 'ResumeAllInterrupts' but only applicable to
|*              Category 2 interrupts.
 ******************************************************************/
_os_SYSINLINE void ResumeOSInterrupts         ( void );

/*******************************************************************
|*
|* FUNCTION:    SuspendOSInterrupts             
|*
|* DESCRIPTION: Like 'SuspendAllInterrupts' but only applicable to
|*              Category 2 interrupts.
 ******************************************************************/
_os_SYSINLINE void SuspendOSInterrupts        ( void );

/*******************************************************************
|*
|* MACRO:       DeclareEvent    
|*
|* DESCRIPTION: 
|*              DeclareEvent serves as an external declaration of an event.
|*              The function and use of this service are similar to that of the
|*              external declaration of variables.
 ******************************************************************/
#define DeclareEvent(var)               extern _os_FASTALLOC const EventMaskType var

/*******************************************************************
|*
|* FUNCTION:    SetEvent                
|*
|* DESCRIPTION: 
|*              The events of task <task_id> are set according to the event
|*              mask <mask>. Calling SetEvent causes the task <TaskID> to
|*              be transferred to the ready state, if it was waiting for at least
|*              one of the events specified in <Mask>.
|*
|* RETURN:
|*              No error,                                       E_OK
|*              Task <task_id> is invalid,                      E_OS_ID
|*              Events cannot be set as the referenced task is in the
|*                       suspended state,                       E_OS_STATE
 ******************************************************************/
_os_SYSINLINE StatusType SetEvent (TaskType TaskID, EventMaskType Mask);

/*******************************************************************
|*
|* FUNCTION:    GetEvent
|*
|* DESCRIPTION:
|*              This service returns the current state of all event bits
|*              of the task <task_id>, not the events that the task is waiting for.
|*              The current status of the event mask of task <task-id> is
|*              copied to <event>.
|*
|* RETURN:
|*              No error,                                               E_OK
|*              Task <TaskID> is invalid,                               E_OS_ID
|*              Referenced task <TaskID> is in the suspended state,     E_OS_STATE
|*
 ******************************************************************/
_os_SYSINLINE StatusType GetEvent ( TaskType TaskID, EventMaskRefType Event);

/*******************************************************************
|*
|* FUNCTION:    WaitEvent
|*
|* DESCRIPTION:
|*               The state of the calling task is set to waiting, unless at least one
|*              of the events specified in <mask> has already been set.
|*              This call enforces rescheduling, if the wait condition occurs. If
|*              rescheduling takes place, the internal resource of the task is
|*              released while the task is in the waiting state.
|*              This service must only be called from the task owning the event.
|*
|* RETURN:
|*              No error,                               E_OK
|*              Calling task occupies resources,        E_OS_RESOURCE
|*              Call at interrupt level,                E_OS_CALLEVEL
 ******************************************************************/
_os_SYSINLINE StatusType WaitEvent ( EventMaskType Mask );

/*******************************************************************
|*
|* FUNCTION:    ClearEvent              
|*
|* DESCRIPTION:
|*              The events of the ask calling ClearEvent are cleared
|*              according to the event mask <mask>.
|*              The system service ClearEvent is restricted to tasks
|*              which own the event.
|*
|* RETURN:
|*              No error,                               E_OK
|*              Call at interrupt level,                E_OS_CALLEVEL
|*
 ******************************************************************/
_os_SYSINLINE StatusType ClearEvent ( EventMaskType Mask );

/**********************************************************************************
|*
|* MACRO:       DeclareResource 
|*
|* DESCRIPTION:
|*               DeclareResource serves as an external declaration of a resource.
|*               The function and use of this service are similar to that
|*               of the external declaration of variables.
|*
 *********************************************************************************/
#define DeclareResource(var)            extern _os_FASTALLOC const ResourceType var


/*********************************************************************************
|*
|* FUNCTION:     GetResource
|*
|* DESCRIPTION:
|*               This call serves to enter critical sections in the code that are
|*               assigned to the resource referenced by <ResID>. A critical
|*               section must always be left using ReleaseResource.
|*               Nested resource occupation is only allowed if the inner critical
|*               sections are completely executed within the surrounding critical
|*               section (strictly stacked, see chapter 8.2, Restrictions when
|*               using resources). Nested occupation of one and the same
|*               resource is also forbidden!
|*               Corresponding calls to GetResource and ReleaseResource
|*               should appear within the same function on the same function
|*               level.
|*               Services which are points of rescheduling (TerminateTask, ChainTask, 
|*               Schedule and WaitEvent) must not be
|*               used in critical sections.
|*               Additionally, critical sections must be left before completion of
|*               an interrupt service routine.
|*               Generally speaking, critical sections should be short.
|*
|* RETURN:
|*              No error                                                        E_OK
|*              Resource <ResID> is invalid,                                    E_OS_ID
|*              Attempt to get resource which is already occupied by any task
|*                      or ISR, or the statically assigned priority of the 
|*                      calling task or interrupt routine is higher than the 
|*                      calculated ceiling priority,                            E_OS_ACCESS.
*
 *********************************************************************************/
_os_SYSINLINE StatusType GetResource ( ResourceType ResID );


/*********************************************************************************
|*
|* FUNCTION:    ReleaseResource         
|*
|* DESCRIPTION:
|*              ReleaseResource is the counterpart of GetResource and
|*              serves to leave critical sections in the code that are assigned to
|*              the resource referenced by <ResID>.
|*              The service may be called from an ISR and from task level.
|*
|* RETURN:
|*              No error,                                               E_OK
|*              Resource <ResID> is invalid,                            E_OS_ID
|*              Attempt to release a resource which is not occupied by 
|*                      any task or ISR, or another resource has to be 
|*                      released before,                                E_OS_NOFUNC
|*              Attempt to release a resource which has a lower ceiling
|*              priority than the statically assigned priority
|*              of the calling task or interrupt routine,               E_OS_ACCESS.
|*
 **********************************************************************************/
_os_SYSINLINE StatusType ReleaseResource ( ResourceType ResID );



/**********************************************************************************
|*
|* MACRO:       DeclareMessage
|*
|* DESCRIPTION:
|*               DeclareMessage serves as an external declaration of a message.
|*               The function and use of this service are similar to that
|*               of the external declaration of variables.
|*
 *********************************************************************************/
#define DeclareMessage(var)            extern _os_FASTALLOC const SymbolicName var


/**********************************************************************************
|*
|* MACRO:       DeclareAppMode
|*
|* DESCRIPTION:
|*               DeclareAppMode serves as an external declaration of an application
|*               mode.
|*               The function and use of this service are similar to that
|*               of the external declaration of variables.
|*
 *********************************************************************************/
#define DeclareAppMode(var)            extern _os_FASTALLOC const AppModeType var


/**********************************************************************************
|*
|* MACRO:       DeclareComAppMode
|*
|* DESCRIPTION:
|*               DeclareComAppMode serves as an external declaration of an application
|*               mode for communication.
|*               The function and use of this service are similar to that
|*               of the external declaration of variables.
|*
 *********************************************************************************/
#define DeclareComAppMode(var)            extern _os_FASTALLOC const COMApplicationModeType var


/***********************************************************************************
|*
|* FUNCTION:    StartCOM                
|*
|* DESCRIPTION:
|*              The service 'StartCOM' starts and initialises the OSEK COM
|*              implementation in the requested application mode <Mode>.
|*              If it fails, initialisation of the OSEK COM implementation
|*              aborts.
|*              'StartCOM' must be called from within a task if an OSEK-compliant
|*              operating system is used.
|*              Before returning, the service 'StartCOM' calls the application
|*              function StartCOMExtension. If the return value of 'StartCOMExtension'
|*              is different from E_OK 'StartCOM' returns also this value.
|*
|* REMARKS:
|*              The hardware and low-level resources used by OSEK COM 
|*              are initialised before 'StartCOM' is called.
|*
|*              'StartCOM' does not enable periodic transmission of messages. 
|*              If needed, 'StartPeriodic' can be called from 'StartCOMExtension'.
|*              
|*
|* RETURN:
|*              No error,                                               E_OK
|*              <Mode> is out of range                                  E_COM_ID
|*
 **********************************************************************************/
_os_SYSINLINE StatusType      StartCOM(COMApplicationModeType Mode);

/***********************************************************************************
|*
|* FUNCTION:    StopCOM         
|*
|* DESCRIPTION:
|*              The service 'StopCOM' causes all OSEK COM activity to cease
|*              immediately. All resources used by OSEK COM are returned
|*              or left in an inactive state.
|*              The shutdown occurs immediatly without waiting for pending operations
|*              to complete, therefore data may be lost.
|*
|*              When 'StopCOM' completes succesfully the system is left in a state
|*              in which 'StartCOM' can be called to re-initialise OSEK COM.
|*
|* REMARKS:
|*              
|*
|* RETURN:
|*              No error,                                               E_OK
|*              <Mode> is out of range                                  E_COM_ID
|*
 **********************************************************************************/
_os_SYSINLINE StatusType      StopCOM(COMShutdownModeType Mode);


/***********************************************************************************
|*
|* FUNCTION:    GetCOMApplicationMode   
|*
|* DESCRIPTION:
|*              This service returns the current COM Application Mode. It may be
|*              used to write mode-dependent application code.
|*
|* REMARKS:
|*              Since the mode is defined in 'StartCOM', undefined behaviour results
|*              if the 'GetCOMApplicationMode' is called before 'StartCOM'.
}*              
|*
|* RETURN:
|*              The COM application mode.
|*
 **********************************************************************************/
_os_SYSINLINE COMApplicationModeType  GetCOMApplicationMode(void);

/***********************************************************************************
|*
|* FUNCTION:    InitMessage     
|*
|* DESCRIPTION:
|*              'InitMessage' initialises the message object identified by
|*              'Message' with the application data referenced by <DataRef>.
|*              This function may be called in the 'StartCOMExtension' in order to
|*              change the default initialisation.
|*              For dynamic-length messages, the length of the message is initialised
|*              to its maximum.
|*              If 'InitMessage' initialises a transmission message object directly
|*              in the I_PDU, additionally byte ordering is performed and both the 
|*              CPU and the network-order Message Callouts are called.
|*
|* RETURN:
|*              No error,                                               E_OK
|*              <Message> is out of range, zero-length message,         E_COM_ID
|*                      or internally transmitted message.
|*
 **********************************************************************************/
_os_SYSINLINE StatusType      InitMessage(SymbolicName, ApplicationDataRef);


/***********************************************************************************
|*
|* FUNCTION:    SendMessage
|*
|* DESCRIPTION: 
|*              Updates the message object identified by <Message> with the application
|*              message referenced by the <DataRef> parameter.
|*              
|*              External Communication: If <Message> has the Triggered Transfer Property,
|*              the update is followed by immediate transmission of the I-PDU associated
|*              with the message except when the message is packed into an I-PDU with 
|*              Periodic Transmission Mode.
|*              If <Message> has the Pending Transfer Property, no transmission is caused 
|*              by the update.
|*              
|*              Internal Communication: The message <Message> is routed to the receiving
|*              part of the Interaction Layer. 
|*
|*              All flags are reseted.
|*
|* RETURN:
|*              No error,                                               E_OK
|*              If <Message> is out of range,                           E_COM_ID
|*              or it is a 'to be received' message,
|*              or dynamic-length
|*              or zero-length message.
 **********************************************************************************/
_os_SYSINLINE StatusType      SendMessage(SymbolicName Message, ApplicationDataRef DataRef);



/***********************************************************************************
|*
|* FUNCTION:    ReceiveMessage
|*
|* DESCRIPTION: 
|*              Updates the application message referenced by <DataRef> with the data
|*              in the message object identified by <Message>.
|*              It resets all flags.
|*              
|* RETURN:
|*              Data in the queued or unqueued message identified by    E_OK
|*              <Message> are available and returned to the application
|*              sucessfulty.            
|*
|*              The queued message is empty                             E_COM_NOMSG
|*
|*              An overflow of the message queue identified by          E_COM_LIMIT
|*              <Message> ocurred since the last call to 'ReceiveMessage'.
|*              At least one message has been discarded since the message queue
|*              filled, the service is nevertheless performed, and the overflow
|*              condition is cleared.
|*              
|*              If <Message> is out of range,                           E_COM_ID
}*              or it refers to 'to be sent' message,
|*              or dynamic-length,
|*              or zero-length message.
 **********************************************************************************/
_os_SYSINLINE StatusType      ReceiveMessage(SymbolicName Message, ApplicationDataRef DataRef);



/***********************************************************************************
|*
|* FUNCTION:    GetMessageStatus        
|*
|* DESCRIPTION:
|*              Returns the current status of the message object <Message>.
|*
|* RETURN:
|*              None of the below conditions                            E_OK
|*
|*              Message queue is empty                                  E_COM_NOMSG
|*
|*              Overflow of the message queue                           E_COM_LIMIT
|*
|*              <Message> is out of range                               E_COM_ID
|*              or not queued message.
|*
 **********************************************************************************/
_os_SYSINLINE StatusType      GetMessageStatus(SymbolicName Message);

/**********************************************************
 * MACRO:       COMCallout
 *
 * DESCRIPTION: It declares a Call Back routine
 *********************************************************/
#define COMCallout(name) CalloutReturnType _os_u_##name(void)



extern OSServiceIdType  _os_errid;
extern int32_t          _os_sys_par[3];
#define OSServiceID_GetResource                         (OSServiceIdType)_os_N_GetResource
#define OSServiceID_ReleaseResource                     (OSServiceIdType)_os_N_ReleaseResource
#define OSServiceID_GetTaskID                           (OSServiceIdType)_os_N_GetTaskID
#define OSServiceID_StartOS                             (OSServiceIdType)_os_N_StartOS
#define OSServiceID_ActivateTask                        (OSServiceIdType)_os_N_ActivateTask
#define OSServiceID_TerminateTask                       (OSServiceIdType)_os_N_TerminateTask
#define OSServiceID_ChainTask                           (OSServiceIdType)_os_N_ChainTask
#define OSServiceID_GetTaskState                        (OSServiceIdType)_os_N_GetTaskState
#define OSServiceID_Schedule                            (OSServiceIdType)_os_N_Schedule
#define OSServiceID_GetActiveApplicationMode            (OSServiceIdType)_os_N_GetActiveApplicationMode
#define OSServiceID_GetAlarmBase                        (OSServiceIdType)_os_N_GetAlarmBase
#define OSServiceID_GetAlarm                            (OSServiceIdType)_os_N_GetAlarm
#define OSServiceID_SetRelAlarm                         (OSServiceIdType)_os_N_SetRelAlarm
#define OSServiceID_SetAbsAlarm                         (OSServiceIdType)_os_N_SetAbsAlarm
#define OSServiceID_CancelAlarm                         (OSServiceIdType)_os_N_CancelAlarm
#define OSServiceID_SetEvent                            (OSServiceIdType)_os_N_SetEvent
#define OSServiceID_GetEvent                            (OSServiceIdType)_os_N_GetEvent
#define OSServiceID_WaitEvent                           (OSServiceIdType)_os_N_WaitEvent
#define OSServiceID_ClearEvent                          (OSServiceIdType)_os_N_ClearEvent
#define OSServiceID_ShutdownOS                          (OSServiceIdType)_os_N_ShutdownOS
#define OSServiceID_IncrementCounter                    (OSServiceIdType)_os_N_IncrementCounter
#define COMServiceID_StartCOM                           (COMServiceIdType)_os_N_StartCOM
#define COMServiceID_StopCOM                            (COMServiceIdType)_os_N_StopCOM
#define COMServiceID_GetCOMApplicationMode              (COMServiceIdType)_os_N_GetCOMApplicationMode
#define COMServiceID_InitMessage                        (COMServiceIdType)_os_N_InitMessage
#define COMServiceID_SendMessage                        (COMServiceIdType)_os_N_SendMessage
#define COMServiceID_ReceiveMessage                     (COMServiceIdType)_os_N_ReceiveMessage
#define COMServiceID_GetMessageStatus                   (COMServiceIdType)_os_N_GetMessageStatus
        
#define OSErrorGetServiceId()                           _os_errid
#define COMErrorGetServiceId()                          _os_errid
#define OSError_GetResource_ResID()                     _os_sys_par[0]
#define OSError_ReleaseResource_ResID()                 _os_sys_par[0]
#define OSError_StartOS_Mode()                          _os_sys_par[0]
#define OSError_ActivateTask_TaskID()                   _os_sys_par[0]
#define OSError_ChainTask_TaskID()                      _os_sys_par[0] 
#define OSError_GetTaskState_TaskID()                   _os_sys_par[0]  
#define OSError_GetTaskState_State()                    _os_sys_par[1]  
#define OSError_GetAlarmBase_AlarmID()                  _os_sys_par[0]
#define OSError_GetAlarmBase_Info()                     _os_sys_par[1]
#define OSError_SetRelAlarm_AlarmID()                   _os_sys_par[0]
#define OSError_SetRelAlarm_increment()                 _os_sys_par[1]
#define OSError_SetRelAlarm_cycle()                     _os_sys_par[2]
#define OSError_SetAbsAlarm_AlarmID()                   _os_sys_par[0]
#define OSError_SetAbsAlarm_start()                     _os_sys_par[1]
#define OSError_SetAbsAlarm_cycle()                     _os_sys_par[2]
#define OSError_CancelAlarm_AlarmID()                   _os_sys_par[0]
#define OSError_GetAlarm_AlarmID()                      _os_sys_par[0]
#define OSError_GetAlarm_Tick()                         _os_sys_par[0]
#define OSError_StartOS_Mode()                          _os_sys_par[0]
#define OSError_SetEvent_TaskID()                       _os_sys_par[0]
#define OSError_SetEvent_Mask()                         _os_sys_par[1]
#define OSError_GetEvent_TaskID()                       _os_sys_par[0]
#define OSError_GetEvent_Event()                        _os_sys_par[1]
#define OSError_WaitEvent_Mask()                        _os_sys_par[0]
#define OSError_ClearEvent_Mask()                       _os_sys_par[0]
#define OSError_IncrementCounter_CounterID()            _os_sys_par[0]
#define COMError_StartCOM_Mode()                        _os_sys_par[0]
#define COMError_StopCOM_Mode()                         _os_sys_par[0]
#define COMError_InitMessage_Message()                  _os_sys_par[0]
#define COMError_InitMessage_DataRef()                  _os_sys_par[1]
#define COMError_SendMessage_Message()                  _os_sys_par[0]
#define COMError_SendMessage_DataRef()                  _os_sys_par[1]
#define COMError_ReceiveMessage_Message()               _os_sys_par[0]
#define COMError_ReceiveMessage_DataRef()               _os_sys_par[1]
#define COMError_GetMessageStatus_Message()             _os_sys_par[0]



/***********************************************************************************
 * NAME:                ErrorHook
 *
 * DESCRIPTION:         The error hook routine (ErrorHook) is called if a 
 *                      system service returns a StatusType value not equal 
 *                      to E_OK. The hook routine 'ErrorHook' is not called 
 *                      if a system service is called from the ErrorHook 
 *                      itself (i.e., a recursive call of error hook never occurs).
 *                      Any possibly occurring error by calling system services 
 *                      from the ErrorHook can only be detected by evaluating 
 *                      the return value.
 **********************************************************************************/     
void ErrorHook (StatusType Error);
 

/**********************************************************************************
 * NAME:                COMErrorHook
 *
 * DESCRIPTION:         The COM error hook routine is called if a COM 
 *                      system service returns a StatusType value not equal 
 *                      to E_OK. The hook routine 'COMErrorHook' is not called 
 *                      if a system service is called from the COMErrorHook 
 *                      itself (i.e., a recursive call of error hook never occurs).
 *                      Any possibly occurring error by calling system services 
 *                      from the COMErrorHook can only be detected by evaluating 
 *                      the return value.
 **********************************************************************************/     
void COMErrorHook (StatusType Error);

/**********************************************************************************
 * NAME:                PreTaskHook
 *
 * DESCRIPTION:         'PreTaskHook' is called before entering the context of a new task
 *                      and may be used for debugging or time measurement
 *                      However, the task is still/already in the running state, and GetTaskId
 *                      will not return INVALID_TASK.
 **********************************************************************************/     
void PreTaskHook (void);

/**********************************************************************************
 * NAME:                PostTaskHook
 *
 * DESCRIPTION:         'PostTaskHook' is called after leaving the context of the old task
 *                      and may be used for debugging or time measurement
 **********************************************************************************/     
void PostTaskHook (void);
        
/**********************************************************************************
 * NAME:                StartupHook
 *
 * DESCRIPTION:         The corresponding hook routine (StartupHook) 
 *                      is called after the operating system start-up
 *                      and before the scheduler is running.
 *
 *                      After the operating system is initialised
 *                      (scheduler is not running), StartOS calls the hook routine 
 *                      'StartupHook', where the user can place the initialisation
 *                      code for all his operating system dependent initialisation.
 *                      In order to structure the initialisation code in 'StartupHook'
 *                      according to the started application mode, the service 
 *                      'GetActiveApplicationMode' is provided. After returning from 
 *                      that hook routine the operating systems enables the interrupts
 *                      and starts the scheduler.
 **********************************************************************************/     
void StartupHook(void); 

/**********************************************************************************
 * NAME:                ShutdownHook
 *
 * DESCRIPTION:         The corresponding hook routine (ShutdownHook) is called when
 *                      a system shutdown is requested by the application or by the
 *                      operating system in case of a severe error.
 *
 *                      When 'ShutdownOS' is called the operating system will call
 *                      the hook routine ShutdownHook and shut down afterwards.
 *                      The user is usually free to define any system behaviour
 *                      in 'ShutdownHook' e.g. not to return from the routine.
 **********************************************************************************/     
void ShutdownHook (StatusType Error);


#endif   /* _OSEK_H */







