/* @(#)irq.h    1.3 */

#ifndef IRQ_H
#define IRQ_H

typedef int             irqflags_t;

/*
 * Disable/enable interrupts in a nested manner with as little overhead
 * as possible. When there is a chance of getting preempted in between
 * then use the irqenter()/irqleave() combo instead.
 *
 * Note that these macros handle the generic case, simply restoring all
 * flags. For the m16c this includes the register bank selection bit and IPL.
 */
#ifdef __CM16C__
/*
 * __asm equivalent:
 * define irqsuspend(x) __asm("stc flg,%0\n\tfclr i" : "=r"(x) : :)
 * define irqrestore(x) __asm("ldc %0,flg" : : "r"(x) :)
 */
# define irqsuspend(x)  do { (x) = __stc_flg(); __fclr(__I); } while (0)
# define irqrestore(x)  __ldc_flg(x)
#endif
#ifndef irqsuspend
# define irqsuspend(x)  do { x = 0; SuspendAllInterrupts(); } while (0)
# define irqrestore(x)  ResumeAllInterrupts()
#endif

/*
 * Create an interrupt context, permitting to do things uninterrupted including
 * kernel calls capable of preempting us. Pre-emption is postponed inside an
 * interrupt context. Unfortunately the RES_SCHEDULER trick cannot be nested
 * yet.
 */
#define irqenter(x)     do { GetResource(RES_SCHEDULER); irqsuspend(x); } while (0)
#define irqleave(x)     do { irqrestore(x); ReleaseResource(RES_SCHEDULER); } while (0)

#endif
