/* @(#)ioreq.h  1.2 */

#ifndef IOREQ_H
#define IOREQ_H

#include <osek/osek.h>
#include <stdint.h>

typedef struct ioreq    ioreq_t;
typedef struct ioqueue  ioqueue_t;

/*
 * A bool_t is useful for documentation purposes -- should be moved to <osek/types.h>
 */
typedef enum
{
        false = 0,
        true  = 1
}                       bool_t;

/* 
 * GENERIC I/O REQUEST
 * ===================
 *
 * Upon completion, the driver will notify the task and activate the next
 * request from the queue. Errors are indicated by a nonzero "err" after
 * the request has completed. These are negative E* constants from <errno.h>.
 * A non-positive length request will simply be notified when its turn comes
 * (no data transfer). It is primarily useful to ensure that previously queued
 * requests are all flushed: queues are emptied by ISRs. The buffer pointer
 * itself is never modified.
 *
 * Notification can be suppressed by specifying INVALID_TASK as task.
 */
struct ioreq
{
        char *          buf;            /* I/O buffer                           */
        int             len;            /* input                                */
        int             pos;            /* output                               */
        int_least8_t    err;            /* 0: success, -errno on failure        */
        TaskType        task;           /* to be notified using "mask" (input)  */
        EventMaskType   mask;           /* the event mask (input)               */
        ioreq_t *       next;           /* device driver internal (queue link)  */
        ioqueue_t *     queue;          /* we belong to                         */
};

/*
 * GENERIC DEVICE I/O QUEUE
 * ========================
 */
struct ioqueue
{
        ioreq_t *       head;                   /* of request queue, NULL when empty */
        ioreq_t *       tail;                   /* of request queue */
};

extern  bool_t          ioreq_put(ioqueue_t *queue, ioreq_t *request);
extern  ioreq_t *       ioreq_get(ioqueue_t *queue);
extern  ioreq_t *       ioreq_abort(ioqueue_t *queue);
extern  void            ioreq_end(ioreq_t *request);

/*
 * GENERIC DEVICE OPERATIONS
 * =========================
 * 
 * A device driver should implement a generic interface for doing device
 * operations. The device handle must be opaque at the outside in order to
 * get device independent prototypes. Replace the "dev_" prefix by the
 * generic name for the device:
 * 
 * void dev_config(void)
 * 
 *      Configure all devices at system startup. This is done in one
 *      call because similar devices might share registers, interrupts
 *      and this way there is no need for a separate driver initialization
 *      call.
 * 
 * void dev_read(device_t dev, ioreq_t *request)
 * 
 *      Add an asynchronous read request to the corresponding device
 *      request queue. The request argument should not be modified by
 *      the caller until the caller is notified through the event mechanism.
 *      Requests are processed in FIFO order.
 * 
 * ioreq_t *dev_read_abort(device_t dev)
 * 
 *      Immediately finish the currently active read request, notifying
 *      the task as usual. The function returns the aborted request pointer,
 *      if there is any.
 * 
 * void dev_write(device_t dev, ioreq_t *request)
 * 
 *      Add an asynchronous write request to the corresponding device
 *      request queue. The request argument should not be modified by the
 *      caller until the caller is notified through the event mechanism.
 *      Requests are processed in FIFO order.
 * 
 * ioreq_t *dev_write_abort(device_t dev)
 * 
 *      Immediately finish the currently active write request, notifying
 *      the task as usual. The function returns the aborted request pointer,
 *      if there is any.
 *
 * There may be more generic operations such as open/close.
 */

#endif
