/**************************************************************************
**                                                                        *
**  FILE        :  strcmp.c                                               *
**                                                                        *
**  DESCRIPTION :  Source file for strcmp() routine                       *
**                 Compares two strings.                                  *
**                 Returns : < 0  -- s1<s2                                *
**                           = 0  -- s1=s2                                *
**                           > 0  -- s1>s2                                *
**                                                                        *
**  Copyright 1996-2009 Altium BV                                         *
**                                                                        *
**************************************************************************/

#include <string.h>
#include <ctype.h>

int
strcmp( const char * s1, const char * s2 )
{
        unsigned char c1, c2;

#if __TRADEOFF__ < 2    /* Optimize for speed */

        const unsigned long     *ulong_s1;
        const unsigned long     *ulong_s2;
        unsigned long           temp;

        if (!(((long)s1 & 0x3) | ((long)s2 & 0x3)))
        {
                /* Both strings are word-aligned
                 */
                ulong_s1 = (unsigned long *)s1;
                ulong_s2 = (unsigned long *)s2;
                temp = *ulong_s1;

                while (temp == *ulong_s2)
                {
                        if ((temp - 0x01010101) & ~temp & 0x80808080)
                        {
                                /* Words are equal and contain a zero-byte so
                                 * end of string and the strings must be equal
                                 */
                                return 0;
                        }
                        temp = *++ulong_s1;
                        ++ulong_s2;
                }
                
                s1 = (char *)ulong_s1;
                s2 = (char *)ulong_s2;
        }

        /* Words are unequal so compare bytes
         */
#endif
        
        do
        {
                c2 = *s2++;
                c1 = *s1++;
        } while ( c1 != '\0' && c1 == c2);
        return( (int)c1 - (int)c2 );
}

/* Case insensitive version: */
int
strcasecmp( const char * s1, const char * s2 )
{
        unsigned char c1, c2;

        do
        {
                c2 = toupper(*s2++);
                c1 = toupper(*s1++);
        } while ( c1 != '\0' && c1 == c2);

        return( (int)c1 - (int)c2 );
}

/* Case insensitive strncmp version: To be moved into strncmp.c */
int
strncasecmp( const char * s1, const char * s2, size_t n )
{
        unsigned char c1, c2;

        while ( n-- )
        {
                c2 = toupper(*s2++);
                c1 = toupper(*s1++);

                if ( c1 == '\0' || c1 != c2 )
                {
                        return( (int)c1 - (int)c2 );
                }
        }

        return( 0 );
}
