------------------------------------------------------------------------
--                                                                      
--                   Copyright (c) 2002 by Mentor Graphics              
--                                                                      
--  This source file is proprietary information of Mentor Graphics,Inc. 
--  It may be distributed in whole without restriction provided that    
--  this copyright statement is not removed from the file and that      
--  any derivative work contains this copyright notice.                 
--       

-- 
-- This file describes type-conversion functions between 
-- various standard types and the IEEE 1164 types std_logic and
-- std_logic_vector.
-- These type-conversion functions are needed to do post-synthesis
-- simulation using the -vhdl_wrapper option.
-- The type-conversions will be applied to the post-synthesis port
-- list to and from the original root entity port types.
--
-- Any type-conversions to and from port types that are not any of the
-- types specified her will have to be written by the user. They could
-- be added to this file if desired.
--

library ieee ;                  -- IEEE
use ieee.std_logic_1164.all ;
library exemplar ;              -- Exemplar
use exemplar.exemplar_1164.all ; 

package typetran is
    function to_type (a:std_logic) return bit ;
    function from_type (a:bit) return std_logic ;

    function to_type (a:std_logic_vector) return bit_vector ;
    function from_type (a:bit_vector)     return std_logic_vector ;

    function to_type (a:std_logic) return boolean ;
    function from_type (a:boolean) return std_logic ;

    function to_signed_int (a: std_logic_vector) return integer ;
    function to_unsigned_int (a: std_logic_vector) return integer ;
    function from_int (a:integer; size:integer)   return std_logic_vector ;

    function to_type (a:std_logic_vector) return std_ulogic_vector ;
    function from_type (a:std_ulogic_vector) return std_logic_vector ;
 
end typetran ;

package body typetran is
    function to_type (a:std_logic) return bit is
    begin
        return to_bit (a) ;
    end ;
 
    function from_type (a:bit) return std_logic is
    begin
        return to_stdulogic (a) ;
    end ;

    function to_type (a:std_logic) return boolean is
    begin
        if (a='1') then
            return TRUE ;
        else 
            return FALSE ;
        end if ;
    end ;
 
    function from_type (a:boolean) return std_logic is
    begin
        if (a=TRUE) then
            return '1' ;
        else 
            return '0' ;
        end if ;
    end ;

    function to_type (a:std_logic_vector) return bit_vector is 
       variable result : bit_vector (a'range) ;
    begin 
       for i in a'range loop
           result(i) := to_type(a(i)) ;
       end loop ;
       return result ;
    end ;

    function from_type (a:bit_vector) return std_logic_vector is
       variable result : std_logic_vector (a'range) ;
    begin
       for i in a'range loop
           result(i) := from_type(a(i)) ;
       end loop ;
       return result ;
    end ; 
     
    function to_signed_int (a:std_logic_vector) return integer is 
       variable temp : integer ;
    begin
        -- Interpret the left most bit as the sign bit 
       if (a(a'left)='1') then
           return - evec2int (comp2(a)) ;
       else 
           return evec2int (a) ;
       end if ;   
    end ;

    function to_unsigned_int (a:std_logic_vector) return integer is 
    begin
       return evec2int (a) ;
    end ;

    function from_int (a:integer; size:integer) 
                                    return std_logic_vector is
    begin
       return int2evec (a,size) ;
    end ;

    function to_type (a:std_logic_vector) return std_ulogic_vector is
    begin
         -- Use type conversion here 
        return std_ulogic_vector (a) ;
    end ;
 
    function from_type (a:std_ulogic_vector) return std_logic_vector is
    begin
         -- Use direct type conversion here 
        return std_logic_vector(a) ;
    end ;
 
end typetran ;
