--------------------------------------------------------------------
--
--
-- PACKAGE : VHDL ASSERTIONS LIBRARY
-- FILE    : ASSERTIONS.VHDL
-- REVISION: 1.00 (20.01.2006)
-- 
-- (c) 2006 Aldec, Inc.
--
--
--------------------------------------------------------------------


library IEEE;
use IEEE.std_logic_1164.all;

--------------------------------------------------------------------
--                     PACKAGE OVA_LOGIC                          --
--------------------------------------------------------------------
package ova_logic is

    ----------------------------------------------------------------
    ----------------------------------------------------------------

    TYPE ova_logic_t IS ( '0', '1', 'z', 'x' );

    ----------------------------------------------------------------

    TYPE ova_vector_t IS ARRAY ( NATURAL RANGE <> ) OF ova_logic_t;

    ----------------------------------------------------------------

    TYPE ova_memory_t IS ARRAY ( NATURAL RANGE <> , NATURAL RANGE <> ) OF ova_logic_t;

    ----------------------------------------------------------------
    ----------------------------------------------------------------

    FUNCTION "/=" ( l,r : ova_logic_t ) RETURN BOOLEAN; 
    
    ----------------------------------------------------------------

    FUNCTION "=" ( l,r : ova_logic_t ) RETURN BOOLEAN; 
    
    ----------------------------------------------------------------

    FUNCTION To_ova_logic_t( s : boolean ) RETURN ova_logic_t;	

    ----------------------------------------------------------------

    FUNCTION To_ova_logic_t( s : BIT ) RETURN ova_logic_t;	

    ----------------------------------------------------------------

    FUNCTION To_ova_logic_t( s : std_logic ) RETURN ova_logic_t;	

    ----------------------------------------------------------------

    FUNCTION To_ova_logic_t ( s : INTEGER ) RETURN ova_logic_t;
 
    ----------------------------------------------------------------

    FUNCTION To_ova_vector_t( s : BIT_VECTOR ) RETURN ova_vector_t;	 	

    ----------------------------------------------------------------

    FUNCTION To_ova_vector_t ( s : std_logic_vector ) RETURN ova_vector_t;	 	

    ----------------------------------------------------------------

    FUNCTION To_ova_vector_t ( s : INTEGER ) RETURN ova_vector_t;

    ----------------------------------------------------------------

    FUNCTION to_ova_vector_t( s : std_logic; width : INTEGER ) RETURN ova_vector_t;	
	
    ----------------------------------------------------------------

    FUNCTION to_ova_vector_t( s : BIT; width : INTEGER ) RETURN ova_vector_t;
    
    ----------------------------------------------------------------

    FUNCTION to_ova_vector_t( s : INTEGER; width : INTEGER ) RETURN ova_vector_t;

    ----------------------------------------------------------------
    ----------------------------------------------------------------
	
    FUNCTION to_bit_vector32( s : INTEGER ) RETURN bit_vector;
	
    ----------------------------------------------------------------

    FUNCTION to_bit_vector64( s : INTEGER ) RETURN bit_vector;
	
    ----------------------------------------------------------------

    FUNCTION to_bit_vector64( s : REAL ) RETURN bit_vector;
	
    ----------------------------------------------------------------
    ----------------------------------------------------------------
	
	
end package;
--------------------------------------------------------------------
--                     PACKAGE OVA_LOGIC END                      --
--------------------------------------------------------------------


--------------------------------------------------------------------
--                   PACKAGE BODY OVA_LOGIC                       --
--------------------------------------------------------------------
package body ova_logic is
--------------------------------------------------------------------
--------------------------------------------------------------------

    FUNCTION "/=" ( l,r : ova_logic_t ) RETURN BOOLEAN IS
    BEGIN		
        CASE l IS
	    WHEN 'x' => CASE r IS
			    WHEN 'x'    => RETURN false;
			    WHEN OTHERS => RETURN true;
		        END CASE;
	    WHEN 'z' => CASE r IS
			    WHEN 'z'    => RETURN false;
			    WHEN OTHERS => RETURN true;
		        END CASE;
	    WHEN '0' => CASE r IS
			    WHEN '0'    => RETURN false;
			    WHEN OTHERS => RETURN true;
		        END CASE;
	    WHEN '1' => CASE r IS
			    WHEN '1'    => RETURN false;
			    WHEN OTHERS => RETURN true;
		        END CASE;
        END CASE;
    END FUNCTION;

    ----------------------------------------------------------------

    FUNCTION "=" ( l,r : ova_logic_t ) RETURN BOOLEAN IS
    BEGIN		
        CASE l IS
	    WHEN 'x' => CASE r IS
			    WHEN 'x'    => RETURN true;
			    WHEN OTHERS => RETURN false;
		        END CASE;
	    WHEN 'z' => CASE r IS
			    WHEN 'z'    => RETURN true;
			    WHEN OTHERS => RETURN false;
		        END CASE;
	    WHEN '0' => CASE r IS
			    WHEN '0'    => RETURN true;
			    WHEN OTHERS => RETURN false;
		        END CASE;
	    WHEN '1' => CASE r IS
			    WHEN '1'    => RETURN true;
			    WHEN OTHERS => RETURN false;
		        END CASE;
        END CASE;
    END FUNCTION;

    ----------------------------------------------------------------

    FUNCTION To_ova_logic_t ( s : BIT ) RETURN ova_logic_t IS
    BEGIN
        CASE s IS
            WHEN '0' => RETURN ('0');
            WHEN '1' => RETURN ('1');
        END CASE;
    END;			

    ----------------------------------------------------------------

    FUNCTION To_ova_logic_t ( s : boolean ) RETURN ova_logic_t IS
    BEGIN
        IF( s = false ) THEN
            RETURN '0';
        ELSE 
            RETURN '1';
        END IF;
    END;			

    ----------------------------------------------------------------

    FUNCTION To_ova_vector_t( s : BIT_VECTOR ) RETURN ova_vector_t IS 
        ALIAS sv : BIT_VECTOR ( s'LENGTH-1 DOWNTO 0 ) IS s;
        VARIABLE result : ova_vector_t ( s'LENGTH-1 DOWNTO 0 );
    BEGIN	
        for i in result'RANGE loop
            CASE sv(i) IS
                WHEN '0' => result(i):='0';
                WHEN '1' => result(i):='1';
            END CASE;
        end loop;
        return result;
    END;

    -----------------------------------------------------------------

    FUNCTION To_ova_logic_t( s : std_logic ) RETURN ova_logic_t IS
    BEGIN
        CASE s IS
            WHEN 'U' => RETURN ('x');
            WHEN 'X' => RETURN ('x'); 
            WHEN '0' => RETURN ('0');
            WHEN '1' => RETURN ('1');
            WHEN 'Z' => RETURN ('z');
            WHEN 'W' => RETURN ('z');
            WHEN 'L' => RETURN ('0');
            WHEN 'H' => RETURN ('1');
            WHEN '-' => RETURN ('x');                
        END CASE;
    END;

    ----------------------------------------------------------------

    FUNCTION To_ova_vector_t( s : std_logic_vector ) RETURN ova_vector_t IS 
        ALIAS sv : std_logic_vector ( s'LENGTH-1 DOWNTO 0 ) IS s;
        VARIABLE result : ova_vector_t ( s'LENGTH-1 DOWNTO 0 );
    BEGIN	
        FOR i IN result'RANGE LOOP
            CASE sv(i) IS
                WHEN '0'|'L' => result(i):='0';
                WHEN '1'|'H' => result(i):='1';	
                WHEN 'U'|'X'|'-' => result(i):='x';
                WHEN 'Z'|'W' => result(i):='z';
            END CASE;
        END LOOP;
        RETURN result;
    END;

    -----------------------------------------------------------------

    FUNCTION To_ova_logic_t( s : INTEGER ) RETURN ova_logic_t IS
    BEGIN
        IF( s = 0 )	THEN
            RETURN '0';
        ELSE
            RETURN '1';
        END IF;
    END;

    ----------------------------------------------------------------

    FUNCTION To_ova_vector_t( s: INTEGER ) RETURN ova_vector_t IS
        VARIABLE result: ova_vector_t (31 DOWNTO 0);
        VARIABLE temp: INTEGER;
    BEGIN
        temp := s;
        FOR i IN 0 TO 31 LOOP
            IF( temp MOD 2 ) = 1 THEN
                result(i) := '1';
            ELSE 
                result(i) := '0';
            END IF;
            
            IF temp > 0 THEN
                temp := temp / 2;
            ELSE
                temp := ( temp - 1 ) / 2; 
            END IF;
        END LOOP;
        RETURN result;
    END;
	
    ----------------------------------------------------------------

    FUNCTION to_ova_vector_t( s : std_logic; width : INTEGER ) RETURN ova_vector_t IS
        VARIABLE result: ova_vector_t( width - 1 DOWNTO 0 );
    BEGIN
        result(0) := to_ova_logic_t( s );
        FOR i IN 1 TO width - 1 LOOP
            result(i) := '0';
        END LOOP;
        RETURN result;
    END;

    ----------------------------------------------------------------

    FUNCTION to_ova_vector_t( s : BIT; width : INTEGER ) RETURN ova_vector_t IS
        VARIABLE result: ova_vector_t( width - 1 DOWNTO 0 );
    BEGIN
        result(0) := to_ova_logic_t( s );
        FOR i IN 1 TO width - 1 LOOP
            result(i) := '0';
        END LOOP;
        RETURN result;
    END;

    ----------------------------------------------------------------

    FUNCTION To_ova_vector_t( s: INTEGER; width : INTEGER ) RETURN ova_vector_t IS
        VARIABLE result: ova_vector_t (width - 1 DOWNTO 0);
        VARIABLE temp: INTEGER;
    BEGIN
        temp := s;
        FOR i IN 0 TO width - 1 LOOP
            IF( temp MOD 2 ) = 1 THEN
                result(i) := '1';
            ELSE 
                result(i) := '0';
            END IF;
            
            IF temp > 0 THEN
                temp := temp / 2;
            ELSE
                temp := ( temp - 1 ) / 2; 
            END IF;
        END LOOP;
        RETURN result;
    END;

    ----------------------------------------------------------------
    ----------------------------------------------------------------
	
    FUNCTION to_bit_vector32( s : INTEGER ) RETURN bit_vector IS
        VARIABLE result: bit_vector( 31 DOWNTO 0 );
        VARIABLE temp: INTEGER;
    BEGIN
    	temp := s;
    	FOR i IN 0 TO 31 LOOP
    	    IF( temp MOD 2 ) = 1 THEN
    	        result(i) := '1';
    	    ELSE
    	        result(i) := '0';
    	    END IF;
    	    
    	    IF temp > 0 THEN
    	        temp := temp / 2;
    	    ELSE
    	        temp := ( temp - 1 ) / 2;
    	    END IF;
    	END LOOP;
        RETURN result;
    END;
	
    ----------------------------------------------------------------

    FUNCTION to_bit_vector64( s : INTEGER ) RETURN bit_vector IS
        VARIABLE result: bit_vector( 63 DOWNTO 0 );
        VARIABLE temp: INTEGER;
    BEGIN
    	temp := s;
    	FOR i IN 0 TO 31 LOOP
    	    IF( temp MOD 2 ) = 1 THEN
    	        result(i) := '1';
    	    ELSE
    	        result(i) := '0';
    	    END IF;
    	    
    	    IF temp > 0 THEN
    	        temp := temp / 2;
    	    ELSE
    	        temp := ( temp - 1 ) / 2;
    	    END IF;
    	END LOOP;
    	
    	FOR j IN 32 TO 63 LOOP 
    	    result(j) := '0';
    	END LOOP;
        RETURN result;
    END;

    ----------------------------------------------------------------

    FUNCTION to_bit_vector64( s : REAL ) RETURN bit_vector IS
        VARIABLE result: bit_vector( 63 DOWNTO 0 );
    BEGIN
    	REPORT "Error AVHDL1000: Conversion from REAL to BIT_VECTOR(63 DOWNTO 0) not implemented yet!";
		IF s /= 0.0 THEN
			RETURN "0000000000000000000000000000000000000000000000000000000000000001";
		ELSE
			RETURN "0000000000000000000000000000000000000000000000000000000000000000"; 		
		END IF;
    END;
    
    ----------------------------------------------------------------
    ----------------------------------------------------------------
	
end package body;	
--------------------------------------------------------------------
--                 PACKAGE BODY OVA_LOGIC END                     --
--------------------------------------------------------------------


--------------------------------------------------------------------
--                     PACKAGE ASSERT_API                         --
--------------------------------------------------------------------
library IEEE;
use IEEE.std_logic_1164.all;

package assert_api is

    ----------------------------------------------------------------
    ----------------------------------------------------------------

    type aldec_internal_pointer is access Integer;

    ----------------------------------------------------------------

    subtype aldec_internal_pointer32 is Integer;

    ----------------------------------------------------------------

    type aldec_internal_integer is range -2147483647 to 2147483647;

    ----------------------------------------------------------------
	
    type aldec_internal_pointer_vector is array ( natural range <> ) OF aldec_internal_pointer;

    ----------------------------------------------------------------

    type aldec_internal_pointer32_vector is array( natural range <> ) of aldec_internal_pointer32;	

    ----------------------------------------------------------------
	
    subtype aldec_internal_void_pointer is aldec_internal_pointer;
	
    ----------------------------------------------------------------

    subtype aldec_internal_time is time;

    ----------------------------------------------------------------
	
    subtype aldec_internal_std_logic is std_logic;
	
    ----------------------------------------------------------------

    IMPURE FUNCTION akernel_func( action : string; matched_var_val : aldec_internal_integer ) RETURN aldec_internal_pointer;
    
    ----------------------------------------------------------------
    
    IMPURE FUNCTION akernel_func( action : string; matched_var_val : std_logic ) RETURN aldec_internal_pointer;
        
    ----------------------------------------------------------------

    IMPURE FUNCTION akernel_func( action : string; start: aldec_internal_time; flags : aldec_internal_integer ) RETURN aldec_internal_pointer;

    ------------------------------------------------------------

    IMPURE FUNCTION akernel_func( action : string; start: aldec_internal_time; assert_name, file_name: string; line, value: integer ) RETURN aldec_internal_pointer;

    ------------------------------------------------------------

    IMPURE FUNCTION akernel_func( action : string; start: aldec_internal_time; assert_name: integer; file_name : string; line, value : integer ) RETURN aldec_internal_pointer;

    ----------------------------------------------------------------
	
    IMPURE FUNCTION akernel_func( action : string; process_handle : aldec_internal_pointer ; value1 : aldec_internal_pointer; value2: std_logic_vector ) RETURN aldec_internal_pointer;

    ----------------------------------------------------------------
    
    IMPURE FUNCTION akernel_func( action : string; process_handle : aldec_internal_pointer ; value2: std_logic_vector ) RETURN aldec_internal_pointer;

    ----------------------------------------------------------------

    FUNCTION akernel_func_itime( action: string; process_handle: aldec_internal_pointer; value: integer ) RETURN aldec_internal_time;

    ----------------------------------------------------------------

    IMPURE FUNCTION akernel_proc_addr( object: aldec_internal_void_pointer ) RETURN aldec_internal_pointer;

    ----------------------------------------------------------------

    FUNCTION akernel_proc_addr32( object: aldec_internal_void_pointer ) RETURN aldec_internal_pointer32;
    
    ----------------------------------------------------------------

    IMPURE FUNCTION akernel_var_addr( object: aldec_internal_void_pointer ) RETURN aldec_internal_pointer;

    ----------------------------------------------------------------

    FUNCTION akernel_var_addr32( object: aldec_internal_void_pointer ) RETURN aldec_internal_pointer32;

    ----------------------------------------------------------------

    PROCEDURE akernel_proc (action : string; start,tick: aldec_internal_time );		
	
    ----------------------------------------------------------------
    
    PROCEDURE akernel_return_value( v: aldec_internal_void_pointer );
	
    ----------------------------------------------------------------
    ----------------------------------------------------------------

end package;
--------------------------------------------------------------------
--                   PACKAGE ASSERT_API END                       --
--------------------------------------------------------------------
