--------------------------------------------------------------------------
--                                                                      --
-- Copyright (c) 1997 - 2009 Aldec, Inc.  All rights reserved.          --
--                                                                      --
-- This source file may be used and distributed without restriction     --
-- provided that this copyright statement is not removed from the file  --
-- and that any derivative work contains this copyright notice.         --
--                                                                      --
--    Package name: RANDOM_PKG                                          --
--                                                                      --
--    Purpose:                                                          --
--     A set of procedures used for random stimulators.					--
--                                                                      --
--                                                                      --
--------------------------------------------------------------------------

library ieee;
use ieee.math_real.all;

package random_pkg is
	procedure uniform_p(
						variable vseed		:		inout integer;
						constant vlow		:		in integer;
						constant vhigh		:		in integer;
						variable result		:		out real
						);
	
	procedure uniform_p(
						variable vseed		:		inout integer;
						constant vlow		:		in real;
						constant vhigh		:		in real;
						variable result		:		out real
						);
	
	procedure normal_p(
						variable vseed		:		inout integer;
						constant mean		:		in integer;
						constant deviation	:		in integer;
						variable result		:		out real
					   );
	
	procedure normal_p(
						variable vseed		:		inout integer;
						constant mean		:		in real;
						constant deviation	:		in real;
						variable result		:		out real
					   );
	
	procedure exponential_p(
						variable vseed		:		inout integer;
						constant mean		:		in integer;
						variable result		:		out real
						);
	
	procedure exponential_p(
						variable vseed		:		inout integer;
						constant mean		:		in real;
						variable result		:		out real
						);
	
	procedure poisson_p(
						variable vseed		:		inout integer;
						constant mean		:		in integer;
						variable result		:		out integer
						);
	
	procedure poisson_p(
						variable vseed		:		inout integer;
						constant mean		:		in real;
						variable result		:		out integer
						);
	
	procedure chi_square_p(
						variable vseed		:		inout integer;
						constant deg_of_free:		in integer;
						variable result		:		out real
						);
	
	procedure t_p(
						variable vseed		:		inout integer;
						constant deg_of_free:		in integer;
						variable result		:		out real
						);
	
	procedure erlangian_p(
						variable vseed		:		inout integer;
						constant k			:		in integer;
						constant mean		:		in integer;
						variable result		:		out real
						);
	
	procedure erlangian_p(
						variable vseed		:		inout integer;
						constant k			:		in integer;
						constant mean		:		in real;
						variable result		:		out real
						);
	
end package random_pkg;

package body random_pkg is
	procedure uniform_p(
						variable vseed		:		inout integer;
						constant vlow		:		in integer;
						constant vhigh		:		in integer;
						variable result		:		out real
						) is
							
		constant d: real := 0.00000011920928955078125;
		variable a, b, c: real;
		variable temp: integer;
	begin 
		if (vseed = 0) then 
			vseed := 259341593;
		end if;
		if (vlow > vhigh) then
			a := 0.0;
			b := 2147483647.0;
		else
			a := real(vlow);
			b := real(vhigh);
		end if;
		vseed := 69069 * vseed + 1;
		
		-- this group of statements relies on IEEE floating point format
		-- integer value is interpreted as mantissa of floting point value
		temp := vseed;
		temp :=	temp/(2**9); -- shift right 9 times
		if (temp < 0) then
			c := 1.5;
			temp := -temp;
		else
			c := 1.0;
		end if;
		for i in 23 downto 1 loop
			if (temp rem 2 = 1) then
				c := c + (1.0 / (2**i));
			end if;
			temp := temp/2;
			exit when temp = 0;
		end loop;
		c := c + (c*d);
		c := ((b - a) * (c - 1.0)) + a;
		result := c;
	end procedure uniform_p;

	procedure uniform_p(
					variable vseed		:		inout integer;
					constant vlow		:		in real;
					constant vhigh		:		in real;
					variable result		:		out real
					) is
						
		constant d: real := 0.00000011920928955078125;
		variable a, b, c: real;
		variable temp: integer;
	begin 
		if (vseed = 0) then 
			vseed := 259341593;
		end if;
		if (vlow > vhigh) then
			a := 0.0;
			b := 2147483647.0;
		else
			a := vlow;
			b := vhigh;
		end if;
		vseed := 69069 * vseed + 1;
		
		-- this group of statements relies on IEEE floating point format
		-- integer value is interpreted as mantissa of floating point value
		temp := vseed;
		temp :=	temp/(2**9); -- shift right 9 times
		if (temp < 0) then
			c := 1.5;
			temp := -temp;
		else
			c := 1.0;
		end if;
		for i in 23 downto 1 loop
			if (temp rem 2 = 1) then
				c := c + (1.0 / (2**i));
			end if;
			temp := temp/2;
			exit when temp = 0;
		end loop;
		c := c + (c*d);
		c := ((b - a) * (c - 1.0)) + a;
		result := c;
	end procedure uniform_p;
	
	procedure normal_p(
						variable vseed		:		inout integer;
						constant mean		:		in integer;
						constant deviation	:		in integer;
						variable result		:		out real
				       ) is
		
		variable v1, v2, s	:	real;
	begin
		s := 1.0;
		while ((s >= 1.0) or (s = 0.0)) loop
			uniform_p(vseed, -1, 1, v1);
			uniform_p(vseed, -1, 1, v2);
			s := v1*v1 + v2*v2;
		end loop;
		s := v1 * SQRT(-2.0 * LOG(s) / s);
		v1 := real(deviation);
		v2 := real(mean);
		result := s * v1 + v2;
	end procedure normal_p;
	
	procedure normal_p(
						variable vseed		:		inout integer;
						constant mean		:		in real;
						constant deviation	:		in real;
						variable result		:		out real
				       ) is
		
		variable v1, v2, s	:	real;
	begin
		s := 1.0;
		while ((s >= 1.0) or (s = 0.0)) loop
			uniform_p(vseed, -1, 1, v1);
			uniform_p(vseed, -1, 1, v2);
			s := v1*v1 + v2*v2;
		end loop;
		s := v1 * SQRT(-2.0 * LOG(s) / s);
		v1 := deviation;
		v2 := mean;
		result := s * v1 + v2;
	end procedure normal_p;
	
	procedure exponential_p(
					variable vseed		:		inout integer;
					constant mean		:		in integer;
					variable result		:		out real
					) is
	
		variable n: real;
	begin
		uniform_p(vseed, 0, 1, n);
		if (n /= 0.0) then
			n := -LOG(n) * real(mean);
		end if;
		result := n;
	end procedure exponential_p;
	
	procedure exponential_p(
					variable vseed		:		inout integer;
					constant mean		:		in real;
					variable result		:		out real
					) is
	
		variable n: real;
	begin
		uniform_p(vseed, 0, 1, n);
		if (n /= 0.0) then
			n := -LOG(n) * mean;
		end if;
		result := n;
	end procedure exponential_p;
	
	procedure poisson_p(
					variable vseed		:		inout integer;
					constant mean		:		in integer;
					variable result		:		out integer
					) is
	
		variable n: integer;
		variable p, q, temp: real;
	begin
		n := 0;
		q := -real(mean);
		p := EXP(q);
		uniform_p(vseed, 0, 1, q);
		while (p < q) loop
			n := n + 1;
			temp := q;
			uniform_p(vseed, 0, 1, q);
			q := q * temp;
		end loop;
		result := n;
	end procedure poisson_p;
	
	procedure poisson_p(
					variable vseed		:		inout integer;
					constant mean		:		in real;
					variable result		:		out integer
					) is
	
		variable n: integer;
		variable p, q, temp: real;
	begin
		n := 0;
		q := -mean;
		p := EXP(q);
		uniform_p(vseed, 0, 1, q);
		while (p < q) loop
			n := n + 1;
			temp := q;
			uniform_p(vseed, 0, 1, q);
			q := q * temp;
		end loop;
		result := n;
	end procedure poisson_p;
	
	procedure chi_square_p(
					variable vseed		:		inout integer;
					constant deg_of_free:		in integer;
					variable result		:		out real
					) is
		variable x, temp: real;
		variable k: integer;
	begin
		if ((deg_of_free mod 2) /= 0) then
			normal_p(vseed, 0, 1, x);
			x := x * x;
		else
			x := 0.0;
		end if;
		k := 2;
		while (k <= deg_of_free) loop
			exponential_p(vseed, 1, temp);
			x := x + 2.0*temp;
			k := k + 2;
		end loop;
		result := x;
	end procedure chi_square_p;
	
	procedure t_p(
					variable vseed		:		inout integer;
					constant deg_of_free:		in integer;
					variable result		:		out real
					) is

		variable x, chi2, div, root: real;
	begin
		chi_square_p(vseed, deg_of_free, chi2);
		div := chi2 / real(deg_of_free);
		root := SQRT(div);
		normal_p(vseed, 0, 1, x);
		x := x / root;
		result := x;
	end procedure t_p;				
	
	procedure erlangian_p(
					variable vseed		:		inout integer;
					constant k			:		in integer;
					constant mean		:		in integer;
					variable result		:		out real
					) is
	
		variable a, b, x, temp: real;
	begin
		x := 1.0;
		for i in 1 to k loop
			uniform_p(vseed, 0, 1, temp);
			x := x * temp;
		end loop;
		a := real(mean);
		b := real(k);
		x := -a * LOG(x) / b;
		result := x;
	end procedure erlangian_p;
			
	procedure erlangian_p(
					variable vseed		:		inout integer;
					constant k			:		in integer;
					constant mean		:		in real;
					variable result		:		out real
					) is
	
		variable a, b, x, temp: real;
	begin
		x := 1.0;
		for i in 1 to k loop
			uniform_p(vseed, 0, 1, temp);
			x := x * temp;
		end loop;
		a := mean;
		b := real(k);
		x := -a * LOG(x) / b;
		result := x;
	end procedure erlangian_p;
			
end random_pkg;
