/*****************************************************************************
 *
 * C Interface for VHDL
 *
 *****************************************************************************/

#ifndef ALD_H
#define ALD_H

#pragma pack(8)

//#ifndef EXTERN
#define EXTERN /*Empty*/
//#define DEFINED_EXTERN
//#endif

#if defined(__STDC__) || defined(__cplusplus)

#ifndef PROTO_PARAMS
#define PROTO_PARAMS(params) params
#define DEFINED_PROTO_PARAMS
#endif
/*#ifndef EXTERN
#define EXTERN
#define DEFINED_EXTERN
#endif
*/
#else

#ifndef PROTO_PARAMS
#define PROTO_PARAMS(params) params
//#define PROTO_PARAMS(params) (/* nothing */)
#define DEFINED_PROTO_PARAMS
#endif
/*#ifndef EXTERN
#define EXTERN extern
#define DEFINED_EXTERN
#endif
*/
#endif /* __STDC__ */


/*****************************************************************************
 * DEFINES
 *****************************************************************************/

#define ald_GetNumRecordElements(type)      ald_TickLength(type)
//#define ald_GetSignalValueIndirect(sig,val) ald_GetArraySignalValue(sig,val)
#define ald_GetVarValueIndirect(var,val)    ald_GetArrayVarValue(var,val)


/*****************************************************************************
 * TYPE DEFINITIONS
 *****************************************************************************/

#if !defined(__INTERNAL_ALDEC_BUILD__)

typedef void * aldDriverIdT;            /* Handle to a signal driver         */
typedef void * aldProcessIdT;           /* Handle to a process               */
typedef void * aldRegionIdT;            /* Handle to a region                */
typedef void * aldSignalIdT;            /* Handle to a signal                */
typedef void * aldTypeIdT;              /* Handle to a type description      */
typedef void * aldVariableIdT;          /* Handle to a variable              */
typedef void * aldPackageIdT;           /* Handle to a package               */

//typedef void * aldComponentIdT;         /* Handle to a component             */

#endif

typedef          int  aldInt32T;
typedef          char aldInt8T;
typedef unsigned int aldUInt32T;

typedef aldInt32T aldDelayT;

typedef void (*aldVoidFuncPtrT)();

/* Types */

typedef enum aldTypeKindEnum_ {
    ALD_TYPE_SCALAR,               /* Integer types                          */
    ALD_TYPE_ARRAY,
    ALD_TYPE_RECORD,
    ALD_TYPE_ENUM,
    ALD_TYPE_INTEGER__not_used,    /* It's defined only for compatibility with other simulators */
    ALD_TYPE_PHYSICAL,
    ALD_TYPE_REAL,
    ALD_TYPE_ACCESS,
    ALD_TYPE_FILE,
    ALD_TYPE_TIME
} aldTypeKindT;

/* Directions */

typedef enum aldDirectionEnum_ {
    ALD_INTERNAL,
    ALD_DIR_IN,
    ALD_DIR_OUT,
    ALD_DIR_INOUT,
    ALD_DIR_BUFFER
} aldDirectionT;

/* Process triggers */

typedef enum aldProcessTriggerEnum_ {
    ALD_ACTIVE,
    ALD_EVENT
} aldProcessTriggerT;

/* Driver modes */

typedef enum aldDriverModeEnum_ {
    ALD_INERTIAL,
    ALD_TRANSPORT
} aldDriverModeT;

/* Force types */

typedef enum aldForceTypeEnum_ {
    ALD_FORCE_DEFAULT,
    ALD_FORCE_DEPOSIT,
    ALD_FORCE_DRIVE,
    ALD_FORCE_FREEZE
} aldForceTypeT;

/* -------------------- Data structure for time values -------------------- */

typedef union aldTime64Union_ {
    struct {
#if defined(_WIN32) || defined(WIN32) || defined(linux)
        unsigned int lo;
        int          hi;
#define ALD_TIME64_INIT(h,l)    {{(l), (h)}}
#else
        int          hi;
        unsigned int lo;
#define ALD_TIME64_INIT(h,l)    {{(h), (l)}}
#endif
    } s;
#if defined(_LP64) || defined(__LP64__)
    long v;
#elif defined (__GNUC__) && !defined(__STRICT_ANSI__)
    long long v;
#else
    double d;
#endif
} aldTime64T;
#define ALD_TIME64_ASGN(t,h,l)    {(t).s.hi = (h); (t).s.lo = (l);}
#define ALD_TIME64_HI32(t)        ((t).s.hi)
#define ALD_TIME64_LO32(t)        ((t).s.lo)


typedef union aldPhys64Union_ {
    struct {
#if defined(_WIN32) || defined(WIN32) || defined(linux)
        unsigned int lo;
        int          hi;
#define ALD_PHYS64_INIT(h,l)    {{(l), (h)}}
#else
        int          hi;
        unsigned int lo;
#define ALD_PHYS64_INIT(h,l)    {{(h), (l)}}
#endif
    } s;
#if defined(_LP64) || defined(__LP64__)
    long v;
#elif defined (__GNUC__) && !defined(__STRICT_ANSI__)
    long long v;
#else
    double d;
#endif
} aldPhys64T;
#define ALD_PHYS64_ASGN(t,h,l)    {(t).s.hi = (h); (t).s.lo = (l);}
#define ALD_PHYS64_HI32(t)        ((t).s.hi)
#define ALD_PHYS64_LO32(t)        ((t).s.lo)


/* ------------------------------------------------------------------------- */

/* Data structure for physical type units */

typedef struct aldPhysicalDataStruct_ aldPhysicalDataT;
struct aldPhysicalDataStruct_ {
    aldPhysicalDataT  * next;           /* Ptr to next unit; NULL at end     */
    char              * unit_name;      /* Name of unit                      */
    aldInt32T           position;       /* Multiple of primary unit          */
};

/* Data structure for ports and generics */

typedef struct aldInterfaceListStruct_ aldInterfaceListT;
struct aldInterfaceListStruct_ {
    char         * name;                /* Simple name of generic/port       */
    aldTypeIdT     type;                /* Type of generic/port              */
    aldDirectionT  port_dir;            /* Direction of port                 */
                                        /* (All generics are INTERNAL)       */
    union {
      aldInt32T    generic_value;       /* Integer/physical/enum generic     */
                                        /* value                             */
      double       generic_value_real;  /* Real generic value                */
      aldTime64T   generic_value_time;  /* Time generic value                */
      aldPhys64T   generic_value_phys;  /* Physical generic value            */
      void       * generic_array_value; /* Array generic value               */
      aldSignalIdT port;                /* Signal ID of port                 */
    } u;
    aldInterfaceListT *nxt;             /* Next generic/port in list         */
};

typedef enum aldRegionType_ {
	ALD_MODULE,
	ALD_INTERFACE,
	ALD_PROGRAM,
	ALD_CLASS,
	ALD_PACKAGE,
	ALD_PRIMITIVE,
	ALD_UDP,
	ALD_BLOCK,
	ALD_GENERATE
} aldRegionT; 

/*****************************************************************************
 * FUNCTION PROTOTYPES
 *****************************************************************************/

#define PROTO(arg) arg

#ifdef __cplusplus
extern "C" {
#endif

/***********
 * Regions *
 ***********/

EXTERN aldRegionIdT        ald_CreateRegion         PROTO((aldRegionIdT parent,
                                                           char *name));
EXTERN aldRegionIdT        ald_FindRegion           PROTO((char *name));
EXTERN aldRegionIdT        ald_FirstLowerRegion     PROTO((aldRegionIdT reg));
EXTERN aldRegionIdT        ald_GetCallingRegion     PROTO((void));
EXTERN aldRegionIdT        ald_GetCurrentRegion     PROTO((void));
EXTERN aldRegionIdT        ald_GetTopRegion         PROTO((void));
EXTERN aldRegionIdT        ald_HigherRegion         PROTO((aldRegionIdT reg));
EXTERN aldRegionIdT        ald_NextRegion           PROTO((aldRegionIdT reg));

EXTERN char              * ald_GetLibraryName       PROTO((aldRegionIdT reg));
EXTERN char              * ald_GetPrimaryName       PROTO((aldRegionIdT reg));
EXTERN char              * ald_GetRegionFullName    PROTO((aldRegionIdT reg));
EXTERN char              * ald_GetRegionName        PROTO((aldRegionIdT reg));
EXTERN aldRegionT          ald_GetRegionType        PROTO((aldRegionIdT reg));
EXTERN char              * ald_GetRegionSourceName  PROTO((aldRegionIdT reg));
EXTERN char              * ald_GetSecondaryName     PROTO((aldRegionIdT reg));

EXTERN int                 ald_GetRegionKind        PROTO((aldRegionIdT reg));

EXTERN aldInterfaceListT * ald_GetGenericList       PROTO((aldRegionIdT reg));

/*************
 * Processes *
 *************/

EXTERN aldProcessIdT  ald_CreateProcess      PROTO((char *name,
                                                    aldVoidFuncPtrT func,
                                                    void *param));
EXTERN aldProcessIdT  ald_FirstProcess       PROTO((aldRegionIdT reg));
EXTERN aldProcessIdT  ald_NextProcess        PROTO((void));

EXTERN char         * ald_GetProcessName     PROTO((aldProcessIdT proc));

EXTERN void           ald_Desensitize        PROTO((aldProcessIdT proc));
EXTERN void           ald_ScheduleWakeup     PROTO((aldProcessIdT proc,
                                                    aldDelayT delay));
EXTERN void           ald_ScheduleWakeup64   PROTO((aldProcessIdT proc,
                                                    aldTime64T delay));
EXTERN void           ald_Sensitize          PROTO((aldProcessIdT proc,
                                                    aldSignalIdT sig,
                                                    aldProcessTriggerT when));

/***********
 * Signals *
 ***********/

EXTERN aldSignalIdT   ald_CreateSignal         PROTO((char *name,
                                                      aldRegionIdT reg,
                                                      aldTypeIdT type));
EXTERN aldSignalIdT   ald_FindPort             PROTO((aldInterfaceListT *ilist,
                                                      char *name));
EXTERN aldSignalIdT   ald_FindSignal           PROTO((char *name));
EXTERN aldSignalIdT   ald_FirstSignal          PROTO((aldRegionIdT reg));
EXTERN aldSignalIdT   ald_NextSignal           PROTO((void));

EXTERN int            ald_ForceSignal          PROTO((aldSignalIdT sigid,
                                                    char         *value_string,
                                                    aldDelayT     delay,
                                                    aldForceTypeT force_type,
                                                    aldInt32T     cancel_period,
                                                    aldInt32T     repeat_period
                                                    ));
EXTERN int            ald_ReleaseSignal        PROTO((aldSignalIdT sigid));

EXTERN void         * ald_GetArraySignalValue  PROTO((aldSignalIdT sig,
                                                    void *buf));
EXTERN void         * ald_GetSignalValueIndirect  PROTO((aldSignalIdT sig,
                                                    void *buf));
EXTERN aldSignalIdT * ald_GetDrivingSignals    PROTO((char *name));
EXTERN aldDirectionT  ald_GetSignalMode        PROTO((aldSignalIdT sig));
EXTERN char         * ald_GetSignalName        PROTO((aldSignalIdT sig));
EXTERN char         * ald_GetSignalNameIndirect  PROTO((aldSignalIdT sig,
                                                        char *buf,
                                                        int length));
EXTERN aldRegionIdT   ald_GetSignalRegion      PROTO((aldSignalIdT sig));
EXTERN aldSignalIdT * ald_GetSignalSubelements PROTO((aldSignalIdT sig,
                                                      aldSignalIdT *buf));
EXTERN aldTypeIdT     ald_GetSignalType        PROTO((aldSignalIdT sig));
EXTERN aldInt32T      ald_GetSignalValue       PROTO((aldSignalIdT sig));

EXTERN void           ald_SetSignalValue       PROTO((aldSignalIdT sig,
                                                      long val));

EXTERN char         * ald_SignalImage          PROTO((aldSignalIdT sig));


/***********
 * Drivers *
 ***********/

EXTERN aldDriverIdT   ald_CreateDriver         PROTO((aldSignalIdT sig));
EXTERN aldDriverIdT   ald_FindDriver           PROTO((aldSignalIdT sig));
EXTERN aldDriverIdT * ald_GetDriverSubelements PROTO((aldDriverIdT drv,
                                                      aldDriverIdT *buf));
EXTERN char        ** ald_GetDriverNames       PROTO((aldSignalIdT sig,
                                                      aldInt32T *length));
EXTERN char         * ald_GetDriverValues      PROTO((aldSignalIdT sig, 
                                                      aldInt32T *length));

EXTERN void           ald_ScheduleDriver       PROTO((aldDriverIdT drv,
                                                      long value,
                                                      aldDelayT delay,
                                                      aldDriverModeT mode));
EXTERN void           ald_ScheduleDriver64     PROTO((aldDriverIdT drv,
                                                      long value,
                                                      aldTime64T delay,
                                                      aldDriverModeT mode));
EXTERN void           ald_SetDriverOwner       PROTO((aldDriverIdT drv,
                                                      aldProcessIdT proc));
EXTERN aldProcessIdT  ald_GetDriverOwner       PROTO((aldDriverIdT drv));
EXTERN aldSignalIdT   ald_GetDriverSignal      PROTO((aldDriverIdT drv));

/*************
 * Variables *
 *************/

EXTERN aldVariableIdT   ald_FindVar            PROTO((char *name));
EXTERN aldVariableIdT   ald_FirstVar           PROTO((aldProcessIdT proc));
EXTERN aldVariableIdT   ald_NextVar            PROTO((void));

EXTERN void           * ald_GetArrayVarValue   PROTO((aldVariableIdT var,
                                                      void *buf));
EXTERN void           * ald_GetVarAddr         PROTO((char *name));
EXTERN char           * ald_GetVarImage        PROTO((char *name));
EXTERN char           * ald_GetVarImageById    PROTO((aldVariableIdT var));
EXTERN char           * ald_GetVarName         PROTO((aldVariableIdT var));
EXTERN aldVariableIdT * ald_GetVarSubelements  PROTO((aldVariableIdT var,
                                                      aldVariableIdT *buf ));
EXTERN aldTypeIdT       ald_GetVarType         PROTO((aldVariableIdT var));
EXTERN aldInt32T        ald_GetVarValue        PROTO((aldVariableIdT var));

EXTERN void             ald_SetVarValue        PROTO((aldVariableIdT var,
                                                      long val));

/*********
 * Types *
 *********/

EXTERN aldTypeIdT      ald_CreateArrayType     PROTO((aldInt32T left,
                                                      aldInt32T right,
                                                      aldTypeIdT elem_type));
EXTERN aldTypeIdT      ald_CreateEnumType      PROTO((aldInt32T size,
                                                      aldInt32T count,
                                                      char **literals));
EXTERN aldTypeIdT      ald_CreateRealType      PROTO((void));
EXTERN aldTypeIdT      ald_CreateScalarType    PROTO((aldInt32T left,
                                                      aldInt32T right));

EXTERN aldTypeIdT      ald_GetArrayElementType PROTO((aldTypeIdT type));
EXTERN char         ** ald_GetEnumValues       PROTO((aldTypeIdT type));
EXTERN aldPhysicalDataT * ald_GetPhysicalData  PROTO((aldTypeIdT type));
EXTERN aldTypeKindT    ald_GetTypeKind         PROTO((aldTypeIdT type));

EXTERN char          * ald_Image               PROTO((void *value,
                                                      aldTypeIdT type));

EXTERN aldInt32T       ald_TickDir             PROTO((aldTypeIdT type));
EXTERN aldInt32T       ald_TickHigh            PROTO((aldTypeIdT type));
EXTERN aldInt32T       ald_TickLeft            PROTO((aldTypeIdT type));
EXTERN aldInt32T       ald_TickLength          PROTO((aldTypeIdT type));
EXTERN aldInt32T       ald_TickLow             PROTO((aldTypeIdT type));
EXTERN aldInt32T       ald_TickRight           PROTO((aldTypeIdT type));

/*************
 * Callbacks *
 *************/

EXTERN void  ald_AddInputReadyCB         PROTO((int file_desc,
                                                aldVoidFuncPtrT func,
                                                void *param));
EXTERN void  ald_AddOutputReadyCB        PROTO((int file_desc,
                                                aldVoidFuncPtrT func,
                                                void *param));

EXTERN void  ald_AddSocketInputReadyCB   PROTO((int socket_desc,
                                                aldVoidFuncPtrT func,
                                                void *param));
EXTERN void  ald_AddSocketOutputReadyCB  PROTO((int socket_desc,
                                                aldVoidFuncPtrT func,
                                                void *param));

EXTERN void  ald_AddEnvCB            PROTO((aldVoidFuncPtrT func, void *param));
EXTERN void  ald_AddLoadDoneCB       PROTO((aldVoidFuncPtrT func, void *param));
EXTERN void  ald_AddQuitCB           PROTO((aldVoidFuncPtrT func, void *param));
EXTERN void  ald_AddRestartCB        PROTO((aldVoidFuncPtrT func, void *param));
EXTERN void  ald_AddRestoreCB        PROTO((aldVoidFuncPtrT func, void *param));
EXTERN void  ald_AddRestoreDoneCB    PROTO((aldVoidFuncPtrT func, void *param));
EXTERN void  ald_AddSaveCB           PROTO((aldVoidFuncPtrT func, void *param));
EXTERN void  ald_AddSimStatusCB      PROTO((aldVoidFuncPtrT func, void *param));

EXTERN void  ald_RemoveEnvCB         PROTO((aldVoidFuncPtrT func, void *param));
EXTERN void  ald_RemoveLoadDoneCB    PROTO((aldVoidFuncPtrT func, void *param));
EXTERN void  ald_RemoveQuitCB        PROTO((aldVoidFuncPtrT func, void *param));
EXTERN void  ald_RemoveRestartCB     PROTO((aldVoidFuncPtrT func, void *param));
EXTERN void  ald_RemoveRestoreCB     PROTO((aldVoidFuncPtrT func, void *param));
EXTERN void  ald_RemoveRestoreDoneCB PROTO((aldVoidFuncPtrT func, void *param));
EXTERN void  ald_RemoveSaveCB        PROTO((aldVoidFuncPtrT func, void *param));
EXTERN void  ald_RemoveSimStatusCB   PROTO((aldVoidFuncPtrT func, void *param));

/*********************
 * Memory Management *
 *********************/

EXTERN void   * ald_Malloc                PROTO((unsigned long size));
EXTERN void   * fli_Alloc                 PROTO((unsigned long size, int alignment));
EXTERN void   * ald_Realloc               PROTO((void *p, unsigned long size));
EXTERN void     ald_Free                  PROTO((void *p));

/* The following function can be used in place of free() to
 * free memory malloc'd by vsim and returned by an ald_* function.
 * It cannot be used to free memory allocated by ald_Malloc() or
 * malloc'd by foreign code.
 */
EXTERN void     ald_VsimFree              PROTO((void *ptr));

/******************
 * Save & Restore *
 ******************/

EXTERN char   * ald_GetCheckpointFilename PROTO((void));
EXTERN int      ald_IsRestore             PROTO((void));
EXTERN int      ald_IsColdRestore         PROTO((void));
EXTERN void     ald_SaveBlock             PROTO((char *p, unsigned long size));
EXTERN void     ald_SaveChar              PROTO((char data));
EXTERN void     ald_SaveLong              PROTO((long data));
EXTERN void     ald_SaveShort             PROTO((short data));
EXTERN void     ald_SaveString            PROTO((char *data));
EXTERN void     ald_RestoreBlock          PROTO((char *p));
EXTERN char     ald_RestoreChar           PROTO((void));
EXTERN long     ald_RestoreLong           PROTO((void));
EXTERN short    ald_RestoreShort          PROTO((void));
EXTERN char   * ald_RestoreString         PROTO((void));
EXTERN void     ald_RestoreProcess        PROTO((aldProcessIdT proc,
                                                 char *name,
                                                 aldVoidFuncPtrT func,
                                                 void *param));

/*****************
 * Time & Events *
 *****************/

EXTERN aldUInt32T    ald_Delta           PROTO((void));
EXTERN aldInt32T     ald_Now             PROTO((void));
EXTERN aldTime64T  * ald_NowIndirect     PROTO((aldTime64T *timep));
EXTERN aldInt32T     ald_NowUpper        PROTO((void));

EXTERN int      ald_GetNextEventTime     PROTO((aldTime64T *timep));
EXTERN int      ald_GetNextNextEventTime PROTO((aldTime64T *timep));
EXTERN int      ald_GetResolutionLimit   PROTO((void));
EXTERN void     ald_GetRunStopTime       PROTO((aldTime64T *timep));

/****************************
 * Communication & Commands *
 ****************************/

EXTERN void     ald_AddCommand           PROTO((char *cmd_name,
                                                aldVoidFuncPtrT func));
EXTERN void     ald_AddTclCommand        PROTO((char *cmd_name,
                                                aldVoidFuncPtrT func,
                                                void *param,
                                                aldVoidFuncPtrT funcDeleteCB));
EXTERN void     ald_Command              PROTO((char *cmd));
EXTERN int      ald_Cmd                  PROTO((char *cmd));
EXTERN void   * ald_Interp               PROTO((void));

EXTERN int      ald_AskStdin             PROTO((char *buf, char *prompt));
EXTERN void     ald_PrintMessage         PROTO((char *msg));
EXTERN void     ald_PrintFormatted       PROTO((char *format, ...));

EXTERN void     ald_Break                PROTO((void));
EXTERN void     ald_FatalError           PROTO((void));
EXTERN void     ald_Quit                 PROTO((void));

/*****************
 * Miscellaneous *
 *****************/

EXTERN char   * ald_GetProductVersion    PROTO((void));

EXTERN char   * ald_FindProjectEntry     PROTO((char *section,
                                                char *name,
                                                int   expand));
EXTERN void     ald_WriteProjectEntry    PROTO((char *key, char *val));

EXTERN int      ald_IsFirstInit          PROTO((void));

/***********
 * Tracing *
 ***********/

EXTERN int      ald_GetTraceLevel        PROTO((void));
EXTERN void     ald_TraceActivate        PROTO((void));
EXTERN void     ald_TraceOff             PROTO((void));
EXTERN void     ald_TraceOn              PROTO((int level, char *tag));
EXTERN void     ald_TraceSkipID          PROTO((int n));
EXTERN void     ald_TraceSuspend         PROTO((void));

/***********
 * Extension *
 ***********/

EXTERN char*    ald_GetTypeName         PROTO((aldTypeIdT type));
EXTERN char*    ald_GetTypeDes          PROTO((aldTypeIdT type));

// The next constants is to be used as source_number parameter to the 
// ald_GetComponentSourceName function depending on which source kind is needed.
static const int COMPONENT_SOURCE_VHDL_ENTITY       = 0;
static const int COMPONENT_SOURCE_VHDL_ARCHITECTURE = 1;
static const int COMPONENT_SOURCE_VERILOG_MODULE    = 0;

EXTERN char*    ald_GetComponentSourceName PROTO((aldRegionIdT region, int source_number));
EXTERN char*    ald_GetComponentSourceType PROTO((aldRegionIdT region, int source_number));
EXTERN char*    ald_GetComponentCompileSourceName PROTO((aldRegionIdT region));
EXTERN int      ald_GetComponentSourceNumber PROTO((aldRegionIdT region));
EXTERN const char*
                ald_GetComponentSourceKind PROTO((aldRegionIdT region, int source_number));
EXTERN aldPackageIdT
                ald_GetNextPackage      PROTO((aldRegionIdT region, aldPackageIdT prev));
EXTERN char*    ald_GetPackageName      PROTO((aldPackageIdT region));
EXTERN char*    ald_GetPackageLibName   PROTO((aldPackageIdT region));
EXTERN char*    ald_GetPackageSource    PROTO((aldPackageIdT region, int bBody));
EXTERN aldVariableIdT 
                ald_GetNextGeneric      PROTO((aldRegionIdT region, aldVariableIdT previous));
EXTERN char*    ald_GetGenericInitValue PROTO((aldRegionIdT region, aldVariableIdT generic));
EXTERN int      spv_printf              PROTO_PARAMS((const char *format,...));

typedef enum {
    HANDLE_TYPE_REGION,
    HANDLE_TYPE_SIGNAL,
    HANDLE_TYPE_VARIABLE
} _FLI_HANDLE_TYPE_;

typedef int* PLI_Handle;

EXTERN PLI_Handle ald_ConvertHandleToPLI PROTO((void* hdl, _FLI_HANDLE_TYPE_ type));

#ifdef __cplusplus
}
#endif

#pragma pack()


#endif /* ALD_H */

/* ***************************** End of Header ***************************** */
