#ifndef ALDEC_RC_H
#define ALDEC_RC_H

#include <stdio.h>
#include <windows.h>

#define VSIM_OFF	0x00
#define VSIM_ON		0x01
#define VSIM_BUSY	0x02

#define CS_OK 0
#define CS_READY CS_OK
#define CS_ERROR -1
#define CS_TIMEOUT -2
#define CS_BUSY -4
#define CS_INV_ARG -8

#define CS_SIM_RDY 1
#define CS_CMD_RDY 2
#define CS_SIM_FINISH 3
//Data Set Ready - Simulator has post data but they have not been processed yet
#define CS_SIM_DSR 4  
#define CS_SIM_ERROR CS_ERROR
#define CS_SIM_TIMEOUT CS_TIMEOUT
#define CS_SIM_BUSY CS_BUSY
#define CS_SIM_QUEUE_FULL -9
#define CS_SIM_INV_HANDLE -10
#define CS_SIM_QUIT -12

#define CONSOLE_ON true
#define CONSOLE_OFF false

typedef struct t_acc_vecval
{
unsigned int aval;
unsigned int bval;
} s_acc_vecval, *p_acc_vecval;

typedef enum ESignalType {
	LOG_TYPE = 0,
	INT_TYPE,
	FLOAT_TYPE
} ESignalType;

typedef struct TSignal {
	char *pname;		
	int pli_handle;	
	int size;			//size of vector	
	int chunks;			//number of 32-bit chunks
	int user;			//Auxilary variable
	s_acc_vecval val[4];
} TSignal;

typedef struct THDLTime {
	unsigned int high;
	unsigned int low;
} THDLTime;

struct TModule;
struct TSig;
struct THDLType;

typedef struct THDLIntRange {
	int left;
	int right;
} THDLIntRange;

typedef struct THDLFloatRange {
	double left;
	double right;
} THDLFloatRange;

typedef struct THDLVector {
	bool is_vect;
	int size;
	int left;
	int right;
} THDLVector;

typedef struct THDLArray {
	int num_dims;
	THDLIntRange *rgs;
	THDLType *base_type;
} THDLArray;

typedef struct THDLRecordField {
	char *name;
	THDLType *type;
	int offset;
} THDLTRecordField;

typedef struct THDLRecord {
	int num_fields;
	THDLRecordField *fields;
} THDLTRecord;

typedef struct THDLPhysRange {
	__int64 low;
	__int64 high;
} THDLPhysRange;

typedef struct THDLPhysUnit {
	char *name;
	__int64 position;
} THDLPhysUnit;

typedef struct THDLPhys {
	int num_units;
	THDLPhysUnit *units;
	THDLPhysRange range;
} THDLPhys;

typedef struct THDLEnum {
	int num_enums;
    char **enumv;
} THDLEnum;

typedef union {
	THDLEnum		e;
	THDLArray		a;
	THDLVector		v;
	THDLIntRange	i;
	THDLFloatRange	f;
	THDLPhys		p;
	THDLRecord		r;
} THDLTypeDesc;

typedef struct THDLType {
	unsigned int handle;
	char *name;
	char kind;
	THDLTypeDesc desc;
	int size;
} THDLType;

typedef union {
	char k; /*short info in verilog */
	THDLType *t; /*full info for VHDL */
} THDLTypeUnion;

typedef void (*TMonitorCb)(TSig *Sig);
typedef void (*TTimeChangeCb)(const THDLTime *Time, void *user);

typedef union {
	 int enumv, *enumvs;
     char smallenumv, *smallenumvs;
     int  intg, *intgs;
     double real, *reals;
	 s_acc_vecval vec, *vecs;
} THDLVal;

typedef struct TSig {
	unsigned int handle;
	char kind;
	char *name;
	TModule *parent;
	THDLType *type;
	//int size;
	TMonitorCb mon_cb;
	void *user;
	unsigned int mon_handle;
	unsigned char trace : 1;
	unsigned char imm_resp : 1; /*immediate simulator response*/
	unsigned char pckg_allowed : 1;
	unsigned char set_val_ack : 1;
	THDLVal value;
} TSig;


typedef struct TParam {
	unsigned int handle;
	char *name;
	TModule *parent;
	int size;
	THDLTypeUnion type;
	THDLVal value;
} TParam;

typedef struct TModule {
	unsigned int handle;
	unsigned int hdl : 2;
	TModule *Parent;
	char *name;	
	char *def_name;
	char *dsgu_name;
	TModule *Modules;
	int ModulesCnt;
	TSig *Ports;
	int PortsCnt;
	TParam *Params;
	int ParamsCnt;
	TSig *Signals;
	int SignalsCnt;
	unsigned int user;
} TModule;

typedef struct TDesign {
	TModule *Modules;
	int ModulesCnt;
} TDesign;

#define PORT_UNSPEC			0
#define PORT_IN				1
#define PORT_OUT			2
#define PORT_INOUT			3
#define PORT_BUFFER			4
#define PORT_LINK			5
#define INTERNAL_SIGNAL		0x10

#define TYPE_UNSPEC			0
#define TYPE_INT			1
#define TYPE_INT_R			2
#define TYPE_FLOAT			3
#define TYPE_FLOAT_R		4
#define TYPE_ENUM			6
#define TYPE_SMALL_ENUM		7
#define TYPE_PHYS			8
#define TYPE_SMALL_PHYS		9
#define TYPE_RECORD			10
#define TYPE_TIME			11
#define TYPE_VL				0x10
#define TYPE_VL_NET			0x11
#define TYPE_VL_REG			0x12
#define TYPE_VL_STRING		0x14
#define TYPE_ARRAY			0x40

#define OBJ_VERILOG			0
#define OBJ_VHDL			1
#define OBJ_EDIF			2

typedef void* TAldecCmd; 

extern "C" {

//Command channel functions
TAldecCmd* StartAldecIPC(char *path, int port, int time_out, bool console);
TAldecCmd* OpenCmdChannel(int port, int attempts);
void CloseCmdChannel(TAldecCmd *cmd_if);

int ExecuteCmd(TAldecCmd *cmd_if, char *cmd);	
int VaExecuteCmd(TAldecCmd *cmd_if, char *cmd, ...);	
int PostCmd(TAldecCmd *cmd_if, char *cmd);	
int VaPostCmd(TAldecCmd *cmd_if, char *cmd, ...);	
int WaitCmd(TAldecCmd *cmd_if, int time_out);
int BreakCmd(TAldecCmd *cmd_if);
int CmdBusy(TAldecCmd *cmd_if);
int GetCmdStatus(TAldecCmd *cmd_if);

int SetRespBufSizeLimit(TAldecCmd *cmd_if, long size_max);
char* GetResponse(TAldecCmd *cmd_if);
int GetResponseId(TAldecCmd *cmd_if);

int Print(TAldecCmd *cmd_if, int kind, char *msg);
int VaPrint(TAldecCmd *cmd_if, int kind, char *msg, ...);
int PrintMessage(TAldecCmd *cmd_if, char *msg);
int VaPrintMessage(TAldecCmd *cmd_if, char *msg, ...);
int PrintWarning(TAldecCmd *cmd_if, char *msg);
int VaPrintWarning(TAldecCmd *cmd_if, char *msg, ...);
int PrintError(TAldecCmd *cmd_if, char *msg);
int VaPrintError(TAldecCmd *cmd_if, char *msg, ...);

int GetSimulatorState(TAldecCmd *cmd_if);
int GetSimulatorPID(TAldecCmd *cmd_if);

int OpenOutFile(TAldecCmd *cmd_if, char *out_fn, bool append);
void CloseOutFile(TAldecCmd *cmd_if);

//Simulation channel functions
int OpenSimChannel(TAldecCmd *cmd, int pid, int time_out);
int CloseSimChannel(TAldecCmd *cmd);

int ExecuteSimQueue(TAldecCmd *cmd);
int PostSimQueue(TAldecCmd *cmd);
int WaitSim(TAldecCmd *cmd_if, int time_out);
int SimBusy(TAldecCmd *cmd_if);

int QueueSetDelay(TAldecCmd *cmd, __int64 d);
int QueueSetLongDelay(TAldecCmd *cmd, __int64 d);
int QueueCalculate(TAldecCmd *cmd);
int QueueFetchValue(TAldecCmd *cmd, TSignal *signal);
int QueueSetValue(TAldecCmd *cmd, TSignal *signal);
int QueueGetSignal(TAldecCmd *cmd, TSignal *signal);
int QueueStop(TAldecCmd *cmd);
int QueueStopAfterRunAndWake(TAldecCmd *cmd);
int QueueStopAfterRun(TAldecCmd *cmd);

//** Design scan **//
TDesign *GetDesignRoot(TAldecCmd *cmd);
int GetRootModule(TAldecCmd *cmd);
int QueueGetChildren(TAldecCmd *cmd, TModule *Ref);
int QueueGetPorts(TAldecCmd *cmd, TModule *Ref);
int QueueGetParameters(TAldecCmd *cmd, TModule *Module);
int QueueGetSignals(TAldecCmd cmd, TModule *Module);
int GetTypes(TAldecCmd *cmd);
int GetDesign(TAldecCmd *cmd);
int QueueGetSigVal(TAldecCmd *cmd, TSig *Sig);
int QueueMonitorOn(TAldecCmd *cmd, TSig *Sig, TMonitorCb MonCb, void *user);
int QueueMonitorOff(TAldecCmd *cmd, TSig *Sig);
TTimeChangeCb SetTimeChangeCb(TAldecCmd *cmd, TTimeChangeCb TimeChgCb, void *user);

}

#endif
