-- ----------------------------------------------------------------------------
-- Common defines
-- ----------------------------------------------------------------------------
Library IEEE;
Use     IEEE.Std_Logic_1164.All;
-- ----------------------------------------------------------------------------

-- ----------------------------------------------------------------------------
Package ReplaceWith_EntityName_WB_I2S_Package Is
    Constant cCoreRevision                 : Std_Logic_Vector(7 Downto 0)  := x"01";
    Constant cCoreSize                     : Integer := ReplaceWith_FifoAddressWidth;
    Constant cFifoWidth                    : Integer := 24;
    Constant cFifoDepth                    : Integer := ReplaceWith_FifoDepth;
    Constant cFifoAddressWidth             : Integer := ReplaceWith_FifoAddressWidth;

    Constant cI2S_WordLength_16            : Std_Logic_Vector(1 Downto 0) := "00";
    Constant cI2S_WordLength_20            : Std_Logic_Vector(1 Downto 0) := "01";
    Constant cI2S_WordLength_24            : Std_Logic_Vector(1 Downto 0) := "10";
    Constant cI2S_WordLength_32            : Std_Logic_Vector(1 Downto 0) := "11";

    Constant cI2S_WordJustification_Normal : Std_Logic_Vector(1 Downto 0) := "00";
    Constant cI2S_WordJustification_Left   : Std_Logic_Vector(1 Downto 0) := "01";
    Constant cI2S_WordJustification_Right  : Std_Logic_Vector(1 Downto 0) := "10";

    Constant cRegAdr_Control               : Std_Logic_Vector(2 Downto 0) := "000";
    Constant cRegAdr_Mode                  : Std_Logic_Vector(2 Downto 0) := "001";
    Constant cRegAdr_Status                : Std_Logic_Vector(2 Downto 0) := "010";
    Constant cRegAdr_Data                  : Std_Logic_Vector(2 Downto 0) := "011";
    Constant cRegAdr_RX_Pointers           : Std_Logic_Vector(2 Downto 0) := "100";
    Constant cRegAdr_TX_Pointers           : Std_Logic_Vector(2 Downto 0) := "101";

    Constant cWordLength_16Hex             : Std_Logic_Vector(5 Downto 0) := "010000";
    Constant cWordLength_20Hex             : Std_Logic_Vector(5 Downto 0) := "010100";
    Constant cWordLength_24Hex             : Std_Logic_Vector(5 Downto 0) := "011000";
    Constant cWordLength_32Hex             : Std_Logic_Vector(5 Downto 0) := "100000";

    Constant cBitCounterTX_16              : Std_Logic_Vector(4 Downto 0) := "01111";
    Constant cBitCounterTX_20              : Std_Logic_Vector(4 Downto 0) := "10011";
    Constant cBitCounterTX_24              : Std_Logic_Vector(4 Downto 0) := "10111";
    Constant cBitCounterTX_32              : Std_Logic_Vector(4 Downto 0) := "10111";
    Constant cBitCounterTXZero             : Std_Logic_Vector(4 Downto 0) := "00000";

    Constant cBitCounterRX_16              : Std_Logic_Vector(5 Downto 0) := "010000";
    Constant cBitCounterRX_20              : Std_Logic_Vector(5 Downto 0) := "010100";
    Constant cBitCounterRX_24              : Std_Logic_Vector(5 Downto 0) := "011000";
    Constant cBitCounterRX_32              : Std_Logic_Vector(5 Downto 0) := "011000";
    Constant cBitCounterRXZero             : Std_Logic_Vector(5 Downto 0) := "000000";

    Constant cBitCounterStart_0            : Std_Logic_Vector(4 Downto 0) := "00000";
    Constant cBitCounterStart_1            : Std_Logic_Vector(4 Downto 0) := "00001";
    Constant cBitCounterStart_8            : Std_Logic_Vector(4 Downto 0) := "01000";
    Constant cBitCounterStart_12           : Std_Logic_Vector(4 Downto 0) := "01100";
    Constant cBitCounterStart_16           : Std_Logic_Vector(4 Downto 0) := "10000";
End Package;
-- ----------------------------------------------------------------------------


-- ----------------------------------------------------------------------------
-- Wishbone Interface
-- ----------------------------------------------------------------------------
-- ----------------------------------------------------------------------------
Library IEEE;
Use     IEEE.Std_Logic_1164.All;
Use     IEEE.Numeric_Std.All;
Use     Work.ReplaceWith_EntityName_WB_I2S_Package.All;
-- ----------------------------------------------------------------------------

-- ----------------------------------------------------------------------------
Entity ReplaceWith_EntityName_WB_I2S_WishboneInterface Is
--    Generic (
--        ReplaceWith_FifoAddressWidth : Integer := 5;
--        24        : Integer := 24;
--        ReplaceWith_FifoDepth        : Integer := 32
--    );
    Port (
        -- Wishbone interface
        CLK_I : In  Std_Logic;
        RST_I : In  Std_Logic;
        STB_I : In  Std_Logic;
        CYC_I : In  Std_Logic;
        DAT_I : In  Std_Logic_Vector(31 Downto 0);
        DAT_O : Out Std_Logic_Vector(31 Downto 0);
        WE_I  : In  Std_Logic;
--        SEL_I : In  Std_Logic_Vector(3 Downto 0);
        ADR_I : In  Std_Logic_Vector(2 Downto 0);
        ACK_O : Out Std_Logic;
        INT_O : Out Std_Logic;

        -- FIFO Controls
        FIFO_RX_CLK            : Out Std_Logic;
        FIFO_TX_CLK            : Out Std_Logic;
        FIFO_RX_DATA           : In  Std_Logic_Vector(23 Downto 0);
        FIFO_TX_DATA           : Out Std_Logic_Vector(23 Downto 0);
        FIFO_RX_WATERMARK_FLAG : In  Std_Logic;
        FIFO_RX_FULL           : In  Std_Logic;
        FIFO_RX_EMPTY          : In  Std_Logic;
        FIFO_TX_WATERMARK_FLAG : In  Std_Logic;
        FIFO_TX_FULL           : In  Std_Logic;
        FIFO_TX_EMPTY          : In  Std_Logic;
        FIFO_TX_RESET          : Out Std_Logic;
        FIFO_RX_RESET          : Out Std_Logic;
        FIFO_RX_WATERMARK      : Out Std_Logic_Vector(ReplaceWith_FifoAddressWidth-1 Downto 0);
        FIFO_TX_WATERMARK      : Out Std_Logic_Vector(ReplaceWith_FifoAddressWidth-1 Downto 0);

        FIFO_RX_POINTER        : In  Std_Logic_Vector(ReplaceWith_FifoAddressWidth-1 Downto 0);
        FIFO_TX_POINTER        : In  Std_Logic_Vector(ReplaceWith_FifoAddressWidth-1 Downto 0);
        FIFO_RX_WRITE_POINTER  : In  Std_Logic_Vector(ReplaceWith_FifoAddressWidth-1 Downto 0);
        FIFO_TX_READ_POINTER  : In  Std_Logic_Vector(ReplaceWith_FifoAddressWidth-1 Downto 0);

        FIFO_TX_WRITE_ENABLE   : Out Std_Logic;
        FIFO_RX_READ_ENABLE    : Out Std_Logic;

        MONO_MODE_IN           : Out Std_Logic;
        MONO_MODE_OUT          : Out Std_Logic;
        MONO_CHANNEL_SELECT    : Out Std_Logic;
        I2S_RUN                : Out Std_Logic;
        I2S_RX_RUN             : Out Std_Logic;
        I2S_TX_RUN             : Out Std_Logic;
        I2S_WORD_LENGTH        : Out Std_Logic_Vector(1 Downto 0);
        I2S_WORD_JUSTIFICATION : Out Std_Logic_Vector(1 Downto 0);

        -- Clock Generation Controls
        CLOCK_PRESCALER        : Out Std_Logic_Vector(7 Downto 0);
        CLOCK_BIT_DIVIDER      : Out Std_Logic_Vector(3 Downto 0);
        CLOCK_ENABLE_PRESCALER : Out Std_Logic
    );
End;
-- ----------------------------------------------------------------------------

-- ----------------------------------------------------------------------------
Architecture RTL Of ReplaceWith_EntityName_WB_I2S_WishboneInterface Is
-- ----------------------------------------------------------------------------

    Signal   WordLength           : Std_Logic_Vector(1 Downto 0);
    Signal   WordJustification    : Std_Logic_Vector(1 Downto 0);

    Signal   WishboneRequest      : Std_Logic;
    Signal   WishboneReadRequest  : Std_Logic;
    Signal   WishboneWriteRequest : Std_Logic;

    Signal   Prescaler            : Std_Logic_Vector(7 Downto 0);
    Signal   Watermark            : Std_Logic_Vector(15 Downto 0);
    Signal   ClockPrescalerEnable : Std_Logic;
    Signal   BitDivider           : Std_Logic_Vector(3 Downto 0);

    Signal   Mode                 : Std_Logic_Vector(6 Downto 0);
    Alias    aMonoOut             : Std_Logic Is Mode(0);
    Alias    aMonoIn              : Std_Logic Is Mode(1);
    Alias    aMonoInSelect        : Std_Logic Is Mode(2);
    Alias    aEnableTX            : Std_Logic Is Mode(3);
    Alias    aEnableRX            : Std_Logic Is Mode(4);
    Alias    aEnableTXInt         : Std_Logic Is Mode(5);
    Alias    aEnableRXInt         : Std_Logic Is Mode(6);

    Signal   FifoTxLR             : Std_Logic;
    Signal   FifoRxLR             : Std_Logic;

    Signal   FifoTxPointer        : Integer Range 0 To ReplaceWith_FifoDepth - 1;
    Signal   FifoRxPointer        : Integer Range 0 To ReplaceWith_FifoDepth - 1;
    Signal   FifoTxReadPointer    : Integer Range 0 To ReplaceWith_FifoDepth - 1;
    Signal   FifoRxWritePointer   : Integer Range 0 To ReplaceWith_FifoDepth - 1;

    Signal   FifoTxReset          : Std_Logic;
    Signal   FifoRxReset          : Std_Logic;
    Signal   ReadAcknowledge      : Std_Logic;
-- ----------------------------------------------------------------------------
Begin
-- ----------------------------------------------------------------------------

    -- ------------------------------------------------------------------------
    FIFO_RX_RESET          <= FifoRxReset Or RST_I Or Not aEnableRX;
    FIFO_TX_RESET          <= FifoTxReset Or RST_I Or Not aEnableTX;
    CLOCK_PRESCALER        <= Prescaler;
    CLOCK_BIT_DIVIDER      <= BitDivider;
    CLOCK_ENABLE_PRESCALER <= ClockPrescalerEnable;
    FIFO_RX_CLK            <= CLK_I;
    FIFO_TX_CLK            <= CLK_I;
    FIFO_RX_WATERMARK      <= Watermark(ReplaceWith_FifoAddressWidth-1 Downto 0);
    FIFO_TX_WATERMARK      <= Watermark(ReplaceWith_FifoAddressWidth-1 Downto 0);
    MONO_CHANNEL_SELECT    <= aMonoInSelect;
    MONO_MODE_IN           <= aMonoIn;
    MONO_MODE_OUT          <= aMonoOut;
    I2S_RUN                <= aEnableTX Or aEnableRX;
    I2S_RX_RUN             <= aEnableRX;
    I2S_TX_RUN             <= aEnableTX;
    I2S_WORD_LENGTH        <= WordLength;
    I2S_WORD_JUSTIFICATION <= WordJustification;
    -- ------------------------------------------------------------------------

    -- ------------------------------------------------------------------------
    INT_O <= '1' When (FIFO_RX_WATERMARK_FLAG='1' And aEnableRX='1' And aEnableRXInt='1') Or
                      (FIFO_TX_WATERMARK_FLAG='1' And aEnableTX='1' And aEnableTXInt='1') Else
             '0';



    ACK_O                <= (STB_I And CYC_I And WE_I) Or ReadAcknowledge;
    WishboneRequest      <= STB_I And CYC_I;
    WishboneReadRequest  <= (WishboneRequest And Not WE_I) And Not ReadAcknowledge;
    WishboneWriteRequest <= (WishboneRequest And     WE_I);
    FIFO_RX_READ_ENABLE  <= '1' When (WishboneReadRequest='1' And ADR_I=cRegAdr_Data) Else '0';
    -- ------------------------------------------------------------------------

    -- ------------------------------------------------------------------------
    FifoTxPointer <= To_Integer(Unsigned(FIFO_TX_POINTER));
    FifoRxPointer <= To_Integer(Unsigned(FIFO_RX_POINTER));
    FifoTxReadPointer <= To_Integer(Unsigned(FIFO_TX_READ_POINTER));
    FifoRxWritePointer <= To_Integer(Unsigned(FIFO_RX_WRITE_POINTER));
    FifoTxLR      <= '1' When (FifoTxPointer Mod 2) = 1 And aMonoIn  = '0' Else '0';
    FifoRxLR      <= '1' When (FifoRxPointer Mod 2) = 1 And aMonoOut = '0' Else '0';
    -- ------------------------------------------------------------------------

    -- ------------------------------------------------------------------------
    WishboneDataRead:
    Process(ADR_I, Watermark, ClockPrescalerEnable, Prescaler, Mode, WordLength, WordJustification,
            FIFO_RX_DATA,
            FIFO_RX_WATERMARK_FLAG, FIFO_RX_FULL, FIFO_RX_EMPTY,
            FIFO_TX_WATERMARK_FLAG, FIFO_TX_FULL, FIFO_TX_EMPTY,
            FifoRxLR, FifoTxLR,
            FifoRxPointer, FifoRxWritePointer,
            FifoRxPointer, FifoTxReadPointer)
    Begin
        Case ADR_I Is
            When cRegAdr_Control =>
--                DAT_O(31 Downto 17) <= (Others=>'0');
                DAT_O(15)           <= ClockPrescalerEnable;
                DAT_O(31 Downto 16) <= Watermark;
                DAT_O( 7 Downto  0) <= Prescaler;
                DAT_O(14 Downto  8) <= (Others=>'0');
            When cRegAdr_Mode =>
                DAT_O(31 Downto 17) <= (Others=>'0');
                DAT_O(16 Downto 15) <= WordJustification;
                DAT_O(14 Downto  8) <= Mode;
                Case WordLength Is
                    When cI2S_WordLength_16 => DAT_O(7 Downto 0) <= "00" & cWordLength_16Hex;
                    When cI2S_WordLength_20 => DAT_O(7 Downto 0) <= "00" & cWordLength_20Hex;
                    When cI2S_WordLength_24 => DAT_O(7 Downto 0) <= "00" & cWordLength_24Hex;
                    When Others             => DAT_O(7 Downto 0) <= "00" & cWordLength_32Hex;
                End Case;
            When cRegAdr_Status =>
                DAT_O(31 Downto 8) <= Std_Logic_Vector(To_Unsigned(cCoreSize, 16)) & cCoreRevision;
                DAT_O(7) <= FIFO_RX_WATERMARK_FLAG;
                DAT_O(6) <= FIFO_RX_FULL;
                DAT_O(5) <= FIFO_RX_EMPTY;
                DAT_O(4) <= FifoRxLR;
                DAT_O(3) <= FIFO_TX_WATERMARK_FLAG;
                DAT_O(2) <= FIFO_TX_FULL;
                DAT_O(1) <= FIFO_TX_EMPTY;
                DAT_O(0) <= FifoTxLR;
            When cRegAdr_RX_Pointers =>
                DAT_O(31 Downto 16) <= Std_Logic_Vector(To_Unsigned(FifoRxPointer, 16));        --Read
                DAT_O(15 Downto  0) <= Std_Logic_Vector(To_Unsigned(FifoRxWritePointer, 16));   --Write
            When cRegAdr_TX_Pointers =>
                DAT_O(31 Downto 16) <= Std_Logic_Vector(To_Unsigned(FifoTxReadPointer, 16));    --REad
                DAT_O(15 Downto  0) <= Std_Logic_Vector(To_Unsigned(FifoTxPointer, 16));        --Write
            When Others =>
                Case WordLength Is
                    When cI2S_WordLength_16 =>
                        DAT_O(31 Downto 16) <= (Others=>FIFO_RX_DATA(15));
                        DAT_O(15 Downto  0) <= FIFO_RX_DATA(15 Downto 0);
                    When cI2S_WordLength_20 =>
                        DAT_O(31 Downto 20) <= (Others=>FIFO_RX_DATA(19));
                        DAT_O(19 Downto  0) <= FIFO_RX_DATA(19 Downto 0);
                    When cI2S_WordLength_24 =>
                        DAT_O(31 Downto 24) <= (Others=>FIFO_RX_DATA(23));
                        DAT_O(23 Downto  0) <= FIFO_RX_DATA(23 Downto 0);
                    When Others =>
                        DAT_O(31 Downto  8) <= FIFO_RX_DATA(23 Downto 0);
                        DAT_O( 7 Downto  0) <= (Others=>'0');
                End Case;
        End Case;
    End Process;
    -- ------------------------------------------------------------------------

    Process(CLK_I)
    Begin
        If Rising_Edge(CLK_I) Then
            If RST_I='1' Then
                ReadAcknowledge <= '0';
            Else
                If ReadAcknowledge='1' Then
                    ReadAcknowledge <= '0';
                Else
                    If STB_I='1' And CYC_I='1' And WE_I='0' Then
                        ReadAcknowledge <= '1';
                    End If;
                End If;
            End If;
        End If;
    End Process;
    -- ------------------------------------------------------------------------
    WishboneDataWrite:
    Process(CLK_I)
    Begin
        If Rising_Edge(CLK_I) Then
            If RST_I='1' Then
                FIFO_TX_WRITE_ENABLE <= '0';
                FifoTxReset          <= '0';
                FifoRxReset          <= '0';
                Watermark            <= (Others=>'0');
                Prescaler            <= (Others=>'0');
                WordLength           <= cI2S_WordLength_24;
                BitDivider           <= (Others=>'0');
                ClockPrescalerEnable <= '0';
                Mode                 <= (Others=>'0');
                WordJustification    <= cI2S_WordJustification_Normal;
            Else
                FIFO_TX_WRITE_ENABLE <= '0';
                FifoRxReset          <= '0';
                FifoTxReset          <= '0';

                If WishboneWriteRequest='1' Then
                    FIFO_TX_WRITE_ENABLE <= '0';
                    FifoRxReset          <= '0';
                    FifoTxReset          <= '0';
                    Case ADR_I Is
                        When cRegAdr_Control =>
                            ClockPrescalerEnable <= DAT_I(15);
                            Watermark            <= DAT_I(31 Downto 16);
                            Prescaler            <= DAT_I( 7 Downto 0);

                        When cRegAdr_Mode =>
                            Mode              <= DAT_I(14 Downto 8);
                            WordJustification <= DAT_I(16 Downto 15);

                            Case DAT_I(5 Downto 0) Is
                                When cWordLength_16hex =>
                                    WordLength <= cI2S_WordLength_16;
                                    BitDivider <= "1000";
                                When cWordLength_20Hex =>
                                    WordLength <= cI2S_WordLength_20;
                                    BitDivider <= "0110";
                                When cWordLength_24Hex =>
                                    WordLength <= cI2S_WordLength_24;
                                    BitDivider <= "0101";
                                When Others =>
                                    WordLength <= cI2S_WordLength_32;
                                    BitDivider <= "0101";
                            End Case;
                            If aMonoOut /= DAT_I(8) Or DAT_I(11) = '0' Then
                                FifoTxReset   <= '1';
                            End If;
                            If aMonoIn /= DAT_I(9) Or DAT_I(12) = '0' Then
                                FifoRxReset <= '1';
                            End If;

                        When cRegAdr_Data =>
                            Case WordLength Is
                                When cI2S_WordLength_16 =>  FIFO_TX_DATA <= DAT_I(15 Downto 0) & "00000000";
                                When cI2S_WordLength_20 =>  FIFO_TX_DATA <= DAT_I(19 Downto 0) & "0000";
                                When cI2S_WordLength_24 =>  FIFO_TX_DATA <= DAT_I(23 Downto 0);
                                When Others             =>  FIFO_TX_DATA <= DAT_I(31 Downto 8);
                            End Case;
                            FIFO_TX_WRITE_ENABLE <= '1';

                        When Others => Null;
                    End Case;
                End If;
            End If;
        End If;
    End Process;
    -- ------------------------------------------------------------------------
-- ----------------------------------------------------------------------------
End RTL;
-- ----------------------------------------------------------------------------


-- ----------------------------------------------------------------------------
Library IEEE;
Use     IEEE.Std_Logic_1164.All;
Use     IEEE.Std_Logic_Unsigned.All;
Use     IEEE.Numeric_Std.All;
Use     Work.ReplaceWith_EntityName_WB_I2S_Package.All;
-- ----------------------------------------------------------------------------

-- ----------------------------------------------------------------------------
Entity ReplaceWith_EntityName_WB_I2S_TimingController Is
    Port (
        CLK                       : In  Std_Logic;
        RST                       : In  Std_Logic;
        I2S_RUN                   : In  Std_Logic;
        CLOCK_ENABLE_PRESCALER    : In  Std_Logic;
        CLOCK_BIT_DIVIDER         : In  Std_Logic_Vector(3 Downto 0);
        CLOCK_PRESCALER           : In  Std_Logic_Vector(7 Downto 0);

        ENABLE_RX                 : Out Std_Logic;
        ENABLE_TX                 : Out Std_Logic;
        RESET_TX                  : Out Std_Logic;
        LOAD_TX                   : Out Std_Logic;
        SAVE_RX                   : Out Std_Logic;
        RX_POINTER                : In  Std_Logic_Vector(ReplaceWith_FifoAddressWidth-1 Downto 0);
        TX_POINTER                : In  Std_Logic_Vector(ReplaceWith_FifoAddressWidth-1 Downto 0);

        I2S_RX_RUN                : In Std_Logic;
        I2S_TX_RUN                : In Std_Logic;
        I2S_RX_WORD_LENGTH        : In Std_Logic_Vector(1 Downto 0);
        I2S_RX_WORD_JUSTIFICATION : In Std_Logic_Vector(1 Downto 0);
        I2S_TX_WORD_LENGTH        : In Std_Logic_Vector(1 Downto 0);
        I2S_TX_WORD_JUSTIFICATION : In Std_Logic_Vector(1 Downto 0);
        WORD_SELECT_TX            : Out Std_Logic;
        WORD_SELECT_RX            : Out Std_Logic;

        MASTER_CLOCK              : Out Std_Logic; -- 256 * LRCLK
        CHANNEL_CLOCK             : Out Std_Logic; -- LRCLK
        BIT_CLOCK                 : Out Std_Logic -- 64 * LRCLK
    );
End;
-- ----------------------------------------------------------------------------

-- ----------------------------------------------------------------------------
Architecture RTL Of ReplaceWith_EntityName_WB_I2S_TimingController Is
-- ----------------------------------------------------------------------------



    Signal MasterClock              : Std_Logic;
    Signal ChannelClock             : Std_Logic;
    Signal BitClock                 : Std_Logic;
    Signal MasterClockEdge          : Std_Logic;
    Signal MasterClockDivider       : Std_Logic_Vector(7 Downto 0);
    Signal ChannelClockDivider      : Std_Logic_Vector(7 Downto 0);
    Signal BitClockDivider          : Std_Logic_Vector(1 Downto 0);
    Signal ChannelClockEdge         : Std_Logic;
    Signal MasterClockFF            : Std_Logic;
    Signal MasterClockEdgeFF        : Std_Logic;
    Signal ChannelClockPulse        : Std_Logic;
    Signal BitClockPulse            : Std_Logic;
    Signal EnableRX                 : Std_Logic;

    Signal BitCounter_RX            : Std_Logic_Vector(5 Downto 0);
    Signal BitCounterIsZero_RX      : Std_Logic;
    Signal BitCounterStart_RX       : Std_Logic_Vector(4 Downto 0);
    Signal BitCounterStartIsZero_RX : Std_Logic;
    Signal BitCounterEnable_RX      : Std_Logic;

    Signal BitCounter_TX            : Std_Logic_Vector(4 Downto 0);
    Signal BitCounterIsZero_TX      : Std_Logic;
    Signal BitCounterStart_TX       : Std_Logic_Vector(4 Downto 0);
    Signal BitCounterStartIsZero_TX : Std_Logic;
    Signal BitCounterEnable_TX      : Std_Logic;

    Signal I2S_SyncronizedRxStart   : Std_Logic;
    Signal I2S_SyncronizedTxStart   : Std_Logic;
    Signal ChannelClockRxSelector   : Std_Logic;
    Signal ChannelClockTxSelector   : Std_Logic;
    Signal SyncronizeRxStart        : Std_Logic;
    Signal SyncronizeTxStart        : Std_Logic;
    Signal LeftChannelLevelRx       : Std_Logic;
    Signal LeftChannelLevelTx       : Std_Logic;
    Signal StartChannelRx           : Std_Logic;
    Signal StartChannelTx           : Std_Logic;
    Signal ChannelClockValidRx      : Std_Logic;
    Signal ChannelClockValidTx      : Std_Logic;
    Signal ChannelStartPointValidRx : Std_Logic;
    Signal ChannelStartPointValidTx : Std_Logic;
    Signal StopClocks               : Std_Logic;
    Signal ChannelClockRunning      : Std_Logic;
    Signal EnableTX                 : Std_Logic;
    Signal LoadTX                   : Std_Logic;
-- ----------------------------------------------------------------------------
Begin
-- ----------------------------------------------------------------------------

    -- ------------------------------------------------------------------------
    MASTER_CLOCK  <= CLK When (I2S_RUN='1' And CLOCK_ENABLE_PRESCALER = '0') Else MasterClockFF;
    CHANNEL_CLOCK <= ChannelClock;
    BIT_CLOCK     <= BitClock;
    MasterClockEdge <= '1'               When (I2S_RUN='1' And CLOCK_ENABLE_PRESCALER='0') Else
                       MasterClockEdgeFF When (I2S_RUN='1' And CLOCK_ENABLE_PRESCALER='1') Else
                       '0';
    WORD_SELECT_TX   <= ChannelClock Xnor LeftChannelLevelTx;
    WORD_SELECT_RX   <= ChannelClock Xnor LeftChannelLevelRx;
    -- ------------------------------------------------------------------------

    -- ------------------------------------------------------------------------
    -- Generate Master Clock MCLK at 256 * Sample rate frequency
    -- ------------------------------------------------------------------------
    MasterClockGenerator:
    Process(CLK)
        Variable Divider : Integer Range 0 To 255;
    Begin
        If Rising_Edge(CLK) Then
            If RST='1' Then
                MasterClock        <= '0';
                MasterClockDivider <= CLOCK_PRESCALER;
                MasterClockEdgeFF  <= '0';
                MasterClockFF      <= '0';
                Divider := 0;
            Else
                If CLOCK_ENABLE_PRESCALER='1' And I2S_RUN='1' Then
                    MasterClockEdgeFF <= '0';
                    Divider := Divider + 1;
                    If Divider = to_integer(unsigned(CLOCK_PRESCALER)) / 2 Then
                        MasterClock <= '1';
                        MasterClockEdgeFF <= '1';
                    End If;
                    If Divider >= to_integer(unsigned(CLOCK_PRESCALER)) Then
                        MasterClock <= '0';
                        Divider := 0;
                        MasterClockEdgeFF <= '1';
                    End If;
                    MasterClockFF <= Not MasterClock;
                Else
                    MasterClock       <= '0';
                    MasterClockEdgeFF <= '0';
                    MasterClockFF     <= '0';
                    MasterClockDivider <= CLOCK_PRESCALER;
                    Divider := 0;
                End If;
            End If;
        End If;
    End Process;
    -- ------------------------------------------------------------------------

    -- ------------------------------------------------------------------------
    -- Left/Right Channel selector - equals Sample Rate frequency
    -- ------------------------------------------------------------------------
    ChannelClockGenerator:
    Process(CLK)
    Begin
        If Rising_Edge(CLK) Then
            If RST='1' Then
                ChannelClock        <= '0';
                ChannelClockDivider <= (Others=>'1');
                ChannelClockEdge    <= '0';
                ChannelClockPulse   <= '0';
                ChannelClockRunning <= '0';
            Else
                If I2S_RUN='1' Then
                    If CLOCK_ENABLE_PRESCALER='0' Or MasterClockEdge='1' Then
                        ChannelClockDivider <= ChannelClockDivider + 1;
                    End If;
                    If ChannelClockDivider = x"00" And MasterClockEdge='1' Then
                        ChannelClock      <= Not ChannelClock;
                        ChannelClockEdge  <= '1';
                        ChannelClockPulse <= '1';
                    Else
                        ChannelClockEdge  <= '0';
                        ChannelClockPulse <= '0';
                    End If;
                Else
                    ChannelClock        <= '0';
                    ChannelClockEdge    <= '0';
                    ChannelClockPulse   <= '0';
                    ChannelClockDivider <= (Others=>'1');
                End If;
                If I2S_RUN='1' Then
                    ChannelClockRunning <= '1';
                ElsIf I2S_RUN='0' And ChannelClockDivider=x"00" And MasterClockEdge='1' Then
                    ChannelClockRunning <= '0';
                End If;
            End If;
        End If;
    End Process;
    -- ------------------------------------------------------------------------

    -- ------------------------------------------------------------------------
    -- Serial Data Clock, 64 times faster than LRCK
    -- ------------------------------------------------------------------------
    BitClockGenerator:
    Process(CLK)
    Begin
        If Rising_Edge(CLK) Then
            If RST='1' Then
                BitClock        <= '1';
                BitClockDivider <= (Others=>'1');
                BitClockPulse   <= '0';
            Else
                If I2S_RUN='1' Then
                    If CLOCK_ENABLE_PRESCALER='0' Or MasterClockEdge='1' Then
                        BitClockDivider <= BitClockDivider + 1;
                    End If;
                    If BitClockDivider=x"0" And MasterClockEdge='1' Then
                        BitClock      <= Not BitClock;
                        BitClockPulse <= '1';
                    Else
                        BitClockPulse <= '0';
                    End If;
                Else
                    BitClockPulse <= '0';
                    BitClock      <= '1';
                    BitClockDivider <= (Others=>'1');
                End If;
            End If;
        End If;
    End Process;
    -- ------------------------------------------------------------------------

    -- ------------------------------------------------------------------------
    -- Depending on mode selected, LRCLK meaning changes.
    -- For standard I2S communication-> LRCLK=0 is left channel, LRCLK=1 is right channel
    -- for left/right justified modes-> LRCLK=1 is left channel, LRCLK=0 is right channel
    -- on top of that we have different pointers for rx and tx, at the start.
    -- ------------------------------------------------------------------------
    LeftChannelLevelRx       <= '0' When I2S_RX_WORD_JUSTIFICATION=cI2S_WordJustification_Normal Else '1';
    LeftChannelLevelTx       <= '0' When I2S_TX_WORD_JUSTIFICATION=cI2S_WordJustification_Normal Else '1';

    StartChannelRx           <= '1' When ((To_Integer(Unsigned(RX_POINTER)) Mod 2) = 0) Else '0';
    StartChannelTx           <= '1' When ((To_Integer(Unsigned(TX_POINTER)) Mod 2) = 0) Else '0';

    ChannelClockValidRx      <= LeftChannelLevelRx  XNor ChannelClock;
    ChannelClockValidTx      <= LeftChannelLevelTx  XNor ChannelClock;

    SyncronizeRxStart <= (StartChannelRx Xnor ChannelClockValidRx);
    SyncronizeTxStart <= (StartChannelTx Xnor ChannelClockValidTx);
    -- ------------------------------------------------------------------------

    -- ------------------------------------------------------------------------
    -- Start on Left Channel, Stop on Right Channel
    -- ------------------------------------------------------------------------
    Process(CLK)
    Begin
        If Rising_Edge(CLK) Then
            If RST='1' Then
                I2S_SyncronizedRxStart <= '0';
                I2S_SyncronizedTxStart <= '0';
                StopClocks             <= '0';
            Else
                If SyncronizeRxStart='1' And ChannelClockPulse='1' Then
                    I2S_SyncronizedRxStart <= I2S_RX_RUN;
                End If;
                If SyncronizeTxStart='1' And ChannelClockPulse='1' Then
                    I2S_SyncronizedTxStart <= I2S_TX_RUN;
                End If;
                StopClocks <= Not I2S_SyncronizedRxStart And Not I2S_SyncronizedTxStart;
            End If;
        End If;
    End Process;
    -- ------------------------------------------------------------------------

    -- ------------------------------------------------------------------------
    -- Transmitter Control
    -- ------------------------------------------------------------------------
    Process(CLK)
    Begin
        If Rising_Edge(CLK) Then
            If RST='1' Or I2S_SyncronizedTxStart='0' Then
                BitCounter_TX <= (Others=>'0');
            Else
                If ChannelClockPulse='1' Then
                    Case I2S_TX_WORD_LENGTH Is
                        When cI2S_WordLength_16 => BitCounter_TX <= cBitCounterTX_16;
                        When cI2S_WordLength_20 => BitCounter_TX <= cBitCounterTX_20;
                        When Others             => BitCounter_TX <= cBitCounterTX_24;
                    End Case;
                Else
                    If BitCounterEnable_TX='1' Then
                        If BitClockPulse='1' And BitCounterIsZero_TX='0' And BitClock='0' Then
                            BitCounter_TX <= BitCounter_TX - 1;
                        End If;
                    End If;
                End If;
            End If;
        End If;
    End Process;
    BitCounterIsZero_TX <= '1' When BitCounter_TX = cBitCounterTXZero Else '0';
    BitCounterEnable_TX <= BitCounterStartIsZero_TX;
    EnableTX            <= BitCounterEnable_TX And BitClockPulse And Not BitClock And Not BitCounterIsZero_TX And I2S_TX_RUN;
    ENABLE_TX           <= EnableTX;
    RESET_TX            <= ((BitCounterisZero_TX And I2S_TX_RUN) And Not EnableTX And Not LoadTx And BitClockPulse) Or RST;
    LoadTx              <= ChannelClockPulse And I2S_TX_RUN And I2S_SyncronizedTxStart;
    LOAD_TX             <= LoadTx;
    -- ------------------------------------------------------------------------

    -- ------------------------------------------------------------------------
    Process(CLK)
    Begin
        If Rising_Edge(CLK) Then
            If RST='1' Or I2S_SyncronizedTxStart='0' Then
                BitCounterStart_TX <= (Others=>'0');
            Else
                If ChannelClockPulse='1' Then
                    Case I2S_TX_WORD_JUSTIFICATION Is
                        When cI2S_WordJustification_Normal => BitCounterStart_TX <= cBitCounterStart_1;
                        When cI2S_WordJustification_Left   => BitCounterStart_TX <= cBitCounterStart_0;
                        --When cI2S_WordJustification_Right  =>
                        When Others =>
                            Case I2S_TX_WORD_LENGTH Is
                                When cI2S_WordLength_16 => BitCounterStart_TX <= cBitCounterStart_16;
                                When cI2S_WordLength_20 => BitCounterStart_TX <= cBitCounterStart_12;
                                When Others             => BitCounterStart_TX <= cBitCounterStart_8;
                            End Case;
                    End Case;
                Else
                    If BitClockPulse='1' And BitClock = '0' And BitCounterStartIsZero_TX='0' Then
                        BitCounterStart_TX <= BitCounterStart_TX - 1;
                    End If;
                End If;
            End If;
        End If;
    End Process;
    BitCounterStartIsZero_TX <= '1' When BitCounterStart_TX = cBitCounterStart_0 Else '0';
    -- ------------------------------------------------------------------------

    -- ------------------------------------------------------------------------
    -- Receiver Control
    -- ------------------------------------------------------------------------
    Process(CLK)
    Begin
        If Rising_Edge(CLK) Then
            If RST='1' Or I2S_SyncronizedRxStart = '0' Then
                BitCounter_RX <= (Others=>'0');
            Else
                If ChannelClockPulse='1' Then
                    Case I2S_RX_WORD_LENGTH Is
                        When cI2S_WordLength_16 => BitCounter_RX <= cBitCounterRX_16;
                        When cI2S_WordLength_20 => BitCounter_RX <= cBitCounterRX_20;
                        When Others             => BitCounter_RX <= cBitCounterRX_24;
                    End Case;
                Else
                    If BitCounterEnable_RX='1' Then
                        If BitClockPulse='1' And BitCounterIsZero_RX='0' And BitClock='1' Then
                            BitCounter_RX <= BitCounter_RX - 1;
                        End If;
                    End If;
                End If;
            End If;
        End If;
    End Process;
    BitCounterIsZero_RX <= '1' When BitCounter_RX = cBitCounterRXZero Else '0';
    BitCounterEnable_RX <= BitCounterStartIsZero_RX;
    ENABLE_RX           <= BitCounterEnable_RX And BitClockPulse And BitClock And Not BitCounterIsZero_RX And I2S_RX_RUN;
    SAVE_RX             <= EnableRx And BitCounterIsZero_RX And I2S_RX_RUN And I2S_SyncronizedRxStart;
    -- ------------------------------------------------------------------------

    -- ------------------------------------------------------------------------
    Process(CLK)
    Begin
        If Rising_Edge(CLK) Then
            If RST='1' Then
                EnableRx <= '0';
            Else
                EnableRx <= BitCounterEnable_RX And BitClockPulse And BitClock And Not BitCounterIsZero_RX;
            End If;
        End If;
    End Process;
    -- ------------------------------------------------------------------------

    -- ------------------------------------------------------------------------
    Process(CLK)
    Begin
        If Rising_Edge(CLK) Then
            If RST='1' Or I2S_SyncronizedRxStart = '0' Then
                BitCounterStart_RX <= (Others=>'0');
            Else
                If ChannelClockPulse='1' Then
                    Case I2S_RX_WORD_JUSTIFICATION Is
                        When cI2S_WordJustification_Normal => BitCounterStart_RX <= cBitCounterStart_1;
                        When cI2S_WordJustification_Left   => BitCounterStart_RX <= cBitCounterStart_0;
                        When Others =>
                            Case I2S_RX_WORD_LENGTH Is
                                When cI2S_WordLength_16 => BitCounterStart_RX <= cBitCounterStart_16;
                                When cI2S_WordLength_20 => BitCounterStart_RX <= cBitCounterStart_12;
                                When Others             => BitCounterStart_RX <= cBitCounterStart_8;
                            End Case;
                    End Case;
                Else
                    If BitClockPulse='1' And BitClock='1' And BitCounterStartIsZero_RX='0' Then
                        BitCounterStart_RX <= BitCounterStart_RX - 1;
                    End If;
                End If;
            End If;
        End If;
    End Process;
    BitCounterStartIsZero_RX <= '1' When BitCounterStart_RX = cBitCounterStart_0 Else '0';
    -- ------------------------------------------------------------------------

-- ----------------------------------------------------------------------------
End RTL;
-- ----------------------------------------------------------------------------




-- ----------------------------------------------------------------------------
Library IEEE;
Use     IEEE.Std_Logic_1164.All;
Use     IEEE.Std_Logic_Unsigned.All;
Use     IEEE.Numeric_Std.All;
Use     Work.ReplaceWith_EntityName_WB_I2S_Package.All;
-- ----------------------------------------------------------------------------

-- ----------------------------------------------------------------------------
Entity ReplaceWith_EntityName_WB_I2S_DualPortFifo Is
Port (
    RESET                  : In  Std_Logic;
    -- Write Port
    WRITE_CLK              : In  Std_Logic;
    WRITE_DATA             : In  Std_Logic_Vector(24 - 1 Downto 0);
    WRITE_ENABLE           : In  Std_Logic;
    WRITE_POINTER          : Out Std_Logic_Vector(ReplaceWith_FifoAddressWidth-1 Downto 0);
    -- Read Port
    READ_CLK               : In  Std_Logic;
    READ_DATA              : Out Std_Logic_Vector(24 - 1 Downto 0);
    READ_ENABLE            : In  Std_Logic;
    READ_POINTER           : Out Std_Logic_Vector(ReplaceWith_FifoAddressWidth-1 Downto 0);
    -- Common Controls
    WISHBONE_RX_SIDE       : In  Std_Logic;
    WISHBONE_TX_SIDE       : In  Std_Logic;
    MONO_MODE              : In  Std_Logic;
    MONO_CHANNEL_SELECT    : In  Std_Logic;
    WORD_SELECT            : In  Std_Logic;
    FIFO_FULL              : Out Std_Logic;
    FIFO_EMPTY             : Out Std_Logic;
    FIFO_WATERMARK_RX      : In  Std_Logic_Vector(ReplaceWith_FifoAddressWidth-1 Downto 0);
    FIFO_WATERMARK_TX      : In  Std_Logic_Vector(ReplaceWith_FifoAddressWidth-1 Downto 0);
    FIFO_WATERMARK_RX_FLAG : Out Std_Logic;
    FIFO_WATERMARK_TX_FLAG : Out Std_Logic
);
End;
-- ----------------------------------------------------------------------------

-- ----------------------------------------------------------------------------
Architecture RTL Of ReplaceWith_EntityName_WB_I2S_DualPortFifo Is
-- ----------------------------------------------------------------------------
--    Type   TFifo Is Array (0 To ReplaceWith_FifoDepth - 1) Of Std_Logic_Vector(24 - 1 Downto 0);
--    Signal Fifo : TFifo;
    -- ------------------------------------------------------------------------
    Component ReplaceWith_MemoryName Is
        Port(
            CLKA  : In  Std_Logic;
            CLKB  : In  Std_Logic;
            WEA   : In  Std_Logic;
            WEB   : In  Std_Logic;
            DINA  : In  Std_Logic_Vector(24 - 1 Downto 0);
            DINB  : In  Std_Logic_Vector(24 - 1 Downto 0);
            DOUTA : Out Std_Logic_Vector(24 - 1 Downto 0);
            DOUTB : Out Std_Logic_Vector(24 - 1 Downto 0);
            ADDRA : In  Std_Logic_Vector(ReplaceWith_FifoAddressWidth-1 Downto 0);
            ADDRB : In  Std_Logic_Vector(ReplaceWith_FifoAddressWidth-1 Downto 0)
        );
    End Component;
    -- ------------------------------------------------------------------------

    Signal ReadPointer      : Integer Range 0 To ReplaceWith_FifoDepth - 1;
    Signal WritePointer     : Integer Range 0 To ReplaceWith_FifoDepth - 1;
    Signal FifoWritePointer : Integer Range 0 To ReplaceWith_FifoDepth - 1;

    Signal ReadAccess      : Std_Logic;
    Signal FifoFull        : Std_Logic;
    Signal FifoEmpty       : Std_Logic;
    Signal WordSelect      : Std_Logic;
    Signal WordSelectFF    : Std_Logic;
    Signal FifoWriteEnable : Std_Logic;
    Signal SignalGND       : Std_Logic;
    Signal SignalDataGND   : Std_Logic_Vector(24 - 1 Downto 0);
    Signal WriteAddress    : Std_Logic_Vector(ReplaceWith_FifoAddressWidth - 1 Downto 0);
    Signal ReadAddress     : Std_Logic_Vector(ReplaceWith_FifoAddressWidth - 1 Downto 0);
    Signal FifoWriteData   : Std_Logic_Vector(24 - 1 Downto 0);
-- ----------------------------------------------------------------------------
Begin
-- ----------------------------------------------------------------------------

    -- ------------------------------------------------------------------------
    FifoFull               <= '1' When WritePointer = ReadPointer                        Else '0';
    FifoEmpty              <= '1' When WritePointer = ((ReadPointer + 1) Mod ReplaceWith_FifoDepth) Else '0';
    FIFO_FULL              <= FifoFull;
    FIFO_EMPTY             <= FifoEmpty;
    FIFO_WATERMARK_RX_FLAG <= '1' When ((ReadPointer - WritePointer) Mod ReplaceWith_FifoDepth) < To_Integer(Unsigned(FIFO_WATERMARK_RX)) Else '0';
    FIFO_WATERMARK_TX_FLAG <= '1' When ((WritePointer - ReadPointer - 1) Mod ReplaceWith_FifoDepth) < To_Integer(Unsigned(FIFO_WATERMARK_TX)) Else '0';
    WRITE_POINTER          <= Std_Logic_Vector(To_Unsigned(WritePointer, ReplaceWith_FifoAddressWidth));
    READ_POINTER           <= Std_Logic_Vector(To_Unsigned(ReadPointer, ReplaceWith_FifoAddressWidth));
    WriteAddress           <= Std_Logic_Vector(To_Unsigned(FifoWritePointer, ReplaceWith_FifoAddressWidth));
    ReadAddress            <= Std_Logic_Vector(To_Unsigned(ReadPointer, ReplaceWith_FifoAddressWidth));
    -- ------------------------------------------------------------------------

    -- ------------------------------------------------------------------------
    Process(WRITE_CLK)
    Begin
        If Rising_Edge(WRITE_CLK) Then
            If RESET='1' Then
                WritePointer     <= 0;
                FifoWritePointer <= 0;
                FifoWriteEnable  <= '0';
                FifoWriteData    <= (Others=>'0');
            Else
                FifoWriteEnable  <= '0';
                FifoWritePointer <= WritePointer;
                If WRITE_ENABLE='1' Then
                    If FifoFull='0' Then
                        If WISHBONE_RX_SIDE='1' Then
                            If MONO_MODE='0' Or (MONO_CHANNEL_SELECT='1' Xor WORD_SELECT='1') Then
                                FifoWriteEnable <= '1';
                                FifoWriteData   <= WRITE_DATA;
--                                If (MONO_MODE='1' Or (WORD_SELECT='1' Xor ((WritePointer Mod 2) = 1))) Then
                                If (MONO_MODE='1' Or (WORD_SELECT='1' Xor ((WritePointer Mod 2) = 0))) Then
                                    WritePointer <= (WritePointer + 1) Mod ReplaceWith_FifoDepth;
                                End If;
                            End If;
                        Else
                        FifoWriteEnable <= '1';
                        FifoWriteData <= WRITE_DATA;
                        WritePointer <= (WritePointer + 1) Mod ReplaceWith_FifoDepth;
                        End If;
                    End If;
                End If;
            End If;
        End If;
    End Process;
    -- ------------------------------------------------------------------------

    -- ------------------------------------------------------------------------
    Process(READ_CLK)
    Begin
        If Rising_Edge(READ_CLK) Then
            If RESET='1' Then
                ReadPointer <= ReplaceWith_FifoDepth - 1;
                ReadAccess  <= '0';
                WordSelect  <= '0';
            Else
                ReadAccess  <= READ_ENABLE;
                WordSelect  <= WORD_SELECT;
                If READ_ENABLE = '1' Then
                    If WISHBONE_TX_SIDE='1' Then
                        If MONO_MODE='0' And (WordSelect='0' Xor (ReadPointer Mod 2) = 0) Then
                            If  FifoEmpty='0' Then
                                ReadPointer <= (ReadPointer + 1) Mod ReplaceWith_FifoDepth;
                            Else
                                ReadPointer <= (ReadPointer - 1) Mod ReplaceWith_FifoDepth;
                            End If;
                        Else
                            If FifoEmpty='0' And WordSelect='0' Then
                                ReadPointer <= (ReadPointer + 1) Mod ReplaceWith_FifoDepth;
                            End If;
                        End If;
                    Else
                        If FifoEmpty='0' Then
                            ReadPointer <= (ReadPointer + 1) Mod ReplaceWith_FifoDepth;
                        End If;
                    End If;
                End If;
            End If;
        End If;
    End Process;
    -- ------------------------------------------------------------------------

    SignalGND     <= '0';
    SignalDataGND <= (Others=>'0');
    -- ------------------------------------------------------------------------
    -- PORTA is Write, PORTB is Read
    -- ------------------------------------------------------------------------
    U_FifoMemory : ReplaceWith_MemoryName Port Map (
        CLKA   => WRITE_CLK,
        CLKB   => READ_CLK,
        WEA    => FifoWriteEnable,
        WEB    => SignalGND,
        DINA   => FifoWriteData,
        DINB   => SignalDataGND,
        DOUTA  => Open,
        DOUTB  => READ_DATA,
        ADDRA  => WriteAddress,
        ADDRB  => ReadAddress
    );
    -- ------------------------------------------------------------------------

-- ----------------------------------------------------------------------------
End RTL;
-- ----------------------------------------------------------------------------


-- ----------------------------------------------------------------------------
Library IEEE;
Use     IEEE.Std_Logic_1164.All;
Use     Work.ReplaceWith_EntityName_WB_I2S_Package.All;
-- ----------------------------------------------------------------------------

-- ----------------------------------------------------------------------------
Entity ReplaceWith_EntityName_WB_I2S_ShiftInOutRegister Is
Port (
    CLK      : In  Std_Logic;
    RST      : In  Std_Logic;
    DIN      : In  Std_Logic;
    DOUT     : Out Std_Logic;
    LOAD     : In  Std_Logic;
    ENABLE   : In  Std_Logic;
    DATA_IN  : In  Std_Logic_Vector(24 - 1 Downto 0);
    DATA_OUT : Out Std_Logic_Vector(24 - 1 Downto 0)
);
End;
-- ----------------------------------------------------------------------------

-- ----------------------------------------------------------------------------
Architecture RTL Of ReplaceWith_EntityName_WB_I2S_ShiftInOutRegister Is
-- ----------------------------------------------------------------------------
    Signal ShiftReg : Std_Logic_Vector(24-1 Downto 0);
-- ----------------------------------------------------------------------------
Begin
-- ----------------------------------------------------------------------------

    -- ------------------------------------------------------------------------
    Process(CLK)
    Begin
        If Rising_Edge(CLK) Then
            If RST='1' Then
                ShiftReg <= (Others=>'0');
            Else
                If LOAD='1' Then
                    ShiftReg <= DATA_IN;
                Else
                    If ENABLE='1' Then
                        ShiftReg <= ShiftReg(24-2 Downto 0) & DIN;
                    End If;
                End If;
            End If;
        End If;
    End Process;
    -- ------------------------------------------------------------------------

    -- ------------------------------------------------------------------------
    DOUT     <= ShiftReg(24-1);
    DATA_OUT <= ShiftReg;
    -- ------------------------------------------------------------------------

-- ----------------------------------------------------------------------------
End RTL;
-- ----------------------------------------------------------------------------


-- ----------------------------------------------------------------------------
-- TOP LEVEL NETLIST
-- ----------------------------------------------------------------------------
Library IEEE;
Use     IEEE.Std_Logic_1164.All;
Use     Work.ReplaceWith_EntityName_WB_I2S_Package.All;

-- ----------------------------------------------------------------------------
Entity ReplaceWith_EntityName Is
Port (
    CLK_I    : In  Std_Logic;
    CLK_BASE : In  Std_Logic;
    RST_I    : In  Std_Logic;
    STB_I    : In  Std_Logic;
    CYC_I    : In  Std_Logic;
    DAT_I    : In  Std_Logic_Vector(31 Downto 0);
    DAT_O    : Out Std_Logic_Vector(31 Downto 0);
    ADR_I    : In  Std_Logic_Vector( 2 Downto 0);
    WE_I     : In  Std_Logic;
    INT_O    : Out Std_Logic;
    ACK_O    : Out Std_Logic;
    SDI      : In  Std_Logic;                       --IncludeIf_RXEnabled
    SDO      : Out Std_Logic;                       --IncludeIf_TXEnabled
    WS       : Out Std_Logic;
    ADCK     : Out Std_Logic;
    SCK      : Out Std_Logic
);
End;
-- ----------------------------------------------------------------------------

-- ----------------------------------------------------------------------------
Architecture RTL Of ReplaceWith_EntityName Is
-- ----------------------------------------------------------------------------
    -- ------------------------------------------------------------------------
    Component ReplaceWith_EntityName_WB_I2S_DualPortFifo Is
    Port (
        RESET                  : In  Std_Logic;
        -- Write Port
        WRITE_CLK              : In  Std_Logic;
        WRITE_DATA             : In  Std_Logic_Vector(24 - 1 Downto 0);
        WRITE_ENABLE           : In  Std_Logic;
        WRITE_POINTER          : Out Std_Logic_Vector(ReplaceWith_FifoAddressWidth-1 Downto 0);
        -- Read Port
        READ_CLK               : In  Std_Logic;
        READ_DATA              : Out Std_Logic_Vector(24 - 1 Downto 0);
        READ_ENABLE            : In  Std_Logic;
        READ_POINTER           : Out Std_Logic_Vector(ReplaceWith_FifoAddressWidth-1 Downto 0);
        -- Common Controls
        WISHBONE_RX_SIDE       : In  Std_Logic;
        WISHBONE_TX_SIDE       : In  Std_Logic;
        MONO_MODE              : In  Std_Logic;
        MONO_CHANNEL_SELECT    : In  Std_Logic;
        WORD_SELECT            : In  Std_Logic;
        FIFO_FULL              : Out Std_Logic;
        FIFO_EMPTY             : Out Std_Logic;
        FIFO_WATERMARK_RX      : In  Std_Logic_Vector(ReplaceWith_FifoAddressWidth-1 Downto 0);
        FIFO_WATERMARK_TX      : In  Std_Logic_Vector(ReplaceWith_FifoAddressWidth-1 Downto 0);
        FIFO_WATERMARK_RX_FLAG : Out Std_Logic;
        FIFO_WATERMARK_TX_FLAG : Out Std_Logic
    );
    End Component;
    -- ------------------------------------------------------------------------

    -- ------------------------------------------------------------------------
    Component ReplaceWith_EntityName_WB_I2S_ShiftInOutRegister Is
    Port (
        CLK      : In  Std_Logic;
        RST      : In  Std_Logic;
        DIN      : In  Std_Logic;
        DOUT     : Out Std_Logic;
        LOAD     : In  Std_Logic;
        ENABLE   : In  Std_Logic;
        DATA_IN  : In  Std_Logic_Vector(24 - 1 Downto 0);
        DATA_OUT : Out Std_Logic_Vector(24 - 1 Downto 0)
    );
    End Component;
    -- ------------------------------------------------------------------------

    -- ------------------------------------------------------------------------
    Component ReplaceWith_EntityName_WB_I2S_TimingController Is
        Port (
            CLK                       : In  Std_Logic;
            RST                       : In  Std_Logic;
            I2S_RUN                   : In  Std_Logic;
            CLOCK_ENABLE_PRESCALER    : In  Std_Logic;
            CLOCK_BIT_DIVIDER         : In  Std_Logic_Vector(3 Downto 0);
            CLOCK_PRESCALER           : In  Std_Logic_Vector(7 Downto 0);

            ENABLE_RX                 : Out Std_Logic;
            ENABLE_TX                 : Out Std_Logic;
            RESET_TX                  : Out Std_Logic;
            LOAD_TX                   : Out Std_Logic;
            SAVE_RX                   : Out Std_Logic;
            RX_POINTER                : In  Std_Logic_Vector(ReplaceWith_FifoAddressWidth-1 Downto 0);
            TX_POINTER                : In  Std_Logic_Vector(ReplaceWith_FifoAddressWidth-1 Downto 0);

            I2S_RX_RUN                : In  Std_Logic;
            I2S_TX_RUN                : In  Std_Logic;
            I2S_RX_WORD_LENGTH        : In  Std_Logic_Vector(1 Downto 0);
            I2S_RX_WORD_JUSTIFICATION : In  Std_Logic_Vector(1 Downto 0);
            I2S_TX_WORD_LENGTH        : In  Std_Logic_Vector(1 Downto 0);
            I2S_TX_WORD_JUSTIFICATION : In  Std_Logic_Vector(1 Downto 0);
            WORD_SELECT_TX            : Out Std_Logic;
            WORD_SELECT_RX            : Out Std_Logic;

            MASTER_CLOCK              : Out Std_Logic; -- 256 * LRCLK
            CHANNEL_CLOCK             : Out Std_Logic; -- LRCLK
            BIT_CLOCK                 : Out Std_Logic -- 64 * LRCLK
        );
    End Component;
    -- ------------------------------------------------------------------------

    -- ------------------------------------------------------------------------
    Component ReplaceWith_EntityName_WB_I2S_WishboneInterface Is
        Port (
            -- Wishbone interface
            CLK_I : In  Std_Logic;
            RST_I : In  Std_Logic;
            STB_I : In  Std_Logic;
            CYC_I : In  Std_Logic;
            DAT_I : In  Std_Logic_Vector(31 Downto 0);
            DAT_O : Out Std_Logic_Vector(31 Downto 0);
            WE_I  : In  Std_Logic;
            ADR_I : In  Std_Logic_Vector(2 Downto 0);
            ACK_O : Out Std_Logic;
            INT_O : Out Std_Logic;

            -- FIFO Controls
            FIFO_RX_CLK            : Out Std_Logic;
            FIFO_TX_CLK            : Out Std_Logic;
            FIFO_RX_DATA           : In  Std_Logic_Vector(24 - 1  Downto 0);
            FIFO_TX_DATA           : Out Std_Logic_Vector(24 - 1  Downto 0);
            FIFO_RX_WATERMARK_FLAG : In  Std_Logic;
            FIFO_RX_FULL           : In  Std_Logic;
            FIFO_RX_EMPTY          : In  Std_Logic;
            FIFO_TX_WATERMARK_FLAG : In  Std_Logic;
            FIFO_TX_FULL           : In  Std_Logic;
            FIFO_TX_EMPTY          : In  Std_Logic;
            FIFO_TX_RESET          : Out Std_Logic;
            FIFO_RX_RESET          : Out Std_Logic;
            FIFO_RX_WATERMARK      : Out Std_Logic_Vector(ReplaceWith_FifoAddressWidth-1 Downto 0);
            FIFO_TX_WATERMARK      : Out Std_Logic_Vector(ReplaceWith_FifoAddressWidth-1 Downto 0);

            FIFO_RX_POINTER        : In  Std_Logic_Vector(ReplaceWith_FifoAddressWidth-1 Downto 0);
            FIFO_TX_POINTER        : In  Std_Logic_Vector(ReplaceWith_FifoAddressWidth-1 Downto 0);
            FIFO_RX_WRITE_POINTER  : In  Std_Logic_Vector(ReplaceWith_FifoAddressWidth-1 Downto 0);
            FIFO_TX_READ_POINTER   : In  Std_Logic_Vector(ReplaceWith_FifoAddressWidth-1 Downto 0);

            FIFO_TX_WRITE_ENABLE   : Out Std_Logic;
            FIFO_RX_READ_ENABLE    : Out Std_Logic;

            MONO_MODE_IN           : Out Std_Logic;
            MONO_MODE_OUT          : Out Std_Logic;
            MONO_CHANNEL_SELECT    : Out Std_Logic;
            I2S_RUN                : Out Std_Logic;
            I2S_RX_RUN             : Out Std_Logic;
            I2S_TX_RUN             : Out Std_Logic;
            I2S_WORD_LENGTH        : Out Std_Logic_Vector(1 Downto 0);
            I2S_WORD_JUSTIFICATION : Out Std_Logic_Vector(1 Downto 0);

            -- Clock Generation Controls
            CLOCK_PRESCALER        : Out Std_Logic_Vector(7 Downto 0);
            CLOCK_BIT_DIVIDER      : Out Std_Logic_Vector(3 Downto 0);
            CLOCK_ENABLE_PRESCALER : Out Std_Logic
        );
    End Component;
    -- ------------------------------------------------------------------------

    Signal SignalGnd            : Std_Logic;
    Signal SignalVcc            : Std_Logic;
    Signal SignalGnd4           : Std_Logic_Vector(3 Downto 0);
    Signal SignalGndData        : Std_Logic_Vector(cFifoWidth - 1 Downto 0);

    Signal ClockEnablePrescaler : Std_Logic;
    Signal ClockPrescaler       : Std_Logic_Vector(7 Downto 0);
    Signal ClockBitDivider      : Std_Logic_Vector(3 Downto 0);
    Signal I2SRun               : Std_Logic;
    Signal I2SRxRun             : Std_Logic;
    Signal I2STxRun             : Std_Logic;
    Signal I2SWordLength        : Std_Logic_Vector(1 Downto 0);
    Signal I2SWordJustification : Std_Logic_Vector(1 Downto 0);

    Signal FifoRxClock          : Std_Logic;
    Signal FifoRxData           : Std_Logic_Vector(cFifoWidth - 1 Downto 0);
    Signal FifoRxWatermarkFlag  : Std_Logic;
    Signal FifoRxFull           : Std_Logic;
    Signal FifoRxEmpty          : Std_Logic;
    Signal FifoRxReset          : Std_Logic;
    Signal FifoRxWatermark      : Std_Logic_Vector(cFifoAddressWidth-1 Downto 0);
    Signal FifoRxReadPointer    : Std_Logic_Vector(cFifoAddressWidth-1 Downto 0);
    Signal FifoRxReadEnable     : Std_Logic;
    Signal FifoRxPointer        : Std_Logic_Vector(cFifoAddressWidth-1 Downto 0);

    Signal FifoTxClock          : Std_Logic;
    Signal FifoTxData           : Std_Logic_Vector(cFifoWidth - 1 Downto 0);
    Signal FifoTxWatermarkFlag  : Std_Logic;
    Signal FifoTxFull           : Std_Logic;
    Signal FifoTxEmpty          : Std_Logic;
    Signal FifoTxReset          : Std_Logic;
    Signal FifoTxWatermark      : Std_Logic_Vector(cFifoAddressWidth-1 Downto 0);
    Signal FifoTxPointer        : Std_Logic_Vector(cFifoAddressWidth-1 Downto 0);
    Signal FifoTxWriteEnable    : Std_Logic;

    Signal FifoRx_RXPointer     : Std_Logic_Vector(cFifoAddressWidth-1 Downto 0);
    Signal FifoTx_TXPointer     : Std_Logic_Vector(cFifoAddressWidth-1 Downto 0);

    Signal MonoModeIn           : Std_Logic;
    Signal MonoModeOut          : Std_Logic;
    Signal MonoChannelSelect    : Std_Logic;

    Signal EnableRx : Std_Logic;
    Signal EnableTx : Std_Logic;
    Signal ResetTx  : Std_Logic;
    Signal LoadTx   : Std_Logic;
    Signal SaveRx   : Std_Logic;

    Signal WordSelectTx : Std_Logic;
    Signal WordSelectRx : Std_Logic;

    Signal FifoRxWriteData : Std_Logic_Vector(cFifoWidth - 1 Downto 0);
    Signal FifoTxReadData  : Std_Logic_Vector(cFifoWidth - 1 Downto 0);


-- ----------------------------------------------------------------------------
Begin
-- ----------------------------------------------------------------------------

    -- ------------------------------------------------------------------------
    -- ------------------------------------------------------------------------
    U_TimingController : ReplaceWith_EntityName_WB_I2S_TimingController
    Port Map (
        CLK                       => CLK_BASE,
        RST                       => RST_I,
        I2S_RUN                   => I2SRun,
        CLOCK_ENABLE_PRESCALER    => ClockEnablePrescaler,
        CLOCK_BIT_DIVIDER         => ClockBitDivider,
        CLOCK_PRESCALER           => ClockPrescaler,

        ENABLE_RX                 => EnableRx,
        ENABLE_TX                 => EnableTx,
        RESET_TX                  => ResetTx,
        LOAD_TX                   => LoadTx,
        SAVE_RX                   => SaveRx,
        RX_POINTER                => FifoRx_RXPointer,
        TX_POINTER                => FifoTx_TXPointer,

        I2S_RX_RUN                => I2SRxRun,
        I2S_TX_RUN                => I2STxRun,
        I2S_RX_WORD_LENGTH        => I2SWordLength,
        I2S_RX_WORD_JUSTIFICATION => I2SWordJustification,
        I2S_TX_WORD_LENGTH        => I2SWordLength,
        I2S_TX_WORD_JUSTIFICATION => I2SWordJustification,

        WORD_SELECT_TX            => WordSelectTx,
        WORD_SELECT_RX            => WordSelectRx,

        MASTER_CLOCK              => ADCK,
        CHANNEL_CLOCK             => WS,
        BIT_CLOCK                 => SCK
    );
    -- ------------------------------------------------------------------------

    -- ------------------------------------------------------------------------ --IncludeIf_TXEnabled
    U_ShiftTX : ReplaceWith_EntityName_WB_I2S_ShiftInOutRegister                --IncludeIf_TXEnabled
    Port Map (                                                                  --IncludeIf_TXEnabled
        CLK      => CLK_BASE,                                                   --IncludeIf_TXEnabled
        RST      => RST_I,                                                      --IncludeIf_TXEnabled
        DIN      => SignalGnd,                                                  --IncludeIf_TXEnabled
        DOUT     => SDO,                                                        --IncludeIf_TXEnabled
        LOAD     => LoadTx,                                                     --IncludeIf_TXEnabled
        ENABLE   => EnableTx,                                                   --IncludeIf_TXEnabled
        DATA_IN  => FifoTxReadData,                                             --IncludeIf_TXEnabled
        DATA_OUT => Open                                                        --IncludeIf_TXEnabled
    );                                                                          --IncludeIf_TXEnabled
    -- ------------------------------------------------------------------------ --IncludeIf_TXEnabled

    -- ------------------------------------------------------------------------ --IncludeIf_RXEnabled
    U_ShiftRX : ReplaceWith_EntityName_WB_I2S_ShiftInOutRegister                --IncludeIf_RXEnabled
    Port Map (                                                                  --IncludeIf_RXEnabled
        CLK      => CLK_BASE,                                                   --IncludeIf_RXEnabled
        RST      => RST_I,                                                      --IncludeIf_RXEnabled
        DIN      => SDI,                                                        --IncludeIf_RXEnabled
        DOUT     => Open,                                                       --IncludeIf_RXEnabled
        LOAD     => SaveRx,                                                     --IncludeIf_RXEnabled
        ENABLE   => EnableRx,                                                   --IncludeIf_RXEnabled
        DATA_IN  => SignalGndData,                                              --IncludeIf_RXEnabled
        DATA_OUT => FifoRxWriteData                                             --IncludeIf_RXEnabled
    );                                                                          --IncludeIf_RXEnabled
    -- ------------------------------------------------------------------------ --IncludeIf_RXEnabled

    -- ------------------------------------------------------------------------          --IncludeIf_TXEnabled
    U_FifoTX : ReplaceWith_EntityName_WB_I2S_DualPortFifo                                  --IncludeIf_TXEnabled
    Port Map (                                                                           --IncludeIf_TXEnabled
        RESET                  => FifoTxReset,                                           --IncludeIf_TXEnabled
        -- Write Port                                                                    --IncludeIf_TXEnabled
        WRITE_CLK              => FifoTxClock,                                           --IncludeIf_TXEnabled
        WRITE_DATA             => FifoTxData,                                            --IncludeIf_TXEnabled
        WRITE_ENABLE           => FifoTxWriteEnable,                                     --IncludeIf_TXEnabled
        WRITE_POINTER          => FifoTxPointer,                                         --IncludeIf_TXEnabled
        -- Read Port                                                                     --IncludeIf_TXEnabled
        READ_CLK               => CLK_BASE,                                              --IncludeIf_TXEnabled
        READ_DATA              => FifoTxReadData,                                        --IncludeIf_TXEnabled
        READ_ENABLE            => LoadTx,                                                --IncludeIf_TXEnabled
        READ_POINTER           => FifoTx_TXPointer,                                      --IncludeIf_TXEnabled
        -- Common Controls                                                               --IncludeIf_TXEnabled
        WISHBONE_RX_SIDE       => SignalGnd,                                             --IncludeIf_TXEnabled
        WISHBONE_TX_SIDE       => SignalVcc,                                             --IncludeIf_TXEnabled
        MONO_MODE              => MonoModeOut,                                           --IncludeIf_TXEnabled
        MONO_CHANNEL_SELECT    => MonoChannelSelect,                                     --IncludeIf_TXEnabled
        WORD_SELECT            => WordSelectTx,                                          --IncludeIf_TXEnabled
        FIFO_FULL              => FifoTxFull,                                            --IncludeIf_TXEnabled
        FIFO_EMPTY             => FifoTxEmpty,                                           --IncludeIf_TXEnabled
        FIFO_WATERMARK_RX      => FifoTxWatermark,                                       --IncludeIf_TXEnabled
        FIFO_WATERMARK_TX      => FifoTxWatermark,                                       --IncludeIf_TXEnabled
        FIFO_WATERMARK_RX_FLAG => Open,                                                  --IncludeIf_TXEnabled
        FIFO_WATERMARK_TX_FLAG => FifoTxWatermarkFlag                                    --IncludeIf_TXEnabled
    );                                                                                   --IncludeIf_TXEnabled
    -- ------------------------------------------------------------------------          --IncludeIf_TXEnabled
    FifoTxPointer       <= (Others=>'0'); --IncludeIf_TXDisabled
    FifoTxWatermarkFlag <= '0';           --IncludeIf_TXDisabled
    FifoTxFull          <= '0';           --IncludeIf_TXDisabled
    FifoTxEmpty         <= '0';           --IncludeIf_TXDisabled

    -- ------------------------------------------------------------------------          --IncludeIf_RXEnabled
    U_FifoRX : ReplaceWith_EntityName_WB_I2S_DualPortFifo                                  --IncludeIf_RXEnabled
    Port Map (                                                                           --IncludeIf_RXEnabled
        RESET                  => FifoRxReset,                                           --IncludeIf_RXEnabled
        -- Write Port                                                                    --IncludeIf_RXEnabled
        WRITE_CLK              => CLK_BASE,                                              --IncludeIf_RXEnabled
        WRITE_DATA             => FifoRxWriteData,                                       --IncludeIf_RXEnabled
        WRITE_ENABLE           => SaveRx,                                                --IncludeIf_RXEnabled
        WRITE_POINTER          => FifoRx_RXPointer,                                      --IncludeIf_RXEnabled
        -- Read Port                                                                     --IncludeIf_RXEnabled
        READ_CLK               => FifoRxClock,                                           --IncludeIf_RXEnabled
        READ_DATA              => FifoRxData,                                            --IncludeIf_RXEnabled
        READ_ENABLE            => FifoRxReadEnable,                                      --IncludeIf_RXEnabled
        READ_POINTER           => FifoRxPointer,                                         --IncludeIf_RXEnabled
        -- Common Controls                                                               --IncludeIf_RXEnabled
        WISHBONE_RX_SIDE       => SignalVcc,                                             --IncludeIf_RXEnabled
        WISHBONE_TX_SIDE       => SignalGnd,                                             --IncludeIf_RXEnabled
        MONO_MODE              => MonoModeIn,                                            --IncludeIf_RXEnabled
        MONO_CHANNEL_SELECT    => MonoChannelSelect,                                     --IncludeIf_RXEnabled
        WORD_SELECT            => WordSelectRx,                                          --IncludeIf_RXEnabled
        FIFO_FULL              => FifoRxFull,                                            --IncludeIf_RXEnabled
        FIFO_EMPTY             => FifoRxEmpty,                                           --IncludeIf_RXEnabled
        FIFO_WATERMARK_RX      => FifoRxWatermark,                                       --IncludeIf_RXEnabled
        FIFO_WATERMARK_TX      => FifoRxWatermark,                                       --IncludeIf_RXEnabled
        FIFO_WATERMARK_RX_FLAG => FifoRxWatermarkFlag,                                   --IncludeIf_RXEnabled
        FIFO_WATERMARK_TX_FLAG => Open                                                   --IncludeIf_RXEnabled
    );                                                                                   --IncludeIf_RXEnabled
    -- ------------------------------------------------------------------------          --IncludeIf_RXEnabled
    FifoRxData          <= (Others=>'0');  --IncludeIf_RxDisabled
    FifoRxPointer       <= (Others=>'0');  --IncludeIf_RxDisabled
    FifoRxWatermarkFlag <= '0';            --IncludeIf_RxDisabled
    FifoRxFull          <= '0';            --IncludeIf_RxDisabled
    FifoRxEmpty         <= '0';            --IncludeIf_RxDisabled

    -- ------------------------------------------------------------------------
    U_WishboneInterface : ReplaceWith_EntityName_WB_I2S_WishboneInterface
    Port Map (
        -- Wishbone interface
        CLK_I                  => CLK_I,
        RST_I                  => RST_I,
        STB_I                  => STB_I,
        CYC_I                  => CYC_I,
        DAT_I                  => DAT_I,
        DAT_O                  => DAT_O,
        WE_I                   => WE_I,
        INT_O                  => INT_O,
        ADR_I                  => ADR_I,
        ACK_O                  => ACK_O,

        -- FIFO Controls
        FIFO_RX_CLK            => FifoRxClock,
        FIFO_RX_DATA           => FifoRxData,
        FIFO_RX_WATERMARK_FLAG => FifoRxWatermarkFlag,
        FIFO_RX_FULL           => FifoRxFull,
        FIFO_RX_EMPTY          => FifoRxEmpty,
        FIFO_RX_RESET          => FifoRxReset,
        FIFO_RX_WATERMARK      => FifoRxWatermark,
        FIFO_RX_POINTER        => FifoRxPointer,
        FIFO_RX_WRITE_POINTER  => FifoRx_RXPointer,
        FIFO_RX_READ_ENABLE    => FifoRxReadEnable,

        FIFO_TX_CLK            => FifoTxClock,
        FIFO_TX_DATA           => FifoTxData,
        FIFO_TX_WATERMARK_FLAG => FifoTxWatermarkFlag,
        FIFO_TX_FULL           => FifoTxFull,
        FIFO_TX_EMPTY          => FifoTxEmpty,
        FIFO_TX_RESET          => FifoTxReset,
        FIFO_TX_WATERMARK      => FifoTxWatermark,
        FIFO_TX_POINTER        => FifoTxPointer,
        FIFO_TX_READ_POINTER  =>  FifoTx_TXPointer,
        FIFO_TX_WRITE_ENABLE   => FifoTxWriteEnable,

        MONO_MODE_IN           => MonoModeIn,
        MONO_MODE_OUT          => MonoModeOut,
        MONO_CHANNEL_SELECT    => MonoChannelSelect,

        I2S_RUN                => I2SRun,
        I2S_RX_RUN             => I2SRxRun,
        I2S_TX_RUN             => I2STxRun,
        I2S_WORD_LENGTH        => I2SWordLength,
        I2S_WORD_JUSTIFICATION => I2SWordJustification,

        -- Clock Generation Controls
        CLOCK_PRESCALER        => ClockPrescaler,
        CLOCK_BIT_DIVIDER      => ClockBitDivider,
        CLOCK_ENABLE_PRESCALER => ClockEnablePrescaler
    );
    -- ------------------------------------------------------------------------

    SignalGnd     <= '0';
    SignalVcc     <= '1';
    SignalGnd4    <= (Others=>'0');
    SignalGndData <= (Others=>'0');


-- ----------------------------------------------------------------------------
End RTL;
-- ----------------------------------------------------------------------------




