------------------------------------------------------------------------------------
Library IEEE;
Use     IEEE.Std_Logic_1164.All;
Use     IEEE.Std_Logic_Unsigned.All;
------------------------------------------------------------------------------------

------------------------------------------------------------------------------------
Entity Configurable_ReplaceWith_Designator Is
   Port (
        ----------------------------------------------------------------------------
        CLK_I          : In    Std_Logic;
        RST_I          : In    Std_Logic;
        INT_I          : In    Std_Logic_Vector(31 DownTo 0);
        ----------------------------------------------------------------------------

        ----------------------------------------------------------------------------
        -- Leave these Out if not including OCDS Unit
        TCK            : In    Std_Logic;                         -- IncludeIf_OCDS
        TDI            : In    Std_Logic;                         -- IncludeIf_OCDS
        TDO            : Out   Std_Logic;                         -- IncludeIf_OCDS
        TMS            : In    Std_Logic;                         -- IncludeIf_OCDS
        TRST           : In    Std_Logic;                         -- IncludeIf_OCDS
        ----------------------------------------------------------------------------

        ----------------------------------------------------------------------------
        IO_ACK_I       : In    Std_Logic;
        IO_ADR_O       : Out   Std_Logic_Vector(23 DownTo 0);
        IO_CYC_O       : Out   Std_Logic;
        IO_DAT_I       : In    Std_Logic_Vector(31 DownTo 0);
        IO_DAT_O       : Out   Std_Logic_Vector(31 DownTo 0);
        IO_SEL_O       : Out   Std_Logic_Vector( 3 DownTo 0);
        IO_STB_O       : Out   Std_Logic;
        IO_WE_O        : Out   Std_Logic;
        IO_CLK_O       : Out   Std_Logic;
        IO_RST_O       : Out   Std_Logic;
        ----------------------------------------------------------------------------

        ----------------------------------------------------------------------------
        ME_ACK_I       : In    Std_Logic;
        ME_ADR_O       : Out   Std_Logic_Vector(31 DownTo 0);
        ME_CYC_O       : Out   Std_Logic;
        ME_DAT_I       : In    Std_Logic_Vector(31 DownTo 0);
        ME_DAT_O       : Out   Std_Logic_Vector(31 DownTo 0);
        ME_SEL_O       : Out   Std_Logic_Vector( 3 DownTo 0);
        ME_STB_O       : Out   Std_Logic;
        ME_WE_O        : Out   Std_Logic;
        ME_CLK_O       : Out   Std_Logic;
        ME_RST_O       : Out   Std_Logic;
        ----------------------------------------------------------------------------

        ----------------------------------------------------------------------------
        -- For Debugging
        ----------------------------------------------------------------------------
        LAX_CPU_I_ReadData        : Out Std_Logic_Vector(31 Downto 0);
        LAX_CPU_I_WaitRequest     : Out Std_Logic;
        LAX_CPU_I_Address         : Out Std_Logic_Vector(31 Downto 0);
        LAX_CPU_I_Read            : Out Std_Logic;
        LAX_CPU_I_ReadDataValid   : Out Std_Logic;
        LAX_CPU_D_Address         : Out Std_Logic_Vector(31 Downto 0);
        LAX_CPU_D_ByteEnable      : Out Std_Logic_Vector(3  Downto 0);
        LAX_CPU_D_Read            : Out Std_Logic;
        LAX_CPU_D_Write           : Out Std_Logic;
        LAX_CPU_D_ReadData        : Out Std_Logic_Vector(31 Downto 0);
        LAX_CPU_D_WriteData       : Out Std_Logic_Vector(31 Downto 0);
        LAX_CPU_D_WaitRequest     : Out Std_Logic;

        LAX_TAP_Ir_Out            : Out Std_Logic_Vector(7 Downto 0);
        LAX_TAP_Irq               : Out Std_Logic;
        LAX_TAP_TDO               : Out Std_Logic;
        LAX_TAP_Clrn              : Out Std_Logic;
        LAX_TAP_Ena               : Out Std_Logic;
        LAX_TAP_Ir_In             : Out Std_Logic_Vector(7 Downto 0);
        LAX_TAP_Raw_TCK           : Out Std_Logic;
        LAX_TAP_Rti               : Out Std_Logic;
        LAX_TAP_Shift             : Out Std_Logic;
        LAX_TAP_TDI               : Out Std_Logic;
        LAX_TAP_Update            : Out Std_Logic;
        LAX_TAP_Usr               : Out Std_Logic;

        LAX_MainMemory_I_Wait     : Out Std_Logic;
        LAX_MainMemory_D_Wait     : Out Std_Logic;
        LAX_Peripheral_I_Wait     : Out Std_Logic;
        LAX_Peripheral_D_Wait     : Out Std_Logic;
        LAX_InternalMem_I_Wait    : Out Std_Logic;
        LAX_InternalMem_D_Wait    : Out Std_Logic;
        LAX_DebugMem_I_Wait       : Out Std_Logic;
        LAX_DebugMem_D_Wait       : Out Std_Logic;
        LAX_Debug_State_IO        : Out Std_Logic_Vector(1 Downto 0);
        LAX_Debug_StateNext_IO    : Out Std_Logic_Vector(1 Downto 0);
        LAX_Debug_Access_IO_Data  : Out Std_Logic;
        LAX_Debug_Access_IO_Instr : Out Std_Logic
        ----------------------------------------------------------------------------
   );
End Configurable_ReplaceWith_Designator;
------------------------------------------------------------------------------------


------------------------------------------------------------------------------------
Architecture RTL Of Configurable_ReplaceWith_Designator Is
------------------------------------------------------------------------------------
    Component Nios2_OCDS
        Port (
            Altium_TCK     : In  Std_Logic;
            Altium_TDI     : In  Std_Logic;
            Altium_TMS     : In  Std_Logic;
            Altium_TRST    : In  Std_Logic;
            Altium_TDO     : Out Std_Logic;

            Reset_n        : In  Std_Logic;

            Debug_Ir_Out   : In  Std_Logic_Vector(7 Downto 0);
            Debug_Irq      : In  Std_Logic;
            Debug_TDO      : In  Std_Logic;
            Debug_Clrn     : Out Std_Logic;
            Debug_Ena      : Out Std_Logic;
            Debug_Ir_In    : Out Std_Logic_Vector(7 Downto 0);
            Debug_Raw_TCK  : Out Std_Logic;
            Debug_Rti      : Out Std_Logic;
            Debug_Shift    : Out Std_Logic;
            Debug_TDI      : Out Std_Logic;
            Debug_Update   : Out Std_Logic;
            Debug_Usr1     : Out Std_Logic
        );
    End Component;

    Component ReplaceWith_Designator_Nios2_Template
        Port (
            --------------------------------------------------------------------
            CLK_I                     : In  Std_Logic;
            RST_I                     : In  Std_Logic;
            INT_I                     : In  Std_Logic_Vector(31 Downto 0);
            --------------------------------------------------------------------

            --------------------------------------------------------------------
            IO_STB_O                  : Out Std_Logic;
            IO_CYC_O                  : Out Std_Logic;
            IO_ACK_I                  : In  Std_Logic;
            IO_ADR_O                  : Out Std_Logic_Vector(23 Downto 0);
            IO_DAT_I                  : In  Std_Logic_Vector(31 Downto 0);
            IO_DAT_O                  : Out Std_Logic_Vector(31 Downto 0);
            IO_SEL_O                  : Out Std_Logic_Vector(3  Downto 0);
            IO_WE_O                   : Out Std_Logic;
            IO_CLK_O                  : Out Std_Logic;
            IO_RST_O                  : Out Std_Logic;
            --------------------------------------------------------------------

            --------------------------------------------------------------------
            DataRamIn                 : In  Std_Logic_Vector(31 Downto 0);
            DataRamOut                : Out Std_Logic_Vector(31 Downto 0);
            DataRamBE                 : Out Std_Logic_Vector(3 Downto 0);
            DataRamWE                 : Out Std_Logic;
            DataRamAddr               : Out Std_Logic_Vector(31 Downto 0);
            InstrRomIn                : In  Std_Logic_Vector(31 Downto 0);
            InstrRomAddr              : Out Std_Logic_Vector(31 Downto 0);
            --------------------------------------------------------------------

            --------------------------------------------------------------------
            ME_STB_O                  : Out Std_Logic;
            ME_CYC_O                  : Out Std_Logic;
            ME_ACK_I                  : In  Std_Logic;
            ME_ADR_O                  : Out Std_Logic_Vector(31 Downto 0);
            ME_DAT_I                  : In  Std_Logic_Vector(31 Downto 0);
            ME_DAT_O                  : Out Std_Logic_Vector(31 Downto 0);
            ME_SEL_O                  : Out Std_Logic_Vector(3  Downto 0);
            ME_WE_O                   : Out Std_Logic;
            ME_CLK_O                  : Out Std_Logic;
            ME_RST_O                  : Out Std_Logic;
            --------------------------------------------------------------------

            --------------------------------------------------------------------
            Debug_Ir_Out              : Out Std_Logic_Vector(7 Downto 0);
            Debug_Irq                 : Out Std_Logic;
            Debug_TDO                 : Out Std_Logic;
            Debug_Clrn                : In  Std_Logic;
            Debug_Ena                 : In  Std_Logic;
            Debug_Ir_In               : In  Std_Logic_Vector(7 Downto 0);
            Debug_Raw_TCK             : In  Std_Logic;
            Debug_Rti                 : In  Std_Logic;
            Debug_Shift               : In  Std_Logic;
            Debug_TDI                 : In  Std_Logic;
            Debug_Update              : In  Std_Logic;
            Debug_Usr1                : In  Std_Logic;
            --------------------------------------------------------------------

            --------------------------------------------------------------------
            TimeBaseIn                : In  Std_Logic_Vector(31 Downto 0);
            TimeBaseOut               : Out Std_Logic_Vector(31 Downto 0);
            TimeBaseAddr              : Out Std_Logic_Vector(1 Downto 0);
            TimeBaseWE                : Out Std_Logic;
            TimeBaseInterrupt         : In  Std_Logic;
            TimeBaseReset             : Out Std_Logic;
            --------------------------------------------------------------------

            --------------------------------------------------------------------
            LAX_CPU_I_ReadData        : Out Std_Logic_Vector(31 Downto 0);
            LAX_CPU_I_WaitRequest     : Out Std_Logic;
            LAX_CPU_I_Address         : Out Std_Logic_Vector(31 Downto 0);
            LAX_CPU_I_Read            : Out Std_Logic;
            LAX_CPU_I_ReadDataValid   : Out Std_Logic;
            LAX_CPU_D_Address         : Out Std_Logic_Vector(31 Downto 0);
            LAX_CPU_D_ByteEnable      : Out Std_Logic_Vector(3  Downto 0);
            LAX_CPU_D_Read            : Out Std_Logic;
            LAX_CPU_D_Write           : Out Std_Logic;
            LAX_CPU_D_ReadData        : Out Std_Logic_Vector(31 Downto 0);
            LAX_CPU_D_WriteData       : Out Std_Logic_Vector(31 Downto 0);
            LAX_CPU_D_WaitRequest     : Out Std_Logic;

            Debug_MainMemory_I_Wait   : Out Std_Logic;
            Debug_MainMemory_D_Wait   : Out Std_Logic;
            Debug_Peripheral_I_Wait   : Out Std_Logic;
            Debug_Peripheral_D_Wait   : Out Std_Logic;
            Debug_InternalMem_I_Wait  : Out Std_Logic;
            Debug_InternalMem_D_Wait  : Out Std_Logic;
            Debug_DebugMem_I_Wait     : Out Std_Logic;
            Debug_DebugMem_D_Wait     : Out Std_Logic;
            Debug_State_IO            : Out Std_Logic_Vector(1 Downto 0);
            Debug_StateNext_IO        : Out Std_Logic_Vector(1 Downto 0);
            Debug_Access_IO_Data      : Out Std_Logic;
            Debug_Access_IO_Instr     : Out Std_Logic
            --------------------------------------------------------------------
        );
    End Component;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    Constant  BootAddressWidth : Integer := ReplaceWith_BootAddressWidth;
    Component Memory_ReplaceWith_Designator
        Port
        (
            ADDRA   : In  Std_Logic_Vector(BootAddressWidth-1 DownTo 0);
            ADDRB   : In  Std_Logic_Vector(BootAddressWidth-1 DownTo 0);
            ByteWEA : In  Std_Logic_Vector(3 DownTo 0);
            ByteWEB : In  Std_Logic_Vector(3 DownTo 0);
            CLKA    : In  Std_Logic;
            CLKB    : In  Std_Logic;
            DINA    : In  Std_Logic_Vector(31 DownTo 0);
            DINB    : In  Std_Logic_Vector(31 DownTo 0);
            DOUTA   : Out Std_Logic_Vector(31 DownTo 0);
            DOUTB   : Out Std_Logic_Vector(31 DownTo 0);
            ENA     : In  Std_Logic;
            ENB     : In  Std_Logic;
            WEA     : In  Std_Logic;
            WEB     : In  Std_Logic
        );
    End Component;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    Component ReplaceWith_Designator_Nios2_TimeBase
        Port (
            -- Global signals
            CLK_I : In Std_Logic;
            RST_I : In Std_Logic;

            DAT_I : In  Std_Logic_Vector(31 Downto 0);
            DAT_O : Out Std_Logic_Vector(31 Downto 0);
            ADR_I : In  Std_Logic_Vector(1 Downto 0);
            WE_I  : In  Std_Logic;

            INT_O : Out Std_Logic
        );
    End Component;
    ----------------------------------------------------------------------------

    Signal Signal_GND4                             : Std_Logic_Vector(3 Downto 0);
    Signal Signal_GND                              : Std_Logic;
    Signal Signal_VCC                              : Std_Logic;

    Signal Boot_Reset_Counter                      : Std_Logic_Vector(7 Downto 0);

    Signal PinSignal_SubPart_LowRAM_ADDRA          : STD_LOGIC_VECTOR(31 downto 0);
    Signal PinSignal_SubPart_LowRAM_ADDRB          : STD_LOGIC_VECTOR(31 downto 0);
    Signal PinSignal_SubPart_LowRAM_ByteWEA        : STD_LOGIC_VECTOR(3 downto 0);
    Signal PinSignal_SubPart_LowRAM_ByteWEB        : STD_LOGIC_VECTOR(3 downto 0);
    Signal PinSignal_SubPart_LowRAM_DINA           : STD_LOGIC_VECTOR(31 downto 0);
    Signal PinSignal_SubPart_LowRAM_DINB           : STD_LOGIC_VECTOR(31 downto 0);
    Signal PinSignal_SubPart_LowRAM_ENA            : STD_LOGIC;
    Signal PinSignal_SubPart_LowRAM_ENB            : STD_LOGIC;
    Signal PinSignal_SubPart_LowRAM_LOWRAM_DO      : STD_LOGIC_VECTOR(31 downto 0);
    Signal PinSignal_SubPart_LowRAM_WEA            : STD_LOGIC;
    Signal PinSignal_SubPart_LowRAM_WEB            : STD_LOGIC;
    Signal PinSignal_SubPart_LowRAM_DOUTA          : STD_LOGIC_VECTOR(31 downto 0);
    Signal PinSignal_SubPart_LowRAM_DOUTB          : STD_LOGIC_VECTOR(31 downto 0);

    Signal PinSignal_SubPart_Nios2_TimeBase_DAT_I  : Std_Logic_Vector(31 Downto 0);
    Signal PinSignal_SubPart_Nios2_TimeBase_DAT_O  : Std_Logic_Vector(31 Downto 0);
    Signal PinSignal_SubPart_Nios2_TimeBase_ADR_I  : Std_Logic_Vector(1 Downto 0);
    Signal PinSignal_SubPart_Nios2_TimeBase_WE_I   : Std_Logic;
    Signal PinSignal_SubPart_Nios2_TimeBase_INT_O  : Std_Logic;
    Signal PinSignal_SubPart_Nios2_TimeBase_RST_I  : Std_Logic;

    ----------------------------------------------------------------------------
    -- For Debugging
    ----------------------------------------------------------------------------
    Signal Debug_Ir_Out_Signal                     : Std_Logic_Vector(7 Downto 0);
    Signal Debug_Irq_Signal                        : Std_Logic;
    Signal Debug_TDO_Signal                        : Std_Logic;
    Signal Debug_Clrn_Signal                       : Std_Logic;
    Signal Debug_Ena_Signal                        : Std_Logic;
    Signal Debug_Ir_In_Signal                      : Std_Logic_Vector(7 Downto 0);
    Signal Debug_Raw_TCK_Signal                    : Std_Logic;
    Signal Debug_Rti_Signal                        : Std_Logic;
    Signal Debug_Shift_Signal                      : Std_Logic;
    Signal Debug_TDI_Signal                        : Std_Logic;
    Signal Debug_Update_Signal                     : Std_Logic;
    Signal Debug_Usr1_Signal                       : Std_Logic;
    Signal Signal_Rst                              : Std_Logic;

    Signal Debug_MainMemory_I_Wait                 : Std_Logic;
    Signal Debug_MainMemory_D_Wait                 : Std_Logic;
    Signal Debug_Peripheral_I_Wait                 : Std_Logic;
    Signal Debug_Peripheral_D_Wait                 : Std_Logic;
    Signal Debug_InternalMem_I_Wait                : Std_Logic;
    Signal Debug_InternalMem_D_Wait                : Std_Logic;
    Signal Debug_DebugMem_I_Wait                   : Std_Logic;
    Signal Debug_DebugMem_D_Wait                   : Std_Logic;
    Signal Debug_State_IO                          : Std_Logic_Vector(1 Downto 0)   ;
    Signal Debug_StateNext_IO                      : Std_Logic_Vector(1 Downto 0)   ;
    Signal Debug_Access_IO_Data                    : Std_Logic;
    Signal Debug_Access_IO_Instr                   : Std_Logic;
    ----------------------------------------------------------------------------
Begin

    ----------------------------------------------------------------------------
    LAX_TAP_Ir_Out                  <= Debug_Ir_Out_Signal;
    LAX_TAP_Irq                     <= Debug_Irq_Signal;
    LAX_TAP_TDO                     <= Debug_TDO_Signal;
    LAX_TAP_Clrn                    <= Debug_Clrn_Signal;
    LAX_TAP_Ena                     <= Debug_Ena_Signal;
    LAX_TAP_Ir_In                   <= Debug_Ir_In_Signal;
    LAX_TAP_Raw_TCK                 <= Debug_Raw_TCK_Signal;
    LAX_TAP_Rti                     <= Debug_Rti_Signal;
    LAX_TAP_Shift                   <= Debug_Shift_Signal;
    LAX_TAP_TDI                     <= Debug_TDI_Signal;
    LAX_TAP_Update                  <= Debug_Update_Signal;
    LAX_TAP_Usr                     <= Debug_Usr1_Signal;


    LAX_MainMemory_I_Wait           <= Debug_MainMemory_I_Wait ;
    LAX_MainMemory_D_Wait           <= Debug_MainMemory_D_Wait ;
    LAX_Peripheral_I_Wait           <= Debug_Peripheral_I_Wait ;
    LAX_Peripheral_D_Wait           <= Debug_Peripheral_D_Wait ;
    LAX_InternalMem_I_Wait          <= Debug_InternalMem_I_Wait;
    LAX_InternalMem_D_Wait          <= Debug_InternalMem_D_Wait;
    LAX_DebugMem_I_Wait             <= Debug_DebugMem_I_Wait   ;
    LAX_DebugMem_D_Wait             <= Debug_DebugMem_D_Wait   ;
    LAX_Debug_State_IO              <= Debug_State_IO;
    LAX_Debug_StateNext_IO          <= Debug_StateNext_IO;
    LAX_Debug_Access_IO_Data        <= Debug_Access_IO_Data;
    LAX_Debug_Access_IO_Instr       <= Debug_Access_IO_Instr;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    Signal_GND4                     <= (Others=>'0');
    Signal_GND                      <= '0';
    Signal_VCC                      <= '1';
    Signal_Rst                      <= Not Boot_Reset_Counter(7);

    PinSignal_SubPart_LowRAM_ByteWEA <= Signal_GND4;
    PinSignal_SubPart_LowRAM_WEA     <= Signal_GND;
    PinSignal_SubPart_LowRAM_ENA     <= Signal_VCC;
    PinSignal_SubPart_LowRAM_ENB     <= Signal_VCC;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    SubPart_OCDS : Nios2_OCDS
    Port Map (
        Altium_TCK     => TCK,
        Altium_TDI     => TDI,
        Altium_TMS     => TMS,
        Altium_TRST    => TRST,
        Altium_TDO     => TDO,

        Reset_n        => Signal_VCC,

        Debug_Ir_Out   => Debug_Ir_Out_Signal,
        Debug_Irq      => Debug_Irq_Signal,
        Debug_TDO      => Debug_TDO_Signal,
        Debug_Clrn     => Debug_Clrn_Signal,
        Debug_Ena      => Debug_Ena_Signal,
        Debug_Ir_In    => Debug_Ir_In_Signal,
        Debug_Raw_TCK  => Debug_Raw_TCK_Signal,
        Debug_Rti      => Debug_Rti_Signal,
        Debug_Shift    => Debug_Shift_Signal,
        Debug_TDI      => Debug_TDI_Signal,
        Debug_Update   => Debug_Update_Signal,
        Debug_Usr1     => Debug_Usr1_Signal
    );
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    SubPart_NIOS2 : ReplaceWith_Designator_Nios2_Template
    Port Map (
        CLK_I         => CLK_I,
        RST_I         => Signal_Rst,
        INT_I         => INT_I,

        IO_STB_O      => IO_STB_O,
        IO_CYC_O      => IO_CYC_O,
        IO_ACK_I      => IO_ACK_I,
        IO_ADR_O      => IO_ADR_O,
        IO_DAT_I      => IO_DAT_I,
        IO_DAT_O      => IO_DAT_O,
        IO_SEL_O      => IO_SEL_O,
        IO_WE_O       => IO_WE_O ,
        IO_CLK_O      => IO_CLK_O,
        IO_RST_O      => IO_RST_O,

        ME_STB_O      => ME_STB_O,
        ME_CYC_O      => ME_CYC_O,
        ME_ACK_I      => ME_ACK_I,
        ME_ADR_O      => ME_ADR_O,
        ME_DAT_I      => ME_DAT_I,
        ME_DAT_O      => ME_DAT_O,
        ME_SEL_O      => ME_SEL_O,
        ME_WE_O       => ME_WE_O ,
        ME_CLK_O      => ME_CLK_O,
        ME_RST_O      => ME_RST_O,

        -- RAMA is for read and RAMB is for write
        InstrRomAddr  => PinSignal_SubPart_LowRAM_ADDRA,
        InstrRomIn    => PinSignal_SubPart_LowRAM_DOUTA,
        --DataRamOut  => PinSignal_SubPart_LowRAM_DINA,
        DataRamAddr   => PinSignal_SubPart_LowRAM_ADDRB,
        DataRamBE     => PinSignal_SubPart_LowRAM_ByteWEB,
        DataRamOut    => PinSignal_SubPart_LowRAM_DINB,
        DataRamIn     => PinSignal_SubPart_LowRAM_DOUTB,
        DataRamWE     => PinSignal_SubPart_LowRAM_WEB,

        Debug_Ir_Out  => Debug_Ir_Out_Signal,
        Debug_Irq     => Debug_Irq_Signal,
        Debug_TDO     => Debug_TDO_Signal,
        Debug_Clrn    => Debug_Clrn_Signal,
        Debug_Ena     => Debug_Ena_Signal,
        Debug_Ir_In   => Debug_Ir_In_Signal,
        Debug_Raw_TCK => Debug_Raw_TCK_Signal,
        Debug_Rti     => Debug_Rti_Signal,
        Debug_Shift   => Debug_Shift_Signal,
        Debug_TDI     => Debug_TDI_Signal,
        Debug_Update  => Debug_Update_Signal,
        Debug_Usr1    => Debug_Usr1_Signal,

        TimeBaseIn                => PinSignal_SubPart_Nios2_TimeBase_DAT_O,
        TimeBaseOut               => PinSignal_SubPart_Nios2_TimeBase_DAT_I,
        TimeBaseAddr              => PinSignal_SubPart_Nios2_TimeBase_ADR_I,
        TimeBaseWE                => PinSignal_SubPart_Nios2_TimeBase_WE_I,
        TimeBaseInterrupt         => PinSignal_SubPart_Nios2_TimeBase_INT_O,
        TimeBaseReset             => PinSignal_SubPart_Nios2_TimeBase_RST_I,

        LAX_CPU_I_ReadData        => LAX_CPU_I_ReadData     ,
        LAX_CPU_I_WaitRequest     => LAX_CPU_I_WaitRequest ,
        LAX_CPU_I_Address         => LAX_CPU_I_Address     ,
        LAX_CPU_I_Read            => LAX_CPU_I_Read         ,
        LAX_CPU_I_ReadDataValid   => LAX_CPU_I_ReadDataValid,
        LAX_CPU_D_Address         => LAX_CPU_D_Address     ,
        LAX_CPU_D_ByteEnable      => LAX_CPU_D_ByteEnable  ,
        LAX_CPU_D_Read            => LAX_CPU_D_Read         ,
        LAX_CPU_D_Write           => LAX_CPU_D_Write        ,
        LAX_CPU_D_ReadData        => LAX_CPU_D_ReadData     ,
        LAX_CPU_D_WriteData       => LAX_CPU_D_WriteData    ,
        LAX_CPU_D_WaitRequest     => LAX_CPU_D_WaitRequest  ,

        Debug_MainMemory_I_Wait    => Debug_MainMemory_I_Wait    ,
        Debug_MainMemory_D_Wait    => Debug_MainMemory_D_Wait    ,
        Debug_Peripheral_I_Wait    => Debug_Peripheral_I_Wait     ,
        Debug_Peripheral_D_Wait    => Debug_Peripheral_D_Wait     ,
        Debug_InternalMem_I_Wait   => Debug_InternalMem_I_Wait    ,
        Debug_InternalMem_D_Wait   => Debug_InternalMem_D_Wait    ,
        Debug_DebugMem_I_Wait      => Debug_DebugMem_I_Wait       ,
        Debug_DebugMem_D_Wait      => Debug_DebugMem_D_Wait       ,
        Debug_State_IO             => Debug_State_IO,
        Debug_StateNext_IO         => Debug_StateNext_IO,
        Debug_Access_IO_Data       => Debug_Access_IO_Data,
        Debug_Access_IO_Instr      => Debug_Access_IO_Instr
   );
   ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    SubPart_BootMemory : Memory_ReplaceWith_Designator
    Port Map
    (
        ADDRA      => PinSignal_SubPart_LowRAM_ADDRA(BootAddressWidth+1 DownTo 2),
        ADDRB      => PinSignal_SubPart_LowRAM_ADDRB(BootAddressWidth+1 DownTo 2),
        ByteWEA    => Signal_GND4,
        ByteWEB    => PinSignal_SubPart_LowRAM_ByteWEB,
        DINA       => (Others => '0'),
        DINB       => PinSignal_SubPart_LowRAM_DINB,
        DOUTA      => PinSignal_SubPart_LowRAM_DOUTA,
        DOUTB      => PinSignal_SubPart_LowRAM_DOUTB,
        ENA        => Signal_VCC,
        ENB        => Signal_VCC,
        WEA        => Signal_GND,
        WEB        => PinSignal_SubPart_LowRAM_WEB,
        CLKA       => CLK_I,
        CLKB       => CLK_I
    );
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    Boot_Reset:
    Process(CLK_I,
            RST_I)
    Begin
        If Rising_Edge(CLK_I) Then
            If RST_I = '1' Then
                Boot_Reset_Counter <= (Others=>'0');
            ElsIf Boot_Reset_Counter(7) = '0' Then
                Boot_Reset_Counter <= Boot_Reset_Counter + 1 ;
            Else
                Boot_Reset_Counter <= Boot_Reset_Counter;
            End If ;
        End if ;
    End Process ;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    SubPart_Nios2_TimeBase : ReplaceWith_Designator_Nios2_TimeBase Port Map (
        CLK_I => CLK_I,
        RST_I => PinSignal_SubPart_Nios2_TimeBase_RST_I,

        DAT_I => PinSignal_SubPart_Nios2_TimeBase_DAT_I,
        DAT_O => PinSignal_SubPart_Nios2_TimeBase_DAT_O,
        ADR_I => PinSignal_SubPart_Nios2_TimeBase_ADR_I,
        WE_I  => PinSignal_SubPart_Nios2_TimeBase_WE_I,

        INT_O => PinSignal_SubPart_Nios2_TimeBase_INT_O
    );
    ----------------------------------------------------------------------------

------------------------------------------------------------------------------------
End RTL;
------------------------------------------------------------------------------------

--------------------------------------------------------------------------------
Library IEEE;
Use IEEE.Std_Logic_1164.all;
Use IEEE.Std_Logic_Unsigned.All;
--------------------------------------------------------------------------------

--------------------------------------------------------------------------------
Entity ReplaceWith_Designator_Nios2_TimeBase Is
    Port
    (
        -- Global signals
        CLK_I : In Std_Logic;
        RST_I : In Std_Logic;

        DAT_I : In  Std_Logic_Vector(31 Downto 0);
        DAT_O : Out Std_Logic_Vector(31 Downto 0);
        ADR_I : In  Std_Logic_Vector(1 Downto 0);
        WE_I  : In  Std_Logic;

        INT_O : Out Std_Logic
    );
End ReplaceWith_Designator_Nios2_TimeBase;
--------------------------------------------------------------------------------

--------------------------------------------------------------------------------
Architecture RTL Of ReplaceWith_Designator_Nios2_TimeBase Is
    Signal    TimeBaseRegister_Count    : Std_Logic_Vector(63 DownTo 0);   -- Generic clock counter
    Signal    TimeBaseRegister_Lo       : Std_Logic_Vector(31 DownTo 0);
    Signal    TimeBaseRegister_Hi       : Std_Logic_Vector(31 DownTo 0);
    Signal    TimeBaseRegister_Transfer : Std_Logic_Vector(31 DownTo 0);

    Signal    TimerControlRegister      : Std_Logic_Vector(2 Downto 0);
    Signal    TimerMaxValueRegister     : Std_Logic_Vector(31 Downto 0);
    Signal    Timer                     : Std_Logic_Vector(31 Downto 0);

    Constant  cTimeBaseRegister_LO      : Std_Logic_Vector(1 Downto 0) := "00";
    Constant  cTimeBaseRegister_HI      : Std_Logic_Vector(1 Downto 0) := "01";
    Constant  cTimerMaxValueRegister    : Std_Logic_Vector(1 Downto 0) := "10";
    Constant  cTimerControlRegister     : Std_Logic_Vector(1 Downto 0) := "11";

    Signal    Interrupt                 : Std_Logic;

    Constant  cTimerEnable              : Integer := 0;
    Constant  cInterruptPending         : Integer := 1;
    Constant  cClearInterrupt           : Integer := 2;
    Constant  cResetTimer               : Integer := 3;
Begin
    ----------------------------------------------------------------------------
    -- 64-bit counter - always counts on every clock.
    TickCounterIncrement:
    Process(CLK_I)
    Begin
        If Rising_Edge(CLK_I) Then
            If RST_I = '1' Then
                TimeBaseRegister_Count <= (Others => '0');
            ElsIf (WE_I = '1') And (ADR_I = cTimerControlRegister) And (DAT_I(cResetTimer) = '1') Then
                TimeBaseRegister_Count <= (Others => '0');
            Else
                TimeBaseRegister_Count <= TimeBaseRegister_Count + 1;
            End If;
       End If;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    TickCounterCapture:
    Process (CLK_I)
    Begin
        If Rising_Edge(CLK_I) Then
           If RST_I = '1' Then
              TimeBaseRegister_Lo       <= (Others => '0');
              TimeBaseRegister_Hi       <= (Others => '0');
           Else
               If WE_I='0' Then
                  Case ADR_I Is
                    When cTimeBaseRegister_LO =>
                     -- Capture Timer Register
                        TimeBaseRegister_LO <= TimeBaseRegister_Count(31 DownTo 0);
                        TimeBaseRegister_Hi <= TimeBaseRegister_Count(63 DownTo 32);
                    When Others =>
                        TimeBaseRegister_Lo <= TimeBaseRegister_Lo;
                        TimeBaseRegister_Hi <= TimeBaseRegister_Hi;
                  End Case;

               End If;
           End If;
        End If;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    TimerIncrement:
    Process(CLK_I)
    Begin
        If Rising_Edge(CLK_I) Then
            If RST_I='1' Then
                Timer <= (Others=>'1');
            Else
                If TimerControlRegister(cTimerEnable) = '1' Then
                    If Timer = TimerMaxValueRegister Then
                        Timer <= (Others=>'0');
                    Else
                        Timer <= Timer + 1;
                    End If;
                End If;
            End If;
        End If;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    TimerInterrupt:
    Process(CLK_I)
    Begin
        If Rising_Edge(CLK_I) Then
            If RST_I='1' Then
                Interrupt <= '0';
            Else
                If TimerControlRegister(cTimerEnable) = '1' Then
                    If Timer = TimerMaxValueRegister Then
                        Interrupt <= '1';
                    ElsIf (WE_I = '1') And (ADR_I = cTimerControlRegister) And (DAT_I(cClearInterrupt) = '1') Then
                        Interrupt <= '0';
                    End If;
                Else
                    Interrupt <= '0';
                End If;
            End If;
        End If;
    End Process;
    ----------------------------------------------------------------------------

    INT_O <= Interrupt;

    ----------------------------------------------------------------------------
    RegisterWrite:
    Process(CLK_I)
    Begin
        If Rising_Edge(CLK_I) Then
            If RST_I='1' Then
                TimerMaxValueRegister <= (Others=>'0');
                TimerControlRegister  <= (Others=>'0');
            Else
                If WE_I='1' Then
                    Case ADR_I Is
                        When cTimerMaxValueRegister =>
                            TimerMaxValueRegister <= DAT_I;
                        When cTimerControlRegister =>
                            TimerControlRegister <= DAT_I(2 Downto 0);
                        When Others =>
                            NULL;
                    End Case;
                End If;
            End If;
        End If;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    RegisterRead :
    Process(TimeBaseRegister_Lo,
            TimeBaseRegister_Count,
            TimerMaxValueRegister,
            ADR_I,
            Interrupt,
            TimerControlRegister)
    Begin
        Case ADR_I Is
            When cTimeBaseRegister_LO =>
                 DAT_O <= TimeBaseRegister_Lo;
            When cTimeBaseRegister_HI =>
                 DAT_O <= TimeBaseRegister_Count(63 Downto 32);
            When cTimerMaxValueRegister =>
                 DAT_O <= TimerMaxValueRegister;
            When cTimerControlRegister =>
                 DAT_O <= "000000000000000000000000000000" & Interrupt & TimerControlRegister(cTimerEnable);
            When Others =>
                 DAT_O <= (Others=>'0');
        End Case;
    End Process;
    ----------------------------------------------------------------------------

End RTL;
--------------------------------------------------------------------------------

------------------------------------------------------------------------------------
Library IEEE;
Use     IEEE.Std_Logic_1164.All;
Use     IEEE.Std_Logic_Unsigned.All;
------------------------------------------------------------------------------------

------------------------------------------------------------------------------------
Entity ReplaceWith_Designator_Nios2_Template Is
    Port (
        --------------------------------------------------------------------
        CLK_I                    : In  Std_Logic;
        RST_I                    : In  Std_Logic;
        INT_I                    : In  Std_Logic_Vector(31 Downto 0);
        --------------------------------------------------------------------

        --------------------------------------------------------------------
        IO_STB_O                 : Out Std_Logic;
        IO_CYC_O                 : Out Std_Logic;
        IO_ACK_I                 : In  Std_Logic;
        IO_ADR_O                 : Out Std_Logic_Vector(23 Downto 0);
        IO_DAT_I                 : In  Std_Logic_Vector(31 Downto 0);
        IO_DAT_O                 : Out Std_Logic_Vector(31 Downto 0);
        IO_SEL_O                 : Out Std_Logic_Vector(3  Downto 0);
        IO_WE_O                  : Out Std_Logic;
        IO_CLK_O                 : Out Std_Logic;
        IO_RST_O                 : Out Std_Logic;
        --------------------------------------------------------------------

        --------------------------------------------------------------------
        DataRamIn                : In  Std_Logic_Vector(31 Downto 0);
        DataRamOut               : Out Std_Logic_Vector(31 Downto 0);
        DataRamBE                : Out Std_Logic_Vector(3  Downto 0);
        DataRamWE                : Out Std_Logic;
        DataRamAddr              : Out Std_Logic_Vector(31 Downto 0);
        InstrRomIn               : In  Std_Logic_Vector(31 Downto 0);
        InstrRomAddr             : Out Std_Logic_Vector(31 Downto 0);
        --------------------------------------------------------------------

        --------------------------------------------------------------------
        ME_STB_O                 : Out Std_Logic;
        ME_CYC_O                 : Out Std_Logic;
        ME_ACK_I                 : In  Std_Logic;
        ME_ADR_O                 : Out Std_Logic_Vector(31 Downto 0);
        ME_DAT_I                 : In  Std_Logic_Vector(31 Downto 0);
        ME_DAT_O                 : Out Std_Logic_Vector(31 Downto 0);
        ME_SEL_O                 : Out Std_Logic_Vector(3  Downto 0);
        ME_WE_O                  : Out Std_Logic;
        ME_CLK_O                 : Out Std_Logic;
        ME_RST_O                 : Out Std_Logic;
        --------------------------------------------------------------------

        --------------------------------------------------------------------
        Debug_Ir_Out             : Out Std_Logic_Vector(7 Downto 0);
        Debug_Irq                : Out Std_Logic;
        Debug_TDO                : Out Std_Logic;
        Debug_Clrn               : In  Std_Logic;
        Debug_Ena                : In  Std_Logic;
        Debug_Ir_In              : In  Std_Logic_Vector(7 Downto 0);
        Debug_Raw_TCK            : In  Std_Logic;
        Debug_Rti                : In  Std_Logic;
        Debug_Shift              : In  Std_Logic;
        Debug_TDI                : In  Std_Logic;
        Debug_Update             : In  Std_Logic;
        Debug_Usr1               : In  Std_Logic;
        --------------------------------------------------------------------

        --------------------------------------------------------------------
        TimeBaseIn               : In  Std_Logic_Vector(31 Downto 0);
        TimeBaseOut              : Out Std_Logic_Vector(31 Downto 0);
        TimeBaseAddr             : Out Std_Logic_Vector(1 Downto 0);
        TimeBaseWE               : Out Std_Logic;
        TimeBaseInterrupt        : In  Std_Logic;
        TimeBaseReset            : Out Std_Logic;
        --------------------------------------------------------------------

        --------------------------------------------------------------------
        LAX_CPU_I_ReadData        : Out Std_Logic_Vector(31 Downto 0);
        LAX_CPU_I_WaitRequest     : Out Std_Logic;
        LAX_CPU_I_Address         : Out Std_Logic_Vector(31 Downto 0);
        LAX_CPU_I_Read            : Out Std_Logic;
        LAX_CPU_I_ReadDataValid   : Out Std_Logic;
        LAX_CPU_D_Address         : Out Std_Logic_Vector(31 Downto 0);
        LAX_CPU_D_ByteEnable      : Out Std_Logic_Vector(3  Downto 0);
        LAX_CPU_D_Read            : Out Std_Logic;
        LAX_CPU_D_Write           : Out Std_Logic;
        LAX_CPU_D_ReadData        : Out Std_Logic_Vector(31 Downto 0);
        LAX_CPU_D_WriteData       : Out Std_Logic_Vector(31 Downto 0);
        LAX_CPU_D_WaitRequest     : Out Std_Logic;

        Debug_MainMemory_I_Wait   : Out Std_Logic;
        Debug_MainMemory_D_Wait   : Out Std_Logic;
        Debug_Peripheral_I_Wait   : Out Std_Logic;
        Debug_Peripheral_D_Wait   : Out Std_Logic;
        Debug_InternalMem_I_Wait  : Out Std_Logic;
        Debug_InternalMem_D_Wait  : Out Std_Logic;
        Debug_DebugMem_I_Wait     : Out Std_Logic;
        Debug_DebugMem_D_Wait     : Out Std_Logic;
        Debug_State_IO            : Out Std_Logic_Vector(1 Downto 0);
        Debug_StateNext_IO        : Out Std_Logic_Vector(1 Downto 0);
        Debug_Access_IO_Data      : Out Std_Logic;
        Debug_Access_IO_Instr     : Out Std_Logic
        --------------------------------------------------------------------
   );
End;
------------------------------------------------------------------------------------

------------------------------------------------------------------------------------
Architecture RTL Of ReplaceWith_Designator_Nios2_Template Is
------------------------------------------------------------------------------------
    --------------------------------------------------------------------
    Component ReplaceWith_MCU Is
        Port (
             Clk             : In  Std_Logic;
             Reset_n         : In  Std_Logic;
             D_Irq           : In  Std_Logic_Vector(31 Downto 0);

             I_ReadData      : In  Std_Logic_Vector(31 Downto 0);
             I_WaitRequest   : In  Std_Logic;
             I_Address       : Out Std_Logic_Vector(31 Downto 0);
             I_Read          : Out Std_Logic;
             I_ReadDataValid : In  Std_Logic;                       -- IncludeIf_HasInstructionCache

             D_Address       : Out Std_Logic_Vector(31 Downto 0);
             D_ByteEnable    : Out Std_Logic_Vector(3  Downto 0);
             D_Read          : Out Std_Logic;
             D_Write         : Out Std_Logic;
             D_ReadData      : In  Std_Logic_Vector(31 Downto 0);
             D_WriteData     : Out Std_Logic_Vector(31 Downto 0);
             D_WaitRequest   : In  Std_Logic;

             Ir_Out          : Out Std_Logic_Vector(7 Downto 0);
             Irq             : Out Std_Logic;
             TDO             : Out Std_Logic;
             Clrn            : In  Std_Logic;
             Ena             : In  Std_Logic;
             Ir_In           : In  Std_Logic_Vector(7 Downto 0);
             Raw_TCK         : In  Std_Logic;
             Rti             : In  Std_Logic;
             Shift           : In  Std_Logic;
            --         TCK             : In  Std_Logic;
             TDI             : In  Std_Logic;
             Update          : In  Std_Logic;
             Usr1            : In  Std_Logic;

             Jtag_Debug_Module_Address             : In  Std_Logic_Vector(8 Downto 0);
             Jtag_Debug_Module_BeginTransfer       : In  Std_Logic;
             Jtag_Debug_Module_ByteEnable          : In  Std_Logic_Vector(3 Downto 0);
             Jtag_Debug_Module_Clk                 : In  Std_Logic;
             Jtag_Debug_Module_DebugAccess         : In  Std_Logic;
             Jtag_Debug_Module_Reset               : In  Std_Logic;
             Jtag_Debug_Module_Select              : In  Std_Logic;
             Jtag_Debug_Module_Write               : In  Std_Logic;
             Jtag_Debug_Module_WriteData           : In  Std_Logic_Vector(31 Downto 0);
             Jtag_Debug_Module_DebugAccess_To_Roms : Out Std_Logic;
             Jtag_Debug_Module_ReadData            : Out Std_Logic_Vector(31 Downto 0);
             Jtag_Debug_Module_ResetRequest        : Out Std_Logic
        );
    End Component;
    --------------------------------------------------------------------
------------------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    Type TStateIO          Is (IO_Idle          , IO_WaitForDMemACK         , IO_WaitForIMemACK         , IO_Dummy);
    Type TStateME          Is (ME_Idle          , ME_WaitForDMemACK         , ME_WaitForIMemACK         , ME_Dummy);
    Type TStateInternalMem Is (InternalMem_Idle , InternalMem_WaitForDMemACK, InternalMem_WaitForIMemACK, InternalMem_Dummy);
    Type TStateDebugMem    Is (DebugMem_Idle    , DebugMem_WaitForDMemACK   , DebugMem_WaitForIMemACK   , DebugMem_Dummy);
    Type TStateTimeBaseMem Is (TimeBaseMem_Idle , TimeBaseMem_WaitForDMemACK, TimeBaseMem_WaitForIMemACK, TimeBaseMem_Dummy);
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    Constant cDebugUnitAddress    : Std_Logic_Vector(31 Downto 0) := x"00FFF000";
    Constant cTimeBaseUnitAddress : Std_Logic_Vector(31 Downto 0) := x"FFFFFFE0";
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    Signal StateIO                      : TStateIO;
    Signal StateIO_Next                 : TStateIO;
    Signal StateME                      : TStateME;
    Signal StateME_Next                 : TStateME;
    Signal StateInternalMem             : TStateInternalMem;
    Signal StateInternalMem_Next        : TStateInternalMem;
    Signal StateDebugMem                : TStateDebugMem;
    Signal StateDebugMem_Next           : TStateDebugMem;
    Signal StateTimeBaseMem             : TStateTimeBaseMem;
    Signal StateTimeBaseMem_Next        : TStateTimeBaseMem;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    Signal Combined_Rst                 : Std_Logic;
    Signal Signal_Rst                   : Std_Logic;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    Signal Instr_ReadData               : Std_Logic_Vector(31 Downto 0);
    Signal Instr_WaitRequest            : Std_Logic;
    Signal Instr_Address                : Std_Logic_Vector(31 Downto 0);
    Signal Instr_Read                   : Std_Logic;
    Signal Instr_ReadDataValid          : Std_Logic;
    Signal Data_Address                 : Std_Logic_Vector(31 Downto 0);
    Signal Data_ByteEnable              : Std_Logic_Vector(3  Downto 0);
    Signal Data_Read                    : Std_Logic;
    Signal Data_Write                   : Std_Logic;
    Signal Data_ReadData                : Std_Logic_Vector(31 Downto 0);
    Signal Data_WriteData               : Std_Logic_Vector(31 Downto 0);
    Signal Data_WaitRequest             : Std_Logic;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    Signal InstrAddressIsInternalMemory : Std_Logic;
    Signal InstrAddressIsPeripheral     : Std_Logic;
    Signal InstrAddressIsExternalMemory : Std_Logic;
    Signal InstrAddressIsDebugMemory    : Std_Logic;
    Signal InstrAddressIsTimeBaseMemory : Std_Logic;
    Signal DataAddressIsInternalMemory  : Std_Logic;
    Signal DataAddressIsPeripheral      : Std_Logic;
    Signal DataAddressIsExternalMemory  : Std_Logic;
    Signal DataAddressIsDebugMemory     : Std_Logic;
    Signal DataAddressIsTimeBaseMemory  : Std_Logic;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    Signal Access_ME_Data               : Std_Logic;
    Signal Access_ME_Instr              : Std_Logic;
    Signal Access_ME                    : Std_Logic;
    Signal Access_IO_Data               : Std_Logic;
    Signal Access_IO_Instr              : Std_Logic;
    Signal Access_InternalMem_Data      : Std_Logic;
    Signal Access_InternalMem_Instr     : Std_Logic;
    Signal Access_DebugMem_Data         : Std_Logic;
    Signal Access_DebugMem_Instr        : Std_Logic;
    Signal Access_TimeBaseMem_Data      : Std_Logic;
    Signal Access_TimeBaseMem_Instr     : Std_Logic;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    Signal TimeOut_IO_Counter           : Std_Logic_Vector(12 Downto 0);
    Signal TimeOut_IO                   : Std_Logic;
    Signal TimeOut_ME_Counter           : Std_Logic_Vector(12 Downto 0);
    Signal TimeOut_ME                   : Std_Logic;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    Signal MainMemory_I_Wait            : Std_Logic;
    Signal MainMemory_D_Wait            : Std_Logic;
    Signal Peripheral_I_Wait            : Std_Logic;
    Signal Peripheral_D_Wait            : Std_Logic;
    Signal InternalMem_I_Wait           : Std_Logic;
    Signal InternalMem_D_Wait           : Std_Logic;
    Signal DebugMem_I_Wait              : Std_Logic;
    Signal DebugMem_D_Wait              : Std_Logic;
    Signal TimeBaseMem_I_Wait           : Std_Logic;
    Signal TimeBaseMem_D_Wait           : Std_Logic;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    Signal MainMemory_Running           : Std_Logic;
    Signal Peripheral_Running           : Std_Logic;
    Signal Peripheral_Acked             : Std_Logic;
    Signal Peripheral_Data_In_Latch     : Std_Logic_Vector(31 downto 0);
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    Signal MainMemory_I_Lock            : Std_Logic;
    Signal Peripheral_I_Lock            : Std_Logic;
    Signal DebugMem_I_Lock              : Std_Logic;
    Signal TimeBaseMem_I_Lock           : Std_Logic;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    Signal Jtag_Debug_Module_Address             : Std_Logic_Vector(8 Downto 0);
    Signal Jtag_Debug_Module_BeginTransfer       : Std_Logic;
    Signal Jtag_Debug_Module_ByteEnable          : Std_Logic_Vector(3 Downto 0);
    Signal Jtag_Debug_Module_Clk                 : Std_Logic;
    Signal Jtag_Debug_Module_DebugAccess         : Std_Logic;
    Signal Jtag_Debug_Module_Reset               : Std_Logic;
    Signal Jtag_Debug_Module_Select              : Std_Logic;
    Signal Jtag_Debug_Module_Write               : Std_Logic;
    Signal Jtag_Debug_Module_WriteData           : Std_Logic_Vector(31 Downto 0);
    Signal Jtag_Debug_Module_DebugAccess_To_Roms : Std_Logic;
    Signal Jtag_Debug_Module_ReadData            : Std_Logic_Vector(31 Downto 0);
    Signal Jtag_Debug_Module_ResetRequest        : Std_Logic;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    Signal DebugMem_DataRequest         : Std_Logic;
    Signal EndianFixed_Data_WriteData   : Std_Logic_Vector(31 Downto 0);
    Signal EndianFixed_Data_ByteEnable  : Std_Logic_Vector(3 Downto 0);
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    Signal Interrupt_Cpu                : Std_Logic_Vector(31 Downto 0);
    ----------------------------------------------------------------------------
Begin
------------------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    LAX_CPU_I_ReadData      <= Instr_ReadData;
    LAX_CPU_I_WaitRequest   <= Instr_WaitRequest;
    LAX_CPU_I_Address       <= Instr_Address;
    LAX_CPU_I_Read          <= Instr_Read;
    LAX_CPU_I_ReadDataValid <= Instr_ReadDataValid;
    LAX_CPU_D_Address       <= Data_Address;
    LAX_CPU_D_ByteEnable    <= Data_ByteEnable;
    LAX_CPU_D_Read          <= Data_Read;
    LAX_CPU_D_Write         <= Data_Write;
    LAX_CPU_D_ReadData      <= Data_ReadData;
    LAX_CPU_D_WriteData     <= Data_WriteData;
    LAX_CPU_D_WaitRequest   <= Data_WaitRequest;

    Debug_MainMemory_I_Wait  <= MainMemory_I_Wait  ;
    Debug_MainMemory_D_Wait  <= MainMemory_D_Wait  ;
    Debug_Peripheral_I_Wait  <= Peripheral_I_Wait  ;
    Debug_Peripheral_D_Wait  <= Peripheral_D_Wait  ;
    Debug_InternalMem_I_Wait <= InternalMem_I_Wait ;
    Debug_InternalMem_D_Wait <= InternalMem_D_Wait ;
    Debug_DebugMem_I_Wait    <= DebugMem_I_Wait    ;
    Debug_DebugMem_D_Wait    <= DebugMem_D_Wait    ;
    ----------------------------------------------------------------------------


    ----------------------------------------------------------------------------
    SubPart_MCU : ReplaceWith_MCU Port Map (
        Clk                => CLK_I,
        Reset_n            => Signal_Rst,
        D_Irq              => Interrupt_Cpu,

        I_ReadData         => Instr_ReadData,
        I_WaitRequest      => Instr_WaitRequest,
        I_Address          => Instr_Address,
        I_Read             => Instr_Read,
        I_ReadDataValid    => Instr_ReadDataValid,      -- IncludeIf_HasInstructionCache

        D_Address          => Data_Address,
        D_ByteEnable       => Data_ByteEnable,
        D_Read             => Data_Read,
        D_Write            => Data_Write,
        D_ReadData         => Data_ReadData,
        D_WriteData        => Data_WriteData,
        D_WaitRequest      => Data_WaitRequest,
--        D_ReadDataValid    => Data_ReadDataValid,       -- IncludeIF_HasDataCache

        Ir_Out             => Debug_Ir_Out,
        Irq                => Debug_Irq,
        TDO                => Debug_TDO,
        Clrn               => Debug_Clrn,
        Ena                => Debug_Ena,
        Ir_In              => Debug_Ir_In,
        Raw_TCK            => Debug_Raw_TCK,
        Rti                => Debug_Rti,
        Shift              => Debug_Shift,
--        TCK                => Debug_TCK,
        TDI                => Debug_TDI,
        Update             => Debug_Update,
        Usr1               => Debug_Usr1,


        Jtag_Debug_Module_Address             => Jtag_Debug_Module_Address,
        Jtag_Debug_Module_BeginTransfer       => Jtag_Debug_Module_BeginTransfer,
        Jtag_Debug_Module_ByteEnable          => Jtag_Debug_Module_ByteEnable,
        Jtag_Debug_Module_Clk                 => Jtag_Debug_Module_Clk,
        Jtag_Debug_Module_DebugAccess         => Jtag_Debug_Module_DebugAccess,
        Jtag_Debug_Module_Reset               => Jtag_Debug_Module_Reset,
        Jtag_Debug_Module_Select              => Jtag_Debug_Module_Select,
        Jtag_Debug_Module_Write               => Jtag_Debug_Module_Write,
        Jtag_Debug_Module_WriteData           => Jtag_Debug_Module_WriteData,
        Jtag_Debug_Module_DebugAccess_To_Roms => Jtag_Debug_Module_DebugAccess_To_Roms,
        Jtag_Debug_Module_ReadData            => Jtag_Debug_Module_ReadData,
        Jtag_Debug_Module_ResetRequest        => Jtag_Debug_Module_ResetRequest
    );
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    Combined_Rst       <= RST_I Or Jtag_Debug_Module_ResetRequest;
    Signal_Rst         <= Not Combined_Rst;
    IO_CLK_O           <= CLK_I;
    IO_RST_O           <= Combined_Rst;
    ME_CLK_O           <= CLK_I;
    ME_RST_O           <= Combined_Rst;
    TimeBaseReset      <= Combined_Rst;
    Interrupt_Cpu      <= INT_I(31 Downto 1) & TimeBaseInterrupt;                --RAF: to be compatible with TSK3K we would have to "Or" TimeBaseInterrupt with INT_I(0)
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    DecodeInstrAddress:
    Process (Instr_Address)
    Begin
        InstrAddressIsDebugMemory    <= '0';
        InstrAddressIsInternalMemory <= '0';
        InstrAddressIsPeripheral     <= '0';
        InstrAddressIsExternalMemory <= '0';
        InstrAddressIsTimeBaseMemory <= '0';

        If Instr_Address(31 Downto 11) = cDebugUnitAddress(31 Downto 11) Then
            InstrAddressIsDebugMemory <= '1';
        ElsIf Instr_Address(31 Downto 4) = cTimeBaseUnitAddress(31 Downto 4) Then
            InstrAddressIsTimeBaseMemory <= '1';
        ElsIf Instr_Address(31 Downto 24) = x"00" Then
            InstrAddressIsInternalMemory <= '1';
        ElsIf Instr_Address(31 Downto 24) = x"FF" Then
            InstrAddressIsPeripheral <= '1'; -- support instruction fetch on the IO side!!! RAF: DO NOT SUPPORT that, just decode to prevent the processor from fetching instructions from this address as we do for tsk3000, unless we'd like to have different IO behaviour on different processors
        Else
            InstrAddressIsExternalMemory <= '1';
        End If;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    DecodeDataAddress:
    Process (Data_Address)
    Begin
        DataAddressIsDebugMemory    <= '0';
        DataAddressIsInternalMemory <= '0';
        DataAddressIsPeripheral     <= '0';
        DataAddressIsExternalMemory <= '0';
        DataAddressIsTimeBaseMemory <= '0';

        If Data_Address(31 Downto 11) = cDebugUnitAddress(31 Downto 11) Then
            DataAddressIsDebugMemory <= '1';
        ElsIf Data_Address(31 Downto 4) = cTimeBaseUnitAddress(31 Downto 4) Then
            DataAddressIsTimeBaseMemory <= '1';
        ElsIf Data_Address(31 Downto 24) = x"00" Then
            DataAddressIsInternalMemory <= '1';
        ElsIf Data_Address(31 Downto 24) = x"FF" Then
            DataAddressIsPeripheral <= '1';
        Else
            DataAddressIsExternalMemory <= '1';
        End If;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    WaitSignals:
    Process (MainMemory_I_Wait,
             MainMemory_D_Wait,
             Peripheral_I_Wait,
             Peripheral_D_Wait,
             InternalMem_I_Wait,
             InternalMem_D_Wait,
             TimeBaseMem_D_Wait,
             TimeBaseMem_I_Wait,
             DebugMem_I_Wait,
             DebugMem_D_Wait,
             Data_WaitRequest,
             Instr_WaitRequest,
             Instr_ReadDataValid,
             Instr_Read)
    Begin
        Data_WaitRequest    <= MainMemory_D_Wait Or Peripheral_D_Wait Or InternalMem_D_Wait Or DebugMem_D_Wait Or TimeBaseMem_D_Wait;
        Instr_WaitRequest   <= MainMemory_I_Wait Or Peripheral_I_Wait Or InternalMem_I_Wait Or DebugMem_I_Wait Or TimeBaseMem_I_Wait;
        Instr_ReadDataValid <= '0';
        If Instr_Read = '1' Then
            Instr_ReadDataValid <= Not (MainMemory_I_Wait Or Peripheral_I_Wait Or InternalMem_I_Wait Or DebugMem_I_Wait Or TimeBaseMem_I_Wait);
        End If;
    End Process;
    ----------------------------------------------------------------------------


    ----------------------------------------------------------------------------
    Data_ReadData_Mux:
    Process(DataAddressIsInternalMemory,
            DataAddressIsPeripheral,
            DataAddressIsInternalMemory,
            DataAddressIsExternalMemory,
            DataAddressIsDebugMemory,
            DataAddressIsTimeBaseMemory,
            Data_ReadData,
            DataRamIn,
            IO_DAT_I,
            ME_DAT_I,
            TimeBaseIn,
            Jtag_Debug_Module_ReadData)
    Begin
        Data_ReadData <= (Others=>'0');

        If DataAddressIsInternalMemory = '1' Then
            Data_ReadData <= DataRamIn;
        ElsIf DataAddressIsPeripheral = '1' Then
            If Peripheral_Acked = '0' Then
                Data_ReadData <= IO_DAT_I;
            Else
                -- use latched data instead of live data,
                -- when the peripheral acked we removed the strobe
                -- so data is not guaranteed to be valid anymore
                Data_ReadData <= Peripheral_Data_In_Latch;
            End If;
        ElsIf DataAddressIsExternalMemory = '1' Then
            Data_ReadData <= ME_DAT_I;
        ElsIf DataAddressIsDebugMemory = '1' Then
            Data_ReadData <= Jtag_Debug_Module_ReadData;
        ElsIf DataAddressIsTimeBaseMemory = '1' Then
            Data_ReadData <= TimeBaseIn;
        End If;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    Instr_ReadData_Mux:
    Process(InstrAddressIsExternalMemory,
            InstrAddressIsInternalMemory,
            InstrAddressIsDebugMemory,
            Instr_ReadData,
            ME_DAT_I,
            InstrRomIn,
            Jtag_Debug_Module_ReadData)
    Begin
        Instr_ReadData <= (Others=>'0');

        If InstrAddressIsExternalMemory = '1' Then
            Instr_ReadData <= ME_DAT_I;
        ElsIf InstrAddressIsInternalMemory = '1' Then
            Instr_ReadData <= InstrRomIn;
        ElsIf InstrAddressIsDebugMemory = '1' Then
            Instr_ReadData <= Jtag_Debug_Module_ReadData;
        End If;
    End Process;
    ----------------------------------------------------------------------------


    ----------------------------------------------------------------------------
    Data_WriteData_Mux:
    Process(Data_WriteData,
            Data_ByteEnable)
    Begin
        EndianFixed_Data_ByteEnable <= Data_ByteEnable;
        EndianFixed_Data_WriteData  <= Data_WriteData;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    -- Debug Memory Interface;
    ----------------------------------------------------------------------------
    Access_DebugMem_Data  <= (Data_Read  Or Data_Write) And DataAddressIsDebugMemory;
    Access_DebugMem_Instr <= (Instr_Read              ) And InstrAddressIsDebugMemory;
    ----------------------------------------------------------------------------

   DebugMem_DataRequest            <= (DataAddressIsDebugMemory And (Data_Read Or Data_Write));
   Jtag_Debug_Module_Address       <= Data_Address(10 Downto 2) When ((DebugMem_DataRequest='1') And (DebugMem_I_Lock='0')) Else Instr_Address(10 Downto 2);
   Jtag_Debug_Module_BeginTransfer <= (InstrAddressIsDebugMemory And Instr_Read) Or (DataAddressIsDebugMemory And (Data_Read Or Data_Write));
   Jtag_Debug_Module_ByteEnable    <= "1111" When (InstrAddressIsDebugMemory='1' And Instr_Read='1') Else Data_ByteEnable;
   Jtag_Debug_Module_Clk           <= CLK_I;
   Jtag_Debug_Module_DebugAccess   <= Jtag_Debug_Module_DebugAccess_To_Roms;
   Jtag_Debug_Module_Reset         <= Combined_Rst;
   Jtag_Debug_Module_Select        <= Jtag_Debug_Module_BeginTransfer;
   Jtag_Debug_Module_Write         <= Data_Write When ((DataAddressIsDebugMemory='1') And (DebugMem_I_Lock='0'))  Else '0';
   Jtag_Debug_Module_WriteData     <= Data_WriteData;


    ----------------------------------------------------------------------------
    DebugMemory_Combinational:
    Process (StateDebugMem,
             Access_DebugMem_Data,
             Access_DebugMem_Instr,
             Instr_Read,
             InstrAddressIsDebugMemory,
             Data_Read,
             Data_Write,
             DataAddressIsDebugMemory)
    Begin
        DebugMem_I_Lock    <= '0';
        DebugMem_I_Wait    <= Access_DebugMem_Instr;
        DebugMem_D_Wait    <= Access_DebugMem_Data;
        StateDebugMem_Next <= StateDebugMem;

        Case StateDebugMem Is
            When DebugMem_Idle =>
                If Access_DebugMem_Data = '1' Then
                    StateDebugMem_Next <= DebugMem_WaitForDMemAck;
                ElsIf Access_DebugMem_Instr = '1' Then
                    StateDebugMem_Next <= DebugMem_WaitForIMemAck;
                Else
                    StateDebugMem_Next <= DebugMem_Idle;
                End If;
            When DebugMem_WaitForDMemAck =>
                DebugMem_D_Wait       <= '0';
                StateDebugMem_Next    <= DebugMem_Idle;
            When DebugMem_WaitForIMemAck =>
                DebugMem_I_Wait       <= '0';
                DebugMem_I_Lock       <= '1';
                StateDebugMem_Next    <= DebugMem_Idle;
            When Others =>
                StateDebugMem_Next    <= DebugMem_Idle;
        End Case;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    DebugMemory_Registered:
    Process(CLK_I,
            Combined_Rst)
    Begin
      If Rising_Edge(CLK_I) Then
         If Combined_Rst='1' Then
            StateDebugMem <= DebugMem_Idle;
         Else
            StateDebugMem <= StateDebugMem_Next;
         End If;
      End If;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    -- Internal Memory Interface;
    ----------------------------------------------------------------------------
    Access_InternalMem_Data  <= (Data_Read  Or Data_Write) And DataAddressIsInternalMemory;
    Access_InternalMem_Instr <= (Instr_Read              ) And InstrAddressIsInternalMemory;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    InternalMemory_Control:
    Process (EndianFixed_Data_WriteData,
             EndianFixed_Data_ByteEnable,
             DataAddressIsInternalMemory,
             Data_Address,
             Data_Write,
             Instr_Address)
    Begin
        DataRamOut   <= EndianFixed_Data_WriteData;
        DataRamBE    <= EndianFixed_Data_ByteEnable;
        DataRamAddr  <= Data_Address;
        InstrRomAddr <= Instr_Address;
        DataRamWE    <= '0';
        If DataAddressIsInternalMemory = '1' Then
            DataRamWE <= Data_Write;
        End If;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    -- 1 clock wait for internal mem (remove in the future, once we have tightly coupled mem. do this by setting InternalMem_Running to 0 permanently)
    -- RAF: shouldn't InternalMem_Running be set in InternalMem_Idle whichever interface is requesting access and released in InternalMem_Wait? (need to check that with avalon spec)
    --      Also, data and instruction interfaces are independed (dualport ram that we use) so we need a separate decoder for instruction and data sides.
    --      According to avalon spec, waitrequest must be asserted within the first clock cycle of transfer (avalon spec p.34), but be carefull here, check if this works.
    ----------------------------------------------------------------------------
    InternalMemory_Combinational:
    Process( StateInternalMem,
             Access_InternalMem_Data,
             Access_InternalMem_Instr,
             InternalMem_I_Wait,
             InternalMem_D_Wait)
    Begin
        InternalMem_I_Wait    <= Access_InternalMem_Instr;
        InternalMem_D_Wait    <= Access_InternalMem_Data;
        StateInternalMem_Next <= StateInternalMem;

        Case StateInternalMem Is
            When InternalMem_Idle =>
                If Access_InternalMem_Data = '1' Then
                    StateInternalMem_Next <= InternalMem_WaitForDMemAck;
                ElsIf Access_InternalMem_Instr = '1' Then
                    StateInternalMem_Next <= InternalMem_WaitForIMemAck;
                Else
                    StateInternalMem_Next <= InternalMem_Idle;
                End If;
            When InternalMem_WaitForDMemAck =>
                InternalMem_D_Wait    <= '0';
                StateInternalMem_Next <= InternalMem_Idle;
            When InternalMem_WaitForIMemAck =>
                InternalMem_I_Wait    <= '0';
                StateInternalMem_Next <= InternalMem_Idle;
            When Others =>
                StateInternalMem_Next <= InternalMem_Idle;
        End Case;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    InternalMemory_IO_Registered:
    Process(CLK_I)
    Begin
       If Rising_Edge(CLK_I) Then
          If Combined_Rst = '1' Then
             StateInternalMem <= InternalMem_Idle;
          Else
             StateInternalMem <= StateInternalMem_Next;
          End If;
       End If;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    -- Wishbone IO Peripheral Interface;
    ----------------------------------------------------------------------------
    Access_IO_Data  <= (Data_Read  Or Data_Write) And DataAddressIsPeripheral;
    Access_IO_Instr <= (Instr_Read              ) And InstrAddressIsPeripheral;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    WishboneIO_Acked:
    Process( CLK_I )

    Begin
       If rising_edge(CLK_I) Then
           If Peripheral_Running = '1' Then
               If IO_ACK_I = '1' Then
                   Peripheral_Acked <= '1';

                   -- remember the current output data from the peripheral,
                   -- now that the peripheral acked we will remove the strobe
                   -- so the data on IO_DAT_I will not be valid anymore
                   Peripheral_Data_In_Latch <= IO_DAT_I;
               End If;
           Else
               Peripheral_Acked <= '0';
           End If;
       End If;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    WishboneIO_Outputs:
    Process( Access_IO_Data,
             Peripheral_Running,
             Peripheral_I_Lock,
             Data_Address,
             EndianFixed_Data_ByteEnable,
             Data_Write,
             EndianFixed_Data_WriteData,
             Instr_Address,
             Peripheral_Acked)

    Begin
       If Peripheral_Acked = '0' Then
           IO_STB_O <= Peripheral_Running;
           IO_CYC_O <= Peripheral_Running;
       Else
           IO_STB_O <= '0';
           IO_CYC_O <= '0';
       End If;

       IO_DAT_O <= EndianFixed_Data_WriteData;

       If (Access_IO_Data = '1') And (Peripheral_I_Lock = '0') Then
          IO_ADR_O <= Data_Address(23 Downto 0);
          IO_SEL_O <= EndianFixed_Data_ByteEnable;
          IO_WE_O  <= Data_Write;
       Else
          IO_ADR_O <= Instr_Address(23 Downto 0);
          IO_SEL_O <= "1111";
          IO_WE_O  <= '0';
       End If;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    WishboneFSM_IO_State_Decoder:
    Process ( StateIO, StateIO_Next )
    Begin
        Case StateIO Is
            When IO_Idle =>
                Debug_State_IO <= "00";
            When IO_WaitForDMemAck =>
                Debug_State_IO <= "01";
            When IO_WaitForIMemAck =>
                Debug_State_IO <= "10";
            When Others =>
                Debug_State_IO <= "11";
        End Case;

        Case StateIO_Next Is
            When IO_Idle =>
                Debug_StateNext_IO <= "00";
            When IO_WaitForDMemAck =>
                Debug_StateNext_IO <= "01";
            When IO_WaitForIMemAck =>
                Debug_StateNext_IO <= "10";
            When Others =>
                Debug_StateNext_IO <= "11";
        End Case;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    WishboneFSM_IO_Combinatorial:
    Process ( Access_IO_Instr,
              Access_IO_Data,
              IO_ACK_I,
              TimeOut_IO,
              Peripheral_I_Wait,
              Peripheral_D_Wait,
              StateIO)
    Begin
        Peripheral_I_Lock  <= '0';
        Peripheral_I_Wait  <= Access_IO_Instr;
        Peripheral_D_Wait  <= Access_IO_Data;
        Peripheral_Running <= Access_IO_Instr Or Access_IO_Data;
        StateIO_Next       <= StateIO;

        Case StateIO Is
            When IO_Idle =>
                If Access_IO_Data = '1' Then
                    StateIO_Next <= IO_WaitForDMemAck;
                ElsIf Access_IO_Instr = '1' Then
                    StateIO_Next <= IO_WaitForIMemAck;
                Else
                    StateIO_Next <= IO_Idle;
                End If;
            When IO_WaitForDMemAck =>
                Peripheral_D_Wait  <= '1';
                Peripheral_Running <= '1';
                If (IO_ACK_I = '1') Or (Peripheral_Acked = '1') Or (TimeOut_IO = '1') Then
                    Peripheral_D_Wait <= '0';
                    StateIO_Next      <= IO_Idle;
                End If;
            When IO_WaitForIMemAck =>
                Peripheral_I_Wait  <= '1';
                Peripheral_I_Lock  <= '1';
                Peripheral_Running <= '1';
                If (IO_ACK_I = '1') Or (Peripheral_Acked = '1') Or (TimeOut_IO = '1') Then
                    Peripheral_I_Wait <= '0';
                    StateIO_Next      <= IO_Idle;
                End If;
            When Others =>
                StateIO_Next <= IO_Idle;
        End Case;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    WishboneFSM_IO_Registered:
    Process(CLK_I,
            Combined_Rst)
    Begin
        If Rising_Edge(CLK_I) Then
            If Combined_Rst='1' Then
                StateIO <= IO_Idle;
            Else
                StateIO <= StateIO_Next;
            End If;
        End If;
    End Process;
    ----------------------------------------------------------------------------


    ----------------------------------------------------------------------------
    -- Wishbone Memory Interface;
    ----------------------------------------------------------------------------
    Access_ME_Data  <= (Data_Read  Or Data_Write) And DataAddressIsExternalMemory;
    Access_ME_Instr <= (Instr_Read              ) And InstrAddressIsExternalMemory;
    Access_ME       <= Access_ME_Data Or Access_ME_Instr;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    WishboneME_Outputs:
    Process( MainMemory_Running,
             MainMemory_I_Lock,
             Data_Address,
             Access_ME_Data,
             EndianFixed_Data_ByteEnable,
             Data_Write,
             EndianFixed_Data_WriteData,
             Instr_Address )

    Begin
       ME_STB_O <= MainMemory_Running;
       ME_CYC_O <= MainMemory_Running;
       ME_DAT_O <= EndianFixed_Data_WriteData;

       If (Access_ME_Data = '1') And (MainMemory_I_Lock = '0') Then
          ME_ADR_O <= Data_Address;
          ME_SEL_O <= EndianFixed_Data_ByteEnable;
          ME_WE_O  <= Data_Write;
       Else
          ME_ADR_O <= Instr_Address;
          ME_SEL_O <= "1111";
          ME_WE_O  <= '0';
       End If;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    WishboneFSM_ME_Combinatorial:
    Process ( Access_ME_Instr,
              Access_ME_Data,
              ME_ACK_I,
              TimeOut_ME,
              MainMemory_I_Wait,
              MainMemory_D_Wait,
              StateME)
    Begin
        MainMemory_I_Wait  <= Access_ME_Instr;
        MainMemory_D_Wait  <= Access_ME_Data;
        MainMemory_Running <= Access_ME_Instr Or Access_ME_Data;
        MainMemory_I_Lock  <= '0';
        StateME_Next       <= StateME;

        Case StateME Is
            When Me_Idle =>
                If Access_ME_Data = '1' Then
                    StateME_Next <= ME_WaitForDMemAck;
                ElsIf Access_ME_Instr = '1' Then
                    StateME_Next <= ME_WaitForIMemAck;
                Else
                    StateME_Next <= Me_Idle;
                End If;
            When ME_WaitForDMemAck =>
                MainMemory_D_Wait  <= '1';
                MainMemory_Running <= '1';
                If (ME_ACK_I = '1') Or (TimeOut_ME = '1') Then
                    MainMemory_D_Wait <= '0';
                    StateME_Next      <= Me_Idle;
                End If;
            When ME_WaitForIMemAck =>
                MainMemory_I_Wait  <= '1';
                MainMemory_I_Lock  <= '1';
                MainMemory_Running <= '1';
                If (ME_ACK_I ='1') Or (TimeOut_ME = '1') Then
                    MainMemory_I_Wait  <= '0';
                    StateME_Next       <= Me_Idle;
                End If;
            When Others =>
                StateME_Next <= Me_Idle;
        End Case;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    WishboneFSM_ME_Registered:
    Process(CLK_I,
            Combined_Rst)
    Begin
       If Rising_Edge(CLK_I) Then
          If Combined_Rst='1' Then
             StateME <= ME_Idle;
          Else
             StateME <= StateME_Next;
          End If;
       End If;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    -- TimeBase Memory Interface;
    ----------------------------------------------------------------------------
    Access_TimeBaseMem_Data  <= (Data_Read  Or Data_Write) And DataAddressIsTimeBaseMemory;
    Access_TimeBaseMem_Instr <= (Instr_Read              ) And InstrAddressIsTimeBaseMemory;
    ----------------------------------------------------------------------------

    Debug_Access_IO_Data  <= Access_TimeBaseMem_Data;
    Debug_Access_IO_Instr <= TimeBaseInterrupt;

    ----------------------------------------------------------------------------
    TimeBaseMemory_Control:
    Process (Data_WriteData,
             Access_TimeBaseMem_Data,
             TimeBaseMem_I_Lock,
             Data_Address,
             Data_Write,
             Instr_Address)
    Begin
        TimeBaseOut <= Data_WriteData;

        If (Access_TimeBaseMem_Data = '1') And (TimeBaseMem_I_Lock = '0') Then
            TimeBaseAddr <= Data_Address(3 Downto 2);
            TimeBaseWE   <= Data_Write;
        Else
            TimeBaseAddr <= Instr_Address(3 Downto 2);
            TimeBaseWE   <= '0';
        End If;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    TimeBaseMemory_Combinational:
    Process (StateTimeBaseMem,
             Access_TimeBaseMem_Data,
             Access_TimeBaseMem_Instr)
    Begin
        TimeBaseMem_I_Lock       <= '0';
        TimeBaseMem_I_Wait       <= Access_TimeBaseMem_Instr;
        TimeBaseMem_D_Wait       <= Access_TimeBaseMem_Data;
        StateTimeBaseMem_Next    <= StateTimeBaseMem;

        Case StateTimeBaseMem Is
            When TimeBaseMem_Idle =>
                If Access_TimeBaseMem_Data = '1' Then
                    StateTimeBaseMem_Next <= TimeBaseMem_WaitForDMemAck;
                ElsIf Access_TimeBaseMem_Instr = '1' Then
                    StateTimeBaseMem_Next <= TimeBaseMem_WaitForIMemAck;
                Else
                    StateTimeBaseMem_Next <= TimeBaseMem_Idle;
                End If;
            When TimeBaseMem_WaitForDMemAck =>
                TimeBaseMem_D_Wait    <= '0';
                StateTimeBaseMem_Next <= TimeBaseMem_Idle;
            When TimeBaseMem_WaitForIMemAck =>
                TimeBaseMem_I_Wait    <= '0';
                TimeBaseMem_I_Lock    <= '1';
                StateTimeBaseMem_Next <= TimeBaseMem_Idle;
            When Others =>
                StateTimeBaseMem_Next <= TimeBaseMem_Idle;
        End Case;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    TimeBaseMemory_Registered:
    Process(CLK_I,
            Combined_Rst)
    Begin
      If Rising_Edge(CLK_I) Then
         If Combined_Rst='1' Then
            StateTimeBaseMem <= TimeBaseMem_Idle;
         Else
            StateTimeBaseMem <= StateTimeBaseMem_Next;
         End If;
      End If;
    End Process;
    ----------------------------------------------------------------------------


    ----------------------------------------------------------------------------
    TimeOutCounter_IO:
    Process(CLK_I,
            TimeOut_IO_Counter)
    Begin
        If Rising_Edge(CLK_I) Then
            If (Combined_Rst = '1') Or ((Peripheral_I_Wait = '0') And (Peripheral_D_Wait = '0')) Then
                TimeOut_IO_Counter <= (Others=>'0') ;
            Else
                TimeOut_IO_Counter <= TimeOut_IO_Counter + 1 ;
            End If ;
        End if ;

        TimeOut_IO <= TimeOut_IO_Counter(12) ;
    End Process ;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    TimeOutCounter_ME:
    Process(CLK_I,
            TimeOut_ME_Counter)
    Begin
        If Rising_Edge(CLK_I) Then
            If (Combined_Rst = '1') Or ((MainMemory_I_Wait = '0') And (MainMemory_D_Wait = '0')) Then
                TimeOut_ME_Counter <= (Others=>'0') ;
            Else
                TimeOut_ME_Counter <= TimeOut_ME_Counter + 1 ;
            End If ;
        End if ;

        TimeOut_ME <= TimeOut_ME_Counter(12) ;
    End Process ;
    ----------------------------------------------------------------------------

------------------------------------------------------------------------------------
End RTL;
------------------------------------------------------------------------------------



