Library ieee;
Use ieee.std_logic_1164.All;

Entity ARM_SYNC Is
    Port (
        CLK_I        : In  Std_Logic ;
        RST_I        : In  Std_Logic ;
        TCK          : In  Std_Logic ;
        TRST         : In  Std_Logic ;
        TMS          : In  Std_Logic ;
        TDI          : In  Std_Logic ;

        ARM_nRESET   : Out Std_Logic ;
        ARM_DBGTCKEN : Out Std_Logic ;
        ARM_DBGTDI   : Out Std_Logic ;
        ARM_DBGTMS   : Out Std_Logic ;
        ARM_DBGnTRST : Out Std_Logic
    );
End ARM_SYNC;

Architecture RTL Of ARM_SYNC Is

    ----------------------------------------------------------------------------
    Component CLKINT
        Port (
            A : In  Std_Logic ;
            Y : Out Std_Logic
        ) ;
    End Component ;
    ----------------------------------------------------------------------------

Signal STARTUP_RST_1, STARTUP_RST_2            : Std_Logic ;
Signal STARTUP_TRST_1, STARTUP_TRST_2          : Std_Logic ;
Signal TCK_1, TCK_2, TCK_3, TCK_2_N3, TCK_N2_3 : Std_Logic ;

Begin

    ----------------------------------------------------------------------------
    -- assign necessary global signals
GLOBAL_ARM_DBGnTRST : CLKINT
    Port Map(
        A => STARTUP_TRST_2,
        Y => ARM_DBGnTRST
    );

GLOBAL_ARM_nRESET : CLKINT
    Port Map(
        A => STARTUP_RST_2,
        Y => ARM_nRESET
    );

GLOBAL_ARM_DBGTCKEN : CLKINT
    Port Map(
        A => TCK_N2_3,
        Y => ARM_DBGTCKEN
    );

    ----------------------------------------------------------------------------
    -- setup reset
SYNC_STARTUP_RST:
    Process (CLK_I,RST_I)
    Begin
        If (RST_I = '1') then
            STARTUP_RST_1 <= '0';
            STARTUP_RST_2 <= '0';
        ElsIf Rising_Edge(CLK_I) Then
            STARTUP_RST_1 <= '1';
            STARTUP_RST_2 <= STARTUP_RST_1;
        End If;
    End Process SYNC_STARTUP_RST;

    ----------------------------------------------------------------------------
    -- setup trst
SYNC_STARTUP_TRST:
    Process (CLK_I,TRST)
    Begin
        If (TRST = '0') Then
            STARTUP_TRST_1 <= '0';
            STARTUP_TRST_2 <= '0';
        ElsIf Rising_Edge(CLK_I) Then
            STARTUP_TRST_1 <= '1';
            STARTUP_TRST_2 <= STARTUP_TRST_1;
        End If;
    End Process SYNC_STARTUP_TRST;

    ----------------------------------------------------------------------------
    -- synchronize tck
SYNC_TCK:
    Process (CLK_I,STARTUP_TRST_2)
    Begin
        If (STARTUP_TRST_2 = '0') Then
            TCK_1 <= '0';
            TCK_2 <= '0';
            TCK_3 <= '0';
        ElsIf Rising_Edge(CLK_I) Then
            TCK_1 <= TCK;
            TCK_2 <= TCK_1;
            TCK_3 <= TCK_2;
        End If;
    End Process Sync_TCK;

    TCK_N2_3 <= (Not (TCK_2) And TCK_3);
    TCK_2_N3 <= (TCK_2 And Not (TCK_3));

    ----------------------------------------------------------------------------
    -- synchronize TMS And TDI
SYNC_TMS_TDI:
    Process (CLK_I,STARTUP_TRST_2)
    Begin
        If (STARTUP_TRST_2 = '0') Then
            ARM_DBGTMS <= '0';
            ARM_DBGTDI <= '0';
        ElsIf Rising_Edge(CLK_I) Then
            If (TCK_2_N3 = '1') Then
                ARM_DBGTMS <= TMS;
                ARM_DBGTDI <= TDI;
            End If;
        End If;
    End Process SYNC_TMS_TDI;

End RTL;

Library ieee;
Use ieee.std_logic_1164.All;

Entity ARM_MEM_CTRL Is
    Port (
        CLK_I      : In  Std_Logic ;
        RST_I      : In  Std_Logic ;

        ME_ACK_I   : In  Std_Logic;
        ME_ADR_O   : Out Std_Logic_Vector(31 DownTo 0);
        ME_CYC_O   : Out Std_Logic;
        ME_DAT_I   : In  Std_Logic_Vector(31 DownTo 0);
        ME_DAT_O   : Out Std_Logic_Vector(31 DownTo 0);
        ME_SEL_O   : Out Std_Logic_Vector( 3 DownTo 0);
        ME_STB_O   : Out Std_Logic;
        ME_WE_O    : Out Std_Logic;
        ME_CLK_O   : Out Std_Logic;
        ME_RST_O   : Out Std_Logic;

        IO_ACK_I   : In  Std_Logic;
        IO_ADR_O   : Out Std_Logic_Vector(23 DownTo 0);
        IO_CYC_O   : Out Std_Logic;
        IO_DAT_I   : In  Std_Logic_Vector(31 DownTo 0);
        IO_DAT_O   : Out Std_Logic_Vector(31 DownTo 0);
        IO_SEL_O   : Out Std_Logic_Vector( 3 DownTo 0);
        IO_STB_O   : Out Std_Logic;
        IO_WE_O    : Out Std_Logic;
        IO_CLK_O   : Out Std_Logic;
        IO_RST_O   : Out Std_Logic;

        INT_ADDR   : Out Std_Logic_Vector(ReplaceWith_BootAddressWidth-1 DownTo 0);
        INT_ByteWE : Out Std_Logic_Vector(3 DownTo 0);
        INT_CLK    : Out Std_Logic;
        INT_DIN    : Out Std_Logic_Vector(31 DownTo 0);
        INT_DOUT   : In  Std_Logic_Vector(31 DownTo 0);
        INT_EN     : Out Std_Logic;
        INT_WE     : Out Std_Logic;

        ARM_ADDR   : In  Std_Logic_Vector(31 DownTo 0);
        ARM_SIZE   : In  Std_Logic_Vector(1 DownTo 0);
        ARM_TRANS  : In  Std_Logic_Vector(1 DownTo 0);
        ARM_WRITE  : In  Std_Logic;
        ARM_RDATA  : Out Std_Logic_Vector(31 DownTo 0);
        ARM_WDATA  : In  Std_Logic_Vector(31 DownTo 0);
        ARM_CLKEN  : Out Std_Logic
    );
End ARM_MEM_CTRL;

Architecture RTL Of ARM_MEM_CTRL Is

    Type STATE_TYPE Is (STATE_IDLE, STATE_READ, STATE_WAIT, STATE_WRITE) ;
    Signal state              : STATE_TYPE ;
    Signal next_state         : STATE_TYPE ;
    Signal state_in_idle      : Std_Logic ;

    Signal WB_ADR_O           : Std_Logic_Vector(31 DownTo 0) ;
    Signal WB_DAT_O           : Std_Logic_Vector(31 DownTo 0) ;
    Signal WB_DAT_I           : Std_Logic_Vector(31 DownTo 0) ;
    Signal WB_SEL_O           : Std_Logic_Vector(3 DownTo 0) ;
    Signal WB_WE_O            : Std_Logic ;
    Signal WB_EN_O            : Std_Logic ;
    Signal WB_ACK_I           : Std_Logic ;

    Signal DEBUG_CLKINT       : Std_Logic ;

    Signal WB_IO              : Std_Logic;
    Signal WB_ME              : Std_Logic;
    Signal WB_INTME           : Std_Logic;

Begin

    ----------------------------------------------------------------------------
    -- only register the address line If we are In idle state
    assign_wb_address :
        Process (CLK_I, RST_I)
        Begin
            If (RST_I = '1') Then
                WB_ADR_O <= (Others => '0') ;
            ElsIf Rising_Edge(CLK_I) Then
                If (state_in_idle = '1') Then
                    WB_ADR_O <= ARM_ADDR ;
                End If ;
            End If ;
        End Process ;

    ----------------------------------------------------------------------------
    -- only register the data Out line If we are In write state
    assign_wb_data_o :
        Process (CLK_I, RST_I, state)
        Begin
            If (RST_I = '1') Then
                WB_DAT_O <= (Others => '0') ;
            ElsIf Rising_Edge(CLK_I) Then
                If (state = STATE_WRITE) Then
                    WB_DAT_O <= ARM_WDATA(7 DownTo 0) & ARM_WDATA(15 DownTo 8) & ARM_WDATA(23 DownTo 16) & ARM_WDATA(31 DownTo 24) ;
                End If ;
            End If ;
        End Process ;

    ----------------------------------------------------------------------------
    -- only register the select line If we are In idle state
    -- assumption: the processor Is In big-endian mode
    assign_wb_sel :
        Process(CLK_I, RST_I)
        Begin
            If (RST_I = '1') Then
                WB_SEL_O <= "1111" ;
            ElsIf Rising_Edge(CLK_I) Then
                If (state_in_idle = '1') Then
                    Case ARM_SIZE Is
                        When "00"   =>
                            Case ARM_ADDR(1 DownTo 0) Is
                                When "00"   => WB_SEL_O <= "1000";
                                When "01"   => WB_SEL_O <= "0100";
                                When "10"   => WB_SEL_O <= "0010";
                                When Others => WB_SEL_O <= "0001";
                            End Case;
                        When "01"   =>
                            Case ARM_ADDR(1) Is
                                When '0'    => WB_SEL_O <= "1100";
                                When Others => WB_SEL_O <= "0011";
                            End Case;
                        When Others => WB_SEL_O <= "1111" ;
                    End Case ;
                End If ;
            End If ;
        End Process ;

    ----------------------------------------------------------------------------
    -- only register the we line If we are In idle state
    assign_we :
        Process(CLK_I, RST_I)
        Begin
            If (RST_I = '1') Then
                WB_WE_O <= '0' ;
            ElsIf Rising_Edge(CLK_I) Then
                If (state_in_idle = '1') Then
                    WB_WE_O <= ARM_WRITE ;
                End If ;
            End If ;
        End Process ;

    ----------------------------------------------------------------------------
    -- assign state to next state
    assign_next_state :
        Process(CLK_I, RST_I)
        Begin
            If RST_I = '1' Then
                state <= STATE_IDLE ;
            ElsIf Rising_Edge(CLK_I) Then
                state <= next_state ;
            End If ;
        End Process ;

    ----------------------------------------------------------------------------
    -- assign the state machine
    assign_state :
        Process(state, ARM_TRANS, ARM_WRITE, WB_ACK_I)
        Begin
            next_state <= state ;
            state_in_idle <= '0' ;
            Case state Is
                When STATE_IDLE  =>
                    state_in_idle <= '1' ;
                    If (ARM_TRANS(1) = '1') Then
                        If (ARM_WRITE = '1') Then
                            next_state <= STATE_WRITE ;
                        Else
                            next_state <= STATE_READ ;
                        End If ;
                    End If ;
                When STATE_READ  =>
                    next_state <= STATE_WAIT ;
                When STATE_WRITE =>
                    next_state <= STATE_WAIT ;
                When STATE_WAIT  =>
                    If (WB_ACK_I = '1') Then
                        next_state <= STATE_IDLE ;
                    End If ;
                When Others =>
                    next_state <= STATE_IDLE ;
            End Case ;
        End Process ;

    ----------------------------------------------------------------------------
    -- assign the value for clock enable
    assign_clken_int :
        Process(state_in_idle)
        Begin
            DEBUG_CLKINT <= '1' ;
            If (state_in_idle = '0') Then
                DEBUG_CLKINT <= '0' ;
            End If ;
        End Process ;
    ----------------------------------------------------------------------------
    -- assign the value for the wb en
    assign_wb_en :
        Process(state)
        Begin
            WB_EN_O <= '0' ;
            Case state Is
                When STATE_READ  =>
                    WB_EN_O <= '1' ;
                When STATE_WAIT  =>
                    WB_EN_O <= '1' ;
                When Others =>
                    WB_EN_O <= '0' ;
            End Case ;
        End Process ;

    ----------------------------------------------------------------------------
    -- mux between the buses
    ME_IO_Bus_Mux :
        Process(WB_ADR_O, WB_WE_O, WB_EN_O, IO_DAT_I, ME_DAT_I, ME_ACK_I, IO_ACK_I)
        Begin
            WB_IO    <= '0';
            WB_ME    <= '0';
            WB_INTME <= '0';
            If WB_ADR_O(31 DownTo 24) = X"FF" Then
                IO_CYC_O  <= WB_EN_O ;
                IO_STB_O  <= WB_EN_O ;

                WB_DAT_I  <= IO_DAT_I ;
                WB_ACK_I  <= IO_ACK_I ;

                INT_EN    <= '0' ;

                ME_CYC_O  <= '0' ;
                ME_STB_O  <= '0' ;
                WB_IO     <= '1';
            ElsIf WB_ADR_O(31 DownTo 24) = X"00" Then
                ME_CYC_O  <= '0' ;
                ME_STB_O  <= '0' ;

                WB_DAT_I  <= INT_DOUT ;
                WB_ACK_I  <= '1' And WB_EN_O ;

                INT_EN    <= WB_EN_O ;

                IO_CYC_O  <= '0' ;
                IO_STB_O  <= '0' ;
                WB_INTME  <= '1';
            Else
                ME_CYC_O  <= WB_EN_O ;
                ME_STB_O  <= WB_EN_O ;

                WB_DAT_I  <= ME_DAT_I ;
                WB_ACK_I  <= ME_ACK_I ;

                INT_EN    <= '0' ;

                IO_CYC_O  <= '0' ;
                IO_STB_O  <= '0' ;
                WB_ME     <= '1';
            End If ;
        End Process ;


    ----------------------------------------------------------------------------
    --
    INT_ADDR      <= WB_ADR_O((ReplaceWith_BootAddressWidth-1)+2 DownTo 2) ;  -- for the memory we always pass word address(msb DownTo 2) And sel lines are for byte address
    INT_ByteWE(3) <= WB_SEL_O(3) And WB_INTME ;
    INT_ByteWE(2) <= WB_SEL_O(2) And WB_INTME ;
    INT_ByteWE(1) <= WB_SEL_O(1) And WB_INTME ;
    INT_ByteWE(0) <= WB_SEL_O(0) And WB_INTME ;
    INT_CLK       <= CLK_I ;
    INT_DIN       <= WB_DAT_O ;
    INT_WE        <= WB_WE_O And WB_EN_O And WB_INTME;

    ----------------------------------------------------------------------------
    -- output the ME bus
    ME_WE_O      <= WB_WE_O And WB_EN_O And WB_ME;
    ME_SEL_O     <= WB_SEL_O ;
    ME_DAT_O     <= WB_DAT_O ;
    ME_ADR_O     <= WB_ADR_O ;
    ME_CLK_O     <= CLK_I ;
    ME_RST_O     <= RST_I ;

    ----------------------------------------------------------------------------
    -- output the IO bus
    IO_WE_O      <= WB_WE_O And WB_EN_O And WB_IO;
    IO_SEL_O     <= WB_SEL_O ;
    IO_DAT_O     <= WB_DAT_O ;
    IO_ADR_O     <= WB_ADR_O(23 DownTo 0) ;
    ARM_CLKEN    <= DEBUG_CLKINT ;
    IO_CLK_O     <= CLK_I ;
    IO_RST_O     <= RST_I ;

    ARM_RDATA <= WB_DAT_I(7 DownTo 0) & WB_DAT_I(15 DownTo 8) & WB_DAT_I(23 DownTo 16) & WB_DAT_I(31 DownTo 24) ;

End RTL ;

Library ieee;
Use ieee.std_logic_1164.All;

Entity Configurable_ReplaceWith_Designator Is

    Port(
    ----------------------------------------------------------------------------
    CLK_I    : In  Std_Logic;
    RST_I    : In  Std_Logic;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    TCK      : In  Std_Logic;
    TDI      : In  Std_Logic;
    TMS      : In  Std_Logic;
    TRST     : In  Std_Logic;
    TDO      : Out Std_Logic;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    IO_ACK_I : In  Std_Logic;
    IO_ADR_O : Out Std_Logic_Vector(23 DownTo 0);
    IO_CYC_O : Out Std_Logic;
    IO_DAT_I : In  Std_Logic_Vector(31 DownTo 0);
    IO_DAT_O : Out Std_Logic_Vector(31 DownTo 0);
    IO_SEL_O : Out Std_Logic_Vector( 3 DownTo 0);
    IO_STB_O : Out Std_Logic;
    IO_WE_O  : Out Std_Logic;
    IO_CLK_O : Out Std_Logic;
    IO_RST_O : Out Std_Logic;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    ME_ACK_I : In  Std_Logic;
    ME_ADR_O : Out Std_Logic_Vector(31 DownTo 0);
    ME_CYC_O : Out Std_Logic;
    ME_DAT_I : In  Std_Logic_Vector(31 DownTo 0);
    ME_DAT_O : Out Std_Logic_Vector(31 DownTo 0);
    ME_SEL_O : Out Std_Logic_Vector( 3 DownTo 0);
    ME_STB_O : Out Std_Logic;
    ME_WE_O  : Out Std_Logic;
    ME_CLK_O : Out Std_Logic;
    ME_RST_O : Out Std_Logic;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    INT_I    : In  Std_Logic_Vector(31 DownTo 0)
    ----------------------------------------------------------------------------
    );
End Configurable_ReplaceWith_Designator;

Architecture RTL Of Configurable_ReplaceWith_Designator Is
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    Component CLKINT
    Port (
        A : In  Std_Logic ;
        Y : Out Std_Logic
    ) ;
    End Component ;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    Component ARM_SYNC Is
    Port (
        CLK_I        : In  Std_Logic ;
        RST_I        : In  Std_Logic ;
        TCK          : In  Std_Logic ;
        TRST         : In  Std_Logic ;
        TMS          : In  Std_Logic ;
        TDI          : In  Std_Logic ;

        ARM_nRESET   : Out Std_Logic ;
        ARM_DBGTCKEN : Out Std_Logic ;
        ARM_DBGTDI   : Out Std_Logic ;
        ARM_DBGTMS   : Out Std_Logic ;
        ARM_DBGnTRST : Out Std_Logic
    );
    End Component;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    Component ARM_MEM_CTRL Is
    Port (
        CLK_I      : In  Std_Logic ;
        RST_I      : In  Std_Logic ;

        ME_ACK_I   : In  Std_Logic;
        ME_ADR_O   : Out Std_Logic_Vector(31 DownTo 0);
        ME_CYC_O   : Out Std_Logic;
        ME_DAT_I   : In  Std_Logic_Vector(31 DownTo 0);
        ME_DAT_O   : Out Std_Logic_Vector(31 DownTo 0);
        ME_SEL_O   : Out Std_Logic_Vector( 3 DownTo 0);
        ME_STB_O   : Out Std_Logic;
        ME_WE_O    : Out Std_Logic;
        ME_CLK_O   : Out Std_Logic;
        ME_RST_O   : Out Std_Logic;

        IO_ACK_I   : In  Std_Logic;
        IO_ADR_O   : Out Std_Logic_Vector(23 DownTo 0);
        IO_CYC_O   : Out Std_Logic;
        IO_DAT_I   : In  Std_Logic_Vector(31 DownTo 0);
        IO_DAT_O   : Out Std_Logic_Vector(31 DownTo 0);
        IO_SEL_O   : Out Std_Logic_Vector( 3 DownTo 0);
        IO_STB_O   : Out Std_Logic;
        IO_WE_O    : Out Std_Logic;
        IO_CLK_O   : Out Std_Logic;
        IO_RST_O   : Out Std_Logic;

        INT_ADDR   : Out Std_Logic_Vector(ReplaceWith_BootAddressWidth-1 DownTo 0);
        INT_ByteWE : Out Std_Logic_Vector(3 DownTo 0);
        INT_CLK    : Out Std_Logic;
        INT_DIN    : Out Std_Logic_Vector(31 DownTo 0);
        INT_DOUT   : In  Std_Logic_Vector(31 DownTo 0);
        INT_EN     : Out Std_Logic;
        INT_WE     : Out Std_Logic;

        ARM_ADDR   : In  Std_Logic_Vector(31 DownTo 0) ;
        ARM_SIZE   : In  Std_Logic_Vector(1 DownTo 0) ;
        ARM_TRANS  : In  Std_Logic_Vector(1 DownTo 0) ;
        ARM_WRITE  : In  Std_Logic ;
        ARM_RDATA  : Out Std_Logic_Vector(31 DownTo 0) ;
        ARM_WDATA  : In  Std_Logic_Vector(31 DownTo 0) ;
        ARM_CLKEN  : Out Std_Logic
    );
    End Component;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    Component Memory_ReplaceWith_Designator
       Port
       (
         ADDR   : In  Std_Logic_Vector(ReplaceWith_BootAddressWidth-1 DownTo 0);
         ByteWE : In  Std_Logic_Vector(3 DownTo 0);
         CLK    : In  Std_Logic;
         DIN    : In  Std_Logic_Vector(31 DownTo 0);
         DOUT   : Out Std_Logic_Vector(31 DownTo 0);
         EN     : In  Std_Logic;
         WE     : In  Std_Logic
       );
    End Component;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    Component A7S
        Port(
            ABORT         : In  Std_Logic;
            CFGBIGEND     : In  Std_Logic;
            CLK           : In  Std_Logic;
            CLKEN         : In  Std_Logic;
            CPA           : In  Std_Logic;
            CPB           : In  Std_Logic;
            DBGBREAK      : In  Std_Logic;
            DBGEN         : In  Std_Logic;
            DBGEXT        : In  Std_Logic_Vector(1 DownTo 0);
            DBGRQ         : In  Std_Logic;
            DBGTCKEN      : In  Std_Logic;
            DBGTDI        : In  Std_Logic;
            DBGTMS        : In  Std_Logic;
            DBGnTRST      : In  Std_Logic;
            RDATA         : In  Std_Logic_Vector(31 DownTo 0);
            nFIQ          : In  Std_Logic;
            nIRQ          : In  Std_Logic;
            nRESET        : In  Std_Logic;
            ADDR          : Out Std_Logic_Vector(31 DownTo 0);
            CPSEQ         : Out Std_Logic;
            CPTBIT        : Out Std_Logic;
            CPnI          : Out Std_Logic;
            CPnMREQ       : Out Std_Logic;
            CPnOPC        : Out Std_Logic;
            CPnTRANS      : Out Std_Logic;
            DBGACK        : Out Std_Logic;
            DBGCOMMRX     : Out Std_Logic;
            DBGCOMMTX     : Out Std_Logic;
            DBGINSTRVALID : Out Std_Logic;
            DBGRNG        : Out Std_Logic_Vector(1 DownTo 0);
            DBGTDO        : Out Std_Logic;
            DBGnEXEC      : Out Std_Logic;
            DBGnTDOEN     : Out Std_Logic;
            DMORE         : Out Std_Logic;
            LOCK          : Out Std_Logic;
            PROT          : Out Std_Logic_Vector(1 DownTo 0);
            SIZE          : Out Std_Logic_Vector(1 DownTo 0);
            TRANS         : Out Std_Logic_Vector(1 DownTo 0);
            WDATA         : Out Std_Logic_Vector(31 DownTo 0);
            WRITE         : Out Std_Logic
        );
    End Component;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    Signal G_CLK_I      : Std_Logic ;

    Signal ARM_nRESET   : Std_Logic ;
    Signal ARM_DBGTCKEN : Std_Logic ;
    Signal ARM_DBGTDI   : Std_Logic ;
    Signal ARM_DBGTMS   : Std_Logic ;
    Signal ARM_DBGnTRST : Std_Logic ;

    Signal ARM_ADDR     : Std_Logic_Vector(31 DownTo 0) ;
    Signal ARM_SIZE     : Std_Logic_Vector(1 DownTo 0) ;
    Signal ARM_TRANS    : Std_Logic_Vector(1 DownTo 0) ;
    Signal ARM_WRITE    : Std_Logic ;
    Signal ARM_RDATA    : Std_Logic_Vector(31 DownTo 0) ;
    Signal ARM_WDATA    : Std_Logic_Vector(31 DownTo 0) ;
    Signal ARM_CLKEN    : Std_Logic ;

    Signal INT_ADDR     : Std_Logic_Vector(ReplaceWith_BootAddressWidth-1 DownTo 0);
    Signal INT_ByteWE   : Std_Logic_Vector(3 DownTo 0);
    Signal INT_CLK      : Std_Logic;
    Signal INT_DIN      : Std_Logic_Vector(31 DownTo 0);
    Signal INT_DOUT     : Std_Logic_Vector(31 DownTo 0);
    Signal INT_EN       : Std_Logic;
    Signal INT_WE       : Std_Logic;

Begin

    GLOBAL_CLK_I : CLKINT
    Port Map(
        A => CLK_I,
        Y => G_CLK_I
    );

    CoreMP7_00 : A7S
        Port Map(
            ABORT         => '0',
            CFGBIGEND     => '0',
            CLK           => G_CLK_I,
            CLKEN         => ARM_CLKEN,
            CPA           => '1',
            CPB           => '1',
            DBGBREAK      => '0',
            DBGEN         => '1',
            DBGEXT        => (Others => '0'),
            DBGRQ         => '0',
            DBGTCKEN      => ARM_DBGTCKEN,
            DBGTDI        => ARM_DBGTDI,
            DBGTMS        => ARM_DBGTMS,
            DBGnTRST      => ARM_DBGnTRST,
            RDATA         => ARM_RDATA,
            nFIQ          => '1',
            nIRQ          => Not INT_I(0),
            nRESET        => ARM_nRESET,
            ADDR          => ARM_ADDR,
            CPSEQ         => open,
            CPTBIT        => open,
            CPnI          => open,
            CPnMREQ       => open,
            CPnOPC        => open,
            CPnTRANS      => open,
            DBGACK        => open,
            DBGCOMMRX     => open,
            DBGCOMMTX     => open,
            DBGINSTRVALID => open,
            DBGRNG        => open,
            DBGTDO        => TDO,
            DBGnEXEC      => open,
            DBGnTDOEN     => open,
            DMORE         => open,
            LOCK          => open,
            PROT          => open,
            SIZE          => ARM_SIZE,
            TRANS         => ARM_TRANS,
            WDATA         => ARM_WDATA,
            WRITE         => ARM_WRITE
        ) ;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    ARM_SYNC_inst : ARM_SYNC
        Port Map (
            CLK_I        => G_CLK_I,
            RST_I        => RST_I,
            TCK          => TCK,
            TRST         => TRST,
            TMS          => TMS,
            TDI          => TDI,
            ARM_nRESET   => ARM_nRESET,
            ARM_DBGTCKEN => ARM_DBGTCKEN,
            ARM_DBGTDI   => ARM_DBGTDI,
            ARM_DBGTMS   => ARM_DBGTMS,
            ARM_DBGnTRST => ARM_DBGnTRST
        ) ;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    ARM_MEM_CTRL_inst : ARM_MEM_CTRL
    Port Map (
        CLK_I      => G_CLK_I    ,
        RST_I      => RST_I      ,

        ME_ACK_I   => ME_ACK_I   ,
        ME_ADR_O   => ME_ADR_O   ,
        ME_CYC_O   => ME_CYC_O   ,
        ME_DAT_I   => ME_DAT_I   ,
        ME_DAT_O   => ME_DAT_O   ,
        ME_SEL_O   => ME_SEL_O   ,
        ME_STB_O   => ME_STB_O   ,
        ME_WE_O    => ME_WE_O    ,
        ME_CLK_O   => ME_CLK_O   ,
        ME_RST_O   => ME_RST_O   ,

        IO_ACK_I   => IO_ACK_I   ,
        IO_ADR_O   => IO_ADR_O   ,
        IO_CYC_O   => IO_CYC_O   ,
        IO_DAT_I   => IO_DAT_I   ,
        IO_DAT_O   => IO_DAT_O   ,
        IO_SEL_O   => IO_SEL_O   ,
        IO_STB_O   => IO_STB_O   ,
        IO_WE_O    => IO_WE_O    ,
        IO_CLK_O   => IO_CLK_O   ,
        IO_RST_O   => IO_RST_O   ,

        INT_ADDR   => INT_ADDR   ,
        INT_ByteWE => INT_ByteWE ,
        INT_CLK    => INT_CLK    ,
        INT_DIN    => INT_DIN    ,
        INT_DOUT   => INT_DOUT   ,
        INT_EN     => INT_EN     ,
        INT_WE     => INT_WE     ,

        ARM_ADDR   => ARM_ADDR   ,
        ARM_SIZE   => ARM_SIZE   ,
        ARM_TRANS  => ARM_TRANS  ,
        ARM_WRITE  => ARM_WRITE  ,
        ARM_RDATA  => ARM_RDATA  ,
        ARM_WDATA  => ARM_WDATA  ,
        ARM_CLKEN  => ARM_CLKEN
    );
    ----------------------------------------------------------------------------

    INT_MEMORY_inst : Memory_ReplaceWith_Designator
    Port Map (
        ADDR   => INT_ADDR  ,
        ByteWE => INT_ByteWE,
        CLK    => INT_CLK   ,
        DIN    => INT_DIN   ,
        DOUT   => INT_DOUT  ,
        EN     => INT_EN    ,
        WE     => INT_WE
    ) ;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
End RTL;





