/**************************************************************************
**                                                                        *
**  FILE        :  bounds.c                                               *
**                                                                        *
**  DESCRIPTION :  Support functions for run-time bounds checking.        *
**                                                                        *
**  Pointer updates resulting in an out-of-bounds address are not         *
**  directly flagged as errors, although their behavior is undefined.     *
**  Instead, a special "illegal" address is returned that triggers an     *
**  error on the next pointer operation. Out-of-bounds pointers may occur *
**  when post-increment with a value larger than 1 or post-decrement are  *
**  being used. When this module is compiled with -DBOUNDS_STRICT,        *
**  pointer updates will be checked more strictly.                        *
**                                                                        *
**  Copyright 1996-2010 Altium BV                                         *
**                                                                        *
**************************************************************************/

#include <stddef.h>
#include <stdlib.h>
#include "runtime-error.h"
#include "bounds-error.h"
#include "bounds.h"

#pragma runtime -malloc,-bounds

extern  __bounds_mem_t  _lc_ub_bounds[0x10001];         /* list of static bounds */
extern  __bounds_mem_t  _lc_ue_bounds[0x10001];

volatile char   __bounds_uninit;        /* for uninitialized pointers */
#ifndef BOUNDS_STRICT
volatile char   __bounds_illegal;       /* for out-of-bounds pointers */
#endif

/* --------------------------------------------------------------------- */

/*
 * splay tree
 *
 * Based on public domain code from Daniel Sleator <sleator@cs.cmu.edu>.
 */

const   int             chunk_size = 50;

typedef struct splay_s  splay_t;

struct  splay_s
{
        splay_t *       left;
        splay_t *       right;
#pragma warning 525
        __bounds_t      low;
        __bounds_t      high;
#pragma warning restore
};

static  splay_t         null;           /* the NULL pointer [0,0] */
static  splay_t *       tree = & null;
static  splay_t *       freelist;
static  splay_t *       chunk_begin;
static  splay_t *       chunk_end;
static  char            init_done;

static  splay_t *       splay_new ( void )
{
        splay_t *       s;

        if      ( freelist )
        {       /* recyle */
                s = freelist;
                freelist = freelist->left;
        }
        else
        {
                if      ( chunk_begin == chunk_end )
                {       /* start a new chunk */
                        chunk_begin =
                           (splay_t *) malloc( chunk_size * sizeof(splay_t) );
                        if      ( ! chunk_begin )
                        {
                                __runtime_error( "out of memory in bounds checker" );
                                exit( 1 );
                        }
                        chunk_end = chunk_begin + chunk_size;
                }
                s = chunk_begin++;
        }
        return s;
}

static  void            splay_del ( splay_t * s )
{
        s->left = freelist;
        freelist = s;
}

static  void            splay_lookup ( __bounds_t addr )
{
        splay_t         tmp;
        splay_t *       l;
        splay_t *       r;
        splay_t *       y;

        if      ( ! tree )
        {
                return;
        }
        tmp.left = NULL;
        tmp.right = NULL;
        l = & tmp;
        r = & tmp;
        for     ( ;; )
        {
                if      ( addr < tree->low )
                {
                        if      ( ! tree->left )
                        {
                                break;
                        }
                        if      ( addr < tree->left->low )
                        {       /* rotate right */
                                y = tree->left;
                                tree->left = y->right;
                                y->right = tree;
                                tree = y;
                                if      ( ! tree->left )
                                {
                                        break;
                                }
                        }
                        /* link right */
                        r->left = tree;
                        r = tree;
                        tree = tree->left;
                }
                else if ( addr > tree->high )
                {
                        if      ( ! tree->right )
                        {
                                break;
                        }
                        if      ( addr > tree->right->high )
                        {       /* rotate left */
                                y = tree->right;
                                tree->right = y->left;
                                y->left = tree;
                                tree = y;
                                if      ( ! tree->right )
                                {
                                        break;
                                }
                        }
                        /* link left */
                        l->right = tree;
                        l = tree;
                        tree = tree->right;
                }
                else
                {
                        break;
                }
        }
        /* assemble */
        l->right = tree->left;
        r->left = tree->right;
        tree->left = tmp.right;
        tree->right = tmp.left;
}

static  void            splay_delete ( __bounds_t addr )
{
        splay_t *       d;

        splay_lookup( addr );
        if      ( ! tree || addr != tree->low )
        {
                return; /* shouldn't happen */
        }
        d = tree;
        if      ( ! tree->left )
        {
                tree = tree->right;
        }
        else
        {
                tree = tree->left;
                splay_lookup( addr );
                tree->right = d->right;
        }
        splay_del( d );
}

static  void            splay_insert ( __bounds_t low, __bounds_t high )
{
        splay_t *       l;
        splay_t *       r;

        l = NULL;
        r = NULL;
        if      ( tree )
        {
                splay_lookup( low );
                if      ( high < tree->low )
                {
                        l = tree->left;
                        r = tree;
                        tree->left = NULL;
                }
                else if ( low > tree->high )
                {
                        l = tree;
                        r = tree->right;
                        tree->right = NULL;
                }
                else if ( low == tree->low && high == tree->high )
                {       /* allow multiple identical ranges caused by overlaying in the linker */
                        return;
                }
                else
                {
                        __runtime_error( "ignored overlapping ranges in bounds checker" );
                        splay_delete( tree->low );
                        return; /* ignore this range */
                }
        }
        tree = splay_new();
        tree->left  = l;
        tree->right = r;
        tree->low   = low;
        tree->high  = high;
}

/* --------------------------------------------------------------------- */

static  void            __bounds_init ( void )
{
        __bounds_mem_t* p;
        __bounds_mem_t* end;

        splay_insert( (__bounds_t) & __bounds_uninit, (__bounds_t) & __bounds_uninit );
#ifndef BOUNDS_STRICT
        splay_insert( (__bounds_t) & __bounds_illegal, (__bounds_t) & __bounds_illegal );
#endif
        p   = _lc_ub_bounds;
        end = _lc_ue_bounds;
        while   ( p < end )
        {
                splay_insert( p[0], p[1] );
                p += 2;
        }
        init_done = 1;
}

static  _Bool           __bounds_check ( __bounds_t addr1, __bounds_t addr2 )
{
        if      ( ! init_done )
        {
                __bounds_init();
        }
        splay_lookup( addr1 );
        return  (  (addr1 >= tree->low && addr1 <= tree->high)
                && (addr2 <  tree->low || addr2 >  tree->high)
                );
}

/* --------------------------------------------------------------------- */

extern  void            __bounds_new ( __bounds_t addr, size_t size )
{
        if      ( ! init_done )
        {
                __bounds_init();
        }
        SAVE_CALLER;
        splay_insert( addr, addr + size );
        CLEAR_CALLER;
}

extern  void            __bounds_del ( __bounds_t addr )
{
        splay_delete( addr );
}

extern  __bounds_t      __bounds_inc ( __bounds_t addr, __bounds_off_t offset )
{
        __bounds_t      new_addr;

        new_addr = addr + offset;
        if      ( __bounds_check( addr, new_addr ) )
        {       /* invalid pointer update */
#ifndef BOUNDS_STRICT
                if      ( tree->low != tree->high )
                {       /* postpone error until next operation */
                        return (__bounds_t) & __bounds_illegal;
                }
#endif
                SAVE_CALLER;
                __bounds_error( 1, tree->low, tree->high, new_addr, NULL );
                CLEAR_CALLER;
        }
        return new_addr;
}

extern  __bounds_t      __bounds_dec ( __bounds_t addr, __bounds_off_t offset )
{
        __bounds_t      new_addr;

        new_addr = addr - offset;
        if      ( __bounds_check( addr, new_addr ) )
        {       /* invalid pointer update */
#ifndef BOUNDS_STRICT
                if      ( tree->low != tree->high )
                {       /* postpone error until next operation */
                        return (__bounds_t) & __bounds_illegal;
                }
#endif
                SAVE_CALLER;
                __bounds_error( 1, tree->low, tree->high, new_addr, NULL );
                CLEAR_CALLER;
        }
        return new_addr;
}

extern  void            __bounds_ref ( __bounds_t addr, size_t size )
{
        if      ( __bounds_check( addr, addr + size ) )
        {       /* access beyond end of object */
                SAVE_CALLER;
                __bounds_error( 2, tree->low, tree->high, NULL, NULL );
                CLEAR_CALLER;
        }
}

extern  void            __bounds_cmp ( __bounds_t addr1, __bounds_t addr2 )
{
        if      (  __bounds_check( addr1, addr2 )
                || __bounds_check( addr2, addr1 )
                )
        {       /* comparing pointers to different objects */
                SAVE_CALLER;
                __bounds_error( 3, NULL, NULL, addr1, addr2 );
                CLEAR_CALLER;
        }
}

extern  void            __bounds_sub ( __bounds_t addr1, __bounds_t addr2 )
{
        if      (  __bounds_check( addr1, addr2 )
                || __bounds_check( addr2, addr1 )
                )
        {       /* subtracting pointers to different objects */
                SAVE_CALLER;
                __bounds_error( 4, NULL, NULL, addr1, addr2 );
                CLEAR_CALLER;
        }
}
