-- ----------------------------------------------------------------------------
Library IEEE;
Use     IEEE.Std_Logic_1164.All;
Use     IEEE.Std_Logic_Unsigned.All;
Use     IEEE.Std_Logic_Arith.All;
-- ----------------------------------------------------------------------------

-- ----------------------------------------------------------------------------
Entity ReplaceWith_Designator Is
    Port (
        -- --------------------------------------------------------------------
        ReplaceWith_SRAMEntityPorts

        ReplaceWith_SDRAMEntityPorts

        ReplaceWith_FLASHEntityPorts

        ReplaceWith_USBEntityPorts


        -- --------------------------------------------------------------------
        --  {ReplaceWith_TopLevelMemoryPorts} Start
        -- External Memory interface
        MEM_A               : Out   Std_Logic_Vector(23 Downto 0);
        MEM_BE              : Out   Std_Logic_Vector( 3 Downto 0);
        MEM_D               : InOut Std_Logic_Vector(31 Downto 0);
        MEM_FLASH_BUSY      : In    Std_Logic;
        MEM_FLASH_E         : Out   Std_Logic;
        MEM_FLASH_RESET     : Out   Std_Logic;
        MEM_OE              : Out   Std_Logic;
        MEM_SDRAM_CAS       : Out   Std_Logic;
        MEM_SDRAM_CKE       : Out   Std_Logic;
        MEM_SDRAM_E         : Out   Std_Logic;
        MEM_SDRAM_RAS       : Out   Std_Logic;
        MEM_SRAM_E          : Out   Std_Logic;
        MEM_W               : Out   Std_Logic;
        MEM_USB_E           : Out   Std_Logic;
        MEM_USB_RESET       : Out   Std_Logic;
        --  {ReplaceWith_TopLevelMemoryPorts} End
        -- --------------------------------------------------------------------

        MEM_CLK             : In Std_Logic
    );
End;
-- ----------------------------------------------------------------------------

-- ----------------------------------------------------------------------------
Architecture RTL Of ReplaceWith_Designator Is
-- ----------------------------------------------------------------------------

    -- ------------------------------------------------------------------------
    Component WB_SHARED_SDRAM Is
        Port (
            -- Timing configuration
            CONFIG_BURST_LENGTH                  : In  Std_Logic_Vector( 2 Downto 0);
            CONFIG_TIMING_TCL                    : In  Std_Logic_Vector( 1 Downto 0);
            CONFIG_TIMING_TMRD                   : In  Std_Logic_Vector( 2 Downto 0);
            CONFIG_TIMING_TRCD                   : In  Std_Logic_Vector( 2 Downto 0);
            CONFIG_TIMING_TRFC                   : In  Std_Logic_Vector( 2 Downto 0);
            CONFIG_TIMING_TRP                    : In  Std_Logic_Vector( 2 Downto 0);
            CONFIG_TIMING_TWR                    : In  Std_Logic_Vector( 2 Downto 0);
            REFRESH_INTERVAL                     : In  Std_Logic_Vector( 2 Downto 0);
            REFRESH_PRESCALER                    : In  Std_Logic_Vector( 7 Downto 0);

            --SRAM interface
            CONTROL_WB_SRAM_ACCESS_GRANTED       : Out Std_Logic;
            CONTROL_WB_SRAM_ACCESS_REQUEST       : In  Std_Logic;
            CONTROL_WB_SRAM_ADR                  : In  Std_Logic_Vector(23 Downto 0);
            CONTROL_WB_SRAM_BE                   : In  Std_Logic_Vector( 3 Downto 0);
            CONTROL_WB_SRAM_D_EN                 : In  Std_Logic_Vector(31 Downto 0);
            CONTROL_WB_SRAM_DAT_IN               : In  Std_Logic_Vector(31 Downto 0);
            CONTROL_WB_SRAM_DAT_OUT              : Out Std_Logic_Vector(31 Downto 0);
            CONTROL_WB_SRAM_OE                   : In  Std_Logic;
            CONTROL_WB_SRAM_TRANSFER             : In  Std_Logic;
            CONTROL_WB_SRAM_TRANSFER_DONE        : In  Std_Logic;
            CONTROL_WB_SRAM_W                    : In  Std_Logic;

            -- SDRAM interface
            CONTROL_WB_SDRAM_ACKNOWLEDGE         : In  Std_Logic;
            CONTROL_WB_SDRAM_BANK_ADDRESS        : In  Std_Logic_Vector( 1 Downto 0);
            CONTROL_WB_SDRAM_COLUMN_ADDRESS      : In  Std_Logic_Vector(12 Downto 0);
            CONTROL_WB_SDRAM_DATA_FROM_MEM       : Out Std_Logic_Vector(31 Downto 0);
            CONTROL_WB_SDRAM_DATA_TO_MEM         : In  Std_Logic_Vector(31 Downto 0);
            CONTROL_WB_SDRAM_MEM_BE              : In  Std_Logic_Vector( 3 Downto 0);
            CONTROL_WB_SDRAM_REQUEST_READ        : In  Std_Logic;
            CONTROL_WB_SDRAM_REQUEST_TRANSFER    : In  Std_Logic;
            CONTROL_WB_SDRAM_ROW_ADDRESS         : In  Std_Logic_Vector(12 Downto 0);
            CONTROL_WB_SDRAM_SDRAM_MODE_REGISTER : In  Std_Logic_Vector(12 Downto 0);
            CONTROL_WB_SDRAM_TRANSFER_DONE       : Out Std_Logic;

            -- FLASH interface
            CONTROL_WB_FLASH_ACCESS_GRANTED      : Out Std_Logic;
            CONTROL_WB_FLASH_ACCESS_REQUEST      : In  Std_Logic;
            CONTROL_WB_FLASH_ADR                 : In  Std_Logic_Vector(23 Downto 0);
            CONTROL_WB_FLASH_BE                  : In  Std_Logic_Vector( 3 Downto 0);
            CONTROL_WB_FLASH_BUSY                : Out Std_Logic;
            CONTROL_WB_FLASH_D_EN                : In  Std_Logic_Vector(31 Downto 0);
            CONTROL_WB_FLASH_DAT_IN              : In  Std_Logic_Vector(31 Downto 0);
            CONTROL_WB_FLASH_DAT_OUT             : Out Std_Logic_Vector(31 Downto 0);
            CONTROL_WB_FLASH_OE                  : In  Std_Logic;
            CONTROL_WB_FLASH_RESET               : In  Std_Logic;
            CONTROL_WB_FLASH_TRANSFER            : In  Std_Logic;
            CONTROL_WB_FLASH_TRANSFER_DONE       : In  Std_Logic;
            CONTROL_WB_FLASH_W                   : In  Std_Logic;

            -- External Memory interface
            MEM_A                                : Out Std_Logic_Vector(23 Downto 0);
            MEM_BE                               : Out Std_Logic_Vector( 3 Downto 0);
            MEM_CLK                              : In  Std_Logic;
            MEM_D_EN                             : Out Std_Logic_Vector(31 Downto 0);
            MEM_D_IN                             : In  Std_Logic_Vector(31 Downto 0);
            MEM_D_OUT                            : Out Std_Logic_Vector(31 Downto 0);
            MEM_FLASH_BUSY                       : In  Std_Logic;
            MEM_FLASH_E                          : Out Std_Logic;
            MEM_FLASH_RESET                      : Out Std_Logic;
            MEM_OE                               : Out Std_Logic;
            MEM_RST                              : In  Std_Logic;
            MEM_SDRAM_CAS                        : Out Std_Logic;
            MEM_SDRAM_CKE                        : Out Std_Logic;
            MEM_SDRAM_E                          : Out Std_Logic;
            MEM_SDRAM_RAS                        : Out Std_Logic;
            MEM_SRAM_E                           : Out Std_Logic;
            MEM_W                                : Out Std_Logic
        );
    End Component;
    -- ------------------------------------------------------------------------


    -- ------------------------------------------------------------------------
    -- External Memory interface    - Static
    -- ------------------------------------------------------------------------
    Signal Port_Mem_d_en                  : Std_Logic_Vector(31 Downto 0);
    Signal Port_Mem_d_in                  : Std_Logic_Vector(31 Downto 0);
    Signal Port_Mem_d_out                 : Std_Logic_Vector(31 Downto 0);
    Signal MEM_SRAM_E_Signal              : Std_Logic;


    Signal PortSramControl_AccessRequest     : Std_Logic;
    Signal PortSramControl_SramAccessGranted : Std_Logic;
    Signal PortSramPort_MemoryAddress        : Std_Logic_Vector(23 Downto 0);
    Signal PortSramPort_MemoryByteEnable     : Std_Logic_Vector( 3 Downto 0);
    Signal PortSramPort_MemoryDataEnable     : Std_Logic_Vector(31 Downto 0);
    Signal PortSramPort_MemoryDataOutput     : Std_Logic_Vector(31 Downto 0);
    Signal PortSramPort_MemoryDataInput      : Std_Logic_Vector(31 Downto 0);
    Signal PortSramPort_MemoryOutputEnable   : Std_Logic;
    Signal PortSramControl_Transfer          : Std_Logic;
    Signal PortSramControl_TransferFinished  : Std_Logic;
    Signal PortSramPort_MemoryWrite          : Std_Logic;
    Signal EnableUsb                         : Std_Logic;
    Signal EnableSram                        : Std_Logic;
    Signal EnableUsbOE : Std_Logic;

    -- ------------------------------------------------------------------------

    -- ------------------------------------------------------------------------
    -- Dynamic Declarations
    -- ------------------------------------------------------------------------

    -- ------------------------------------------------------------------------
    -- SDRAM Declarations - Dynamic
    -- ------------------------------------------------------------------------

    ReplaceWith_SDRAMDeclareSignals

    -- ------------------------------------------------------------------------

    -- ------------------------------------------------------------------------
    -- SRAM Declarations - Dynamic
    -- ------------------------------------------------------------------------
    Type TSramState Is (
        SramState_AddressSetup,
        SramState_Write_Cycle0_Pulse,
        SramState_Write_Cycle0_Wait,
        SramState_Write_Cycle1_Setup,
        SramState_Write_Cycle1_Pulse,

        SramState_Read_Cycle0_Latch,
        SramState_Read_Cycle1_Setup,

        SramState_Write_Pulse,
        SramState_Write_ACK,

        SramState_Read_ACK,
        SramState_WaitForAck
    );

    Signal SramState      : TSramState;
    Signal SramState_Next : TSramState;

    SubType TSramByteLane Is Std_Logic_Vector(3 Downto 0);
    SubType TSramOutBits  Is Std_Logic_Vector(4 Downto 0);

    Constant cSramWrite_Word                 : TSramByteLane := "1111";
    Constant cSramWrite_LoHalf               : TSramByteLane := "0011";
    Constant cSramWrite_HiHalf               : TSramByteLane := "1100";
    Constant cSramWrite_Byte0                : TSramByteLane := "0001";
    Constant cSramWrite_Byte1                : TSramByteLane := "0010";
    Constant cSramWrite_Byte2                : TSramByteLane := "0100";
    Constant cSramWrite_Byte3                : TSramByteLane := "1000";
    Constant cSramWrite_Bytes012             : TSramByteLane := "0111";
    Constant cSramWrite_Bytes123             : TSramByteLane := "1110";
    Constant cSramWrite_Bytes12              : TSramByteLane := "0110";
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    Constant cSramOutBits_Write_Word         : TSramOutBits  := "00000";

    Constant cSramOutBits_Write_LoHalf       : TSramOutBits  := "11000";
    Constant cSramOutBits_Write_HiHalf       : TSramOutBits  := "00110";
    Constant cSramOutBits_Write_Byte0        : TSramOutBits  := "11100";
    Constant cSramOutBits_Write_Byte1        : TSramOutBits  := "11010";
    Constant cSramOutBits_Write_Byte2        : TSramOutBits  := "10110";
    Constant cSramOutBits_Write_Byte3        : TSramOutBits  := "01110";
    Constant cSramOutBits_Write_Bytes012     : TSramOutBits  := "10000";
    Constant cSramOutBits_Write_Bytes123     : TSramOutBits  := "00010";
    Constant cSramOutBits_Write_Bytes12      : TSramOutBits  := "10010";

    Constant cSramOutBits_Read               : TSramOutBits  := "00000";
    Constant cSramOutBits_None               : TSramOutBits  := "11111";

    Constant cSramTimerIsZero                : Std_Logic_Vector(2 Downto 0) := "000";

    Signal   SramOutBits                     : TSramOutBits;
    Signal   SramOutBits_Write               : TSramOutBits;




    ReplaceWith_SRAMDeclareSignals

    -- ------------------------------------------------------------------------

    -- ------------------------------------------------------------------------
    -- FLASH Declarations - Dynamic
    -- ------------------------------------------------------------------------
    Type TFLashState Is (
        FlashState_Idle,
        FlashState_Read_AddressSetup,
        FlashState_Read_CSActive,
        FlashState_Read_OEActive,
        FlashState_Read_Hold,
        FlashState_Read_HoldLast,

        FlashState_Write_AddressSetup,
        FlashState_Write_CSActive,
        FlashState_Write_WEActive,
        FlashState_Write_WEInactive,
        FlashState_Write_WEInactiveLast,
        FlashState_Write_Ack,
        FlashState_Write_Hold,
        FlashState_Write_HoldLast,
        FlashState_WaitForAck
    );
    Signal   FlashState      : TFlashState;
    Signal   FlashState_Next : TFlashState;

    Subtype  TFlashStateDelayCounter Is Integer Range 0 To 7;
    Constant cFlashTimerIsZero : Std_Logic_Vector(2 Downto 0) := "000";


    ----------------------------------------------------------------------------
    ReplaceWith_FLASHDeclareSignals
    ----------------------------------------------------------------------------


    -- ------------------------------------------------------------------------
    -- Usb Declarations - Dynamic
    -- ------------------------------------------------------------------------
    Type TUsbState Is (
        UsbState_AddressSetup,
        UsbState_Write_Cycle0_Pulse,
        UsbState_Write_Cycle0_Wait,
        UsbState_Write_Cycle1_Setup,
        UsbState_Write_Cycle1_Pulse,

        UsbState_Read_Cycle0_Latch,
        UsbState_Read_Cycle1_Setup,

        UsbState_Write_Pulse,
        UsbState_Write_ACK,

        UsbState_Read_ACK,
        UsbState_WaitForAck
    );

    Signal UsbState      : TUsbState;
    Signal UsbState_Next : TUsbState;

    SubType TUsbByteLane Is Std_Logic_Vector(3 Downto 0);
    SubType TUsbOutBits  Is Std_Logic_Vector(4 Downto 0);

    Constant cUsbWrite_Word                 : TUsbByteLane := "1111";
    Constant cUsbWrite_LoHalf               : TUsbByteLane := "0011";
    Constant cUsbWrite_HiHalf               : TUsbByteLane := "1100";
    Constant cUsbWrite_Byte0                : TUsbByteLane := "0001";
    Constant cUsbWrite_Byte1                : TUsbByteLane := "0010";
    Constant cUsbWrite_Byte2                : TUsbByteLane := "0100";
    Constant cUsbWrite_Byte3                : TUsbByteLane := "1000";
    Constant cUsbWrite_Bytes012             : TUsbByteLane := "0111";
    Constant cUsbWrite_Bytes123             : TUsbByteLane := "1110";
    Constant cUsbWrite_Bytes12              : TUsbByteLane := "0110";
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    Constant cUsbOutBits_Write_Word         : TUsbOutBits  := "00000";

    Constant cUsbOutBits_Write_LoHalf       : TUsbOutBits  := "11000";
    Constant cUsbOutBits_Write_HiHalf       : TUsbOutBits  := "00110";
    Constant cUsbOutBits_Write_Byte0        : TUsbOutBits  := "11100";
    Constant cUsbOutBits_Write_Byte1        : TUsbOutBits  := "11010";
    Constant cUsbOutBits_Write_Byte2        : TUsbOutBits  := "10110";
    Constant cUsbOutBits_Write_Byte3        : TUsbOutBits  := "01110";
    Constant cUsbOutBits_Write_Bytes012     : TUsbOutBits  := "10000";
    Constant cUsbOutBits_Write_Bytes123     : TUsbOutBits  := "00010";
    Constant cUsbOutBits_Write_Bytes12      : TUsbOutBits  := "10010";

    Constant cUsbOutBits_Read               : TUsbOutBits  := "00000";
    Constant cUsbOutBits_None               : TUsbOutBits  := "11111";

    Constant cUsbTimerIsZero                : Std_Logic_Vector(2 Downto 0) := "000";

    Signal   UsbOutBits                     : TUsbOutBits;
    Signal   UsbOutBits_Write               : TUsbOutBits;




    ReplaceWith_USBDeclareSignals

    -- ------------------------------------------------------------------------


-- ----------------------------------------------------------------------------
Begin
-- ----------------------------------------------------------------------------

    -- ------------------------------------------------------------------------
    -- WB_SHARED_SDRAM component instance
    -- ------------------------------------------------------------------------
    U_MEMORY_CONTROLLER : WB_SHARED_SDRAM Port Map (
        -- Timing configuration
        CONFIG_BURST_LENGTH                  => cCONFIG_BURST_LENGTH,
        CONFIG_TIMING_TCL                    => cCONFIG_TIMING_TCL,
        CONFIG_TIMING_TMRD                   => cCONFIG_TIMING_TMRD,
        CONFIG_TIMING_TRCD                   => cCONFIG_TIMING_TRCD,
        CONFIG_TIMING_TRFC                   => cCONFIG_TIMING_TRFC,
        CONFIG_TIMING_TRP                    => cCONFIG_TIMING_TRP,
        CONFIG_TIMING_TWR                    => cCONFIG_TIMING_TWR,
        REFRESH_INTERVAL                     => cREFRESH_INTERVAL,
        REFRESH_PRESCALER                    => cREFRESH_PRESCALER,

        --SRAM interface
        CONTROL_WB_SRAM_ACCESS_GRANTED       => PortSramControl_SramAccessGranted,
        CONTROL_WB_SRAM_ACCESS_REQUEST       => PortSramControl_AccessRequest,
        CONTROL_WB_SRAM_ADR                  => PortSramPort_MemoryAddress,
        CONTROL_WB_SRAM_BE                   => PortSramPort_MemoryByteEnable,
        CONTROL_WB_SRAM_D_EN                 => PortSramPort_MemoryDataEnable,
        CONTROL_WB_SRAM_DAT_IN               => PortSramPort_MemoryDataOutput,
        CONTROL_WB_SRAM_DAT_OUT              => PortSramPort_MemoryDataInput,
        CONTROL_WB_SRAM_OE                   => PortSramPort_MemoryOutputEnable,
        CONTROL_WB_SRAM_TRANSFER             => PortSramControl_Transfer,
        CONTROL_WB_SRAM_TRANSFER_DONE        => PortSramControl_TransferFinished,
        CONTROL_WB_SRAM_W                    => PortSramPort_MemoryWrite,

        -- SDRAM interface
        CONTROL_WB_SDRAM_ACKNOWLEDGE         => SdramControl_WishboneAcknowledge,
        CONTROL_WB_SDRAM_BANK_ADDRESS        => SdramWishbone_BankAddress,
        CONTROL_WB_SDRAM_COLUMN_ADDRESS      => SdramWishbone_ColumnAddress,
        CONTROL_WB_SDRAM_DATA_FROM_MEM       => SdramWishbone_DataOutput,
        CONTROL_WB_SDRAM_DATA_TO_MEM         => SdramWishbone_DataInput,
        CONTROL_WB_SDRAM_MEM_BE              => SdramWishbone_ByteSelect,
        CONTROL_WB_SDRAM_REQUEST_READ        => SdramWishbone_RequestRead,
        CONTROL_WB_SDRAM_REQUEST_TRANSFER    => SdramControl_RequestTransfer,
        CONTROL_WB_SDRAM_ROW_ADDRESS         => SdramWishbone_RowAddress,
        CONTROL_WB_SDRAM_SDRAM_MODE_REGISTER => SdramWishbone_ModeRegister,
        CONTROL_WB_SDRAM_TRANSFER_DONE       => SdramControl_TransferDone,

        -- FLASH interface
        CONTROL_WB_FLASH_ACCESS_GRANTED      => FlashControl_FlashAccessGranted,
        CONTROL_WB_FLASH_ACCESS_REQUEST      => FlashControl_AccessRequest,
        CONTROL_WB_FLASH_ADR                 => FlashPort_MemoryAddress,
        CONTROL_WB_FLASH_BE                  => FlashPort_MemoryByteEnable,
        CONTROL_WB_FLASH_BUSY                => FlashPort_Busy,
        CONTROL_WB_FLASH_D_EN                => FlashPort_MemoryDataEnable,
        CONTROL_WB_FLASH_DAT_IN              => FlashPort_MemoryDataOutput,
        CONTROL_WB_FLASH_DAT_OUT             => FlashPort_MemoryDataInput,
        CONTROL_WB_FLASH_OE                  => FlashPort_MemoryOutputEnable,
        CONTROL_WB_FLASH_RESET               => FlashPort_MemoryReset,
        CONTROL_WB_FLASH_TRANSFER            => FlashControl_Transfer,
--        CONTROL_WB_FLASH_TRANSFER_DONE       => FlashControl_TransferDone,
        CONTROL_WB_FLASH_TRANSFER_DONE       => FlashControl_TransferFinished,
        CONTROL_WB_FLASH_W                   => FlashPort_MemoryWrite,

        -- External Memory interface
        -- External Memory interface
        MEM_A                                => MEM_A,
        MEM_BE                               => MEM_BE,
        MEM_D_EN                             => Port_Mem_d_en,
        MEM_D_IN                             => Port_Mem_d_in,
        MEM_D_OUT                            => Port_Mem_d_out,
        MEM_FLASH_BUSY                       => MEM_FLASH_BUSY,
        MEM_FLASH_E                          => MEM_FLASH_E,
        MEM_FLASH_RESET                      => MEM_FLASH_RESET,
        MEM_OE                               => MEM_OE,
        MEM_RST                              => SDRAM_RST_I,
        MEM_SDRAM_CAS                        => MEM_SDRAM_CAS,
        MEM_SDRAM_CKE                        => MEM_SDRAM_CKE,
        MEM_SDRAM_E                          => MEM_SDRAM_E,
        MEM_SDRAM_RAS                        => MEM_SDRAM_RAS,
        MEM_SRAM_E                           => MEM_SRAM_E_Signal,
        MEM_W                                => MEM_W,

        MEM_CLK                              => MEM_CLK
    );
    -- ------------------------------------------------------------------------



-- ----------------------------------------------------------------------------
    -- ------------------------------------------------------------------------
    -- SDRAM logic
    -- ------------------------------------------------------------------------
    ReplaceWith_SdramWishboneInterface

    ReplaceWith_SdramMemoryController
-- ----------------------------------------------------------------------------


-- ----------------------------------------------------------------------------
    -- ------------------------------------------------------------------------
    -- SRAM logic
    -- ------------------------------------------------------------------------

    ReplaceWith_SRAMWishboneInterface

    ReplaceWith_SRAMMemoryController
-- ----------------------------------------------------------------------------
    -- ------------------------------------------------------------------------
    -- FLASH logic
    -- ------------------------------------------------------------------------
    ReplaceWith_FLASHWishboneInterface

    ReplaceWith_FLASHMemoryController
-- ----------------------------------------------------------------------------

-- ----------------------------------------------------------------------------
    -- ------------------------------------------------------------------------
    -- Usb logic
    -- ------------------------------------------------------------------------

    ReplaceWith_USBWishboneInterface

    ReplaceWith_USBMemoryController

-- ----------------------------------------------------------------------------

    Process(MEM_CLK)
    Begin
        If Rising_Edge(MEM_CLK) Then
            If SDRAM_RST_I='1' Then
                EnableSram <= '0';
                EnableUsb  <= '0';
            Else
                If SRAMControl_RequestTransfer='1' Then
                    EnableSram <= '1';
                ElsIf SramControl_TransferFinished='1' Then
                    EnableSram <= '0';
                ElsIf USBControl_RequestTransfer='1' Then
                    EnableUsb <= '1';
                ElsIf UsbControl_TransferFinished='1' Then
                    EnableUsb <= '0';
                End IF;
            End If;
        End If;
    End Process;
        PortSramControl_AccessRequest     <= SramControl_AccessRequest     When EnableSram='1' Else UsbControl_AccessRequest;
        PortSramPort_MemoryAddress        <= SramPort_MemoryAddress        When EnableSram='1' Else UsbPort_MemoryAddress;
        PortSramPort_MemoryByteEnable     <= SramPort_MemoryByteEnable     When EnableSram='1' Else UsbPort_MemoryByteEnable;
        PortSramPort_MemoryDataEnable     <= SramPort_MemoryDataEnable     When EnableSram='1' Else UsbPort_MemoryDataEnable;
        PortSramPort_MemoryDataOutput     <= SramPort_MemoryDataOutput     When EnableSram='1' Else UsbPort_MemoryDataOutput;
        PortSramPort_MemoryOutputEnable   <= SramPort_MemoryOutputEnable   When EnableSram='1' Else UsbPort_MemoryOutputEnable;
        PortSramControl_Transfer          <= SramControl_Transfer          When EnableSram='1' Else UsbControl_Transfer;
        PortSramPort_MemoryWrite          <= SramPort_MemoryWrite          When EnableSram='1' Else UsbPort_MemoryWrite;

        PortSramControl_TransferFinished  <= SramControl_TransferFinished  When EnableSram='1' Else UsbControl_TransferFinished;

        SramControl_SramAccessGranted     <= PortSramControl_SramAccessGranted When EnableSram='1' Else '0';
        UsbControl_UsbAccessGranted       <= PortSramControl_SramAccessGranted When EnableUsb='1'  Else '0';
        SramPort_MemoryDataInput          <= PortSramPort_MemoryDataInput;
        UsbPort_MemoryDataInput           <= PortSramPort_MemoryDataInput;
        MEM_SRAM_E                        <= MEM_SRAM_E_Signal When EnableSram='1' Else '1';
        MEM_USB_E                         <= '0'               When (EnableUsb ='1' And EnableUsbOE='1')  Else '1'; --IncludeIf_Usb IncludeIf_DisableUsb
        MEM_USB_RESET                     <= '1';                                           --IncludeIf_Usb IncludeIf_DisableUsb

    -- ------------------------------------------------------------------------
    -- MEMORY logic
    -- ------------------------------------------------------------------------
    -- {ReplaceWith_Memory_Logic} Start
    -- ------------------------------------------------------------------------
    ReplaceWith_MemoryController



--    Port_Mem_d_in( 0) <= MEM_D( 0);
--    Port_Mem_d_in( 1) <= MEM_D( 1);
--    Port_Mem_d_in( 2) <= MEM_D( 2);
--    Port_Mem_d_in( 3) <= MEM_D( 3);
--    Port_Mem_d_in( 4) <= MEM_D( 4);
--    Port_Mem_d_in( 5) <= MEM_D( 5);
--    Port_Mem_d_in( 6) <= MEM_D( 6);
--    Port_Mem_d_in( 7) <= MEM_D( 7);
--    Port_Mem_d_in( 8) <= MEM_D( 8);
--    Port_Mem_d_in( 9) <= MEM_D( 9);
--    Port_Mem_d_in(10) <= MEM_D(10);
--    Port_Mem_d_in(11) <= MEM_D(11);
--    Port_Mem_d_in(12) <= MEM_D(12);
--    Port_Mem_d_in(13) <= MEM_D(13);
--    Port_Mem_d_in(14) <= MEM_D(14);
--    Port_Mem_d_in(15) <= MEM_D(15);
--    Port_Mem_d_in(16) <= MEM_D(16);
--    Port_Mem_d_in(17) <= MEM_D(17);
--    Port_Mem_d_in(18) <= MEM_D(18);
--    Port_Mem_d_in(19) <= MEM_D(19);
--    Port_Mem_d_in(20) <= MEM_D(20);
--    Port_Mem_d_in(21) <= MEM_D(21);
--    Port_Mem_d_in(22) <= MEM_D(22);
--    Port_Mem_d_in(23) <= MEM_D(23);
--    Port_Mem_d_in(24) <= MEM_D(24);
--    Port_Mem_d_in(25) <= MEM_D(25);
--    Port_Mem_d_in(26) <= MEM_D(26);
--    Port_Mem_d_in(27) <= MEM_D(27);
--    Port_Mem_d_in(28) <= MEM_D(28);
--    Port_Mem_d_in(29) <= MEM_D(29);
--    Port_Mem_d_in(30) <= MEM_D(30);
--    Port_Mem_d_in(31) <= MEM_D(31);
--
--
--    MEM_D( 0) <= Port_Mem_d_out( 0) When Port_Mem_d_en( 0) = '0' Else 'Z';
--    MEM_D( 1) <= Port_Mem_d_out( 1) When Port_Mem_d_en( 1) = '0' Else 'Z';
--    MEM_D( 2) <= Port_Mem_d_out( 2) When Port_Mem_d_en( 2) = '0' Else 'Z';
--    MEM_D( 3) <= Port_Mem_d_out( 3) When Port_Mem_d_en( 3) = '0' Else 'Z';
--    MEM_D( 4) <= Port_Mem_d_out( 4) When Port_Mem_d_en( 4) = '0' Else 'Z';
--    MEM_D( 5) <= Port_Mem_d_out( 5) When Port_Mem_d_en( 5) = '0' Else 'Z';
--    MEM_D( 6) <= Port_Mem_d_out( 6) When Port_Mem_d_en( 6) = '0' Else 'Z';
--    MEM_D( 7) <= Port_Mem_d_out( 7) When Port_Mem_d_en( 7) = '0' Else 'Z';
--    MEM_D( 8) <= Port_Mem_d_out( 8) When Port_Mem_d_en( 8) = '0' Else 'Z';
--    MEM_D( 9) <= Port_Mem_d_out( 9) When Port_Mem_d_en( 9) = '0' Else 'Z';
--    MEM_D(10) <= Port_Mem_d_out(10) When Port_Mem_d_en(10) = '0' Else 'Z';
--    MEM_D(11) <= Port_Mem_d_out(11) When Port_Mem_d_en(11) = '0' Else 'Z';
--    MEM_D(12) <= Port_Mem_d_out(12) When Port_Mem_d_en(12) = '0' Else 'Z';
--    MEM_D(13) <= Port_Mem_d_out(13) When Port_Mem_d_en(13) = '0' Else 'Z';
--    MEM_D(14) <= Port_Mem_d_out(14) When Port_Mem_d_en(14) = '0' Else 'Z';
--    MEM_D(15) <= Port_Mem_d_out(15) When Port_Mem_d_en(15) = '0' Else 'Z';
--    MEM_D(16) <= Port_Mem_d_out(16) When Port_Mem_d_en(16) = '0' Else 'Z';
--    MEM_D(17) <= Port_Mem_d_out(17) When Port_Mem_d_en(17) = '0' Else 'Z';
--    MEM_D(18) <= Port_Mem_d_out(18) When Port_Mem_d_en(18) = '0' Else 'Z';
--    MEM_D(19) <= Port_Mem_d_out(19) When Port_Mem_d_en(19) = '0' Else 'Z';
--    MEM_D(20) <= Port_Mem_d_out(20) When Port_Mem_d_en(20) = '0' Else 'Z';
--    MEM_D(21) <= Port_Mem_d_out(21) When Port_Mem_d_en(21) = '0' Else 'Z';
--    MEM_D(22) <= Port_Mem_d_out(22) When Port_Mem_d_en(22) = '0' Else 'Z';
--    MEM_D(23) <= Port_Mem_d_out(23) When Port_Mem_d_en(23) = '0' Else 'Z';
--    MEM_D(24) <= Port_Mem_d_out(24) When Port_Mem_d_en(24) = '0' Else 'Z';
--    MEM_D(25) <= Port_Mem_d_out(25) When Port_Mem_d_en(25) = '0' Else 'Z';
--    MEM_D(26) <= Port_Mem_d_out(26) When Port_Mem_d_en(26) = '0' Else 'Z';
--    MEM_D(27) <= Port_Mem_d_out(27) When Port_Mem_d_en(27) = '0' Else 'Z';
--    MEM_D(28) <= Port_Mem_d_out(28) When Port_Mem_d_en(28) = '0' Else 'Z';
--    MEM_D(29) <= Port_Mem_d_out(29) When Port_Mem_d_en(29) = '0' Else 'Z';
--    MEM_D(30) <= Port_Mem_d_out(30) When Port_Mem_d_en(30) = '0' Else 'Z';
--    MEM_D(31) <= Port_Mem_d_out(31) When Port_Mem_d_en(31) = '0' Else 'Z';

    -- ------------------------------------------------------------------------
    -- {ReplaceWith_Memory_Logic} End
    -- ------------------------------------------------------------------------
-- ----------------------------------------------------------------------------

-- ----------------------------------------------------------------------------
End RTL;
-- ----------------------------------------------------------------------------

