-- ----------------------------------------------------------------------------
Library IEEE;
Use     IEEE.Std_Logic_1164.All;
Use     IEEE.Std_Logic_Unsigned.All;
Use     IEEE.Std_Logic_Arith.All;
-- ----------------------------------------------------------------------------

-- ----------------------------------------------------------------------------
Entity ReplaceWith_Designator Is
    Port (
        -- --------------------------------------------------------------------
        ReplaceWith_SramEntityPorts

	ReplaceWith_SdramEntityPorts

	ReplaceWith_FlashEntityPorts


        -- --------------------------------------------------------------------
        --  {ReplaceWith_TopLevelMemoryPorts} Start
        -- External Memory interface
        MEM_A               : Out   Std_Logic_Vector(23 Downto 0);
        MEM_BE              : Out   Std_Logic_Vector( 3 Downto 0);
        MEM_D               : InOut Std_Logic_Vector(31 Downto 0);
        MEM_FLASH_BUSY      : In    Std_Logic;
        MEM_FLASH_E         : Out   Std_Logic;
        MEM_FLASH_RESET     : Out   Std_Logic;
        MEM_OE              : Out   Std_Logic;
        MEM_SDRAM_CAS       : Out   Std_Logic;
        MEM_SDRAM_CKE       : Out   Std_Logic;
        MEM_SDRAM_E         : Out   Std_Logic;
        MEM_SDRAM_RAS       : Out   Std_Logic;
        MEM_SRAM_E          : Out   Std_Logic;
        MEM_W               : Out   Std_Logic;
        --  {ReplaceWith_TopLevelMemoryPorts} End
        -- --------------------------------------------------------------------

        MEM_CLK             : In Std_Logic
    );
End;
-- ----------------------------------------------------------------------------

-- ----------------------------------------------------------------------------
Architecture RTL Of ReplaceWith_Designator Is
-- ----------------------------------------------------------------------------

    -- ------------------------------------------------------------------------
    Component WB_SHARED_SDRAM Is
        Port (
            -- Timing configuration
            CONFIG_BURST_LENGTH                  : In  Std_Logic_Vector( 2 Downto 0);
            CONFIG_TIMING_TCL                    : In  Std_Logic_Vector( 1 Downto 0);
            CONFIG_TIMING_TMRD                   : In  Std_Logic_Vector( 2 Downto 0);
            CONFIG_TIMING_TRCD                   : In  Std_Logic_Vector( 2 Downto 0);
            CONFIG_TIMING_TRFC                   : In  Std_Logic_Vector( 2 Downto 0);
            CONFIG_TIMING_TRP                    : In  Std_Logic_Vector( 2 Downto 0);
            CONFIG_TIMING_TWR                    : In  Std_Logic_Vector( 2 Downto 0);
            REFRESH_INTERVAL                     : In  Std_Logic_Vector( 2 Downto 0);
            REFRESH_PRESCALER                    : In  Std_Logic_Vector( 7 Downto 0);

            --SRAM interface
            CONTROL_WB_SRAM_ACCESS_GRANTED       : Out Std_Logic;
            CONTROL_WB_SRAM_ACCESS_REQUEST       : In  Std_Logic;
            CONTROL_WB_SRAM_ADR                  : In  Std_Logic_Vector(23 Downto 0);
            CONTROL_WB_SRAM_BE                   : In  Std_Logic_Vector( 3 Downto 0);
            CONTROL_WB_SRAM_D_EN                 : In  Std_Logic_Vector(31 Downto 0);
            CONTROL_WB_SRAM_DAT_IN               : In  Std_Logic_Vector(31 Downto 0);
            CONTROL_WB_SRAM_DAT_OUT              : Out Std_Logic_Vector(31 Downto 0);
            CONTROL_WB_SRAM_OE                   : In  Std_Logic;
            CONTROL_WB_SRAM_TRANSFER             : In  Std_Logic;
            CONTROL_WB_SRAM_TRANSFER_DONE        : In  Std_Logic;
            CONTROL_WB_SRAM_W                    : In  Std_Logic;

            -- SDRAM interface
            CONTROL_WB_SDRAM_ACKNOWLEDGE         : In  Std_Logic;
            CONTROL_WB_SDRAM_BANK_ADDRESS        : In  Std_Logic_Vector( 1 Downto 0);
            CONTROL_WB_SDRAM_COLUMN_ADDRESS      : In  Std_Logic_Vector(12 Downto 0);
            CONTROL_WB_SDRAM_DATA_FROM_MEM       : Out Std_Logic_Vector(31 Downto 0);
            CONTROL_WB_SDRAM_DATA_TO_MEM         : In  Std_Logic_Vector(31 Downto 0);
            CONTROL_WB_SDRAM_MEM_BE              : In  Std_Logic_Vector( 3 Downto 0);
            CONTROL_WB_SDRAM_REQUEST_READ        : In  Std_Logic;
            CONTROL_WB_SDRAM_REQUEST_TRANSFER    : In  Std_Logic;
            CONTROL_WB_SDRAM_ROW_ADDRESS         : In  Std_Logic_Vector(13 Downto 0);
            CONTROL_WB_SDRAM_SDRAM_MODE_REGISTER : In  Std_Logic_Vector(12 Downto 0);
            CONTROL_WB_SDRAM_TRANSFER_DONE       : Out Std_Logic;

            -- FLASH interface
            CONTROL_WB_FLASH_ACCESS_GRANTED      : Out Std_Logic;
            CONTROL_WB_FLASH_ACCESS_REQUEST      : In  Std_Logic;
            CONTROL_WB_FLASH_ADR                 : In  Std_Logic_Vector(23 Downto 0);
            CONTROL_WB_FLASH_BE                  : In  Std_Logic_Vector( 3 Downto 0);
            CONTROL_WB_FLASH_BUSY                : Out Std_Logic;
            CONTROL_WB_FLASH_D_EN                : In  Std_Logic_Vector(31 Downto 0);
            CONTROL_WB_FLASH_DAT_IN              : In  Std_Logic_Vector(31 Downto 0);
            CONTROL_WB_FLASH_DAT_OUT             : Out Std_Logic_Vector(31 Downto 0);
            CONTROL_WB_FLASH_OE                  : In  Std_Logic;
            CONTROL_WB_FLASH_RESET               : In  Std_Logic;
            CONTROL_WB_FLASH_TRANSFER            : In  Std_Logic;
            CONTROL_WB_FLASH_TRANSFER_DONE       : In  Std_Logic;
            CONTROL_WB_FLASH_W                   : In  Std_Logic;

            -- External Memory interface
            MEM_A                                : Out Std_Logic_Vector(23 Downto 0);
            MEM_BE                               : Out Std_Logic_Vector( 3 Downto 0);
            MEM_CLK                              : In  Std_Logic;
            MEM_D_EN                             : Out Std_Logic_Vector(31 Downto 0);
            MEM_D_IN                             : In  Std_Logic_Vector(31 Downto 0);
            MEM_D_OUT                            : Out Std_Logic_Vector(31 Downto 0);
            MEM_FLASH_BUSY                       : In  Std_Logic;
            MEM_FLASH_E                          : Out Std_Logic;
            MEM_FLASH_RESET                      : Out Std_Logic;
            MEM_OE                               : Out Std_Logic;
            MEM_RST                              : In  Std_Logic;
            MEM_SDRAM_CAS                        : Out Std_Logic;
            MEM_SDRAM_CKE                        : Out Std_Logic;
            MEM_SDRAM_E                          : Out Std_Logic;
            MEM_SDRAM_RAS                        : Out Std_Logic;
            MEM_SRAM_E                           : Out Std_Logic;
            MEM_W                                : Out Std_Logic
        );
    End Component;
    -- ------------------------------------------------------------------------


    -- ------------------------------------------------------------------------
    -- Timing Configuration
    -- ------------------------------------------------------------------------
    -- {ReplaceWith_SdramTimingConfiguration} Start
    --Constant cCONFIG_BURST_LENGTH  : Std_Logic_Vector( 2 Downto 0) := ReplaceWith_CONFIG_BURST_LENGTH;
    --Constant cCONFIG_TIMING_TCL    : Std_Logic_Vector( 1 Downto 0) := ReplaceWith_CONFIG_TIMING_TCL;
    --Constant cCONFIG_TIMING_TMRD   : Std_Logic_Vector( 2 Downto 0) := ReplaceWith_CONFIG_TIMING_TMRD;
    --Constant cCONFIG_TIMING_TRCD   : Std_Logic_Vector( 2 Downto 0) := ReplaceWith_CONFIG_TIMING_TRCD;
    --Constant cCONFIG_TIMING_TRFC   : Std_Logic_Vector( 2 Downto 0) := ReplaceWith_CONFIG_TIMING_TRFC;
    --Constant cCONFIG_TIMING_TRP    : Std_Logic_Vector( 2 Downto 0) := ReplaceWith_CONFIG_TIMING_TRP;
    --Constant cCONFIG_TIMING_TWR    : Std_Logic_Vector( 2 Downto 0) := ReplaceWith_CONFIG_TIMING_TWR;
    --Constant cREFRESH_INTERVAL     : Std_Logic_Vector( 2 Downto 0) := ReplaceWith_REFRESH_INTERVAL;
    --Constant cREFRESH_PRESCALER    : Std_Logic_Vector( 7 Downto 0) := ReplaceWith_REFRESH_PRESCALER;
    Constant cCONFIG_BURST_LENGTH  : Std_Logic_Vector( 2 Downto 0) := "000";
    Constant cCONFIG_TIMING_TCL    : Std_Logic_Vector( 1 Downto 0) := "11";
    Constant cCONFIG_TIMING_TMRD   : Std_Logic_Vector( 2 Downto 0) := "001";
    Constant cCONFIG_TIMING_TRCD   : Std_Logic_Vector( 2 Downto 0) := "001";
    Constant cCONFIG_TIMING_TRFC   : Std_Logic_Vector( 2 Downto 0) := "001";
    Constant cCONFIG_TIMING_TRP    : Std_Logic_Vector( 2 Downto 0) := "001";
    Constant cCONFIG_TIMING_TWR    : Std_Logic_Vector( 2 Downto 0) := "001";
    Constant cREFRESH_INTERVAL     : Std_Logic_Vector( 2 Downto 0) := "001";
    Constant cREFRESH_PRESCALER    : Std_Logic_Vector( 7 Downto 0) := "10010001";

    Constant cCONFIG_BURST_MODE    : Std_Logic := '1';
    Constant cCONFIG_BURST_TYPE    : Std_Logic := '0';


    -- {ReplaceWith_SdramTimingConfiguration} End
    -- ------------------------------------------------------------------------

    -- ------------------------------------------------------------------------
    -- SRAM interface signal declaration - Static
    -- ------------------------------------------------------------------------
    Signal Port_Sram_ACCESS_GRANTED       : Std_Logic;
    Signal Port_Sram_ACCESS_REQUEST       : Std_Logic;
    Signal Port_Sram_ADR                  : Std_Logic_Vector(23 Downto 0);
    Signal Port_Sram_BE                   : Std_Logic_Vector( 3 Downto 0);
    Signal Port_Sram_D_EN                 : Std_Logic_Vector(31 Downto 0);
    Signal Port_Sram_DAT_IN               : Std_Logic_Vector(31 Downto 0);
    Signal Port_Sram_DAT_OUT              : Std_Logic_Vector(31 Downto 0);
    Signal Port_Sram_OE                   : Std_Logic;
    Signal Port_Sram_TRANSFER             : Std_Logic;
    Signal Port_Sram_TRANSFER_DONE        : Std_Logic;
    Signal Port_Sram_W                    : Std_Logic;
    -- ------------------------------------------------------------------------

    -- ------------------------------------------------------------------------
    -- SDRAM interface signal declaration - Static
    -- ------------------------------------------------------------------------
    Signal Port_Sdram_Acknowledge         : Std_Logic;
    Signal Port_Sdram_Bank_address        : Std_Logic_Vector( 1 Downto 0);
    Signal Port_Sdram_Column_address      : Std_Logic_Vector(12 Downto 0);
    Signal Port_Sdram_Data_from_mem       : Std_Logic_Vector(31 Downto 0);
    Signal Port_Sdram_Data_to_mem         : Std_Logic_Vector(31 Downto 0);
    Signal Port_Sdram_Mem_be              : Std_Logic_Vector( 3 Downto 0);
    Signal Port_Sdram_Request_read        : Std_Logic;
    Signal Port_Sdram_Request_transfer    : Std_Logic;
    Signal Port_Sdram_Row_address         : Std_Logic_Vector(13 Downto 0);
    Signal Port_Sdram_Sdram_mode_register : Std_Logic_Vector(12 Downto 0);
    Signal Port_Sdram_Transfer_done       : Std_Logic;
    -- ------------------------------------------------------------------------

    -- ------------------------------------------------------------------------
    -- FLASH interface signal declaration - Static
    -- ------------------------------------------------------------------------
    Signal Port_Flash_Access_granted      : Std_Logic;
    Signal Port_Flash_Access_request      : Std_Logic;
    Signal Port_Flash_Adr                 : Std_Logic_Vector(23 Downto 0);
    Signal Port_Flash_Be                  : Std_Logic_Vector( 3 Downto 0);
    Signal Port_Flash_Busy                : Std_Logic;
    Signal Port_Flash_D_en                : Std_Logic_Vector(31 Downto 0);
    Signal Port_Flash_Dat_in              : Std_Logic_Vector(31 Downto 0);
    Signal Port_Flash_Dat_out             : Std_Logic_Vector(31 Downto 0);
    Signal Port_Flash_Oe                  : Std_Logic;
    Signal Port_Flash_Reset               : Std_Logic;
    Signal Port_Flash_Transfer            : Std_Logic;
    Signal Port_Flash_Transfer_done       : Std_Logic;
    Signal Port_Flash_W                   : Std_Logic;
    -- ------------------------------------------------------------------------

    -- ------------------------------------------------------------------------
    -- External Memory interface    - Static
    -- ------------------------------------------------------------------------
    Signal Port_Mem_a                     : Std_Logic_Vector(23 Downto 0);
    Signal Port_Mem_be                    : Std_Logic_Vector( 3 Downto 0);
    Signal Port_Mem_d_en                  : Std_Logic_Vector(31 Downto 0);
    Signal Port_Mem_d_in                  : Std_Logic_Vector(31 Downto 0);
    Signal Port_Mem_d_out                 : Std_Logic_Vector(31 Downto 0);
    Signal Port_Mem_flash_busy            : Std_Logic;
    Signal Port_Mem_flash_e               : Std_Logic;
    Signal Port_Mem_flash_reset           : Std_Logic;
    Signal Port_Mem_oe                    : Std_Logic;
    Signal Port_Mem_rst                   : Std_Logic;
    Signal Port_Mem_sdram_cas             : Std_Logic;
    Signal Port_Mem_sdram_cke             : Std_Logic;
    Signal Port_Mem_sdram_e               : Std_Logic;
    Signal Port_Mem_sdram_ras             : Std_Logic;
    Signal Port_Mem_sram_e                : Std_Logic;
    Signal Port_Mem_w                     : Std_Logic;
    -- ------------------------------------------------------------------------

    -- ------------------------------------------------------------------------
    -- Dynamic Declarations
    -- ------------------------------------------------------------------------

    -- ------------------------------------------------------------------------
    -- SDRAM Declarations - Dynamic
    -- ------------------------------------------------------------------------
    -- {ReplaceWith_Sdram_ArchitecturalDeclaration} Start
    Signal SdramWishbone_AcknowledgeSignal    : Std_Logic;
    Signal SdramWishbone_TransferFinished_ff0 : Std_Logic;
    Signal SdramWishbone_TransferFinished     : Std_Logic;
    Signal SdramControl_TransferDone          : Std_Logic;
    Signal SdramWishbone_Request              : Std_Logic;
    Signal SdramWishbone_Acknowledge          : Std_Logic;
    Signal SdramWishbone_Transfer             : Std_Logic;
    Signal SdramWishbone_DataInput            : Std_Logic_Vector(31 Downto 0);
    Signal SdramWishbone_Address              : Std_Logic_Vector(25 Downto 0);
    Signal SdramWishbone_ByteSelect           : Std_Logic_Vector( 3 Downto 0);
    Signal SdramWishbone_RequestRead          : Std_Logic;
    Signal SdramWishbone_RequestWrite         : Std_Logic;
    Signal SdramWishbone_DataOutput           : Std_Logic_Vector(31 Downto 0);
    Signal SdramWishbone_RowAddress           : Std_Logic_Vector(13 Downto 0);
    Signal SdramWishbone_ColumnAddress        : Std_Logic_Vector(12 Downto 0);
    Signal SdramWishbone_BankAddress          : Std_Logic_Vector( 1 Downto 0);
    Signal SdramWishbone_ModeRegister         : Std_Logic_Vector(12 Downto 0);



    -- {ReplaceWith_Sdram_ArchitecturalDeclaration} End
    -- ------------------------------------------------------------------------

    -- ------------------------------------------------------------------------
    -- SRAM Declarations - Dynamic
    -- ------------------------------------------------------------------------
    -- {ReplaceWith_Sram_ArchitecturalDeclaration} Start
    ----------------------------------------------------------------------------
    -- Constants for byte lanes - always the same
    ----------------------------------------------------------------------------

    Constant cCONFIG_SRAM_ADDRESS_SETUP : Std_Logic_Vector(2 Downto 0) := "001";
    Constant cCONFIG_SRAM_WRITE_PULSE   : Std_Logic_Vector(2 Downto 0) := "001";
    Constant cCONFIG_SRAM_WRITE_HOLD    : Std_Logic_Vector(2 Downto 0) := "001";


    Type TSramState Is (
        SramState_AddressSetup,
        SramState_Write_Cycle0_Pulse,
        SramState_Write_Cycle0_Wait,
        SramState_Write_Cycle1_Setup,
        SramState_Write_Cycle1_Pulse,

        SramState_Read_Cycle0_Latch,
        SramState_Read_Cycle1_Setup,

        SramState_Write_Pulse,
        SramState_Write_ACK,

        SramState_Read_ACK,
        SramState_WaitForAck
    );

    Signal SramState      : TSramState;
    Signal SramState_Next : TSramState;

    SubType TSramByteLane Is Std_Logic_Vector(3 Downto 0);
    SubType TSramOutBits  Is Std_Logic_Vector(4 Downto 0);

    Constant cSramWrite_Word                 : TSramByteLane := "1111";
    Constant cSramWrite_LoHalf               : TSramByteLane := "0011";
    Constant cSramWrite_HiHalf               : TSramByteLane := "1100";
    Constant cSramWrite_Byte0                : TSramByteLane := "0001";
    Constant cSramWrite_Byte1                : TSramByteLane := "0010";
    Constant cSramWrite_Byte2                : TSramByteLane := "0100";
    Constant cSramWrite_Byte3                : TSramByteLane := "1000";
    Constant cSramWrite_Bytes012             : TSramByteLane := "0111";
    Constant cSramWrite_Bytes123             : TSramByteLane := "1110";
    Constant cSramWrite_Bytes12              : TSramByteLane := "0110";
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    Constant cSramOutBits_Write_Word         : TSramOutBits  := "00000";

    Constant cSramOutBits_Write_LoHalf       : TSramOutBits  := "11000";
    Constant cSramOutBits_Write_HiHalf       : TSramOutBits  := "00110";
    Constant cSramOutBits_Write_Byte0        : TSramOutBits  := "11100";
    Constant cSramOutBits_Write_Byte1        : TSramOutBits  := "11010";
    Constant cSramOutBits_Write_Byte2        : TSramOutBits  := "10110";
    Constant cSramOutBits_Write_Byte3        : TSramOutBits  := "01110";
    Constant cSramOutBits_Write_Bytes012     : TSramOutBits  := "10000";
    Constant cSramOutBits_Write_Bytes123     : TSramOutBits  := "00010";
    Constant cSramOutBits_Write_Bytes12      : TSramOutBits  := "10010";

    Constant cSramOutBits_Read               : TSramOutBits  := "00000";
    Constant cSramOutBits_None               : TSramOutBits  := "11111";

    Constant cSramTimerIsZero                : Std_Logic_Vector(2 Downto 0) := "000";

    Signal   SramOutBits                     : TSramOutBits;
    Signal   SramOutBits_Write               : TSramOutBits;

    Signal   SramRequest                     : Std_Logic;
    Signal   SramRequestRead                 : Std_Logic;
    Signal   SramRequestWrite                : Std_Logic;

    Signal   SramWriteActive_Cycle           : Std_Logic;
    Signal   SramWE                          : Std_Logic;
    Signal   SramOE                          : Std_Logic;


    Signal  SramTimer                        : Std_Logic_Vector(2 Downto 0);
    Signal  SramTimerIsZero                  : Std_Logic;
    Signal  SramLoadAddressDelay             : Std_Logic;
    Signal  SramLoadWritePulseDelay          : Std_Logic;
    Signal  SramLoadWriteHoldDelay           : Std_Logic;
    Signal  SramSRAM_Acknowledge             : Std_Logic;

    Signal  SramSramAccessRequestff0         : Std_Logic;
    Signal  SramSramAccessRequestff1         : Std_Logic;
    Signal  SramSramAccessRequestff2         : Std_Logic;
    Signal  SramSramAccessRequestEdge        : Std_Logic;
    Signal  SramWB_ACKNOWLEDGE_SYNC          : Std_Logic;

    Signal  SramLatchedData                  : Std_Logic_Vector(15 Downto 0);
    Signal  SramCycle_Number                 : Std_Logic;

    Signal  SramWishbone_TransferFinished_ff0 : Std_Logic;
    Signal  SramWishbone_TransferFinished_ff1 : Std_Logic;
    Signal  SramWishbone_TransferFinished     : Std_Logic;
    Signal  SramWishbone_AcknowledgeSignal    : Std_Logic;
    Signal  SramWishbone_RequestTransfer      : Std_Logic;

    Signal  SramControl_TransferFinished      : Std_Logic;
    Signal  SramWishbone_Request              : Std_Logic;

    Signal  SramWishbone_DataInput            : Std_Logic_Vector(31 Downto 0);
    Signal  SramWishbone_DataOutput           : Std_Logic_Vector(31 Downto 0);
    Signal  SramWishbone_Address              : Std_Logic_Vector(23 Downto 0);
    Signal  SramWishbone_ByteSelect           : Std_Logic_Vector( 3 Downto 0);
    Signal  SramWishbone_WriteEnable          : Std_Logic;
    Signal  SramWishbone_RequestRead          : Std_Logic;
    Signal  SramWishbone_RequestWrite         : Std_Logic;

    Signal  SramPort_MemoryByteEnable         : Std_Logic_Vector( 3 Downto 0);
    Signal  SramPort_MemoryAddress            : Std_Logic_Vector(23 Downto 0);

    Signal  SramControl_AccessRequest         : Std_Logic;
    Signal  SramAccessRequest_ff0             : Std_Logic;
    Signal  SramAccessRequest_ff1             : Std_Logic;
    Signal  SramAccessRequest_ff2             : Std_Logic;
    Signal  SramWishbone_AcknowledgeSync      : Std_Logic;

    Signal  SramAcknowledge                   : Std_Logic;
    Signal  SramControl_SramAccessGranted     : Std_Logic;
    Signal  SramPort_MemoryWrite              : Std_Logic;
    Signal  SramPort_MemoryOutputEnable       : Std_Logic;
    Signal  SramPort_MemoryDataOutput         : Std_Logic_Vector(31 Downto 0);
    Signal  SramPort_MemoryDatainput          : Std_Logic_Vector(31 Downto 0);
    Signal  SramPort_MemoryDataEnable         : Std_Logic_Vector(31 Downto 0);
    Signal  SramControl_TransferDone          : Std_Logic;
    Signal  SramControl_Transfer              : Std_Logic;
    Signal  SramControl_InternalDataWidth     : Std_Logic;
    Signal  SramAccessRequestEdge             : Std_Logic;
      -- {ReplaceWith_Sram_ArchitecturalDeclaration} End
    -- ------------------------------------------------------------------------

    -- ------------------------------------------------------------------------
    -- FLASH Declarations - Dynamic
    -- ------------------------------------------------------------------------
    -- {ReplaceWith_Flash_ArchitecturalDeclaration} Start
    --Constant cCONFIG_FLASH_ADDRESS_SETUP : Std_Logic_Vector(2 Downto 0) := ReplaceWith_FlashConstants_AddressSetup;
    --Constant cCONFIG_FLASH_WRITE_PULSE   : Std_Logic_Vector(2 Downto 0) := ReplaceWith_FlashConstants_WritePulse;
    --Constant cCONFIG_FLASH_WRITE_HOLD    : Std_Logic_Vector(2 Downto 0) := ReplaceWith_FlashConstants_WriteHold;

    Constant cCONFIG_FLASH_ADDRESS_SETUP : Std_Logic_Vector(2 Downto 0) := "110";
    Constant cCONFIG_FLASH_WRITE_PULSE   : Std_Logic_Vector(2 Downto 0) := "110";
    Constant cCONFIG_FLASH_WRITE_HOLD    : Std_Logic_Vector(2 Downto 0) := "110";

    Type TFLashState Is (
        FlashState_Idle,
        FlashState_Read_AddressSetup,
        FlashState_Read_CSActive,
        FlashState_Read_OEActive,
        FlashState_Read_Hold,
        FlashState_Read_HoldLast,

        FlashState_Write_AddressSetup,
        FlashState_Write_CSActive,
        FlashState_Write_WEActive,
        FlashState_Write_WEInactive,
        FlashState_Write_WEInactiveLast,
        FlashState_Write_Ack,
        FlashState_Write_Hold,
        FlashState_Write_HoldLast,
        FlashState_WaitForAck
    );
    Signal   FlashState      : TFlashState;
    Signal   FlashState_Next : TFlashState;

    Subtype  TFlashStateDelayCounter Is Integer Range 0 To 7;
    Constant cFlashTimerIsZero : Std_Logic_Vector(2 Downto 0) := "000";

    ----------------------------------------------------------------------------
    Signal   FlashRequest                : Std_Logic;
    Signal   FlashRequestRead            : Std_Logic;
    Signal   FlashRequestWrite           : Std_Logic;
    Signal   FlashSkipCycle_0            : Std_Logic;
    Signal   FlashSkipCycle_1            : Std_Logic;
    ----------------------------------------------------------------------------
    Signal   FlashIsMemActive            : Std_Logic;
    Signal   FlashIsCSActive             : Std_Logic;
    Signal   FlashIsWEActive             : Std_Logic;
    Signal   FlashIsOEActive             : Std_Logic;
    Signal   FlashIsWriting              : Std_Logic;
    Signal   FlashLoadReadData           : Std_Logic;
    Signal   FlashWishboneAck            : Std_Logic;
    ----------------------------------------------------------------------------
    Signal   FlashCycle                  : Std_Logic;
    Signal   FlashResetCycle             : Std_Logic;
    Signal   FlashIncrementCycle         : Std_Logic;
    Signal   FlashIsLastCycle            : Std_Logic;
    ----------------------------------------------------------------------------
    Signal   FlashReadDataLoWord_Reg     : Std_Logic_Vector(15 Downto 0);
    ----------------------------------------------------------------------------
    Signal   FlashLoadTimerAddressSetup  : Std_Logic;
    Signal   FlashLoadTimerReadPulse     : Std_Logic;
    Signal   FlashLoadTimerWritePulse    : Std_Logic;
    Signal   FlashLoadTimerWriteHold     : Std_Logic;
    Signal   FlashTimer                  : Std_Logic_Vector(2 Downto 0);
    Signal   FlashTimerIsZero            : Std_Logic;
    Signal   FlashRequestReadBusyFlag    : Std_Logic;
    Signal   FlashFlashAccessRequestff0  : Std_Logic;
    Signal   FlashFlashAccessRequestff1  : Std_Logic;
    Signal   FlashFlashAccessRequestff2  : Std_Logic;
    Signal   FlashFlashAccessRequestEdge : Std_Logic;
    Signal   FlashWB_ACKNOWLEDGE_SYNC    : Std_Logic;
    Signal   Flash_Acknowledge      : Std_Logic;
    ----------------------------------------------------------------------------


    Signal FlashControl_Transfer              : Std_Logic;
    Signal FlashAcknowledge                   : Std_Logic;
    Signal FlashControl_TransferFinished      : Std_Logic;
    Signal FlashWishbone_TransferFinished_ff0 : Std_Logic;
    Signal FlashWishbone_TransferFinished_ff1 : Std_Logic;
    Signal FlashWishbone_TransferFinished     : Std_Logic;
    Signal FlashWishbone_AcknowledgeSignal    : Std_Logic;
    Signal FlashWishbone_RequestTransfer      : Std_Logic;

    Signal FlashWishbone_DataOutput           : Std_Logic_Vector(31 Downto 0);
    Signal FlashWishbone_DataInput            : Std_Logic_Vector(31 Downto 0);
    Signal FlashWishbone_Address              : Std_Logic_Vector(24 Downto 0);
    Signal FlashWishbone_Request              : Std_Logic;
    Signal FlashWishbone_ByteSelect           : Std_Logic_Vector( 3 Downto 0);
    Signal FlashWishbone_WriteEnable          : Std_Logic;
    Signal FlashWishbone_RequestRead          : Std_Logic;
    Signal FlashWishbone_RequestWrite         : Std_Logic;

    Signal FlashPort_MemoryAddress            : Std_Logic_Vector(23 Downto 0);
    Signal FlashPort_MemoryDataEnable         : Std_Logic_Vector(31 Downto 0);
    Signal FlashPort_MemoryDataOutput         : Std_Logic_Vector(31 Downto 0);
    Signal FlashPort_MemoryByteEnable         : Std_Logic_Vector( 3 Downto 0);
    Signal FlashPort_MemoryWrite              : Std_Logic;
    Signal FlashPort_MemoryOutputEnable       : Std_Logic;
    Signal FlashPort_MemoryReset              : Std_Logic;

    Signal FlashControl_AccessRequest         : Std_Logic;
    Signal FlashAccessRequest_ff0             : Std_Logic;
    Signal FlashAccessRequest_ff1             : Std_Logic;
    Signal FlashAccessRequest_ff2             : Std_Logic;
    Signal FlashWishbone_AcknowledgeSync      : Std_Logic;
    Signal FlashControl_TransferDone          : Std_Logic;

    Signal FlashControl_FlashAccessGranted    : Std_Logic;
    Signal FlashPort_MemoryDataInput          : Std_Logic_Vector(31 Downto 0);
    Signal FlashPort_ByteEnable               : Std_Logic_Vector(3 Downto 0);

    Signal FlashPort_Busy                     : Std_Logic;

    Signal FlashAccessRequestEdge             : Std_Logic;
    -- {ReplaceWith_Flash_ArchitecturalDeclaration} End
    -- ------------------------------------------------------------------------

-- ----------------------------------------------------------------------------
Begin
-- ----------------------------------------------------------------------------

    -- ------------------------------------------------------------------------
    -- WB_SHARED_SDRAM component instance
    -- ------------------------------------------------------------------------
    U_MEMORY_CONTROLLER : WB_SHARED_SDRAM Port Map (
        -- Timing configuration
        CONFIG_BURST_LENGTH                  => cCONFIG_BURST_LENGTH,
        CONFIG_TIMING_TCL                    => cCONFIG_TIMING_TCL,
        CONFIG_TIMING_TMRD                   => cCONFIG_TIMING_TMRD,
        CONFIG_TIMING_TRCD                   => cCONFIG_TIMING_TRCD,
        CONFIG_TIMING_TRFC                   => cCONFIG_TIMING_TRFC,
        CONFIG_TIMING_TRP                    => cCONFIG_TIMING_TRP,
        CONFIG_TIMING_TWR                    => cCONFIG_TIMING_TWR,
        REFRESH_INTERVAL                     => cREFRESH_INTERVAL,
        REFRESH_PRESCALER                    => cREFRESH_PRESCALER,

        --SRAM interface
        CONTROL_WB_SRAM_ACCESS_GRANTED       => SramControl_SramAccessGranted,
        CONTROL_WB_SRAM_ACCESS_REQUEST       => SramControl_AccessRequest,
        CONTROL_WB_SRAM_ADR                  => SramPort_MemoryAddress,
        CONTROL_WB_SRAM_BE                   => SramPort_MemoryByteEnable,
        CONTROL_WB_SRAM_D_EN                 => SramPort_MemoryDataEnable,
        CONTROL_WB_SRAM_DAT_IN               => SramPort_MemoryDataOutput,
        CONTROL_WB_SRAM_DAT_OUT              => SramPort_MemoryDataInput,
        CONTROL_WB_SRAM_OE                   => SramPort_MemoryOutputEnable,
        CONTROL_WB_SRAM_TRANSFER             => SramControl_Transfer,
        CONTROL_WB_SRAM_TRANSFER_DONE        => SramControl_TransferFinished,
        CONTROL_WB_SRAM_W                    => SramPort_MemoryWrite,

        -- SDRAM interface
        CONTROL_WB_SDRAM_ACKNOWLEDGE         => SdramWishbone_TransferFinished,
        CONTROL_WB_SDRAM_BANK_ADDRESS        => SdramWishbone_BankAddress,
        CONTROL_WB_SDRAM_COLUMN_ADDRESS      => SdramWishbone_ColumnAddress,
        CONTROL_WB_SDRAM_DATA_FROM_MEM       => SdramWishbone_DataOutput,
        CONTROL_WB_SDRAM_DATA_TO_MEM         => SdramWishbone_DataInput,
        CONTROL_WB_SDRAM_MEM_BE              => SdramWishbone_ByteSelect,
        CONTROL_WB_SDRAM_REQUEST_READ        => SdramWishbone_RequestRead,
        CONTROL_WB_SDRAM_REQUEST_TRANSFER    => SdramWishbone_Transfer,
        CONTROL_WB_SDRAM_ROW_ADDRESS         => SdramWishbone_RowAddress,
        CONTROL_WB_SDRAM_SDRAM_MODE_REGISTER => SdramWishbone_ModeRegister,
        CONTROL_WB_SDRAM_TRANSFER_DONE       => SdramControl_TransferDone,

        -- FLASH interface
        CONTROL_WB_FLASH_ACCESS_GRANTED      => FlashControl_FlashAccessGranted,
        CONTROL_WB_FLASH_ACCESS_REQUEST      => FlashControl_AccessRequest,
        CONTROL_WB_FLASH_ADR                 => FlashPort_MemoryAddress,
        CONTROL_WB_FLASH_BE                  => FlashPort_MemoryByteEnable,
        CONTROL_WB_FLASH_BUSY                => FlashPort_Busy,
        CONTROL_WB_FLASH_D_EN                => FlashPort_MemoryDataEnable,
        CONTROL_WB_FLASH_DAT_IN              => FlashPort_MemoryDataOutput,
        CONTROL_WB_FLASH_DAT_OUT             => FlashPort_MemoryDataInput,
        CONTROL_WB_FLASH_OE                  => FlashPort_MemoryOutputEnable,
        CONTROL_WB_FLASH_RESET               => FlashPort_MemoryReset,
        CONTROL_WB_FLASH_TRANSFER            => FlashControl_Transfer,
--        CONTROL_WB_FLASH_TRANSFER_DONE       => FlashControl_TransferDone,
        CONTROL_WB_FLASH_TRANSFER_DONE       => FlashControl_TransferFinished,
        CONTROL_WB_FLASH_W                   => FlashPort_MemoryWrite,

        -- External Memory interface
        MEM_A                                => Port_Mem_a,
        MEM_BE                               => Port_Mem_be,
        MEM_D_EN                             => Port_Mem_d_en,
        MEM_D_IN                             => Port_Mem_d_in,
        MEM_D_OUT                            => Port_Mem_d_out,
        MEM_FLASH_BUSY                       => Port_Mem_flash_busy,
        MEM_FLASH_E                          => Port_Mem_flash_e,
        MEM_FLASH_RESET                      => Port_Mem_flash_reset,
        MEM_OE                               => Port_Mem_oe,
        MEM_RST                              => Port_Mem_rst,
        MEM_SDRAM_CAS                        => Port_Mem_sdram_cas,
        MEM_SDRAM_CKE                        => Port_Mem_sdram_cke,
        MEM_SDRAM_E                          => Port_Mem_sdram_e,
        MEM_SDRAM_RAS                        => Port_Mem_sdram_ras,
        MEM_SRAM_E                           => Port_Mem_sram_e,
        MEM_W                                => Port_Mem_w,

        MEM_CLK                              => MEM_CLK
    );
    -- ------------------------------------------------------------------------



-- ----------------------------------------------------------------------------
    -- ------------------------------------------------------------------------
    -- SDRAM logic
    -- ------------------------------------------------------------------------
    -- {ReplaceWith_SDRAM_Logic} Start
    -- ------------------------------------------------------------------------
    -- {ReplaceWith_SDRAM_Logic_WishboneInterface} Start
    Process(WB_SDRAM_CLK_I)
    Begin
        If Rising_Edge(WB_SDRAM_CLK_I) Then
            If WB_SDRAM_CLK_I='1' Then
                SdramWishbone_AcknowledgeSignal    <= '0';
                SdramWishbone_TransferFinished_ff0 <= '0';
                SdramWishbone_TransferFinished     <= '0';
            Else
                SdramWishbone_AcknowledgeSignal    <= SdramWishbone_TransferFinished;
                SdramWishbone_TransferFinished_ff0 <= SdramControl_TransferDone;
                SdramWishbone_TransferFinished     <= SdramWishbone_TransferFinished_ff0 And Not SdramWishbone_AcknowledgeSignal And Not SdramWishbone_TransferFinished And SdramWishbone_Request;
            End If;
        End If;
    End Process;
    WB_SDRAM_ACK_O            <= SdramWishbone_AcknowledgeSignal;
    SdramWishbone_Acknowledge <= SdramWishbone_TransferFinished;

    Process(WB_SDRAM_CLK_I)
    Begin
        If Rising_Edge(WB_SDRAM_CLK_I) Then
            If WB_SDRAM_RST_I='1' Then
                SdramWishbone_Transfer <= '0';
            Else
                SdramWishbone_Transfer <= SdramWishbone_Request And Not SdramWishbone_AcknowledgeSignal And Not SdramWishbone_TransferFinished;
            End If;
        End If;
    End Process;

    Process(WB_SDRAM_CLK_I)
    Begin
        If Rising_Edge(WB_SDRAM_CLK_I) Then
            If WB_SDRAM_RST_I='1' Then
                SdramWishbone_Request      <= '0';
                WB_SDRAM_DAT_O             <= (Others=>'0');
                SdramWishbone_DataInput    <= (Others=>'0');
                SdramWishbone_Address      <= (Others=>'0');
                SdramWishbone_ByteSelect   <= (Others=>'0');
                SdramWishbone_RequestRead  <= '0';
                SdramWishbone_RequestWrite <= '0';
            Else
                SdramWishbone_Request <= WB_SDRAM_CYC_I And WB_SDRAM_STB_I And Not SdramWishbone_AcknowledgeSignal And Not SdramWishbone_TransferFinished;
                If SdramWishbone_TransferFinished='1' Then
                    WB_SDRAM_DAT_O <= SdramWishbone_DataOutput;
                End If;
                SdramWishbone_DataInput    <=     WB_SDRAM_DAT_I;
                SdramWishbone_Address      <=     WB_SDRAM_ADR_I;
                SdramWishbone_ByteSelect   <=     WB_SDRAM_SEL_I;
                SdramWishbone_RequestRead  <= Not WB_SDRAM_WE_I;
                SdramWishbone_RequestWrite <=     WB_SDRAM_WE_I;
            End If;
        End If;
    End Process;
    -- {ReplaceWith_SDRAM_Logic_WishboneInterface} End
    -- ------------------------------------------------------------------------
    -- {ReplaceWith_SDRAM_Logic_AddressLogic} Start

    -- Row Address
    SdramWishbone_RowAddress <= '0' & SdramWishbone_Address(23 Downto 11);
    -- Column Address
    SdramWishbone_ColumnAddress <= "0000" & SdramWishbone_Address(10 Downto 2);
    -- Bank Address
    SdramWishbone_BankAddress <= SdramWishbone_Address(25 Downto 24);

    -- Congif Opcode [12:0]:
    --           000 0 00 000 0 000
    -- RESERVED: 000 - -- --- - ---     ; must be set to all zero
    -- WB      : --- 1 -- --- - ---     ; Write Burst Mode: 0-Programmed Burst Length, 1-Single Location Access
    -- Op Mode : --- - 00 --- - ---     ; Operation Mode: 00 - Standard Operation, all other states reserved
    -- CAS lat : --- - -- cfg - ---     ; CAS Latency, configurable, {1'b0, CONGIG_CL_VALUE}
    -- BT      : --- - -- --- 0 ---     ; Burst Type, 0-Sequential, 1-Interleaved
    -- BL      : --- - -- --- - 000     ; Burst Length, bits 2 1 0  BT= 0           1
    --                                                       0 0 0      1           1
    --                                                       0 0 1      2           2
    --                                                       0 1 0      4           4
    --                                                       0 1 1      8           8
    --                                                       1 0 0      Reserved    Reserved
    --                                                       1 0 1      Reserved    Reserved
    --                                                       1 1 0      Reserved    Reserved
    --                                                       1 1 1      Full Page   Reserved

    SdramWishbone_ModeRegister <= "000" & cCONFIG_BURST_MODE & "000" & cCONFIG_TIMING_TCL & cCONFIG_BURST_TYPE & cCONFIG_BURST_LENGTH;

    -- {ReplaceWith_SDRAM_Logic_AddressLogic} End
    -- ------------------------------------------------------------------------
    -- {ReplaceWith_SDRAM_Logic} End
    -- ------------------------------------------------------------------------
-- ----------------------------------------------------------------------------


-- ----------------------------------------------------------------------------
    -- ------------------------------------------------------------------------
    -- SRAM logic
    -- ------------------------------------------------------------------------
    -- {ReplaceWith_SRAM_Logic} Start
    -- ------------------------------------------------------------------------
    -- {ReplaceWith_SRAM_Logic_WishboneInterface} Start
    Process(WB_SRAM_CLK_I)
    Begin
        If Rising_Edge(WB_SRAM_CLK_I) Then
            If WB_SRAM_RST_I='1' Then
                SramWishbone_TransferFinished_ff0 <= '0';
                SramWishbone_TransferFinished_ff1 <= '0';
                SramWishbone_TransferFinished     <= '0';
                SramWishbone_AcknowledgeSignal    <= '0';
                SramWishbone_RequestTransfer      <= '0';
            Else
                SramWishbone_TransferFinished_ff0 <= SramControl_TransferDone;
                SramWishbone_TransferFinished_ff1 <= SramWishbone_TransferFinished_ff0;
                SramWishbone_TransferFinished     <= SramWishbone_TransferFinished_ff0 And Not SramWishbone_AcknowledgeSignal And Not SramWishbone_TransferFinished And SramWishbone_Request;
                SramWishbone_AcknowledgeSignal    <= SramWishbone_TransferFinished;
                SramWishbone_RequestTransfer      <= SramWishbone_Request And Not SramWishbone_AcknowledgeSignal And Not SramWishbone_TransferFinished;
            End If;
        End If;
    End Process;
    WB_SRAM_ACK_O <= SramWishbone_AcknowledgeSignal;

    Process(WB_SRAM_CLK_I)
    Begin
        If Rising_Edge(WB_SRAM_CLK_I) Then
            If WB_SRAM_RST_I='1' Then
                SramWishbone_Request      <= '0';
                SramWishbone_DataInput    <= (Others=>'0');
                SramWishbone_Address      <= (Others=>'0');
                SramWishbone_ByteSelect   <= (Others=>'0');
                SramWishbone_WriteEnable  <= '0';
                SramWishbone_RequestRead  <= '0';
                SramWishbone_RequestWrite <= '0';
                WB_SRAM_DAT_O             <= (Others=>'0');
            Else
                SramWishbone_Request      <= WB_SRAM_CYC_I And WB_SRAM_STB_I And Not SramWishbone_AcknowledgeSignal And Not SramWishbone_TransferFinished;
                SramWishbone_DataInput    <= WB_SRAM_DAT_I;
                SramWishbone_Address      <= "0000" & WB_SRAM_ADR_I;
                SramWishbone_ByteSelect   <= WB_SRAM_SEL_I;
                SramWishbone_WriteEnable  <= WB_SRAM_WE_I;
                SramWishbone_RequestRead  <= Not WB_SRAM_WE_I;
                SramWishbone_RequestWrite <= WB_SRAM_WE_I;
                If SramWishbone_TransferFinished='1' Then
                    WB_SRAM_DAT_O <= SramWishbone_DataOutput;
                End If;
            End If;
        End If;
    End Process;
    -- {ReplaceWith_SRAM_Logic_WishboneInterface} End
    -- ------------------------------------------------------------------------
    -- {ReplaceWith_SRAM_Logic_TimingController} Start

    SramControl_InternalDataWidth <= '1';

    Process(MEM_CLK)
    Begin
        If Rising_Edge(MEM_CLK) Then
            If WB_SRAM_RST_I='1' Then
                SramTimer <= (Others=>'0');
            Else
                   If SramLoadAddressDelay='1'    Then SramTimer <= cCONFIG_SRAM_ADDRESS_SETUP;
                ElsIf SramLoadWritePulseDelay='1' Then SramTimer <= cCONFIG_SRAM_WRITE_PULSE;
                ElsIf SramLoadWriteHoldDelay='1'  Then SramTimer <= cCONFIG_SRAM_WRITE_HOLD;
                ElsIf SramTimerIsZero='0'         Then SramTimer <= SramTimer - 1;
                End If;
            End If;
        End If;
    End Process;
    SramTimerIsZero <= '1' When SramTimer=cSramTimerIsZero Else '0';

    Process(SramOutBits,
            SramControl_InternalDataWidth,
            SramCycle_Number)
    Begin
        If SramControl_InternalDataWidth='1' Then
            SramPort_MemoryByteEnable(3 Downto 2) <= "00";
            If SramCycle_Number='0' Then
                SramPort_MemoryByteEnable(1 Downto 0) <= SramOutBits(4) & SramOutBits(3);
            Else
                SramPort_MemoryByteEnable(1 Downto 0) <= SramOutBits(2) & SramOutBits(1);
            End If;
        Else
            SramPort_MemoryByteEnable <= SramOutBits(4) & SramOutBits(3) & SramOutBits(2) & SramOutBits(1);
        End If;
    End Process;

    Process(SramWishbone_Address,
            SramControl_InternalDataWidth,
            SramCycle_Number)
    Begin
        If SramControl_InternalDataWidth='1' Then
            SramPort_MemoryAddress(23 Downto 2) <= SramWishbone_Address(22 Downto 1);
            SramPort_MemoryAddress(1) <= SramCycle_Number;
            SramPort_MemoryAddress(0) <= '0';
        Else
            SramPort_MemoryAddress <= SramWishbone_Address;
        End If;
    End Process;

    Process(SramPort_MemoryDataInput,
            SramControl_InternalDataWidth,
            SramLatchedData)
    Begin
        If SramControl_InternalDataWidth='1' Then
            SramWishbone_DataOutput <= SramLatchedData & SramPort_MemoryDataInput(15 Downto 0);
        Else
            SramWishbone_DataOutput <= SramPort_MemoryDataInput;
        End if;
    End Process;

    Process(MEM_CLK)
    Begin
        If Rising_Edge(MEM_CLK) Then
            If WB_SRAM_RST_I='1' Then
                SramLatchedData <= (Others=>'0');
            Else
                If SramState=SramState_Read_Cycle0_Latch Then
                    SramLatchedData <= SramPort_MemoryDataInput(15 Downto 0);
                End If;
            End If;
        End If;
    End Process;

    Process(MEM_CLK)
    Begin
        If Rising_Edge(MEM_CLK) Then
            If WB_SRAM_RST_I='1' Then
                SramControl_AccessRequest    <= '0';
                SramAccessRequest_ff0        <= '0';
                SramAccessRequest_ff1        <= '0';
                SramAccessRequest_ff2        <= '0';
                SramWishbone_AcknowledgeSync <= '0';
            Else
                SramAccessRequest_ff0        <= SramWishbone_Request;
                SramAccessRequest_ff1        <= SramAccessRequest_ff0;
                SramAccessRequest_ff2        <= SramAccessRequest_ff1;
                SramWishbone_AcknowledgeSync <= SramWishbone_AcknowledgeSignal;

                If SramAccessRequestEdge='1' Then
                    SramControl_AccessRequest <= '1';
                ElsIf SramAcknowledge='1' Then
                    SramControl_AccessRequest <= '0';
                End If;
            End If;
        End If;
    End Process;

    SramAccessRequestEdge <= '1' When SramAccessRequest_ff2='0' And SramAccessRequest_ff1='1' And SramAccessRequest_ff0='1' Else '0';
    ----------------------------------------------------------------------------
--    SramRequest      <= (SramWishbone_RequestRead Or SramWishbone_RequestWrite) And SramControl_SramAccessGranted;
--    SramRequestRead  <= SramRequest And SramWishbone_RequestRead;
--    SramRequestWrite <= SramRequest And SramWishbone_RequestWrite;
    ----------------------------------------------------------------------------
    Process(MEM_CLK)
    Begin
        If Rising_Edge(MEM_CLK) Then
            SramRequest      <= (SramWishbone_RequestRead Or SramWishbone_RequestWrite) And SramControl_SramAccessGranted;
            SramRequestRead  <= SramRequest And SramWishbone_RequestRead;
            SramRequestWrite <= SramRequest And SramWishbone_RequestWrite;
        End If;
    End Process;

    ----------------------------------------------------------------------------
    SramPort_MemoryWrite        <= SramWE;
    SramPort_MemoryOutputEnable <= SramOE;
    --MEM_D_O <= DATA_IN;
    Process( SramWishbone_DataInput,
             SramCycle_Number)
    Begin
        If SramControl_InternalDataWidth='1' Then
            SramPort_MemoryDataOutput(31 Downto 16) <= (Others=>'0');
            If SramCycle_Number='0' Then
                SramPort_MemoryDataOutput(15 Downto 0) <= SramWishbone_DataInput(31 Downto 16);
            Else
                SramPort_MemoryDataOutput(15 Downto 0) <= SramWishbone_DataInput(15 Downto 0);
            End If;
        Else
            SramPort_MemoryDataOutput <= SramWishbone_DataInput;
        End If;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    SramMemory_OuptutControls:
    Process(SramWishbone_ByteSelect,
            SramOutBits_Write,
            SramWriteActive_Cycle)
    Begin
        If SramWriteActive_Cycle = '1' Then
           SramOutBits   <= SramOutBits_Write;
           SramOE        <= '1';
           SramPort_MemoryDataEnable <= (Others=>'0');
        Else
           SramOutBits   <= cSramOutBits_Read;
           SramOE        <= '0';
           SramPort_MemoryDataEnable <= (Others=>'1');
        End If;

        Case SramWishbone_ByteSelect Is
           When cSramWrite_Word     => SramOutBits_Write <= cSramOutBits_Write_Word;
           When cSramWrite_LoHalf   => SramOutBits_Write <= cSramOutBits_Write_LoHalf;
           When cSramWrite_HiHalf   => SramOutBits_Write <= cSramOutBits_Write_HiHalf;
           When cSramWrite_Byte0    => SramOutBits_Write <= cSramOutBits_Write_Byte0;
           When cSramWrite_Byte1    => SramOutBits_Write <= cSramOutBits_Write_Byte1;
           When cSramWrite_Byte2    => SramOutBits_Write <= cSramOutBits_Write_Byte2;
           When cSramWrite_Byte3    => SramOutBits_Write <= cSramOutBits_Write_Byte3;
           When cSramWrite_Bytes012 => SramOutBits_Write <= cSramOutBits_Write_Bytes012;
           When cSramWrite_Bytes123 => SramOutBits_Write <= cSramOutBits_Write_Bytes123;
           When cSramWrite_Bytes12  => SramOutBits_Write <= cSramOutBits_Write_Bytes12;
           When Others              => SramOutBits_Write <= cSramOutBits_None;
        End Case;
    End Process;
    ----------------------------------------------------------------------------

    Process(MEM_CLK)
    Begin
        If Rising_Edge(MEM_CLK) Then
            If WB_SRAM_RST_I='1' Then
                SramControl_TransferDone <= '0';
            Else
                If SramAcknowledge='1' Then
                    SramControl_TransferDone <= '1';
                ElsIf SramWishbone_AcknowledgeSync='1' Then
                    SramControl_TransferDone <= '0';
                End If;
            End If;
        End If;
    End Process;

    SramFSM_Combinational_Single:
    Process(SramRequestRead,
            SramRequestWrite,
            SramState,
            SramTimerIsZero,
            SramWishbone_AcknowledgeSync,
            SramControl_InternalDataWidth)
    Begin
        ------------------------------------------------------------------------
        -- Default outputs
        SramWriteActive_Cycle        <= '0';
        SramWE                       <= '1';
        SramState_Next               <= SramState;
        SramLoadAddressDelay         <= '0';
        SramLoadWritePulseDelay      <= '0';
        SramLoadWriteHoldDelay       <= '0';
        SramControl_Transfer         <= '0';
        SramControl_TransferFinished <= '0';
        SramAcknowledge              <= '0';
        SramCycle_Number             <= '0';

        ------------------------------------------------------------------------

        ------------------------------------------------------------------------
        Case SramState Is
           ---------------------------------------------------------------------
           When SramState_AddressSetup =>
                If SramRequestWrite = '1' Then                --Address is being setup in this phases
                    SramWriteActive_Cycle <= '1';
                    SramLoadWritePulseDelay <= '1';
                    SramControl_Transfer <= '1';
                    If SramControl_InternalDataWidth='1' Then
                        SramState_Next <= SramState_Write_Cycle0_Pulse;
                    Else
                        SramState_Next <= SramState_Write_Pulse;
                    End If;
                ElsIf SramRequestRead = '1' Then
                    SramLoadAddressDelay <= '1';
                    SramControl_Transfer <= '1';
                    If SramControl_InternalDataWidth='1' Then
                        SramState_Next <= SramState_Read_Cycle0_Latch;
                    Else
                        SramState_Next <= SramState_Read_ACK;
                    End If;
                End If;
           ---------------------------------------------------------------------

           ---------------------------------------------------------------------
           When SramState_Write_Cycle0_Pulse =>
                SramControl_Transfer <= '1';
                SramWriteActive_Cycle <= '1';
                SramWE <= '0';
                If SramTimerIsZero='1' Then
                    SramState_Next <= SramState_Write_Cycle0_Wait;
                    SramLoadWriteHoldDelay <= '1';
                End If;

           When SramState_Write_Cycle0_Wait =>
                SramControl_Transfer <= '1';
                SramWriteActive_Cycle <= '1';
                If SramTimerIsZero='1' Then
                    SramState_Next <= SramState_Write_Cycle1_Setup;
                    SramLoadAddressDelay <= '1';
                End If;

           When SramState_Write_Cycle1_Setup =>
                SramControl_Transfer <= '1';
                SramWriteActive_Cycle <= '1';
                SramCycle_Number      <= '1';
                If SramTImerIsZero='1' Then
                    SramState_Next        <= SramState_Write_cycle1_Pulse;
                    SramLoadWritePulseDelay <= '1';
                End If;

           When SramState_Write_Cycle1_Pulse =>
                SramControl_Transfer <= '1';
                SramWriteActive_Cycle <= '1';
                SramWE                <= '0';
                SramCycle_Number      <= '1';
                If SramTimerIsZero='1' Then
                    SramState_Next        <= SramState_Write_ACK;
                    SramLoadWriteHoldDelay <= '1';
                End If;
           ---------------------------------------------------------------------

           ---------------------------------------------------------------------
           When SramState_Write_Pulse =>
                SramControl_Transfer <= '1';
                SramWriteActive_Cycle <= '1';
                SramWE                <= '0';
                If SramTimerIsZero='1' Then
                    SramState_Next        <= SramState_Write_ACK;
                    SramLoadWriteHoldDelay <= '1';
                End If;

           When SramState_Write_ACK =>
                SramControl_Transfer <= '1';
                SramWriteActive_Cycle <= '1';
                If SramTimerIsZero='1' Then
                    SramAcknowledge <= '1';
                    SramState_Next  <= SramState_WaitForAck;
                End if;
           ---------------------------------------------------------------------

           ---------------------------------------------------------------------
           When SramState_Read_Cycle0_Latch =>
                SramControl_Transfer <= '1';
                If SramTimerIsZero='1' Then
                    SramState_Next    <= SramState_Read_Cycle1_Setup;
                    SramLoadAddressDelay <= '1';
                End If;

           When SramState_Read_Cycle1_Setup =>
                SramControl_Transfer <= '1';
                SramCycle_Number  <= '1';
                If SramTimerIsZero='1' Then
                    SramState_Next    <= SramState_Read_ACK;
                End If;
           ---------------------------------------------------------------------

           ---------------------------------------------------------------------
           When SramState_Read_ACK =>
                SramControl_Transfer <= '1';
                SramCycle_Number  <= '1';
                If SramTimerIsZero='1' Then
                    SramAcknowledge <= '1';
                    SramState_Next    <= SramState_WaitForAck;
                End If;
           ---------------------------------------------------------------------

            When SramState_WaitForAck =>
                SramControl_Transfer <= '1';
                SramCycle_Number  <= '1';
                If SramWishbone_AcknowledgeSync='1' Then
                    SramState_Next <= SramState_AddressSetup;
                    SramControl_TransferFinished <= '1';
                End If;

           When Others =>
                SramState_Next <= SramState_AddressSetup;

        End Case;
    End Process;
    ----------------------------------------------------------------------------

    SramFSM_Synchronous:
    Process(MEM_CLK)
    Begin
        If Rising_Edge(MEM_CLK) Then
           If WB_SRAM_RST_I = '1' Then
              SramState <= SramState_AddressSetup;
           Else
              SramState <= SramState_Next;
           End If;
        End If;
    End Process;
    ----------------------------------------------------------------------------




    -- {ReplaceWith_SRAM_Logic_TimingController} End
    -- ------------------------------------------------------------------------
    -- {ReplaceWith_SRAM_Logic} End
    -- ------------------------------------------------------------------------
-- ----------------------------------------------------------------------------


-- ----------------------------------------------------------------------------
    -- ------------------------------------------------------------------------
    -- FLASH logic
    -- ------------------------------------------------------------------------
    -- {ReplaceWith_FLASH_Logic} Start
    -- ------------------------------------------------------------------------
    -- {ReplaceWith_FLASH_Logic_WishboneInterface} Start
    Process(WB_FLASH_CLK_I)
    Begin
        If Rising_Edge(WB_FLASH_CLK_I) Then
            If WB_FLASH_RST_I='1' Then
                FlashWishbone_TransferFinished_ff0 <= '0';
                FlashWishbone_TransferFinished_ff1 <= '0';
                FlashWishbone_TransferFinished     <= '0';
                FlashWishbone_AcknowledgeSignal    <= '0';
                FlashWishbone_RequestTransfer      <= '0';
            Else
                FlashWishbone_TransferFinished_ff0 <= FlashControl_TransferFinished;
                FlashWishbone_TransferFinished_ff1 <= FlashWishbone_TransferFinished_ff0;
                FlashWishbone_TransferFinished     <= FlashWishbone_TransferFinished_ff0 And Not FlashWishbone_AcknowledgeSignal And Not FlashWishbone_TransferFinished And FlashWishbone_Request;
                FlashWishbone_AcknowledgeSignal    <= FlashWishbone_TransferFinished;
                FlashWishbone_RequestTransfer      <= FlashWishbone_Request And Not FlashWishbone_AcknowledgeSignal And Not FlashWishbone_TransferFinished;
            End If;
        End If;
    End Process;
    WB_FLASH_ACK_O <= FlashWishbone_AcknowledgeSignal;

    Process(WB_FLASH_CLK_I)
    Begin
        If Rising_Edge(WB_FLASH_CLK_I) Then
            If WB_FLASH_RST_I='1' Then
                FlashWishbone_Request      <= '0';
                FlashWishbone_DataInput    <= (Others=>'0');
                FlashWishbone_Address      <= (Others=>'0');
                FlashWishbone_ByteSelect   <= (Others=>'0');
                FlashWishbone_WriteEnable  <= '0';
                FlashWishbone_RequestRead  <= '0';
                FlashWishbone_RequestWrite <= '0';
                WB_FLASH_DAT_O             <= (Others=>'0');
            Else
                FlashWishbone_Request      <= WB_FLASH_CYC_I And WB_FLASH_STB_I And Not FlashWishbone_AcknowledgeSignal And Not FlashWishbone_TransferFinished;
                FlashWishbone_DataInput    <= WB_FLASH_DAT_I;
                FlashWishbone_Address      <= WB_FLASH_ADR_I;
                FlashWishbone_ByteSelect   <= WB_FLASH_SEL_I;
                FlashWishbone_WriteEnable  <= WB_FLASH_WE_I;
                FlashWishbone_RequestRead  <= Not WB_FLASH_WE_I;
                FlashWishbone_RequestWrite <= WB_FLASH_WE_I;
                If FlashWishbone_TransferFinished='1' Then
                    WB_FLASH_DAT_O <= FlashWishbone_DataOutput;
                End If;
            End If;
        End If;
    End Process;
    -- {ReplaceWith_FLASH_Logic_WishboneInterface} End

    -- ------------------------------------------------------------------------

    -- {ReplaceWith_FLASH_Logic_TimingController} Start

    FlashPort_MemoryByteEnable <= (Others=>'1');
    ----------------------------------------------------------------------------
    Process(MEM_CLK)
    Begin
        If Rising_Edge(MEM_CLK) Then
            If WB_FLASH_RST_I='1' Then
                FlashPort_MemoryAddress <= (Others=>'0');
            Else
                FlashPort_MemoryAddress(0) <= FlashCycle;
                FlashPort_MemoryAddress(23 Downto 1) <= FlashWishbone_Address(24 Downto 2);
            End If;
        End If;
    End Process;
    ----------------------------------------------------------------------------
    ----------------------------------------------------------------------------
    FlashFlashDriveMemoryDataAndByteEnable:
    Process(FlashCycle, FlashIsWriting, FlashWishbone_DataInput)
    Begin
        FlashPort_MemoryDataEnable <= (Others=>'1');
        FlashPort_MemoryDataOutput <= (Others=>'0');
        If FlashIsWriting = '1' Then
            If FlashCycle = '0' Then
                FlashPort_MemoryDataOutput(15 Downto 0)  <= FlashWishbone_DataInput(31 Downto 16);
                FlashPort_MemoryDataOutput(31 Downto 16) <= (Others=>'0');
            Else
                FlashPort_MemoryDataOutput(15 Downto 0)  <= FlashWishbone_DataInput(15 Downto 0);
                FlashPort_MemoryDataOutput(31 Downto 16) <= (Others=>'0');
            End If;
            FlashPort_MemoryDataEnable <= (Others=>'0');
        End If;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    FlashDriveMemoryControl:
    Process(FlashIsCSActive, FlashIsWEActive, FlashIsOEActive, WB_FLASH_RST_I, FlashIsMemActive)
    Begin
        FlashPort_MemoryWrite         <= Not FlashIsWEActive;
        FlashPort_MemoryOutputEnable  <= Not FlashIsOEActive;
        FlashPort_MemoryReset         <= Not WB_FLASH_RST_I;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    FlashDriveWishboneSignals:
    Process(FlashReadDataLoWord_Reg,
            FlashPort_MemoryDataInput,
            FlashWishboneAck,
            FlashSkipCycle_1,
            FlashRequestRead)
    Begin
         If FlashSkipCycle_1='1' Then
             FlashWishbone_DataOutput(31 Downto 16) <= FlashPort_MemoryDataInput(15 Downto 0);
             FlashWishbone_DataOutput(15 Downto  0) <= FlashReadDataLoWord_Reg;
         Else
             FlashWishbone_DataOutput(31 Downto 16) <= FlashReadDataLoWord_Reg;
             FlashWishbone_DataOutput(15 Downto 0) <= FlashPort_MemoryDataInput(15 Downto 0);
         End If;
    End Process;
    ----------------------------------------------------------------------------


    Process(MEM_CLK)
    Begin
        If Rising_Edge(MEM_CLK) Then
            If WB_FLASH_RST_I='1' Then
                FlashControl_AccessRequest  <= '0';
                FlashAccessRequest_ff0 <= '0';
                FlashAccessRequest_ff1 <= '0';
                FlashAccessRequest_ff2 <= '0';
                FlashWishbone_AcknowledgeSync  <= '0';
            Else
                FlashAccessRequest_ff0 <= FlashWishbone_Request;
                FlashAccessRequest_ff1 <= FlashAccessRequest_ff0;
                FlashAccessRequest_ff2 <= FlashAccessRequest_ff1;
                FlashWishbone_AcknowledgeSync   <= FlashWishbone_TransferFinished;

                If FlashAccessRequestEdge='1' Then
                    FlashControl_AccessRequest <= '1';
                ElsIf FlashAcknowledge='1' Then
                    FlashControl_AccessRequest <= '0';
                End If;
            End If;
        End If;
    End Process;

    FlashAccessRequestEdge <= '1' When FlashAccessRequest_ff2='0' And FlashAccessRequest_ff1='1' And FlashAccessRequest_ff0='1' Else '0';

    Process(MEM_CLK)
    Begin
        If Rising_Edge(MEM_CLK) Then
            If WB_FLASH_RST_I='1' Then
                FlashControl_TransferDone <= '0';
            Else
                If FlashAcknowledge='1' Then
                    FlashControl_TransferDone <= '1';
                ElsIf FlashWishbone_AcknowledgeSync='1' Then
                    FlashControl_TransferDone <= '0';
                End If;
            End If;
        End If;
    End Process;


--    ----------------------------------------------------------------------------
--    FlashInterpretWishboneRequest:
--    Process(FlashWishbone_RequestRead,
--            FlashWishbone_RequestWrite,
--            FlashControl_FlashAccessGranted,
--            FlashRequest,
--            FlashWishbone_ByteSelect)
--    Begin
--        FlashRequest              <= (FlashWishbone_RequestRead Or FlashWishbone_RequestWrite) And FlashControl_FlashAccessGranted;
--        FlashRequestRead          <= FlashRequest And FlashWishbone_RequestRead;
--        FlashRequestWrite         <= FlashRequest And FlashWishbone_RequestWrite;
--        FlashSkipCycle_0          <= Not (FlashWishbone_ByteSelect(2) Or FlashWishbone_ByteSelect(3));
--        FlashSkipCycle_1          <= Not (FlashWishbone_ByteSelect(0) Or FlashWishbone_ByteSelect(1));
--    End Process;
--    ----------------------------------------------------------------------------

    Process(MEM_CLK)
    Begin
        If Rising_Edge(MEM_CLK) Then
            FlashRequest              <= (FlashWishbone_RequestRead Or FlashWishbone_RequestWrite) And FlashControl_FlashAccessGranted;
            FlashRequestRead          <= FlashRequest And FlashWishbone_RequestRead;
            FlashRequestWrite         <= FlashRequest And FlashWishbone_RequestWrite;
            FlashSkipCycle_0          <= Not (FlashWishbone_ByteSelect(2) Or FlashWishbone_ByteSelect(3));
            FlashSkipCycle_1          <= Not (FlashWishbone_ByteSelect(0) Or FlashWishbone_ByteSelect(1));
        End If;
    End Process;

    ----------------------------------------------------------------------------
    FlashCycleControl:
    Process(MEM_CLK,
            WB_FLASH_RST_I,
            FlashCycle,
            FlashSkipCycle_0,
            FlashSkipCycle_1)
    Begin
        If FlashCycle = '0' And FlashSkipCycle_1 = '0' Then
            FlashIsLastCycle <= '0';
        Else
            FlashIsLastCycle <= '1';
        End If;

        If WB_FLASH_RST_I = '1' Then
            FlashCycle <= '0';
        ElsIf Rising_Edge(MEM_CLK) Then
            If FlashResetCycle = '1' Then
                If FlashSkipCycle_0 = '1' Then
                    FlashCycle <= '1';
                Else
                    FlashCycle <= '0';
                End If;
            ElsIf FlashIncrementCycle = '1' Then
                FlashCycle <= '1';
            End If;
        End If;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    FlashRegisterWishboneData:
    Process(MEM_CLK)
    Begin
        If Rising_Edge(MEM_CLK) Then
           If FlashLoadReadData = '1' Then
               FlashReadDataLoWord_Reg <= FlashPort_MemoryDataInput(15 Downto 0);
           End If;
        End If;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    Process(MEM_CLK)
    Begin
        If Rising_Edge(MEM_CLK) Then
            If WB_FLASH_RST_I='1' Then
                FlashTimer <= (Others=>'0');
            Else
                   If FlashLoadTimerAddressSetup ='1' Then FlashTimer <= cCONFIG_FLASH_ADDRESS_SETUP;
                ElsIf FlashLoadTimerReadPulse    ='1' Then FlashTimer <= cCONFIG_FLASH_ADDRESS_SETUP;
                ElsIf FlashLoadTimerWritePulse   ='1' Then FlashTimer <= cCONFIG_FLASH_WRITE_PULSE;
                ElsIf FlashLoadTimerWriteHold    ='1' Then FlashTimer <= cCONFIG_FLASH_WRITE_HOLD;
                ElsIf FlashTimerIsZero           ='0' Then FlashTimer <= FlashTimer - 1;
                End If;
            End If;
        End If;
    End Process;
    ----------------------------------------------------------------------------
    FlashTimerIsZero <= '1' When FlashTimer=cFlashTimerIsZero Else '0';
    ----------------------------------------------------------------------------
    FlashFsm_Combinational:
    Process(FlashState,
            FlashRequestRead,
            FlashRequestWrite,
            FlashCycle,
            FlashIsLastcycle,
            FlashTimerIsZero,
            FlashWishbone_AcknowledgeSync)
    Begin
        ------------------------------------------------------------------------
        FlashIsMemActive              <= '1';
        FlashIsCSActive               <= '0';
        FlashIsWEActive               <= '0';
        FlashIsOEActive               <= '0';
        FlashIsWriting                <= '0';
        FlashLoadReadData             <= '0';
        FlashWishboneAck              <= '0';
        FlashResetCycle               <= '0';
        FlashIncrementCycle           <= '0';
        FlashState_Next               <= FlashState;
        FlashLoadTimerAddressSetup    <= '0';
        FlashLoadTimerReadPulse       <= '0';
        FlashLoadTimerWritePulse      <= '0';
        FlashLoadTimerWriteHold       <= '0';
        FlashControl_Transfer         <= '1';
        FlashAcknowledge              <= '0';
        FlashControl_TransferFinished <= '0';
        ------------------------------------------------------------------------

        ------------------------------------------------------------------------
        Case FlashState Is
            --------------------------------------------------------------------
            When FlashState_Idle =>
                FlashControl_Transfer <= '0';
                If FlashRequestRead = '1' Then
                    FlashLoadTimerAddressSetup <= '1';
                    FlashResetCycle            <= '1';
                    FlashState_Next            <= FlashState_Read_AddressSetup;
                ElsIf FlashRequestWrite = '1' Then
                    FlashIsWriting             <= '1';
                    FlashResetCycle            <= '1';
                    FlashState_Next            <= FlashState_Write_AddressSetup;
                    FlashLoadTimerAddressSetup <= '1';
                Else
                    FlashIsMemActive           <= '0';
                End If;
            --------------------------------------------------------------------

            --------------------------------------------------------------------
            When FlashState_Read_AddressSetup =>
                FlashIsCSActive <= '1';
                If FlashTimerIsZero = '1' Then
                    FlashLoadTimerReadPulse    <= '1';
                    FlashIsCSActive <= '1';
                    FlashState_Next            <= FlashState_Read_OEActive;
                End If;

            When FlashState_Read_OEActive =>
                FlashIsCSActive <= '1';
                FlashIsOEActive <= '1';
                If FlashTimerIsZero = '1' Then
                    If FlashIsLastCycle = '1' Then
                        Flash_Acknowledge <= '1';
                        FlashState_Next   <= FlashState_WaitForAck;
                    Else
                        FlashLoadReadData <= '1';
                        FlashState_Next   <= FlashState_Read_Hold;
                    End If;
                End If;

            When FlashState_Read_Hold =>
                FlashIsCSActive <= '1';
                If FlashTimerIsZero = '1' Then
                    FlashIncrementCycle        <= '1';
                    FlashState_Next            <= FlashState_Read_AddressSetup;
                    FlashLoadTimerAddressSetup <= '1';
                End If;

            When FlashState_Read_HoldLast =>
                FlashIsCSActive <= '1';
                If FlashTimerIsZero = '1' Then
                    FlashState_Next            <= FlashState_Idle;
                End If;
            --------------------------------------------------------------------

            --------------------------------------------------------------------
            When FlashState_Write_AddressSetup =>
                FlashIsWriting <= '1';
                FlashIsCSActive <= '1';
                If FlashTimerIsZero = '1' Then
                    FlashLoadTimerWritePulse <= '1';
                    FlashState_Next            <= FlashState_Write_WEActive;
                End If;

            When FlashState_Write_WEActive =>
                FlashIsWriting  <= '1';
                FlashIsCSActive <= '1';
                FlashIsWEActive <= '1';
                If FlashTimerIsZero = '1' Then
                    If FlashIsLastCycle = '0' Then
                        FlashLoadTimerWriteHold <= '1';
                        FlashState_Next            <= FlashState_Write_WEInactive;
                    Else
                        FlashLoadTimerWriteHold <= '1';
                        FlashState_Next            <= FlashState_Write_Hold;
                    End If;
                End If;

            When FlashState_Write_Ack =>
                FlashIsWriting             <= '1';
                FlashIsCSActive            <= '1';
                Flash_Acknowledge     <= '1';
                FlashState_Next            <= FlashState_WaitForAck;

            When FlashState_Write_WEInactive =>
                FlashIsWriting  <= '1';
                FlashIsCSActive <= '1';
                If FlashTimerIsZero = '1' Then
                    FlashIncrementCycle <= '1';
                    FlashLoadTimerAddressSetup <= '1';
                    FlashState_Next            <= FlashState_Write_AddressSetup;
                End If;

            When FlashState_Write_Hold =>
                FlashIsCSActive <= '1';
                FlashIsWriting <= '1';
                If FlashTimerIsZero = '1' Then
                    FlashState_Next            <= FlashState_Write_Ack;
                End If;

            When FlashState_WaitForAck =>
                FlashIsCSActive <= '1';
                If FlashWishbone_AcknowledgeSync='1' Then
                    FlashControl_TransferFinished <= '1';
                    FlashState_Next <= FlashState_Idle;
                End If;
            --------------------------------------------------------------------
            When Others =>
                FlashState_Next <= FlashState_Idle;
            --------------------------------------------------------------------

        End Case;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    FlashFsm_Synchronous:
    Process(MEM_CLK)
    Begin
        If Rising_Edge(MEM_CLK) Then
            If WB_FLASH_RST_I='1' Then
                FlashState <= FlashState_Idle;
            Else
                FlashState  <= FlashState_Next;
            End If;
        End If;
    End Process;
    ----------------------------------------------------------------------------
    -- {ReplaceWith_FLASH_Logic_TimingController} End
    -- ------------------------------------------------------------------------
    -- {ReplaceWith_FLASH_Logic} End
    -- ------------------------------------------------------------------------
-- ----------------------------------------------------------------------------

-- ----------------------------------------------------------------------------
    -- ------------------------------------------------------------------------
    -- MEMORY logic
    -- ------------------------------------------------------------------------
    -- {ReplaceWith_Memory_Logic} Start
    -- ------------------------------------------------------------------------

    Port_Mem_d_in( 0) <= MEM_D( 0);
    Port_Mem_d_in( 1) <= MEM_D( 1);
    Port_Mem_d_in( 2) <= MEM_D( 2);
    Port_Mem_d_in( 3) <= MEM_D( 3);
    Port_Mem_d_in( 4) <= MEM_D( 4);
    Port_Mem_d_in( 5) <= MEM_D( 5);
    Port_Mem_d_in( 6) <= MEM_D( 6);
    Port_Mem_d_in( 7) <= MEM_D( 7);
    Port_Mem_d_in( 8) <= MEM_D( 8);
    Port_Mem_d_in( 9) <= MEM_D( 9);
    Port_Mem_d_in(10) <= MEM_D(10);
    Port_Mem_d_in(11) <= MEM_D(11);
    Port_Mem_d_in(12) <= MEM_D(12);
    Port_Mem_d_in(13) <= MEM_D(13);
    Port_Mem_d_in(14) <= MEM_D(14);
    Port_Mem_d_in(15) <= MEM_D(15);
    Port_Mem_d_in(16) <= MEM_D(16);
    Port_Mem_d_in(17) <= MEM_D(17);
    Port_Mem_d_in(18) <= MEM_D(18);
    Port_Mem_d_in(19) <= MEM_D(19);
    Port_Mem_d_in(20) <= MEM_D(20);
    Port_Mem_d_in(21) <= MEM_D(21);
    Port_Mem_d_in(22) <= MEM_D(22);
    Port_Mem_d_in(23) <= MEM_D(23);
    Port_Mem_d_in(24) <= MEM_D(24);
    Port_Mem_d_in(25) <= MEM_D(25);
    Port_Mem_d_in(26) <= MEM_D(26);
    Port_Mem_d_in(27) <= MEM_D(27);
    Port_Mem_d_in(28) <= MEM_D(28);
    Port_Mem_d_in(29) <= MEM_D(29);
    Port_Mem_d_in(30) <= MEM_D(30);
    Port_Mem_d_in(31) <= MEM_D(31);


    MEM_D( 0) <= Port_Mem_d_out( 0) When Port_Mem_d_en( 0) = '0' Else 'Z';
    MEM_D( 1) <= Port_Mem_d_out( 1) When Port_Mem_d_en( 1) = '0' Else 'Z';
    MEM_D( 2) <= Port_Mem_d_out( 2) When Port_Mem_d_en( 2) = '0' Else 'Z';
    MEM_D( 3) <= Port_Mem_d_out( 3) When Port_Mem_d_en( 3) = '0' Else 'Z';
    MEM_D( 4) <= Port_Mem_d_out( 4) When Port_Mem_d_en( 4) = '0' Else 'Z';
    MEM_D( 5) <= Port_Mem_d_out( 5) When Port_Mem_d_en( 5) = '0' Else 'Z';
    MEM_D( 6) <= Port_Mem_d_out( 6) When Port_Mem_d_en( 6) = '0' Else 'Z';
    MEM_D( 7) <= Port_Mem_d_out( 7) When Port_Mem_d_en( 7) = '0' Else 'Z';
    MEM_D( 8) <= Port_Mem_d_out( 8) When Port_Mem_d_en( 8) = '0' Else 'Z';
    MEM_D( 9) <= Port_Mem_d_out( 9) When Port_Mem_d_en( 9) = '0' Else 'Z';
    MEM_D(10) <= Port_Mem_d_out(10) When Port_Mem_d_en(10) = '0' Else 'Z';
    MEM_D(11) <= Port_Mem_d_out(11) When Port_Mem_d_en(11) = '0' Else 'Z';
    MEM_D(12) <= Port_Mem_d_out(12) When Port_Mem_d_en(12) = '0' Else 'Z';
    MEM_D(13) <= Port_Mem_d_out(13) When Port_Mem_d_en(13) = '0' Else 'Z';
    MEM_D(14) <= Port_Mem_d_out(14) When Port_Mem_d_en(14) = '0' Else 'Z';
    MEM_D(15) <= Port_Mem_d_out(15) When Port_Mem_d_en(15) = '0' Else 'Z';
    MEM_D(16) <= Port_Mem_d_out(16) When Port_Mem_d_en(16) = '0' Else 'Z';
    MEM_D(17) <= Port_Mem_d_out(17) When Port_Mem_d_en(17) = '0' Else 'Z';
    MEM_D(18) <= Port_Mem_d_out(18) When Port_Mem_d_en(18) = '0' Else 'Z';
    MEM_D(19) <= Port_Mem_d_out(19) When Port_Mem_d_en(19) = '0' Else 'Z';
    MEM_D(20) <= Port_Mem_d_out(20) When Port_Mem_d_en(20) = '0' Else 'Z';
    MEM_D(21) <= Port_Mem_d_out(21) When Port_Mem_d_en(21) = '0' Else 'Z';
    MEM_D(22) <= Port_Mem_d_out(22) When Port_Mem_d_en(22) = '0' Else 'Z';
    MEM_D(23) <= Port_Mem_d_out(23) When Port_Mem_d_en(23) = '0' Else 'Z';
    MEM_D(24) <= Port_Mem_d_out(24) When Port_Mem_d_en(24) = '0' Else 'Z';
    MEM_D(25) <= Port_Mem_d_out(25) When Port_Mem_d_en(25) = '0' Else 'Z';
    MEM_D(26) <= Port_Mem_d_out(26) When Port_Mem_d_en(26) = '0' Else 'Z';
    MEM_D(27) <= Port_Mem_d_out(27) When Port_Mem_d_en(27) = '0' Else 'Z';
    MEM_D(28) <= Port_Mem_d_out(28) When Port_Mem_d_en(28) = '0' Else 'Z';
    MEM_D(29) <= Port_Mem_d_out(29) When Port_Mem_d_en(29) = '0' Else 'Z';
    MEM_D(30) <= Port_Mem_d_out(30) When Port_Mem_d_en(30) = '0' Else 'Z';
    MEM_D(31) <= Port_Mem_d_out(31) When Port_Mem_d_en(31) = '0' Else 'Z';

    -- ------------------------------------------------------------------------
    -- {ReplaceWith_Memory_Logic} End
    -- ------------------------------------------------------------------------
-- ----------------------------------------------------------------------------

-- ----------------------------------------------------------------------------
End RTL;
-- ----------------------------------------------------------------------------

