-- ---------------------------------------------------------------------------------------------------------------------------------------------
-- Global Package                                                                                                                 Global Package
-- ---------------------------------------------------------------------------------------------------------------------------------------------
Library IEEE;
Use     IEEE.Std_Logic_1164.All;
Use     IEEE.Std_Logic_Arith.All;
-- ---------------------------------------------------------------------------------------------------------------------------------------------

-- -----------------------------------------------------------------------------
Package Configurable_ReplaceWith_Designator_GlobalConfig Is
    Constant cWishboneAddressWidth  : Integer := ReplaceWith_WishboneAddressWidth;
    Constant cSdramDataWidth        : Integer := ReplaceWith_SdramDataWidth;
    Constant cSdramAddressWidth     : Integer := ReplaceWith_SdramAddressWidth;
    Constant cSdramBankAddressWidth : Integer := ReplaceWith_SdramBankAddressWidth;
    Constant cSdramByteEnableWidth  : Integer := ReplaceWith_SdramByteEnableWidth;
    Constant cMemAddressWidth       : Integer := ReplaceWith_MemAddressWidth;
    Constant cTimerWidth            : Integer :=  3;
    Constant cCACHE_ADR_WIDTH       : Integer := 11;

    -- Different IC geometries are resolved here
    Constant cFlushBankAddressMsb              : Integer := ReplaceWith_FlushBankAddressMsb;
    Constant cFlushBankAddressLsb              : Integer := ReplaceWith_FlushBankAddressLsb;
    Constant cFlushRowAddressMsb               : Integer := ReplaceWith_FlushRowAddressMsb;
    Constant cFlushRowAddressLsb               : Integer := ReplaceWith_FlushRowAddressLsb;
    Constant cFlushIndexRowAddressMsb          : Integer := ReplaceWith_FlushIndexRowAddressMsb;
    Constant cFlushIndexRowAddressLsb          : Integer := ReplaceWith_FlushIndexRowAddressLsb;
    Constant cFlushIndexMsb                    : Integer := ReplaceWith_FlushIndexMsb;
    Constant cFlushIndexLsb                    : Integer := ReplaceWith_FlushIndexLsb;
    Constant cReplaceBankAddressMsb            : Integer := ReplaceWith_ReplaceBankAddressMsb;
    Constant cReplaceBankAddressLsb            : Integer := ReplaceWith_ReplaceBankAddressLsb;
    Constant cReplaceRowAddressMsb             : Integer := ReplaceWith_ReplaceRowAddressMsb;
    Constant cReplaceRowAddressLsb             : Integer := ReplaceWith_ReplaceRowAddressLsb;
    Constant cReplaceColumnAddressMsb          : Integer := ReplaceWith_ReplaceColumnAddressMsb;
    Constant cReplaceColumnAddressLsb          : Integer := ReplaceWith_ReplaceColumnAddressLsb;

    Constant cReplaceColumnAddressPaddingWidth : Integer := ReplaceWith_ReplaceColumnAddressPaddingWidth;
    Constant cReplaceColumnAddressPadding      : Std_Logic_Vector(cReplaceColumnAddressPaddingWidth - 1 Downto 0) := Conv_Std_Logic_Vector(0, cReplaceColumnAddressPaddingWidth);

    Constant cFlushColumnAddressPaddingWidth   : Integer := ReplaceWith_FlushColumnAddressPaddingWidth;
    Constant cFlushColumnAddressPadding        : Std_Logic_Vector(cFlushColumnAddressPaddingWidth - 1 Downto 0) := Conv_Std_Logic_Vector(0, cFlushColumnAddressPaddingWidth);

    Constant cSdramMemoryAddressPaddingWidth   : Integer := ReplaceWith_SdramMemoryAddressPaddingWidth;
    Constant cSdramMemoryAddressPadding        : Std_Logic_Vector(cSdramMemoryAddressPaddingWidth - 1 Downto 0) := Conv_Std_Logic_Vector(0, cSdramMemoryAddressPaddingWidth);

    Constant cTM_tCL                  : Integer := ReplaceWith_tCL;
    Constant cTM_tRP                  : Integer := ReplaceWith_tRP;
    Constant cTM_tWR                  : Integer := ReplaceWith_tWR;
    Constant cTM_tRFC                 : Integer := ReplaceWith_tRFC;
    Constant cTM_tMRD                 : Integer := ReplaceWith_tMRD;
    Constant cTM_tRCD                 : Integer := ReplaceWith_tRCD;
    Constant cTM_Refresh              : Integer := ReplaceWith_Refresh;
    Constant cTimerValue_tRP        : Std_Logic_Vector(cTimerWidth - 1 Downto 0) := Conv_Std_Logic_Vector(cTM_tRP  - 1, cTimerWidth);
    Constant cTimerValue_tWR        : Std_Logic_Vector(cTimerWidth - 1 Downto 0) := Conv_Std_Logic_Vector(cTM_tWR  - 1, cTimerWidth);
    Constant cTimerValue_tRFC       : Std_Logic_Vector(cTimerWidth - 1 Downto 0) := Conv_Std_Logic_Vector(cTM_tRFC - 1, cTimerWidth);
    Constant cTimerValue_tMRD       : Std_Logic_Vector(cTimerWidth - 1 Downto 0) := Conv_Std_Logic_Vector(cTM_tMRD - 1, cTimerWidth);
    Constant cTimerValue_tRCD       : Std_Logic_Vector(cTimerWidth - 1 Downto 0) := Conv_Std_Logic_Vector(cTM_tRCD - 1, cTimerWidth);
    Constant cTimerValue_tCL        : Std_Logic_Vector(cTimerWidth - 1 Downto 0) := Conv_Std_Logic_Vector(cTM_tCL  - 1, cTimerWidth);
    Constant cTimerValue_Refresh    : Std_Logic_Vector(10 Downto 0) := Conv_Std_Logic_Vector(cTM_Refresh - 1, 11);

    Constant cBL_8                  : Integer   :=  3;
    Constant cBT_INTERLEAVED        : Std_Logic := '1';
    Constant cOP_MODE_NORMAL        : Integer   :=  0;
    Constant cWB_BURST              : Std_Logic := '0';
    Constant cRESERVED              : Integer   :=  0;
    -- -------------------------------------------------------------------------
    Constant cMODE_BURST_LENGTH     : Std_Logic_Vector( 2 Downto 0) := Conv_Std_Logic_Vector(cBL_8, 3);
    Constant cMODE_BURST_TYPE       : Std_Logic                     := cBT_INTERLEAVED;
    Constant cMODE_CAS_LATENCY      : Std_Logic_Vector( 2 Downto 0) := Conv_Std_Logic_Vector(cTM_tCL, 3);
    Constant cMODE_OPERATING_MODE   : Std_Logic_Vector( 1 Downto 0) := Conv_Std_Logic_Vector(cOP_MODE_NORMAL, 2);
    Constant cMODE_WRITE_BURST_MODE : Std_Logic                     := cWB_BURST;
    Constant cMODE_RESERVED         : Std_Logic_Vector( 2 Downto 0) := Conv_Std_Logic_Vector(cRESERVED, 3);
    Constant cLoadModeRegister      : Std_Logic_Vector(12 Downto 0) := cMODE_RESERVED         &
                                                                       cMODE_WRITE_BURST_MODE &
                                                                       cMODE_OPERATING_MODE   &
                                                                       cMODE_CAS_LATENCY      &
                                                                       cMODE_BURST_TYPE       &
                                                                       cMODE_BURST_LENGTH;

    -- -------------------------------------------------------------------------
    Constant cCACHE_CMD_ADR_CMD     : Std_Logic_Vector(1 Downto 0) := "00";  -- Command we pass to sdram engine
    Constant cCACHE_CMD_ADR_LINE    : Std_Logic_Vector(1 Downto 0) := "01";  -- Sdram: Used for Line Replace Address, For Asynch Port this is our Wishbone Address
    Constant cCACHE_CMD_ASYNCH_DATA : Std_Logic_Vector(1 Downto 0) := "10";  -- AsynchPort: Data Buffer (both ways)
    -- -------------------------------------------------------------------------
    Constant cCACHE_FLAG_CACHE_GO             : Integer := 31;
    Constant cCACHE_FLAG_REPLACE              : Integer := 30;
    Constant cCACHE_FLAG_FLUSH                : Integer := 29;
    Constant cCACHE_FLAG_DONE_ALL             : Integer := 28;
    Constant cCACHE_FLAG_DONE_REPLACE         : Integer := 27;
    Constant cCACHE_FLAG_DONE_FLUSH           : Integer := 26;
    Constant cCACHE_FLAG_DONE_REPLACE_CURRENT : Integer := 25;
    Constant cCACHE_FLAG_DONE_FLUSH_CURRENT   : Integer := 24;
    Constant cCACHE_FLAG_DONE_STARTUP         : Integer := 23;
    Constant cCACHE_FLAG_FLUSH_FOR_READ       : Integer := 22;
    Constant cCACHE_FLAG_FAST_REPLACE         : Integer := 21;
    Constant cCACHE_FLAG_FLUSH_FOR_WRITE      : Integer := 20;
    Constant cCACHE_FLAG_MEMORY_INITIALIZED   : Integer := 19;
    Constant cCACHE_FLAG_BE_MSB               : Integer := 18;
    Constant cCACHE_FLAG_BE_LSB               : Integer := 15;
    Constant cCACHE_FLAG_REQUEST_ASYNCH_PORT : Integer := 23;
    Constant cCACHE_FLAG_ASYNCH_PORT_GRANTED : Integer := 4;
    Constant cCACHE_FLAG_REQUEST_USB         : Integer := 2;
    Constant cCACHE_FLAG_REQUEST_FLASH       : Integer := 1;
    Constant cCACHE_FLAG_REQUEST_SRAM        : Integer := 0;
    -- -------------------------------------------------------------------------


    Constant cCACHE_ADR_LSB                : Integer := ReplaceWith_CacheAddressLsb; -- 4 for x32, 3 for x16

    Constant cWbPageRequestPadWidth        : Integer := ReplaceWith_WbPageRequestPadWidth;
    Constant cWbReplaceDestinationPadWidth : Integer := ReplaceWith_WbReplaceDestinationPadWidth;
    Constant cWbPageRequestPad             : Std_Logic_Vector(cWbPageRequestPadWidth        - 1 Downto 0) := Conv_Std_Logic_Vector(0, cWbPageRequestPadWidth);
    Constant cWbReplaceDestinationPad      : Std_Logic_Vector(cWbReplaceDestinationPadWidth - 1 Downto 0) := Conv_Std_Logic_Vector(0, cWbReplaceDestinationPadWidth);

    Constant cWishboneSramAddressWidth  : Integer := ReplaceWith_WishboneSramAddressWidth;
    Constant cWishboneFlashAddressWidth : Integer := ReplaceWith_WishboneFlashAddressWidth;
    Constant cWishboneUsbAddressWidth   : Integer := ReplaceWith_WishboneUsbAddressWidth;
End Package;
-- -----------------------------------------------------------------------------


-- ---------------------------------------------------------------------------------------------------------------------------------------------
-- Data Cache                                                                                                                         Data Cache
-- ---------------------------------------------------------------------------------------------------------------------------------------------
Library IEEE;
Use     IEEE.Std_Logic_1164.All;
-- ---------------------------------------------------------------------------------------------------------------------------------------------

-- -----------------------------------------------------------------------------
Entity Configurable_ReplaceWith_Designator_DataCache Is
Port (
     CLKA     : In  Std_Logic;
     DINA     : In  Std_Logic_Vector(31 Downto 0);
     DOUTA    : Out Std_Logic_Vector(31 Downto 0);
     ADDRA    : In  Std_Logic_Vector(10 Downto 0);
     BYTE_WEA : In  Std_Logic_Vector( 3 Downto 0);
     WEA      : In  Std_Logic;

     CLKB     : In  Std_Logic;
     DINB     : In  Std_Logic_Vector(31 Downto 0);
     DOUTB    : Out Std_Logic_Vector(31 Downto 0);
     ADDRB    : In  Std_Logic_Vector(10 Downto 0);
     BYTE_WEB : In  Std_Logic_Vector( 3 Downto 0);
     WEB      : In  Std_Logic
);
End;
-- -----------------------------------------------------------------------------

-- -----------------------------------------------------------------------------
Architecture RTL Of Configurable_ReplaceWith_Designator_DataCache Is
-- -----------------------------------------------------------------------------
    -- -------------------------------------------------------------------------
    Component Configurable_ReplaceWith_Designator_ByteMemory Is
        Port(
            CLKA    : In  Std_Logic;
            CLKB    : In  Std_Logic;
            WEA     : In  Std_Logic;
            WEB     : In  Std_Logic;
            DINA    : In  Std_Logic_Vector( 7 Downto 0);
            DINB    : In  Std_Logic_Vector( 7 Downto 0);
            DOUTA   : Out Std_Logic_Vector( 7 Downto 0);
            DOUTB   : Out Std_Logic_Vector( 7 Downto 0);
            ADDRA   : In  Std_Logic_Vector(10 Downto 0);
            ADDRB   : In  Std_Logic_Vector(10 Downto 0)
        );
    End Component;
    -- -------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    Signal B3DoutA        : Std_Logic_Vector(7 Downto 0);
    Signal B2DoutA        : Std_Logic_Vector(7 Downto 0);
    Signal B1DoutA        : Std_Logic_Vector(7 Downto 0);
    Signal B0DoutA        : Std_Logic_Vector(7 Downto 0);
    Signal B3DoutB        : Std_Logic_Vector(7 Downto 0);
    Signal B2DoutB        : Std_Logic_Vector(7 Downto 0);
    Signal B1DoutB        : Std_Logic_Vector(7 Downto 0);
    Signal B0DoutB        : Std_Logic_Vector(7 Downto 0);
    Signal B3WriteEnableA : Std_Logic;
    Signal B2WriteEnableA : Std_Logic;
    Signal B1WriteEnableA : Std_Logic;
    Signal B0WriteEnableA : Std_Logic;
    Signal B3WriteEnableB : Std_Logic;
    Signal B2WriteEnableB : Std_Logic;
    Signal B1WriteEnableB : Std_Logic;
    Signal B0WriteEnableB : Std_Logic;
    -- -------------------------------------------------------------------------

-- -----------------------------------------------------------------------------
Begin
-- -----------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    B3 : Configurable_ReplaceWith_Designator_ByteMemory Port Map (
       CLKA  => CLKA,
       CLKB  => CLKB,
       WEA   => B3WriteEnableA,
       WEB   => B3WriteEnableB,
       DINA  => DINA(31 Downto 24),
       DINB  => DINB(31 Downto 24),
       DOUTA => B3DoutA,
       DOUTB => B3DoutB,
       ADDRA => ADDRA,
       ADDRB => ADDRB
    );
    -- -------------------------------------------------------------------------
    B2 : Configurable_ReplaceWith_Designator_ByteMemory Port Map (
       CLKA  => CLKA,
       CLKB  => CLKB,
       WEA   => B2WriteEnableA,
       WEB   => B2WriteEnableB,
       DINA  => DINA(23 Downto 16),
       DINB  => DINB(23 Downto 16),
       DOUTA => B2DoutA,
       DOUTB => B2DoutB,
       ADDRA => ADDRA,
       ADDRB => ADDRB
    );
    -- -------------------------------------------------------------------------
    B1 : Configurable_ReplaceWith_Designator_ByteMemory Port Map (
       CLKA  => CLKA,
       CLKB  => CLKB,
       WEA   => B1WriteEnableA,
       WEB   => B1WriteEnableB,
       DINA  => DINA(15 Downto 8),
       DINB  => DINB(15 Downto 8),
       DOUTA => B1DoutA,
       DOUTB => B1DoutB,
       ADDRA => ADDRA,
       ADDRB => ADDRB
    );
    -- -------------------------------------------------------------------------
    B0 : Configurable_ReplaceWith_Designator_ByteMemory Port Map (
       CLKA  => CLKA,
       CLKB  => CLKB,
       WEA   => B0WriteEnableA,
       WEB   => B0WriteEnableB,
       DINA  => DINA(7 Downto 0),
       DINB  => DINB(7 Downto 0),
       DOUTA => B0DoutA,
       DOUTB => B0DoutB,
       ADDRA => ADDRA,
       ADDRB => ADDRB
    );
    -- -------------------------------------------------------------------------
    DOUTA <= B3DoutA & B2DoutA & B1DoutA & B0DoutA;
    B3WriteEnableA   <= WEA And BYTE_WEA(3);
    B2WriteEnableA   <= WEA And BYTE_WEA(2);
    B1WriteEnableA   <= WEA And BYTE_WEA(1);
    B0WriteEnableA   <= WEA And BYTE_WEA(0);
    -- -------------------------------------------------------------------------
    DOUTB <= B3DoutB & B2DoutB & B1DoutB & B0DoutB;
    B3WriteEnableB   <= WEB And BYTE_WEB(3);
    B2WriteEnableB   <= WEB And BYTE_WEB(2);
    B1WriteEnableB   <= WEB And BYTE_WEB(1);
    B0WriteEnableB   <= WEB And BYTE_WEB(0);
    -- -------------------------------------------------------------------------

-- -----------------------------------------------------------------------------
End RTL;
-- -----------------------------------------------------------------------------

-- ---------------------------------------------------------------------------------------------------------------------------------------------
-- Tag Cache                                                                                                                           Tag Cache
-- ---------------------------------------------------------------------------------------------------------------------------------------------
Library IEEE;
Use     IEEE.Std_Logic_1164.All;
-- ---------------------------------------------------------------------------------------------------------------------------------------------

-- -----------------------------------------------------------------------------
Entity Configurable_ReplaceWith_Designator_TagCache Is
Port (
    CLKA  : In  Std_Logic;
    DINA  : In  Std_Logic_Vector(19 Downto 0);
    DOUTA : Out Std_Logic_Vector(19 Downto 0);
    ADDRA : In  Std_Logic_Vector( 7 Downto 0);
    WEA   : In  Std_Logic;

    CLKB  : In  Std_Logic;
    DINB  : In  Std_Logic_Vector(19 Downto 0);
    DOUTB : Out Std_Logic_Vector(19 Downto 0);
    ADDRB : In  Std_Logic_Vector( 7 Downto 0);
    WEB   : In  Std_Logic
);
End;
-- -----------------------------------------------------------------------------

-- -----------------------------------------------------------------------------
Architecture RTL Of Configurable_ReplaceWith_Designator_TagCache Is
-- -----------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    Component Configurable_ReplaceWith_Designator_TagMemory Is
    Port (
        CLKA  : In  Std_Logic;
        CLKB  : In  Std_Logic;
        WEA   : In  Std_Logic;
        WEB   : In  Std_Logic;
        DINA  : In  Std_Logic_Vector(19 Downto 0);
        DINB  : In  Std_Logic_Vector(19 Downto 0);
        DOUTA : Out Std_Logic_Vector(19 Downto 0);
        DOUTB : Out Std_Logic_Vector(19 Downto 0);
        ADDRA : In  Std_Logic_Vector( 7 Downto 0);
        ADDRB : In  Std_Logic_Vector( 7 Downto 0)
    );
    End Component;
    -- -------------------------------------------------------------------------


-- -----------------------------------------------------------------------------
Begin
-- -----------------------------------------------------------------------------

    U_Configurable_ReplaceWith_Designator_TagCache : Configurable_ReplaceWith_Designator_TagMemory Port Map (
        CLKA  => CLKA,
        CLKB  => CLKB,
        WEA   => WEA,
        WEB   => WEB,
        DINA  => DINA,
        DINB  => DINB,
        DOUTA => DOUTA,
        DOUTB => DOUTB,
        ADDRA => ADDRA,
        ADDRB => ADDRB
    );

-- -----------------------------------------------------------------------------
End RTL;
-- -----------------------------------------------------------------------------


-- ---------------------------------------------------------------------------------------------------------------------------------------------
-- Cmd Pipe                                                                                                                             Cmd Pipe
-- ---------------------------------------------------------------------------------------------------------------------------------------------
Library IEEE;
Use     IEEE.Std_Logic_1164.All;
-- ---------------------------------------------------------------------------------------------------------------------------------------------

-- -----------------------------------------------------------------------------
Entity Configurable_ReplaceWith_Designator_CommandPipe Is
Port (
    CLKA  : In  Std_Logic;
    DINA  : In  Std_Logic_Vector(31 Downto 0);
    DOUTA : Out Std_Logic_Vector(31 Downto 0);
    ADDRA : In  Std_Logic_Vector( 1 Downto 0);
    WEA   : In  Std_Logic;

    CLKB  : In  Std_Logic;
    DINB  : In  Std_Logic_Vector(31 Downto 0);
    DOUTB : Out Std_Logic_Vector(31 Downto 0);
    ADDRB : In  Std_Logic_Vector( 1 Downto 0);
    WEB   : In  Std_Logic
);
End;
-- -----------------------------------------------------------------------------

-- -----------------------------------------------------------------------------
Architecture RTL Of Configurable_ReplaceWith_Designator_CommandPipe Is
-- -----------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    Component Configurable_ReplaceWith_Designator_CmdMemory Is
    Port (
        CLKA  : In  Std_Logic;
        CLKB  : In  Std_Logic;
        WEA   : In  Std_Logic;
        WEB   : In  Std_Logic;
        DINA  : In  Std_Logic_Vector(31 Downto 0);
        DINB  : In  Std_Logic_Vector(31 Downto 0);
        DOUTA : Out Std_Logic_Vector(31 Downto 0);
        DOUTB : Out Std_Logic_Vector(31 Downto 0);
        ADDRA : In  Std_Logic_Vector( 1 Downto 0);
        ADDRB : In  Std_Logic_Vector( 1 Downto 0)
    );
    End Component;
    -- -------------------------------------------------------------------------

-- -----------------------------------------------------------------------------
Begin
-- -----------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    U_Configurable_ReplaceWith_Designator_CmdMemory : Configurable_ReplaceWith_Designator_CmdMemory Port Map (
        CLKA  => CLKA,
        CLKB  => CLKB,
        WEA   => WEA,
        WEB   => WEB,
        DINA  => DINA,
        DINB  => DINB,
        DOUTA => DOUTA,
        DOUTB => DOUTB,
        ADDRA => ADDRA,
        ADDRB => ADDRB
    );
    -- -------------------------------------------------------------------------

-- -----------------------------------------------------------------------------
End RTL;
-- -----------------------------------------------------------------------------


-- ---------------------------------------------------------------------------------------------------------------------------------------------
-- Asynchronous Port Select                                                                                                   Asynch Port Select
-- ---------------------------------------------------------------------------------------------------------------------------------------------

-- -----------------------------------------------------------------------------
Library IEEE;
Use     IEEE.Std_Logic_1164.All;
-- -----------------------------------------------------------------------------

-- -----------------------------------------------------------------------------
Entity Configurable_ReplaceWith_Designator_ASYNCH_PORT_SELECT Is
Port (
    RST_I         : In Std_Logic;
    CLK_I         : In Std_Logic;

    --SRAM WISHBONE
    SRAM_STB_O    : Out Std_Logic;
    SRAM_CYC_O    : Out Std_Logic;
    SRAM_DAT_O    : Out Std_Logic_Vector(31 Downto 0);
    SRAM_DAT_I    : In  Std_Logic_Vector(31 Downto 0);
    SRAM_ADR_O    : Out Std_Logic_Vector(23 Downto 0);
    SRAM_SEL_O    : Out Std_Logic_Vector(3 Downto 0);
    SRAM_WE_O     : Out Std_Logic;
    SRAM_ACK_I    : In  Std_Logic;
    SRAM_RST_O    : Out Std_Logic;
    SRAM_CLK_O    : Out Std_Logic;
    --SRAM MEMORY
    SRAM_MEM_D_O  : Out Std_Logic_Vector(31 Downto 0);
    SRAM_MEM_D_I  : In  Std_Logic_Vector(31 Downto 0);
    SRAM_MEM_D_E  : In  Std_Logic;
    SRAM_MEM_A_I  : In  Std_Logic_Vector(23 Downto 0);
    SRAM_MEM_BE_W : In  Std_Logic_Vector(3 Downto 0);
    SRAM_MEM_OE   : In  Std_Logic;
    SRAM_MEM_WE   : In  Std_Logic;
    SRAM_MEM_CE   : In  Std_Logic;
    -- FLASH WISHBONE
    FLASH_STB_O   : Out Std_Logic;
    FLASH_CYC_O   : Out Std_Logic;
    FLASH_DAT_O   : Out Std_Logic_Vector(31 Downto 0);
    FLASH_DAT_I   : In  Std_Logic_Vector(31 Downto 0);
    FLASH_ADR_O   : Out Std_Logic_Vector(23 Downto 0);
    FLASH_SEL_O   : Out Std_Logic_Vector(3 Downto 0);
    FLASH_WE_O    : Out Std_Logic;
    FLASH_ACK_I   : In  Std_Logic;
    FLASH_RST_O   : Out Std_Logic;
    FLASH_CLK_O   : Out Std_Logic;
    -- FLASH MEMORY
    FLASH_MEM_D_O : Out Std_Logic_Vector(31 Downto 0);
    FLASH_MEM_D_I : In  Std_Logic_Vector(31 Downto 0);
    FLASH_MEM_D_E : In  Std_Logic;
    FLASH_MEM_A_I : In Std_Logic_Vector(23 Downto 0);
    FLASH_MEM_OE  : In Std_Logic;
    FLASH_MEM_WE  : In Std_Logic;
    FLASH_MEM_CE  : In  Std_Logic;
    --USB WISHBONE
    USB_STB_O    : Out Std_Logic;
    USB_CYC_O    : Out Std_Logic;
    USB_DAT_O    : Out Std_Logic_Vector(31 Downto 0);
    USB_DAT_I    : In  Std_Logic_Vector(31 Downto 0);
    USB_ADR_O    : Out Std_Logic_Vector(23 Downto 0);
    USB_SEL_O    : Out Std_Logic_Vector(3 Downto 0);
    USB_WE_O     : Out Std_Logic;
    USB_ACK_I    : In  Std_Logic;
    USB_RST_O    : Out Std_Logic;
    USB_CLK_O    : Out Std_Logic;
    --USB MEMORY
    USB_MEM_D_O  : Out Std_Logic_Vector(31 Downto 0);
    USB_MEM_D_I  : In  Std_Logic_Vector(31 Downto 0);
    USB_MEM_D_E  : In  Std_Logic;
    USB_MEM_A_I  : In  Std_Logic_Vector(23 Downto 0);
    USB_MEM_BE_W : In  Std_Logic_Vector(3 Downto 0);
    USB_MEM_OE   : In  Std_Logic;
    USB_MEM_WE   : In  Std_Logic;
    USB_MEM_CE   : In  Std_Logic;
    USB_MEM_RESET : In Std_Logic;

    -- engine side
    ASYNCH_PORT_MEM_DATA_I        : in    std_logic_vector(31 downto 0);
    ASYNCH_PORT_MEM_ADR_O         : out   std_logic_vector(23 downto 0);
    ASYNCH_PORT_MEM_BE_O          : out   std_logic_vector(3 downto 0);
    ASYNCH_PORT_MEM_DATA_O        : out   std_logic_vector(31 downto 0);
    ASYNCH_PORT_MEM_DATA_WRITE_O  : out   std_logic;
    ASYNCH_PORT_MEM_FLASH_CS_O    : out   std_logic;
    ASYNCH_PORT_MEM_FLASH_RESET_O : out   std_logic;
    ASYNCH_PORT_MEM_OE_O          : out   std_logic;
    ASYNCH_PORT_MEM_SRAM_CS_O     : out   std_logic;
    ASYNCH_PORT_MEM_USB_CS_O      : out   std_logic;
    ASYNCH_PORT_MEM_USB_RESET_O   : out   std_logic;
    ASYNCH_PORT_MEM_WE_O          : out   std_logic;
    -- wishbone side
    TO_MEM_CTRL_ACK_O             : out   std_logic;
    TO_MEM_CTRL_ADR_I             : in    std_logic_vector(23 downto 0);
    TO_MEM_CTRL_DATA_I            : in    std_logic_vector(31 downto 0);
    TO_MEM_CTRL_DATA_O            : out   std_logic_vector(31 downto 0);
    TO_MEM_CTRL_FLASH_GO          : in    std_logic;
    TO_MEM_CTRL_SEL_O             : in    std_logic_vector(3 downto 0);
    TO_MEM_CTRL_SRAM_GO           : in    std_logic;
    TO_MEM_CTRL_USB_GO            : in    std_logic;
    TO_MEM_CTRL_WE_I              : In    Std_Logic
);
End;
-- -----------------------------------------------------------------------------

-- -----------------------------------------------------------------------------
Architecture RTL Of Configurable_ReplaceWith_Designator_ASYNCH_PORT_SELECT Is
-- -----------------------------------------------------------------------------
    Signal DataOutput : Std_Logic_Vector(31 Downto 0);
-- -----------------------------------------------------------------------------
Begin
-- -----------------------------------------------------------------------------

    -- wishbone side
    TO_MEM_CTRL_ACK_O  <= SRAM_ACK_I Or USB_ACK_I Or FLASH_ACK_I;

    -- -------------------------------------------------------------------------
    --SRAM FAKE_WISHBONE
    -- -------------------------------------------------------------------------
    SRAM_STB_O   <= TO_MEM_CTRL_SRAM_GO;
    SRAM_CYC_O   <= TO_MEM_CTRL_SRAM_GO;
    SRAM_DAT_O   <= TO_MEM_CTRL_DATA_I;
    SRAM_ADR_O   <= TO_MEM_CTRL_ADR_I;
    SRAM_SEL_O   <= TO_MEM_CTRL_SEL_O;
    SRAM_WE_O    <= TO_MEM_CTRL_WE_I;
    SRAM_CLK_O   <= CLK_I;
    SRAM_MEM_D_O <= ASYNCH_PORT_MEM_DATA_I;
    -- -------------------------------------------------------------------------
    FLASH_STB_O   <= TO_MEM_CTRL_FLASH_GO;
    FLASH_CYC_O   <= TO_MEM_CTRL_FLASH_GO;
    FLASH_DAT_O   <= TO_MEM_CTRL_DATA_I;
    FLASH_ADR_O   <= TO_MEM_CTRL_ADR_I;
    FLASH_SEL_O   <= TO_MEM_CTRL_SEL_O;
    FLASH_WE_O    <= TO_MEM_CTRL_WE_I;
    FLASH_CLK_O   <= CLK_I;
    FLASH_MEM_D_O <= ASYNCH_PORT_MEM_DATA_I;
    -- -------------------------------------------------------------------------
    USB_STB_O   <= TO_MEM_CTRL_USB_GO;
    USB_CYC_O   <= TO_MEM_CTRL_USB_GO;
    USB_DAT_O   <= TO_MEM_CTRL_DATA_I;
    USB_ADR_O   <= TO_MEM_CTRL_ADR_I;
    USB_SEL_O   <= TO_MEM_CTRL_SEL_O;
    USB_WE_O    <= TO_MEM_CTRL_WE_I;
    USB_CLK_O   <= CLK_I;
    USB_MEM_D_O <= ASYNCH_PORT_MEM_DATA_I;
    -- -------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    -- -------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    Process(TO_MEM_CTRL_SRAM_GO, TO_MEM_CTRL_USB_GO, TO_MEM_CTRL_FLASH_GO,
            SRAM_MEM_A_I, SRAM_MEM_BE_W, SRAM_MEM_D_I, SRAM_MEM_D_E, SRAM_MEM_CE, SRAM_MEM_WE, SRAM_MEM_OE,
            USB_MEM_A_I, USB_MEM_BE_W, USB_MEM_D_I, USB_MEM_D_E, USB_MEM_CE, USB_MEM_WE, USB_MEM_OE,
            FLASH_MEM_A_I, FLASH_MEM_D_I, FLASH_MEM_D_E, FLASH_MEM_CE, FLASH_MEM_WE, FLASH_MEM_OE)
    Begin
         ASYNCH_PORT_MEM_ADR_O        <= (Others=>'0');
         ASYNCH_PORT_MEM_BE_O         <= (Others=>'1');
         ASYNCH_PORT_MEM_DATA_O       <= (Others=>'0');
         ASYNCH_PORT_MEM_DATA_WRITE_O <= '0';
         ASYNCH_PORT_MEM_FLASH_CS_O   <= '1';
         ASYNCH_PORT_MEM_SRAM_CS_O    <= '1';
         ASYNCH_PORT_MEM_USB_CS_O     <= '1';
         ASYNCH_PORT_MEM_WE_O         <= '1';
         ASYNCH_PORT_MEM_OE_O         <= '1';
         If TO_MEM_CTRL_SRAM_GO='1' Then
             ASYNCH_PORT_MEM_ADR_O        <= SRAM_MEM_A_I;
             ASYNCH_PORT_MEM_BE_O         <= SRAM_MEM_BE_W;
             ASYNCH_PORT_MEM_DATA_O       <= SRAM_MEM_D_I;
             ASYNCH_PORT_MEM_DATA_WRITE_O <= SRAM_MEM_D_E;
             ASYNCH_PORT_MEM_SRAM_CS_O    <= SRAM_MEM_CE;
             ASYNCH_PORT_MEM_WE_O         <= SRAM_MEM_WE;
             ASYNCH_PORT_MEM_OE_O         <= SRAM_MEM_OE;
         ElsIf TO_MEM_CTRL_USB_GO='1' Then
             ASYNCH_PORT_MEM_ADR_O        <= USB_MEM_A_I;
             ASYNCH_PORT_MEM_BE_O         <= USB_MEM_BE_W;
             ASYNCH_PORT_MEM_DATA_O       <= USB_MEM_D_I;
             ASYNCH_PORT_MEM_DATA_WRITE_O <= USB_MEM_D_E;
             ASYNCH_PORT_MEM_USB_CS_O     <= USB_MEM_CE;
             ASYNCH_PORT_MEM_WE_O         <= USB_MEM_WE;
             ASYNCH_PORT_MEM_OE_O         <= USB_MEM_OE;
         ElsIf TO_MEM_CTRL_FLASH_GO='1' Then
             ASYNCH_PORT_MEM_ADR_O        <= FLASH_MEM_A_I;
             ASYNCH_PORT_MEM_DATA_O       <= FLASH_MEM_D_I;
             ASYNCH_PORT_MEM_DATA_WRITE_O <= FLASH_MEM_D_E;
             ASYNCH_PORT_MEM_FLASH_CS_O   <= FLASH_MEM_CE;
             ASYNCH_PORT_MEM_WE_O         <= FLASH_MEM_WE;
             ASYNCH_PORT_MEM_OE_O         <= FLASH_MEM_OE;
         End If;
    End Process;
    -- -------------------------------------------------------------------------
    ASYNCH_PORT_MEM_FLASH_RESET_O <= '1';
    ASYNCH_PORT_MEM_USB_RESET_O   <= USB_MEM_RESET;
    -- -------------------------------------------------------------------------
    Process(CLK_I)
    Begin
        If Rising_Edge(CLK_I) Then
            FLASH_RST_O <= RST_I;
            SRAM_RST_O  <= RST_I;
            USB_RST_O   <= RST_I;
            If TO_MEM_CTRL_SRAM_GO='1' Or SRAM_ACK_I='1' Then
                  DataOutput <= SRAM_DAT_I;
            ElsIf TO_MEM_CTRL_USB_GO='1'  Or USB_ACK_I='1' Then
                  DataOutput <= USB_DAT_I;
            ElsIf TO_MEM_CTRL_FLASH_GO='1' Or FLASH_ACK_I='1'Then
                  DataOutput <= FLASH_DAT_I;
            End If;
        End If;
    End Process;

    TO_MEM_CTRL_DATA_O <= DataOutput;
    -- -------------------------------------------------------------------------
-- -----------------------------------------------------------------------------
End;
-- -----------------------------------------------------------------------------

-- ---------------------------------------------------------------------------------------------------------------------------------------------
-- Wishbone Interface                                                                                                         Wishbone Interface
-- ---------------------------------------------------------------------------------------------------------------------------------------------
Library IEEE;
Use     IEEE.Std_Logic_1164.All;
Use     Work.Configurable_ReplaceWith_Designator_GlobalConfig.All;
-- ---------------------------------------------------------------------------------------------------------------------------------------------

-- -----------------------------------------------------------------------------
Entity Configurable_ReplaceWith_Designator_WishboneInterface Is
Port (
     Debug_WishboneState : Out Std_Logic_Vector(3 Downto 0);
    -- Cache Command Interface
    CACHE_CMD_CLK     : Out Std_Logic;
    CACHE_CMD_WE_O    : Out Std_Logic;
    CACHE_CMD_DAT_I   : In  Std_Logic_Vector(31 Downto 0);
    CACHE_CMD_DAT_O   : Out Std_Logic_Vector(31 Downto 0);
    CACHE_CMD_ADR_O   : Out Std_Logic_Vector( 1 Downto 0);

    -- Dual Port RAM Wishbone Data Cache Interface
    DP_RAM_DAT_CLK_O  : Out Std_Logic;
    DP_RAM_DAT_RST_O  : Out Std_Logic;
    DP_RAM_DAT_DATA_I : In  Std_Logic_Vector(31 Downto 0);
    DP_RAM_DAT_DATA_O : Out Std_Logic_Vector(31 Downto 0);
    DP_RAM_DAT_ADR_O  : Out Std_Logic_Vector(cCACHE_ADR_WIDTH - 1 Downto 0);
    DP_RAM_DAT_BE_O   : Out Std_Logic_Vector( 3 Downto 0);
    DP_RAM_DAT_WE_O   : Out Std_Logic;

    -- Dual Port RAM Cache Tag Interface
    DP_RAM_TAG_CLK_O  : Out Std_Logic;
    DP_RAM_TAG_RST_O  : Out Std_Logic;
    DP_RAM_TAG_DATA_I : In  Std_Logic_Vector(19 Downto 0);
    DP_RAM_TAG_DATA_O : Out Std_Logic_Vector(19 Downto 0);
    DP_RAM_TAG_ADR_O  : Out Std_Logic_Vector(cCACHE_ADR_WIDTH - 4 Downto 0);
    DP_RAM_TAG_BE_O   : Out Std_Logic_Vector( 3 Downto 0);
    DP_RAM_TAG_WE_O   : Out Std_Logic;

    ASYNCH_PORT_MEM_DATA_I   : In  Std_Logic_Vector(31 Downto 0);
    ASYNCH_PORT_MEM_DATA_O   : Out Std_Logic_Vector(31 Downto 0);
    ASYNCH_PORT_MEM_ADR_O    : Out Std_Logic_Vector(23 Downto 0);
    ASYNCH_PORT_MEM_SEL_O    : Out Std_Logic_Vector( 3 Downto 0);
    ASYNCH_PORT_MEM_SRAM_GO  : Out Std_Logic;
    ASYNCH_PORT_MEM_FLASH_GO : Out Std_Logic;
    ASYNCH_PORT_MEM_USB_GO   : Out Std_Logic;
    ASYNCH_PORT_MEM_ACK_I    : In  Std_Logic;
    ASYNCH_PORT_MEM_WRITE    : Out Std_Logic;

    -- Sram Wishbone Interface
    WB_SRAM_CLK_I : In  Std_Logic;
    WB_SRAM_RST_I : In  Std_Logic;
    WB_SRAM_STB_I : In  Std_Logic;
    WB_SRAM_CYC_I : In  Std_Logic;
    WB_SRAM_ACK_O : Out Std_Logic;
    WB_SRAM_ADR_I : In  Std_Logic_Vector(cWishboneSramAddressWidth - 1 Downto 0);
    WB_SRAM_DAT_I : In  Std_Logic_Vector(31 Downto 0);
    WB_SRAM_DAT_O : Out Std_Logic_Vector(31 Downto 0);
    WB_SRAM_SEL_I : In  Std_Logic_Vector( 3 Downto 0);
    WB_SRAM_WE_I  : In  Std_Logic;
    -- Flash Wishbone Interface
    WB_FLASH_CLK_I : In  Std_Logic;
    WB_FLASH_RST_I : In  Std_Logic;
    WB_FLASH_STB_I : In  Std_Logic;
    WB_FLASH_CYC_I : In  Std_Logic;
    WB_FLASH_ACK_O : Out Std_Logic;
    WB_FLASH_ADR_I : In  Std_Logic_Vector(cWishboneFlashAddressWidth - 1 Downto 0);
    WB_FLASH_DAT_I : In  Std_Logic_Vector(31 Downto 0);
    WB_FLASH_DAT_O : Out Std_Logic_Vector(31 Downto 0);
    WB_FLASH_SEL_I : In  Std_Logic_Vector( 3 Downto 0);
    WB_FLASH_WE_I  : In  Std_Logic;
    -- Usb Wishbone Interface                                                    --Incl__udeIf_ComponentVersion_2
    WB_USB_CLK_I : In  Std_Logic;                                                --Incl__udeIf_ComponentVersion_2
    WB_USB_RST_I : In  Std_Logic;                                                --Incl__udeIf_ComponentVersion_2
    WB_USB_STB_I : In  Std_Logic;                                                --Incl__udeIf_ComponentVersion_2
    WB_USB_CYC_I : In  Std_Logic;                                                --Incl__udeIf_ComponentVersion_2
    WB_USB_ACK_O : Out Std_Logic;                                                --Incl__udeIf_ComponentVersion_2
    WB_USB_ADR_I : In  Std_Logic_Vector(cWishboneUsbAddressWidth - 1 Downto 0);  --Incl__udeIf_ComponentVersion_2
    WB_USB_DAT_I : In  Std_Logic_Vector(31 Downto 0);                            --Incl__udeIf_ComponentVersion_2
    WB_USB_DAT_O : Out Std_Logic_Vector(31 Downto 0);                            --Incl__udeIf_ComponentVersion_2
    WB_USB_SEL_I : In  Std_Logic_Vector( 3 Downto 0);                            --Incl__udeIf_ComponentVersion_2
    WB_USB_WE_I  : In  Std_Logic;                                                --Incl__udeIf_ComponentVersion_2
    -- Sdram Wishbone Interface
    WB_SDRAM_CLK_I          : In  Std_Logic;
    WB_SDRAM_RST_I          : In  Std_Logic;
    WB_SDRAM_STB_I          : In  Std_Logic;
    WB_SDRAM_CYC_I          : In  Std_Logic;
    WB_SDRAM_ACK_O          : Out Std_Logic;
    WB_SDRAM_ADR_I          : In  Std_Logic_Vector(cWishboneAddressWidth - 1 Downto 0);
    WB_SDRAM_DAT_I          : In  Std_Logic_Vector(31 Downto 0);
    WB_SDRAM_DAT_O          : Out Std_Logic_Vector(31 Downto 0);
    WB_SDRAM_SEL_I          : In  Std_Logic_Vector( 3 Downto 0);
    WB_SDRAM_WE_I           : In  Std_Logic
);
End;
-- -----------------------------------------------------------------------------

-- -----------------------------------------------------------------------------
Architecture RTL Of Configurable_ReplaceWith_Designator_WishboneInterface Is
-- -----------------------------------------------------------------------------

    Type TCacheState Is ( sCacheIdle,
                          sCacheAsynchPortInProgress,
                          cCacheAsynchPortClose,
                          sCacheHitCheckRead,
                          sCacheHitCheckWrite,
                          sCacheWaitForLineReplace,
                          sCacheWaitForLineFlushed );

    Signal CacheState     : TCacheState;
    Signal CacheStateNext : TCacheState;

    Signal WishboneRequest            : Std_Logic;
    Signal WishboneWrite              : Std_Logic;
    Signal WishboneRead               : Std_Logic;
    Signal DataIndex                  : Std_Logic_Vector(cCACHE_ADR_WIDTH - 1 Downto 0);
    Signal TagIndex                   : Std_Logic_Vector(cCACHE_ADR_WIDTH - 4 Downto 0);

    Signal PageActual                 : Std_Logic_Vector(19 Downto 0);
    Signal PageRequested              : Std_Logic_Vector(19 Downto 0);
    Signal PageRequested_Reg          : Std_Logic_Vector(19 Downto 0);
    Signal CacheHit                   : Std_Logic;

    Signal DoWishBoneWriteAcknowledge : Std_Logic;
    Signal DoWishboneReadAcknowledge  : Std_Logic;

    Signal DoCacheWrite               : Std_Logic;

    Signal DoCache_PrepareReplaceLine : Std_Logic;
    Signal DoCache_ReplaceLine        : Std_Logic;
    Signal DoCache_FlushLine          : Std_Logic;

    Signal CacheDone_All              : Std_Logic;
    Signal CacheDone_Replace          : Std_Logic;
    Signal CacheDone_Flush            : Std_Logic;
    Signal Cache_Status               : Std_Logic_Vector(31 Downto 0);
    Signal Cache_Write                : Std_Logic;
    Signal Cache_Command              : Std_Logic_Vector(31 Downto 0);
    Signal Cache_CommandAsynchPort    : Std_Logic_Vector(31 Downto 0);
    Signal Cache_Data_Out             : Std_Logic_Vector(31 Downto 0);
    Signal Cache_Address              : Std_Logic_Vector( 1 Downto 0);
    Signal CommandRequest             : Std_Logic;
    Signal CacheCanGo                 : Std_Logic;

    Signal ReplaceDestinationAddress         : Std_Logic_Vector(31 Downto 0);
    Signal DoAsynchPort_Acknowledge          : Std_Logic;

    Signal DoAsynchPort_WriteAddress  : Std_Logic;
    Signal DoAsynchPort_WriteData     : Std_Logic;
    Signal DoAsynchPort_WriteCommand  : Std_Logic;
    Signal DoAsynchPort_ReadData      : Std_Logic;
    Signal DoAsynchPort_Write         : Std_Logic;

    Signal AsynchPortGo           : Std_Logic;
    Signal AsynchPort_UsbGo       : Std_Logic;
    Signal AsynchPort_SramGo      : Std_Logic;
    Signal AsynchPort_FlashGo     : Std_Logic;

    Signal CacheSelect_DoWrite    : Std_Logic;

    Signal MemoryIsInitialized    : Std_Logic;

    Signal Command_SelectLines    : Std_Logic_Vector(3 Downto 0);
    Signal Command_DataIndex      : Std_Logic_Vector(cCACHE_ADR_WIDTH - 1 Downto 0);

    -- Sram
    Signal SramWishboneRequest    : Std_Logic;
    Signal SramWishboneWrite      : Std_Logic;
    Signal SramWishboneRead       : Std_Logic;
    -- Flash
    Signal FlashWishboneRequest   : Std_Logic;
    Signal FlashWishboneWrite     : Std_Logic;
    Signal FlashWishboneRead      : Std_Logic;
    -- Usb
    Signal UsbWishboneRequest     : Std_Logic;
    Signal UsbWishboneWrite       : Std_Logic;
    Signal UsbWishboneRead        : Std_Logic;

    Signal AsynchPortCanGo        : Std_Logic;

    Signal DoAsynchPort_RequestAccess : Std_Logic;
    Signal AsynchPortGranted : Std_Logic;
    Signal DoCloseAsynchPort : Std_Logic;
    Signal SramAccessRequest  : Std_Logic;
    Signal UsbAccessRequest   : Std_Logic;
    Signal FlashAccessRequest : Std_Logic;
    Signal DoSdramCacheGo : Std_Logic;

-- -----------------------------------------------------------------------------
Begin
-- -----------------------------------------------------------------------------

-- -----------------------------------------------------------------------------
-- Sram
-- -----------------------------------------------------------------------------
    SramWishboneRequest <= WB_SRAM_STB_I And WB_SRAM_CYC_I;          --IncludeIf_Sram
    SramWishboneWrite   <= SramWishboneRequest And     WB_SRAM_WE_I; --IncludeIf_Sram
    SramWishboneRead    <= SramWishboneRequest And Not WB_SRAM_WE_I; --IncludeIf_Sram

    SramWishboneRequest <= '0'; --ExcludeIf_Sram
    SramWishboneWrite   <= '0'; --ExcludeIf_Sram
    SramWishboneRead    <= '0'; --ExcludeIf_Sram
-- -----------------------------------------------------------------------------
-- Flash
-- -----------------------------------------------------------------------------
    FlashWishboneRequest <= WB_FLASH_STB_I And WB_FLASH_CYC_I;           --IncludeIf_Flash
    FlashWishboneWrite   <= FlashWishboneRequest And     WB_FLASH_WE_I;  --IncludeIf_Flash
    FlashWishboneRead    <= FlashWishboneRequest And Not WB_FLASH_WE_I;  --IncludeIf_Flash

    FlashWishboneRequest <= '0';  --ExcludeIf_Flash
    FlashWishboneWrite   <= '0';  --ExcludeIf_Flash
    FlashWishboneRead    <= '0';  --ExcludeIf_Flash
-- -----------------------------------------------------------------------------
-- Usb
-- -----------------------------------------------------------------------------
    UsbWishboneRequest <= WB_USB_STB_I And WB_USB_CYC_I;            --IncludeIf_Usb
    UsbWishboneWrite   <= UsbWishboneRequest And     WB_USB_WE_I;   --IncludeIf_Usb
    UsbWishboneRead    <= UsbWishboneRequest And Not WB_USB_WE_I;   --IncludeIf_Usb

    UsbWishboneRequest <= '0';   --ExcludeIf_Usb
    UsbWishboneWrite   <= '0';   --ExcludeIf_Usb
    UsbWishboneRead    <= '0';   --ExcludeIf_Usb
-- -----------------------------------------------------------------------------

-- -----------------------------------------------------------------------------
-- -----------------------------------------------------------------------------
    AsynchPortGo <= AsynchPort_UsbGo Or AsynchPort_SramGo Or AsynchPort_FlashGo;

-- -----------------------------------------------------------------------------
-- Priority Resolver
-- -----------------------------------------------------------------------------


    -- -------------------------------------------------------------------------
    pWishbonePriorityResolver:
    Process(UsbWishboneRequest,
            SramWishboneRequest,
            UsbWishboneWrite,      --Include_If_Usb
            SramWishboneWrite,
            FlashWishboneWrite,
            DoAsynchPort_Acknowledge,
            FlashWishboneRequest)
    Begin
        AsynchPort_UsbGo     <= '0';
        AsynchPort_SramGo    <= '0';
        AsynchPort_FlashGo   <= '0';
        DoAsynchPort_Write   <= '0';

        If SramWishboneRequest  = '1' Then
            AsynchPort_SramGo       <= '1';
            DoAsynchPort_Write      <= SramWishboneWrite;
        ElsIf UsbWishboneRequest   = '1' Then                --Include_If_Usb
            AsynchPort_UsbGo       <= '1';                   --Include_If_Usb
            DoAsynchPort_Write     <= UsbWishboneWrite;      --Include_If_Usb
        ElsIf FlashWishboneRequest = '1' Then
            AsynchPort_FlashGo       <= '1';
            DoAsynchPort_Write       <= FlashWishboneWrite;
        End If;
    End Process;
    -- -------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    Process(WB_SDRAM_CLK_I)
    Begin
        If Rising_Edge(WB_SDRAM_CLK_I) Then
            WB_FLASH_ACK_O <= '0';
            WB_SRAM_ACK_O  <= '0';
            WB_USB_ACK_O   <= '0';                            --Include_If_Usb

            ASYNCH_PORT_MEM_ADR_O    <= (Others=>'0');
            ASYNCH_PORT_MEM_SEL_O    <= (Others=>'0');
            ASYNCH_PORT_MEM_WRITE    <= '0';
            SramAccessRequest        <= '0';
            UsbAccessRequest         <= '0';
            FlashAccessRequest       <= '0';

             If SramWishboneRequest= '1' Then
                  ASYNCH_PORT_MEM_DATA_O <= WB_SRAM_DAT_I;
                  ASYNCH_PORT_MEM_ADR_O(cWishboneSramAddressWidth - 3 Downto 0) <= WB_SRAM_ADR_I(cWishboneSramAddressWidth - 1 Downto 2);
                  WB_SRAM_ACK_O          <= ASYNCH_PORT_MEM_ACK_I;
                  ASYNCH_PORT_MEM_SEL_O  <= WB_SRAM_SEL_I;
                  ASYNCH_PORT_MEM_WRITE  <= SramWishboneWrite;
                  SramAccessRequest      <= '1';
            ElsIf UsbWishboneRequest='1' Then                                                                                             --Include_If_Usb
                  ASYNCH_PORT_MEM_DATA_O <= WB_USB_DAT_I;                                                                                 --Include_If_Usb
                  ASYNCH_PORT_MEM_ADR_O(cWishboneUsbAddressWidth - 3 Downto 0) <= WB_USB_ADR_I(cWishboneUsbAddressWidth - 1 Downto 2);    --Include_If_Usb
                  WB_USB_ACK_O          <= ASYNCH_PORT_MEM_ACK_I;                                                                         --Include_If_Usb
                  ASYNCH_PORT_MEM_SEL_O <= WB_USB_SEL_I;                                                                                  --Include_If_Usb
                  ASYNCH_PORT_MEM_WRITE  <= UsbWishboneWrite;                                                                             --Include_If_Usb
                  UsbAccessRequest       <= '1';                                                                                          --Include_If_Usb
            ElsIf FlashWishboneRequest='1' Then
                  ASYNCH_PORT_MEM_DATA_O <= WB_FLASH_DAT_I;
                  ASYNCH_PORT_MEM_ADR_O(cWishboneFlashAddressWidth - 3 Downto 0) <= WB_FLASH_ADR_I(cWishboneFlashAddressWidth - 1 Downto 2);
                  WB_FLASH_ACK_O <= FlashWishboneRequest And ASYNCH_PORT_MEM_ACK_I;
                  ASYNCH_PORT_MEM_SEL_O   <= WB_FLASH_SEL_I;
                  ASYNCH_PORT_MEM_WRITE  <= FlashWishboneWrite;
                  FlashAccessRequest     <= '1';
            End If;

            If SramWishboneRequest = '1' And ASYNCH_PORT_MEM_ACK_I='1' Then
                WB_SRAM_DAT_O          <= ASYNCH_PORT_MEM_DATA_I;
            End If;

            If UsbWishboneRequest = '1' And ASYNCH_PORT_MEM_ACK_I='1' Then --Include_If_Usb
                WB_USB_DAT_O          <= ASYNCH_PORT_MEM_DATA_I;                                                                        --Include_If_Usb
            End If;                                                         --Include_If_Usb

            If FlashWishboneRequest = '1' And ASYNCH_PORT_MEM_ACK_I='1' Then
                  WB_FLASH_DAT_O          <= ASYNCH_PORT_MEM_DATA_I;
            End If;


        End If;
    End Process;
    ASYNCH_PORT_MEM_SRAM_GO <= AsynchPortGranted And Not ASYNCH_PORT_MEM_ACK_I And Not DoCloseAsynchPort And SramAccessRequest;
    ASYNCH_PORT_MEM_USB_GO  <= AsynchPortGranted And Not ASYNCH_PORT_MEM_ACK_I And Not DoCloseAsynchPort And UsbAccessRequest; -- Include_If_Usb
    ASYNCH_PORT_MEM_FLASH_GO <= AsynchPortGranted And Not ASYNCH_PORT_MEM_ACK_I And Not DoCloseAsynchPort And FlashAccessRequest;


-- -----------------------------------------------------------------------------
-- SDRAM Section
-- -----------------------------------------------------------------------------
    -- -------------------------------------------------------------------------
    CACHE_CMD_CLK   <= WB_SDRAM_CLK_I;
    CACHE_CMD_ADR_O <= Cache_Address;
    CACHE_CMD_DAT_O <= Cache_Data_Out;
    CACHE_CMD_WE_O  <= Cache_Write;
    Cache_Status    <= CACHE_CMD_DAT_I;
    -- -------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    Cache_Data_Out         <= ReplaceDestinationAddress When DoCache_PrepareReplaceLine = '1' Else
                              Cache_CommandAsynchPort   When DoAsynchPort_WriteCommand  = '1' Else
                              Cache_Command;

    Cache_Write            <= CacheSelect_DoWrite;
    Cache_Address          <= cCACHE_CMD_ADR_LINE    When DoCache_PrepareReplaceLine='1' Else
                              cCACHE_CMD_ADR_CMD;
    CacheSelect_DoWrite    <= DoCache_PrepareReplaceLine Or
                              CommandRequest             Or
                              DoAsynchPort_WriteCommand;

    Command_SelectLines <= WB_SDRAM_SEL_I;
    Command_DataIndex   <= DataIndex;
    -- -------------------------------------------------------------------------

    DoSdramCacheGo <= DoCache_ReplaceLine And DoCache_FlushLine;
    -- -------------------------------------------------------------------------
    Cache_Command(cCACHE_FLAG_CACHE_GO)                             <= WishboneRequest And DoSdramCacheGo;
    Cache_Command(cCACHE_FLAG_REPLACE)                              <= DoCache_ReplaceLine;
    Cache_Command(cCACHE_FLAG_FLUSH)                                <= DoCache_FlushLine;
    Cache_Command(cCACHE_FLAG_FLUSH_FOR_READ)                       <= WishboneRead;
    Cache_Command(cCACHE_FLAG_FAST_REPLACE)                         <= Not PageActual(19) And PageActual(18) And WishboneRead; -- 21
    Cache_Command(cCACHE_FLAG_FLUSH_FOR_WRITE)                      <= WishboneWrite;
    Cache_Command(cCACHE_FLAG_MEMORY_INITIALIZED)                   <= '0';
    Cache_Command(cCACHE_FLAG_BE_MSB     Downto cCACHE_FLAG_BE_LSB) <= Command_SelectLines;
    Cache_Command(cCACHE_FLAG_BE_LSB - 1 Downto cCACHE_ADR_WIDTH)   <= (Others=>'0');
    Cache_Command(cCACHE_ADR_WIDTH   - 1 Downto  0)                 <= Command_DataIndex;
    Cache_Command(cCACHE_FLAG_REQUEST_ASYNCH_PORT)                  <= '0';
    -- -------------------------------------------------------------------------
    Cache_CommandAsynchPort(cCACHE_FLAG_CACHE_GO)                             <= '0';
    Cache_CommandAsynchPort(cCACHE_FLAG_REPLACE)                              <= '0';
    Cache_CommandAsynchPort(cCACHE_FLAG_FLUSH)                                <= '0';
    Cache_CommandAsynchPort(cCACHE_FLAG_FLUSH_FOR_READ)                       <= '0';
    Cache_CommandAsynchPort(cCACHE_FLAG_FAST_REPLACE)                         <= '0'; -- 21
    Cache_CommandAsynchPort(cCACHE_FLAG_FLUSH_FOR_WRITE)                      <= '0';
    Cache_CommandAsynchPort(cCACHE_FLAG_MEMORY_INITIALIZED)                   <= '0';
    Cache_CommandAsynchPort(cCACHE_FLAG_BE_MSB     Downto cCACHE_FLAG_BE_LSB) <= (Others=>'0');
    Cache_CommandAsynchPort(cCACHE_FLAG_BE_LSB - 1 Downto cCACHE_ADR_WIDTH)   <= (Others=>'0');
    Cache_CommandAsynchPort(cCACHE_FLAG_REQUEST_ASYNCH_PORT)                  <= DoAsynchPort_RequestAccess;
    Cache_CommandAsynchPort(cCACHE_FLAG_ASYNCH_PORT_GRANTED)                  <= '0';
    Cache_CommandAsynchPort(cCACHE_FLAG_REQUEST_USB)                          <= UsbWishboneRequest;
    Cache_CommandAsynchPort(cCACHE_FLAG_REQUEST_FLASH)                        <= FlashWishboneRequest;
    Cache_CommandAsynchPort(cCACHE_FLAG_REQUEST_SRAM)                         <= SramWishboneRequest;
    -- -------------------------------------------------------------------------
    AsynchPortGranted <= Cache_Status(cCACHE_FLAG_ASYNCH_PORT_GRANTED) And Cache_Status(cCACHE_FLAG_REQUEST_ASYNCH_PORT) And Not Cache_Status(cCACHE_FLAG_CACHE_GO);

    -- -------------------------------------------------------------------------
    CommandRequest    <= DoCache_ReplaceLine Or DoCache_FlushLine;
    -- -------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    CacheDone_All     <= Cache_Status(cCACHE_FLAG_DONE_ALL);
    CacheDone_Replace <= Cache_Status(cCACHE_FLAG_DONE_REPLACE_CURRENT);
    CacheDone_Flush   <= Cache_Status(cCACHE_FLAG_DONE_FLUSH);
    -- -------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    WishboneRequest <= WB_SDRAM_STB_I  And WB_SDRAM_CYC_I;
    WishboneWrite   <= WishboneRequest And     WB_SDRAM_WE_I;
    WishboneRead    <= WishboneRequest And Not WB_SDRAM_WE_I;
    -- -------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    DataIndex     <= WB_SDRAM_ADR_I(cCACHE_ADR_WIDTH + 1 Downto 2);
    TagIndex      <= WB_SDRAM_ADR_I(cCACHE_ADR_WIDTH + 1 Downto 5);
    -- -------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    WB_SDRAM_DAT_O <= DP_RAM_DAT_DATA_I;
    WB_SDRAM_ACK_O <= DoWishboneWriteAcknowledge Or DoWishboneReadAcknowledge;
    -- -------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    -- Cache data side.
    -- -------------------------------------------------------------------------
    DP_RAM_DAT_CLK_O  <= WB_SDRAM_CLK_I;
    DP_RAM_DAT_RST_O  <= WB_SDRAM_RST_I;
    DP_RAM_DAT_DATA_O <= WB_SDRAM_DAT_I;
    DP_RAM_DAT_ADR_O  <= DataIndex;
    DP_RAM_DAT_BE_O   <= WB_SDRAM_SEL_I;
    DP_RAM_DAT_WE_O   <= DoCacheWrite;
    -- -------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    -- Cache Tag side.
    -- -------------------------------------------------------------------------
    DP_RAM_TAG_CLK_O  <= WB_SDRAM_CLK_I;
    DP_RAM_TAG_RST_O  <= WB_SDRAM_RST_I;
    DP_RAM_TAG_ADR_O  <= TagIndex;
    DP_RAM_TAG_DATA_O <= PageRequested;
    DP_RAM_TAG_BE_O   <= (Others=>'1');
    DP_RAM_TAG_WE_O   <= DoCacheWrite;
    PageActual        <= DP_RAM_TAG_DATA_I;
    -- -------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    pPageRequested:
    Process(WB_SDRAM_CLK_I)
    Begin
        If Rising_Edge(WB_SDRAM_CLK_I) Then
            If WB_SDRAM_RST_I='1' Then
                PageRequested_Reg <= (Others=>'0');
            Else
                PageRequested_Reg <= '1' & '1' & cWbPageRequestPad & WB_SDRAM_ADR_I(cWishboneAddressWidth - 1 Downto cCACHE_ADR_WIDTH + 2);
            End If;
        End If;
    End Process;
    PageRequested             <= '1' & '1' & cWbPageRequestPad & WB_SDRAM_ADR_I(cWishboneAddressWidth - 1 Downto cCACHE_ADR_WIDTH + 2);
    ReplaceDestinationAddress <= cWbReplaceDestinationPad & WB_SDRAM_ADR_I(cWishboneAddressWidth - 1 Downto 2);

    -- -------------------------------------------------------------------------
    MemoryIsInitialized <= Cache_Status(cCACHE_FLAG_MEMORY_INITIALIZED);
    CacheCanGo <= '1' When CacheDone_All='1' And MemoryIsInitialized = '1' Else
                  '0';
    -- -------------------------------------------------------------------------
    CacheHit <= '1' When (PageActual(18 Downto 0) = PageRequested_Reg(18 Downto 0)) Else '0';
    -- -------------------------------------------------------------------------

    AsynchPortCanGo <= '1' When CacheDone_All='1' And MemoryIsInitialized='1' And DoCloseAsynchPort = '0' Else '0'; --todo

    -- -------------------------------------------------------------------------
    pCacheFSM_Combinatorial:
    Process(CacheState,
            WishboneRead,
            AsynchPortCanGo,
            WishboneWrite,
            CacheDone_All,
            CacheDone_Replace,
            CacheDone_Flush,
            CacheCanGo,
            AsynchPortGo,
            DoAsynchPort_Write,
            WishboneRequest,
            ASYNCH_PORT_MEM_ACK_I,
            CacheHit)
    Begin
        CacheStateNext             <= CacheState;
        DoWishboneReadAcknowledge  <= '0';
        DoWishboneWriteAcknowledge <= '0';

        DoCache_PrepareReplaceLine <= '0';
        DoCache_ReplaceLine        <= '0';
        DoCache_FlushLine          <= '0';

        DoCacheWrite               <= '0';
        DoAsynchPort_Acknowledge   <= '0';
        DoAsynchPort_WriteAddress  <= '0';
        DoAsynchPort_WriteData     <= '0';
        DoAsynchPort_WriteCommand  <= '0';
        DoAsynchPort_ReadData      <= '0';

        DoAsynchPort_WriteCommand  <= '0';
        DoAsynchPort_RequestAccess <= '0';

        DoCloseAsynchPort <= '0';
        -- ---------------------------------------------------------------------
        Case CacheState Is
        -- ---------------------------------------------------------------------
            -- -----------------------------------------------------------------
            When sCacheIdle =>
                 If WishboneRequest='1' Then                                       -- Do Sdram
                     If CacheCanGo='1' Then
                          If WishboneRead='1' Then
                              CacheStateNext             <= sCacheHitCheckRead;
                              DoCache_PrepareReplaceLine <= '1';
                          ElsIf WishboneWrite='1' Then
                              DoCache_PrepareReplaceLine <= '1';
                              CacheStateNext             <= sCacheHitCheckWrite;
                          End If;
                     End If;
                 ElsIf AsynchPortGo='1' Then                                       -- Do AsynchPort
                    If AsynchPortCanGo='1' Then
                       DoAsynchPort_WriteCommand  <= '1';
                       DoAsynchPort_RequestAccess <= '1';
                       CacheStateNext             <= sCacheAsynchPortInProgress;
                    End If;
                 End If;
            -- -----------------------------------------------------------------
            When sCacheAsynchPortInProgress =>
                 If ASYNCH_PORT_MEM_ACK_I='1' Then
                    DoAsynchPort_WriteCommand <= '1';
                    DoCloseAsynchPort <= '1';
                    CacheStateNext <= cCacheAsynchPortClose;
                 End If;
            -- -----------------------------------------------------------------
            When cCacheAsynchPortClose =>
                 DoCloseAsynchPort <= '1';
                 CacheStateNext <= sCacheIdle;
            -- -----------------------------------------------------------------
            When sCacheHitCheckRead =>
                 If CacheHit='1' Then
                     CacheStateNext            <= sCacheIdle;
                     DoWishboneReadAcknowledge <= '1';
                 Else
                    If CacheDone_All='1' Then
                        CacheStateNext      <= sCacheWaitForLineReplace;
                        DoCache_ReplaceLine <= '1';
                        DoCache_FlushLine   <= '1';
                    End If;
                 End If;
            -- -----------------------------------------------------------------
            When sCacheHitCheckWrite =>
                 If CacheHit='1' Then
                     CacheStateNext             <= sCacheIdle;
                     DoWishboneWriteAcknowledge <= '1';
                     DoCacheWrite               <= '1';
                 Else
                    If CacheDone_All='1' Then
                        CacheStateNext      <= sCacheWaitForLineFlushed;
                        DoCache_ReplaceLine <= '1';
                        DoCache_FlushLine   <= '1';
                     End If;
                 End If;
            -- -----------------------------------------------------------------
            When sCacheWaitForLineReplace =>
                 If CacheDone_Replace = '1' Then
                    CacheStateNext            <= sCacheIdle;
                    DoWishboneReadAcknowledge <= '1';
                 End If;
            -- -----------------------------------------------------------------
            When sCacheWaitForLineFlushed =>
                 If CacheDone_Flush = '1' Then
                    CacheStateNext             <= sCacheIdle;
                    DoWishboneWriteAcknowledge <= '1';
                    DoCacheWrite <= '1';
                 End If;
            -- -----------------------------------------------------------------
            When Others =>
                 CacheStateNext <= sCacheIdle;
        -- ---------------------------------------------------------------------
        End Case;
        -- ---------------------------------------------------------------------
    End Process;
    -- -------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    pCacheFSM_Synchronous:
    Process(WB_SDRAM_CLK_I)
    Begin
        If Rising_Edge(WB_SDRAM_CLK_I) Then
            If WB_SDRAM_RST_I='1' Then
                CacheState  <= sCacheIdle;
            Else
                CacheState  <= CacheStateNext;
            End If;
        End If;
    End Process;
    -- -------------------------------------------------------------------------
    Debug_WishboneState <= x"1" When CacheState = sCacheIdle                 Else
                           x"2" When CacheState = sCacheAsynchPortInProgress Else
                           x"3" When CacheState = sCacheHitCheckRead         Else
                           x"4" When CacheState = sCacheHitCheckWrite        Else
                           x"5" When CacheState = sCacheWaitForLineReplace   Else
                           x"6" When CacheState = sCacheWaitForLineFlushed   Else
                           x"0";

-- -----------------------------------------------------------------------------
End RTL;
-- -----------------------------------------------------------------------------

-- ---------------------------------------------------------------------------------------------------------------------------------------------
-- Top Level For Component                                                                                               Top Level For Component
-- ---------------------------------------------------------------------------------------------------------------------------------------------
Library IEEE;
Use     IEEE.Std_Logic_1164.All;
Use     Work.Configurable_ReplaceWith_Designator_GlobalConfig.All;
-- ---------------------------------------------------------------------------------------------------------------------------------------------

-- -----------------------------------------------------------------------------
Entity Configurable_ReplaceWith_Designator Is
Port (
     Debug_WishboneState : Out Std_Logic_Vector(3 Downto 0);
              Debug_SdramState : Out Std_Logic_Vector(3 Downto 0);
    MEM_CLK         : In    Std_Logic;
    -- Sram Wishbone Interface                                                     --IncludeIf_Sram
    SRAM_CLK_I     : In  Std_Logic;                                                --IncludeIf_Sram
    SRAM_RST_I     : In  Std_Logic;                                                --IncludeIf_Sram
    SRAM_STB_I     : In  Std_Logic;                                                --IncludeIf_Sram
    SRAM_CYC_I     : In  Std_Logic;                                                --IncludeIf_Sram
    SRAM_ACK_O     : Out Std_Logic;                                                --IncludeIf_Sram
    SRAM_ADR_I     : In  Std_Logic_Vector(cWishboneSramAddressWidth - 1 Downto 0); --IncludeIf_Sram
    SRAM_DAT_I     : In  Std_Logic_Vector(31 Downto 0);                            --IncludeIf_Sram
    SRAM_DAT_O     : Out Std_Logic_Vector(31 Downto 0);                            --IncludeIf_Sram
    SRAM_SEL_I     : In  Std_Logic_Vector( 3 Downto 0);                            --IncludeIf_Sram
    SRAM_WE_I      : In  Std_Logic;                                                --IncludeIf_Sram
    -- Flash Wishbone Interface                                                    --IncludeIf_Flash
    FLASH_CLK_I    : In  Std_Logic;                                                --IncludeIf_Flash
    FLASH_RST_I    : In  Std_Logic;                                                --IncludeIf_Flash
    FLASH_STB_I    : In  Std_Logic;                                                --IncludeIf_Flash
    FLASH_CYC_I    : In  Std_Logic;                                                --IncludeIf_Flash
    FLASH_ACK_O    : Out Std_Logic;                                                --IncludeIf_Flash
    FLASH_ADR_I    : In  Std_Logic_Vector(cWishboneFlashAddressWidth - 1 Downto 0);--IncludeIf_Flash
    FLASH_DAT_I    : In  Std_Logic_Vector(31 Downto 0);                            --IncludeIf_Flash
    FLASH_DAT_O    : Out Std_Logic_Vector(31 Downto 0);                            --IncludeIf_Flash
    FLASH_SEL_I    : In  Std_Logic_Vector( 3 Downto 0);                            --IncludeIf_Flash
    FLASH_WE_I     : In  Std_Logic;                                                --IncludeIf_Flash
    -- Usb Wishbone Interface                                                      --IncludeIf_Usb
    USB_CLK_I      : In  Std_Logic;                                                --IncludeIf_Usb
    USB_RST_I      : In  Std_Logic;                                                --IncludeIf_Usb
    USB_STB_I      : In  Std_Logic;                                                --IncludeIf_Usb
    USB_CYC_I      : In  Std_Logic;                                                --IncludeIf_Usb
    USB_ACK_O      : Out Std_Logic;                                                --IncludeIf_Usb
    USB_ADR_I      : In  Std_Logic_Vector(cWishboneUsbAddressWidth - 1 Downto 0);  --IncludeIf_Usb
    USB_DAT_I      : In  Std_Logic_Vector(31 Downto 0);                            --IncludeIf_Usb
    USB_DAT_O      : Out Std_Logic_Vector(31 Downto 0);                            --IncludeIf_Usb
    USB_SEL_I      : In  Std_Logic_Vector( 3 Downto 0);                            --IncludeIf_Usb
    USB_WE_I       : In  Std_Logic;                                                --IncludeIf_Usb
    -- Sdram Wishbone Interface
    SDRAM_CLK_I    : In  Std_Logic;
    SDRAM_RST_I    : In  Std_Logic;
    SDRAM_STB_I    : In  Std_Logic;
    SDRAM_CYC_I    : In  Std_Logic;
    SDRAM_ACK_O    : Out Std_Logic;
    SDRAM_ADR_I    : In  Std_Logic_Vector(cWishboneAddressWidth - 1 Downto 0);
    SDRAM_DAT_I    : In  Std_Logic_Vector(31 Downto 0);
    SDRAM_DAT_O    : Out Std_Logic_Vector(31 Downto 0);
    SDRAM_SEL_I    : In  Std_Logic_Vector( 3 Downto 0);
    SDRAM_WE_I     : In  Std_Logic;


    MEM_D           : InOut Std_Logic_Vector(cSdramDataWidth - 1 Downto 0);
    MEM_A           : Out   Std_Logic_Vector(cMemAddressWidth - 1 Downto 0);
    MEM_OE          : Out    Std_Logic;
    MEM_W           : Out   Std_Logic;
    MEM_BE          : Out   Std_Logic_Vector(cSdramByteEnableWidth - 1 Downto 0);
    MEM_SDRAM_CKE   : Out Std_Logic;
    MEM_SDRAM_RAS   : Out Std_Logic;
    MEM_SDRAM_CAS   : Out Std_Logic;
    MEM_SDRAM_E     : Out Std_Logic;

    MEM_USB_RESET   : Out Std_Logic; -- IncludeIf_ComponentVersion_2
    MEM_USB_E       : Out Std_Logic; -- IncludeIf_ComponentVersion_2

    MEM_FLASH_BUSY  : In Std_Logic;
    MEM_FLASH_RESET : Out Std_Logic;
    MEM_FLASH_E     : Out Std_Logic;
    MEM_SRAM_E      : Out Std_Logic
);
End;
-- -----------------------------------------------------------------------------

-- -----------------------------------------------------------------------------
Architecture RTL Of Configurable_ReplaceWith_Designator Is
-- -----------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    Component Configurable_ReplaceWith_Designator_DataCache Is
    Port (
         CLKA     : In  Std_Logic;
         DINA     : In  Std_Logic_Vector(31 Downto 0);
         DOUTA    : Out Std_Logic_Vector(31 Downto 0);
         ADDRA    : In  Std_Logic_Vector(10 Downto 0);
         BYTE_WEA : In  Std_Logic_Vector( 3 Downto 0);
         WEA      : In  Std_Logic;

         CLKB     : In  Std_Logic;
         DINB     : In  Std_Logic_Vector(31 Downto 0);
         DOUTB    : Out Std_Logic_Vector(31 Downto 0);
         ADDRB    : In  Std_Logic_Vector(10 Downto 0);
         BYTE_WEB : In  Std_Logic_Vector( 3 Downto 0);
         WEB      : In  Std_Logic
    );
    End Component;
    -- -------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    Component Configurable_ReplaceWith_Designator_TagCache Is
    Port (
        CLKA  : In  Std_Logic;
        DINA  : In  Std_Logic_Vector(19 Downto 0);
        DOUTA : Out Std_Logic_Vector(19 Downto 0);
        ADDRA : In  Std_Logic_Vector( 7 Downto 0);
        WEA   : In  Std_Logic;

        CLKB  : In  Std_Logic;
        DINB  : In  Std_Logic_Vector(19 Downto 0);
        DOUTB : Out Std_Logic_Vector(19 Downto 0);
        ADDRB : In  Std_Logic_Vector( 7 Downto 0);
        WEB   : In  Std_Logic
    );
    End Component;
    -- -------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    Component Configurable_ReplaceWith_Designator_CommandPipe Is
    Port (
        CLKA  : In  Std_Logic;
        DINA  : In  Std_Logic_Vector(31 Downto 0);
        DOUTA : Out Std_Logic_Vector(31 Downto 0);
        ADDRA : In  Std_Logic_Vector( 1 Downto 0);
        WEA   : In  Std_Logic;

        CLKB  : In  Std_Logic;
        DINB  : In  Std_Logic_Vector(31 Downto 0);
        DOUTB : Out Std_Logic_Vector(31 Downto 0);
        ADDRB : In  Std_Logic_Vector( 1 Downto 0);
        WEB   : In  Std_Logic
    );
    End Component;
    -- -------------------------------------------------------------------------


    -- -------------------------------------------------------------------------
    Component Configurable_ReplaceWith_Designator_WishboneInterface Is
    Port (
    Debug_WishboneState : Out Std_Logic_Vector(3 Downto 0);
        -- Cache Command Interface
        CACHE_CMD_CLK     : Out Std_Logic;
        CACHE_CMD_WE_O    : Out Std_Logic;
        CACHE_CMD_DAT_I   : In  Std_Logic_Vector(31 Downto 0);
        CACHE_CMD_DAT_O   : Out Std_Logic_Vector(31 Downto 0);
        CACHE_CMD_ADR_O   : Out Std_Logic_Vector( 1 Downto 0);

        -- Dual Port RAM Wishbone Data Cache Interface
        DP_RAM_DAT_CLK_O  : Out Std_Logic;
        DP_RAM_DAT_RST_O  : Out Std_Logic;
        DP_RAM_DAT_DATA_I : In  Std_Logic_Vector(31 Downto 0);
        DP_RAM_DAT_DATA_O : Out Std_Logic_Vector(31 Downto 0);
        DP_RAM_DAT_ADR_O  : Out Std_Logic_Vector(cCACHE_ADR_WIDTH - 1 Downto 0);
        DP_RAM_DAT_BE_O   : Out Std_Logic_Vector( 3 Downto 0);
        DP_RAM_DAT_WE_O   : Out Std_Logic;

        -- Dual Port RAM Cache Tag Interface
        DP_RAM_TAG_CLK_O  : Out Std_Logic;
        DP_RAM_TAG_RST_O  : Out Std_Logic;
        DP_RAM_TAG_DATA_I : In  Std_Logic_Vector(19 Downto 0);
        DP_RAM_TAG_DATA_O : Out Std_Logic_Vector(19 Downto 0);
        DP_RAM_TAG_ADR_O  : Out Std_Logic_Vector(cCACHE_ADR_WIDTH - 4 Downto 0);
        DP_RAM_TAG_BE_O   : Out Std_Logic_Vector( 3 Downto 0);
        DP_RAM_TAG_WE_O   : Out Std_Logic;

        ASYNCH_PORT_MEM_DATA_I   : In  Std_Logic_Vector(31 Downto 0);
        ASYNCH_PORT_MEM_DATA_O   : Out Std_Logic_Vector(31 Downto 0);
        ASYNCH_PORT_MEM_ADR_O    : Out Std_Logic_Vector(23 Downto 0);
        ASYNCH_PORT_MEM_SEL_O    : Out Std_Logic_Vector( 3 Downto 0);
        ASYNCH_PORT_MEM_SRAM_GO  : Out Std_Logic;
        ASYNCH_PORT_MEM_FLASH_GO : Out Std_Logic;
        ASYNCH_PORT_MEM_USB_GO   : Out Std_Logic;
        ASYNCH_PORT_MEM_ACK_I    : In  Std_Logic;
        ASYNCH_PORT_MEM_WRITE    : Out Std_Logic;

        -- Sram Wishbone Interface
        WB_SRAM_CLK_I : In  Std_Logic;
        WB_SRAM_RST_I : In  Std_Logic;
        WB_SRAM_STB_I : In  Std_Logic;
        WB_SRAM_CYC_I : In  Std_Logic;
        WB_SRAM_ACK_O : Out Std_Logic;
        WB_SRAM_ADR_I : In  Std_Logic_Vector(cWishboneSramAddressWidth - 1 Downto 0);
        WB_SRAM_DAT_I : In  Std_Logic_Vector(31 Downto 0);
        WB_SRAM_DAT_O : Out Std_Logic_Vector(31 Downto 0);
        WB_SRAM_SEL_I : In  Std_Logic_Vector( 3 Downto 0);
        WB_SRAM_WE_I  : In  Std_Logic;
        -- Flash Wishbone Interface
        WB_FLASH_CLK_I : In  Std_Logic;
        WB_FLASH_RST_I : In  Std_Logic;
        WB_FLASH_STB_I : In  Std_Logic;
        WB_FLASH_CYC_I : In  Std_Logic;
        WB_FLASH_ACK_O : Out Std_Logic;
        WB_FLASH_ADR_I : In  Std_Logic_Vector(cWishboneFlashAddressWidth - 1 Downto 0);
        WB_FLASH_DAT_I : In  Std_Logic_Vector(31 Downto 0);
        WB_FLASH_DAT_O : Out Std_Logic_Vector(31 Downto 0);
        WB_FLASH_SEL_I : In  Std_Logic_Vector( 3 Downto 0);
        WB_FLASH_WE_I  : In  Std_Logic;
        -- Usb Wishbone Interface                                                    --Include_If_ComponentVersion_2
        WB_USB_CLK_I : In  Std_Logic;                                                --Include_If_ComponentVersion_2
        WB_USB_RST_I : In  Std_Logic;                                                --Include_If_ComponentVersion_2
        WB_USB_STB_I : In  Std_Logic;                                                --Include_If_ComponentVersion_2
        WB_USB_CYC_I : In  Std_Logic;                                                --Include_If_ComponentVersion_2
        WB_USB_ACK_O : Out Std_Logic;                                                --Include_If_ComponentVersion_2
        WB_USB_ADR_I : In  Std_Logic_Vector(cWishboneUsbAddressWidth - 1 Downto 0);  --Include_If_ComponentVersion_2
        WB_USB_DAT_I : In  Std_Logic_Vector(31 Downto 0);                            --Include_If_ComponentVersion_2
        WB_USB_DAT_O : Out Std_Logic_Vector(31 Downto 0);                            --Include_If_ComponentVersion_2
        WB_USB_SEL_I : In  Std_Logic_Vector( 3 Downto 0);                            --Include_If_ComponentVersion_2
        WB_USB_WE_I  : In  Std_Logic;                                                --Include_If_ComponentVersion_2
        -- Sdram Wishbone Interface
        WB_SDRAM_CLK_I          : In  Std_Logic;
        WB_SDRAM_RST_I          : In  Std_Logic;
        WB_SDRAM_STB_I          : In  Std_Logic;
        WB_SDRAM_CYC_I          : In  Std_Logic;
        WB_SDRAM_ACK_O          : Out Std_Logic;
        WB_SDRAM_ADR_I          : In  Std_Logic_Vector(cWishboneAddressWidth - 1 Downto 0);
        WB_SDRAM_DAT_I          : In  Std_Logic_Vector(31 Downto 0);
        WB_SDRAM_DAT_O          : Out Std_Logic_Vector(31 Downto 0);
        WB_SDRAM_SEL_I          : In  Std_Logic_Vector( 3 Downto 0);
        WB_SDRAM_WE_I           : In  Std_Logic
    );
    End Component;
    -- -----------------------------------------------------------------------------
    -- -------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    Component ReplaceWith_SdramEngineName Is
    Port (
         Debug_SdramState : Out Std_Logic_Vector(3 Downto 0);
        -- System
        MEM_CLK                         : In  Std_Logic;
        MEM_RST                         : In  Std_Logic;
        -- Data Cache
        CACHE_DAT_CLK_O                 : Out Std_Logic;
        CACHE_DAT_BE_O                  : Out Std_Logic_Vector(3 Downto 0);
        CACHE_DAT_WE_O                  : Out Std_Logic;
        CACHE_DAT_ADR_O                 : Out Std_Logic_Vector(10 Downto 0);
        CACHE_DAT_DATA_I                : In  Std_Logic_Vector(31 Downto 0);
        CACHE_DAT_DATA_O                : Out Std_Logic_Vector(31 Downto 0);
        -- Tag Cache
        CACHE_TAG_CLK_O                 : Out Std_Logic;
        CACHE_TAG_WE_O                  : Out Std_Logic;
        CACHE_TAG_ADR_O                 : Out Std_Logic_Vector(7 Downto 0);
        CACHE_TAG_DATA_I                : In  Std_Logic_Vector(19 Downto 0);
        CACHE_TAG_DATA_O                : Out Std_Logic_Vector(19 Downto 0);
        -- Command Pipe
        CACHE_CMD_CLK_O                 : Out Std_Logic;
        CACHE_CMD_WE_O                  : Out Std_Logic;
        CACHE_CMD_ADR_O                 : Out Std_Logic_Vector( 1 Downto 0);
        CACHE_CMD_DATA_O                : Out Std_Logic_Vector(31 Downto 0);
        CACHE_CMD_DATA_I                : In  Std_Logic_Vector(31 Downto 0);
        -- Timer Configuration
        TIMER_VALUE_TRP                 : In  Std_Logic_Vector(2 Downto 0);
        TIMER_VALUE_TWR                 : In  Std_Logic_Vector(2 Downto 0);
        TIMER_VALUE_TRFC                : In  Std_Logic_Vector(2 Downto 0);
        TIMER_VALUE_TMRD                : In  Std_Logic_Vector(2 Downto 0);
        TIMER_VALUE_TRCD                : In  Std_Logic_Vector(2 Downto 0);
        TIMER_VALUE_TCL                 : In  Std_Logic_Vector(2 Downto 0);
        -- Memory Mode Register
        LOAD_MODE_REGISTER              : In  Std_Logic_Vector(12 Downto 0);
        -- Address Resover for diff memory geometries
        RESOLVED_BANK_ADDRESS_FLUSH     : In  Std_Logic_Vector( 1 Downto 0);
        RESOLVED_ROW_ADDRESS_FLUSH      : In  Std_Logic_Vector(12 Downto 0);
        RESOLVED_COLUMN_ADDRESS_FLUSH   : In  Std_Logic_Vector(12 Downto 0);
        RESOLVED_BANK_ADDRESS_REPLACE   : In  Std_Logic_Vector( 1 Downto 0);
        RESOLVED_ROW_ADDRESS_REPLACE    : In  Std_Logic_Vector(12 Downto 0);
        RESOLVED_COLUMN_ADDRESS_REPLACE : In  Std_Logic_Vector(12 Downto 0);

        RESOLVER_FLUSH_ADDRESS          : Out Std_Logic_Vector(19 Downto 0);
        RESOLVER_REPLACE_ADDRESS        : Out Std_Logic_Vector(31 Downto 0);
        RESOLVER_FLUSH_INDEX            : Out Std_Logic_Vector(10 Downto 0);
        -- Refresh engine reload value
        REFRESH_RELOAD_VALUE            : In  Std_Logic_Vector(10 Downto 0);

        ASYNCH_PORT_MEM_ADR_I           : In  STD_LOGIC_VECTOR(23 DOWNTO 0);
        ASYNCH_PORT_MEM_BE_I            : In  STD_LOGIC_VECTOR(3 DOWNTO 0);
        ASYNCH_PORT_MEM_DATA_I          : In  STD_LOGIC_VECTOR(31 DOWNTO 0);
        ASYNCH_PORT_MEM_DATA_O          : Out STD_LOGIC_VECTOR(31 DOWNTO 0);
        ASYNCH_PORT_MEM_DATA_WRITE_I    : In  STD_LOGIC;
        ASYNCH_PORT_MEM_FLASH_CS_I      : In  STD_LOGIC;
        ASYNCH_PORT_MEM_FLASH_RESET_I   : In  STD_LOGIC;
        ASYNCH_PORT_MEM_OE_I            : In  STD_LOGIC;
        ASYNCH_PORT_MEM_SRAM_CS_I       : In  STD_LOGIC;
        ASYNCH_PORT_MEM_USB_CS_I        : In  STD_LOGIC;
        ASYNCH_PORT_MEM_USB_RESET_I     : In  STD_LOGIC;
        ASYNCH_PORT_MEM_WE_I            : In  STD_LOGIC;

        FLASH_CS                        : Out STD_LOGIC;
        FLASH_RESET                     : Out STD_LOGIC;
        MEM_ADR_BOTTOM                  : Out STD_LOGIC;
        MEM_ADR_TOP                     : Out STD_LOGIC_VECTOR(7 DOWNTO 0);
        MEM_OE                          : Out STD_LOGIC;
        SRAM_CS                         : Out STD_LOGIC;
        USB_CS                          : Out STD_LOGIC;
        USB_RESET                       : Out STD_LOGIC;

        -- Memory Bus
        SDRAM_DATA_O                    : Out Std_Logic_Vector(31 Downto 0);
        SDRAM_DATA_I                    : In  Std_Logic_Vector(31 Downto 0);
        SDRAM_DATA_E                    : Out Std_Logic_Vector(31 Downto 0);
        SDRAM_A                         : Out Std_Logic_Vector(12 Downto 0);
        SDRAM_BA                        : Out Std_Logic_Vector( 1 Downto 0);
        SDRAM_W                         : Out Std_Logic;
        SDRAM_BE                        : Out Std_Logic_Vector( 3 Downto 0);
        SDRAM_CKE                       : Out Std_Logic;
        SDRAM_RAS                       : Out Std_Logic;
        SDRAM_CAS                       : Out Std_Logic;
        SDRAM_E                         : Out Std_Logic
    );
    End Component;
    -- -------------------------------------------------------------------------

    --------------------------------------------------------------------------------
    Component ReplaceWith_Designator_Sram_x32 Is
    Port
       (
         SRAM_D_O  : Out   Std_Logic_Vector(31 Downto 0);
         SRAM_D_I  : In    Std_Logic_Vector(31 Downto 0);
         SRAM_D_E  : Out   Std_Logic;
         SRAM_A    : Out   Std_Logic_Vector(23 DownTo 0);
         SRAM_OE   : Out   Std_Logic;
         SRAM_WE   : Out   Std_Logic;
         SRAM_CE   : Out   Std_Logic;
         SRAM_BE_E : Out   Std_Logic_Vector(3 Downto 0);
         ---------------------------------------------------------------------------

         ---------------------------------------------------------------------------
         -- Wishbone Slave Port
         ---------------------------------------------------------------------------
         CLK_I      : In    Std_Logic;
         RST_I      : In    Std_Logic;
         CYC_I      : In    Std_Logic;
         STB_I      : In    Std_Logic;
         ACK_O      : Out   Std_Logic;
         ADR_I      : In    Std_Logic_Vector(23 DownTo 0);
         DAT_I      : In    Std_Logic_Vector(31 DownTo 0);
         DAT_O      : Out   Std_Logic_Vector(31 DownTo 0);
         WE_I       : In    Std_Logic;
         SEL_I      : In    Std_Logic_Vector(3 DownTo 0)
       );

    End Component;
    --------------------------------------------------------------------------------

    --------------------------------------------------------------------------------
    Component ReplaceWith_Designator_USB_X32 Is Port
       (

         MEM_D_O       : Out   Std_Logic_Vector(31 Downto 0);                          -- IncludeIf_Shared
         MEM_D_I       : In    Std_Logic_Vector(31 Downto 0);                          -- IncludeIf_Shared
         MEM_D_E       : Out   Std_Logic;                                              -- IncludeIf_Shared

         MEM_A         : Out   Std_Logic_Vector(23 Downto 0);                          -- IncludeIf_FullAddress
         MEM_OE        : Out   Std_Logic;
         MEM_W         : Out   Std_Logic;
         MEM_BE        : Out   Std_Logic_Vector(3 Downto 0);
         MEM_USB_E     : Out   Std_Logic;
         MEM_USB_RESET : Out   Std_Logic;
         ---------------------------------------------------------------------------

         ---------------------------------------------------------------------------
         -- Wishbone Slave Port
         ---------------------------------------------------------------------------
         USB_CLK_I      : In    Std_Logic;
         USB_RST_I      : In    Std_Logic;
         USB_CYC_I      : In    Std_Logic;
         USB_STB_I      : In    Std_Logic;
         USB_ACK_O      : Out   Std_Logic;
         USB_ADR_I      : In    Std_Logic_Vector(23 DownTo 0);
         USB_DAT_I      : In    Std_Logic_Vector(31 DownTo 0);
         USB_DAT_O      : Out   Std_Logic_Vector(31 DownTo 0);
         USB_WE_I       : In    Std_Logic;
         USB_SEL_I      : In    Std_Logic_Vector(3 DownTo 0)
       );

    End Component;
    --------------------------------------------------------------------------------

    --------------------------------------------------------------------------------
    Component ReplaceWith_Designator_Flash_x16 Is Port
       (
         ----------------------------------------------------------------------------- IncludeIf_1_Device_X_16
         FLASH_D_O  : Out Std_Logic_Vector(31 DownTo 0);                             -- IncludeIf_1_Device_X_16
         FLASH_D_I  : In  Std_Logic_Vector(31 DownTo 0);                             -- IncludeIf_1_Device_X_16
         FLASH_D_E  : Out Std_Logic;
         FLASH_A    : Out   Std_Logic_Vector(23 DownTo 0);   -- IncludeIf_1_Device_X_16
         FLASH_OE   : Out   Std_Logic;                                               -- IncludeIf_1_Device_X_16
         FLASH_WE   : Out   Std_Logic;                                               -- IncludeIf_1_Device_X_16
         FLASH_CE   : Out   Std_Logic;                                               -- IncludeIf_1_Device_X_16
         ----------------------------------------------------------------------------- IncludeIf_1_Device_X_16

         ---------------------------------------------------------------------------
         -- Wishbone Slave Port
         ---------------------------------------------------------------------------
         CLK_I      : In    Std_Logic;
         RST_I      : In    Std_Logic;
         CYC_I      : In    Std_Logic;
         STB_I      : In    Std_Logic;
         ACK_O      : Out   Std_Logic;
         ADR_I      : In    Std_Logic_Vector(23 DownTo 0);
         DAT_I      : In    Std_Logic_Vector(31 DownTo 0);
         DAT_O      : Out   Std_Logic_Vector(31 DownTo 0);
         WE_I       : In    Std_Logic;
         SEL_I      : In    Std_Logic_Vector(3  DownTo 0)
       );

    End Component;
    --------------------------------------------------------------------------------

    --------------------------------------------------------------------------------
    Component Configurable_ReplaceWith_Designator_ASYNCH_PORT_SELECT Is
    Port (
        RST_I         : In Std_Logic;
        CLK_I         : In Std_Logic;

        --SRAM WISHBONE
        SRAM_STB_O    : Out Std_Logic;
        SRAM_CYC_O    : Out Std_Logic;
        SRAM_DAT_O    : Out Std_Logic_Vector(31 Downto 0);
        SRAM_DAT_I    : In  Std_Logic_Vector(31 Downto 0);
        SRAM_ADR_O    : Out Std_Logic_Vector(23 Downto 0);
        SRAM_SEL_O    : Out Std_Logic_Vector(3 Downto 0);
        SRAM_WE_O     : Out Std_Logic;
        SRAM_ACK_I    : In  Std_Logic;
        SRAM_RST_O    : Out Std_Logic;
        SRAM_CLK_O    : Out Std_Logic;
        --SRAM MEMORY
        SRAM_MEM_D_O  : Out Std_Logic_Vector(31 Downto 0);
        SRAM_MEM_D_I  : In  Std_Logic_Vector(31 Downto 0);
        SRAM_MEM_D_E  : In  Std_Logic;
        SRAM_MEM_A_I  : In  Std_Logic_Vector(23 Downto 0);
        SRAM_MEM_BE_W : In  Std_Logic_Vector(3 Downto 0);
        SRAM_MEM_OE   : In  Std_Logic;
        SRAM_MEM_WE   : In  Std_Logic;
        SRAM_MEM_CE   : In  Std_Logic;
        -- FLASH WISHBONE
        FLASH_STB_O   : Out Std_Logic;
        FLASH_CYC_O   : Out Std_Logic;
        FLASH_DAT_O   : Out Std_Logic_Vector(31 Downto 0);
        FLASH_DAT_I   : In  Std_Logic_Vector(31 Downto 0);
        FLASH_ADR_O   : Out Std_Logic_Vector(23 Downto 0);
        FLASH_SEL_O   : Out Std_Logic_Vector(3 Downto 0);
        FLASH_WE_O    : Out Std_Logic;
        FLASH_ACK_I   : In  Std_Logic;
        FLASH_RST_O   : Out Std_Logic;
        FLASH_CLK_O   : Out Std_Logic;
        -- FLASH MEMORY
        FLASH_MEM_D_O : Out Std_Logic_Vector(31 Downto 0);
        FLASH_MEM_D_I : In  Std_Logic_Vector(31 Downto 0);
        FLASH_MEM_D_E : In  Std_Logic;
        FLASH_MEM_A_I : In Std_Logic_Vector(23 Downto 0);
        FLASH_MEM_OE  : In Std_Logic;
        FLASH_MEM_WE  : In Std_Logic;
        FLASH_MEM_CE  : In  Std_Logic;
        --USB WISHBONE
        USB_STB_O    : Out Std_Logic;
        USB_CYC_O    : Out Std_Logic;
        USB_DAT_O    : Out Std_Logic_Vector(31 Downto 0);
        USB_DAT_I    : In  Std_Logic_Vector(31 Downto 0);
        USB_ADR_O    : Out Std_Logic_Vector(23 Downto 0);
        USB_SEL_O    : Out Std_Logic_Vector(3 Downto 0);
        USB_WE_O     : Out Std_Logic;
        USB_ACK_I    : In  Std_Logic;
        USB_RST_O    : Out Std_Logic;
        USB_CLK_O    : Out Std_Logic;
        --USB MEMORY
        USB_MEM_D_O  : Out Std_Logic_Vector(31 Downto 0);
        USB_MEM_D_I  : In  Std_Logic_Vector(31 Downto 0);
        USB_MEM_D_E  : In  Std_Logic;
        USB_MEM_A_I  : In  Std_Logic_Vector(23 Downto 0);
        USB_MEM_BE_W : In  Std_Logic_Vector(3 Downto 0);
        USB_MEM_OE   : In  Std_Logic;
        USB_MEM_WE   : In  Std_Logic;
        USB_MEM_CE   : In  Std_Logic;
        USB_MEM_RESET : In Std_Logic;

        -- engine side
        ASYNCH_PORT_MEM_DATA_I        : in    std_logic_vector(31 downto 0);
        ASYNCH_PORT_MEM_ADR_O         : out   std_logic_vector(23 downto 0);
        ASYNCH_PORT_MEM_BE_O          : out   std_logic_vector(3 downto 0);
        ASYNCH_PORT_MEM_DATA_O        : out   std_logic_vector(31 downto 0);
        ASYNCH_PORT_MEM_DATA_WRITE_O  : out   std_logic;
        ASYNCH_PORT_MEM_FLASH_CS_O    : out   std_logic;
        ASYNCH_PORT_MEM_FLASH_RESET_O : out   std_logic;
        ASYNCH_PORT_MEM_OE_O          : out   std_logic;
        ASYNCH_PORT_MEM_SRAM_CS_O     : out   std_logic;
        ASYNCH_PORT_MEM_USB_CS_O      : out   std_logic;
        ASYNCH_PORT_MEM_USB_RESET_O   : out   std_logic;
        ASYNCH_PORT_MEM_WE_O          : out   std_logic;
        -- wishbone side
        TO_MEM_CTRL_ACK_O             : out   std_logic;
        TO_MEM_CTRL_ADR_I             : in    std_logic_vector(23 downto 0);
        TO_MEM_CTRL_DATA_I            : in    std_logic_vector(31 downto 0);
        TO_MEM_CTRL_DATA_O            : out   std_logic_vector(31 downto 0);
        TO_MEM_CTRL_FLASH_GO          : in    std_logic;
        TO_MEM_CTRL_SEL_O             : in    std_logic_vector(3 downto 0);
        TO_MEM_CTRL_SRAM_GO           : in    std_logic;
        TO_MEM_CTRL_USB_GO            : in    std_logic;
        TO_MEM_CTRL_WE_I              : In    Std_Logic
    );
    End Component;
    --------------------------------------------------------------------------------


    -- Timing Configuration
    Signal TimerValue_tRP   : Std_Logic_Vector(cTimerWidth - 1 Downto 0);
    Signal TimerValue_tWR   : Std_Logic_Vector(cTimerWidth - 1 Downto 0);
    Signal TimerValue_tRFC  : Std_Logic_Vector(cTimerWidth - 1 Downto 0);
    Signal TimerValue_tMRD  : Std_Logic_Vector(cTimerWidth - 1 Downto 0);
    Signal TimerValue_tRCD  : Std_Logic_Vector(cTimerWidth - 1 Downto 0);
    Signal TimerValue_tCL   : Std_Logic_Vector(cTimerWidth - 1 Downto 0);
    Signal LoadModeRegister : Std_Logic_Vector(12 Downto 0);

    -- Address Resolver
    Signal FinalBankAddress_Flush     : Std_Logic_Vector( 1 Downto 0);
    Signal FinalRowAddress_Flush      : Std_Logic_Vector(12 Downto 0);
    Signal FinalColumnAddress_Flush   : Std_Logic_Vector(12 Downto 0);
    Signal FinalBankAddress_Replace   : Std_Logic_Vector( 1 Downto 0);
    Signal FinalRowAddress_Replace    : Std_Logic_Vector(12 Downto 0);
    Signal FinalColumnAddress_Replace : Std_Logic_Vector(12 Downto 0);

    Signal ToResolve_FlushAddress     : Std_Logic_Vector(19 Downto 0);
    Signal ToResolve_ReplaceAddress   : Std_Logic_Vector(31 Downto 0);
    Signal ToResolve_FlushIndex       : Std_Logic_Vector(10 Downto 0);

    Signal DataCache_WbClk    : Std_Logic;
    Signal DataCache_WbDin    : Std_Logic_Vector(31 Downto 0);
    Signal DataCache_WbDout   : Std_Logic_Vector(31 Downto 0);
    Signal DataCache_WbAddr   : Std_Logic_Vector(10 Downto 0);
    Signal DataCache_WbByteW  : Std_Logic_Vector( 3 Downto 0);
    Signal DataCache_WbWe     : Std_Logic;

    Signal DataCache_MemClk   : Std_Logic;
    Signal DataCache_MemDin   : Std_Logic_Vector(31 Downto 0);
    Signal DataCache_MemDout  : Std_Logic_Vector(31 Downto 0);
    Signal DataCache_MemAddr  : Std_Logic_Vector(10 Downto 0);
    Signal DataCache_MemByteW : Std_Logic_Vector( 3 Downto 0);
    Signal DataCache_MemWe    : Std_Logic;

    Signal TagCache_WbClk     : Std_Logic;
    Signal TagCache_WbDin     : Std_Logic_Vector(19 Downto 0);
    Signal TagCache_WbDout    : Std_Logic_Vector(19 Downto 0);
    Signal TagCache_WbAddr    : Std_Logic_Vector( 7 Downto 0);
    Signal TagCache_WbWe      : Std_Logic;

    Signal TagCache_MemClk    : Std_Logic;
    Signal TagCache_MemDin    : Std_Logic_Vector(19 Downto 0);
    Signal TagCache_MemDout   : Std_Logic_Vector(19 Downto 0);
    Signal TagCache_MemAddr   : Std_Logic_Vector( 7 Downto 0);
    Signal TagCache_MemWe     : Std_Logic;

    Signal CmdCache_WbClk     : Std_Logic;
    Signal CmdCache_WbDin     : Std_Logic_Vector(31 Downto 0);
    Signal CmdCache_WbDout    : Std_Logic_Vector(31 Downto 0);
    Signal CmdCache_WbAddr    : Std_Logic_Vector( 1 Downto 0);
    Signal CmdCache_WbWe      : Std_Logic;

    Signal CmdCache_MemClk    : Std_Logic;
    Signal CmdCache_MemDin    : Std_Logic_Vector(31 Downto 0);
    Signal CmdCache_MemDout   : Std_Logic_Vector(31 Downto 0);
    Signal CmdCache_MemAddr   : Std_Logic_Vector( 1 Downto 0);
    Signal CmdCache_MemWe     : Std_Logic;

    Signal SdramDataout     : Std_Logic_Vector(31 Downto 0);
    Signal SdramDataIn      : Std_Logic_Vector(31 Downto 0);
    Signal SdramDataEnable  : Std_Logic_Vector(31 Downto 0);
    Signal SdramAddress     : Std_Logic_Vector(12 Downto 0);
    Signal SdramBankAddress : Std_Logic_Vector( 1 Downto 0);
    Signal SdramWrite       : Std_Logic;
    Signal SdramByteEnable  : Std_Logic_Vector(3 Downto 0);
    Signal SdramClockEnable : Std_Logic;
    Signal SdramNRas        : Std_Logic;
    Signal SdramNCas        : Std_Logic;
    Signal SdramChipSelect  : Std_Logic;

    Signal TimerValue_Refresh : Std_Logic_Vector(10 Downto 0);

    Signal Signal_GND             : Std_Logic;
    Signal Signal_GND_4           : Std_Logic_Vector( 3 Downto 0);
    Signal Signal_GND_32          : Std_Logic_Vector(31 Downto 0);
    Signal Signal_GND_SRAM_ADR_I  : Std_Logic_Vector(cWishboneSramAddressWidth - 1 Downto 0);
    Signal Signal_GND_FLASH_ADR_I : Std_Logic_Vector(cWishboneFlashAddressWidth - 1 Downto 0);
    Signal Signal_GND_USB_ADR_I   : Std_Logic_Vector(cWishboneUsbAddressWidth - 1 Downto 0);

    Signal AsynchPort_Sram_D_O     : Std_Logic_Vector(31 Downto 0);
    Signal AsynchPort_Sram_D_I     : Std_Logic_Vector(31 Downto 0);
    Signal AsynchPort_Sram_D_E     : Std_Logic;
    Signal AsynchPort_Sram_A       : Std_Logic_Vector(23 Downto 0);
    Signal AsynchPort_Sram_OE      : Std_Logic;
    Signal AsynchPort_Sram_CE      : Std_Logic;
    Signal AsynchPort_Sram_WE      : Std_Logic;
    Signal AsynchPort_Sram_BE_E    : Std_Logic_Vector(3 Downto 0);
    Signal AsynchPort_SramWb_CLK_I : Std_Logic;
    Signal AsynchPort_SramWb_RST_I : Std_Logic;
    Signal AsynchPort_SramWb_CYC_I : Std_Logic;
    Signal AsynchPort_SramWb_STB_I : Std_Logic;
    Signal AsynchPort_SramWb_ACK_O : Std_Logic;
    Signal AsynchPort_SramWb_ADR_I : Std_Logic_Vector(23 DownTo 0);
    Signal AsynchPort_SramWb_DAT_I : Std_Logic_Vector(31 DownTo 0);
    Signal AsynchPort_SramWb_DAT_O : Std_Logic_Vector(31 DownTo 0);
    Signal AsynchPort_SramWb_WE_I  : Std_Logic;
    Signal AsynchPort_SramWb_SEL_I : Std_Logic_Vector(3 DownTo 0);

    Signal AsynchPort_Usb_MEM_D_O       : Std_Logic_Vector(31 Downto 0);                          -- IncludeIf_Shared
    Signal AsynchPort_Usb_MEM_D_I       : Std_Logic_Vector(31 Downto 0);                          -- IncludeIf_Shared
    Signal AsynchPort_Usb_MEM_D_E       : Std_Logic;                                              -- IncludeIf_Shared
    Signal AsynchPort_Usb_MEM_A         : Std_Logic_Vector(23 Downto 0);                          -- IncludeIf_FullAddress
    Signal AsynchPort_Usb_MEM_OE        : Std_Logic;
    Signal AsynchPort_Usb_MEM_W         : Std_Logic;
    Signal AsynchPort_Usb_MEM_BE        : Std_Logic_Vector(3 Downto 0);
    Signal AsynchPort_Usb_MEM_USB_E     : Std_Logic;
    Signal AsynchPort_Usb_MEM_USB_RESET : Std_Logic;

    Signal AsynchPort_UsbWb_USB_CLK_I   : Std_Logic;
    Signal AsynchPort_UsbWb_USB_RST_I   : Std_Logic;
    Signal AsynchPort_UsbWb_USB_CYC_I   : Std_Logic;
    Signal AsynchPort_UsbWb_USB_STB_I   : Std_Logic;
    Signal AsynchPort_UsbWb_USB_ACK_O   : Std_Logic;
    Signal AsynchPort_UsbWb_USB_ADR_I   : Std_Logic_Vector(23 DownTo 0);
    Signal AsynchPort_UsbWb_USB_DAT_I   : Std_Logic_Vector(31 DownTo 0);
    Signal AsynchPort_UsbWb_USB_DAT_O   : Std_Logic_Vector(31 DownTo 0);
    Signal AsynchPort_UsbWb_USB_WE_I    : Std_Logic;
    Signal AsynchPort_UsbWb_USB_SEL_I   : Std_Logic_Vector(3 DownTo 0);

    Signal AsynchPort_Flash_FLASH_D_O  : Std_Logic_Vector(31 DownTo 0);
    Signal AsynchPort_Flash_FLASH_D_I  : Std_Logic_Vector(31 DownTo 0);
    Signal AsynchPort_Flash_FLASH_D_E  : Std_Logic;
    Signal AsynchPort_Flash_FLASH_A    : Std_Logic_Vector(23 DownTo 0);
    Signal AsynchPort_Flash_FLASH_OE   : Std_Logic;
    Signal AsynchPort_Flash_FLASH_WE   : Std_Logic;
    Signal AsynchPort_Flash_FLASH_CE   : Std_Logic;
    Signal AsynchPort_FlashWb_CLK_I    : Std_Logic;
    Signal AsynchPort_FlashWb_RST_I    : Std_Logic;
    Signal AsynchPort_FlashWb_CYC_I    : Std_Logic;
    Signal AsynchPort_FlashWb_STB_I    : Std_Logic;
    Signal AsynchPort_FlashWb_ACK_O    : Std_Logic;
    Signal AsynchPort_FlashWb_ADR_I    : Std_Logic_Vector(23 DownTo 0);
    Signal AsynchPort_FlashWb_DAT_I    : Std_Logic_Vector(31 DownTo 0);
    Signal AsynchPort_FlashWb_DAT_O    : Std_Logic_Vector(31 DownTo 0);
    Signal AsynchPort_FlashWb_WE_I     : Std_Logic;
    Signal AsynchPort_FlashWb_SEL_I    : Std_Logic_Vector(3  DownTo 0);


    -- NewAsynchPort
    Signal AsynchPortMemDataI : Std_Logic_Vector(31 Downto 0);
    Signal AsynchPortMemDataO : Std_Logic_Vector(31 Downto 0);
    Signal AsynchPortAdrO     : Std_Logic_Vector(23 Downto 0);
    Signal AsynchPortSelO     : Std_Logic_Vector( 3 Downto 0);
    Signal AsynchPortSramGo   : Std_Logic;
    Signal AsynchPortFlashGo  : Std_Logic;
    Signal AsynchPortUsbGo    : Std_Logic;
    Signal AsynchPortAckI     : Std_Logic;
    Signal AsynchPortWrite    : Std_Logic;

    Signal SdramAsynchPortMemAdrI          : STD_LOGIC_VECTOR(23 DOWNTO 0);
    Signal SdramAsynchPortMemBeI           : STD_LOGIC_VECTOR(3 DOWNTO 0);
    Signal SdramAsynchPortMemDataI         : STD_LOGIC_VECTOR(31 DOWNTO 0);
    Signal SdramAsynchPortMemDataO         : STD_LOGIC_VECTOR(31 DOWNTO 0);
    Signal SdramAsynchPortMemDataWriteI    : STD_LOGIC;
    Signal SdramAsynchPortMemFlashCsI      : STD_LOGIC;
    Signal SdramAsynchPortMemFlashResetI   : STD_LOGIC;
    Signal SdramAsynchPortMemOeI           : STD_LOGIC;
    Signal SdramAsynchPortMemSramCsI       : STD_LOGIC;
    Signal SdramAsynchPortMemUsbCsI        : STD_LOGIC;
    Signal SdramAsynchPortMemUsbResetI     : STD_LOGIC;
    Signal SdramAsynchPortMemWeI           : STD_LOGIC;

    Signal SdramFlashCs                      : STD_LOGIC;
    Signal SdramFlashReset                   : STD_LOGIC;
    Signal SdramMemAdrBottom                 : STD_LOGIC;
    Signal SdramMemAdrTop                    : STD_LOGIC_VECTOR(7 DOWNTO 0);
    Signal SdramMemOe                        : STD_LOGIC;
    Signal SdramSramCs                       : STD_LOGIC;
    Signal SdramUsbCs                        : STD_LOGIC;
    Signal SdramUsbReset                     : STD_LOGIC;





-- -----------------------------------------------------------------------------
Begin
-- -----------------------------------------------------------------------------

    Signal_GND             <= '0';
    Signal_GND_4           <= (Others=>'0');
    Signal_GND_32          <= (Others=>'0');
    Signal_GND_SRAM_ADR_I  <= (Others=>'0');
    Signal_GND_FLASH_ADR_I <= (Others=>'0');
    Signal_GND_USB_ADR_I   <= (Others=>'0');

    -- -------------------------------------------------------------------------
    U_DataCache : Configurable_ReplaceWith_Designator_DataCache Port Map (
         CLKA     => DataCache_WbClk,
         DINA     => DataCache_WbDin,
         DOUTA    => DataCache_WbDout,
         ADDRA    => DataCache_WbAddr,
         BYTE_WEA => DataCache_WbByteW,
         WEA      => DataCache_WbWe,

         CLKB     => DataCache_MemClk,
         DINB     => DataCache_MemDin,
         DOUTB    => DataCache_MemDout,
         ADDRB    => DataCache_MemAddr,
         BYTE_WEB => DataCache_MemByteW,
         WEB      => DataCache_MemWe
    );
    -- -------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    U_TagCache : Configurable_ReplaceWith_Designator_TagCache Port Map (
        CLKA  => TagCache_WbClk,
        DINA  => TagCache_WbDin,
        DOUTA => TagCache_WbDout,
        ADDRA => TagCache_WbAddr,
        WEA   => TagCache_WbWe,

        CLKB  => TagCache_MemClk,
        DINB  => TagCache_MemDin,
        DOUTB => TagCache_MemDout,
        ADDRB => TagCache_MemAddr,
        WEB   => TagCache_MemWe
    );
    -- -------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    U_CommandCache : Configurable_ReplaceWith_Designator_CommandPipe Port Map (
        CLKA  => CmdCache_WbClk,
        DINA  => CmdCache_WbDin,
        DOUTA => CmdCache_WbDout,
        ADDRA => CmdCache_WbAddr,
        WEA   => CmdCache_WbWe,

        CLKB  => CmdCache_MemClk,
        DINB  => CmdCache_MemDin,
        DOUTB => CmdCache_MemDout,
        ADDRB => CmdCache_MemAddr,
        WEB   => CmdCache_MemWe
    );
    -- -------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    U_WishboneInterface : Configurable_ReplaceWith_Designator_WishboneInterface Port Map (
    Debug_WishboneState => Debug_WishboneState,
        -- Cache Command Interface
        CACHE_CMD_CLK     => CmdCache_WbClk,
        CACHE_CMD_WE_O    => CmdCache_WbWe,
        CACHE_CMD_DAT_I   => CmdCache_WbDout,
        CACHE_CMD_DAT_O   => CmdCache_WbDin,
        CACHE_CMD_ADR_O   => CmdCache_WbAddr,

        -- Dual Port RAM Wishbone Data Cache Interface
        DP_RAM_DAT_CLK_O  => DataCache_WbClk,
        DP_RAM_DAT_RST_O  => Open,
        DP_RAM_DAT_DATA_I => DataCache_WbDout,
        DP_RAM_DAT_DATA_O => DataCache_WbDin,
        DP_RAM_DAT_ADR_O  => DataCache_WbAddr,
        DP_RAM_DAT_BE_O   => DataCache_WbByteW,
        DP_RAM_DAT_WE_O   => DataCache_WbWe,

        -- Dual Port RAM Cache Tag Interface
        DP_RAM_TAG_CLK_O  => TagCache_WbClk,
        DP_RAM_TAG_RST_O  => Open,
        DP_RAM_TAG_DATA_I => TagCache_WbDout,
        DP_RAM_TAG_DATA_O => TagCache_WbDin,
        DP_RAM_TAG_ADR_O  => TagCache_WbAddr,
        DP_RAM_TAG_BE_O   => Open,
        DP_RAM_TAG_WE_O   => TagCache_WbWe,

        ASYNCH_PORT_MEM_DATA_I   => AsynchPortMemDataI,
        ASYNCH_PORT_MEM_DATA_O   => AsynchPortMemDataO,
        ASYNCH_PORT_MEM_ADR_O    => AsynchPortAdrO,
        ASYNCH_PORT_MEM_SEL_O    => AsynchPortSelO,
        ASYNCH_PORT_MEM_SRAM_GO  => AsynchPortSramGo,
        ASYNCH_PORT_MEM_FLASH_GO => AsynchPortFlashGo,
        ASYNCH_PORT_MEM_USB_GO   => AsynchPortUsbGo,
        ASYNCH_PORT_MEM_ACK_I    => AsynchPortAckI,
        ASYNCH_PORT_MEM_WRITE    => AsynchPortWrite,


        -- If Any Interface is Not Enabled then disable it at this level
        -- Sram Wishbone Interface
        WB_SRAM_CLK_I     => SRAM_CLK_I,                                           --IncludeIf_Sram
        WB_SRAM_RST_I     => SRAM_RST_I,                                           --IncludeIf_Sram
        WB_SRAM_STB_I     => SRAM_STB_I,                                           --IncludeIf_Sram
        WB_SRAM_CYC_I     => SRAM_CYC_I,                                           --IncludeIf_Sram
        WB_SRAM_ACK_O     => SRAM_ACK_O,                                           --IncludeIf_Sram
        WB_SRAM_ADR_I     => SRAM_ADR_I,                                           --IncludeIf_Sram
        WB_SRAM_DAT_I     => SRAM_DAT_I,                                           --IncludeIf_Sram
        WB_SRAM_DAT_O     => SRAM_DAT_O,                                           --IncludeIf_Sram
        WB_SRAM_SEL_I     => SRAM_SEL_I,                                           --IncludeIf_Sram
        WB_SRAM_WE_I      => SRAM_WE_I,                                            --IncludeIf_Sram
        -- Sram Wishbone Interface
        WB_SRAM_CLK_I     => SDRAM_CLK_I,                                          --ExcludeIf_Sram
        WB_SRAM_RST_I     => SDRAM_RST_I,                                          --ExcludeIf_Sram
        WB_SRAM_STB_I     => Signal_GND,                                           --ExcludeIf_Sram
        WB_SRAM_CYC_I     => Signal_GND,                                           --ExcludeIf_Sram
        WB_SRAM_ACK_O     => Open,                                                 --ExcludeIf_Sram
        WB_SRAM_ADR_I     => Signal_GND_SRAM_ADR_I,                                --ExcludeIf_Sram
        WB_SRAM_DAT_I     => Signal_GND_32,                                        --ExcludeIf_Sram
        WB_SRAM_DAT_O     => Open,                                                 --ExcludeIf_Sram
        WB_SRAM_SEL_I     => Signal_GND_4,                                         --ExcludeIf_Sram
        WB_SRAM_WE_I      => Signal_GND,                                           --ExcludeIf_Sram
        -- Flash Wishbone Interface                                             --IncludeIf_Flash
        WB_FLASH_CLK_I    => FLASH_CLK_I,                                          --IncludeIf_Flash
        WB_FLASH_RST_I    => FLASH_RST_I,                                          --IncludeIf_Flash
        WB_FLASH_STB_I    => FLASH_STB_I,                                          --IncludeIf_Flash
        WB_FLASH_CYC_I    => FLASH_CYC_I,                                          --IncludeIf_Flash
        WB_FLASH_ACK_O    => FLASH_ACK_O,                                          --IncludeIf_Flash
        WB_FLASH_ADR_I    => FLASH_ADR_I,                                          --IncludeIf_Flash
        WB_FLASH_DAT_I    => FLASH_DAT_I,                                          --IncludeIf_Flash
        WB_FLASH_DAT_O    => FLASH_DAT_O,                                          --IncludeIf_Flash
        WB_FLASH_SEL_I    => FLASH_SEL_I,                                          --IncludeIf_Flash
        WB_FLASH_WE_I     => FLASH_WE_I,                                           --IncludeIf_Flash
        -- Flash Wishbone Interface                                             --ExcludeIf_Flash
        WB_FLASH_CLK_I    => SDRAM_CLK_I,                                          --ExcludeIf_Flash
        WB_FLASH_RST_I    => SDRAM_RST_I,                                          --ExcludeIf_Flash
        WB_FLASH_STB_I    => Signal_GND,                                           --ExcludeIf_Flash
        WB_FLASH_CYC_I    => Signal_GND,                                           --ExcludeIf_Flash
        WB_FLASH_ACK_O    => Open,                                                 --ExcludeIf_Flash
        WB_FLASH_ADR_I    => Signal_GND_FLASH_ADR_I,                               --ExcludeIf_Flash
        WB_FLASH_DAT_I    => Signal_GND_32,                                        --ExcludeIf_Flash
        WB_FLASH_DAT_O    => Open,                                                 --ExcludeIf_Flash
        WB_FLASH_SEL_I    => Signal_GND_4,                                         --ExcludeIf_Flash
        WB_FLASH_WE_I     => Signal_GND,                                           --ExcludeIf_Flash
--        -- Flash Wishbone Interface

        -- Usb Wishbone Interface
        WB_USB_CLK_I      => USB_CLK_I,                                            --IncludeIf_Usb
        WB_USB_RST_I      => USB_RST_I,                                            --IncludeIf_Usb
        WB_USB_STB_I      => USB_STB_I,                                            --IncludeIf_Usb
        WB_USB_CYC_I      => USB_CYC_I,                                            --IncludeIf_Usb
        WB_USB_ACK_O      => USB_ACK_O,                                            --IncludeIf_Usb
        WB_USB_ADR_I      => USB_ADR_I,                                            --IncludeIf_Usb
        WB_USB_DAT_I      => USB_DAT_I,                                            --IncludeIf_Usb
        WB_USB_DAT_O      => USB_DAT_O,                                            --IncludeIf_Usb
        WB_USB_SEL_I      => USB_SEL_I,                                            --IncludeIf_Usb
        WB_USB_WE_I       => USB_WE_I,                                             --IncludeIf_Usb
        -- Usb Wishbone Interface
        WB_USB_CLK_I      => SDRAM_CLK_I,                                          --ExcludeIf_Usb
        WB_USB_RST_I      => SDRAM_RST_I,                                          --ExcludeIf_Usb
        WB_USB_STB_I      => Signal_GND,                                           --ExcludeIf_Usb
        WB_USB_CYC_I      => Signal_GND,                                           --ExcludeIf_Usb
        WB_USB_ACK_O      => Open,                                                 --ExcludeIf_Usb
        WB_USB_ADR_I      => Signal_GND_USB_ADR_I,                                 --ExcludeIf_Usb
        WB_USB_DAT_I      => Signal_GND_32,                                        --ExcludeIf_Usb
        WB_USB_DAT_O      => Open,                                                 --ExcludeIf_Usb
        WB_USB_SEL_I      => Signal_GND_4,                                         --ExcludeIf_Usb
        WB_USB_WE_I       => Signal_GND,                                           --ExcludeIf_Usb

        -- Sdram Wishbone Interface
        WB_SDRAM_CLK_I    => SDRAM_CLK_I,
        WB_SDRAM_RST_I    => SDRAM_RST_I,
        WB_SDRAM_STB_I    => SDRAM_STB_I,
        WB_SDRAM_CYC_I    => SDRAM_CYC_I,
        WB_SDRAM_ACK_O    => SDRAM_ACK_O,
        WB_SDRAM_ADR_I    => SDRAM_ADR_I,
        WB_SDRAM_DAT_I    => SDRAM_DAT_I,
        WB_SDRAM_DAT_O    => SDRAM_DAT_O,
        WB_SDRAM_SEL_I    => SDRAM_SEL_I,
        WB_SDRAM_WE_I     => SDRAM_WE_I
    );
    -- -------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    U_SdramEngine : ReplaceWith_SdramEngineName Port Map (
    Debug_SdramState => Debug_SdramState,
        -- System
        MEM_CLK                         => MEM_CLK,
        MEM_RST                         => SDRAM_RST_I,
        -- Data Cache
        CACHE_DAT_CLK_O                 => DataCache_MemClk,
        CACHE_DAT_BE_O                  => DataCache_MemByteW,
        CACHE_DAT_WE_O                  => DataCache_MemWe,
        CACHE_DAT_ADR_O                 => DataCache_MemAddr,
        CACHE_DAT_DATA_I                => DataCache_MemDout,
        CACHE_DAT_DATA_O                => DataCache_MemDin,
        -- Tag Cache
        CACHE_TAG_CLK_O                 => TagCache_MemClk,
        CACHE_TAG_WE_O                  => TagCache_MemWe,
        CACHE_TAG_ADR_O                 => TagCache_MemAddr,
        CACHE_TAG_DATA_I                => TagCache_MemDout,
        CACHE_TAG_DATA_O                => TagCache_MemDin,
        -- Command Pipe
        CACHE_CMD_CLK_O                 => CmdCache_MemClk,
        CACHE_CMD_WE_O                  => CmdCache_MemWe,
        CACHE_CMD_ADR_O                 => CmdCache_MemAddr,
        CACHE_CMD_DATA_O                => CmdCache_MemDin,
        CACHE_CMD_DATA_I                => CmdCache_MemDout,
        -- Timer Configuration
        TIMER_VALUE_TRP                 => TimerValue_tRP,
        TIMER_VALUE_TWR                 => TimerValue_tWR,
        TIMER_VALUE_TRFC                => TimerValue_tRFC,
        TIMER_VALUE_TMRD                => TimerValue_tMRD,
        TIMER_VALUE_TRCD                => TimerValue_tRCD,
        TIMER_VALUE_TCL                 => TimerValue_tCL,
        -- Memory Mode Register
        LOAD_MODE_REGISTER              => LoadModeRegister,
        -- Address Resover for diff memory geometries
        RESOLVED_BANK_ADDRESS_FLUSH     => FinalBankAddress_Flush,
        RESOLVED_ROW_ADDRESS_FLUSH      => FinalRowAddress_Flush,
        RESOLVED_COLUMN_ADDRESS_FLUSH   => FinalColumnAddress_Flush,
        RESOLVED_BANK_ADDRESS_REPLACE   => FinalBankAddress_Replace,
        RESOLVED_ROW_ADDRESS_REPLACE    => FinalRowAddress_Replace,
        RESOLVED_COLUMN_ADDRESS_REPLACE => FinalColumnAddress_Replace,

        RESOLVER_FLUSH_ADDRESS          => ToResolve_FlushAddress,
        RESOLVER_REPLACE_ADDRESS        => ToResolve_ReplaceAddress,
        RESOLVER_FLUSH_INDEX            => ToResolve_FlushIndex,

        -- Refresh engine reload value
        REFRESH_RELOAD_VALUE            => TimerValue_Refresh,


        ASYNCH_PORT_MEM_ADR_I           => SdramAsynchPortMemAdrI,
        ASYNCH_PORT_MEM_BE_I            => SdramAsynchPortMemBeI,
        ASYNCH_PORT_MEM_DATA_I          => SdramAsynchPortMemDataI,
        ASYNCH_PORT_MEM_DATA_O          => SdramAsynchPortMemDataO,
        ASYNCH_PORT_MEM_DATA_WRITE_I    => SdramAsynchPortMemDataWriteI,
        ASYNCH_PORT_MEM_FLASH_CS_I      => SdramAsynchPortMemFlashCsI,
        ASYNCH_PORT_MEM_FLASH_RESET_I   => SdramAsynchPortMemFlashResetI,
        ASYNCH_PORT_MEM_OE_I            => SdramAsynchPortMemOeI,
        ASYNCH_PORT_MEM_SRAM_CS_I       => SdramAsynchPortMemSramCsI,
        ASYNCH_PORT_MEM_USB_CS_I        => SdramAsynchPortMemUsbCsI,
        ASYNCH_PORT_MEM_USB_RESET_I     => SdramAsynchPortMemUsbResetI,
        ASYNCH_PORT_MEM_WE_I            => SdramAsynchPortMemWeI,


        FLASH_CS                        => SdramFlashCs,
        FLASH_RESET                     => SdramFlashReset,
        MEM_ADR_BOTTOM                  => SdramMemAdrBottom,
        MEM_ADR_TOP                     => SdramMemAdrTop,
        MEM_OE                          => SdramMemOe,
        SRAM_CS                         => SdramSramCs,
        USB_CS                          => SdramUsbCs,
        USB_RESET                       => SdramUsbReset,

        -- Memory Bus
        SDRAM_DATA_O                    => SdramDataOut,
        SDRAM_DATA_I                    => SdramDataIn,
        SDRAM_DATA_E                    => SdramDataEnable,
        SDRAM_A                         => SdramAddress,
        SDRAM_BA                        => SdramBankAddress,
        SDRAM_W                         => SdramWrite,
        SDRAM_BE                        => SdramByteEnable,
        SDRAM_CKE                       => SdramClockEnable,
        SDRAM_RAS                       => SdramNRas,
        SDRAM_CAS                       => SdramNCas,
        SDRAM_E                         => SdramChipSelect
    );
    -- -------------------------------------------------------------------------


    --------------------------------------------------------------------------------
    U_ReplaceWith_Designator_Sram_x32: ReplaceWith_Designator_Sram_x32 Port Map (
         SRAM_D_O  => AsynchPort_Sram_D_O,
         SRAM_D_I  => AsynchPort_Sram_D_I,
         SRAM_D_E  => AsynchPort_Sram_D_E,
         SRAM_A    => AsynchPort_Sram_A,
         SRAM_OE   => AsynchPort_Sram_OE,
         SRAM_WE   => AsynchPort_Sram_WE,
         SRAM_CE   => AsynchPort_Sram_CE,
         SRAM_BE_E => AsynchPort_Sram_BE_E,
         ---------------------------------------------------------------------------

         ---------------------------------------------------------------------------
         -- Wishbone Slave Port
         ---------------------------------------------------------------------------
         CLK_I => AsynchPort_SramWb_CLK_I,
         RST_I => AsynchPort_SramWb_RST_I,
         CYC_I => AsynchPort_SramWb_CYC_I,
         STB_I => AsynchPort_SramWb_STB_I,
         ACK_O => AsynchPort_SramWb_ACK_O,
         ADR_I => AsynchPort_SramWb_ADR_I,
         DAT_I => AsynchPort_SramWb_DAT_I,
         DAT_O => AsynchPort_SramWb_DAT_O,
         WE_I  => AsynchPort_SramWb_WE_I,
         SEL_I => AsynchPort_SramWb_SEL_I
       );

    --------------------------------------------------------------------------------

    --------------------------------------------------------------------------------
    U_ReplaceWith_Designator_USB_X32: ReplaceWith_Designator_USB_X32 Port Map (

         MEM_D_O       => AsynchPort_Usb_MEM_D_O      ,
         MEM_D_I       => AsynchPort_Usb_MEM_D_I      ,
         MEM_D_E       => AsynchPort_Usb_MEM_D_E      ,
         MEM_A         => AsynchPort_Usb_MEM_A        ,
         MEM_OE        => AsynchPort_Usb_MEM_OE       ,
         MEM_W         => AsynchPort_Usb_MEM_W        ,
         MEM_BE        => AsynchPort_Usb_MEM_BE       ,
         MEM_USB_E     => AsynchPort_Usb_MEM_USB_E    ,
         MEM_USB_RESET => AsynchPort_Usb_MEM_USB_RESET,
         ---------------------------------------------------------------------------

         ---------------------------------------------------------------------------
         -- Wishbone Slave Port
         ---------------------------------------------------------------------------
         USB_CLK_I => AsynchPort_UsbWb_USB_CLK_I,
         USB_RST_I => AsynchPort_UsbWb_USB_RST_I,
         USB_CYC_I => AsynchPort_UsbWb_USB_CYC_I,
         USB_STB_I => AsynchPort_UsbWb_USB_STB_I,
         USB_ACK_O => AsynchPort_UsbWb_USB_ACK_O,
         USB_ADR_I => AsynchPort_UsbWb_USB_ADR_I,
         USB_DAT_I => AsynchPort_UsbWb_USB_DAT_I,
         USB_DAT_O => AsynchPort_UsbWb_USB_DAT_O,
         USB_WE_I  => AsynchPort_UsbWb_USB_WE_I ,
         USB_SEL_I => AsynchPort_UsbWb_USB_SEL_I
       );
    --------------------------------------------------------------------------------


    --------------------------------------------------------------------------------
    U_ReplaceWith_Designator_Flash_x16: ReplaceWith_Designator_Flash_x16 Port Map (
         ---------------------------------------------------------------------------
         FLASH_D_O  => AsynchPort_Flash_FLASH_D_O,
         FLASH_D_I  => AsynchPort_Flash_FLASH_D_I,
         FLASH_D_E  => AsynchPort_Flash_FLASH_D_E,
         FLASH_A    => AsynchPort_Flash_FLASH_A,
         FLASH_OE   => AsynchPort_Flash_FLASH_OE,
         FLASH_WE   => AsynchPort_Flash_FLASH_WE,
         FLASH_CE   => AsynchPort_Flash_FLASH_CE,
         ---------------------------------------------------------------------------
         CLK_I => AsynchPort_FlashWb_CLK_I,
         RST_I => AsynchPort_FlashWb_RST_I,
         CYC_I => AsynchPort_FlashWb_CYC_I,
         STB_I => AsynchPort_FlashWb_STB_I,
         ACK_O => AsynchPort_FlashWb_ACK_O,
         ADR_I => AsynchPort_FlashWb_ADR_I,
         DAT_I => AsynchPort_FlashWb_DAT_I,
         DAT_O => AsynchPort_FlashWb_DAT_O,
         WE_I  => AsynchPort_FlashWb_WE_I,
         SEL_I => AsynchPort_FlashWb_SEL_I
       );
    --------------------------------------------------------------------------------

    --------------------------------------------------------------------------------
    U_Configurable_ReplaceWith_Designator_ASYNCH_PORT_SELECT: Configurable_ReplaceWith_Designator_ASYNCH_PORT_SELECT Port Map (
        RST_I         => SDRAM_RST_I,
        CLK_I         => SDRAM_CLK_I,
         ---------------------------------------------------------------------------
        --SRAM WISHBONE
        SRAM_STB_O    => AsynchPort_SramWb_STB_I,
        SRAM_CYC_O    => AsynchPort_SramWb_CYC_I,
        SRAM_DAT_O    => AsynchPort_SramWb_DAT_I,
        SRAM_DAT_I    => AsynchPort_SramWb_DAT_O,
        SRAM_ADR_O    => AsynchPort_SramWb_ADR_I,
        SRAM_SEL_O    => AsynchPort_SramWb_SEL_I,
        SRAM_WE_O     => AsynchPort_SramWb_WE_I,
        SRAM_ACK_I    => AsynchPort_SramWb_ACK_O,
        SRAM_RST_O    => AsynchPort_SramWb_RST_I,
        SRAM_CLK_O    => AsynchPort_SramWb_CLK_I,
        --SRAM MEMORY
        SRAM_MEM_D_O  => AsynchPort_Sram_D_I,
        SRAM_MEM_D_I  => AsynchPort_Sram_D_O,
        SRAM_MEM_D_E  => AsynchPort_Sram_D_E,
        SRAM_MEM_A_I  => AsynchPort_Sram_A,
        SRAM_MEM_BE_W => AsynchPort_Sram_BE_E,
        SRAM_MEM_OE   => AsynchPort_Sram_OE,
        SRAM_MEM_WE   => AsynchPort_Sram_WE,
        SRAM_MEM_CE   => AsynchPort_Sram_CE,

        -- FLASH WISHBONE
        FLASH_STB_O   => AsynchPort_FlashWb_STB_I,
        FLASH_CYC_O   => AsynchPort_FlashWb_CYC_I,
        FLASH_DAT_O   => AsynchPort_FlashWb_DAT_I,
        FLASH_DAT_I   => AsynchPort_FlashWb_DAT_O,
        FLASH_ADR_O   => AsynchPort_FlashWb_ADR_I,
        FLASH_SEL_O   => AsynchPort_FlashWb_SEL_I,
        FLASH_WE_O    => AsynchPort_FlashWb_WE_I,
        FLASH_ACK_I   => AsynchPort_FlashWb_ACK_O,
        FLASH_RST_O   => AsynchPort_FlashWb_RST_I,
        FLASH_CLK_O   => AsynchPort_FlashWb_CLK_I,
        -- FLASH MEMORY
        FLASH_MEM_D_O => AsynchPort_Flash_FLASH_D_I,
        FLASH_MEM_D_I => AsynchPort_Flash_FLASH_D_O,
        FLASH_MEM_D_E => AsynchPort_Flash_FLASH_D_E,
        FLASH_MEM_A_I => AsynchPort_Flash_FLASH_A,
        FLASH_MEM_OE  => AsynchPort_Flash_FLASH_OE,
        FLASH_MEM_WE  => AsynchPort_Flash_FLASH_WE,
        FLASH_MEM_CE  => AsynchPort_Flash_FLASH_CE,

            --USB WISHBONE
        USB_STB_O    => AsynchPort_UsbWb_USB_STB_I,
        USB_CYC_O    => AsynchPort_UsbWb_USB_CYC_I,
        USB_DAT_O    => AsynchPort_UsbWb_USB_DAT_I,
        USB_DAT_I    => AsynchPort_UsbWb_USB_DAT_O,
        USB_ADR_O    => AsynchPort_UsbWb_USB_ADR_I,
        USB_SEL_O    => AsynchPort_UsbWb_USB_SEL_I,
        USB_WE_O     => AsynchPort_UsbWb_USB_WE_I,
        USB_ACK_I    => AsynchPort_UsbWb_USB_ACK_O,
        USB_RST_O    => AsynchPort_UsbWb_USB_RST_I,
        USB_CLK_O    => AsynchPort_UsbWb_USB_CLK_I,
        --USB MEMORY
        USB_MEM_D_O   => AsynchPort_Usb_MEM_D_I,
        USB_MEM_D_I   => AsynchPort_Usb_MEM_D_O,
        USB_MEM_D_E   => AsynchPort_Usb_MEM_D_E,
        USB_MEM_A_I   => AsynchPort_Usb_MEM_A,
        USB_MEM_BE_W  => AsynchPort_Usb_MEM_BE,
        USB_MEM_OE    => AsynchPort_Usb_MEM_OE,
        USB_MEM_WE    => AsynchPort_Usb_MEM_W,
        USB_MEM_CE    => AsynchPort_Usb_MEM_USB_E,
        USB_MEM_RESET => AsynchPort_Usb_MEM_USB_RESET,

        -- engine side
        ASYNCH_PORT_MEM_DATA_I        => SdramAsynchPortMemDataO,          --
        ASYNCH_PORT_MEM_ADR_O         => SdramAsynchPortMemAdrI,           --
        ASYNCH_PORT_MEM_BE_O          => SdramAsynchPortMemBeI,            --
        ASYNCH_PORT_MEM_DATA_O        => SdramAsynchPortMemDataI,          --
        ASYNCH_PORT_MEM_DATA_WRITE_O  => SdramAsynchPortMemDataWriteI,     --
        ASYNCH_PORT_MEM_FLASH_CS_O    => SdramAsynchPortMemFlashCsI,       --
        ASYNCH_PORT_MEM_FLASH_RESET_O => SdramAsynchPortMemFlashResetI,    --
        ASYNCH_PORT_MEM_OE_O          => SdramAsynchPortMemOeI,            --
        ASYNCH_PORT_MEM_SRAM_CS_O     => SdramAsynchPortMemSramCsI,        --
        ASYNCH_PORT_MEM_USB_CS_O      => SdramAsynchPortMemUsbCsI,         --
        ASYNCH_PORT_MEM_USB_RESET_O   => SdramAsynchPortMemUsbResetI,      --
        ASYNCH_PORT_MEM_WE_O          => SdramAsynchPortMemWeI,            --
        -- wishbone side
        TO_MEM_CTRL_ACK_O             => AsynchPortAckI,
        TO_MEM_CTRL_ADR_I             => AsynchPortAdrO,
        TO_MEM_CTRL_DATA_I            => AsynchPortMemDataO,
        TO_MEM_CTRL_DATA_O            => AsynchPortMemDataI,
        TO_MEM_CTRL_FLASH_GO          => AsynchPortFlashGo,
        TO_MEM_CTRL_SEL_O             => AsynchPortSelO,
        TO_MEM_CTRL_SRAM_GO           => AsynchPortSramGo,
        TO_MEM_CTRL_USB_GO            => AsynchPortUsbGo,
        TO_MEM_CTRL_WE_I              => AsynchPortWrite
    );
    --------------------------------------------------------------------------------


    -- -------------------------------------------------------------------------
    -- Provide timing configuration
    -- -------------------------------------------------------------------------
    TimerValue_tRP     <= cTimerValue_tRP;
    TimerValue_tWR     <= cTimerValue_tWR;
    TimerValue_tRFC    <= cTimerValue_tRFC;
    TimerValue_tMRD    <= cTimerValue_tMRD;
    TimerValue_tRCD    <= cTimerValue_tRCD;
    TimerValue_tCL     <= cTimerValue_tCL;
    TimerValue_Refresh <= cTimerValue_Refresh;
    LoadModeRegister   <= cLoadModeRegister;
    -- -------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    -- Resolve addresses
    -- -------------------------------------------------------------------------
    FinalBankAddress_Flush     <= ToResolve_FlushAddress(cFlushBankAddressMsb Downto cFlushBankAddressLsb);
    FinalRowAddress_Flush      <= ToResolve_FlushAddress(cFlushRowAddressMsb Downto cFlushRowAddressLsb) & ToResolve_FlushIndex(cFlushIndexRowAddressMsb Downto cFlushIndexRowAddressLsb);
    FinalColumnAddress_Flush   <= cFlushColumnAddressPadding & ToResolve_FlushIndex(cFlushIndexMsb Downto cFlushIndexLsb);

    FinalBankAddress_Replace   <= ToResolve_ReplaceAddress(cReplaceBankAddressMsb Downto cReplaceBankAddressLsb);
    FinalRowAddress_Replace    <= ToResolve_ReplaceAddress(cReplaceRowAddressMsb Downto cReplaceRowAddressLsb);
    FinalColumnAddress_Replace <= cReplaceColumnAddressPadding & ToResolve_ReplaceAddress(cReplaceColumnAddressMsb Downto cReplaceColumnAddressLsb);
    -- -------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    Process(SdramDataOut, SdramDataEnable)
    Begin
        For I In 0 To 31 Loop
            If SdramDataEnable(I) = '1' Then
                MEM_D(I) <= SdramDataOut(I);
            Else
                MEM_D(I) <= 'Z';
            End If;
        End Loop;
    End Process;
    SdramDataIn <= MEM_D;
    -- -------------------------------------------------------------------------
    MEM_A          <= SdramMemAdrTop & SdramBankAddress & SdramAddress & SdramMemAdrBottom;
    -- -------------------------------------------------------------------------
    MEM_W          <= SdramWrite;
    MEM_BE         <= SdramByteEnable;
    MEM_SDRAM_CKE  <= SdramClockEnable;
    MEM_SDRAM_RAS  <= SdramNRas;
    MEM_SDRAM_CAS  <= SdramNCas;
    MEM_SDRAM_E    <= SdramChipSelect;
    -- -------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    MEM_OE          <= SdramMemOe;
    MEM_FLASH_RESET <= '1';
    MEM_FLASH_E     <= SdramFlashCs;
    MEM_SRAM_E      <= SdramSramCs;
    -- -------------------------------------------------------------------------
    MEM_USB_RESET   <= SdramUsbReset;  -- IncludeIf_Usb
    MEM_USB_E       <= SdramUsbCs; -- IncludeIf_Usb
    -- -------------------------------------------------------------------------

-- -----------------------------------------------------------------------------
End RTL;
-- -----------------------------------------------------------------------------

--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
Library IEEE;
Use IEEE.Std_Logic_1164.all;
Use IEEE.std_logic_unsigned.all;
Use IEEE.Std_Logic_Arith.All;
Use Work.Configurable_ReplaceWith_Designator_GlobalConfig.All;

--------------------------------------------------------------------------------
Entity ReplaceWith_Designator_Flash_x16 Is Port
   (
     ----------------------------------------------------------------------------- IncludeIf_1_Device_X_16
     FLASH_D_O  : Out Std_Logic_Vector(31 DownTo 0);                             -- IncludeIf_1_Device_X_16
     FLASH_D_I  : In  Std_Logic_Vector(31 DownTo 0);                             -- IncludeIf_1_Device_X_16
     FLASH_D_E  : Out Std_Logic;
     FLASH_A    : Out   Std_Logic_Vector(23 DownTo 0);   -- IncludeIf_1_Device_X_16
     FLASH_OE   : Out   Std_Logic;                                               -- IncludeIf_1_Device_X_16
     FLASH_WE   : Out   Std_Logic;                                               -- IncludeIf_1_Device_X_16
     FLASH_CE   : Out   Std_Logic;                                               -- IncludeIf_1_Device_X_16
     ----------------------------------------------------------------------------- IncludeIf_1_Device_X_16

     ---------------------------------------------------------------------------
     -- Wishbone Slave Port
     ---------------------------------------------------------------------------
     CLK_I      : In    Std_Logic;
     RST_I      : In    Std_Logic;
     CYC_I      : In    Std_Logic;
     STB_I      : In    Std_Logic;
     ACK_O      : Out   Std_Logic;
     ADR_I      : In    Std_Logic_Vector(23 DownTo 0);
     DAT_I      : In    Std_Logic_Vector(31 DownTo 0);
     DAT_O      : Out   Std_Logic_Vector(31 DownTo 0);
     WE_I       : In    Std_Logic;
     SEL_I      : In    Std_Logic_Vector(3  DownTo 0)
   );

End ReplaceWith_Designator_Flash_x16;
--------------------------------------------------------------------------------

------------------------------------------------------------------------------------ IncludeIf_2_Device_X_8
----                   ByteLanes[3..0]   Cycle  WE0  WE1  CE       OE  A0         -- IncludeIf_2_Device_X_8
----                                     B3     B2   B1   B0       OE  A0         -- IncludeIf_2_Device_X_8
------------------------------------------------------------------------------    -- IncludeIf_2_Device_X_8
----Read             |     XXXX          0      1    1    0        0   0          -- IncludeIf_2_Device_X_8
----                 |                   1      1    1    0        0   1          -- IncludeIf_2_Device_X_8
----None             |     XXXX          0      1    1    1        1   0          -- IncludeIf_2_Device_X_8
----                 |                   1      1    1    1        1   1          -- IncludeIf_2_Device_X_8
----Write word       |     1111          0      0    0    0        1   0          -- IncludeIf_2_Device_X_8
----                 |                   1      0    0    0        1   1          -- IncludeIf_2_Device_X_8
----Write low half   |     0011          0      1    1    0        1   0          -- IncludeIf_2_Device_X_8
----                 |                   1      0    0    0        1   1          -- IncludeIf_2_Device_X_8
----Write high half  |     1100          0      0    0    0        1   0          -- IncludeIf_2_Device_X_8
----                 |                   1      1    1    0        1   1          -- IncludeIf_2_Device_X_8
----Write Byte0      |     0001          0      0    1    0        1   0          -- IncludeIf_2_Device_X_8
----                 |                   1      1    1    0        1   1          -- IncludeIf_2_Device_X_8
----Write Byte1      |     0010          0      1    0    0        1   0          -- IncludeIf_2_Device_X_8
----                 |                   1      1    1    0        1   1          -- IncludeIf_2_Device_X_8
----Write Byte2      |     0100          0      1    1    0        1   0          -- IncludeIf_2_Device_X_8
----                 |                   1      0    1    0        1   1          -- IncludeIf_2_Device_X_8
----Write Byte3      |     1000          0      1    1    0        1   0          -- IncludeIf_2_Device_X_8
----                 |                   1      1    0    0        1   0          -- IncludeIf_2_Device_X_8
----Write Bytes012   |     0111          0      0    0    0        1   0          -- IncludeIf_2_Device_X_8
----                 |                   1      0    1    0        1   1          -- IncludeIf_2_Device_X_8
----Write Bytes123   |     1110          0      1    0    0        1   0          -- IncludeIf_2_Device_X_8
----                 |                   1      0    0    0        1   0          -- IncludeIf_2_Device_X_8
----Write Bytes12    |     0110          0      1    0    0        1   0          -- IncludeIf_2_Device_X_8
----                 |                   1      0    1    0        1   1          -- IncludeIf_2_Device_X_8
------------------------------------------------------------------------------------ IncludeIf_2_Device_X_8
--                   ByteLanes[3..0]   Cycle  LB   UB   CE       OE  WE  A0     -- IncludeIf_1_Device_X_16
--                                     B3     B2   B1   B0       OE  WE  A0     -- IncludeIf_1_Device_X_16
---------------------------------------------------------------------------------- IncludeIf_1_Device_X_16
--Read             |     XXXX          0      0    0    0        0   1   0      -- IncludeIf_1_Device_X_16
--                 |                   1      0    0    0        0   1   1      -- IncludeIf_1_Device_X_16
--None             |     XXXX          0      1    1    1        1   1   0      -- IncludeIf_1_Device_X_16
--                 |                   1      1    1    1        1   1   1      -- IncludeIf_1_Device_X_16
--Write word       |     1111          0      0    0    0        1   0   0      -- IncludeIf_1_Device_X_16
--                 |                   1      0    0    0        1   0   1      -- IncludeIf_1_Device_X_16
--Write low half   |     0011          0      1    1    1        1   0   0      -- IncludeIf_1_Device_X_16
--                 |                   1      0    0    0        1   0   1      -- IncludeIf_1_Device_X_16
--Write high half  |     1100          0      0    0    0        1   0   0      -- IncludeIf_1_Device_X_16
--                 |                   1      1    1    1        1   0   1      -- IncludeIf_1_Device_X_16
--Write Byte0      |     0001          0      0    1    0        1   0   0      -- IncludeIf_1_Device_X_16
--                 |                   1      1    1    1        1   0   1      -- IncludeIf_1_Device_X_16
--Write Byte1      |     0010          0      1    0    0        1   0   0      -- IncludeIf_1_Device_X_16
--                 |                   1      1    1    1        1   0   1      -- IncludeIf_1_Device_X_16
--Write Byte2      |     0100          0      1    1    1        1   0   0      -- IncludeIf_1_Device_X_16
--                 |                   1      0    1    0        1   0   1      -- IncludeIf_1_Device_X_16
--Write Byte3      |     1000          0      1    1    1        1   0   0      -- IncludeIf_1_Device_X_16
--                 |                   1      1    0    1        1   0   0      -- IncludeIf_1_Device_X_16
--Write Bytes012   |     0111          0      0    0    0        1   0          -- IncludeIf_1_Device_X_16
--                 |                   1      0    1    0        1   0          -- IncludeIf_1_Device_X_16
--Write Bytes123   |     1110          0      1    0    0        1   0          -- IncludeIf_1_Device_X_16
--                 |                   1      0    0    0        1   0          -- IncludeIf_1_Device_X_16
--Write Bytes12    |     0110          0      1    0    0        1   0          -- IncludeIf_1_Device_X_16
--                 |                   1      0    1    0        1   0          -- IncludeIf_1_Device_X_16
--------------------------------------------------------------------------------
Architecture RTL of ReplaceWith_Designator_Flash_x16 Is
    Type TState Is ( State_AddressSetup,
                     State_Write_Cycle0_Setup,
                     State_Write_Cycle0_Pulse,
                     State_Write_Cycle0_Hold,
                     State_Write_Cycle0_ACK,
                     State_Write_Cycle1_Setup,
                     State_Write_Cycle1_Pulse,
                     State_Write_Cycle1_ACK,

                     State_Read_Cycle0_Latch,
                     State_Read_Cycle1_Setup,
                     State_Read_Cycle1_ACK,
                     State_Read_Cycle0_ACK);

    Signal   State      : TState;
    Signal   State_Next : TState;

    SubType  TCycleAndByteLane   Is Std_Logic_Vector(4 DownTo 0);
    SubType  TOutBits            Is Std_Logic_Vector(2 DownTo 0);

    ----------------------------------------------------------------------------
    -- Constants for byte lanes - always the same
    ----------------------------------------------------------------------------
    Constant Write_Word_Cycle0           : TCycleAndByteLane := "01111";
    Constant Write_Word_Cycle1           : TCycleAndByteLane := "11111";

    Constant Write_LoHalf_Cycle0         : TCycleAndByteLane := "01100";
    Constant Write_LoHalf_Cycle1         : TCycleAndByteLane := "11100";
    Constant Write_HiHalf_Cycle0         : TCycleAndByteLane := "00011";
    Constant Write_HiHalf_Cycle1         : TCycleAndByteLane := "10011";

    Constant Write_Byte0_Cycle0          : TCycleAndByteLane := "01000";
    Constant Write_Byte0_Cycle1          : TCycleAndByteLane := "11000";
    Constant Write_Byte1_Cycle0          : TCycleAndByteLane := "00100";
    Constant Write_Byte1_Cycle1          : TCycleAndByteLane := "10100";
    Constant Write_Byte2_Cycle0          : TCycleAndByteLane := "00010";
    Constant Write_Byte2_Cycle1          : TCycleAndByteLane := "10010";
    Constant Write_Byte3_Cycle0          : TCycleAndByteLane := "00001";
    Constant Write_Byte3_Cycle1          : TCycleAndByteLane := "10001";
    Constant Write_Bytes012_Cycle0       : TCycleAndByteLane := "01110";
    Constant Write_Bytes012_Cycle1       : TCycleAndByteLane := "11110";
    Constant Write_Bytes123_Cycle0       : TCycleAndByteLane := "00111";
    Constant Write_Bytes123_Cycle1       : TCycleAndByteLane := "10111";
    Constant Write_Bytes12_Cycle0        : TCycleAndByteLane := "00110";
    Constant Write_Bytes12_Cycle1        : TCycleAndByteLane := "10110";
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    -- Mapping of byte lanes to SRAM Controls
    ----------------------------------------------------------------------------
    Constant OutBits_Write_Word            : TOutBits  := "000";

--    Constant OutBits_Write_Cycle0_LoHalf   : TOutBits  := "000";                  -- IncludeIf_2_Device_X_8
--    Constant OutBits_Write_Cycle1_LoHalf   : TOutBits  := "110";                  -- IncludeIf_2_Device_X_8
--    Constant OutBits_Write_Cycle0_HiHalf   : TOutBits  := "110";                  -- IncludeIf_2_Device_X_8
--    Constant OutBits_Write_Cycle1_HiHalf   : TOutBits  := "000";                  -- IncludeIf_2_Device_X_8
--    Constant OutBits_Write_Cycle0_Byte0    : TOutBits  := "010";                  -- IncludeIf_2_Device_X_8
--    Constant OutBits_Write_Cycle1_Byte0    : TOutBits  := "110";                  -- IncludeIf_2_Device_X_8
--    Constant OutBits_Write_Cycle0_Byte1    : TOutBits  := "100";                  -- IncludeIf_2_Device_X_8
--    Constant OutBits_Write_Cycle1_Byte1    : TOutBits  := "110";                  -- IncludeIf_2_Device_X_8
--    Constant OutBits_Write_Cycle0_Byte2    : TOutBits  := "110";                  -- IncludeIf_2_Device_X_8
--    Constant OutBits_Write_Cycle1_Byte2    : TOutBits  := "010";                  -- IncludeIf_2_Device_X_8
--    Constant OutBits_Write_Cycle0_Byte3    : TOutBits  := "110";                  -- IncludeIf_2_Device_X_8
--    Constant OutBits_Write_Cycle1_Byte3    : TOutBits  := "100";                  -- IncludeIf_2_Device_X_8
--
--    Constant OutBits_Write_Cycle0_Bytes012 : TOutBits  := "000";                  -- IncludeIf_2_Device_X_8
--    Constant OutBits_Write_Cycle1_Bytes012 : TOutBits  := "010";                  -- IncludeIf_2_Device_X_8
--    Constant OutBits_Write_Cycle0_Bytes123 : TOutBits  := "100";                  -- IncludeIf_2_Device_X_8
--    Constant OutBits_Write_Cycle1_Bytes123 : TOutBits  := "000";                  -- IncludeIf_2_Device_X_8
--    Constant OutBits_Write_Cycle0_Bytes12  : TOutBits  := "100";                  -- IncludeIf_2_Device_X_8
--    Constant OutBits_Write_Cycle1_Bytes12  : TOutBits  := "010";                  -- IncludeIf_2_Device_X_8
--
--
--                                                                                -- IncludeIf_2_Device_X_8
--    Constant OutBits_Read                : TOutBits  := "000";                  -- IncludeIf_2_Device_X_8
--    Constant OutBits_None                : TOutBits  := "111";                  -- IncludeIf_2_Device_X_8
--
    Constant OutBits_Write_Cycle0_LoHalf : TOutBits  := "000";                  -- IncludeIf_1_Device_X_16
    Constant OutBits_Write_Cycle1_LoHalf : TOutBits  := "111";                  -- IncludeIf_1_Device_X_16
    Constant OutBits_Write_Cycle0_HiHalf : TOutBits  := "111";                  -- IncludeIf_1_Device_X_16
    Constant OutBits_Write_Cycle1_HiHalf : TOutBits  := "000";                  -- IncludeIf_1_Device_X_16
    Constant OutBits_Write_Cycle0_Byte0  : TOutBits  := "010";                  -- IncludeIf_1_Device_X_16
    Constant OutBits_Write_Cycle1_Byte0  : TOutBits  := "111";                  -- IncludeIf_1_Device_X_16
    Constant OutBits_Write_Cycle0_Byte1  : TOutBits  := "100";                  -- IncludeIf_1_Device_X_16
    Constant OutBits_Write_Cycle1_Byte1  : TOutBits  := "111";                  -- IncludeIf_1_Device_X_16
    Constant OutBits_Write_Cycle0_Byte2  : TOutBits  := "111";                  -- IncludeIf_1_Device_X_16
    Constant OutBits_Write_Cycle1_Byte2  : TOutBits  := "010";                  -- IncludeIf_1_Device_X_16
    Constant OutBits_Write_Cycle0_Byte3  : TOutBits  := "111";                  -- IncludeIf_1_Device_X_16
    Constant OutBits_Write_Cycle1_Byte3  : TOutBits  := "100";                  -- IncludeIf_1_Device_X_16
    
    Constant OutBits_Write_Cycle0_Bytes012 : TOutBits := "000";                  -- IncludeIf_1_Device_X_16
    Constant OutBits_Write_Cycle1_Bytes012 : TOutBits := "010";                  -- IncludeIf_1_Device_X_16
    Constant OutBits_Write_Cycle0_Bytes123 : TOutBits := "100";                  -- IncludeIf_1_Device_X_16
    Constant OutBits_Write_Cycle1_Bytes123 : TOutBits := "000";                  -- IncludeIf_1_Device_X_16
    Constant OutBits_Write_Cycle0_Bytes12  : TOutBits := "100";                  -- IncludeIf_1_Device_X_16
    Constant OutBits_Write_Cycle1_Bytes12  : TOutBits := "010";                  -- IncludeIf_1_Device_X_16

                                                                                -- IncludeIf_1_Device_X_16
    Constant OutBits_Read                : TOutBits  := "000";                  -- IncludeIf_1_Device_X_16
    Constant OutBits_None                : TOutBits  := "111";                  -- IncludeIf_1_Device_X_16

    Signal   OutBits                     : TOutBits;
    Signal   OutBits_Write               : TOutBits;

    Signal   Request                     : Std_Logic;
    Signal   RequestRead                 : Std_Logic;
    Signal   RequestWrite                : Std_Logic;

    Signal   Cycle_Number                : Std_Logic;

    Signal   WriteActive_Cycle           : Std_Logic;
    Signal   WE                          : Std_Logic;
    Signal   OE                          : Std_Logic;
    Signal   ACK                         : Std_Logic;

    Signal   CycleAndSel                 : TCycleAndByteLane;

    Signal   OutData_Latched             : Std_Logic_Vector(15 Downto 0);
    --Signal   ADDR                        : Std_Logic_Vector(ReplaceWith_InputAddressWidth-3 Downto 0);

    Constant cTimer_Zero          : Std_Logic_Vector(4 Downto 0) := "00000";
    Constant cTimer_WriteSetup    : Std_Logic_Vector(4 Downto 0) := Conv_Std_Logic_Vector(ReplaceWith_FlashWriteSetup, 5) ;
    Constant cTimer_WritePulse    : Std_Logic_Vector(4 Downto 0) := Conv_Std_Logic_Vector(ReplaceWith_FlashWritePulse, 5) ;
    Constant cTimer_WriteHold     : Std_Logic_Vector(4 Downto 0) := Conv_Std_Logic_Vector(ReplaceWith_FlashWriteHold, 5) ;
    Constant cTimer_ReadSetup     : Std_Logic_Vector(4 Downto 0) := Conv_Std_Logic_Vector(ReplaceWith_FlashReadSetup, 5) ;
--    Constant cTimer_WriteSetup    : Std_Logic_Vector(4 Downto 0) := "01100" ;
--    Constant cTimer_WritePulse    : Std_Logic_Vector(4 Downto 0) := "01100" ;
--    Constant cTimer_WriteHold     : Std_Logic_Vector(4 Downto 0) := "01100" ;
--    Constant cTimer_ReadSetup     : Std_Logic_Vector(4 Downto 0) := "01100" ;

    Signal Timer                  : Std_Logic_Vector(4 Downto 0);
    Signal TimerIsZero            : Std_Logic;

    Signal LoadTimer_ReadSetup    : Std_Logic;
    Signal LoadTimer_WriteSetup   : Std_Logic;
    Signal LoadTimer_WritePulse   : Std_Logic;
    Signal LoadTimer_WriteHold    : Std_Logic;

    Signal SkipCycle0 : Std_Logic;
    Signal SkipCycle1 : Std_Logic;
    Signal WbAcknowledge : Std_Logic;


Begin

     Process(CLK_I)
     Begin
          If Rising_Edge(CLK_I) Then
              FLASH_CE   <= OutBits(0) Or (Not Request);
              ----------------------------------------------------------------------------
              FLASH_A(23 DownTo 1) <= ADR_I(22 DownTo 0); -- IncludeIf_1_Device_X_16
              FLASH_A(0)           <= Cycle_Number;
              ----------------------------------------------------------------------------
              ----------------------------------------------------------------------------
              FLASH_WE <= WE;                                                              -- IncludeIf_1_Device_X_16
              FLASH_OE <= OE;                                                              -- IncludeIf_1_Device_X_16
              ----------------------------------------------------------------------------
              ----------------------------------------------------------------------------
              Request      <= STB_I And CYC_I And Not ACK And Not WbAcknowledge;
              RequestRead  <= STB_I And CYC_I And (Not WE_I) And Not ACK And Not WbAcknowledge;
              RequestWrite <= STB_I And CYC_I And (    WE_I) And Not ACK And Not WbAcknowledge;

              FLASH_D_O(31 Downto 16) <= (Others=>'0');
              If WriteActive_Cycle='1' Then
                 FLASH_D_E <= '1';
                 If Cycle_Number='1' Then
                    FLASH_D_O(15 Downto 0) <= DAT_I(15 Downto 0);
                 Else
                     FLASH_D_O(15 Downto 0) <= DAT_I(31 Downto 16);
                 End If;
              Else
                  FLASH_D_E <= '0';
              End If;
              WbAcknowledge <= ACK;
              ----------------------------------------------------------------------------
          End If;
     End Process;

--     FLASH_D_O  : Out Std_Logic_Vector(31 DownTo 0);                             -- IncludeIf_1_Device_X_16
--     FLASH_D_E  : Out Std_Logic;

    ----------------------------------------------------------------------------


    ----------------------------------------------------------------------------
    DAT_O(15 DownTo  0) <= FLASH_D_I(15 Downto 0);                                              -- IncludeIf_1_Device_X_16
    DAT_O(31 DownTo 16) <= OutData_Latched;                                      -- IncludeIf_1_Device_X_16
    ----------------------------------------------------------------------------

--    ----------------------------------------------------------------------------
--    ACK_O <= ACK;
      ACK_O <= WbAcknowledge;
--    ----------------------------------------------------------------------------



    ----------------------------------------------------------------------------
    CycleAndSel <= Cycle_Number & SEL_I;
    SkipCycle1  <= Not SEL_I(1) And Not SEL_I(0);
    SkipCycle0  <= Not SEL_I(3) And Not SEL_I(2);
    ----------------------------------------------------------------------------
    ----------------------------------------------------------------------------
    Memory_OuptutControls:
    Process(SEL_I,
            DAT_I,
            State,
            OutBits_Write,
            WriteActive_Cycle,
            Cycle_Number,
            CycleAndSel)
    Begin
        If WriteActive_Cycle = '1' Then
            OutBits    <= OutBits_Write;
            OE         <= '1';
        Else
           OutBits    <= OutBits_Read;
           OE         <= '0';
        End If;

        Case CycleAndSel Is
           When Write_Word_Cycle0      => OutBits_Write <= OutBits_Write_Word;
           When Write_Word_Cycle1      => OutBits_Write <= OutBits_Write_Word;
           When Write_LoHalf_Cycle0    => OutBits_Write <= OutBits_Write_Cycle0_LoHalf;
           When Write_LoHalf_Cycle1    => OutBits_Write <= OutBits_Write_Cycle1_LoHalf;
           When Write_HiHalf_Cycle0    => OutBits_Write <= OutBits_Write_Cycle0_HiHalf;
           When Write_HiHalf_Cycle1    => OutBits_Write <= OutBits_Write_Cycle1_HiHalf;
           When Write_Byte0_Cycle0     => OutBits_Write <= OutBits_Write_Cycle0_Byte0;
           When Write_Byte0_Cycle1     => OutBits_Write <= OutBits_Write_Cycle1_Byte0;
           When Write_Byte1_Cycle0     => OutBits_Write <= OutBits_Write_Cycle0_Byte1;
           When Write_Byte1_Cycle1     => OutBits_Write <= OutBits_Write_Cycle1_Byte1;
           When Write_Byte2_Cycle0     => OutBits_Write <= OutBits_Write_Cycle0_Byte2;
           When Write_Byte2_Cycle1     => OutBits_Write <= OutBits_Write_Cycle1_Byte2;
           When Write_Byte3_Cycle0     => OutBits_Write <= OutBits_Write_Cycle0_Byte3;
           When Write_Byte3_Cycle1     => OutBits_Write <= OutBits_Write_Cycle1_Byte3;
           When Write_Bytes012_Cycle0  => OutBits_Write <= OutBits_Write_Cycle0_Bytes012;
           When Write_Bytes012_Cycle1  => OutBits_Write <= OutBits_Write_Cycle1_Bytes012;
           When Write_Bytes123_Cycle0  => OutBits_Write <= OutBits_Write_Cycle0_Bytes123;
           When Write_Bytes123_Cycle1  => OutBits_Write <= OutBits_Write_Cycle1_Bytes123;
           When Write_Bytes12_Cycle0   => OutBits_Write <= OutBits_Write_Cycle0_Bytes12;
           When Write_Bytes12_Cycle1   => OutBits_Write <= OutBits_Write_Cycle1_Bytes12;
           When Others                 => OutBits_Write <= OutBits_None;
        End Case;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    FSM_Combinational_Single:
    Process(RequestRead,
            RequestWrite,
            State,
            TimerIsZero,
            SkipCycle0,
            SkipCycle1)
    Begin
        ------------------------------------------------------------------------
        -- Default outputs
        WriteActive_Cycle <= '0';
        WE                <= '1';
        ACK               <= '0';
        Cycle_Number      <= '0';
        State_Next        <= State;
        LoadTimer_ReadSetup  <= '0';
        LoadTimer_WriteSetup <= '0';
        LoadTimer_WritePulse <= '0';
        LoadTimer_WriteHold  <= '0';

        ------------------------------------------------------------------------

        ------------------------------------------------------------------------
        Case State Is
           ---------------------------------------------------------------------
           When State_AddressSetup =>
                If RequestWrite = '1' Then                --Address is being setup in this phases
                   WriteActive_Cycle <= '1';
                   If SkipCycle0='1' Then
                        State_Next        <= State_Write_Cycle1_Setup;
                   Else
                        State_Next        <= State_Write_Cycle0_Setup;
                   End If;
                   LoadTimer_WriteSetup <= '1';
                 ElsIf RequestRead = '1' Then
                    If SkipCycle0='1' Then
                        State_Next <= State_Read_Cycle1_Setup;
                    Else
                       State_Next <= State_Read_Cycle0_Latch;
                    End If;
                   LoadTimer_ReadSetup <= '1';
                End If;
           ---------------------------------------------------------------------

           ---------------------------------------------------------------------
           When State_Write_Cycle0_Setup =>
                WriteActive_Cycle <= '1';
                If TimerIsZero='1' Then
                    State_Next <= State_Write_Cycle0_Pulse;
                    LoadTimer_WritePulse <= '1';
                End If;

           When State_Write_Cycle0_Pulse =>
                WriteActive_Cycle <= '1';
                WE                <= '0';
                If TimerIsZero='1' Then
                    If SkipCycle1='1' Then
                        State_Next <= State_Write_Cycle0_ACK;
                    Else
                        State_Next        <= State_Write_Cycle0_Hold;
                    End If;
                    LoadTimer_WriteHold <= '1';
                End If;

           When State_Write_Cycle0_Hold =>
                WriteActive_Cycle <= '1';
                If TimerIsZero='1' Then
                    State_Next        <= State_Write_Cycle1_Setup;
                    LoadTimer_WriteSetup <= '1';
                End If;

           When State_Write_Cycle1_Setup =>
                WriteActive_Cycle <= '1';
                Cycle_Number      <= '1';
                If TimerIsZero='1' Then
                    State_Next        <= State_Write_cycle1_Pulse;
                    LoadTimer_WritePulse <= '1';
                End If;

           When State_Write_Cycle1_Pulse =>
                WriteActive_Cycle <= '1';
                WE                <= '0';
                Cycle_Number      <= '1';
                If TimerIsZero='1' Then
                    State_Next        <= State_Write_Cycle1_ACK;
                    LoadTimer_WriteHold <= '1';
                End If;

           When State_Write_Cycle0_ACK =>
                WriteActive_Cycle <= '1';
                If TimerIsZero='1' Then
                    ACK <= '1';
                    State_Next <= State_AddressSetup;
                End If;

           When State_Write_Cycle1_ACK =>
                WriteActive_Cycle <= '1';
                Cycle_Number      <= '1';
                If TimerIsZero='1' Then
                    ACK               <= '1';
                    State_Next        <= State_AddressSetup;
                End If;
           ---------------------------------------------------------------------

           ---------------------------------------------------------------------
           When State_Read_Cycle0_Latch =>
                If TimerIsZero='1' Then
                    If SkipCycle1='1' Then
                        State_Next <= State_Read_Cycle0_ACK;
                    Else
                        State_Next    <= State_Read_Cycle1_Setup;
                    End If;
                    LoadTimer_ReadSetup <= '1';
                End If;

           When State_Read_Cycle1_Setup =>
                Cycle_Number  <= '1';
                If TimerIsZero='1' Then
                    State_Next    <= State_Read_Cycle1_ACK;
                End If;
           When State_Read_Cycle0_ACK =>
                ACK <= '1';
                State_Next <= State_AddressSetup;

           When State_Read_Cycle1_ACK =>                    -- Assert ACK, Master will read the data one clock cycle later
                ACK           <= '1';
                Cycle_Number  <= '1';
                State_Next    <= State_AddressSetup;
           ---------------------------------------------------------------------

           When Others =>
                State_Next <= State_AddressSetup;

        End Case;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    FSM_Synchronous:
    Process(CLK_I,RST_I,State_Next)
    Begin
        If Rising_Edge(CLK_I) Then
           If RST_I = '1' Then
              State  <= State_AddressSetup;
           Else
              State <= State_Next;
           End If;
        End If;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    FSM_LatchData:
    Process(CLK_I,State,FLASH_D_I)                                                 -- IncludeIf_1_Device_X_16
    Begin
        If Rising_Edge(CLK_I) Then
           If State = State_Read_Cycle0_Latch Then
               OutData_Latched <= FLASH_D_I(15 Downto 0);                                       -- IncludeIf_1_Device_X_16
           End If;
        End If;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    Process(CLK_I)
    Begin
        If Rising_Edge(CLK_I) Then
            If RST_I='1' Then
                Timer <= (Others=>'0');
            Else
                   If LoadTimer_ReadSetup ='1' Then Timer <= cTimer_ReadSetup;
                ElsIf LoadTimer_WriteSetup='1' Then Timer <= cTimer_WriteSetup;
                ElsIf LoadTimer_WritePulse='1' Then Timer <= cTimer_WritePulse;
                ElsIf LoadTimer_WriteHold ='1' Then Timer <= cTimer_WriteHold;
                ElsIf TimerIsZero         ='0' Then Timer <= Timer - 1;
                End If;
            End If;
        End If;
    End Process;
    TimerIsZero <= '1' When Timer=cTimer_Zero Else '0';
    ----------------------------------------------------------------------------


End RTL;
--------------------------------------------------------------------------------

Library IEEE;
Use IEEE.Std_Logic_1164.all;
Use IEEE.std_logic_unsigned.all;
Use IEEE.Std_Logic_Arith.All;
Use Work.Configurable_ReplaceWith_Designator_GlobalConfig.All;
--------------------------------------------------------------------------------
Entity ReplaceWith_Designator_Sram_x32 Is Port
   (
     SRAM_D_O  : Out   Std_Logic_Vector(31 Downto 0);
     SRAM_D_I  : In    Std_Logic_Vector(31 Downto 0);
     SRAM_D_E  : Out   Std_Logic;
     SRAM_A    : Out   Std_Logic_Vector(23 DownTo 0);
     SRAM_OE   : Out   Std_Logic;
     SRAM_WE   : Out   Std_Logic;
     SRAM_CE   : Out   Std_Logic;
     SRAM_BE_E : Out   Std_Logic_Vector(3 Downto 0);
     ---------------------------------------------------------------------------

     ---------------------------------------------------------------------------
     -- Wishbone Slave Port
     ---------------------------------------------------------------------------
     CLK_I      : In    Std_Logic;
     RST_I      : In    Std_Logic;
     CYC_I      : In    Std_Logic;
     STB_I      : In    Std_Logic;
     ACK_O      : Out   Std_Logic;
     ADR_I      : In    Std_Logic_Vector(23 DownTo 0);
     DAT_I      : In    Std_Logic_Vector(31 DownTo 0);
     DAT_O      : Out   Std_Logic_Vector(31 DownTo 0);
     WE_I       : In    Std_Logic;
     SEL_I      : In    Std_Logic_Vector(3 DownTo 0)
   );

End ReplaceWith_Designator_Sram_x32;
--------------------------------------------------------------------------------

--------------------------------------------------------------------------------
Architecture RTL of ReplaceWith_Designator_Sram_x32 Is
    Type TState Is
        (
         State_AddressSetup  ,
         State_Write_Setup,
         State_Write_Pulse   ,     -- WE goes low during this state
         State_Write_Hold,
         State_Write_ACK     ,     -- WE goes high. Must keep seperate form addresses etc in case WE goes high before address, CE etc
         State_Read_ACK
        );

    Signal   State      : TState;
    Signal   State_Next : TState;

    SubType  TByteLane   Is Std_Logic_Vector(3 DownTo 0);
    SubType  TOutBits    Is Std_Logic_Vector(4 DownTo 0);

    ----------------------------------------------------------------------------
    -- Constants for byte lanes - always the same
    ----------------------------------------------------------------------------
    Constant Write_Word           : TByteLane := "1111";
    Constant Write_LoHalf         : TByteLane := "0011";
    Constant Write_HiHalf         : TByteLane := "1100";
    Constant Write_Byte0          : TByteLane := "0001";
    Constant Write_Byte1          : TByteLane := "0010";
    Constant Write_Byte2          : TByteLane := "0100";
    Constant Write_Byte3          : TByteLane := "1000";
    Constant Write_Bytes012       : TByteLane := "0111";
    Constant Write_Bytes123       : TByteLane := "1110";
    Constant Write_Bytes12        : TByteLane := "0110";
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    Constant OutBits_Write_Word     : TOutBits  := "00000";

    Constant OutBits_Write_LoHalf   : TOutBits  := "11000";
    Constant OutBits_Write_HiHalf   : TOutBits  := "00110";
    Constant OutBits_Write_Byte0    : TOutBits  := "11100";
    Constant OutBits_Write_Byte1    : TOutBits  := "11010";
    Constant OutBits_Write_Byte2    : TOutBits  := "10110";
    Constant OutBits_Write_Byte3    : TOutBits  := "01110";
    Constant OutBits_Write_Bytes012 : TOutBits  := "10000";
    Constant OutBits_Write_Bytes123 : TOutBits  := "00010";
    Constant OutBits_Write_Bytes12  : TOutBits  := "10010";

    Constant OutBits_Read           : TOutBits  := "00000";
    Constant OutBits_None           : TOutBits  := "11111";

    Signal   OutBits              : TOutBits;
    Signal   OutBits_Write        : TOutBits;

    Signal   Request              : Std_Logic;
    Signal   RequestRead          : Std_Logic;
    Signal   RequestWrite         : Std_Logic;

    Signal   WriteActive_Cycle    : Std_Logic;
    Signal   WE                   : Std_Logic;
    Signal   OE                   : Std_Logic;
    Signal   ACK                  : Std_Logic;

    Constant cTimer_Zero            : Std_Logic_Vector(4 Downto 0) := "00000";
    Constant cTimer_WriteSetup      : Std_Logic_Vector(4 Downto 0) := Conv_Std_Logic_Vector(ReplaceWith_SramWriteSetup + 1, 5) ;
    Constant cTimer_WritePulse      : Std_Logic_Vector(4 Downto 0) := Conv_Std_Logic_Vector(ReplaceWith_SramWritePulse, 5) ;
    Constant cTimer_WriteHold       : Std_Logic_Vector(4 Downto 0) := Conv_Std_Logic_Vector(ReplaceWith_SramWriteHold, 5) ;
    Constant cTimer_ReadSetup       : Std_Logic_Vector(4 Downto 0) := Conv_Std_Logic_Vector(ReplaceWith_SramReadSetup + 1, 5) ;

    Signal Timer                  : Std_Logic_Vector(4 Downto 0);
    Signal TimerIsZero            : Std_Logic;

    Signal LoadTimer_ReadSetup    : Std_Logic;
    Signal LoadTimer_WriteSetup   : Std_Logic;
    Signal LoadTimer_WritePulse   : Std_Logic;
    Signal LoadTimer_WriteHold    : Std_Logic;
    Signal SRAM_DataEnable : Std_Logic;
    Signal WbAcknowledge : Std_Logic;


Begin

     Process(CLK_I)
     Begin
          If Rising_Edge(CLK_I) Then
              ----------------------------------------------------------------------------
              SRAM_BE_E <= OutBits(4) & OutBits(3) & OutBits(2) & OutBits(1);
              ----------------------------------------------------------------------------
              ----------------------------------------------------------------------------
              SRAM_CE   <= OutBits(0) Or (Not Request);
              ----------------------------------------------------------------------------
              ----------------------------------------------------------------------------
              SRAM_A(23 Downto 1)    <= "000" & ADR_I(19 DownTo 0);
              SRAM_A(0) <= '0';
              ----------------------------------------------------------------------------
              ----------------------------------------------------------------------------
              SRAM_WE <= WE;
              SRAM_OE <= OE;
              ----------------------------------------------------------------------------
              SRAM_D_E <= SRAM_DataEnable;
              SRAM_D_O <= DAT_I;
              Request      <= STB_I And CYC_I And Not ACK And Not WbAcknowledge;
              RequestRead  <= STB_I And CYC_I And (Not WE_I) And Not ACK And Not WbAcknowledge;
              RequestWrite <= STB_I And CYC_I And (    WE_I) And Not ACK And Not WbAcknowledge;
              WbAcknowledge <= ACK;
          End If;
     End Process;

    ----------------------------------------------------------------------------
    DAT_O <= SRAM_D_I;
    ----------------------------------------------------------------------------

--    ----------------------------------------------------------------------------
--    ACK_O <= ACK;
      ACK_O <= WbAcknowledge;
--    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    ----------------------------------------------------------------------------


    ----------------------------------------------------------------------------
    Memory_OuptutControls:
    Process(SEL_I,
            DAT_I,
            State,
            OutBits_Write,
            WriteActive_Cycle)
    Begin
        If WriteActive_Cycle = '1' Then
           OutBits   <= OutBits_Write;
           OE        <= '1';
           SRAM_DataEnable <= '1';
        Else
           OutBits   <= OutBits_Read;
           OE        <= '0';
           SRAM_DataEnable <= '0';
        End If;

        Case SEL_I Is
           When Write_Word     => OutBits_Write <= OutBits_Write_Word;
           When Write_LoHalf   => OutBits_Write <= OutBits_Write_LoHalf;
           When Write_HiHalf   => OutBits_Write <= OutBits_Write_HiHalf;
           When Write_Byte0    => OutBits_Write <= OutBits_Write_Byte0;
           When Write_Byte1    => OutBits_Write <= OutBits_Write_Byte1;
           When Write_Byte2    => OutBits_Write <= OutBits_Write_Byte2;
           When Write_Byte3    => OutBits_Write <= OutBits_Write_Byte3;
           When Write_Bytes012 => OutBits_Write <= OutBits_Write_Bytes012;
           When Write_Bytes123 => OutBits_Write <= OutBits_Write_Bytes123;
           When Write_Bytes12  => OutBits_Write <= OutBits_Write_Bytes12;
           When Others         => OutBits_Write <= OutBits_None;
        End Case;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    FSM_Combinational_Single:
    Process(RequestRead,
            RequestWrite,
            State,
            TimerIsZero)
    Begin
        ------------------------------------------------------------------------
        -- Default outputs
        WriteActive_Cycle <= '0';
        WE                <= '1';
        ACK               <= '0';
        State_Next        <= State;
        LoadTimer_ReadSetup  <= '0';
        LoadTimer_WriteSetup <= '0';
        LoadTimer_WritePulse <= '0';
        LoadTimer_WriteHold  <= '0';
        ------------------------------------------------------------------------

        ------------------------------------------------------------------------
        Case State Is
           ---------------------------------------------------------------------
           When State_AddressSetup =>
                If RequestWrite = '1' Then                --Address is being setup in this phases
                   WriteActive_Cycle <= '1';
                   State_Next        <= State_Write_Setup;                       -- IncludeIf_WriteSetup
                   LoadTimer_WriteSetup <= '1';                                  -- IncludeIf_WriteSetup
                 ElsIf RequestRead = '1' Then
                   LoadTimer_ReadSetup <= '1';                                   -- IncludeIf_ReadSetup
                   State_Next <= State_Read_ACK;
                End If;
           ---------------------------------------------------------------------

           ---------------------------------------------------------------------
           When State_Write_Setup =>
                WriteActive_Cycle <= '1';
                If TimerIsZero='1' Then
                    State_Next <= State_Write_Pulse;
                    LoadTimer_WritePulse <= '1';                                 -- IncludeIf_WritePulse
                End If;

           When State_Write_Pulse =>
                WriteActive_Cycle <= '1';
                WE                <= '0';
                If TimerIsZero='1' Then                                           -- IncludeIf_WriteHold
                    State_Next        <= State_Write_Hold;                        -- IncludeIf_WriteHold
                    LoadTimer_WriteHold <= '1';                                   -- IncludeIf_WriteHold
                End If;                                                           -- IncludeIf_WriteHold

           When State_Write_Hold =>
                WriteActive_Cycle <= '1';
                If TimerIsZero='1' Then
                    State_Next  <= State_Write_ACK;
                End If;


           When State_Write_ACK =>
                WriteActive_Cycle <= '1';
                ACK               <= '1';
                State_Next        <= State_AddressSetup;
           ---------------------------------------------------------------------

           ---------------------------------------------------------------------
           When State_Read_ACK =>                    -- Assert ACK, Master will read the data one clock cycle later
                If TimerIsZero='1' Then                                            -- IncludeIf_ReadSetup
                    ACK           <= '1';
                    State_Next    <= State_AddressSetup;
                End If;                                                            -- IncludeIf_ReadSetup
           ---------------------------------------------------------------------

           When Others =>
                State_Next <= State_AddressSetup;

        End Case;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    FSM_Synchronous:
    Process(CLK_I)
    Begin
        If Rising_Edge(CLK_I) Then
           If RST_I = '1' Then
              State  <= State_AddressSetup;
           Else
              State <= State_Next;
           End If;
        End If;
    End Process;
    ----------------------------------------------------------------------------
    ----------------------------------------------------------------------------
    Process(CLK_I)
    Begin
        If Rising_Edge(CLK_I) Then
            If RST_I='1' Then
                Timer <= (Others=>'0');
            Else
                   If LoadTimer_ReadSetup ='1' Then Timer <= cTimer_ReadSetup;
                ElsIf LoadTimer_WritePulse='1' Then Timer <= cTimer_WritePulse;
                ElsIf LoadTimer_WriteSetup='1' Then Timer <= cTimer_WriteSetup;
                ElsIf LoadTimer_WriteHold ='1' Then Timer <= cTimer_WriteHold;
                ElsIf TimerIsZero         ='0' Then Timer <= Timer - 1;
                End If;
            End If;
        End If;
    End Process;
    TimerIsZero <= '1' When Timer=cTimer_Zero Else '0';
    ----------------------------------------------------------------------------

End RTL;
--------------------------------------------------------------------------------
Library IEEE;
Use IEEE.Std_Logic_1164.all;
Use IEEE.std_logic_unsigned.all;
Use IEEE.Std_Logic_Arith.All;
Use Work.Configurable_ReplaceWith_Designator_GlobalConfig.All;

--------------------------------------------------------------------------------
Entity ReplaceWith_Designator_USB_X32 Is Port
   (

     MEM_D_O       : Out   Std_Logic_Vector(31 Downto 0);                          -- IncludeIf_Shared
     MEM_D_I       : In    Std_Logic_Vector(31 Downto 0);                          -- IncludeIf_Shared
     MEM_D_E       : Out   Std_Logic;                                              -- IncludeIf_Shared

     MEM_A         : Out   Std_Logic_Vector(23 Downto 0);                          -- IncludeIf_FullAddress
     MEM_OE        : Out   Std_Logic;
     MEM_W         : Out   Std_Logic;
     MEM_BE        : Out   Std_Logic_Vector(3 Downto 0);
     MEM_USB_E     : Out   Std_Logic;
     MEM_USB_RESET : Out   Std_Logic;
     ---------------------------------------------------------------------------

     ---------------------------------------------------------------------------
     -- Wishbone Slave Port
     ---------------------------------------------------------------------------
     USB_CLK_I      : In    Std_Logic;
     USB_RST_I      : In    Std_Logic;
     USB_CYC_I      : In    Std_Logic;
     USB_STB_I      : In    Std_Logic;
     USB_ACK_O      : Out   Std_Logic;
     USB_ADR_I      : In    Std_Logic_Vector(23 DownTo 0);
     USB_DAT_I      : In    Std_Logic_Vector(31 DownTo 0);
     USB_DAT_O      : Out   Std_Logic_Vector(31 DownTo 0);
     USB_WE_I       : In    Std_Logic;
     USB_SEL_I      : In    Std_Logic_Vector(3 DownTo 0)
   );

End ReplaceWith_Designator_USB_X32;
--------------------------------------------------------------------------------

--------------------------------------------------------------------------------
Architecture RTL of ReplaceWith_Designator_USB_X32 Is
    Type TState Is
        (
         State_AddressSetup  ,

         State_Write_Pulse   ,     -- WE goes low during this state
         State_Write_ACK     ,     -- WE goes high. Must keep seperate form addresses etc in case WE goes high before address, CE etc

         State_Read_ACK
        );

    Signal   State      : TState;
    Signal   State_Next : TState;

    SubType  TByteLane   Is Std_Logic_Vector(3 DownTo 0);
    SubType  TOutBits    Is Std_Logic_Vector(4 DownTo 0);

    ----------------------------------------------------------------------------
    -- Constants for byte lanes - always the same
    ----------------------------------------------------------------------------
    Constant Write_Word           : TByteLane := "1111";
    Constant Write_LoHalf         : TByteLane := "0011";
    Constant Write_HiHalf         : TByteLane := "1100";
    Constant Write_Byte0          : TByteLane := "0001";
    Constant Write_Byte1          : TByteLane := "0010";
    Constant Write_Byte2          : TByteLane := "0100";
    Constant Write_Byte3          : TByteLane := "1000";
    Constant Write_Bytes012       : TByteLane := "0111";
    Constant Write_Bytes123       : TByteLane := "1110";
    Constant Write_Bytes12        : TByteLane := "0110";
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    Constant OutBits_Write_Word     : TOutBits  := "00000";

    Constant OutBits_Write_LoHalf   : TOutBits  := "11000";
    Constant OutBits_Write_HiHalf   : TOutBits  := "00110";
    Constant OutBits_Write_Byte0    : TOutBits  := "11100";
    Constant OutBits_Write_Byte1    : TOutBits  := "11010";
    Constant OutBits_Write_Byte2    : TOutBits  := "10110";
    Constant OutBits_Write_Byte3    : TOutBits  := "01110";
    Constant OutBits_Write_Bytes012 : TOutBits  := "10000";
    Constant OutBits_Write_Bytes123 : TOutBits  := "00010";
    Constant OutBits_Write_Bytes12  : TOutBits  := "10010";

    Constant OutBits_Read           : TOutBits  := "00000";
    Constant OutBits_None           : TOutBits  := "11111";

    Signal   OutBits              : TOutBits;
    Signal   OutBits_Write        : TOutBits;

    Signal   Request              : Std_Logic;
    Signal   RequestRead          : Std_Logic;
    Signal   RequestWrite         : Std_Logic;

    Signal   WriteActive_Cycle    : Std_Logic;
    Signal   WE                   : Std_Logic;
    Signal   OE                   : Std_Logic;
    Signal   ACK                  : Std_Logic;
    Signal   ADDR                 : Std_Logic_Vector(19-3 Downto 0);

    Constant cAddressSetupDelay : Std_Logic_Vector(5-1 Downto 0) := Conv_Std_Logic_Vector(ReplaceWith_UsbAddressSetup + 1, 5);
    Constant cWritePulseDelay   : Std_Logic_Vector(5-1 Downto 0) := Conv_Std_Logic_Vector(ReplaceWith_UsbWritePulse, 5);
    Constant cWriteHoldDelay    : Std_Logic_Vector(5-1 Downto 0) := Conv_Std_Logic_Vector(ReplaceWith_UsbWriteHold, 5);
    Constant cTimerIsZero       : Std_Logic_Vector(5-1 Downto 0) := Conv_Std_Logic_Vector(0, 5);

    Signal  Timer               : Std_Logic_Vector(5-1 Downto 0);
    Signal  TimerIsZero         : Std_Logic;
    Signal  LoadAddressDelay    : Std_Logic;
    Signal  LoadWritePulseDelay : Std_Logic;
    Signal  LoadWriteHoldDelay  : Std_Logic;
    Signal WbAcknowledge : Std_Logic;
Begin

     Process(USB_CLK_I)
     Begin
          If Rising_Edge(USB_CLK_I) Then
              ----------------------------------------------------------------------------
              MEM_BE <= OutBits(4) & OutBits(3) & OutBits(2) & OutBits(1);
              ----------------------------------------------------------------------------
              MEM_USB_E   <= OutBits(0) Or (Not Request);
              ----------------------------------------------------------------------------
              MEM_A    <= "000000" &  ADDR(17-1 Downto 0) & '0'; -- IncludeIf_FullAddress
              ----------------------------------------------------------------------------
              ----------------------------------------------------------------------------
              Request      <= USB_STB_I And USB_CYC_I And Not ACK And Not WbAcknowledge;
              RequestRead  <= USB_STB_I And USB_CYC_I And (Not USB_WE_I) And Not ACK And Not WbAcknowledge;
              RequestWrite <= USB_STB_I And USB_CYC_I And (    USB_WE_I) And Not ACK And Not WbAcknowledge;
              ----------------------------------------------------------------------------

              MEM_OE <= OE;
              MEM_W  <= WE;
              If WriteActive_Cycle='1' Then
                  MEM_D_O   <= USB_DAT_I; -- IncludeIf_Shared
                  MEM_D_E   <= '1';        -- IncludeIf_Shared
              Else
                  MEM_D_O   <= (Others=>'0'); -- IncludeIf_Shared
                  MEM_D_E   <= '0';             -- IncludeIf_Shared
              End If;
              WbAcknowledge <= ACK;
          End If;
     End Process;
     USB_ACK_O <= WbAcknowledge;
     MEM_USB_RESET <= Not USB_RST_I;


    ----------------------------------------------------------------------------
    Process(USB_CLK_I)
    Begin
        If Rising_Edge(USB_CLK_I) Then
            If USB_RST_I='1' Then
                Timer <= (Others=>'0');
            Else
                   If LoadAddressDelay='1'    Then Timer <= cAddressSetupDelay;
                ElsIf LoadWritePulseDelay='1' Then Timer <= cWritePulseDelay;
                ElsIf LoadWriteHoldDelay='1'  Then Timer <= cWriteHoldDelay;
                ElsIf TimerIsZero='0'         Then Timer <= Timer - 1;
                End If;
            End If;
        End If;
    End Process;
    TimerIsZero <= '1' When Timer=cTimerIsZero Else '0';
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    ADDR <= USB_ADR_I(16 DownTo 0);
    ----------------------------------------------------------------------------


    ----------------------------------------------------------------------------

    USB_DAT_O <= MEM_D_I; -- IncludeIf_Shared
    ----------------------------------------------------------------------------


    OE <= '0' When ((RequestRead='1') And (State=State_Read_ACK)) Else '1';

    ----------------------------------------------------------------------------
    Memory_OuptutControls:
    Process(USB_SEL_I,
            USB_DAT_I,
            State,
            OutBits_Write,
            WriteActive_Cycle)
    Begin
        If WriteActive_Cycle = '1' Then
           OutBits   <= OutBits_Write;
        Else
           OutBits   <= OutBits_Read;
        End If;

        Case USB_SEL_I Is
           When Write_Word     => OutBits_Write <= OutBits_Write_Word;
           When Write_LoHalf   => OutBits_Write <= OutBits_Write_LoHalf;
           When Write_HiHalf   => OutBits_Write <= OutBits_Write_HiHalf;
           When Write_Byte0    => OutBits_Write <= OutBits_Write_Byte0;
           When Write_Byte1    => OutBits_Write <= OutBits_Write_Byte1;
           When Write_Byte2    => OutBits_Write <= OutBits_Write_Byte2;
           When Write_Byte3    => OutBits_Write <= OutBits_Write_Byte3;
           When Write_Bytes012 => OutBits_Write <= OutBits_Write_Bytes012;
           When Write_Bytes123 => OutBits_Write <= OutBits_Write_Bytes123;
           When Write_Bytes12  => OutBits_Write <= OutBits_Write_Bytes12;
           When Others         => OutBits_Write <= OutBits_None;
        End Case;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    FSM_Combinational_Single:
    Process(RequestRead,
            RequestWrite,
            State,
            TimerIsZero)
    Begin
        ------------------------------------------------------------------------
        -- Default outputs
        WriteActive_Cycle <= '0';
        WE                <= '1';
        ACK               <= '0';
        State_Next        <= State;
        LoadAddressDelay    <= '0';
        LoadWritePulseDelay <= '0';
        LoadWriteHoldDelay  <= '0';

        ------------------------------------------------------------------------

        ------------------------------------------------------------------------
        Case State Is
           ---------------------------------------------------------------------
           When State_AddressSetup =>
                If RequestWrite = '1' Then                --Address is being setup in this phases
                    WriteActive_Cycle <= '1';
                    State_Next        <= State_Write_Pulse;
                    LoadWritePulseDelay <= '1';
                ElsIf RequestRead = '1' Then
                    State_Next <= State_Read_ACK;
                    LoadAddressDelay <= '1';
                End If;
           ---------------------------------------------------------------------

           ---------------------------------------------------------------------
           When State_Write_Pulse =>
                WriteActive_Cycle <= '1';
                WE                <= '0';
                If TimerIsZero='1' Then
                    State_Next        <= State_Write_ACK;
                    LoadWriteHoldDelay <= '1';
                End If;

           When State_Write_ACK =>
                WriteActive_Cycle <= '1';
                If TimerIsZero='1' Then
                    ACK               <= '1';
                    State_Next        <= State_AddressSetup;
                End if;
           ---------------------------------------------------------------------

           ---------------------------------------------------------------------
           When State_Read_ACK =>                    -- Assert ACK, Master will read the data one clock cycle later
                If TimerIsZero='1' Then
                    ACK           <= '1';
                    State_Next    <= State_AddressSetup;
                End If;
           ---------------------------------------------------------------------

           When Others =>
                State_Next <= State_AddressSetup;

        End Case;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    FSM_Synchronous:
    Process(USB_CLK_I,USB_RST_I,State_Next)
    Begin
        If Rising_Edge(USB_CLK_I) Then
           If USB_RST_I = '1' Then
              State  <= State_AddressSetup;
           Else
              State <= State_Next;
           End If;
        End If;
    End Process;
    ----------------------------------------------------------------------------


End RTL;
--------------------------------------------------------------------------------
