Library IEEE;
Use IEEE.Std_Logic_1164.all;
Use IEEE.std_logic_unsigned.all;

--------------------------------------------------------------------------------
Entity ReplaceWith_Designator_USB_X32 Is Port
   (

     MEM_D_O       : Out   Std_Logic_Vector(31 Downto 0);                          -- IncludeIf_Shared
     MEM_D_I       : In    Std_Logic_Vector(31 Downto 0);                          -- IncludeIf_Shared
     MEM_D_E       : Out   Std_Logic;                                              -- IncludeIf_Shared

     MEM_A         : Out   Std_Logic_Vector(23 Downto 0);                          -- IncludeIf_FullAddress
     MEM_OE        : Out   Std_Logic;
     MEM_W         : Out   Std_Logic;
     MEM_BE        : Out   Std_Logic_Vector(3 Downto 0);
     MEM_USB_E     : Out   Std_Logic;
     MEM_USB_RESET : Out   Std_Logic;
     ---------------------------------------------------------------------------

     ---------------------------------------------------------------------------
     -- Wishbone Slave Port
     ---------------------------------------------------------------------------
     USB_CLK_I      : In    Std_Logic;
     USB_RST_I      : In    Std_Logic;
     USB_CYC_I      : In    Std_Logic;
     USB_STB_I      : In    Std_Logic;
     USB_ACK_O      : Out   Std_Logic;
     USB_ADR_I      : In    Std_Logic_Vector(23 DownTo 0);
     USB_DAT_I      : In    Std_Logic_Vector(31 DownTo 0);
     USB_DAT_O      : Out   Std_Logic_Vector(31 DownTo 0);
     USB_WE_I       : In    Std_Logic;
     USB_SEL_I      : In    Std_Logic_Vector(3 DownTo 0)
   );

End ReplaceWith_Designator_USB_X32;
--------------------------------------------------------------------------------

--------------------------------------------------------------------------------
Architecture RTL of ReplaceWith_Designator_USB_X32 Is
    Type TState Is
        (
         State_AddressSetup  ,

         State_Write_Pulse   ,     -- WE goes low during this state
         State_Write_ACK     ,     -- WE goes high. Must keep seperate form addresses etc in case WE goes high before address, CE etc

         State_Read_ACK
        );

    Signal   State      : TState;
    Signal   State_Next : TState;

    SubType  TByteLane   Is Std_Logic_Vector(3 DownTo 0);
    SubType  TOutBits    Is Std_Logic_Vector(4 DownTo 0);

    ----------------------------------------------------------------------------
    -- Constants for byte lanes - always the same
    ----------------------------------------------------------------------------
    Constant Write_Word           : TByteLane := "1111";
    Constant Write_LoHalf         : TByteLane := "0011";
    Constant Write_HiHalf         : TByteLane := "1100";
    Constant Write_Byte0          : TByteLane := "0001";
    Constant Write_Byte1          : TByteLane := "0010";
    Constant Write_Byte2          : TByteLane := "0100";
    Constant Write_Byte3          : TByteLane := "1000";
    Constant Write_Bytes012       : TByteLane := "0111";
    Constant Write_Bytes123       : TByteLane := "1110";
    Constant Write_Bytes12        : TByteLane := "0110";
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    Constant OutBits_Write_Word     : TOutBits  := "00000";

    Constant OutBits_Write_LoHalf   : TOutBits  := "11000";
    Constant OutBits_Write_HiHalf   : TOutBits  := "00110";
    Constant OutBits_Write_Byte0    : TOutBits  := "11100";
    Constant OutBits_Write_Byte1    : TOutBits  := "11010";
    Constant OutBits_Write_Byte2    : TOutBits  := "10110";
    Constant OutBits_Write_Byte3    : TOutBits  := "01110";
    Constant OutBits_Write_Bytes012 : TOutBits  := "10000";
    Constant OutBits_Write_Bytes123 : TOutBits  := "00010";
    Constant OutBits_Write_Bytes12  : TOutBits  := "10010";

    Constant OutBits_Read           : TOutBits  := "00000";
    Constant OutBits_None           : TOutBits  := "11111";

    Signal   OutBits              : TOutBits;
    Signal   OutBits_Write        : TOutBits;

    Signal   Request              : Std_Logic;
    Signal   RequestRead          : Std_Logic;
    Signal   RequestWrite         : Std_Logic;

    Signal   WriteActive_Cycle    : Std_Logic;
    Signal   WE                   : Std_Logic;
    Signal   OE                   : Std_Logic;
    Signal   ACK                  : Std_Logic;
    Signal   ADDR                 : Std_Logic_Vector(19-3 Downto 0);

    Constant cAddressSetupDelay : Std_Logic_Vector(2-1 Downto 0) := "10";
    Constant cWritePulseDelay   : Std_Logic_Vector(2-1 Downto 0) := "10";
    Constant cWriteHoldDelay    : Std_Logic_Vector(2-1 Downto 0) := "10";
    Constant cTimerIsZero       : Std_Logic_Vector(2-1 Downto 0) := "00";

    Signal  Timer               : Std_Logic_Vector(2-1 Downto 0);
    Signal  TimerIsZero         : Std_Logic;
    Signal  LoadAddressDelay    : Std_Logic;
    Signal  LoadWritePulseDelay : Std_Logic;
    Signal  LoadWriteHoldDelay  : Std_Logic;
Begin
    ----------------------------------------------------------------------------
    Process(USB_CLK_I)
    Begin
        If Rising_Edge(USB_CLK_I) Then
            If USB_RST_I='1' Then
                Timer <= (Others=>'0');
            Else
                   If LoadAddressDelay='1'    Then Timer <= cAddressSetupDelay;
                ElsIf LoadWritePulseDelay='1' Then Timer <= cWritePulseDelay;
                ElsIf LoadWriteHoldDelay='1'  Then Timer <= cWriteHoldDelay;
                ElsIf TimerIsZero='0'         Then Timer <= Timer - 1;
                End If;
            End If;
        End If;
    End Process;
    TimerIsZero <= '1' When Timer=cTimerIsZero Else '0';
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    ADDR <= USB_ADR_I(16 DownTo 0);
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    MEM_BE <= OutBits(4) & OutBits(3) & OutBits(2) & OutBits(1);
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    MEM_USB_E   <= OutBits(0) Or (Not Request);
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------

    MEM_A    <= "000000" &  ADDR(17-1 Downto 0) & '0'; -- IncludeIf_FullAddress
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------

    USB_DAT_O <= MEM_D_I; -- IncludeIf_Shared
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    USB_ACK_O <= ACK;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    Request      <= USB_STB_I And USB_CYC_I;
    RequestRead  <= USB_STB_I And USB_CYC_I And (Not USB_WE_I);
    RequestWrite <= USB_STB_I And USB_CYC_I And (    USB_WE_I);
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    MEM_W         <= WE;
    MEM_OE        <= OE;
    MEM_USB_RESET <= Not USB_RST_I;
    ----------------------------------------------------------------------------

    OE <= '0' When ((RequestRead='1') And (State=State_Read_ACK)) Else '1';

    ----------------------------------------------------------------------------
    Memory_OuptutControls:
    Process(USB_SEL_I,
            USB_DAT_I,
            State,
            OutBits_Write,
            WriteActive_Cycle)
    Begin
        If WriteActive_Cycle = '1' Then
           OutBits   <= OutBits_Write;

           MEM_D_O   <= USB_DAT_I; -- IncludeIf_Shared
           MEM_D_E   <= '1';        -- IncludeIf_Shared
        Else
           OutBits   <= OutBits_Read;
           MEM_D_O   <= (Others=>'0'); -- IncludeIf_Shared

           MEM_D_E   <= '0';             -- IncludeIf_Shared
        End If;

        Case USB_SEL_I Is
           When Write_Word     => OutBits_Write <= OutBits_Write_Word;
           When Write_LoHalf   => OutBits_Write <= OutBits_Write_LoHalf;
           When Write_HiHalf   => OutBits_Write <= OutBits_Write_HiHalf;
           When Write_Byte0    => OutBits_Write <= OutBits_Write_Byte0;
           When Write_Byte1    => OutBits_Write <= OutBits_Write_Byte1;
           When Write_Byte2    => OutBits_Write <= OutBits_Write_Byte2;
           When Write_Byte3    => OutBits_Write <= OutBits_Write_Byte3;
           When Write_Bytes012 => OutBits_Write <= OutBits_Write_Bytes012;
           When Write_Bytes123 => OutBits_Write <= OutBits_Write_Bytes123;
           When Write_Bytes12  => OutBits_Write <= OutBits_Write_Bytes12;
           When Others         => OutBits_Write <= OutBits_None;
        End Case;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    FSM_Combinational_Single:
    Process(RequestRead,
            RequestWrite,
            State,
            TimerIsZero)
    Begin
        ------------------------------------------------------------------------
        -- Default outputs
        WriteActive_Cycle <= '0';
        WE                <= '1';
        ACK               <= '0';
        State_Next        <= State;
        LoadAddressDelay    <= '0';
        LoadWritePulseDelay <= '0';
        LoadWriteHoldDelay  <= '0';

        ------------------------------------------------------------------------

        ------------------------------------------------------------------------
        Case State Is
           ---------------------------------------------------------------------
           When State_AddressSetup =>
                If RequestWrite = '1' Then                --Address is being setup in this phases
                    WriteActive_Cycle <= '1';
                    State_Next        <= State_Write_Pulse;
                    LoadWritePulseDelay <= '1';
                ElsIf RequestRead = '1' Then
                    State_Next <= State_Read_ACK;
                    LoadAddressDelay <= '1';
                End If;
           ---------------------------------------------------------------------

           ---------------------------------------------------------------------
           When State_Write_Pulse =>
                WriteActive_Cycle <= '1';
                WE                <= '0';
                If TimerIsZero='1' Then
                    State_Next        <= State_Write_ACK;
                    LoadWriteHoldDelay <= '1';
                End If;

           When State_Write_ACK =>
                WriteActive_Cycle <= '1';
                If TimerIsZero='1' Then
                    ACK               <= '1';
                    State_Next        <= State_AddressSetup;
                End if;
           ---------------------------------------------------------------------

           ---------------------------------------------------------------------
           When State_Read_ACK =>                    -- Assert ACK, Master will read the data one clock cycle later
                If TimerIsZero='1' Then
                    ACK           <= '1';
                    State_Next    <= State_AddressSetup;
                End If;
           ---------------------------------------------------------------------

           When Others =>
                State_Next <= State_AddressSetup;

        End Case;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    FSM_Synchronous:
    Process(USB_CLK_I,USB_RST_I,State_Next)
    Begin
        If Rising_Edge(USB_CLK_I) Then
           If USB_RST_I = '1' Then
              State  <= State_AddressSetup;
           Else
              State <= State_Next;
           End If;
        End If;
    End Process;
    ----------------------------------------------------------------------------


End RTL;
--------------------------------------------------------------------------------
