-- ---------------------------------------------------------------------------------------------------------------------------------------------
-- Global Package                                                                                                                 Global Package
-- ---------------------------------------------------------------------------------------------------------------------------------------------
Library IEEE;
Use     IEEE.Std_Logic_1164.All;
Use     IEEE.Std_Logic_Arith.All;
-- ---------------------------------------------------------------------------------------------------------------------------------------------

-- -----------------------------------------------------------------------------
Package Configurable_ReplaceWith_Designator_GlobalConfig Is
    Constant cWishboneAddressWidth  : Integer := ReplaceWith_WishboneAddressWidth;
    Constant cSdramDataWidth        : Integer := ReplaceWith_SdramDataWidth;
    Constant cSdramAddressWidth     : Integer := ReplaceWith_SdramAddressWidth;
    Constant cSdramBankAddressWidth : Integer := ReplaceWith_SdramBankAddressWidth;
    Constant cSdramByteEnableWidth  : Integer := ReplaceWith_SdramByteEnableWidth;
    Constant cMemAddressWidth       : Integer := ReplaceWith_MemAddressWidth;
    Constant cTimerWidth            : Integer :=  3;
    Constant cCACHE_ADR_WIDTH       : Integer := 11;

    -- Different IC geometries are resolved here
    Constant cFlushBankAddressMsb              : Integer := ReplaceWith_FlushBankAddressMsb;
    Constant cFlushBankAddressLsb              : Integer := ReplaceWith_FlushBankAddressLsb;
    Constant cFlushRowAddressMsb               : Integer := ReplaceWith_FlushRowAddressMsb;
    Constant cFlushRowAddressLsb               : Integer := ReplaceWith_FlushRowAddressLsb;
    Constant cFlushIndexRowAddressMsb          : Integer := ReplaceWith_FlushIndexRowAddressMsb;
    Constant cFlushIndexRowAddressLsb          : Integer := ReplaceWith_FlushIndexRowAddressLsb;
    Constant cFlushIndexMsb                    : Integer := ReplaceWith_FlushIndexMsb;
    Constant cFlushIndexLsb                    : Integer := ReplaceWith_FlushIndexLsb;
    Constant cReplaceBankAddressMsb            : Integer := ReplaceWith_ReplaceBankAddressMsb;
    Constant cReplaceBankAddressLsb            : Integer := ReplaceWith_ReplaceBankAddressLsb;
    Constant cReplaceRowAddressMsb             : Integer := ReplaceWith_ReplaceRowAddressMsb;
    Constant cReplaceRowAddressLsb             : Integer := ReplaceWith_ReplaceRowAddressLsb;
    Constant cReplaceColumnAddressMsb          : Integer := ReplaceWith_ReplaceColumnAddressMsb;
    Constant cReplaceColumnAddressLsb          : Integer := ReplaceWith_ReplaceColumnAddressLsb;

    Constant cReplaceColumnAddressPaddingWidth : Integer := ReplaceWith_ReplaceColumnAddressPaddingWidth;
    Constant cReplaceColumnAddressPadding      : Std_Logic_Vector(cReplaceColumnAddressPaddingWidth - 1 Downto 0) := Conv_Std_Logic_Vector(0, cReplaceColumnAddressPaddingWidth);

    Constant cFlushColumnAddressPaddingWidth   : Integer := ReplaceWith_FlushColumnAddressPaddingWidth;
    Constant cFlushColumnAddressPadding        : Std_Logic_Vector(cFlushColumnAddressPaddingWidth - 1 Downto 0) := Conv_Std_Logic_Vector(0, cFlushColumnAddressPaddingWidth);

    Constant cSdramMemoryAddressPaddingWidth   : Integer := ReplaceWith_SdramMemoryAddressPaddingWidth;
    Constant cSdramMemoryAddressPadding        : Std_Logic_Vector(cSdramMemoryAddressPaddingWidth - 1 Downto 0) := Conv_Std_Logic_Vector(0, cSdramMemoryAddressPaddingWidth);

    Constant cTM_tCL                  : Integer := ReplaceWith_tCL;
    Constant cTM_tRP                  : Integer := ReplaceWith_tRP;
    Constant cTM_tWR                  : Integer := ReplaceWith_tWR;
    Constant cTM_tRFC                 : Integer := ReplaceWith_tRFC;
    Constant cTM_tMRD                 : Integer := ReplaceWith_tMRD;
    Constant cTM_tRCD                 : Integer := ReplaceWith_tRCD;
    Constant cTM_Refresh              : Integer := ReplaceWith_Refresh;
    Constant cTimerValue_tRP        : Std_Logic_Vector(cTimerWidth - 1 Downto 0) := Conv_Std_Logic_Vector(cTM_tRP  - 1, cTimerWidth);
    Constant cTimerValue_tWR        : Std_Logic_Vector(cTimerWidth - 1 Downto 0) := Conv_Std_Logic_Vector(cTM_tWR  - 1, cTimerWidth);
    Constant cTimerValue_tRFC       : Std_Logic_Vector(cTimerWidth - 1 Downto 0) := Conv_Std_Logic_Vector(cTM_tRFC - 1, cTimerWidth);
    Constant cTimerValue_tMRD       : Std_Logic_Vector(cTimerWidth - 1 Downto 0) := Conv_Std_Logic_Vector(cTM_tMRD - 1, cTimerWidth);
    Constant cTimerValue_tRCD       : Std_Logic_Vector(cTimerWidth - 1 Downto 0) := Conv_Std_Logic_Vector(cTM_tRCD - 1, cTimerWidth);
    Constant cTimerValue_tCL        : Std_Logic_Vector(cTimerWidth - 1 Downto 0) := Conv_Std_Logic_Vector(cTM_tCL  - 1, cTimerWidth);
    Constant cTimerValue_Refresh    : Std_Logic_Vector(10 Downto 0) := Conv_Std_Logic_Vector(cTM_Refresh - 1, 11);

    Constant cBL_8                  : Integer   :=  3;
    Constant cBT_INTERLEAVED        : Std_Logic := '1';
    Constant cBT_SEQUENTIAL         : Std_Logic := '0';
    Constant cOP_MODE_NORMAL        : Integer   :=  0;
    Constant cWB_BURST              : Std_Logic := '0';
    Constant cRESERVED              : Integer   :=  0;
    -- -------------------------------------------------------------------------
    Constant cMODE_BURST_LENGTH     : Std_Logic_Vector( 2 Downto 0) := Conv_Std_Logic_Vector(cBL_8, 3);
    Constant cMODE_BURST_TYPE       : Std_Logic                     := cBT_INTERLEAVED;
    Constant cMODE_CAS_LATENCY      : Std_Logic_Vector( 2 Downto 0) := Conv_Std_Logic_Vector(cTM_tCL, 3);
    Constant cMODE_OPERATING_MODE   : Std_Logic_Vector( 1 Downto 0) := Conv_Std_Logic_Vector(cOP_MODE_NORMAL, 2);
    Constant cMODE_WRITE_BURST_MODE : Std_Logic                     := cWB_BURST;
    Constant cMODE_RESERVED         : Std_Logic_Vector( 2 Downto 0) := Conv_Std_Logic_Vector(cRESERVED, 3);
    Constant cLoadModeRegister      : Std_Logic_Vector(12 Downto 0) := cMODE_RESERVED         &
                                                                       cMODE_WRITE_BURST_MODE &
                                                                       cMODE_OPERATING_MODE   &
                                                                       cMODE_CAS_LATENCY      &
                                                                       cMODE_BURST_TYPE       &
                                                                       cMODE_BURST_LENGTH;

    Constant cCACHE_FLAG_CACHE_GO             : Integer := 31;
    Constant cCACHE_FLAG_REPLACE              : Integer := 30;
    Constant cCACHE_FLAG_FLUSH                : Integer := 29;
    Constant cCACHE_FLAG_DONE_ALL             : Integer := 28;
    Constant cCACHE_FLAG_DONE_REPLACE         : Integer := 27;
    Constant cCACHE_FLAG_DONE_FLUSH           : Integer := 26;
    Constant cCACHE_FLAG_DONE_REPLACE_CURRENT : Integer := 25;
    Constant cCACHE_FLAG_DONE_FLUSH_CURRENT   : Integer := 24;
    Constant cCACHE_FLAG_DONE_STARTUP         : Integer := 23;
    Constant cCACHE_FLAG_FLUSH_FOR_READ       : Integer := 22;
    Constant cCACHE_FLAG_FAST_REPLACE         : Integer := 21;
    Constant cCACHE_FLAG_FLUSH_FOR_WRITE      : Integer := 20;
    Constant cCACHE_FLAG_MEMORY_INITIALIZED   : Integer := 19;
    Constant cCACHE_FLAG_BE_MSB               : Integer := 18;
    Constant cCACHE_FLAG_BE_LSB               : Integer := 15;

    Constant cCACHE_CMD_ADR_CMD  : Std_Logic_Vector(1 Downto 0) := "00";
    Constant cCACHE_CMD_ADR_LINE : Std_Logic_Vector(1 Downto 0) := "01";


    Constant cCACHE_ADR_LSB                : Integer := ReplaceWith_CacheAddressLsb; -- 4 for x32, 3 for x16, 2 for x8

    Constant cWbPageRequestPadWidth        : Integer := ReplaceWith_WbPageRequestPadWidth;
    Constant cWbReplaceDestinationPadWidth : Integer := ReplaceWith_WbReplaceDestinationPadWidth;
    Constant cWbPageRequestPad             : Std_Logic_Vector(cWbPageRequestPadWidth        - 1 Downto 0) := Conv_Std_Logic_Vector(0, cWbPageRequestPadWidth);
    Constant cWbReplaceDestinationPad      : Std_Logic_Vector(cWbReplaceDestinationPadWidth - 1 Downto 0) := Conv_Std_Logic_Vector(0, cWbReplaceDestinationPadWidth);
End Package;
-- -----------------------------------------------------------------------------


-- ---------------------------------------------------------------------------------------------------------------------------------------------
-- Data Cache                                                                                                                         Data Cache
-- ---------------------------------------------------------------------------------------------------------------------------------------------
Library IEEE;
Use     IEEE.Std_Logic_1164.All;
-- ---------------------------------------------------------------------------------------------------------------------------------------------

-- -----------------------------------------------------------------------------
Entity Configurable_ReplaceWith_Designator_DataCache Is
Port (
     CLKA     : In  Std_Logic;
     DINA     : In  Std_Logic_Vector(31 Downto 0);
     DOUTA    : Out Std_Logic_Vector(31 Downto 0);
     ADDRA    : In  Std_Logic_Vector(10 Downto 0);
     BYTE_WEA : In  Std_Logic_Vector( 3 Downto 0);
     WEA      : In  Std_Logic;

     CLKB     : In  Std_Logic;
     DINB     : In  Std_Logic_Vector( 7 Downto 0);
     DOUTB    : Out Std_Logic_Vector( 7 Downto 0);
     ADDRB    : In  Std_Logic_Vector(12 Downto 0);
     BYTE_WEB : In  Std_Logic;
     WEB      : In  Std_Logic
);
End;
-- -----------------------------------------------------------------------------

-- -----------------------------------------------------------------------------
Architecture RTL Of Configurable_ReplaceWith_Designator_DataCache Is
-- -----------------------------------------------------------------------------
    -- -------------------------------------------------------------------------
    Component Configurable_ReplaceWith_Designator_ByteMemory Is
        Port(
            CLKA    : In  Std_Logic;
            CLKB    : In  Std_Logic;
            WEA     : In  Std_Logic;
            WEB     : In  Std_Logic;
            DINA    : In  Std_Logic_Vector( 7 Downto 0);
            DINB    : In  Std_Logic_Vector( 7 Downto 0);
            DOUTA   : Out Std_Logic_Vector( 7 Downto 0);
            DOUTB   : Out Std_Logic_Vector( 7 Downto 0);
            ADDRA   : In  Std_Logic_Vector(10 Downto 0);
            ADDRB   : In  Std_Logic_Vector(10 Downto 0)
        );
    End Component;
    -- -------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    Signal B3DoutA : Std_Logic_Vector(7 Downto 0);
    Signal B2DoutA : Std_Logic_Vector(7 Downto 0);
    Signal B1DoutA : Std_Logic_Vector(7 Downto 0);
    Signal B0DoutA : Std_Logic_Vector(7 Downto 0);

    Signal B3DoutB : Std_Logic_Vector(7 Downto 0);
    Signal B2DoutB : Std_Logic_Vector(7 Downto 0);
    Signal B1DoutB : Std_Logic_Vector(7 Downto 0);
    Signal B0DoutB : Std_Logic_Vector(7 Downto 0);

    Signal B3WriteEnableA : Std_Logic;
    Signal B2WriteEnableA : Std_Logic;
    Signal B1WriteEnableA : Std_Logic;
    Signal B0WriteEnableA : Std_Logic;

    Signal B3WriteEnableB : Std_Logic;
    Signal B2WriteEnableB : Std_Logic;
    Signal B1WriteEnableB : Std_Logic;
    Signal B0WriteEnableB : Std_Logic;

    Constant cByte3 : Std_Logic_Vector(1 Downto 0) := "00";
    Constant cByte2 : Std_Logic_Vector(1 Downto 0) := "01";
    Constant cByte1 : Std_Logic_Vector(1 Downto 0) := "10";
    Constant cByte0 : Std_Logic_Vector(1 Downto 0) := "11";


    Signal ByteSelect  : Std_Logic_Vector(1 Downto 0);
    Signal Byte3Select : Std_Logic;
    Signal Byte2Select : Std_Logic;
    Signal Byte1Select : Std_Logic;
    Signal Byte0Select : Std_Logic;
    -- -------------------------------------------------------------------------

-- -----------------------------------------------------------------------------
Begin
-- -----------------------------------------------------------------------------

    B3 : Configurable_ReplaceWith_Designator_ByteMemory Port Map (
       CLKA  => CLKA,
       CLKB  => CLKB,
       WEA   => B3WriteEnableA,
       WEB   => B3WriteEnableB,
       DINA  => DINA(31 Downto 24),
       DINB  => DINB,
       DOUTA => B3DoutA,
       DOUTB => B3DoutB,
       ADDRA => ADDRA,
       ADDRB => ADDRB(12 Downto 2)
    );

    B2 : Configurable_ReplaceWith_Designator_ByteMemory Port Map (
       CLKA  => CLKA,
       CLKB  => CLKB,
       WEA   => B2WriteEnableA,
       WEB   => B2WriteEnableB,
       DINA  => DINA(23 Downto 16),
       DINB  => DINB,
       DOUTA => B2DoutA,
       DOUTB => B2DoutB,
       ADDRA => ADDRA,
       ADDRB => ADDRB(12 Downto 2)
    );

    B1 : Configurable_ReplaceWith_Designator_ByteMemory Port Map (
       CLKA  => CLKA,
       CLKB  => CLKB,
       WEA   => B1WriteEnableA,
       WEB   => B1WriteEnableB,
       DINA  => DINA(15 Downto 8),
       DINB  => DINB,
       DOUTA => B1DoutA,
       DOUTB => B1DoutB,
       ADDRA => ADDRA,
       ADDRB => ADDRB(12 Downto 2)
    );

    B0 : Configurable_ReplaceWith_Designator_ByteMemory Port Map (
       CLKA  => CLKA,
       CLKB  => CLKB,
       WEA   => B0WriteEnableA,
       WEB   => B0WriteEnableB,
       DINA  => DINA(7 Downto 0),
       DINB  => DINB,
       DOUTA => B0DoutA,
       DOUTB => B0DoutB,
       ADDRA => ADDRA,
       ADDRB => ADDRB(12 Downto 2)
    );
    DOUTA <= B3DoutA & B2DoutA & B1DoutA & B0DoutA;

    B3WriteEnableA   <= WEA And BYTE_WEA(3);
    B2WriteEnableA   <= WEA And BYTE_WEA(2);
    B1WriteEnableA   <= WEA And BYTE_WEA(1);
    B0WriteEnableA   <= WEA And BYTE_WEA(0);

    --------------------
    ByteSelect  <= ADDRB(1 Downto 0);
    Byte3Select <= '1' When ByteSelect=cByte3 Else '0';
    Byte2Select <= '1' When ByteSelect=cByte2 Else '0';
    Byte1Select <= '1' When ByteSelect=cByte1 Else '0';
    Byte0Select <= '1' When ByteSelect=cByte0 Else '0';

    B3WriteEnableB   <= WEB And BYTE_WEB And Byte3Select;
    B2WriteEnableB   <= WEB And BYTE_WEB And Byte2Select;
    B1WriteEnableB   <= WEB And BYTE_WEB And Byte1Select;
    B0WriteEnableB   <= WEB And BYTE_WEB And Byte0Select;

    DOUTB <= B0DoutB When Byte3Select = '1' Else
             B1DoutB When Byte2Select = '1' Else
             B2DoutB When Byte1Select = '1' Else
             B3DoutB;
-- -----------------------------------------------------------------------------
End RTL;
-- -----------------------------------------------------------------------------

-- ---------------------------------------------------------------------------------------------------------------------------------------------
-- Tag Cache                                                                                                                           Tag Cache
-- ---------------------------------------------------------------------------------------------------------------------------------------------
Library IEEE;
Use     IEEE.Std_Logic_1164.All;
-- ---------------------------------------------------------------------------------------------------------------------------------------------

-- -----------------------------------------------------------------------------
Entity Configurable_ReplaceWith_Designator_TagCache Is
Port (
    CLKA  : In  Std_Logic;
    DINA  : In  Std_Logic_Vector(19 Downto 0);
    DOUTA : Out Std_Logic_Vector(19 Downto 0);
    ADDRA : In  Std_Logic_Vector( 9 Downto 0);
    WEA   : In  Std_Logic;

    CLKB  : In  Std_Logic;
    DINB  : In  Std_Logic_Vector(19 Downto 0);
    DOUTB : Out Std_Logic_Vector(19 Downto 0);
    ADDRB : In  Std_Logic_Vector( 9 Downto 0);
    WEB   : In  Std_Logic
);
End;
-- -----------------------------------------------------------------------------

-- -----------------------------------------------------------------------------
Architecture RTL Of Configurable_ReplaceWith_Designator_TagCache Is
-- -----------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    Component Configurable_ReplaceWith_Designator_TagMemory Is
    Port (
        CLKA  : In  Std_Logic;
        CLKB  : In  Std_Logic;
        WEA   : In  Std_Logic;
        WEB   : In  Std_Logic;
        DINA  : In  Std_Logic_Vector(19 Downto 0);
        DINB  : In  Std_Logic_Vector(19 Downto 0);
        DOUTA : Out Std_Logic_Vector(19 Downto 0);
        DOUTB : Out Std_Logic_Vector(19 Downto 0);
        ADDRA : In  Std_Logic_Vector( 9 Downto 0);
        ADDRB : In  Std_Logic_Vector( 9 Downto 0)
    );
    End Component;
    -- -------------------------------------------------------------------------


-- -----------------------------------------------------------------------------
Begin
-- -----------------------------------------------------------------------------

    U_Configurable_ReplaceWith_Designator_TagCache : Configurable_ReplaceWith_Designator_TagMemory Port Map (
        CLKA  => CLKA,
        CLKB  => CLKB,
        WEA   => WEA,
        WEB   => WEB,
        DINA  => DINA,
        DINB  => DINB,
        DOUTA => DOUTA,
        DOUTB => DOUTB,
        ADDRA => ADDRA,
        ADDRB => ADDRB
    );

-- -----------------------------------------------------------------------------
End RTL;
-- -----------------------------------------------------------------------------


-- ---------------------------------------------------------------------------------------------------------------------------------------------
-- Cmd Pipe                                                                                                                             Cmd Pipe
-- ---------------------------------------------------------------------------------------------------------------------------------------------
Library IEEE;
Use     IEEE.Std_Logic_1164.All;
-- ---------------------------------------------------------------------------------------------------------------------------------------------

-- -----------------------------------------------------------------------------
Entity Configurable_ReplaceWith_Designator_CommandPipe Is
Port (
    CLKA  : In  Std_Logic;
    DINA  : In  Std_Logic_Vector(31 Downto 0);
    DOUTA : Out Std_Logic_Vector(31 Downto 0);
    ADDRA : In  Std_Logic_Vector( 1 Downto 0);
    WEA   : In  Std_Logic;

    CLKB  : In  Std_Logic;
    DINB  : In  Std_Logic_Vector(31 Downto 0);
    DOUTB : Out Std_Logic_Vector(31 Downto 0);
    ADDRB : In  Std_Logic_Vector( 1 Downto 0);
    WEB   : In  Std_Logic
);
End;
-- -----------------------------------------------------------------------------

-- -----------------------------------------------------------------------------
Architecture RTL Of Configurable_ReplaceWith_Designator_CommandPipe Is
-- -----------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    Component Configurable_ReplaceWith_Designator_CmdMemory Is
    Port (
        CLKA  : In  Std_Logic;
        CLKB  : In  Std_Logic;
        WEA   : In  Std_Logic;
        WEB   : In  Std_Logic;
        DINA  : In  Std_Logic_Vector(31 Downto 0);
        DINB  : In  Std_Logic_Vector(31 Downto 0);
        DOUTA : Out Std_Logic_Vector(31 Downto 0);
        DOUTB : Out Std_Logic_Vector(31 Downto 0);
        ADDRA : In  Std_Logic_Vector( 1 Downto 0);
        ADDRB : In  Std_Logic_Vector( 1 Downto 0)
    );
    End Component;
    -- -------------------------------------------------------------------------

-- -----------------------------------------------------------------------------
Begin
-- -----------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    U_Configurable_ReplaceWith_Designator_CmdMemory : Configurable_ReplaceWith_Designator_CmdMemory Port Map (
        CLKA  => CLKA,
        CLKB  => CLKB,
        WEA   => WEA,
        WEB   => WEB,
        DINA  => DINA,
        DINB  => DINB,
        DOUTA => DOUTA,
        DOUTB => DOUTB,
        ADDRA => ADDRA,
        ADDRB => ADDRB
    );
    -- -------------------------------------------------------------------------

-- -----------------------------------------------------------------------------
End RTL;
-- -----------------------------------------------------------------------------



-- ---------------------------------------------------------------------------------------------------------------------------------------------
-- Wishbone Interface                                                                                                         Wishbone Interface
-- ---------------------------------------------------------------------------------------------------------------------------------------------
Library IEEE;
Use     IEEE.Std_Logic_1164.All;
Use     Work.Configurable_ReplaceWith_Designator_GlobalConfig.All;
-- ---------------------------------------------------------------------------------------------------------------------------------------------
-- -----------------------------------------------------------------------------
Entity Configurable_ReplaceWith_Designator_WishboneInterface Is
Port (
    -- Cache Command Interface
    CACHE_CMD_CLK     : Out Std_Logic;
    CACHE_CMD_WE_O    : Out Std_Logic;
    CACHE_CMD_DAT_I   : In  Std_Logic_Vector(31 Downto 0);
    CACHE_CMD_DAT_O   : Out Std_Logic_Vector(31 Downto 0);
    CACHE_CMD_ADR_O   : Out Std_Logic_Vector( 1 Downto 0);

    -- Dual Port RAM Wishbone Data Cache Interface
    DP_RAM_DAT_CLK_O  : Out Std_Logic;
    DP_RAM_DAT_RST_O  : Out Std_Logic;
    DP_RAM_DAT_DATA_I : In  Std_Logic_Vector(31 Downto 0);
    DP_RAM_DAT_DATA_O : Out Std_Logic_Vector(31 Downto 0);
    DP_RAM_DAT_ADR_O  : Out Std_Logic_Vector(cCACHE_ADR_WIDTH - 1 Downto 0);
    DP_RAM_DAT_BE_O   : Out Std_Logic_Vector( 3 Downto 0);
    DP_RAM_DAT_WE_O   : Out Std_Logic;

    -- Dual Port RAM Cache Tag Interface
    DP_RAM_TAG_CLK_O  : Out Std_Logic;
    DP_RAM_TAG_RST_O  : Out Std_Logic;
    DP_RAM_TAG_DATA_I : In  Std_Logic_Vector(19 Downto 0);
    DP_RAM_TAG_DATA_O : Out Std_Logic_Vector(19 Downto 0);
    DP_RAM_TAG_ADR_O  : Out Std_Logic_Vector(cCACHE_ADR_WIDTH - 2 Downto 0);
    DP_RAM_TAG_BE_O   : Out Std_Logic_Vector( 3 Downto 0);
    DP_RAM_TAG_WE_O   : Out Std_Logic;

    -- Wishbone Interface
    WB_CLK_I          : In  Std_Logic;
    WB_RST_I          : In  Std_Logic;
    WB_STB_I          : In  Std_Logic;
    WB_CYC_I          : In  Std_Logic;
    WB_ACK_O          : Out Std_Logic;
    WB_ADR_I          : In  Std_Logic_Vector(cWishboneAddressWidth - 1 Downto 0);
    WB_DAT_I          : In  Std_Logic_Vector(31 Downto 0);
    WB_DAT_O          : Out Std_Logic_Vector(31 Downto 0);
    WB_SEL_I          : In  Std_Logic_Vector( 3 Downto 0);
    WB_WE_I           : In  Std_Logic
);
End;
-- -----------------------------------------------------------------------------

-- -----------------------------------------------------------------------------
Architecture RTL Of Configurable_ReplaceWith_Designator_WishboneInterface Is
-- -----------------------------------------------------------------------------

    Type TCacheState Is ( sCacheIdle,
                          sCacheHitCheckRead,
                          sCacheHitCheckWrite,
                          sCacheWaitForLineReplace,
                          sCacheWaitForLineFlushed );

    Signal CacheState     : TCacheState;
    Signal CacheStateNext : TCacheState;

    Signal WishboneRequest            : Std_Logic;
    Signal WishboneWrite              : Std_Logic;
    Signal WishboneRead               : Std_Logic;

    Signal DataIndex                  : Std_Logic_Vector(cCACHE_ADR_WIDTH - 1 Downto 0);
    Signal TagIndex                   : Std_Logic_Vector(cCACHE_ADR_WIDTH - 2 Downto 0);

    Signal PageActual                 : Std_Logic_Vector(19 Downto 0);
    Signal PageRequested              : Std_Logic_Vector(19 Downto 0);
    Signal PageRequested_Reg          : Std_Logic_Vector(19 Downto 0);
    Signal CacheHit                   : Std_Logic;

    Signal DoWishBoneWriteAcknowledge : Std_Logic;
    Signal DoWishboneReadAcknowledge  : Std_Logic;

    Signal DoCacheWrite               : Std_Logic;

    Signal DoCache_PrepareReplaceLine : Std_Logic;
    Signal DoCache_ReplaceLine        : Std_Logic;
    Signal DoCache_FlushLine          : Std_Logic;

    Signal CacheDone_All              : Std_Logic;
    Signal CacheDone_Replace          : Std_Logic;
    Signal CacheDone_Flush            : Std_Logic;
    Signal Cache_Status               : Std_Logic_Vector(31 Downto 0);
    Signal Cache_Write                : Std_Logic;
    Signal Cache_Command              : Std_Logic_Vector(31 Downto 0);
    Signal Cache_Data_Out             : Std_Logic_Vector(31 Downto 0);
    Signal Cache_Address              : Std_Logic_Vector( 1 Downto 0);
    Signal CommandRequest             : Std_Logic;
    Signal CacheCanGo                 : Std_Logic;

    Signal ReplaceDestinationAddress  : Std_Logic_Vector(31 Downto 0);
    Signal MemoryIsInitialized        : Std_Logic;
-- -----------------------------------------------------------------------------
Begin
-- -----------------------------------------------------------------------------
    -- -------------------------------------------------------------------------
    CACHE_CMD_CLK   <= WB_CLK_I;
    CACHE_CMD_ADR_O <= Cache_Address;
    CACHE_CMD_DAT_O <= Cache_Data_Out;
    CACHE_CMD_WE_O  <= Cache_Write;
    Cache_Status    <= CACHE_CMD_DAT_I;
    -- -------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    Cache_Data_Out <= ReplaceDestinationAddress When DoCache_PrepareReplaceLine = '1' Else
                      Cache_Command;
    Cache_Write    <= DoCache_PrepareReplaceLine Or CommandRequest;
    Cache_Address  <= cCACHE_CMD_ADR_LINE When DoCache_PrepareReplaceLine = '1' Else cCACHE_CMD_ADR_CMD;
    -- -------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    Cache_Command(cCACHE_FLAG_CACHE_GO)                             <= '1';
    Cache_Command(cCACHE_FLAG_REPLACE)                              <= DoCache_ReplaceLine;
    Cache_Command(cCACHE_FLAG_FLUSH)                                <= DoCache_FlushLine;
    Cache_Command(cCACHE_FLAG_FLUSH_FOR_READ)                       <= WishboneRead;
    Cache_Command(cCACHE_FLAG_FAST_REPLACE)                         <= Not PageActual(19) And PageActual(18) And WishboneRead; -- 21
    Cache_Command(cCACHE_FLAG_FLUSH_FOR_WRITE)                      <= WishboneWrite;
    Cache_Command(cCACHE_FLAG_MEMORY_INITIALIZED)                   <= '0';
    Cache_Command(cCACHE_FLAG_BE_MSB     Downto cCACHE_FLAG_BE_LSB) <= WB_SEL_I;
    Cache_Command(cCACHE_FLAG_BE_LSB - 1 Downto cCACHE_ADR_WIDTH)   <= (Others=>'0');
    Cache_Command(cCACHE_ADR_WIDTH   - 1 Downto  0)                 <= DataIndex;
    Cache_Command(cCACHE_FLAG_DONE_ALL)                             <= '0';	
    -- -------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    CommandRequest    <= DoCache_ReplaceLine Or DoCache_FlushLine;
    -- -------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    CacheDone_All     <= Cache_Status(cCACHE_FLAG_DONE_ALL);
    CacheDone_Replace <= Cache_Status(cCACHE_FLAG_DONE_REPLACE_CURRENT);
    CacheDone_Flush   <= Cache_Status(cCACHE_FLAG_DONE_FLUSH);
    -- -------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    WishboneRequest <= WB_STB_I And WB_CYC_I;
    WishboneWrite   <= WishboneRequest And     WB_WE_I;
    WishboneRead    <= WishboneRequest And Not WB_WE_I;
    -- -------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    DataIndex     <= WB_ADR_I(cCACHE_ADR_WIDTH + 1 Downto 2);
    TagIndex      <= WB_ADR_I(cCACHE_ADR_WIDTH + 1 Downto 3);
    -- -------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    WB_DAT_O <= DP_RAM_DAT_DATA_I;
    WB_ACK_O <= DoWishboneWriteAcknowledge Or DoWishboneReadAcknowledge;
    -- -------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    -- Cache data side.
    -- -------------------------------------------------------------------------
    DP_RAM_DAT_CLK_O  <= WB_CLK_I;
    DP_RAM_DAT_RST_O  <= WB_RST_I;
    DP_RAM_DAT_DATA_O <= WB_DAT_I;
    DP_RAM_DAT_ADR_O  <= DataIndex;
    DP_RAM_DAT_BE_O   <= WB_SEL_I;
    DP_RAM_DAT_WE_O   <= DoCacheWrite;
    -- -------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    -- Cache Tag side.
    -- -------------------------------------------------------------------------
    DP_RAM_TAG_CLK_O  <= WB_CLK_I;
    DP_RAM_TAG_RST_O  <= WB_RST_I;
    DP_RAM_TAG_ADR_O  <= TagIndex;
    DP_RAM_TAG_DATA_O <= PageRequested;
    DP_RAM_TAG_BE_O   <= (Others=>'1');
    DP_RAM_TAG_WE_O   <= DoCacheWrite;
    PageActual        <= DP_RAM_TAG_DATA_I;
    -- -------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    Process(WB_CLK_I)
    Begin
        If Rising_Edge(WB_CLK_I) Then
            If WB_RST_I='1' Then
                PageRequested_Reg <= (Others=>'0');
            Else
                PageRequested_Reg <= '1' & '1' & cWbPageRequestPad & WB_ADR_I(cWishboneAddressWidth - 1 Downto cCACHE_ADR_WIDTH + 2);
            End If;
        End If;
    End Process;
    PageRequested             <= '1' & '1' & cWbPageRequestPad & WB_ADR_I(cWishboneAddressWidth - 1 Downto cCACHE_ADR_WIDTH + 2);
    ReplaceDestinationAddress <= cWbReplaceDestinationPad & WB_ADR_I(cWishboneAddressWidth - 1 Downto 2);
    -- -------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    MemoryIsInitialized <= Cache_Status(cCACHE_FLAG_MEMORY_INITIALIZED);
    CacheCanGo <= '1' When CacheDone_All='0' And TagIndex /= Cache_Status(cCACHE_ADR_WIDTH - 1 Downto 1) And MemoryIsInitialized = '1' Else
                  '1' When CacheDone_All='1' Else
                  '0';

    -- -------------------------------------------------------------------------
    CacheHit <= '1' When (PageActual(18 Downto 0) = PageRequested_Reg(18 Downto 0)) Else '0';
    -- -------------------------------------------------------------------------
    -- -------------------------------------------------------------------------
    Process(CacheState,
            WishboneRead,
            WishboneWrite,
            CacheDone_All,
            CacheDone_Replace,
            CacheDone_Flush,
            CacheCanGo,
            CacheHit)
    Begin
        CacheStateNext             <= CacheState;
        DoWishboneReadAcknowledge  <= '0';
        DoWishboneWriteAcknowledge <= '0';

        DoCache_PrepareReplaceLine <= '0';
        DoCache_ReplaceLine        <= '0';
        DoCache_FlushLine          <= '0';

        DoCacheWrite               <= '0';

        -- ---------------------------------------------------------------------
        Case CacheState Is
        -- ---------------------------------------------------------------------
            -- -----------------------------------------------------------------
            When sCacheIdle =>
                 If CacheCanGo='1' Then
                      If WishboneRead='1' Then
                          CacheStateNext             <= sCacheHitCheckRead;
                          DoCache_PrepareReplaceLine <= '1';
                      ElsIf WishboneWrite='1' Then
                          DoCache_PrepareReplaceLine <= '1';
                          CacheStateNext             <= sCacheHitCheckWrite;
                      End If;
                 End If;
            -- -----------------------------------------------------------------
            When sCacheHitCheckRead =>
                 If CacheHit='1' Then
                     CacheStateNext            <= sCacheIdle;
                     DoWishboneReadAcknowledge <= '1';
                 Else
                    If CacheDone_All='1' Then
                        CacheStateNext      <= sCacheWaitForLineReplace;
                        DoCache_ReplaceLine <= '1';
                        DoCache_FlushLine   <= '1';
                    End If;
                 End If;
            -- -----------------------------------------------------------------
            When sCacheHitCheckWrite =>
                 If CacheHit='1' Then
                     CacheStateNext             <= sCacheIdle;
                     DoWishboneWriteAcknowledge <= '1';
                     DoCacheWrite               <= '1';
                 Else
                    If CacheDone_All='1' Then
                        CacheStateNext      <= sCacheWaitForLineFlushed;
                        DoCache_ReplaceLine <= '1';
                        DoCache_FlushLine   <= '1';
                     End If;
                 End If;
            -- -----------------------------------------------------------------
            When sCacheWaitForLineReplace =>
                 If CacheDone_Replace = '1' Then
                    CacheStateNext            <= sCacheIdle;
                    DoWishboneReadAcknowledge <= '1';
                 End If;
            -- -----------------------------------------------------------------
            When sCacheWaitForLineFlushed =>
                 If CacheDone_Flush = '1' Then
                    CacheStateNext             <= sCacheIdle;
                    DoWishboneWriteAcknowledge <= '1';
                    DoCacheWrite <= '1';
                 End If;
            -- -----------------------------------------------------------------
            When Others =>
                 CacheStateNext <= sCacheIdle;
        -- ---------------------------------------------------------------------
        End Case;
        -- ---------------------------------------------------------------------
    End Process;
    -- -------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    Process(WB_CLK_I)
    Begin
        If Rising_Edge(WB_CLK_I) Then
            If WB_RST_I='1' Then
                CacheState  <= sCacheIdle;
            Else
                CacheState  <= CacheStateNext;
            End If;
        End If;
    End Process;
    -- -------------------------------------------------------------------------
-- -----------------------------------------------------------------------------
End RTL;
-- -----------------------------------------------------------------------------

-- ---------------------------------------------------------------------------------------------------------------------------------------------
-- Top Level For Component                                                                                               Top Level For Component
-- ---------------------------------------------------------------------------------------------------------------------------------------------
Library IEEE;
Use     IEEE.Std_Logic_1164.All;
Use     Work.Configurable_ReplaceWith_Designator_GlobalConfig.All;
-- ---------------------------------------------------------------------------------------------------------------------------------------------

-- -----------------------------------------------------------------------------
Entity Configurable_ReplaceWith_Designator Is
Port (
    MEM_CLK       : In    Std_Logic;
    -- Wishbone Interface
    SDRAM_CLK_I   : In    Std_Logic;
    SDRAM_RST_I   : In    Std_Logic;
    SDRAM_ACK_O   : Out   Std_Logic;
    SDRAM_STB_I   : In    Std_Logic;
    SDRAM_CYC_I   : In    Std_Logic;
    SDRAM_ADR_I   : In    Std_Logic_Vector(cWishboneAddressWidth - 1 Downto 0);
    SDRAM_DAT_O   : Out   Std_Logic_Vector(31 Downto 0);
    SDRAM_DAT_I   : In    Std_Logic_Vector(31 Downto 0);
    SDRAM_SEL_I   : In    Std_Logic_Vector( 3 Downto 0);
    SDRAM_WE_I    : In    Std_Logic;

    MEM_D         : InOut Std_Logic_Vector(cSdramDataWidth - 1 Downto 0);
    MEM_A         : Out   Std_Logic_Vector(cMemAddressWidth - 1 Downto 0);
    MEM_W         : Out   Std_Logic;
    MEM_BE        : Out   Std_Logic;
    MEM_SDRAM_CKE : Out Std_Logic;
    MEM_SDRAM_RAS : Out Std_Logic;
    MEM_SDRAM_CAS : Out Std_Logic;
    MEM_SDRAM_E   : Out Std_Logic
);
End;
-- -----------------------------------------------------------------------------

-- -----------------------------------------------------------------------------
Architecture RTL Of Configurable_ReplaceWith_Designator Is
-- -----------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    Component Configurable_ReplaceWith_Designator_DataCache Is
    Port (
         CLKA     : In  Std_Logic;
         DINA     : In  Std_Logic_Vector(31 Downto 0);
         DOUTA    : Out Std_Logic_Vector(31 Downto 0);
         ADDRA    : In  Std_Logic_Vector(10 Downto 0);
         BYTE_WEA : In  Std_Logic_Vector( 3 Downto 0);
         WEA      : In  Std_Logic;

         CLKB     : In  Std_Logic;
         DINB     : In  Std_Logic_Vector( 7 Downto 0);
         DOUTB    : Out Std_Logic_Vector( 7 Downto 0);
         ADDRB    : In  Std_Logic_Vector(12 Downto 0);
         BYTE_WEB : In  Std_Logic;
         WEB      : In  Std_Logic
    );
    End Component;
    -- -------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    Component Configurable_ReplaceWith_Designator_TagCache Is
    Port (
        CLKA  : In  Std_Logic;
        DINA  : In  Std_Logic_Vector(19 Downto 0);
        DOUTA : Out Std_Logic_Vector(19 Downto 0);
        ADDRA : In  Std_Logic_Vector( 9 Downto 0);
        WEA   : In  Std_Logic;

        CLKB  : In  Std_Logic;
        DINB  : In  Std_Logic_Vector(19 Downto 0);
        DOUTB : Out Std_Logic_Vector(19 Downto 0);
        ADDRB : In  Std_Logic_Vector( 9 Downto 0);
        WEB   : In  Std_Logic
    );
    End Component;
    -- -------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    Component Configurable_ReplaceWith_Designator_CommandPipe Is
    Port (
        CLKA  : In  Std_Logic;
        DINA  : In  Std_Logic_Vector(31 Downto 0);
        DOUTA : Out Std_Logic_Vector(31 Downto 0);
        ADDRA : In  Std_Logic_Vector( 1 Downto 0);
        WEA   : In  Std_Logic;

        CLKB  : In  Std_Logic;
        DINB  : In  Std_Logic_Vector(31 Downto 0);
        DOUTB : Out Std_Logic_Vector(31 Downto 0);
        ADDRB : In  Std_Logic_Vector( 1 Downto 0);
        WEB   : In  Std_Logic
    );
    End Component;
    -- -------------------------------------------------------------------------


    -- -------------------------------------------------------------------------
    Component Configurable_ReplaceWith_Designator_WishboneInterface Is
    Port (
        -- Cache Command Interface
        CACHE_CMD_CLK     : Out Std_Logic;
        CACHE_CMD_WE_O    : Out Std_Logic;
        CACHE_CMD_DAT_I   : In  Std_Logic_Vector(31 Downto 0);
        CACHE_CMD_DAT_O   : Out Std_Logic_Vector(31 Downto 0);
        CACHE_CMD_ADR_O   : Out Std_Logic_Vector( 1 Downto 0);

        -- Dual Port RAM Wishbone Data Cache Interface
        DP_RAM_DAT_CLK_O  : Out Std_Logic;
        DP_RAM_DAT_RST_O  : Out Std_Logic;
        DP_RAM_DAT_DATA_I : In  Std_Logic_Vector(31 Downto 0);
        DP_RAM_DAT_DATA_O : Out Std_Logic_Vector(31 Downto 0);
        DP_RAM_DAT_ADR_O  : Out Std_Logic_Vector(cCACHE_ADR_WIDTH - 1 Downto 0);
        DP_RAM_DAT_BE_O   : Out Std_Logic_Vector( 3 Downto 0);
        DP_RAM_DAT_WE_O   : Out Std_Logic;

        -- Dual Port RAM Cache Tag Interface
        DP_RAM_TAG_CLK_O  : Out Std_Logic;
        DP_RAM_TAG_RST_O  : Out Std_Logic;
        DP_RAM_TAG_DATA_I : In  Std_Logic_Vector(19 Downto 0);
        DP_RAM_TAG_DATA_O : Out Std_Logic_Vector(19 Downto 0);
        DP_RAM_TAG_ADR_O  : Out Std_Logic_Vector(cCACHE_ADR_WIDTH - cCACHE_ADR_LSB Downto 0);
        DP_RAM_TAG_BE_O   : Out Std_Logic_Vector( 3 Downto 0);
        DP_RAM_TAG_WE_O   : Out Std_Logic;

        -- Wishbone Interface
        WB_CLK_I          : In  Std_Logic;
        WB_RST_I          : In  Std_Logic;
        WB_STB_I          : In  Std_Logic;
        WB_CYC_I          : In  Std_Logic;
        WB_ACK_O          : Out Std_Logic;
        WB_ADR_I          : In  Std_Logic_Vector(cWishboneAddressWidth - 1 Downto 0);
        WB_DAT_I          : In  Std_Logic_Vector(31 Downto 0);
        WB_DAT_O          : Out Std_Logic_Vector(31 Downto 0);
        WB_SEL_I          : In  Std_Logic_Vector( 3 Downto 0);
        WB_WE_I           : In  Std_Logic
    );
    End Component;
    -- -------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    Component ReplaceWith_SdramEngineName Is
    Port (
        -- System
        MEM_CLK                         : In  Std_Logic;
        MEM_RST                         : In  Std_Logic;
        -- Data Cache
        CACHE_DAT_CLK_O                 : Out Std_Logic;
        CACHE_DAT_BE_O                  : Out Std_Logic;
        CACHE_DAT_WE_O                  : Out Std_Logic;
        CACHE_DAT_ADR_O                 : Out Std_Logic_Vector(12 Downto 0);
        CACHE_DAT_DATA_I                : In  Std_Logic_Vector( 7 Downto 0);
        CACHE_DAT_DATA_O                : Out Std_Logic_Vector( 7 Downto 0);
        -- Tag Cache
        CACHE_TAG_CLK_O                 : Out Std_Logic;
        CACHE_TAG_WE_O                  : Out Std_Logic;
        CACHE_TAG_ADR_O                 : Out Std_Logic_Vector( 9 Downto 0);
        CACHE_TAG_DATA_I                : In  Std_Logic_Vector(19 Downto 0);
        CACHE_TAG_DATA_O                : Out Std_Logic_Vector(19 Downto 0);
        -- Command Pipe
        CACHE_CMD_CLK_O                 : Out Std_Logic;
        CACHE_CMD_WE_O                  : Out Std_Logic;
        CACHE_CMD_ADR_O                 : Out Std_Logic_Vector( 1 Downto 0);
        CACHE_CMD_DATA_O                : Out Std_Logic_Vector(31 Downto 0);
        CACHE_CMD_DATA_I                : In  Std_Logic_Vector(31 Downto 0);
        -- Timer Configuration
        TIMER_VALUE_TRP                 : In  Std_Logic_Vector(2 Downto 0);
        TIMER_VALUE_TWR                 : In  Std_Logic_Vector(2 Downto 0);
        TIMER_VALUE_TRFC                : In  Std_Logic_Vector(2 Downto 0);
        TIMER_VALUE_TMRD                : In  Std_Logic_Vector(2 Downto 0);
        TIMER_VALUE_TRCD                : In  Std_Logic_Vector(2 Downto 0);
        TIMER_VALUE_TCL                 : In  Std_Logic_Vector(2 Downto 0);
        -- Memory Mode Register
        LOAD_MODE_REGISTER              : In  Std_Logic_Vector(12 Downto 0);
        -- Address Resover for diff memory geometries
        RESOLVED_BANK_ADDRESS_FLUSH     : In  Std_Logic_Vector( 1 Downto 0);
        RESOLVED_ROW_ADDRESS_FLUSH      : In  Std_Logic_Vector(12 Downto 0);
        RESOLVED_COLUMN_ADDRESS_FLUSH   : In  Std_Logic_Vector(12 Downto 0);
        RESOLVED_BANK_ADDRESS_REPLACE   : In  Std_Logic_Vector( 1 Downto 0);
        RESOLVED_ROW_ADDRESS_REPLACE    : In  Std_Logic_Vector(12 Downto 0);
        RESOLVED_COLUMN_ADDRESS_REPLACE : In  Std_Logic_Vector(12 Downto 0);

        RESOLVER_FLUSH_ADDRESS          : Out Std_Logic_Vector(19 Downto 0);
        RESOLVER_REPLACE_ADDRESS        : Out Std_Logic_Vector(31 Downto 0);
        RESOLVER_FLUSH_INDEX            : Out Std_Logic_Vector(10 Downto 0);
        -- Refresh engine reload value
        REFRESH_RELOAD_VALUE            : In  Std_Logic_Vector(10 Downto 0);
        -- Memory Bus
        SDRAM_DATA_O                    : Out Std_Logic_Vector( 7 Downto 0);
        SDRAM_DATA_I                    : In  Std_Logic_Vector( 7 Downto 0);
        SDRAM_DATA_E                    : Out Std_Logic_Vector( 7 Downto 0);
        SDRAM_A                         : Out Std_Logic_Vector(12 Downto 0);
        SDRAM_BA                        : Out Std_Logic_Vector( 1 Downto 0);
        SDRAM_W                         : Out Std_Logic;
        SDRAM_BE                        : Out Std_Logic;
        SDRAM_CKE                       : Out Std_Logic;
        SDRAM_RAS                       : Out Std_Logic;
        SDRAM_CAS                       : Out Std_Logic;
        SDRAM_E                         : Out Std_Logic
    );
    End Component;
    -- -------------------------------------------------------------------------

    -- Timing Configuration
    Signal TimerValue_tRP   : Std_Logic_Vector(cTimerWidth - 1 Downto 0);
    Signal TimerValue_tWR   : Std_Logic_Vector(cTimerWidth - 1 Downto 0);
    Signal TimerValue_tRFC  : Std_Logic_Vector(cTimerWidth - 1 Downto 0);
    Signal TimerValue_tMRD  : Std_Logic_Vector(cTimerWidth - 1 Downto 0);
    Signal TimerValue_tRCD  : Std_Logic_Vector(cTimerWidth - 1 Downto 0);
    Signal TimerValue_tCL   : Std_Logic_Vector(cTimerWidth - 1 Downto 0);
    Signal LoadModeRegister : Std_Logic_Vector(12 Downto 0);

    -- Address Resolver
    Signal FinalBankAddress_Flush     : Std_Logic_Vector( 1 Downto 0);
    Signal FinalRowAddress_Flush      : Std_Logic_Vector(12 Downto 0);
    Signal FinalColumnAddress_Flush   : Std_Logic_Vector(12 Downto 0);
    Signal FinalBankAddress_Replace   : Std_Logic_Vector( 1 Downto 0);
    Signal FinalRowAddress_Replace    : Std_Logic_Vector(12 Downto 0);
    Signal FinalColumnAddress_Replace : Std_Logic_Vector(12 Downto 0);

    Signal ToResolve_FlushAddress     : Std_Logic_Vector(19 Downto 0);
    Signal ToResolve_ReplaceAddress   : Std_Logic_Vector(31 Downto 0);
    Signal ToResolve_FlushIndex       : Std_Logic_Vector(10 Downto 0);

    Signal DataCache_WbClk    : Std_Logic;
    Signal DataCache_WbDin    : Std_Logic_Vector(31 Downto 0);
    Signal DataCache_WbDout   : Std_Logic_Vector(31 Downto 0);
    Signal DataCache_WbAddr   : Std_Logic_Vector(10 Downto 0);
    Signal DataCache_WbByteW  : Std_Logic_Vector( 3 Downto 0);
    Signal DataCache_WbWe     : Std_Logic;

    Signal DataCache_MemClk   : Std_Logic;
    Signal DataCache_MemDin   : Std_Logic_Vector( 7 Downto 0);
    Signal DataCache_MemDout  : Std_Logic_Vector( 7 Downto 0);
    Signal DataCache_MemAddr  : Std_Logic_Vector(12 Downto 0);
    Signal DataCache_MemByteW : Std_Logic;
    Signal DataCache_MemWe    : Std_Logic;

    Signal TagCache_WbClk     : Std_Logic;
    Signal TagCache_WbDin     : Std_Logic_Vector(19 Downto 0);
    Signal TagCache_WbDout    : Std_Logic_Vector(19 Downto 0);
    Signal TagCache_WbAddr    : Std_Logic_Vector( 9 Downto 0);
    Signal TagCache_WbWe      : Std_Logic;

    Signal TagCache_MemClk    : Std_Logic;
    Signal TagCache_MemDin    : Std_Logic_Vector(19 Downto 0);
    Signal TagCache_MemDout   : Std_Logic_Vector(19 Downto 0);
    Signal TagCache_MemAddr   : Std_Logic_Vector( 9 Downto 0);
    Signal TagCache_MemWe     : Std_Logic;

    Signal CmdCache_WbClk     : Std_Logic;
    Signal CmdCache_WbDin     : Std_Logic_Vector(31 Downto 0);
    Signal CmdCache_WbDout    : Std_Logic_Vector(31 Downto 0);
    Signal CmdCache_WbAddr    : Std_Logic_Vector( 1 Downto 0);
    Signal CmdCache_WbWe      : Std_Logic;

    Signal CmdCache_MemClk    : Std_Logic;
    Signal CmdCache_MemDin    : Std_Logic_Vector(31 Downto 0);
    Signal CmdCache_MemDout   : Std_Logic_Vector(31 Downto 0);
    Signal CmdCache_MemAddr   : Std_Logic_Vector( 1 Downto 0);
    Signal CmdCache_MemWe     : Std_Logic;

    Signal SdramDataout     : Std_Logic_Vector( 7 Downto 0);
    Signal SdramDataIn      : Std_Logic_Vector( 7 Downto 0);
    Signal SdramDataEnable  : Std_Logic_Vector( 7 Downto 0);
    Signal SdramAddress     : Std_Logic_Vector(12 Downto 0);
    Signal SdramBankAddress : Std_Logic_Vector( 1 Downto 0);
    Signal SdramWrite       : Std_Logic;
    Signal SdramByteEnable  : Std_Logic;
    Signal SdramClockEnable : Std_Logic;
    Signal SdramNRas        : Std_Logic;
    Signal SdramNCas        : Std_Logic;
    Signal SdramChipSelect  : Std_Logic;

    Signal TimerValue_Refresh : Std_Logic_Vector(10 Downto 0);

-- -----------------------------------------------------------------------------
Begin
-- -----------------------------------------------------------------------------


    -- -------------------------------------------------------------------------
    U_DataCache : Configurable_ReplaceWith_Designator_DataCache Port Map (
         CLKA     => DataCache_WbClk,
         DINA     => DataCache_WbDin,
         DOUTA    => DataCache_WbDout,
         ADDRA    => DataCache_WbAddr,
         BYTE_WEA => DataCache_WbByteW,
         WEA      => DataCache_WbWe,

         CLKB     => DataCache_MemClk,
         DINB     => DataCache_MemDin,
         DOUTB    => DataCache_MemDout,
         ADDRB    => DataCache_MemAddr,
         BYTE_WEB => DataCache_MemByteW,
         WEB      => DataCache_MemWe
    );
    -- -------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    U_TagCache : Configurable_ReplaceWith_Designator_TagCache Port Map (
        CLKA  => TagCache_WbClk,
        DINA  => TagCache_WbDin,
        DOUTA => TagCache_WbDout,
        ADDRA => TagCache_WbAddr,
        WEA   => TagCache_WbWe,

        CLKB  => TagCache_MemClk,
        DINB  => TagCache_MemDin,
        DOUTB => TagCache_MemDout,
        ADDRB => TagCache_MemAddr,
        WEB   => TagCache_MemWe
    );
    -- -------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    U_CommandCache : Configurable_ReplaceWith_Designator_CommandPipe Port Map (
        CLKA  => CmdCache_WbClk,
        DINA  => CmdCache_WbDin,
        DOUTA => CmdCache_WbDout,
        ADDRA => CmdCache_WbAddr,
        WEA   => CmdCache_WbWe,

        CLKB  => CmdCache_MemClk,
        DINB  => CmdCache_MemDin,
        DOUTB => CmdCache_MemDout,
        ADDRB => CmdCache_MemAddr,
        WEB   => CmdCache_MemWe
    );
    -- -------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    U_WishboneInterface : Configurable_ReplaceWith_Designator_WishboneInterface Port Map (
        -- Cache Command Interface
        CACHE_CMD_CLK     => CmdCache_WbClk,
        CACHE_CMD_WE_O    => CmdCache_WbWe,
        CACHE_CMD_DAT_I   => CmdCache_WbDout,
        CACHE_CMD_DAT_O   => CmdCache_WbDin,
        CACHE_CMD_ADR_O   => CmdCache_WbAddr,

        -- Dual Port RAM Wishbone Data Cache Interface
        DP_RAM_DAT_CLK_O  => DataCache_WbClk,
        DP_RAM_DAT_RST_O  => Open,
        DP_RAM_DAT_DATA_I => DataCache_WbDout,
        DP_RAM_DAT_DATA_O => DataCache_WbDin,
        DP_RAM_DAT_ADR_O  => DataCache_WbAddr,
        DP_RAM_DAT_BE_O   => DataCache_WbByteW,
        DP_RAM_DAT_WE_O   => DataCache_WbWe,

        -- Dual Port RAM Cache Tag Interface
        DP_RAM_TAG_CLK_O  => TagCache_WbClk,
        DP_RAM_TAG_RST_O  => Open,
        DP_RAM_TAG_DATA_I => TagCache_WbDout,
        DP_RAM_TAG_DATA_O => TagCache_WbDin,
        DP_RAM_TAG_ADR_O  => TagCache_WbAddr,
        DP_RAM_TAG_BE_O   => Open,
        DP_RAM_TAG_WE_O   => TagCache_WbWe,

        -- Wishbone Interface
        WB_CLK_I          => SDRAM_CLK_I,
        WB_RST_I          => SDRAM_RST_I,
        WB_STB_I          => SDRAM_STB_I,
        WB_CYC_I          => SDRAM_CYC_I,
        WB_ACK_O          => SDRAM_ACK_O,
        WB_ADR_I          => SDRAM_ADR_I,
        WB_DAT_I          => SDRAM_DAT_I,
        WB_DAT_O          => SDRAM_DAT_O,
        WB_SEL_I          => SDRAM_SEL_I,
        WB_WE_I           => SDRAM_WE_I
    );
    -- -------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    U_SdramEngine : ReplaceWith_SdramEngineName Port Map (
        -- System
        MEM_CLK                         => MEM_CLK,
        MEM_RST                         => SDRAM_RST_I,
        -- Data Cache
        CACHE_DAT_CLK_O                 => DataCache_MemClk,
        CACHE_DAT_BE_O                  => DataCache_MemByteW,
        CACHE_DAT_WE_O                  => DataCache_MemWe,
        CACHE_DAT_ADR_O                 => DataCache_MemAddr,
        CACHE_DAT_DATA_I                => DataCache_MemDout,
        CACHE_DAT_DATA_O                => DataCache_MemDin,
        -- Tag Cache
        CACHE_TAG_CLK_O                 => TagCache_MemClk,
        CACHE_TAG_WE_O                  => TagCache_MemWe,
        CACHE_TAG_ADR_O                 => TagCache_MemAddr,
        CACHE_TAG_DATA_I                => TagCache_MemDout,
        CACHE_TAG_DATA_O                => TagCache_MemDin,
        -- Command Pipe
        CACHE_CMD_CLK_O                 => CmdCache_MemClk,
        CACHE_CMD_WE_O                  => CmdCache_MemWe,
        CACHE_CMD_ADR_O                 => CmdCache_MemAddr,
        CACHE_CMD_DATA_O                => CmdCache_MemDin,
        CACHE_CMD_DATA_I                => CmdCache_MemDout,
        -- Timer Configuration
        TIMER_VALUE_TRP                 => TimerValue_tRP,
        TIMER_VALUE_TWR                 => TimerValue_tWR,
        TIMER_VALUE_TRFC                => TimerValue_tRFC,
        TIMER_VALUE_TMRD                => TimerValue_tMRD,
        TIMER_VALUE_TRCD                => TimerValue_tRCD,
        TIMER_VALUE_TCL                 => TimerValue_tCL,
        -- Memory Mode Register
        LOAD_MODE_REGISTER              => LoadModeRegister,
        -- Address Resover for diff memory geometries
        RESOLVED_BANK_ADDRESS_FLUSH     => FinalBankAddress_Flush,
        RESOLVED_ROW_ADDRESS_FLUSH      => FinalRowAddress_Flush,
        RESOLVED_COLUMN_ADDRESS_FLUSH   => FinalColumnAddress_Flush,
        RESOLVED_BANK_ADDRESS_REPLACE   => FinalBankAddress_Replace,
        RESOLVED_ROW_ADDRESS_REPLACE    => FinalRowAddress_Replace,
        RESOLVED_COLUMN_ADDRESS_REPLACE => FinalColumnAddress_Replace,

        RESOLVER_FLUSH_ADDRESS          => ToResolve_FlushAddress,
        RESOLVER_REPLACE_ADDRESS        => ToResolve_ReplaceAddress,
        RESOLVER_FLUSH_INDEX            => ToResolve_FlushIndex,

        -- Refresh engine reload value
        REFRESH_RELOAD_VALUE            => TimerValue_Refresh,
        -- Memory Bus
        SDRAM_DATA_O                    => SdramDataOut,
        SDRAM_DATA_I                    => SdramDataIn,
        SDRAM_DATA_E                    => SdramDataEnable,
        SDRAM_A                         => SdramAddress,
        SDRAM_BA                        => SdramBankAddress,
        SDRAM_W                         => SdramWrite,
        SDRAM_BE                        => SdramByteEnable,
        SDRAM_CKE                       => SdramClockEnable,
        SDRAM_RAS                       => SdramNRas,
        SDRAM_CAS                       => SdramNCas,
        SDRAM_E                         => SdramChipSelect
    );
    -- -------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    -- Provide timing configuration
    -- -------------------------------------------------------------------------
    TimerValue_tRP     <= cTimerValue_tRP;
    TimerValue_tWR     <= cTimerValue_tWR;
    TimerValue_tRFC    <= cTimerValue_tRFC;
    TimerValue_tMRD    <= cTimerValue_tMRD;
    TimerValue_tRCD    <= cTimerValue_tRCD;
    TimerValue_tCL     <= cTimerValue_tCL;
    TimerValue_Refresh <= cTimerValue_Refresh;
    LoadModeRegister   <= cLoadModeRegister;
    -- -------------------------------------------------------------------------

    -- -------------------------------------------------------------------------
    -- Resolve addresses
    -- -------------------------------------------------------------------------
    FinalBankAddress_Flush     <= ToResolve_FlushAddress(cFlushBankAddressMsb Downto cFlushBankAddressLsb);
    FinalRowAddress_Flush      <= ToResolve_FlushAddress(cFlushRowAddressMsb Downto cFlushRowAddressLsb) & ToResolve_FlushIndex(cFlushIndexRowAddressMsb Downto cFlushIndexRowAddressLsb);
    FinalColumnAddress_Flush   <= cFlushColumnAddressPadding & ToResolve_FlushIndex(cFlushIndexMsb Downto cFlushIndexLsb) & "00";

    FinalBankAddress_Replace   <= ToResolve_ReplaceAddress(cReplaceBankAddressMsb Downto cReplaceBankAddressLsb);
    FinalRowAddress_Replace    <= ToResolve_ReplaceAddress(cReplaceRowAddressMsb Downto cReplaceRowAddressLsb);
    FinalColumnAddress_Replace <= cReplaceColumnAddressPadding & ToResolve_ReplaceAddress(cReplaceColumnAddressMsb Downto cReplaceColumnAddressLsb) & "00";
    -- -------------------------------------------------------------------------


    -- -------------------------------------------------------------------------
    Process(SdramDataOut, SdramDataEnable)
    Begin
        For I In 0 To 7 Loop
            If SdramDataEnable(I) = '1' Then
                MEM_D(I) <= SdramDataOut(I);
            Else
                MEM_D(I) <= 'Z';
            End If;
        End Loop;
    End Process;
    SdramDataIn <= MEM_D;

    MEM_A  <= cSdramMemoryAddressPadding & SdramBankAddress & SdramAddress & '0';
    -- -------------------------------------------------------------------------
    MEM_W          <= SdramWrite;
    MEM_BE         <= SdramByteEnable;
    MEM_SDRAM_CKE  <= SdramClockEnable;
    MEM_SDRAM_RAS  <= SdramNRas;
    MEM_SDRAM_CAS  <= SdramNCas;
    MEM_SDRAM_E    <= SdramChipSelect;
    -- -------------------------------------------------------------------------

-- -----------------------------------------------------------------------------
End RTL;
-- -----------------------------------------------------------------------------

