Library IEEE;
Use IEEE.Std_Logic_1164.all;
Use IEEE.std_logic_unsigned.all;

--------------------------------------------------------------------------------
Entity Configurable_ReplaceWith_Designator Is Port
   (
     ----------------------------------------------------------------------------- IncludeIf_2_Device_X_8
     SRAM0_D   : InOut Std_Logic_Vector( 7 DownTo 0);                           -- IncludeIf_2_Device_X_8
     SRAM1_D   : InOut Std_Logic_Vector( 7 DownTo 0);                           -- IncludeIf_2_Device_X_8
     SRAM_A    : Out   Std_Logic_Vector(ReplaceWith_AddressWidth-1 DownTo 0);   -- IncludeIf_2_Device_X_8
     SRAM0_OE  : Out   Std_Logic;                                               -- IncludeIf_2_Device_X_8
     SRAM1_OE  : Out   Std_Logic;                                               -- IncludeIf_2_Device_X_8
     SRAM0_WE  : Out   Std_Logic;                                               -- IncludeIf_2_Device_X_8
     SRAM1_WE  : Out   Std_Logic;                                               -- IncludeIf_2_Device_X_8
     SRAM_CE   : Out   Std_Logic;                                               -- IncludeIf_2_Device_X_8
     ----------------------------------------------------------------------------- IncludeIf_2_Device_X_8
     ----------------------------------------------------------------------------- IncludeIf_1_Device_X_16
     SRAM_D    : InOut Std_Logic_Vector(15 DownTo 0);                           -- IncludeIf_1_Device_X_16
     SRAM_A    : Out   Std_Logic_Vector(ReplaceWith_AddressWidth-1 DownTo 0);   -- IncludeIf_1_Device_X_16
     SRAM_OE   : Out   Std_Logic;                                               -- IncludeIf_1_Device_X_16
     SRAM_WE   : Out   Std_Logic;                                               -- IncludeIf_1_Device_X_16
     SRAM_CE   : Out   Std_Logic;                                               -- IncludeIf_1_Device_X_16
     SRAM_LB   : Out   Std_Logic;                                               -- IncludeIf_1_Device_X_16
     SRAM_UB   : Out   Std_Logic;                                               -- IncludeIf_1_Device_X_16
     ----------------------------------------------------------------------------- IncludeIf_1_Device_X_16

     ---------------------------------------------------------------------------
     -- Wishbone Slave Port
     ---------------------------------------------------------------------------
     CLK_I      : In    Std_Logic;
     RST_I      : In    Std_Logic;
     CYC_I      : In    Std_Logic;
     STB_I      : In    Std_Logic;
     ACK_O      : Out   Std_Logic;
     ADR_I      : In    Std_Logic_Vector(ReplaceWith_InputAddressWidth-1 DownTo 0);
     DAT_I      : In    Std_Logic_Vector(31 DownTo 0);
     DAT_O      : Out   Std_Logic_Vector(31 DownTo 0);
     WE_I       : In    Std_Logic;
     SEL_I      : In    Std_Logic_Vector(3  DownTo 0)
   );

End Configurable_ReplaceWith_Designator;
--------------------------------------------------------------------------------
    
---------------------------------------------------------------------------------- IncludeIf_2_Device_X_8
--                   ByteLanes[3..0]   Cycle  WE0  WE1  CE       OE  A0         -- IncludeIf_2_Device_X_8
--                                     B3     B2   B1   B0       OE  A0         -- IncludeIf_2_Device_X_8
----------------------------------------------------------------------------    -- IncludeIf_2_Device_X_8
--Read             |     XXXX          0      1    1    0        0   0          -- IncludeIf_2_Device_X_8
--                 |                   1      1    1    0        0   1          -- IncludeIf_2_Device_X_8
--None             |     XXXX          0      1    1    1        1   0          -- IncludeIf_2_Device_X_8
--                 |                   1      1    1    1        1   1          -- IncludeIf_2_Device_X_8
--Write word       |     1111          0      0    0    0        1   0          -- IncludeIf_2_Device_X_8
--                 |                   1      0    0    0        1   1          -- IncludeIf_2_Device_X_8
--Write low half   |     0011          0      1    1    0        1   0          -- IncludeIf_2_Device_X_8
--                 |                   1      0    0    0        1   1          -- IncludeIf_2_Device_X_8
--Write high half  |     1100          0      0    0    0        1   0          -- IncludeIf_2_Device_X_8
--                 |                   1      1    1    0        1   1          -- IncludeIf_2_Device_X_8
--Write Byte0      |     0001          0      0    1    0        1   0          -- IncludeIf_2_Device_X_8
--                 |                   1      1    1    0        1   1          -- IncludeIf_2_Device_X_8
--Write Byte1      |     0010          0      1    0    0        1   0          -- IncludeIf_2_Device_X_8
--                 |                   1      1    1    0        1   1          -- IncludeIf_2_Device_X_8
--Write Byte2      |     0100          0      1    1    0        1   0          -- IncludeIf_2_Device_X_8
--                 |                   1      0    1    0        1   1          -- IncludeIf_2_Device_X_8
--Write Byte3      |     1000          0      1    1    0        1   0          -- IncludeIf_2_Device_X_8
--                 |                   1      1    0    0        1   0          -- IncludeIf_2_Device_X_8
--Write Bytes012   |     0111          0      0    0    0        1   0          -- IncludeIf_2_Device_X_8
--                 |                   1      0    1    0        1   1          -- IncludeIf_2_Device_X_8
--Write Bytes123   |     1110          0      1    0    0        1   0          -- IncludeIf_2_Device_X_8
--                 |                   1      0    0    0        1   0          -- IncludeIf_2_Device_X_8
--Write Bytes12    |     0110          0      1    0    0        1   0          -- IncludeIf_2_Device_X_8
--                 |                   1      0    1    0        1   1          -- IncludeIf_2_Device_X_8
---------------------------------------------------------------------------------- IncludeIf_2_Device_X_8
--                   ByteLanes[3..0]   Cycle  LB   UB   CE       OE  WE  A0     -- IncludeIf_1_Device_X_16
--                                     B3     B2   B1   B0       OE  WE  A0     -- IncludeIf_1_Device_X_16
---------------------------------------------------------------------------------- IncludeIf_1_Device_X_16
--Read             |     XXXX          0      0    0    0        0   1   0      -- IncludeIf_1_Device_X_16
--                 |                   1      0    0    0        0   1   1      -- IncludeIf_1_Device_X_16
--None             |     XXXX          0      1    1    1        1   1   0      -- IncludeIf_1_Device_X_16
--                 |                   1      1    1    1        1   1   1      -- IncludeIf_1_Device_X_16
--Write word       |     1111          0      0    0    0        1   0   0      -- IncludeIf_1_Device_X_16
--                 |                   1      0    0    0        1   0   1      -- IncludeIf_1_Device_X_16
--Write low half   |     0011          0      1    1    1        1   0   0      -- IncludeIf_1_Device_X_16
--                 |                   1      0    0    0        1   0   1      -- IncludeIf_1_Device_X_16
--Write high half  |     1100          0      0    0    0        1   0   0      -- IncludeIf_1_Device_X_16
--                 |                   1      1    1    1        1   0   1      -- IncludeIf_1_Device_X_16
--Write Byte0      |     0001          0      0    1    0        1   0   0      -- IncludeIf_1_Device_X_16
--                 |                   1      1    1    1        1   0   1      -- IncludeIf_1_Device_X_16
--Write Byte1      |     0010          0      1    0    0        1   0   0      -- IncludeIf_1_Device_X_16
--                 |                   1      1    1    1        1   0   1      -- IncludeIf_1_Device_X_16
--Write Byte2      |     0100          0      1    1    1        1   0   0      -- IncludeIf_1_Device_X_16
--                 |                   1      0    1    0        1   0   1      -- IncludeIf_1_Device_X_16
--Write Byte3      |     1000          0      1    1    1        1   0   0      -- IncludeIf_1_Device_X_16
--                 |                   1      1    0    1        1   0   0      -- IncludeIf_1_Device_X_16
--Write Bytes012   |     0111          0      0    0    0        1   0          -- IncludeIf_1_Device_X_16
--                 |                   1      0    1    0        1   0          -- IncludeIf_1_Device_X_16
--Write Bytes123   |     1110          0      1    0    0        1   0          -- IncludeIf_1_Device_X_16
--                 |                   1      0    0    0        1   0          -- IncludeIf_1_Device_X_16
--Write Bytes12    |     0110          0      1    0    0        1   0          -- IncludeIf_1_Device_X_16
--                 |                   1      0    1    0        1   0          -- IncludeIf_1_Device_X_16
--------------------------------------------------------------------------------
Architecture RTL of Configurable_ReplaceWith_Designator Is
    Type TState Is
        (
         State_AddressSetup  ,
         State_Write_Cycle0_Setup,
         State_Write_Cycle0_Pulse   ,     -- WE goes low during this state
         State_Write_Cycle0_Wait    ,
         State_Write_Cycle1_Setup   ,
         State_Write_Cycle1_Pulse   ,     -- WE goes low during this state
         State_Write_Cycle1_ACK     ,     -- WE goes high. Must keep seperate form addresses etc in case WE goes high before address, CE etc

         State_Read_Cycle0_Latch    ,
         State_Read_Cycle1_Setup    ,
         State_Read_Cycle1_ACK
        );

    Signal   State      : TState;
    Signal   State_Next : TState;

    SubType  TCycleAndByteLane   Is Std_Logic_Vector(4 DownTo 0);
    SubType  TOutBits            Is Std_Logic_Vector(2 DownTo 0);

    ----------------------------------------------------------------------------
    -- Constants for byte lanes - always the same
    ----------------------------------------------------------------------------
    Constant Write_Word_Cycle0           : TCycleAndByteLane := "01111";
    Constant Write_Word_Cycle1           : TCycleAndByteLane := "11111";
    Constant Write_LoHalf_Cycle0         : TCycleAndByteLane := "00011";
    Constant Write_LoHalf_Cycle1         : TCycleAndByteLane := "10011";
    Constant Write_HiHalf_Cycle0         : TCycleAndByteLane := "01100";
    Constant Write_HiHalf_Cycle1         : TCycleAndByteLane := "11100";
    Constant Write_Byte0_Cycle0          : TCycleAndByteLane := "00001";
    Constant Write_Byte0_Cycle1          : TCycleAndByteLane := "10001";
    Constant Write_Byte1_Cycle0          : TCycleAndByteLane := "00010";
    Constant Write_Byte1_Cycle1          : TCycleAndByteLane := "10010";
    Constant Write_Byte2_Cycle0          : TCycleAndByteLane := "00100";
    Constant Write_Byte2_Cycle1          : TCycleAndByteLane := "10100";
    Constant Write_Byte3_Cycle0          : TCycleAndByteLane := "01000";
    Constant Write_Byte3_Cycle1          : TCycleAndByteLane := "11000";
    Constant Write_Bytes012_Cycle0       : TCycleAndByteLane := "00111";
    Constant Write_Bytes012_Cycle1       : TCycleAndByteLane := "10111";
    Constant Write_Bytes123_Cycle0       : TCycleAndByteLane := "01110";
    Constant Write_Bytes123_Cycle1       : TCycleAndByteLane := "11110";
    Constant Write_Bytes12_Cycle0        : TCycleAndByteLane := "00110";
    Constant Write_Bytes12_Cycle1        : TCycleAndByteLane := "10110";
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    -- Mapping of byte lanes to SRAM Controls
    ----------------------------------------------------------------------------
    Constant OutBits_Write_Word            : TOutBits  := "000";

    Constant OutBits_Write_Cycle0_LoHalf   : TOutBits  := "000";                  -- IncludeIf_2_Device_X_8
    Constant OutBits_Write_Cycle1_LoHalf   : TOutBits  := "110";                  -- IncludeIf_2_Device_X_8
    Constant OutBits_Write_Cycle0_HiHalf   : TOutBits  := "110";                  -- IncludeIf_2_Device_X_8
    Constant OutBits_Write_Cycle1_HiHalf   : TOutBits  := "000";                  -- IncludeIf_2_Device_X_8
    Constant OutBits_Write_Cycle0_Byte0    : TOutBits  := "010";                  -- IncludeIf_2_Device_X_8
    Constant OutBits_Write_Cycle1_Byte0    : TOutBits  := "110";                  -- IncludeIf_2_Device_X_8
    Constant OutBits_Write_Cycle0_Byte1    : TOutBits  := "100";                  -- IncludeIf_2_Device_X_8
    Constant OutBits_Write_Cycle1_Byte1    : TOutBits  := "110";                  -- IncludeIf_2_Device_X_8
    Constant OutBits_Write_Cycle0_Byte2    : TOutBits  := "110";                  -- IncludeIf_2_Device_X_8
    Constant OutBits_Write_Cycle1_Byte2    : TOutBits  := "010";                  -- IncludeIf_2_Device_X_8
    Constant OutBits_Write_Cycle0_Byte3    : TOutBits  := "110";                  -- IncludeIf_2_Device_X_8
    Constant OutBits_Write_Cycle1_Byte3    : TOutBits  := "100";                  -- IncludeIf_2_Device_X_8
    
    Constant OutBits_Write_Cycle0_Bytes012 : TOutBits  := "000";                  -- IncludeIf_2_Device_X_8
    Constant OutBits_Write_Cycle1_Bytes012 : TOutBits  := "010";                  -- IncludeIf_2_Device_X_8
    Constant OutBits_Write_Cycle0_Bytes123 : TOutBits  := "100";                  -- IncludeIf_2_Device_X_8
    Constant OutBits_Write_Cycle1_Bytes123 : TOutBits  := "000";                  -- IncludeIf_2_Device_X_8
    Constant OutBits_Write_Cycle0_Bytes12  : TOutBits  := "100";                  -- IncludeIf_2_Device_X_8
    Constant OutBits_Write_Cycle1_Bytes12  : TOutBits  := "010";                  -- IncludeIf_2_Device_X_8
    
    
                                                                                -- IncludeIf_2_Device_X_8
    Constant OutBits_Read                : TOutBits  := "000";                  -- IncludeIf_2_Device_X_8
    Constant OutBits_None                : TOutBits  := "111";                  -- IncludeIf_2_Device_X_8
    Constant OutBits_Write_Cycle0_LoHalf : TOutBits  := "000";                  -- IncludeIf_1_Device_X_16
    Constant OutBits_Write_Cycle1_LoHalf : TOutBits  := "111";                  -- IncludeIf_1_Device_X_16
    Constant OutBits_Write_Cycle0_HiHalf : TOutBits  := "111";                  -- IncludeIf_1_Device_X_16
    Constant OutBits_Write_Cycle1_HiHalf : TOutBits  := "000";                  -- IncludeIf_1_Device_X_16
    Constant OutBits_Write_Cycle0_Byte0  : TOutBits  := "010";                  -- IncludeIf_1_Device_X_16
    Constant OutBits_Write_Cycle1_Byte0  : TOutBits  := "111";                  -- IncludeIf_1_Device_X_16
    Constant OutBits_Write_Cycle0_Byte1  : TOutBits  := "100";                  -- IncludeIf_1_Device_X_16
    Constant OutBits_Write_Cycle1_Byte1  : TOutBits  := "111";                  -- IncludeIf_1_Device_X_16
    Constant OutBits_Write_Cycle0_Byte2  : TOutBits  := "111";                  -- IncludeIf_1_Device_X_16
    Constant OutBits_Write_Cycle1_Byte2  : TOutBits  := "010";                  -- IncludeIf_1_Device_X_16
    Constant OutBits_Write_Cycle0_Byte3  : TOutBits  := "111";                  -- IncludeIf_1_Device_X_16
    Constant OutBits_Write_Cycle1_Byte3  : TOutBits  := "100";                  -- IncludeIf_1_Device_X_16
    
    Constant OutBits_Write_Cycle0_Bytes012 : TOutBits := "000";                  -- IncludeIf_1_Device_X_16
    Constant OutBits_Write_Cycle1_Bytes012 : TOutBits := "010";                  -- IncludeIf_1_Device_X_16
    Constant OutBits_Write_Cycle0_Bytes123 : TOutBits := "100";                  -- IncludeIf_1_Device_X_16
    Constant OutBits_Write_Cycle1_Bytes123 : TOutBits := "000";                  -- IncludeIf_1_Device_X_16
    Constant OutBits_Write_Cycle0_Bytes12  : TOutBits := "100";                  -- IncludeIf_1_Device_X_16
    Constant OutBits_Write_Cycle1_Bytes12  : TOutBits := "010";                  -- IncludeIf_1_Device_X_16

                                                                                -- IncludeIf_1_Device_X_16
    Constant OutBits_Read                : TOutBits  := "000";                  -- IncludeIf_1_Device_X_16
    Constant OutBits_None                : TOutBits  := "111";                  -- IncludeIf_1_Device_X_16

    Signal   OutBits                     : TOutBits;
    Signal   OutBits_Write               : TOutBits;

    Signal   Request                     : Std_Logic;
    Signal   RequestRead                 : Std_Logic;
    Signal   RequestWrite                : Std_Logic;

    Signal   Cycle_Number                : Std_Logic;

    Signal   WriteActive_Cycle           : Std_Logic;
    Signal   WE                          : Std_Logic;
    Signal   OE                          : Std_Logic;
    Signal   ACK                         : Std_Logic;

    Signal   CycleAndSel                 : TCycleAndByteLane;

    Signal   OutData_Latched             : Std_Logic_Vector(15 Downto 0);
    Signal   ADDR                        : Std_Logic_Vector(ReplaceWith_InputAddressWidth-3 Downto 0);

    Constant cTimer_Zero            : Std_Logic_Vector(4 Downto 0) := "00000";
    Constant cTimer_WriteSetup      : Std_Logic_Vector(4 Downto 0) := ReplaceWith_SramWriteSetup ;
    Constant cTimer_WritePulse      : Std_Logic_Vector(4 Downto 0) := ReplaceWith_SramWritePulse ;
    Constant cTimer_WriteHold       : Std_Logic_Vector(4 Downto 0) := ReplaceWith_SramWriteHold ;
    Constant cTimer_ReadSetup       : Std_Logic_Vector(4 Downto 0) := ReplaceWith_SramReadSetup ;

    Signal Timer                  : Std_Logic_Vector(4 Downto 0);
    Signal TimerIsZero            : Std_Logic;

    Signal LoadTimer_ReadSetup    : Std_Logic;
    Signal LoadTimer_WriteSetup   : Std_Logic;
    Signal LoadTimer_WritePulse   : Std_Logic;
    Signal LoadTimer_WriteHold    : Std_Logic;

Begin
    ----------------------------------------------------------------------------
    ADDR <= ADR_I(ReplaceWith_InputAddressWidth-1 DownTo 2);
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    SRAM0_WE  <= OutBits(2) Or WE;                                              -- IncludeIf_2_Device_X_8
    SRAM1_WE  <= OutBits(1) Or WE;                                              -- IncludeIf_2_Device_X_8
    SRAM_LB   <= OutBits(2);                                                    -- IncludeIf_1_Device_X_16
    SRAM_UB   <= OutBits(1);                                                    -- IncludeIf_1_Device_X_16
    SRAM_CE   <= OutBits(0) Or (Not Request);
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    SRAM_A(ReplaceWith_AddressWidth-1 DownTo 1) <= ADDR(ReplaceWith_AddressWidth-2 DownTo 0); -- IncludeIf_2_Device_X_8
    SRAM_A(ReplaceWith_AddressWidth-1 DownTo 1) <= ADDR(ReplaceWith_AddressWidth-2 DownTo 0); -- IncludeIf_1_Device_X_16
    SRAM_A(0)           <= Cycle_Number;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    DAT_O(15 DownTo  0) <= OutData_Latched;
    DAT_O(23 DownTo 16) <= SRAM0_D;                                             -- IncludeIf_2_Device_X_8
    DAT_O(31 DownTo 24) <= SRAM1_D;                                             -- IncludeIf_2_Device_X_8
    DAT_O(31 DownTo 16) <= SRAM_D;                                              -- IncludeIf_1_Device_X_16
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    ACK_O <= ACK;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    Request      <= STB_I And CYC_I;
    RequestRead  <= STB_I And CYC_I And (Not WE_I);
    RequestWrite <= STB_I And CYC_I And (    WE_I);
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    SRAM0_OE <= OE;                                                             -- IncludeIf_2_Device_X_8
    SRAM1_OE <= OE;                                                             -- IncludeIf_2_Device_X_8
    SRAM_WE <= WE;                                                              -- IncludeIf_1_Device_X_16
    SRAM_OE <= OE;                                                              -- IncludeIf_1_Device_X_16
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    CycleAndSel <= Cycle_Number & SEL_I;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    Memory_OuptutControls:
    Process(SEL_I,
            DAT_I,
            State,
            OutBits_Write,
            WriteActive_Cycle,
            Cycle_Number,
            CycleAndSel)
    Begin
        If WriteActive_Cycle = '1' Then
            OutBits    <= OutBits_Write;
            OE         <= '1';
            If Cycle_Number = '0' Then
                SRAM0_D    <= DAT_I( 7 DownTo  0);                              -- IncludeIf_2_Device_X_8
                SRAM1_D    <= DAT_I(15 DownTo  8);                              -- IncludeIf_2_Device_X_8
                SRAM_D    <= DAT_I(15 DownTo  0);                               -- IncludeIf_1_Device_X_16
            Else
                SRAM0_D    <= DAT_I(23 DownTo 16);                              -- IncludeIf_2_Device_X_8
                SRAM1_D    <= DAT_I(31 DownTo 24);                              -- IncludeIf_2_Device_X_8
                SRAM_D    <= DAT_I(31 DownTo 16);                               -- IncludeIf_1_Device_X_16
            End If;
        Else
           OutBits    <= OutBits_Read;
           OE         <= '0';
           SRAM0_D    <= (Others => 'Z');                                       -- IncludeIf_2_Device_X_8
           SRAM1_D    <= (Others => 'Z');                                       -- IncludeIf_2_Device_X_8
           SRAM_D     <= (Others => 'Z');                                       -- IncludeIf_1_Device_X_16
        End If;

        Case CycleAndSel Is
           When Write_Word_Cycle0      => OutBits_Write <= OutBits_Write_Word;
           When Write_Word_Cycle1      => OutBits_Write <= OutBits_Write_Word;
           When Write_LoHalf_Cycle0    => OutBits_Write <= OutBits_Write_Cycle0_LoHalf;
           When Write_LoHalf_Cycle1    => OutBits_Write <= OutBits_Write_Cycle1_LoHalf;
           When Write_HiHalf_Cycle0    => OutBits_Write <= OutBits_Write_Cycle0_HiHalf;
           When Write_HiHalf_Cycle1    => OutBits_Write <= OutBits_Write_Cycle1_HiHalf;
           When Write_Byte0_Cycle0     => OutBits_Write <= OutBits_Write_Cycle0_Byte0;
           When Write_Byte0_Cycle1     => OutBits_Write <= OutBits_Write_Cycle1_Byte0;
           When Write_Byte1_Cycle0     => OutBits_Write <= OutBits_Write_Cycle0_Byte1;
           When Write_Byte1_Cycle1     => OutBits_Write <= OutBits_Write_Cycle1_Byte1;
           When Write_Byte2_Cycle0     => OutBits_Write <= OutBits_Write_Cycle0_Byte2;
           When Write_Byte2_Cycle1     => OutBits_Write <= OutBits_Write_Cycle1_Byte2;
           When Write_Byte3_Cycle0     => OutBits_Write <= OutBits_Write_Cycle0_Byte3;
           When Write_Byte3_Cycle1     => OutBits_Write <= OutBits_Write_Cycle1_Byte3;
           When Write_Bytes012_Cycle0  => OutBits_Write <= OutBits_Write_Cycle0_Bytes012;
           When Write_Bytes012_Cycle1  => OutBits_Write <= OutBits_Write_Cycle1_Bytes012;
           When Write_Bytes123_Cycle0  => OutBits_Write <= OutBits_Write_Cycle0_Bytes123;
           When Write_Bytes123_Cycle1  => OutBits_Write <= OutBits_Write_Cycle1_Bytes123;
           When Write_Bytes12_Cycle0   => OutBits_Write <= OutBits_Write_Cycle0_Bytes12;
           When Write_Bytes12_Cycle1   => OutBits_Write <= OutBits_Write_Cycle1_Bytes12;
           When Others                 => OutBits_Write <= OutBits_None;
        End Case;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    FSM_Combinational_Single:
    Process(RequestRead,
            RequestWrite,
            State,
            TimerIsZero)
    Begin
        ------------------------------------------------------------------------
        -- Default outputs
        WriteActive_Cycle <= '0';
        WE                <= '1';
        ACK               <= '0';
        Cycle_Number      <= '0';
        State_Next        <= State;
        LoadTimer_ReadSetup  <= '0';
        LoadTimer_WriteSetup <= '0';
        LoadTimer_WritePulse <= '0';
        LoadTimer_WriteHold  <= '0';
        ------------------------------------------------------------------------

        ------------------------------------------------------------------------
        Case State Is
           ---------------------------------------------------------------------
           When State_AddressSetup =>
                If RequestWrite = '1' Then                --Address is being setup in this phases
                   WriteActive_Cycle <= '1';
                   State_Next        <= State_Write_Cycle0_Pulse;               -- ExcludeIf_WriteSetup
                   State_Next        <= State_Write_Cycle0_Setup;               -- IncludeIf_WriteSetup
                   LoadTimer_WriteSetup <= '1';                                 -- IncludeIf_WriteSetup
                 ElsIf RequestRead = '1' Then
                   LoadTimer_ReadSetup <= '1';                                  -- IncludeIf_ReadSetup
                   State_Next <= State_Read_Cycle0_Latch;
                End If;
           ---------------------------------------------------------------------

           ---------------------------------------------------------------------
           When State_Write_Cycle0_Setup =>
                WriteActive_Cycle <= '1';
                If TimerIsZero='1' Then
                   State_Next <= State_Write_Cycle0_Pulse;
                   LoadTimer_WritePulse <= '1';                                 -- IncludeIf_WritePulse
                End If;
           When State_Write_Cycle0_Pulse =>
                WriteActive_Cycle <= '1';
                WE                <= '0';
                If TimerIsZero = '1' Then
                   State_Next        <= State_Write_Cycle0_Wait;
                   LoadTimer_WriteHold <= '1';                                  -- IncludeIf_WriteHold
                End If;

           When State_Write_Cycle0_Wait =>
                WriteActive_Cycle <= '1';
                If TimerIsZero = '1' Then
                   State_Next        <= State_Write_Cycle1_Setup;
                   LoadTimer_WriteSetup <= '1';                                 -- IncludeIf_WriteSetup
                End If;

           When State_Write_Cycle1_Setup =>
                WriteActive_Cycle <= '1';
                Cycle_Number      <= '1';
                If TimerIsZero = '1' Then
                   State_Next        <= State_Write_cycle1_Pulse;
                   LoadTimer_WritePulse <= '1';                                 -- IncludeIf_WritePulse
                End If;

           When State_Write_Cycle1_Pulse =>
                WriteActive_Cycle <= '1';
                WE                <= '0';
                Cycle_Number      <= '1';
                If TimerIsZero = '1' Then
                   State_Next        <= State_Write_Cycle1_ACK;
                   LoadTimer_WriteHold <= '1';                                  -- IncludeIf_WriteHold
                End If;

           When State_Write_Cycle1_ACK =>
                WriteActive_Cycle <= '1';
                Cycle_Number      <= '1';
                If TimerIsZero='1' Then
                   ACK               <= '1';
                   State_Next        <= State_AddressSetup;
                End If;
           ---------------------------------------------------------------------

           ---------------------------------------------------------------------
           When State_Read_Cycle0_Latch =>
                If TimerIsZero='1' Then
                   State_Next    <= State_Read_Cycle1_Setup;
                End If;

           When State_Read_Cycle1_Setup =>
                Cycle_Number  <= '1';
                LoadTimer_ReadSetup <= '1';                                  -- IncludeIf_ReadSetup
                State_Next    <= State_Read_Cycle1_ACK;

           When State_Read_Cycle1_ACK =>                    -- Assert ACK, Master will read the data one clock cycle later
                Cycle_Number  <= '1';
                If TimerIsZero='1' Then
                   ACK           <= '1';
                   State_Next    <= State_AddressSetup;
                End If;
           ---------------------------------------------------------------------

           When Others =>
                State_Next <= State_AddressSetup;

        End Case;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    FSM_Synchronous:
    Process(CLK_I)
    Begin
        If Rising_Edge(CLK_I) Then
           If RST_I = '1' Then
              State  <= State_AddressSetup;
           Else
              State <= State_Next;
           End If;
        End If;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    FSM_LatchData:
    Process(CLK_I)
    Begin
        If Rising_Edge(CLK_I) Then
           If State = State_Read_Cycle0_Latch Then
               OutData_Latched( 7 DownTo 0) <= SRAM0_D;                         -- IncludeIf_2_Device_X_8
               OutData_Latched(15 DownTo 8) <= SRAM1_D;                         -- IncludeIf_2_Device_X_8
               OutData_Latched <= SRAM_D;                                       -- IncludeIf_1_Device_X_16
           End If;
        End If;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    Process(CLK_I)
    Begin
        If Rising_Edge(CLK_I) Then
            If RST_I='1' Then
                Timer <= (Others=>'0');
            Else
                   If LoadTimer_ReadSetup ='1' Then Timer <= cTimer_ReadSetup;
                ElsIf LoadTimer_WritePulse='1' Then Timer <= cTimer_WritePulse;
                ElsIf LoadTimer_WriteSetup='1' Then Timer <= cTimer_WriteSetup;
                ElsIf LoadTimer_WriteHold ='1' Then Timer <= cTimer_WriteHold;
                ElsIf TimerIsZero         ='0' Then Timer <= Timer - 1;
                End If;
            End If;
        End If;
    End Process;
    TimerIsZero <= '1' When Timer=cTimer_Zero Else '0';
    ----------------------------------------------------------------------------



End RTL;
--------------------------------------------------------------------------------

