Library IEEE;
Use IEEE.Std_Logic_1164.all;
Use IEEE.std_logic_unsigned.all;

--------------------------------------------------------------------------------
Entity Configurable_ReplaceWith_Designator Is Port
   (
     ---------------------------------------------------------------------------
     MEM_D      : InOut Std_Logic_Vector( 7 DownTo 0);                          -- ExcludeIf_NotDisableAllPorts
     MEM_D      : InOut Std_Logic_Vector(31 Downto 0);                          -- ExcludeIf_DisableAllPorts
     MEM_A      : Out   Std_Logic_Vector(ReplaceWith_AddressWidth-1 DownTo 0);  -- IncludeIf_PartialAddress
     MEM_A      : Out   Std_Logic_Vector(23 Downto 0);                          -- IncludeIf_FullAddress
     MEM_OE     : Out   Std_Logic;
     MEM_W      : Out   Std_Logic;
     MEM_SRAM_E : Out   Std_Logic;
     MEM_BE     : Out   Std_Logic_Vector(3 Downto 0);                           -- ExcludeIf_DisableAllPorts


     MEM_SDRAM_CKE   : Out Std_Logic;   -- ExcludeIf_DisableAllPorts
     MEM_SDRAM_RAS   : Out Std_Logic;   -- ExcludeIf_DisableAllPorts
     MEM_SDRAM_CAS   : Out Std_Logic;   -- ExcludeIf_DisableAllPorts
     MEM_SDRAM_E     : Out Std_Logic;   -- ExcludeIf_DisableAllPorts
     MEM_FLASH_BUSY  : In  Std_Logic;   -- ExcludeIf_DisableAllPorts
     MEM_FLASH_RESET : Out Std_Logic;   -- ExcludeIf_DisableAllPorts
     MEM_FLASH_E     : Out Std_Logic;   -- ExcludeIf_DisableAllPorts
     ---------------------------------------------------------------------------

     ---------------------------------------------------------------------------
     -- Wishbone Slave Port
     ---------------------------------------------------------------------------
     SRAM_CLK_I      : In    Std_Logic;
     SRAM_RST_I      : In    Std_Logic;
     SRAM_CYC_I      : In    Std_Logic;
     SRAM_STB_I      : In    Std_Logic;
     SRAM_ACK_O      : Out   Std_Logic;
     SRAM_ADR_I      : In    Std_Logic_Vector(ReplaceWith_InputAddressWidth-1 DownTo 0);
     SRAM_DAT_I      : In    Std_Logic_Vector(31 DownTo 0);
     SRAM_DAT_O      : Out   Std_Logic_Vector(31 DownTo 0);
     SRAM_WE_I       : In    Std_Logic;
     SRAM_SEL_I      : In    Std_Logic_Vector( 3 DownTo 0)
   );

End Configurable_ReplaceWith_Designator;
--------------------------------------------------------------------------------

--------------------------------------------------------------------------------
--                   ByteLanes[3..0]   Cycle       WE   CE       OE  A1  A0
--                                     B3   B2     B1   B0       OE  A1  A0
------------------------------------------------------------------------------
--Read             |     XXXX          0    0      1    0        0   0   0
--                 |                   0    1      1    0        0   0   1
--                 |                   1    0      1    0        0   1   0
--                 |                   1    1      1    0        0   1   1
--None             |     XXXX          0    0      1    1        1   0   0
--                 |                   0    1      1    1        1   0   1
--                 |                   1    0      1    1        1   1   0
--                 |                   1    1      1    1        1   1   1
--Write word       |     1111          0    0      0    0        1   0   0
--                 |                   0    1      0    0        1   0   1
--                 |                   1    0      0    0        1   1   0
--                 |                   1    1      0    0        1   1   1
--Write low half   |     0011          0    0      0    0        1   0   0
--                 |                   0    1      0    0        1   0   1
--                 |                   1    0      1    0        1   1   0
--                 |                   1    1      1    0        1   1   1
--Write high half  |     1100          0    0      1    0        1   0   0
--                 |                   0    1      1    0        1   0   1
--                 |                   1    0      0    0        1   1   0
--                 |                   1    1      0    0        1   1   1
--Write Byte0      |     0001          0    0      0    0        1   0   0
--                 |                   0    1      1    0        1   0   1
--                 |                   1    0      1    0        1   1   0
--                 |                   1    1      1    0        1   1   1
--Write Byte1      |     0010          0    0      1    0        1   0   0
--                 |                   0    1      0    0        1   0   1
--                 |                   1    0      1    0        1   1   0
--                 |                   1    1      1    0        1   1   1
--Write Byte2      |     0100          0    0      1    0        1   0   0
--                 |                   0    1      1    0        1   0   1
--                 |                   1    0      0    0        1   1   0
--                 |                   1    1      1    0        1   1   1
--Write Byte3      |     1000          0    0      1    0        1   0   0
--                 |                   0    1      1    0        1   0   1
--                 |                   1    0      1    0        1   1   0
--                 |                   1    1      0    0        1   1   1

--------------------------------------------------------------------------------
Architecture RTL of Configurable_ReplaceWith_Designator Is
    Type TState Is
        (
         State_AddressSetup  ,

         State_Write_Cycle0_Pulse   ,     -- WE goes low during this state
         State_Write_Cycle0_Wait    ,
         State_Write_Cycle1_Setup   ,
         State_Write_Cycle1_Pulse   ,     -- WE goes low during this state
         State_Write_Cycle1_Wait    ,
         State_Write_Cycle2_Setup   ,
         State_Write_Cycle2_Pulse   ,     -- WE goes low during this state
         State_Write_Cycle2_Wait    ,
         State_Write_Cycle3_Setup   ,
         State_Write_Cycle3_Pulse   ,     -- WE goes low during this state
         State_Write_Cycle3_ACK     ,     -- WE goes high. Must keep seperate form addresses etc in case WE goes high before address, CE etc

         State_Read_Cycle0_Latch    ,
         State_Read_Cycle1_Setup    ,
         State_Read_Cycle1_Latch    ,
         State_Read_Cycle2_Setup    ,
         State_Read_Cycle2_Latch    ,
         State_Read_Cycle3_Setup    ,
         State_Read_Cycle3_ACK
        );

    SubType  TByteLane Is Std_Logic_Vector(3 Downto 0);

    Signal   State      : TState;
    Signal   State_Next : TState;

    Signal   Request                     : Std_Logic;
    Signal   RequestRead                 : Std_Logic;
    Signal   RequestWrite                : Std_Logic;

    Signal   Cycle_Number                : Std_Logic_Vector(1 DownTo 0);

    Signal   WriteActive_Cycle           : Std_Logic;
    Signal   WE_IfByteSel                : Std_Logic;
    Signal   ACK                         : Std_Logic;

    Signal   CurrentSel                  : Std_Logic;

    Signal   OutData_Latched             : Std_Logic_Vector(23 Downto 0);
    Signal   ADDR                        : Std_Logic_Vector(ReplaceWith_InputAddressWidth-3 Downto 0);


    Constant cAddressSetupDelay : Std_Logic_Vector(ReplaceWith_TimerWidth-1 Downto 0) := ReplaceWith_cAddressSetupDelay;
    Constant cWritePulseDelay   : Std_Logic_Vector(ReplaceWith_TimerWidth-1 Downto 0) := ReplaceWith_cWritePulseDelay;
    Constant cWriteHoldDelay    : Std_Logic_Vector(ReplaceWith_TimerWidth-1 Downto 0) := ReplaceWith_cWriteHoldDelay;
    Constant cTimerIsZero       : Std_Logic_Vector(ReplaceWith_TimerWidth-1 Downto 0) := ReplaceWith_cTimerIsZero;

    Signal  Timer               : Std_Logic_Vector(ReplaceWith_TimerWidth-1 Downto 0);
    Signal  TimerIsZero         : Std_Logic;
    Signal  LoadAddressDelay    : Std_Logic;
    Signal  LoadWritePulseDelay : Std_Logic;
    Signal  LoadWriteHoldDelay  : Std_Logic;
    Signal MEM_A_Signal : Std_Logic_Vector(23 Downto 0);

Begin
    ----------------------------------------------------------------------------
    Process(SRAM_CLK_I)
    Begin
        If Rising_Edge(SRAM_CLK_I) Then
            If SRAM_RST_I='1' Then
                Timer <= (Others=>'0');
            Else
                   If LoadAddressDelay='1'    Then Timer <= cAddressSetupDelay;
                ElsIf LoadWritePulseDelay='1' Then Timer <= cWritePulseDelay;
                ElsIf LoadWriteHoldDelay='1'  Then Timer <= cWriteHoldDelay;
                ElsIf TimerIsZero='0'         Then Timer <= Timer - 1;
                End If;
            End If;
        End If;
    End Process;
    TimerIsZero <= '1' When Timer=cTimerIsZero Else '0';
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    ADDR <= SRAM_ADR_I(ReplaceWith_InputAddressWidth-1 DownTo 2);
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    MEM_W   <= WE_IfByteSel Or Not CurrentSel;
    MEM_SRAM_E   <= Not Request;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    MEM_A(ReplaceWith_AddressWidth-1 DownTo 2) <= ADDR(ReplaceWith_AddressWidth-1 DownTo 2);-- IncludeIf_PartialAddress
    MEM_A( 1 Downto 0) <= Cycle_Number;                                                     -- IncludeIf_PartialAddress


    MEM_A_Signal   <= ReplaceWith_AddressPadding '0' & ADDR(ReplaceWith_AddressWidth-3 Downto 0) ;   -- IncludeIf_FullAddress
    MEM_A(23 DownTo 3) <= MEM_A_Signal(20 Downto 0);                                                -- IncludeIf_FullAddress
    MEM_A( 2 Downto 0) <= Cycle_Number & '0';                                                       -- IncludeIf_FullAddress
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    SRAM_DAT_O(23 DownTo  0) <= OutData_Latched;
    SRAM_DAT_O(31 DownTo 24) <= MEM_D(7 Downto 0);
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    SRAM_ACK_O <= ACK;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    Request      <= SRAM_STB_I And SRAM_CYC_I;
    RequestRead  <= SRAM_STB_I And SRAM_CYC_I And (Not SRAM_WE_I);
    RequestWrite <= SRAM_STB_I And SRAM_CYC_I And (    SRAM_WE_I);
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    MEM_OE <= WriteActive_Cycle;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    Select_SelLink:
    Process(SRAM_SEL_I,
            Cycle_Number)
    Begin
        Case Cycle_Number Is
           When "00"   => CurrentSel <= SRAM_SEL_I(0);
           When "01"   => CurrentSel <= SRAM_SEL_I(1);
           When "10"   => CurrentSel <= SRAM_SEL_I(2);
           When "11"   => CurrentSel <= SRAM_SEL_I(3);
           When Others => Null;
        End Case;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    Memory_OuptutControls:
    Process(SRAM_DAT_I,
            WriteActive_Cycle,
            Cycle_Number)
    Begin
        If WriteActive_Cycle = '1' Then
            Case Cycle_Number Is
               When "00"   => MEM_D(7 Downto 0) <= SRAM_DAT_I( 7 DownTo  0);
               When "01"   => MEM_D(7 Downto 0) <= SRAM_DAT_I(15 DownTo  8);
               When "10"   => MEM_D(7 Downto 0) <= SRAM_DAT_I(23 DownTo 16);
               When "11"   => MEM_D(7 Downto 0) <= SRAM_DAT_I(31 DownTo 24);
               When Others => Null;
            End Case;
        Else
           MEM_D     <= (Others => 'Z');
        End If;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    FSM_Combinational_Single:
    Process(RequestRead,
            RequestWrite,
            State,
            TimerIsZero)
    Begin
        ------------------------------------------------------------------------
        -- Default outputs
        WriteActive_Cycle   <= '0';
        WE_IfByteSel        <= '1';
        ACK                 <= '0';
        Cycle_Number        <= "00";
        State_Next          <= State;
        LoadWritePulseDelay <= '0';
        LoadAddressDelay    <= '0';
        LoadWriteHoldDelay  <= '0';
        ------------------------------------------------------------------------

        ------------------------------------------------------------------------
        Case State Is
           ---------------------------------------------------------------------
           When State_AddressSetup =>
                If RequestWrite = '1' Then                --Address is being setup in this phases
                   WriteActive_Cycle <= '1';
                   State_Next        <= State_Write_Cycle0_Pulse;
                   LoadWritePulseDelay <= '1';
                 ElsIf RequestRead = '1' Then
                   LoadAddressDelay <= '1';
                   State_Next <= State_Read_Cycle0_Latch;
                End If;
           ---------------------------------------------------------------------

           ---------------------------------------------------------------------
           When State_Write_Cycle0_Pulse =>
                WriteActive_Cycle <= '1';
                WE_IfByteSel                <= '0';
                If TimerIsZero='1' Then
                    State_Next        <= State_Write_Cycle0_Wait;
                    LoadWriteHoldDelay <= '1';
                End If;

           When State_Write_Cycle0_Wait =>
                WriteActive_Cycle <= '1';
                If TimerIsZero='1' Then
                    State_Next        <= State_Write_Cycle1_Setup;
                    LoadAddressDelay <= '1';
                End If;

           When State_Write_Cycle1_Setup =>
                WriteActive_Cycle <= '1';
                Cycle_Number      <= "01";
                If TimerIsZero='1' Then
                    State_Next        <= State_Write_Cycle1_Pulse;
                    LoadWritePulseDelay <= '1';
                End If;

           When State_Write_Cycle1_Pulse =>
                WriteActive_Cycle <= '1';
                WE_IfByteSel      <= '0';
                Cycle_Number      <= "01";
                If TimerIsZero='1' Then
                    State_Next        <= State_Write_Cycle1_Wait;
                    LoadWriteHoldDelay <= '1';
                End If;

           When State_Write_Cycle1_Wait =>
                WriteActive_Cycle <= '1';
                Cycle_Number      <= "01";
                If TimerIsZero='1' Then
                    State_Next        <= State_Write_Cycle2_Setup;
                    LoadAddressDelay <= '1';
                End If;

           When State_Write_Cycle2_Setup =>
                WriteActive_Cycle <= '1';
                Cycle_Number      <= "10";
                If TimerIsZero='1' Then
                    State_Next        <= State_Write_Cycle2_Pulse;
                    LoadWritePulseDelay <= '1';
                End if;

           When State_Write_Cycle2_Pulse =>
                WriteActive_Cycle <= '1';
                WE_IfByteSel      <= '0';
                Cycle_Number      <= "10";
                If TimerIsZero='1' Then
                    State_Next        <= State_Write_Cycle2_Wait;
                    LoadWriteHoldDelay <= '1';
                End If;

           When State_Write_Cycle2_Wait =>
                WriteActive_Cycle <= '1';
                Cycle_Number      <= "10";
                If TimerIsZero='1' Then
                    State_Next        <= State_Write_Cycle3_Setup;
                    LoadAddressDelay <= '1';
                End If;

           When State_Write_Cycle3_Setup =>
                WriteActive_Cycle <= '1';
                Cycle_Number      <= "11";
                If TimerIsZero='1' Then
                    State_Next        <= State_Write_Cycle3_Pulse;
                    LoadWritePulseDelay <= '1';
                End If;

           When State_Write_Cycle3_Pulse =>
                WriteActive_Cycle <= '1';
                WE_IfByteSel      <= '0';
                Cycle_Number      <= "11";
                If TimerIsZero='1' Then
                    State_Next        <= State_Write_Cycle3_ACK;
                    LoadWriteHoldDelay <= '1';
                End If;

           When State_Write_Cycle3_ACK =>
                WriteActive_Cycle <= '1';
                Cycle_Number      <= "11";
                If TimerIsZero='1' Then
                    ACK               <= '1';
                    State_Next        <= State_AddressSetup;
                End If;
           ---------------------------------------------------------------------

           ---------------------------------------------------------------------
           When State_Read_Cycle0_Latch =>
                Cycle_Number  <= "00";
                If TimerIsZero='1' Then
                    LoadAddressDelay <= '1';
                    State_Next    <= State_Read_Cycle1_Setup;
                End If;

           When State_Read_Cycle1_Setup =>
                Cycle_Number <= "01";
                If TimerIsZero='1' Then
                    State_Next <= State_Read_Cycle1_Latch;
                End If;
           When State_Read_Cycle1_Latch =>
                Cycle_Number  <= "01";
                LoadAddressDelay <= '1';
                State_Next    <= State_Read_Cycle2_Setup;

           When State_Read_Cycle2_Setup =>
                Cycle_Number <= "10";
                If TimerIsZero='1' Then
                    State_Next <= State_Read_Cycle2_Latch;
                End If;
           When State_Read_Cycle2_Latch =>
                 Cycle_Number  <= "10";
                 LoadAddressDelay <= '1';
                 State_Next    <= State_Read_Cycle3_Setup;

           When State_Read_Cycle3_Setup =>
                Cycle_Number <= "11";
                If TimerIsZero='1' Then
                    State_Next <= State_Read_Cycle3_ACK;
                End If;

           When State_Read_Cycle3_ACK =>
                Cycle_Number  <= "11";
                ACK <= '1';
                State_Next <= State_AddressSetup;
           ---------------------------------------------------------------------

           When Others =>
                State_Next <= State_AddressSetup;

        End Case;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    FSM_Synchronous:
    Process(SRAM_CLK_I)
    Begin
        If Rising_Edge(SRAM_CLK_I) Then
           If SRAM_RST_I = '1' Then
              State  <= State_AddressSetup;
           Else
              State <= State_Next;
           End If;
        End If;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    FSM_LatchData:
    Process(SRAM_CLK_I)
    Begin
        If Rising_Edge(SRAM_CLK_I) Then
           Case State Is
              When State_Read_Cycle0_Latch => OutData_Latched( 7 DownTo  0) <= MEM_D(7 Downto 0);
              When State_Read_Cycle1_Latch => OutData_Latched(15 DownTo  8) <= MEM_D(7 Downto 0);
              When State_Read_Cycle2_Latch => OutData_Latched(23 DownTo 16) <= MEM_D(7 Downto 0);
              When Others                  => Null;
           End Case;
        End If;
    End Process;
    ----------------------------------------------------------------------------

     MEM_SDRAM_CKE   <= '1';   -- ExcludeIf_DisableAllPorts
     MEM_SDRAM_RAS   <= '1';   -- ExcludeIf_DisableAllPorts
     MEM_SDRAM_CAS   <= '1';   -- ExcludeIf_DisableAllPorts
     MEM_SDRAM_E     <= '1';   -- ExcludeIf_DisableAllPorts
     MEM_FLASH_RESET <= '1';   -- ExcludeIf_DisableAllPorts
     MEM_FLASH_E     <= '1';   -- ExcludeIf_DisableAllPorts
     MEM_BE          <= (Others=>'0');                           -- ExcludeIf_DisableAllPorts
End RTL;
--------------------------------------------------------------------------------

