Library IEEE;
Use IEEE.Std_Logic_1164.all;
Use IEEE.std_logic_unsigned.all;

--------------------------------------------------------------------------------
Entity Configurable_ReplaceWith_Designator Is Port
   (

     -----------------------------------------------------------------------------
     MEM_D      : InOut Std_Logic_Vector(31 Downto 0);                          -- ExcludeIf_DisableAllPorts
     MEM_D      : InOut Std_Logic_Vector(15 Downto 0);                          -- ExcludeIf_NotDisableAllPorts
     MEM_A      : Out   Std_Logic_Vector(ReplaceWith_AddressWidth-1 DownTo 0);  -- IncludeIf_PartialAddress
     MEM_A      : Out   Std_Logic_Vector(23 Downto 0);                          -- IncludeIf_FullAddress
     MEM_OE     : Out   Std_Logic;
     MEM_W      : Out   Std_Logic;
     MEM_SRAM_E : Out   Std_Logic;
     MEM_BE     : Out   Std_Logic_Vector(1 Downto 0);                          -- ExcludeIf_NotDisableAllPorts
     MEM_BE     : Out   Std_Logic_Vector(3 Downto 0);                          -- ExcludeIf_DisableAllPorts

     -----------------------------------------------------------------------------

     MEM_SDRAM_CKE   : Out Std_Logic;   -- ExcludeIf_DisableAllPorts
     MEM_SDRAM_RAS   : Out Std_Logic;   -- ExcludeIf_DisableAllPorts
     MEM_SDRAM_CAS   : Out Std_Logic;   -- ExcludeIf_DisableAllPorts
     MEM_SDRAM_E     : Out Std_Logic;   -- ExcludeIf_DisableAllPorts
     MEM_FLASH_E     : Out Std_Logic;   -- ExcludeIf_DisableAllPorts
     MEM_FLASH_BUSY  : In  Std_Logic;   -- ExcludeIf_DisableAllPorts
     MEM_FLASH_RESET : Out Std_Logic;   -- ExcludeIf_DisableAllPorts
     MEM_USB_E       : Out Std_Logic;   -- ExcludeIf_DisableAllPorts
     MEM_USB_RESET   : Out Std_Logic;   -- ExcludeIf_DisableAllPorts

     ---------------------------------------------------------------------------
     -- Wishbone Slave Port
     ---------------------------------------------------------------------------
     SRAM_CLK_I      : In    Std_Logic;
     SRAM_RST_I      : In    Std_Logic;
     SRAM_CYC_I      : In    Std_Logic;
     SRAM_STB_I      : In    Std_Logic;
     SRAM_ACK_O      : Out   Std_Logic;
     SRAM_ADR_I      : In    Std_Logic_Vector(ReplaceWith_InputAddressWidth-1 DownTo 0);
     SRAM_DAT_I      : In    Std_Logic_Vector(31 DownTo 0);
     SRAM_DAT_O      : Out   Std_Logic_Vector(31 DownTo 0);
     SRAM_WE_I       : In    Std_Logic;
     SRAM_SEL_I      : In    Std_Logic_Vector(3  DownTo 0)
   );

End Configurable_ReplaceWith_Designator;
--------------------------------------------------------------------------------
    
--                   ByteLanes[3..0]   Cycle  LB   UB   CE       OE  WE  A0     -- IncludeIf_1_Device_X_16
--                                     B3     B2   B1   B0       OE  WE  A0     -- IncludeIf_1_Device_X_16
---------------------------------------------------------------------------------- IncludeIf_1_Device_X_16
--Read             |     XXXX          0      0    0    0        0   1   0      -- IncludeIf_1_Device_X_16
--                 |                   1      0    0    0        0   1   1      -- IncludeIf_1_Device_X_16
--None             |     XXXX          0      1    1    1        1   1   0      -- IncludeIf_1_Device_X_16
--                 |                   1      1    1    1        1   1   1      -- IncludeIf_1_Device_X_16
--Write word       |     1111          0      0    0    0        1   0   0      -- IncludeIf_1_Device_X_16
--                 |                   1      0    0    0        1   0   1      -- IncludeIf_1_Device_X_16
--Write low half   |     0011          0      1    1    1        1   0   0      -- IncludeIf_1_Device_X_16
--                 |                   1      0    0    0        1   0   1      -- IncludeIf_1_Device_X_16
--Write high half  |     1100          0      0    0    0        1   0   0      -- IncludeIf_1_Device_X_16
--                 |                   1      1    1    1        1   0   1      -- IncludeIf_1_Device_X_16
--Write Byte0      |     0001          0      0    1    0        1   0   0      -- IncludeIf_1_Device_X_16
--                 |                   1      1    1    1        1   0   1      -- IncludeIf_1_Device_X_16
--Write Byte1      |     0010          0      1    0    0        1   0   0      -- IncludeIf_1_Device_X_16
--                 |                   1      1    1    1        1   0   1      -- IncludeIf_1_Device_X_16
--Write Byte2      |     0100          0      1    1    1        1   0   0      -- IncludeIf_1_Device_X_16
--                 |                   1      0    1    0        1   0   1      -- IncludeIf_1_Device_X_16
--Write Byte3      |     1000          0      1    1    1        1   0   0      -- IncludeIf_1_Device_X_16
--                 |                   1      1    0    1        1   0   0      -- IncludeIf_1_Device_X_16
--Write Bytes012   |     0111          0      0    0    0        1   0          -- IncludeIf_1_Device_X_16
--                 |                   1      0    1    0        1   0          -- IncludeIf_1_Device_X_16
--Write Bytes123   |     1110          0      1    0    0        1   0          -- IncludeIf_1_Device_X_16
--                 |                   1      0    0    0        1   0          -- IncludeIf_1_Device_X_16
--Write Bytes12    |     0110          0      1    0    0        1   0          -- IncludeIf_1_Device_X_16
--                 |                   1      0    1    0        1   0          -- IncludeIf_1_Device_X_16
--------------------------------------------------------------------------------
Architecture RTL of Configurable_ReplaceWith_Designator Is
    Type TState Is
        (
         State_AddressSetup  ,

         State_Write_Cycle0_Pulse   ,     -- WE goes low during this state
         State_Write_Cycle0_Wait    ,
         State_Write_Cycle1_Setup   ,
         State_Write_Cycle1_Pulse   ,     -- WE goes low during this state
         State_Write_Cycle1_ACK     ,     -- WE goes high. Must keep seperate form addresses etc in case WE goes high before address, CE etc

         State_Read_Cycle0_Latch    ,
         State_Read_Cycle1_Setup    ,
         State_Read_Cycle1_ACK
        );

    Signal   State      : TState;
    Signal   State_Next : TState;

    SubType  TCycleAndByteLane   Is Std_Logic_Vector(4 DownTo 0);
    SubType  TOutBits            Is Std_Logic_Vector(2 DownTo 0);

    ----------------------------------------------------------------------------
    -- Constants for byte lanes - always the same
    ----------------------------------------------------------------------------
    Constant Write_Word_Cycle0           : TCycleAndByteLane := "01111";
    Constant Write_Word_Cycle1           : TCycleAndByteLane := "11111";
    Constant Write_LoHalf_Cycle0         : TCycleAndByteLane := "00011";
    Constant Write_LoHalf_Cycle1         : TCycleAndByteLane := "10011";
    Constant Write_HiHalf_Cycle0         : TCycleAndByteLane := "01100";
    Constant Write_HiHalf_Cycle1         : TCycleAndByteLane := "11100";
    Constant Write_Byte0_Cycle0          : TCycleAndByteLane := "00001";
    Constant Write_Byte0_Cycle1          : TCycleAndByteLane := "10001";
    Constant Write_Byte1_Cycle0          : TCycleAndByteLane := "00010";
    Constant Write_Byte1_Cycle1          : TCycleAndByteLane := "10010";
    Constant Write_Byte2_Cycle0          : TCycleAndByteLane := "00100";
    Constant Write_Byte2_Cycle1          : TCycleAndByteLane := "10100";
    Constant Write_Byte3_Cycle0          : TCycleAndByteLane := "01000";
    Constant Write_Byte3_Cycle1          : TCycleAndByteLane := "11000";
    Constant Write_Bytes012_Cycle0       : TCycleAndByteLane := "00111";
    Constant Write_Bytes012_Cycle1       : TCycleAndByteLane := "10111";
    Constant Write_Bytes123_Cycle0       : TCycleAndByteLane := "01110";
    Constant Write_Bytes123_Cycle1       : TCycleAndByteLane := "11110";
    Constant Write_Bytes12_Cycle0        : TCycleAndByteLane := "00110";
    Constant Write_Bytes12_Cycle1        : TCycleAndByteLane := "10110";
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    -- Mapping of byte lanes to SRAM Controls
    ----------------------------------------------------------------------------
    Constant OutBits_Write_Word            : TOutBits  := "000";

    Constant OutBits_Write_Cycle0_LoHalf : TOutBits  := "000";                  -- IncludeIf_1_Device_X_16
    Constant OutBits_Write_Cycle1_LoHalf : TOutBits  := "111";                  -- IncludeIf_1_Device_X_16
    Constant OutBits_Write_Cycle0_HiHalf : TOutBits  := "111";                  -- IncludeIf_1_Device_X_16
    Constant OutBits_Write_Cycle1_HiHalf : TOutBits  := "000";                  -- IncludeIf_1_Device_X_16
    Constant OutBits_Write_Cycle0_Byte0  : TOutBits  := "010";                  -- IncludeIf_1_Device_X_16
    Constant OutBits_Write_Cycle1_Byte0  : TOutBits  := "111";                  -- IncludeIf_1_Device_X_16
    Constant OutBits_Write_Cycle0_Byte1  : TOutBits  := "100";                  -- IncludeIf_1_Device_X_16
    Constant OutBits_Write_Cycle1_Byte1  : TOutBits  := "111";                  -- IncludeIf_1_Device_X_16
    Constant OutBits_Write_Cycle0_Byte2  : TOutBits  := "111";                  -- IncludeIf_1_Device_X_16
    Constant OutBits_Write_Cycle1_Byte2  : TOutBits  := "010";                  -- IncludeIf_1_Device_X_16
    Constant OutBits_Write_Cycle0_Byte3  : TOutBits  := "111";                  -- IncludeIf_1_Device_X_16
    Constant OutBits_Write_Cycle1_Byte3  : TOutBits  := "100";                  -- IncludeIf_1_Device_X_16
    
    Constant OutBits_Write_Cycle0_Bytes012 : TOutBits := "000";                  -- IncludeIf_1_Device_X_16
    Constant OutBits_Write_Cycle1_Bytes012 : TOutBits := "010";                  -- IncludeIf_1_Device_X_16
    Constant OutBits_Write_Cycle0_Bytes123 : TOutBits := "100";                  -- IncludeIf_1_Device_X_16
    Constant OutBits_Write_Cycle1_Bytes123 : TOutBits := "000";                  -- IncludeIf_1_Device_X_16
    Constant OutBits_Write_Cycle0_Bytes12  : TOutBits := "100";                  -- IncludeIf_1_Device_X_16
    Constant OutBits_Write_Cycle1_Bytes12  : TOutBits := "010";                  -- IncludeIf_1_Device_X_16

                                                                                -- IncludeIf_1_Device_X_16
    Constant OutBits_Read                : TOutBits  := "000";                  -- IncludeIf_1_Device_X_16
    Constant OutBits_None                : TOutBits  := "111";                  -- IncludeIf_1_Device_X_16

    Signal   OutBits                     : TOutBits;
    Signal   OutBits_Write               : TOutBits;

    Signal   Request                     : Std_Logic;
    Signal   RequestRead                 : Std_Logic;
    Signal   RequestWrite                : Std_Logic;

    Signal   Cycle_Number                : Std_Logic;

    Signal   WriteActive_Cycle           : Std_Logic;
    Signal   WE                          : Std_Logic;
    Signal   OE                          : Std_Logic;
    Signal   ACK                         : Std_Logic;

    Signal   CycleAndSel                 : TCycleAndByteLane;

    Signal   OutData_Latched             : Std_Logic_Vector(15 Downto 0);
    Signal   ADDR                        : Std_Logic_Vector(ReplaceWith_InputAddressWidth-3 Downto 0);

    Constant cAddressSetupDelay : Std_Logic_Vector(ReplaceWith_TimerWidth-1 Downto 0) := ReplaceWith_cAddressSetupDelay;
    Constant cWritePulseDelay   : Std_Logic_Vector(ReplaceWith_TimerWidth-1 Downto 0) := ReplaceWith_cWritePulseDelay;
    Constant cWriteHoldDelay    : Std_Logic_Vector(ReplaceWith_TimerWidth-1 Downto 0) := ReplaceWith_cWriteHoldDelay;
    Constant cTimerIsZero       : Std_Logic_Vector(ReplaceWith_TimerWidth-1 Downto 0) := ReplaceWith_cTimerIsZero;

    Signal  Timer               : Std_Logic_Vector(ReplaceWith_TimerWidth-1 Downto 0);
    Signal  TimerIsZero         : Std_Logic;
    Signal  LoadAddressDelay    : Std_Logic;
    Signal  LoadWritePulseDelay : Std_Logic;
    Signal  LoadWriteHoldDelay  : Std_Logic;
    Signal MEM_A_Signal : Std_Logic_Vector(23 Downto 0);
Begin

    ----------------------------------------------------------------------------
    Process(SRAM_CLK_I)
    Begin
        If Rising_Edge(SRAM_CLK_I) Then
            If SRAM_RST_I='1' Then
                Timer <= (Others=>'0');
            Else
                   If LoadAddressDelay='1'    Then Timer <= cAddressSetupDelay;
                ElsIf LoadWritePulseDelay='1' Then Timer <= cWritePulseDelay;
                ElsIf LoadWriteHoldDelay='1'  Then Timer <= cWriteHoldDelay;
                ElsIf TimerIsZero='0'         Then Timer <= Timer - 1;
                End If;
            End If;
        End If;
    End Process;
    TimerIsZero <= '1' When Timer=cTimerIsZero Else '0';
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    ADDR <= SRAM_ADR_I(ReplaceWith_InputAddressWidth-1 DownTo 2);
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    MEM_BE(1)   <= OutBits(2);                                                    -- IncludeIf_1_Device_X_16
    MEM_BE(0)   <= OutBits(1);                                                    -- IncludeIf_1_Device_X_16
    MEM_SRAM_E  <= OutBits(0) Or (Not Request);
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    MEM_A(ReplaceWith_AddressWidth-1 DownTo 1) <= ADDR(ReplaceWith_AddressWidth-2 DownTo 0); -- IncludeIf_PartialAddress
    MEM_A(0)           <= Cycle_Number;                                                      -- IncludeIf_PartialAddress

    MEM_A_Signal <= ReplaceWith_AddressPadding '0' & ADDR(ReplaceWith_AddressWidth-2 Downto 0); -- IncludeIf_FullAddress
    MEM_A(23 Downto 2) <= MEM_A_Signal(21 Downto 0);                                            -- IncludeIf_FullAddress
    MEM_A(1  Downto 0) <= Cycle_Number & '0';                                                   -- IncludeIf_FullAddress
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    SRAM_DAT_O(15 DownTo  0) <= OutData_Latched;
    SRAM_DAT_O(31 DownTo 16) <= MEM_D(15 Downto 0);                                              -- IncludeIf_1_Device_X_16
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    SRAM_ACK_O <= ACK;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    Request      <= SRAM_STB_I And SRAM_CYC_I;
    RequestRead  <= SRAM_STB_I And SRAM_CYC_I And (Not SRAM_WE_I);
    RequestWrite <= SRAM_STB_I And SRAM_CYC_I And (    SRAM_WE_I);
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    MEM_W  <= WE;                                                              -- IncludeIf_1_Device_X_16
    MEM_OE <= OE;                                                              -- IncludeIf_1_Device_X_16
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    CycleAndSel <= Cycle_Number & SRAM_SEL_I;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    Memory_OuptutControls:
    Process(SRAM_SEL_I,
            SRAM_DAT_I,
            State,
            OutBits_Write,
            WriteActive_Cycle,
            Cycle_Number,
            CycleAndSel)
    Begin

        MEM_D <= (Others=>'Z');

        If WriteActive_Cycle = '1' Then
            OutBits    <= OutBits_Write;
            OE         <= '1';
            If Cycle_Number = '0' Then
                MEM_D(15 Downto 0) <= SRAM_DAT_I(15 DownTo  0);                               -- IncludeIf_1_Device_X_16
            Else
                MEM_D(15 Downto 0) <= SRAM_DAT_I(31 DownTo 16);                               -- IncludeIf_1_Device_X_16
            End If;
        Else
           OutBits    <= OutBits_Read;
           OE         <= '0';
           MEM_D     <= (Others => 'Z');                                       -- IncludeIf_1_Device_X_16
        End If;

        Case CycleAndSel Is
           When Write_Word_Cycle0      => OutBits_Write <= OutBits_Write_Word;
           When Write_Word_Cycle1      => OutBits_Write <= OutBits_Write_Word;
           When Write_LoHalf_Cycle0    => OutBits_Write <= OutBits_Write_Cycle0_LoHalf;
           When Write_LoHalf_Cycle1    => OutBits_Write <= OutBits_Write_Cycle1_LoHalf;
           When Write_HiHalf_Cycle0    => OutBits_Write <= OutBits_Write_Cycle0_HiHalf;
           When Write_HiHalf_Cycle1    => OutBits_Write <= OutBits_Write_Cycle1_HiHalf;
           When Write_Byte0_Cycle0     => OutBits_Write <= OutBits_Write_Cycle0_Byte0;
           When Write_Byte0_Cycle1     => OutBits_Write <= OutBits_Write_Cycle1_Byte0;
           When Write_Byte1_Cycle0     => OutBits_Write <= OutBits_Write_Cycle0_Byte1;
           When Write_Byte1_Cycle1     => OutBits_Write <= OutBits_Write_Cycle1_Byte1;
           When Write_Byte2_Cycle0     => OutBits_Write <= OutBits_Write_Cycle0_Byte2;
           When Write_Byte2_Cycle1     => OutBits_Write <= OutBits_Write_Cycle1_Byte2;
           When Write_Byte3_Cycle0     => OutBits_Write <= OutBits_Write_Cycle0_Byte3;
           When Write_Byte3_Cycle1     => OutBits_Write <= OutBits_Write_Cycle1_Byte3;
           When Write_Bytes012_Cycle0  => OutBits_Write <= OutBits_Write_Cycle0_Bytes012;
           When Write_Bytes012_Cycle1  => OutBits_Write <= OutBits_Write_Cycle1_Bytes012;
           When Write_Bytes123_Cycle0  => OutBits_Write <= OutBits_Write_Cycle0_Bytes123;
           When Write_Bytes123_Cycle1  => OutBits_Write <= OutBits_Write_Cycle1_Bytes123;
           When Write_Bytes12_Cycle0   => OutBits_Write <= OutBits_Write_Cycle0_Bytes12;
           When Write_Bytes12_Cycle1   => OutBits_Write <= OutBits_Write_Cycle1_Bytes12;
           When Others                 => OutBits_Write <= OutBits_None;
        End Case;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    FSM_Combinational_Single:
    Process(RequestRead,
            RequestWrite,
            State,
            TimerIsZero)
    Begin
        ------------------------------------------------------------------------
        -- Default outputs
        WriteActive_Cycle <= '0';
        WE                <= '1';
        ACK               <= '0';
        Cycle_Number      <= '0';
        State_Next        <= State;
        LoadAddressDelay    <= '0';
        LoadWritePulseDelay <= '0';
        LoadWriteHoldDelay  <= '0';

        ------------------------------------------------------------------------

        ------------------------------------------------------------------------
        Case State Is
           ---------------------------------------------------------------------
           When State_AddressSetup =>
                If RequestWrite = '1' Then                --Address is being setup in this phases
                   WriteActive_Cycle <= '1';
                   State_Next        <= State_Write_Cycle0_Pulse;
                   LoadWritePulseDelay <= '1';
                 ElsIf RequestRead = '1' Then
                   State_Next <= State_Read_Cycle0_Latch;
                   LoadAddressDelay <= '1';
                End If;
           ---------------------------------------------------------------------

           ---------------------------------------------------------------------
           When State_Write_Cycle0_Pulse =>
                WriteActive_Cycle <= '1';
                WE                <= '0';
                If TimerIsZero='1' Then
                    State_Next        <= State_Write_Cycle0_Wait;
                    LoadWriteHoldDelay <= '1';
                End If;

           When State_Write_Cycle0_Wait =>
                WriteActive_Cycle <= '1';
                If TimerIsZero='1' Then
                    State_Next        <= State_Write_Cycle1_Setup;
                    LoadAddressDelay <= '1';
                End If;

           When State_Write_Cycle1_Setup =>
                WriteActive_Cycle <= '1';
                Cycle_Number      <= '1';
                If TimerIsZero='1' Then
                    State_Next        <= State_Write_cycle1_Pulse;
                    LoadWritePulseDelay <= '1';
                End If;

           When State_Write_Cycle1_Pulse =>
                WriteActive_Cycle <= '1';
                WE                <= '0';
                Cycle_Number      <= '1';
                If TimerIsZero='1' Then
                    State_Next         <= State_Write_Cycle1_ACK;
                    LoadWriteHoldDelay <= '1';
                End If;

           When State_Write_Cycle1_ACK =>
                WriteActive_Cycle <= '1';
                Cycle_Number      <= '1';
                If TimerIsZero='1' Then
                    ACK               <= '1';
                    State_Next        <= State_AddressSetup;
                End If;
           ---------------------------------------------------------------------

           ---------------------------------------------------------------------
           When State_Read_Cycle0_Latch =>
                If TimerIsZero='1' Then
                    State_Next    <= State_Read_Cycle1_Setup;
                    LoadAddressDelay <= '1';
                End If;

           When State_Read_Cycle1_Setup =>
                Cycle_Number  <= '1';
                If TimerIsZero='1' Then
                    State_Next    <= State_Read_Cycle1_ACK;
                    LoadAddressDelay <= '1';
                End If;

           When State_Read_Cycle1_ACK =>                    -- Assert ACK, Master will read the data one clock cycle later
                Cycle_Number  <= '1';
                If TimerIsZero='1' Then
                    ACK           <= '1';
                    State_Next    <= State_AddressSetup;
                End If;
           ---------------------------------------------------------------------

           When Others =>
                State_Next <= State_AddressSetup;

        End Case;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    FSM_Synchronous:
    Process(SRAM_CLK_I)
    Begin
        If Rising_Edge(SRAM_CLK_I) Then
           If SRAM_RST_I = '1' Then
              State  <= State_AddressSetup;
           Else
              State <= State_Next;
           End If;
        End If;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    FSM_LatchData:
    Process(SRAM_CLK_I)                                                 -- IncludeIf_1_Device_X_16
    Begin
        If Rising_Edge(SRAM_CLK_I) Then
           If State = State_Read_Cycle0_Latch Then
               OutData_Latched <= MEM_D(15 Downto 0);                   -- IncludeIf_1_Device_X_16
           End If;
        End If;
    End Process;
    ----------------------------------------------------------------------------


     MEM_SDRAM_CKE      <= '1';     -- ExcludeIf_DisableAllPorts
     MEM_SDRAM_RAS      <= '1';     -- ExcludeIf_DisableAllPorts
     MEM_SDRAM_CAS      <= '1';     -- ExcludeIf_DisableAllPorts
     MEM_SDRAM_E        <= '1';     -- ExcludeIf_DisableAllPorts
     MEM_FLASH_RESET    <= '1';     -- ExcludeIf_DisableAllPorts
     MEM_FLASH_E        <= '1';     -- ExcludeIf_DisableAllPorts
     MEM_BE(3 Downto 2) <= "11";    -- ExcludeIf_DisableAllPorts
     MEM_USB_E          <= '1';     -- ExcludeIf_DisableAllPorts
     MEM_USB_RESET      <= '1';     -- ExcludeIf_DisableAllPorts
End RTL;
--------------------------------------------------------------------------------

