Library IEEE;
Use IEEE.Std_Logic_1164.all;
Use IEEE.std_logic_unsigned.all;
Use IEEE.std_logic_arith.all;

--------------------------------------------------------------------------------
Entity Configurable_ReplaceWith_Designator Is Port
   (
     ---------------------------------------------------------------------------
     -- Memory Interface
     ---------------------------------------------------------------------------
     -----------------------------------------------------------------------------
     MEM_D      : InOut Std_Logic_Vector(31 Downto 0);                          -- ExcludeIf_DisableAllPorts     ExcludeIf_Shared
     MEM_D      : InOut Std_Logic_Vector(15 Downto 0);                          -- ExcludeIf_NotDisableAllPorts  ExcludeIf_Shared
     MEM_D_O    : Out   Std_Logic_Vector(31 Downto 0);                          -- IncludeIf_Shared
     MEM_D_I    : In    Std_Logic_Vector(31 Downto 0);                          -- IncludeIf_Shared
     MEM_D_E    : Out   Std_Logic;                                              -- IncludeIf_Shared
     MEM_A      : Out   Std_Logic_Vector(ReplaceWith_AddressWidth-1 DownTo 0);  -- IncludeIf_PartialAddress
     MEM_A      : Out   Std_Logic_Vector(23 Downto 0);                          -- IncludeIf_FullAddress
     MEM_OE     : Out   Std_Logic;
     MEM_W      : Out   Std_Logic;
     MEM_FLASH_E     : Out Std_Logic;
     MEM_FLASH_BUSY  : In  Std_Logic;
     MEM_FLASH_RESET : Out Std_Logic;

     -----------------------------------------------------------------------------

     MEM_SDRAM_CKE   : Out Std_Logic;                    -- ExcludeIf_DisableAllPorts
     MEM_SDRAM_RAS   : Out Std_Logic;                    -- ExcludeIf_DisableAllPorts
     MEM_SDRAM_CAS   : Out Std_Logic;                    -- ExcludeIf_DisableAllPorts
     MEM_SDRAM_E     : Out Std_Logic;                    -- ExcludeIf_DisableAllPorts
     MEM_SRAM_E      : Out Std_Logic;                    -- ExcludeIf_DisableAllPorts
     MEM_BE          : Out Std_Logic_Vector(3 Downto 0); -- ExcludeIf_DisableAllPorts


     ---------------------------------------------------------------------------
     -- Wishbone Slave Port
     ---------------------------------------------------------------------------
     FLASH_CLK_I      : In    Std_Logic;
     FLASH_RST_I      : In    Std_Logic;
     FLASH_CYC_I      : In    Std_Logic;
     FLASH_STB_I      : In    Std_Logic;
     FLASH_ACK_O      : Out   Std_Logic;
     FLASH_ADR_I      : In    Std_Logic_Vector(24 DownTo 0);
     FLASH_DAT_I      : In    Std_Logic_Vector(31 DownTo 0);
     FLASH_DAT_O      : Out   Std_Logic_Vector(31 DownTo 0);
     FLASH_WE_I       : In    Std_Logic;
     FLASH_SEL_I      : In    Std_Logic_Vector( 3 DownTo 0)
   );

End Configurable_ReplaceWith_Designator;
--------------------------------------------------------------------------------

--------------------------------------------------------------------------------
Architecture RTL of Configurable_ReplaceWith_Designator Is
    ----------------------------------------------------------------------------
    Type TState Is
        (
         State_Idle                 ,

         State_Read_AddressSetup    , -- needed even if no ReadCSActiveDelay for second cycle
         State_Read_CSActive        , -- only include if there is a OEActiveDelay
         State_Read_OEActive        ,
         State_Read_Hold            , -- only include if there is a ReadHoldDelay
         State_Read_HoldLast        , -- only include if there is a ReadHoldDelay

         State_Write_AddressSetup   , -- needed even if no WriteCSActiveDelay for second cycle
         State_Write_CSActive       , -- only include if there is a WEActiveDelay
         State_Write_WEActive       ,
         State_Write_WEInactive     , -- only include if there is a WECSInactiveDelay
         State_Write_WEInactiveLast , -- only include if there is a WECSInactiveDelay
         State_Write_Ack            ,
         State_Write_Hold           , -- only include if there is a WriteHoldDelay
         State_Write_HoldLast       ,  -- only include if there is a WriteHoldDelay
         State_BusyFlag_Ack            -- Acknowledge for BusyFlag
        );

    Signal   State      : TState;
    Signal   State_Next : TState;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    SubType  TStateDelayCounter  Is Integer Range 0 To 7;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    -- Interpreted Wishbone signals
    ----------------------------------------------------------------------------
    Signal   Request                     : Std_Logic;
    Signal   RequestRead                 : Std_Logic;
    Signal   RequestWrite                : Std_Logic;
    Signal   SkipCycle_0                 : Std_Logic;
    Signal   SkipCycle_1                 : Std_Logic;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    -- Main state machine outputs
    ----------------------------------------------------------------------------
    Signal   IsMemActive                 : Std_Logic;
    Signal   IsCSActive                  : Std_Logic;
    Signal   IsWEActive                  : Std_Logic;
    Signal   IsOEActive                  : Std_Logic;
    Signal   IsWriting                   : Std_Logic;
    Signal   LoadReadData                : Std_Logic;
    Signal   WishboneAck                 : Std_Logic;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    -- Cycle control signals
    ----------------------------------------------------------------------------
    Signal   Cycle                       : Std_Logic;
    Signal   ResetCycle                  : Std_Logic;
    Signal   IncrementCycle              : Std_Logic;
    Signal   IsLastCycle                 : Std_Logic;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    Signal   ReadDataLoWord_Reg          : Std_Logic_Vector(15 Downto 0);
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    ----------------------------------------------------------------------------
    Constant cAddressSetupDelay : Std_Logic_Vector(ReplaceWith_TimerWidth-1 Downto 0) := ReplaceWith_cAddressSetupDelay;
    Constant cWritePulseDelay   : Std_Logic_Vector(ReplaceWith_TimerWidth-1 Downto 0) := ReplaceWith_cWritePulseDelay;
    Constant cWriteHoldDelay    : Std_Logic_Vector(ReplaceWith_TimerWidth-1 Downto 0) := ReplaceWith_cWriteHoldDelay;
    Constant cTimerIsZero       : Std_Logic_Vector(ReplaceWith_TimerWidth-1 Downto 0) := ReplaceWith_cTimerIsZero;

    Signal ReadBusyFlag          : Std_Logic;
    Signal LoadTimerAddressSetup : Std_Logic;
    Signal LoadTimerReadPulse    : Std_Logic;
    Signal LoadTimerWritePulse   : Std_Logic;
    Signal LoadTimerWriteHold    : Std_Logic;
    Signal Timer                 : Std_Logic_Vector(ReplaceWith_TimerWidth-1 Downto 0);
    Signal TimerIsZero           : Std_Logic;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
Begin
    ----------------------------------------------------------------------------
    DriveMemoryAddress:
    Process(Cycle, FLASH_ADR_I)
    Begin
        MEM_A(0) <= Cycle;
        MEM_A(23 DownTo 1) <= FLASH_ADR_I(24 DownTo 2);
    End Process;
    ----------------------------------------------------------------------------
    ----------------------------------------------------------------------------
    DriveMemoryDataAndByteEnable:
    Process(Cycle, IsWriting, FLASH_DAT_I, FLASH_SEL_I)
    Begin
        MEM_D   <= (Others => 'Z'); -- ExcludeIf_Shared
        MEM_D_E <= '0';             -- IncludeIf_Shared
        MEM_D_O <= (Others => '0'); -- IncludeIf_Shared

        If IsWriting = '1' Then
            If Cycle = '0' Then
                MEM_D(15 Downto 0) <= FLASH_DAT_I(31 DownTo 16); -- ExcludeIf_Shared
                MEM_D_O(15 Downto 0) <= FLASH_DAT_I(31 Downto 16); --IncludeIf_Shared
                MEM_D_O(31 Downto 16) <= (Others=>'0');  --IncludeIf_Shared
            Else
                MEM_D(15 Downto 0) <= FLASH_DAT_I(15 DownTo  0); -- ExcludeIf_Shared
                MEM_D_O(15 Downto 0) <= FLASH_DAT_I(15 Downto 0); -- IncludeIf_Shared
                MEM_D_O(31 Downto 16) <= (Others=>'0');  --IncludeIf_Shared
            End If;
            MEM_D_E <= '1'; -- IncludeIf_Shared
        End If;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    DriveMemoryControl:
    Process(IsCSActive, IsWEActive, IsOEActive, FLASH_RST_I, IsMemActive)
    Begin
        MEM_FLASH_E     <= Not IsCSActive;
        MEM_W      <= Not IsWEActive;
        MEM_OE     <= Not IsOEActive;
        MEM_FLASH_RESET  <= Not FLASH_RST_I;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    DriveWishboneSignals:
    Process(ReadDataLoWord_Reg,
            MEM_D,  -- ExcludeIf_Shared
            MEM_D_I, -- IncludeIf_Shared
            WishboneAck,
            SkipCycle_1,
            RequestRead)
    Begin
         If SkipCycle_1='1' Then
             FLASH_DAT_O(31 DownTo 16) <= MEM_D(15 Downto 0); -- ExcludeIf_Shared
             FLASH_DAT_O(31 Downto 16) <= MEM_D_I(15 Downto 0); -- IncludeIf_Shared
             FLASH_DAT_O(15 Downto  0) <= ReadDataLoWord_Reg;
         Else
             FLASH_DAT_O(31 Downto 16) <= ReadDataLoWord_Reg;
             FLASH_DAT_O(15 Downto  0) <= MEM_D(15 Downto 0); -- ExcludeIf_Shared
             FLASH_DAT_O(15 Downto 0) <= MEM_D_I(15 Downto 0); -- IncludeIf_Shared
         End If;
        FLASH_ACK_O               <= WishboneAck;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    InterpretWishboneRequest:
    Process(FLASH_STB_I, FLASH_CYC_I, FLASH_WE_I, Request, FLASH_SEL_I)
    Begin
        Request        <= FLASH_STB_I And FLASH_CYC_I;
        RequestRead    <= Request And (Not FLASH_WE_I);
        RequestWrite   <= Request And (    FLASH_WE_I);
        SkipCycle_0    <= Not (FLASH_SEL_I(2) Or FLASH_SEL_I(3));
        SkipCycle_1    <= Not (FLASH_SEL_I(0) Or FLASH_SEL_I(1));
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    CycleControl:
    Process(FLASH_CLK_I, FLASH_RST_I, Cycle, SkipCycle_0, SkipCycle_1)
    Begin
        If Cycle = '0' And SkipCycle_1 = '0' Then
            IsLastCycle <= '0';
        Else
            IsLastCycle <= '1';
        End If;

        If FLASH_RST_I = '1' Then
            Cycle <= '0';
        ElsIf Rising_Edge(FLASH_CLK_I) Then
            If ResetCycle = '1' Then
                If SkipCycle_0 = '1' Then
                    Cycle <= '1';
                Else
                    Cycle <= '0';
                End If;
            ElsIf IncrementCycle = '1' Then
                Cycle <= '1';
            End If;
        End If;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    RegisterWishboneData:
    Process(FLASH_CLK_I)
    Begin
        If Rising_Edge(FLASH_CLK_I) Then
           If LoadReadData = '1' Then
               ReadDataLoWord_Reg <= MEM_D(15 Downto 0); -- ExcludeIf_Shared
               ReadDataLoWord_Reg <= MEM_D_I(15 Downto 0); -- IncludeIf_Shared
           End If;
        End If;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    Process(FLASH_CLK_I)
    Begin
        If Rising_Edge(FLASH_CLK_I) Then
            If FLASH_RST_I='1' Then
                Timer <= (Others=>'0');
            Else
                   If LoadTimerAddressSetup ='1' Then Timer <= cAddressSetupDelay;
                ElsIf LoadTimerReadPulse    ='1' Then Timer <= cAddressSetupDelay;
                ElsIf LoadTimerWritePulse   ='1' Then Timer <= cWritePulseDelay;
                ElsIf LoadTimerWriteHold    ='1' Then Timer <= cWriteHoldDelay;
                ElsIf TimerIsZero           ='0' Then Timer <= Timer - 1;
                End If;
            End If;
        End If;
    End Process;
    ----------------------------------------------------------------------------
    TimerIsZero <= '1' When Timer=cTimerIsZero Else '0';
    ----------------------------------------------------------------------------
    StateMachine_Combinational:
    Process(State,
            RequestRead,
            RequestWrite,
            Cycle,
            IsLastcycle,
            ReadBusyFlag,
            TimerIsZero)
    Begin
        ------------------------------------------------------------------------
        IsMemActive                 <= '1';
        IsCSActive                  <= '0';
        IsWEActive                  <= '0';
        IsOEActive                  <= '0';
        IsWriting                   <= '0';
        LoadReadData                <= '0';
        WishboneAck                 <= '0';
        ResetCycle                  <= '0';
        IncrementCycle              <= '0';
        State_Next                  <= State;
        LoadTimerAddressSetup       <= '0';
        LoadTimerReadPulse          <= '0';
        LoadTimerWritePulse         <= '0';
        LoadTimerWriteHold          <= '0';
        ------------------------------------------------------------------------

        ------------------------------------------------------------------------
        Case State Is
            --------------------------------------------------------------------
            When State_Idle =>
                If ReadBusyFlag='1' And RequestRead='1' Then
                    State_Next <= State_BusyFlag_Ack;
                ElsIf RequestRead = '1' Then
                    LoadTimerAddressSetup <= '1';
                    ResetCycle            <= '1';
                    State_Next            <= State_Read_AddressSetup;
                ElsIf RequestWrite = '1' Then
                    IsWriting             <= '1';
                    ResetCycle            <= '1';
                    State_Next            <= State_Write_AddressSetup;
                    LoadTimerAddressSetup <= '1';
                Else
                    IsMemActive           <= '0';
                End If;
            --------------------------------------------------------------------

            --------------------------------------------------------------------
            When State_Read_AddressSetup =>
                IsCSActive <= '1';
                If TimerIsZero = '1' Then
                    LoadTimerReadPulse    <= '1';
                    IsCSActive <= '1';
                    State_Next            <= State_Read_OEActive;
                End If;

            When State_Read_OEActive =>
                IsCSActive <= '1';
                IsOEActive <= '1';
                If TimerIsZero = '1' Then
                    If IsLastCycle = '1' Then
                        WishboneAck  <= '1';
                        State_Next   <= State_Read_HoldLast;
                    Else
                        LoadReadData <= '1';
                        State_Next   <= State_Read_Hold;
                    End If;
                End If;

            When State_Read_Hold =>
                IsCSActive <= '1';
                If TimerIsZero = '1' Then
                    IncrementCycle        <= '1';
                    State_Next            <= State_Read_AddressSetup;
                    LoadTimerAddressSetup <= '1';
                End If;

            When State_Read_HoldLast =>
                IsCSActive <= '1';
                If TimerIsZero = '1' Then
                    State_Next            <= State_Idle;
                End If;
            --------------------------------------------------------------------

            --------------------------------------------------------------------
            When State_Write_AddressSetup =>
                IsWriting <= '1';
                IsCSActive <= '1';
                If TimerIsZero = '1' Then
                    LoadTimerWritePulse <= '1';
                    State_Next            <= State_Write_WEActive;
                End If;

            When State_Write_WEActive =>
                IsWriting  <= '1';
                IsCSActive <= '1';
                IsWEActive <= '1';
                If TimerIsZero = '1' Then
                    If IsLastCycle = '0' Then
                        LoadTimerWriteHold <= '1';
                        State_Next            <= State_Write_WEInactive;
                    Else
                        LoadTimerWriteHold <= '1';
                        State_Next            <= State_Write_Hold;
                    End If;
                End If;

            When State_Write_Ack =>
                IsWriting             <= '1';
                IsCSActive            <= '1';
                WishboneAck           <= '1';
                State_Next            <= State_Idle;

            When State_Write_WEInactive =>
                IsWriting  <= '1';
                IsCSActive <= '1';
                If TimerIsZero = '1' Then
                    IncrementCycle <= '1';
                    LoadTimerAddressSetup <= '1';
                    State_Next            <= State_Write_AddressSetup;
                End If;

            When State_Write_Hold =>
                IsCSActive <= '1';
                IsWriting <= '1';
                If TimerIsZero = '1' Then
                    State_Next            <= State_Write_Ack;
                End If;

            --------------------------------------------------------------------
            When State_BusyFlag_Ack =>
                State_Next  <= State_Idle;
                WishboneAck <= '1';
            --------------------------------------------------------------------
            When Others =>
                State_Next <= State_Idle;
            --------------------------------------------------------------------

        End Case;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    StateMachine_Synchronous:
    Process(FLASH_CLK_I)
    Begin
        If Rising_Edge(FLASH_CLK_I) Then
            If FLASH_RST_I='1' Then
                State <= State_Idle;
            Else
                State  <= State_Next;
            End If;
        End If;
    End Process;
    ----------------------------------------------------------------------------
     MEM_SDRAM_CKE   <= '1';                    -- ExcludeIf_DisableAllPorts
     MEM_SDRAM_RAS   <= '1';                    -- ExcludeIf_DisableAllPorts
     MEM_SDRAM_CAS   <= '1';                    -- ExcludeIf_DisableAllPorts
     MEM_SDRAM_E     <= '1';                    -- ExcludeIf_DisableAllPorts
     MEM_SRAM_E      <= '1';                    -- ExcludeIf_DisableAllPorts
     MEM_BE          <= "1111";                 -- ExcludeIf_DisableAllPorts


End RTL;
--------------------------------------------------------------------------------

