Library IEEE;
Use IEEE.Std_Logic_1164.all;
Use IEEE.std_logic_unsigned.all;

--------------------------------------------------------------------------------
Entity Configurable_ReplaceWith_Designator Is Port
   (
     ---------------------------------------------------------------------------
     FLASH_D    : InOut Std_Logic_Vector( 7 DownTo 0);
     FLASH_A    : Out   Std_Logic_Vector(ReplaceWith_AddressWidth-1 DownTo 0);
     FLASH_OE   : Out   Std_Logic;
     FLASH_WE   : Out   Std_Logic;
     FLASH_CE   : Out   Std_Logic;
     ---------------------------------------------------------------------------

     ---------------------------------------------------------------------------
     -- Wishbone Slave Port
     ---------------------------------------------------------------------------
     CLK_I      : In    Std_Logic;
     RST_I      : In    Std_Logic;
     CYC_I      : In    Std_Logic;
     STB_I      : In    Std_Logic;
     ACK_O      : Out   Std_Logic;
     ADR_I      : In    Std_Logic_Vector(ReplaceWith_InputAddressWidth-1 DownTo 0);
     DAT_I      : In    Std_Logic_Vector(31 DownTo 0);
     DAT_O      : Out   Std_Logic_Vector(31 DownTo 0);
     WE_I       : In    Std_Logic;
     SEL_I      : In    Std_Logic_Vector( 3 DownTo 0)
   );

End Configurable_ReplaceWith_Designator;
--------------------------------------------------------------------------------

--------------------------------------------------------------------------------
--                   ByteLanes[3..0]   Cycle       WE   CE       OE  A1  A0
--                                     B3   B2     B1   B0       OE  A1  A0
------------------------------------------------------------------------------
--Read             |     XXXX          0    0      1    0        0   0   0
--                 |                   0    1      1    0        0   0   1
--                 |                   1    0      1    0        0   1   0
--                 |                   1    1      1    0        0   1   1
--None             |     XXXX          0    0      1    1        1   0   0
--                 |                   0    1      1    1        1   0   1
--                 |                   1    0      1    1        1   1   0
--                 |                   1    1      1    1        1   1   1
--Write word       |     1111          0    0      0    0        1   0   0
--                 |                   0    1      0    0        1   0   1
--                 |                   1    0      0    0        1   1   0
--                 |                   1    1      0    0        1   1   1
--Write low half   |     0011          0    0      0    0        1   0   0
--                 |                   0    1      0    0        1   0   1
--                 |                   1    0      1    0        1   1   0
--                 |                   1    1      1    0        1   1   1
--Write high half  |     1100          0    0      1    0        1   0   0
--                 |                   0    1      1    0        1   0   1
--                 |                   1    0      0    0        1   1   0
--                 |                   1    1      0    0        1   1   1
--Write Byte0      |     0001          0    0      0    0        1   0   0
--                 |                   0    1      1    0        1   0   1
--                 |                   1    0      1    0        1   1   0
--                 |                   1    1      1    0        1   1   1
--Write Byte1      |     0010          0    0      1    0        1   0   0
--                 |                   0    1      0    0        1   0   1
--                 |                   1    0      1    0        1   1   0
--                 |                   1    1      1    0        1   1   1
--Write Byte2      |     0100          0    0      1    0        1   0   0
--                 |                   0    1      1    0        1   0   1
--                 |                   1    0      0    0        1   1   0
--                 |                   1    1      1    0        1   1   1
--Write Byte3      |     1000          0    0      1    0        1   0   0
--                 |                   0    1      1    0        1   0   1
--                 |                   1    0      1    0        1   1   0
--                 |                   1    1      0    0        1   1   1

--------------------------------------------------------------------------------
Architecture RTL of Configurable_ReplaceWith_Designator Is

   Constant State_AddressSetup              : Std_Logic_Vector(4 Downto 0) := "00000";
   Constant State_Write_Cycle0_Write_Setup  : Std_Logic_Vector(4 Downto 0) := "00001";
   Constant State_Write_Cycle0_Write_Pulse  : Std_Logic_Vector(4 Downto 0) := "00010";
   Constant State_Write_Cycle0_Write_Hold   : Std_Logic_Vector(4 Downto 0) := "00011";
   Constant State_Write_Cycle0_ACK          : Std_Logic_Vector(4 Downto 0) := "00100";
   Constant State_Write_Cycle1_Write_Setup  : Std_Logic_Vector(4 Downto 0) := "00101";
   Constant State_Write_Cycle1_Write_Pulse  : Std_Logic_Vector(4 Downto 0) := "00110";
   Constant State_Write_Cycle1_Write_Hold   : Std_Logic_Vector(4 Downto 0) := "00111";
   Constant State_Write_Cycle1_ACK          : Std_Logic_Vector(4 Downto 0) := "01000";
   Constant State_Write_Cycle2_Write_Setup  : Std_Logic_Vector(4 Downto 0) := "01001";
   Constant State_Write_Cycle2_Write_Pulse  : Std_Logic_Vector(4 Downto 0) := "01010";
   Constant State_Write_Cycle2_Write_Hold   : Std_Logic_Vector(4 Downto 0) := "01011";
   Constant State_Write_Cycle2_ACK          : Std_Logic_Vector(4 Downto 0) := "01100";
   Constant State_Write_Cycle3_Write_Setup  : Std_Logic_Vector(4 Downto 0) := "01101";
   Constant State_Write_Cycle3_Write_Pulse  : Std_Logic_Vector(4 Downto 0) := "01110";
   Constant State_Write_Cycle3_Write_Hold   : Std_Logic_Vector(4 Downto 0) := "01111";
   Constant State_Read_Cycle0_Latch         : Std_Logic_Vector(4 Downto 0) := "10000";
   Constant State_Read_Cycle1_Setup         : Std_Logic_Vector(4 Downto 0) := "10001";
   Constant State_Read_Cycle1_Latch         : Std_Logic_Vector(4 Downto 0) := "10010";
   Constant State_Read_Cycle2_Setup         : Std_Logic_Vector(4 Downto 0) := "10011";
   Constant State_Read_Cycle2_Latch         : Std_Logic_Vector(4 Downto 0) := "10100";
   Constant State_Read_Cycle3_Setup         : Std_Logic_Vector(4 Downto 0) := "10101";
   Constant State_Read_Cycle3_ACK           : Std_Logic_Vector(4 Downto 0) := "10110";


    SubType  TByteLane Is Std_Logic_Vector(3 Downto 0);

    Signal State      : Std_Logic_Vector(4 Downto 0);
    Signal State_Next : Std_Logic_Vector(4 Downto 0);
    
    Signal   Request                     : Std_Logic;
    Signal   RequestRead                 : Std_Logic;
    Signal   RequestWrite                : Std_Logic;

    Signal   Cycle_Number                : Std_Logic_Vector(1 DownTo 0);

    Signal   WriteActive_Cycle           : Std_Logic;
    Signal   WE_IfByteSel                : Std_Logic;
    Signal   ACK                         : Std_Logic;

    Signal   CurrentSel                  : Std_Logic;

    Signal   OutData_Latched             : Std_Logic_Vector(23 Downto 0);
    Signal   ADDR                        : Std_Logic_Vector(ReplaceWith_InputAddressWidth-3 Downto 0);

    Constant cTimer_Zero            : Std_Logic_Vector(4 Downto 0) := "00000";
    Constant cTimer_WriteSetup      : Std_Logic_Vector(4 Downto 0) := ReplaceWith_FlashWriteSetup ;
    Constant cTimer_WritePulse      : Std_Logic_Vector(4 Downto 0) := ReplaceWith_FlashWritePulse ;
    Constant cTimer_WriteHold       : Std_Logic_Vector(4 Downto 0) := ReplaceWith_FlashWriteHold ;
    Constant cTimer_ReadSetup       : Std_Logic_Vector(4 Downto 0) := ReplaceWith_FlashReadSetup ;


    Signal Timer                  : Std_Logic_Vector(4 Downto 0);
    Signal TimerIsZero            : Std_Logic;

    Signal LoadTimer_ReadSetup    : Std_Logic;
    Signal LoadTimer_WriteSetup   : Std_Logic;
    Signal LoadTimer_WritePulse   : Std_Logic;
    Signal LoadTimer_WriteHold    : Std_Logic;

    Signal SkipCycle0 : Std_Logic;
    Signal SkipCycle1 : Std_Logic;
    Signal SkipCycle2 : Std_Logic;
    Signal SkipCycle3 : Std_Logic;
Begin
    ----------------------------------------------------------------------------
    ADDR <= ADR_I(ReplaceWith_InputAddressWidth-1 DownTo 2);
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    FLASH_WE  <= WE_IfByteSel Or Not CurrentSel;
    FLASH_CE  <= Not Request;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    FLASH_A(ReplaceWith_AddressWidth-1 DownTo 2) <= ADDR(ReplaceWith_AddressWidth-3 DownTo 0);
    FLASH_A( 1 Downto 0) <= Cycle_Number;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    DAT_O( 7 Downto 0) <= FLASH_D;
    DAT_O(31 DownTo 8) <= OutData_Latched;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    ACK_O <= ACK;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    Request      <= STB_I And CYC_I;
    RequestRead  <= STB_I And CYC_I And (Not WE_I);
    RequestWrite <= STB_I And CYC_I And (    WE_I);
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    SkipCycle0 <= Not SEL_I(3);
    SkipCycle1 <= Not SEL_I(2);
    SkipCycle2 <= Not SEL_I(1);
    SkipCycle3 <= Not SEL_I(0);
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    FLASH_OE <= WriteActive_Cycle;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    Select_SelLink:
    Process(SEL_I,
            Cycle_Number)
    Begin
        Case Cycle_Number Is
           When "00"   => CurrentSel <= SEL_I(3);
           When "01"   => CurrentSel <= SEL_I(2);
           When "10"   => CurrentSel <= SEL_I(1);
           When "11"   => CurrentSel <= SEL_I(0);
           When Others => Null;
        End Case;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    Memory_OuptutControls:
    Process(DAT_I,
            WriteActive_Cycle,
            Cycle_Number)
    Begin
        If WriteActive_Cycle = '1' Then
            Case Cycle_Number Is
               When "00"   => FLASH_D <= DAT_I(31 DownTo 24);
               When "01"   => FLASH_D <= DAT_I(23 DownTo 16);
               When "10"   => FLASH_D <= DAT_I(15 DownTo  8);
               When "11"   => FLASH_D <= DAT_I( 7 DownTo  0);
               When Others => Null;
            End Case;
        Else
           FLASH_D     <= (Others => 'Z');
        End If;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    FSM_Combinational_Single:
    Process(RequestRead,
            RequestWrite,
            State,
            TimerIsZero,
            SkipCycle0,
            SkipCycle1,
            SkipCycle2,
            SkipCycle3)
    Begin
        ------------------------------------------------------------------------
        -- Default outputs
        WriteActive_Cycle    <= '0';
        WE_IfByteSel         <= '1';
        ACK                  <= '0';
        Cycle_Number         <= "00";
        State_Next           <= State;
        LoadTimer_ReadSetup  <= '0';
        LoadTimer_WriteSetup <= '0';
        LoadTimer_WritePulse <= '0';
        LoadTimer_WriteHold  <= '0';
        ------------------------------------------------------------------------

        ------------------------------------------------------------------------
        Case State Is
           ---------------------------------------------------------------------
           When State_AddressSetup =>
                If RequestWrite = '1' Then                --Address is being setup in this phases
                   WriteActive_Cycle <= '1';
                   If SkipCycle0='1' Then
                        If SkipCycle1='1' Then
                            If SkipCycle2='1' Then
                                State_Next <= State_Write_Cycle3_Write_Setup;
                            Else
                                State_Next <= State_Write_Cycle2_Write_Setup;
                            End If;
                        Else
                            State_Next <= State_Write_Cycle1_Write_Setup;
                        End If;
                   Else
                        State_Next        <= State_Write_Cycle0_Write_Setup;
                   End If;

                   LoadTimer_WriteSetup <= '1';
                 ElsIf RequestRead = '1' Then
                    If SkipCycle0='1' Then
                        If SkipCycle1='1' Then
                            If SkipCycle2='1' Then
                                State_Next <= State_Read_Cycle3_Setup;
                            Else
                                State_Next <= State_Read_Cycle2_Setup;
                            End If;
                        Else
                            State_Next <= State_Read_Cycle1_Setup;
                        End If;
                    Else
                        State_Next <= State_Read_Cycle0_Latch;
                    End If;
                   LoadTimer_ReadSetup <= '1';
                End If;
           ---------------------------------------------------------------------

           ---------------------------------------------------------------------
           When State_Write_Cycle0_Write_Setup =>
                WriteActive_Cycle <= '1';
                If TimerIsZero='1' Then
                    LoadTimer_WritePulse <= '1';
                    State_Next <= State_Write_Cycle0_Write_Pulse;
                End If;


           When State_Write_Cycle0_Write_Pulse =>
                WriteActive_Cycle <= '1';
                WE_IfByteSel      <= '0';
                If TimerIsZero='1' Then
                    State_Next <= State_Write_Cycle0_Write_Hold;
                    LoadTimer_WriteHold <= '1';
                End If;

           When State_Write_Cycle0_Write_Hold =>
                WriteActive_Cycle <= '1';
                If TimerIsZero='1' Then
                    If SkipCycle1='1' Then
                        If SkipCycle2='1' Then
                            If SkipCycle3='1' Then
                                State_Next <= State_Write_Cycle0_ACK;
                            Else
                                State_Next <= State_Write_Cycle3_Write_Setup;
                            End If;
                        Else
                            State_Next <= State_Write_Cycle2_Write_Setup;
                        End If;
                    Else
                        State_Next <= State_Write_Cycle1_Write_Setup;
                    End If;
                    LoadTimer_WriteSetup <= '1';
                End If;


           When State_Write_Cycle1_Write_Setup =>
                Cycle_Number      <= "01";
                WriteActive_Cycle <= '1';
                If TimerIsZero='1' Then
                    LoadTimer_WritePulse <= '1';
                    State_Next <= State_Write_Cycle1_Write_Pulse;
                End If;

           When State_Write_Cycle1_Write_Pulse =>
                Cycle_Number      <= "01";
                WriteActive_Cycle <= '1';
                WE_IfByteSel      <= '0';
                If TimerIsZero='1' Then
                    State_Next <= State_Write_Cycle1_Write_Hold;
                    LoadTimer_WriteHold <= '1';
                End If;

           When State_Write_Cycle1_Write_Hold =>
                Cycle_Number      <= "01";
                WriteActive_Cycle <= '1';
                If TimerIsZero='1' Then
                    If SkipCycle2='1' Then
                        If SkipCycle3='1' Then
                            State_Next <= State_Write_Cycle1_ACK;
                        Else
                            State_Next <= State_Write_Cycle3_Write_Setup;
                        End If;
                    Else
                        State_Next <= State_Write_Cycle2_Write_Setup;
                    End If;
                    LoadTimer_WriteSetup <= '1';
                End If;


           When State_Write_Cycle2_Write_Setup =>
                Cycle_Number      <= "10";
                WriteActive_Cycle <= '1';
                If TimerIsZero='1' Then
                    LoadTimer_WritePulse <= '1';
                    State_Next <= State_Write_Cycle2_Write_Pulse;
                End If;

           When State_Write_Cycle2_Write_Pulse =>
                Cycle_Number      <= "10";
                WriteActive_Cycle <= '1';
                WE_IfByteSel      <= '0';
                If TimerIsZero='1' Then
                    State_Next <= State_Write_Cycle2_Write_Hold;
                    LoadTimer_WriteHold <= '1';
                End If;

           When State_Write_Cycle2_Write_Hold =>
                Cycle_Number      <= "10";
                WriteActive_Cycle <= '1';
                If TimerIsZero='1' Then
                    If SkipCycle3='1' Then
                        State_Next <= State_Write_Cycle2_ACK;
                    Else
                        State_Next <= State_Write_Cycle3_Write_Setup;
                    End If;
                    LoadTimer_WriteSetup <= '1';
                End If;


           When State_Write_Cycle3_Write_Setup =>
                Cycle_Number      <= "11";
                WriteActive_Cycle <= '1';
                If TimerIsZero='1' Then
                    LoadTimer_WritePulse <= '1';
                    State_Next <= State_Write_Cycle3_Write_Pulse;
                End If;

           When State_Write_Cycle3_Write_Pulse =>
                Cycle_Number      <= "11";
                WriteActive_Cycle <= '1';
                WE_IfByteSel      <= '0';
                If TimerIsZero='1' Then
                    State_Next <= State_Write_Cycle3_Write_Hold;
                    LoadTimer_WriteHold <= '1';
                End If;

           When State_Write_Cycle3_Write_Hold =>
                Cycle_Number      <= "11";
                WriteActive_Cycle <= '1';
                If TimerIsZero='1' Then
                    State_Next <= State_AddressSetup;
                    ACK <= '1';
                End If;

           When State_Write_Cycle0_ACK =>
                Cycle_Number <= "00";
                WriteActive_Cycle <= '1';
                If TimerIsZero='1' Then
                    State_Next <= State_AddressSetup;
                    ACK <= '1';
                End If;
           When State_Write_Cycle1_ACK =>
                Cycle_Number <= "01";
                WriteActive_Cycle <= '1';
                If TimerIsZero='1' Then
                    State_Next <= State_AddressSetup;
                    ACK <= '1';
                End If;
           When State_Write_Cycle2_ACK =>
                Cycle_Number <= "10";
                WriteActive_Cycle <= '1';
                If TimerIsZero='1' Then
                    State_Next <= State_AddressSetup;
                    ACK <= '1';
                End If;
           ---------------------------------------------------------------------

           ---------------------------------------------------------------------
           When State_Read_Cycle0_Latch =>
                If TimerIsZero='1' Then
                    State_Next    <= State_Read_Cycle1_Setup;
                End If;

           When State_Read_Cycle1_Setup =>
                Cycle_Number  <= "01";
                State_Next    <= State_Read_Cycle1_Latch;
                LoadTimer_ReadSetup <= '1';

           When State_Read_Cycle1_Latch =>
                Cycle_Number  <= "01";
                If TimerIsZero='1' Then
                    If SkipCycle2='1' Then
                        If SkipCycle3='1' Then
                            ACK <= '1';
                            State_Next <= State_AddressSetup;
                        Else
                            State_Next <= State_Read_Cycle3_Setup;
                        End If;
                    Else
                        State_Next    <= State_Read_Cycle2_Setup;
                    End If;
                End If;

           When State_Read_Cycle2_Setup =>
                Cycle_Number  <= "10";
                State_Next    <= State_Read_Cycle2_Latch;
                LoadTimer_ReadSetup <= '1';

           When State_Read_Cycle2_Latch =>
                Cycle_Number  <= "10";
                If TimerIsZero='1' Then
                    If SkipCycle3='1' Then
                        ACK <= '1';
                        State_Next <= State_AddressSetup;
                    Else
                        State_Next    <= State_Read_Cycle3_Setup;
                    End If;
                End If;

           When State_Read_Cycle3_Setup =>
                Cycle_Number  <= "11";
                State_Next    <= State_Read_Cycle3_ACK;
                LoadTimer_ReadSetup <= '1';

           When State_Read_Cycle3_ACK =>                    -- Assert ACK, Master will read the data one clock cycle later
                Cycle_Number  <= "11";
                If TimerIsZero='1' Then
                    ACK           <= '1';
                    State_Next    <= State_AddressSetup;
                End If;
           ---------------------------------------------------------------------

           When Others =>
                State_Next <= State_AddressSetup;

        End Case;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    FSM_Synchronous:
    Process(CLK_I)
    Begin
        If Rising_Edge(CLK_I) Then
           If RST_I = '1' Then
              State  <= State_AddressSetup;
           Else
              State <= State_Next;
           End If;
        End If;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    FSM_LatchData:
    Process(CLK_I)
    Begin
        If Rising_Edge(CLK_I) Then
           Case State Is
              When State_Read_Cycle0_Latch => OutData_Latched(23 DownTo 16) <= FLASH_D;
              When State_Read_Cycle1_Latch => OutData_Latched(15 DownTo  8) <= FLASH_D;
              When State_Read_Cycle2_Latch => OutData_Latched( 7 DownTo  0) <= FLASH_D;
              When Others                  => Null;
           End Case;
        End If;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    Process(CLK_I)
    Begin
        If Rising_Edge(CLK_I) Then
            If RST_I='1' Then
                Timer <= (Others=>'0');
            Else
                   If LoadTimer_ReadSetup ='1' Then Timer <= cTimer_ReadSetup;
                ElsIf LoadTimer_WriteSetup='1' Then Timer <= cTimer_WriteSetup;
                ElsIf LoadTimer_WritePulse='1' Then Timer <= cTimer_WritePulse;
                ElsIf LoadTimer_WriteHold ='1' Then Timer <= cTimer_WriteHold;
                ElsIf TimerIsZero         ='0' Then Timer <= Timer - 1;
                End If;
            End If;
        End If;
    End Process;
    TimerIsZero <= '1' When Timer=cTimer_Zero Else '0';
    ----------------------------------------------------------------------------


End RTL;
--------------------------------------------------------------------------------

