Library IEEE;
Use IEEE.Std_Logic_1164.all;
Use IEEE.std_logic_unsigned.all;

--------------------------------------------------------------------------------
Entity Configurable_ReplaceWith_Designator Is Port
   (
     FLASH_D    : InOut Std_Logic_Vector(31 Downto 0);
     FLASH_A    : Out   Std_Logic_Vector(ReplaceWith_AddressWidth-1 DownTo 0);
     FLASH_OE   : Out   Std_Logic;
     FLASH_WE   : Out   Std_Logic;
     FLASH_CE   : Out   Std_Logic;
     ---------------------------------------------------------------------------

     ---------------------------------------------------------------------------
     -- Wishbone Slave Port
     ---------------------------------------------------------------------------
     CLK_I      : In    Std_Logic;
     RST_I      : In    Std_Logic;
     CYC_I      : In    Std_Logic;
     STB_I      : In    Std_Logic;
     ACK_O      : Out   Std_Logic;
     ADR_I      : In    Std_Logic_Vector(ReplaceWith_InputAddressWidth-1 DownTo 0);
     DAT_I      : In    Std_Logic_Vector(31 DownTo 0);
     DAT_O      : Out   Std_Logic_Vector(31 DownTo 0);
     WE_I       : In    Std_Logic;
     SEL_I      : In    Std_Logic_Vector(3 DownTo 0)
   );

End Configurable_ReplaceWith_Designator;
--------------------------------------------------------------------------------

--------------------------------------------------------------------------------
Architecture RTL of Configurable_ReplaceWith_Designator Is
    Type TState Is ( State_AddressSetup,
                     State_Write_Setup,
                     State_Write_Pulse,
                     State_Write_Hold,
                     State_Write_ACK,
                     State_Read_ACK);

    Signal   State      : TState;
    Signal   State_Next : TState;

    SubType  TByteLane   Is Std_Logic_Vector(3 DownTo 0);
    SubType  TOutBits    Is Std_Logic_Vector(4 DownTo 0);

    ----------------------------------------------------------------------------
    -- Constants for byte lanes - always the same
    ----------------------------------------------------------------------------
    Constant Write_Word           : TByteLane := "1111";
    Constant Write_LoHalf         : TByteLane := "0011";
    Constant Write_HiHalf         : TByteLane := "1100";
    Constant Write_Byte0          : TByteLane := "0001";
    Constant Write_Byte1          : TByteLane := "0010";
    Constant Write_Byte2          : TByteLane := "0100";
    Constant Write_Byte3          : TByteLane := "1000";
    Constant Write_Bytes012       : TByteLane := "0111";
    Constant Write_Bytes123       : TByteLane := "1110";
    Constant Write_Bytes12        : TByteLane := "0110";
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    Constant OutBits_Write_Word     : TOutBits  := "00000";
                                    
    Constant OutBits_Write_LoHalf   : TOutBits  := "11000";
    Constant OutBits_Write_HiHalf   : TOutBits  := "00110";
    Constant OutBits_Write_Byte0    : TOutBits  := "11100";
    Constant OutBits_Write_Byte1    : TOutBits  := "11010";
    Constant OutBits_Write_Byte2    : TOutBits  := "10110";
    Constant OutBits_Write_Byte3    : TOutBits  := "01110";
    Constant OutBits_Write_Bytes012 : TOutBits  := "10000";
    Constant OutBits_Write_Bytes123 : TOutBits  := "00010";
    Constant OutBits_Write_Bytes12  : TOutBits  := "10010";

    Constant OutBits_Read           : TOutBits  := "00000";
    Constant OutBits_None           : TOutBits  := "11111";


    Signal   OutBits              : TOutBits;
    Signal   OutBits_Write        : TOutBits;

    Signal   Request              : Std_Logic;
    Signal   RequestRead          : Std_Logic;
    Signal   RequestWrite         : Std_Logic;

    Signal   WriteActive_Cycle    : Std_Logic;
    Signal   WE                   : Std_Logic;
    Signal   OE                   : Std_Logic;
    Signal   ACK                  : Std_Logic;
    Signal   ADDR                 : Std_Logic_Vector(ReplaceWith_InputAddressWidth-3 Downto 0);

    Constant cTimer_Zero            : Std_Logic_Vector(4 Downto 0) := "00000";
    Constant cTimer_WriteSetup      : Std_Logic_Vector(4 Downto 0) := ReplaceWith_FlashWriteSetup ;
    Constant cTimer_WritePulse      : Std_Logic_Vector(4 Downto 0) := ReplaceWith_FlashWritePulse ;
    Constant cTimer_WriteHold       : Std_Logic_Vector(4 Downto 0) := ReplaceWith_FlashWriteHold ;
    Constant cTimer_ReadSetup       : Std_Logic_Vector(4 Downto 0) := ReplaceWith_FlashReadSetup ;

    Signal Timer                  : Std_Logic_Vector(4 Downto 0);
    Signal TimerIsZero            : Std_Logic;

    Signal LoadTimer_ReadSetup    : Std_Logic;
    Signal LoadTimer_WriteSetup   : Std_Logic;
    Signal LoadTimer_WritePulse   : Std_Logic;
    Signal LoadTimer_WriteHold    : Std_Logic;

Begin
    ----------------------------------------------------------------------------
    ADDR <= ADR_I(ReplaceWith_InputAddressWidth-1 DownTo 2);
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    FLASH_CE   <= OutBits(0) Or (Not Request);
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    FLASH_A    <= ADDR(ReplaceWith_AddressWidth-1 DownTo 0);
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    DAT_O <= FLASH_D;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    ACK_O <= ACK;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    Request      <= STB_I And CYC_I;
    RequestRead  <= STB_I And CYC_I And (Not WE_I);
    RequestWrite <= STB_I And CYC_I And (    WE_I);
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    FLASH_WE <= WE;
    FLASH_OE <= OE;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    Memory_OuptutControls:
    Process(SEL_I,
            DAT_I,
            State,
            OutBits_Write,
            WriteActive_Cycle)
    Begin
        If WriteActive_Cycle = '1' Then
           OutBits   <= OutBits_Write;
           OE        <= '1';
           FLASH_D    <= DAT_I;
        Else
           OutBits   <= OutBits_Read;
           OE        <= '0';
           FLASH_D    <= (Others => 'Z');
        End If;

        Case SEL_I Is
           When Write_Word     => OutBits_Write <= OutBits_Write_Word;
           When Write_LoHalf   => OutBits_Write <= OutBits_Write_LoHalf;
           When Write_HiHalf   => OutBits_Write <= OutBits_Write_HiHalf;
           When Write_Byte0    => OutBits_Write <= OutBits_Write_Byte0;
           When Write_Byte1    => OutBits_Write <= OutBits_Write_Byte1;
           When Write_Byte2    => OutBits_Write <= OutBits_Write_Byte2;
           When Write_Byte3    => OutBits_Write <= OutBits_Write_Byte3;
           When Write_Bytes012 => OutBits_Write <= OutBits_Write_Bytes012;
           When Write_Bytes123 => OutBits_Write <= OutBits_Write_Bytes123;
           When Write_Bytes12  => OutBits_Write <= OutBits_Write_Bytes12;
           When Others         => OutBits_Write <= OutBits_None;
        End Case;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    FSM_Combinational_Single:
    Process(RequestRead,
            RequestWrite,
            State,
            TimerIsZero)
    Begin
        ------------------------------------------------------------------------
        -- Default outputs
        WriteActive_Cycle    <= '0';
        WE                   <= '1';
        ACK                  <= '0';
        State_Next           <= State;
        LoadTimer_ReadSetup  <= '0';
        LoadTimer_WriteSetup <= '0';
        LoadTimer_WritePulse <= '0';
        LoadTimer_WriteHold  <= '0';
        ------------------------------------------------------------------------

        Case State Is
           ---------------------------------------------------------------------
           When State_AddressSetup =>
                If RequestWrite = '1' Then                --Address is being setup in this phases
                   WriteActive_Cycle <= '1';
                   LoadTimer_WriteSetup <= '1';
                   State_Next <= State_Write_Setup;
                 ElsIf RequestRead = '1' Then
                   LoadTimer_ReadSetup <= '1';
                   State_Next <= State_Read_ACK;
                End If;
           ---------------------------------------------------------------------

           ---------------------------------------------------------------------
           When State_Write_Setup =>
                WriteActive_Cycle <= '1';
                If TimerIsZero='1' Then
                    State_Next <= State_Write_Pulse;
                    LoadTimer_WritePulse <= '1';
                End If;

           When State_Write_Pulse =>
                WriteActive_Cycle <= '1';
                WE                <= '0';
                If TimerIsZero='1' Then
                    State_Next        <= State_Write_Hold;
                    LoadTimer_WriteHold <= '1';
                End If;

           When State_Write_Hold =>
                WriteActive_Cycle <= '1';
                If TimerIsZero='1' Then
                    State_Next  <= State_Write_ACK;
                End If;

           When State_Write_ACK =>
                WriteActive_Cycle <= '1';
                ACK               <= '1';
           ---------------------------------------------------------------------

           ---------------------------------------------------------------------
           When State_Read_ACK =>                    -- Assert ACK, Master will read the data one clock cycle later
                If TimerIsZero='1' Then
                    ACK           <= '1';
                    State_Next    <= State_AddressSetup;
                End if;
           ---------------------------------------------------------------------

           When Others =>
                State_Next <= State_AddressSetup;

        End Case;
    End Process;
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    Process(CLK_I)
    Begin
        If Rising_Edge(CLK_I) Then
            If RST_I='1' Then
                Timer <= (Others=>'0');
            Else
                   If LoadTimer_ReadSetup ='1' Then Timer <= cTimer_ReadSetup;
                ElsIf LoadTimer_WritePulse='1' Then Timer <= cTimer_WritePulse;
                ElsIf LoadTimer_WriteSetup='1' Then Timer <= cTimer_WriteSetup;
                ElsIf LoadTimer_WriteHold ='1' Then Timer <= cTimer_WriteHold;
                ElsIf TimerIsZero         ='0' Then Timer <= Timer - 1;
                End If;
            End If;
        End If;
    End Process;
    TimerIsZero <= '1' When Timer=cTimer_Zero Else '0';
    ----------------------------------------------------------------------------

    ----------------------------------------------------------------------------
    FSM_Synchronous:
    Process(CLK_I)
    Begin
        If Rising_Edge(CLK_I) Then
           If RST_I = '1' Then
              State  <= State_AddressSetup;
           Else
              State <= State_Next;
           End If;
        End If;
    End Process;
    ----------------------------------------------------------------------------

End RTL;
--------------------------------------------------------------------------------

