//----------------------------------------------------------------------------
//
// cppOutputPolygons.cpp
//
// Description:
//
// This sample program provides a template to output tesselated data from a
// Studio wire file. It reads a wire file and outputs a single wavefront OBJ
// file. Note that animation is not supported.
//
// All routines for this app are in this souce file. Follow the source starting
// at main for easiest understanding.
//
//----------------------------------------------------------------------------

//..........................................................................//
//    Inclusions
//..........................................................................//

#include <stdio.h>
#include <vector>
namespace std {};
using namespace std;

#include <AlUniverse.h>

#include <AlDagNode.h>
#include <AlGroupNode.h>

#include <AlMesh.h>
#include <AlMeshNode.h>
#include <AlTesselate.h>

//..........................................................................//
//    Typedefs
//..........................................................................//

typedef vector<AlDagNode*> dagNodeVector;
typedef dagNodeVector::iterator dagNodeIterator;

typedef vector<AlMesh*> meshVector;
typedef meshVector::iterator meshIterator;

typedef enum 
{
    fast,
    accurate
} tesselator;

//..........................................................................//
//    Local Variables
//..........................................................................//

const int pathLen = 1024;

//..........................................................................//
//    Local Functions
//..........................................................................//

// printError
// Prints 'errStr' to stderr followed by the optional parameter paramStr.
void printError( const char * errStr, const char * paramStr = 0 )
{
    if(paramStr != 0)
    {
        fprintf(stderr,errStr,paramStr);
    }
    else
    {
        fprintf(stderr, "%s", errStr);
    }
}

// printUsage
// Prints a message to stderr informing the user how to use this application.
void printUsage( char **argv )
{

    printError("usage: %s -i <inputfile> -o <outputfile>\n"
               "          [-t] [-f] [-a]\n", argv[0]);
    printError("Where: -i <inputfile>   specifies a valid Alias wire file for\n"
               "                        input.\n"
               "       -o <outputfile>  specifies the name of the obj file for\n"
               "                        output.\n"
               "\n"
               "       -f (default)     specifies the fast tesselator.\n"
               "       -a               specifies the accurate tesselator.\n"
               "\n"
               "       -t (default=0.1) specifies the tesselation tolerance.\n"
               "\n"
               "Description:\n"
               "  This program retrieves a wire file, tesselates all tesselatable\n"
               "  surfaces and outputs one obj file.\n");
}

// parseArguments
// Parses the argument list passed to the app.
void parseArguments( int argc, char **argv, char * inputFile, char * outputFile, tesselator &tessType, double &tolerance )
{
    for(int ii = 1; ii < argc; ii++)
    {
        if (argv[ii][0] == '-')
        {
            switch(argv[ii][1])
            {
                // inputFile
                case 'i':
                case 'I':
                {
                    if((ii+1) < argc)
                    {
                        strncpy(inputFile,argv[ii+1],pathLen);
                    }
                    break;
                }
                // outputFile
                case 'o':
                case 'O':
                {
                    if((ii+1) < argc)
                    {
                        strncpy(outputFile,argv[ii+1],pathLen);
                    }
                    break;
                }
                
                // tessType
                case 'f':
                case 'F':
                {
                    tessType = fast;
                    break;
                }
                case 'a':
                case 'A':
                {
                    tessType = accurate;
                    break;
                }

                // tolerance
                case 't':
                case 'T':
                {
                    if((ii+1) < argc)
                    {
                        tolerance = atof(argv[ii+1]);
                    }
                    break;
                }
                
                default:
                {
                    break;
                }
            }
        }// if ( argv[ii][0] == '-' )
    }// for(int ii = 1; ii < (argc-1); ii++)
}

// recurseDagForLeaves
// Recurses the DAG looking for leaf nodes starting with 'dagNode'. If found
// the leaves are added to the end of 'leaves'. This function provies sample
// code for traversing the DAG. All the nodes retrieved from the api are copies.
// We must therefore delete all node pointers returned from the api. However,
// we need to store the leaf nodes for later use. To get around this problem,
// we delete all non-leaf nodes while traversing the DAG. We then delete the
// leaves once we've finished our processing.
void recurseDagForLeaves( dagNodeVector & leaves, AlDagNode * dagNode )
{
    // Filter invalid nodes.
    if(AlIsValid(dagNode) == FALSE)
    {
        return;
    }

    // If this is a leaf we dont want to delete the current node.
    bool isLeaf = false;

    // Process the current node.
    switch(dagNode->type())
    {
        // Push all leaf nodes into 'leaves'
        case(kShellNodeType):
        case(kSurfaceNodeType):
        {
            leaves.push_back(dagNode);
            isLeaf = true;
            break;
        }

        // Traverse down through groups
        case(kGroupNodeType):
        {
            AlGroupNode * groupNode = dagNode->asGroupNodePtr();
            if(AlIsValid(groupNode)==TRUE)
            {
                AlDagNode * childNode = groupNode->childNode();
                if(AlIsValid(groupNode)==TRUE)
                {
                    recurseDagForLeaves(leaves,childNode);
                }
            }
            break;
        }
    }

    // Grab the next sibling before deleting the node.
    AlDagNode * siblingNode = dagNode->nextNode();

    // Delete the current node if it isn't a leaf.
    if(isLeaf == false)
    {
        delete dagNode;
    }
    
    // Go to the next sibling.
    if(AlIsValid(siblingNode)==TRUE)
    {
        recurseDagForLeaves(leaves,siblingNode);
    }
}

// getDagLeaves
// Recurses the entire DAG for leaf nodes using 'recurseDagForLeaves'.
dagNodeVector getDagLeaves( void )
{
    dagNodeVector leaves;
    recurseDagForLeaves(leaves,AlUniverse::firstDagNode());
    return leaves;
}

// deleteDagNodeVector
// Everything retrieved from the api is a copy and must be deleted after use.
void deleteDagNodeVector( dagNodeVector & nodes )
{
    for(dagNodeIterator ii = nodes.begin(); ii != nodes.end(); ii++)
    {
        delete *ii;
    }

    nodes.clear();
}

// tesselateDagLeaves
// Tesselates the geometry of 'leaves' using the 'tessType' tesselator with 
// the appropriate 'tolerance'.
dagNodeVector tesselateDagLeaves( dagNodeVector dagLeaves, tesselator tessType, double tolerance )
{
    dagNodeVector tesselatedNodes;
    
    for(dagNodeIterator ii = dagLeaves.begin(); ii != dagLeaves.end(); ii++)
    {
        AlDagNode * newNode;
        statusCode tessStatus;

        switch(tessType)
        {
            case(accurate):
            {
                tessStatus = AlTesselate::chordHeightDeviationAccurate(newNode,*ii,tolerance);
                break;
            }
            case(fast):
            default:
            {
                tessStatus = AlTesselate::chordHeightDeviationFast(newNode,*ii,tolerance);
                break;
            }
        }

        if((tessStatus == sSuccess) && (AlIsValid(newNode) == TRUE))
        {
            tesselatedNodes.push_back(newNode);
        }
    }

    return tesselatedNodes;
}

// getMeshesFromNodes
// Returns a vector of mesh pointers from a vector of dag node pointers.
// This routine assumes that the dag nodes contain meshes.
meshVector getMeshesFromNodes( dagNodeVector nodes )
{
    meshVector meshes;
    
    for(dagNodeIterator ii = nodes.begin(); ii != nodes.end(); ii++)
    {
        AlMeshNode * newMeshNode = (AlMeshNode*)(*ii);
        if(AlIsValid(newMeshNode) == TRUE )
        {
            AlMesh * newMesh = newMeshNode->mesh();
            if(AlIsValid(newMesh) == TRUE)
            {
                meshes.push_back(newMesh);
            }
        }
    }

    return meshes;
}

// deleteMeshVector
// Everything retrieved from the api is a copy and must be deleted after use.
void deleteMeshVector( meshVector & meshes )
{
    for(meshIterator ii = meshes.begin(); ii != meshes.end(); ii++)
    {
        delete *ii;
    }

    meshes.clear();
}

// writeMeshesToObj
// Writes the geometry information contained in 'meshes' to 'outputFile'.
// No error checking is done on the mesh pointers.
void writeMeshesToObj( meshVector meshes, char * outputFile )
{
    FILE * fp = fopen(outputFile,"w");

    // Check for io errors.
    if(fp == 0)
    {
        printError("Could not open file for write: ",outputFile);
        return;
    }

    // Write a header.
    fprintf(fp,"# Wavefront OBJ generated by cppOutputPolygons.exe.\n");

    // An index to iterate through meshes.
    int ii = 0;

    // Write the vertices.
    for(ii = 0; ii != meshes.size(); ii++)
    {
        int numVertices = meshes[ii]->numberOfVertices();
        const float * vertices = meshes[ii]->vertices();

        // Write a comment into the file
        fprintf(fp,"# Writing vertices for mesh %d.\n",ii);
                
        for(int jj = 0; jj < numVertices; jj++)
        {
            const float * curVertex = &vertices[3*jj];
            fprintf(fp,"v %g %g %g\n",curVertex[0],curVertex[1],curVertex[2]);
        }
    }

    // Write the normals.
    for(ii = 0; ii != meshes.size(); ii++)
    {
        int numNormals = meshes[ii]->numberOfVertices();
        const float * normals = meshes[ii]->normals();

        // Write a comment into the file
        fprintf(fp,"# Writing normals for mesh %d.\n",ii);
 
        for(int jj = 0; jj < numNormals; jj++)
        {
            const float * curNormal = &normals[3*jj];
            fprintf(fp,"vn %g %g %g\n",curNormal[0],curNormal[1],curNormal[2]);
        }
    }

    // Write the uvs (texture coords).
    for(ii = 0; ii != meshes.size(); ii++)
    {
        int numUVs = meshes[ii]->numberOfVertices();
        const float * uvs = meshes[ii]->uvs();

        // Write a comment into the file
        fprintf(fp,"# Writing uvs for mesh %d.\n",ii);

        // The mesh may not contain uvs; we have to accomodate for this
        // possibility.
        if(uvs != 0)
        {
            for(int jj = 0; jj < numUVs; jj++)
            {
                const float * curUV = &uvs[2*jj];
                fprintf(fp,"vt %g %g\n",curUV[0],curUV[1]);
            }
        }
        else
        {
            for(int jj = 0; jj < numUVs; jj++)
            {
                fprintf( fp, "vt %g %g\n", 0.0, 0.0 );
            }
        }
    }

    // Write out the faces.
    // We need to update the face indices accordingly as we go from
    // mesh to mesh.
    int indexCounter = 0;
    for(ii = 0; ii != meshes.size(); ii++)
    {
        int numTriangles = meshes[ii]->numberOfTriangles();
        const int * triangles = meshes[ii]->triangles();

        // Write a comment into the file
        fprintf(fp,"# Writing faces for mesh %d.\n",ii);

        for(int jj = 0; jj < numTriangles; jj++)
        {
            const int * curTriangle = &triangles[3*jj];

            // Triangles in obj files use 1 based indexing.
            int v1 = indexCounter + curTriangle[0] + 1;
            int v2 = indexCounter + curTriangle[1] + 1;
            int v3 = indexCounter + curTriangle[2] + 1;

            fprintf(fp, "f %d/%d/%d %d/%d/%d %d/%d/%d\n",
                    v1, v1, v1,
                    v2, v2, v2,
                    v3, v3, v3
                    );
        }

        // Update the index count so that the next mesh references
        // it's vertices accordingly.
        indexCounter += meshes[ii]->numberOfVertices();
    }

    fclose(fp);
}

//..........................................................................//
//    Main
//..........................................................................//

int main( int argc, char **argv )
{
    // Check for an invalid argument list argument list.
    if (argc <= 1)
    {
        printUsage(argv);
        exit(0);
    }

    // .wire file for input.
    char inputFile[pathLen] = "";
    // .obj file for output.
    char outputFile[pathLen] = "";
    // The type of tesselator to be used.
    tesselator tessType = fast;
    // The tolerance of the tesselator.
    double tolerance = 0.1;
    
    // Initialize arguments.
    parseArguments(argc,argv,inputFile,outputFile,tessType,tolerance);

    // Initialize Alias.
    AlUniverse::initialize();
    if (AlUniverse::retrieve(inputFile) != sSuccess)
    {
        printError("Could not retrieve %s\n",inputFile);
        exit(-1);
    }

    // Parse and extract the DAG leaf nodes.
    dagNodeVector dagLeaves = getDagLeaves();

    // Tesselate the nodes and remove the old DAG leaves.
    dagNodeVector tesselatedNodes = tesselateDagLeaves(dagLeaves,tessType,tolerance);
    deleteDagNodeVector(dagLeaves);

    // Get the meshes from the dag nodes. Note that removing the mesh's DAG.
    // will also removes the meshes, so we have to do it later.
    meshVector meshes = getMeshesFromNodes(tesselatedNodes);

    // Write the resulting meshs to OBJ and remove the meshes from the DAG.
    writeMeshesToObj(meshes,outputFile);
    deleteMeshVector(meshes);
    deleteDagNodeVector(tesselatedNodes);

    return 0;
}

//----------------------------------------------------------------------------
// cppOutputPolygons.cpp
//----------------------------------------------------------------------------
