#include <stdio.h>
#include <stdlib.h>
#include <AlUniverse.h>
#include <AlCluster.h>
#include <AlClusterNode.h>
#include <AlClusterMember.h>
#include <AlDagNode.h>
#include <AlGroupNode.h>
#include <AlSurface.h>
#include <AlSurfaceNode.h>
#include <AlSurfaceCV.h>
#include <AlFace.h>
#include <AlFaceNode.h>
#include <AlCurve.h>
#include <AlCurveNode.h>
#include <AlCurveCV.h>
#include <AlTM.h>

#include <AlPrint.h>

void cluster_make( char* );
void print_all_clusters( void );
void addCVsToCluster( AlCluster*, AlDagNode* );

static double globalPercent = 0.0;

//
// This program reads in a file, then makes two new clusters consisting
// of everything in the file.  In stage one all root level dag nodes are
// added to cluster1, which is an exclusive cluster.  Anything currently
// in a cluster will not be added to cluster1.  In the second stage all
// clusters are made into multi clusters then all root dag nodes are
// added to a new cluster, cluster2.  Both clusters are printed at each
// stage of development, as appropriate.
//
int
main( int argc, char** argv )
{
	int which;

	if( argc < 2 )
	{
		fprintf( stderr, "usage: %s <filename> {<filename>}* \n",argv[0] );
		fprintf( stderr,
			"Description:\n"
			"  This program reads in a file and makes new clusters\n"
			"  out of all objects, giving an example of how to use\n"
			"  the AlCluster class in cluster creation.\n"
			);
		exit( 0 );
	}

	AlUniverse::initialize( kYUp );

	for( which = 1; which < argc; which++ )
		{
		fprintf( stdout, "Testing file %s\n", argv[which] );
		cluster_make( argv[which] );
		}

    return 0;
}

void
cluster_make( char *filename )
{
	char	newname[256];

	switch( AlUniverse::retrieve( filename ) )
	{
		case sSuccess:
			fprintf( stdout, "Retrieve of '%s' succeeded\n", filename );
		break;

		case sFailure:
			fprintf( stdout, "Retrieve failed\n" );
			return;
		default:
			fprintf( stdout, "Retrieve failed for unknown reason\n" );
			return;
	}

	// Try to add all dag nodes to the first cluster
	//
	AlCluster*	cluster1 = new AlCluster();
	double		percent = 0.1;
	if( cluster1 != (AlCluster*) 0 )
		{
		cluster1->create();
		cluster1->setClusterRestrict( kExclusiveCluster );
		fprintf( stdout, "Created cluster 1 as %p\n", cluster1 );

		AlDagNode *tmpNode = 0;
		for( AlDagNode *dagNode = AlUniverse::firstDagNode();
			dagNode != (AlDagNode*) 0;
			dagNode = tmpNode
			)
			{
			dagNode->addToCluster( cluster1, percent );
			percent += 0.1;
			tmpNode = dagNode->nextNode();
			delete dagNode;
			};

		delete cluster1;
		}
	else
		fprintf( stdout, "ERROR: Failed to create first cluster\n" );

	fprintf( stdout, "One Cluster Created\n" );
	fprintf( stdout, "===================\n" );
	print_all_clusters();

	// Create the second cluster with nothing in it yet
	//
	AlCluster*	cluster2 = new AlCluster();
	if( cluster2 != (AlCluster*) 0 )
		{
		cluster2->create();
		cluster2->setClusterRestrict( kExclusiveCluster );
		}
	else
		fprintf( stderr, "ERROR: Failed to create second cluster\n" );

	fprintf( stdout, "Still One Empty Cluster\n" );
	fprintf( stdout, "=======================\n" );
	print_all_clusters();

	// Change all clusters to Multi-cluster and add all CVs to the newest
	// cluster (cluster2).
	//
	AlCluster	*cluster = AlUniverse::firstCluster();
	if( cluster != (AlCluster*) 0 )
		{
		do
			{
			cluster->setClusterRestrict( kMultiCluster );
			} while( cluster->nextClusterD() == sSuccess );
		delete cluster;
		}

	if( cluster2 != (AlCluster*) 0 )
		{
		globalPercent = 0.05;
		addCVsToCluster( cluster2, AlUniverse::firstDagNode() );
		}

	delete cluster2;
	fprintf( stdout, "Two Clusters\n" );
	fprintf( stdout, "============\n" );
	print_all_clusters();

	// Store the transformed data
	//
	sprintf( newname, "%s.CL", filename );
	AlUniverse::store( newname );
	return;
}

void
print_all_clusters( void )
//
//	Description:
//		Walk the list of all clusters in the universe and print them out
//
{

	fprintf( stdout, "List of Clusters:\n" );
	fprintf( stdout, "=================\n" );
	AlCluster	*cluster = AlUniverse::firstCluster();
		 
	fprintf( stdout, "{\n" );
	int i = 1;
	if( cluster != (AlCluster*) 0 )
		{
		do
			{
			fprintf( stdout, "Now printing cluster %d \n", i );
			fprintf( stdout, "{\n" );
			AlTM tm = AlTM::identity();
			printAlCluster( cluster, tm );
			fprintf( stdout, "}\n" );
			i++;
			} while( cluster->nextClusterD() == sSuccess );
		delete cluster;
		}
	fprintf( stdout, "}\n" );
	return;
}

void
addCVsToCluster(
	AlCluster*	cluster,
	AlDagNode*	dagNode )
{
	/*********   FIREWALL   *********/
	if( NULL == dagNode )
		return;

	// If the given dag node is a cluster node, then print out its name
	//
	switch( dagNode->type() )
	{
		case kCurveNodeType:
		{
			AlCurveNode	*curveNode = dagNode->asCurveNodePtr();
			if( curveNode == NULL ) break;

			AlCurve *curve = curveNode->curve();
    		AlCurveCV *curveCV = curve->firstCV();
		    if( curveCV != (AlCurveCV*) 0 )
				{
				do
					{
					curveCV->addToCluster( cluster, globalPercent );
					globalPercent += 0.05;
					} while( curveCV->nextD() == sSuccess );
				delete curveCV;
				}
			delete curve;
		}
		break;

		case kFaceNodeType:
		{
			AlFaceNode*	faceNode = dagNode->asFaceNodePtr();
			if( faceNode == NULL ) break;

			AlFace	*face = faceNode->firstFace();
			if( face != (AlCurve *) 0 )
			{
				AlCurveCV *curveCV;
				do
				{
					curveCV = face->firstCV();
					if( curveCV != (AlCurveCV*) 0 )
					{
						do
						{
							curveCV->addToCluster( cluster, globalPercent );
							globalPercent += 0.05;
						} while( curveCV->nextD() == sSuccess );
						delete curveCV;
					}
				} while( face->nextFaceD() == sSuccess );
				delete face;
			}
		}
		break;

		case kSurfaceNodeType:
		{
			AlSurfaceNode	*surfaceNode = dagNode->asSurfaceNodePtr();
			if( surfaceNode == NULL ) break;

			AlSurface *surface = surfaceNode->surface();
			if( surface == (AlSurface*) 0 )
				{
				delete surfaceNode;
				break;
				}

			AlSurfaceCV* lastCV = surface->firstCV();
			if( lastCV != (AlSurfaceCV*) 0 )
				 {
				 AlSurfaceCV* surfaceCV;
				 do
					{
					surfaceCV = lastCV;
					do
						{
						surfaceCV->addToCluster( cluster, globalPercent );
						globalPercent += 0.05;
						} while( surfaceCV->nextInVD() == sSuccess );

					delete surfaceCV;
					} while( lastCV->nextInUD() == sSuccess );
				//delete lastCV;
				}
			delete surface;
		}
		break;

		case kGroupNodeType:
		{
			// if the starting node is a group dag node, check its
			// list of children to see if a child is a cluster node
			//
			AlGroupNode *groupNode = dagNode->asGroupNodePtr();
			if( groupNode != (AlGroupNode *) 0 )
				{
				// Traverse the dag.  For the child under this dag, 
				// call addCVsToCluster().
				//
				AlDagNode *childNode = groupNode->childNode();
				if( childNode != (AlDagNode *) 0 )
					{
					addCVsToCluster( cluster, childNode );
					delete childNode;
					}
				}
		}
		break;

		default:
		break;
	}

	// Continue to traverse the dag.
	//
	AlDagNode *siblingNode = dagNode->nextNode();
	if( siblingNode != (AlDagNode *) 0 )
		{
		addCVsToCluster( cluster, siblingNode );
		delete siblingNode;
		}

	return;
}
