
#include <stdio.h>
#include <AlUniverse.h>
#include <AlCurve.h>
#include <AlCurveNode.h>
#include <AlSurface.h>
#include <AlSurfaceNode.h>

//
//	This example illustrates how to call the AlCurve and AlSurface
//	creation routines to make periodic, open and closed curves and 
//	surfaces.
//	This code covers the redundant knot cases required by two of
//	the ::create() methods.
//

//
//	Prototypes
//
static statusCode addCurveToDag( AlCurve *curve, const char *nodeName );
static statusCode addSurfaceToDag( AlSurface *surf, const char *nodeName );

//
//	Example code to show the creation of curves with the
//	::create() method that does not require redundant
//	knot values.
//

void createPeriodicCurve()
{ 
	statusCode result = sFailure;
	AlCurve *curve = new AlCurve;
	if ( curve == NULL )
		return;
			
	int degree = 3;	
	curveFormType curveForm = kPeriodic;
		
	// For a period curve and the non redundant knots creation method, 
	//	numKnots = numControlPts + 1
	int numKnots = 6;		
	const double knotVector[] = { 0, 1, 2, 3, 4, 5 };						 
	int numControlPts = 5;	
	
	// Don't specify the overlapping control points.
	const double controlPoints[][4] = 
	{ 
		{ 0, 0, 0, 1 },
		{ 1, 0, 0, 1 },
		{ 1, 2, 0, 1 },
		{ 2, 5, 0, 1 },
		{ 3, 9, 0, 1 }
	};	
	
	// Same length as the number of controlPoints.  Values can either be 1,2 or 3 but
	// the first and last spots must be 1.
	const int multiplicity[] = { 1, 1, 1, 1, 1 };  
														
	result = curve->create( degree, curveForm, numKnots, knotVector, numControlPts, controlPoints, multiplicity );
	if ( result != sSuccess )
		printf("createPeriodicCurve: failed....\n");
	
	if ( result == sSuccess )
		addCurveToDag( curve, "PeriodicCurve" );

	if ( curve != NULL )
		delete curve;
}

void createOpenCurve()
{ 
	statusCode result = sFailure;
	AlCurve *curve = new AlCurve;
	if ( curve == NULL )
		return;

	int degree = 3;	
	curveFormType curveForm = kOpen;
		
	// For a open curve and the non redundant knots creation method, 
	//	 numKnots = numControlPts - deg + 1
	int numKnots = 3;		
	const double knotVector[] = { 0, 1, 2 };						 
	int numControlPts = 5;	
	const double controlPoints[][4] = 
	{ 
		{ 0, 0, 0, 1 },
		{ 1, 0, 0, 1 },
		{ 1, 2, 0, 1 },
		{ 2, 5, 0, 1 },
		{ 3, 9, 0, 1 }
	};
		
	// Same length as the number of controlPoints.  Values can either be 1,2 or 3 but
	// the first and last spots must be 1.
	const int multiplicity[] = { 1, 1, 1, 1, 1 };  
														
	result = curve->create( degree, curveForm, numKnots, knotVector, numControlPts, controlPoints, multiplicity );
	if ( result != sSuccess )
		printf("createOpenCurve: failed....\n");

	if ( result == sSuccess )
		addCurveToDag( curve, "OpenCurve" );

	if ( curve != NULL )
		delete curve;
}

void createClosedCurve()
{ 
	statusCode result = sFailure;
	AlCurve *curve = new AlCurve;
	if ( curve == NULL )
		return;

	int degree = 3;	
	curveFormType curveForm = kClosed;	
	
	// For a closed curve and the non redundant knots creation method, 
	//	 numKnots = numControlPts - deg + 1
	int numKnots = 3;		
	const double knotVector[] = { 0, 1, 2 };						 
	int numControlPts = 5;
	
	const double controlPoints[][4] = 
	{ 
		{ 0, 0, 0, 1 },
		{ 1, 0, 0, 1 },
		{ 1, 2, 0, 1 },
		{ 2, 5, 0, 1 },
		{ 0, 0, 0, 1 }	// The last point overlaps the first	
	};
		
	// Same length as the number of controlPoints.  Values can either be 1,2 or 3 but
	// the first and last spots must be 1.
	const int multiplicity[] = { 1, 1, 1, 1, 1 };  
														
	result = curve->create( degree, curveForm, numKnots, knotVector, numControlPts, controlPoints, multiplicity );
	if ( result != sSuccess )
		printf("createClosedCurve: failed....\n");

	if ( result == sSuccess )
		addCurveToDag( curve, "ClosedCurve" );

	if ( curve != NULL )
		delete curve;
}
 
//
//	Example code to show the creation of curves with the
//	::create() method that does requires redundant
//	knot values.
//

void createPeriodicCurve2()
{ 
	statusCode result = sFailure;
	AlCurve *curve = new AlCurve;
	if ( curve == NULL )
		return;

	int degree = 3;	
	curveFormType curveForm = kPeriodic;
		
	// For a periodic curve and the redundant knots creation method, 
	//	 numKnots = numControlPts + deg + 1
	const int numKnots = 14;	
	
	//  For i = 0 to 2*deg-1 (n = numberSpans)
	//			Knot    -  Knot  = Knot      - Knot
	//			   i+1         i       i+n+1       i+n
	//
	//	This is the same as saying that the knot differences
	//	must be the same for the knot positions described 
	//	by the equation.
	const double knotVector[numKnots] = { -4, -3, -2, -1, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 };
							 
	int numControlPts = 10;	
	const double controlPoints[][4] = 
	{ 
		{ 0, 0, 0, 1 },
		{ 0, 0, 0, 1 },
		{ 0, 0, 0, 1 },
		{ 1, 0, 0, 1 },
		{ 1, 2, 0, 1 },
		{ 2, 5, 0, 1 },
		{ 3, 9, 0, 1 },
		{ 0, 0, 0, 1 },	// Here we specify the overlapping points.
		{ 0, 0, 0, 1 },
		{ 0, 0, 0, 1 },
	};	
														
	result = curve->create( degree, curveForm, knotVector, numControlPts, controlPoints );
	if ( result != sSuccess )
		printf("createPeriodicCurve2: failed....\n");

	if ( result == sSuccess )
		addCurveToDag( curve, "PeriodicCurve2" );

	if ( curve != NULL )
		delete curve;
}

void createOpenCurve2()
{ 
	statusCode result = sFailure;
	AlCurve *curve = new AlCurve;
	if ( curve == NULL )
		return;

	int degree = 3;	
	curveFormType curveForm = kOpen;
		
	// For an open curve and the redundant knots creation method, 
	//	 numKnots = numControlPts + deg + 1
	const int numKnots = 9;
	
	// We are specifying the redundant knot values.  The knot
	// values bounding the first or last spans must be the same.		
	const double knotVector[numKnots] = { 0, 0, 0, 0, 2, 4, 4, 4, 4 };						 
	int numControlPts = 5;	
	const double controlPoints[][4] = 
	{ 
		{ 0, 0, 0, 1 },
		{ 1, 0, 0, 1 },
		{ 1, 2, 0, 1 },
		{ 2, 5, 0, 1 },
		{ 3, 9, 0, 1 },
	};	
														
	result = curve->create( degree, curveForm, knotVector, numControlPts, controlPoints );
	if ( result != sSuccess )
		printf("createOpenCurve2: failed....\n");

	if ( result == sSuccess )
		addCurveToDag( curve, "OpenCurve2" );

	if ( curve != NULL )
		delete curve;
}

void createClosedCurve2()
{ 
	statusCode result = sFailure;
	AlCurve *curve = new AlCurve;
	if ( curve == NULL )
		return;

	int degree = 3;	
	curveFormType curveForm = kClosed;	
	
	// For a closed curve and the redundant knots creation method,
	//	 numKnots = numControlPts + deg + 1
	const int numKnots = 9;		

	// We are specifying the redundant knot values.  The knot
	// values bounding the first or last spans must be the same.		
	const double knotVector[numKnots] = { 0, 0, 0, 0, 2, 4, 4, 4, 4 };						 
	int numControlPts = 5;
		
	const double controlPoints[][4] = 
	{ 
		{ 0, 0, 0, 1 },
		{ 1, 0, 0, 1 },
		{ 1, 2, 0, 1 },
		{ 2, 5, 0, 1 },
		{ 0, 0, 0, 1 }	// First and last points overlap.	
	};	
														
	result = curve->create( degree, curveForm, knotVector, numControlPts, controlPoints );
	if ( result != sSuccess )
		printf("createClosedCurve2: failed....\n");

	if ( result == sSuccess )
		addCurveToDag( curve, "ClosedCurve2" );

	if ( curve != NULL )
		delete curve;
}

//
//	Example code to show the creation of surfaces with the
//	::create() method that does not require redundant
//	knot values.
//

void createPeriodicSurface()
{ 
	statusCode result = sFailure;
	AlSurface *surf = new AlSurface;
	if ( surf == NULL )
		return;
		
	int uDeg = 3;
	int vDeg = 3;
	curveFormType uForm = kPeriodic;
	curveFormType vForm = kPeriodic;
	
	// For a periodic surface and the non redundant creation method, 
	//	numKnots = numControlPts + 1
	int uNumKnots = 6; 
	int vNumKnots = 6;
	
	const double uKnotVector[] = { 0, 1, 2, 3, 4, 5 };
	const double vKnotVector[] = { 0, 1, 2, 3, 4, 5 };
	int uNumControlPts = 5;
	int vNumControlPts = 5;
	
	// Do not need to specify the overlapping points.
	// Array is of type controlPoints[uNumControlPts][vNumControlPts][4]
	const double controlPoints[][4] = {
		{ 0.0, 0.0, 0.0, 1.0 }, // first u row
		{ 1.0, 0.0, 0.0, 1.0 },
		{ 2.0, 0.0, 0.0, 1.0 },
		{ 3.0, 0.0, 0.0, 1.0 },
		{ 4.0, 0.0, 0.0, 1.0 },
		
		{ 0.0, 1.0, 1.0, 1.0 }, // second u row
		{ 1.0, 1.0, 1.0, 1.0 },
		{ 2.0, 1.0, 1.0, 1.0 },
		{ 3.0, 1.0, 1.0, 1.0 },
		{ 4.0, 1.0, 1.0, 1.0 },
		
		{ 0.0, 2.0, 2.0, 1.0 }, // third u row
 		{ 1.0, 2.0, 2.0, 1.0 },
		{ 2.0, 2.0, 2.0, 1.0 },
		{ 3.0, 2.0, 2.0, 1.0 },
		{ 4.0, 2.0, 2.0, 1.0 },
		
		{ 0.0, 3.0, 1.0, 1.0 }, // fourth u row
		{ 1.0, 3.0, 1.0, 1.0 },
		{ 2.0, 3.0, 1.0, 1.0 },
		{ 3.0, 3.0, 1.0, 1.0 },
		{ 4.0, 3.0, 1.0, 1.0 },
		
		{ 0.0, 4.0, 0.0, 1.0 }, // fifth u row
		{ 1.0, 4.0, 0.0, 1.0 },
		{ 2.0, 4.0, 0.0, 1.0 },
		{ 3.0, 4.0, 0.0, 1.0 },
		{ 4.0, 4.0, 0.0, 1.0 }	
	};
	
	// Multiplicity arrays are the same size as as the control points.
	const int uMultiplicity[] = { 1, 1, 1, 1, 1 };
	const int vMultiplicity[] = { 1, 1, 1, 1, 1 };
	
	result = surf->create( uDeg, vDeg, uForm, vForm, uNumKnots, vNumKnots,
								uKnotVector, vKnotVector, 
								uNumControlPts, vNumControlPts, (double *) controlPoints,
								uMultiplicity, vMultiplicity );
	if ( result != sSuccess )
		printf("createPeriodicSurface: failed....\n");

	if ( result == sSuccess )
		addSurfaceToDag( surf, "PeriodicSurface" );

	if ( surf != NULL )
		delete surf;
}

void createOpenSurface()
{ 
	statusCode result = sFailure;
	AlSurface *surf = new AlSurface;
	if ( surf == NULL )
		return;

	int uDeg = 3;
	int vDeg = 3;
	curveFormType uForm = kOpen;
	curveFormType vForm = kOpen;
	
	// For a open surface and the non redundant creation method, 
	//	 numKnots = numControlPts - degree + 1
	int uNumKnots = 3; 
	int vNumKnots = 3;
	const double uKnotVector[] = { 0, 1, 2 };
	const double vKnotVector[] = { 0, 1, 2 };
	int uNumControlPts = 5;
	int vNumControlPts = 5;
	
	// Array is of type controlPoints[uNumControlPts][vNumControlPts][4]
	const double controlPoints[][4] = {
		{ 0.0, 0.0, 0.0, 1.0 }, // first u row
		{ 1.0, 0.0, 0.0, 1.0 },
		{ 2.0, 0.0, 0.0, 1.0 },
		{ 3.0, 0.0, 0.0, 1.0 },
		{ 4.0, 0.0, 0.0, 1.0 },
		
		{ 0.0, 1.0, 1.0, 1.0 }, // second u row
		{ 1.0, 1.0, 1.0, 1.0 },
		{ 2.0, 1.0, 1.0, 1.0 },
		{ 3.0, 1.0, 1.0, 1.0 },
		{ 4.0, 1.0, 1.0, 1.0 },
		
		{ 0.0, 2.0, 2.0, 1.0 }, // third u row
 		{ 1.0, 2.0, 2.0, 1.0 },
		{ 2.0, 2.0, 2.0, 1.0 },
		{ 3.0, 2.0, 2.0, 1.0 },
		{ 4.0, 2.0, 2.0, 1.0 },
		
		{ 0.0, 3.0, 1.0, 1.0 }, // fourth u row
		{ 1.0, 3.0, 1.0, 1.0 },
		{ 2.0, 3.0, 1.0, 1.0 },
		{ 3.0, 3.0, 1.0, 1.0 },
		{ 4.0, 3.0, 1.0, 1.0 },
		
		{ 0.0, 4.0, 0.0, 1.0 }, // fifth u row
		{ 1.0, 4.0, 0.0, 1.0 },
		{ 2.0, 4.0, 0.0, 1.0 },
		{ 3.0, 4.0, 0.0, 1.0 },
		{ 4.0, 4.0, 0.0, 1.0 }
	
	};

	// Multiplicity arrays are the same size as as the control points.
	const int uMultiplicity[] = { 1, 1, 1, 1, 1 };
	const int vMultiplicity[] = { 1, 1, 1, 1, 1 };
	
	result = surf->create( uDeg, vDeg, uForm, vForm, uNumKnots, vNumKnots,
								uKnotVector, vKnotVector, 
								uNumControlPts, vNumControlPts, (double *) controlPoints,
								uMultiplicity, vMultiplicity );
	if ( result != sSuccess )
		printf("createOpenSurface: failed....\n");
								
	if ( result == sSuccess )
		addSurfaceToDag( surf, "OpenSurface" );

	if ( surf != NULL )
		delete surf;
}

void createClosedSurface()
{ 
	statusCode result = sFailure;
	AlSurface *surf = new AlSurface;
	if ( surf == NULL )
		return;

	int uDeg = 3;
	int vDeg = 3;
	curveFormType uForm = kClosed;
	curveFormType vForm = kClosed;
	
	// For a closed surface and the non redundant creation method, 
	//	 numKnots = numControlPts - degree + 1
	int uNumKnots = 3; 
	int vNumKnots = 3;
	const double uKnotVector[] = { 0, 1, 2 };
	const double vKnotVector[] = { 0, 1, 2 };
	int uNumControlPts = 5;
	int vNumControlPts = 5;
	
	// The first u row repeats at the end of the array
	// Array is of type controlPoints[uNumControlPts][vNumControlPts][4]
	const double controlPoints[][4] = {
		{ 0.0, 0.0, 0.0, 1.0 }, // first u row
		{ 1.0, 0.0, 0.0, 1.0 },
		{ 2.0, 0.0, 0.0, 1.0 },
		{ 3.0, 0.0, 0.0, 1.0 },
		{ 0.0, 0.0, 0.0, 1.0 },
		
		{ 0.0, 0.0, 0.0, 1.0 }, // second u row
		{ 1.0, 1.0, 1.0, 1.0 },
		{ 2.0, 1.0, 1.0, 1.0 },
		{ 3.0, 1.0, 1.0, 1.0 },
		{ 0.0, 0.0, 0.0, 1.0 },
		
		{ 0.0, 0.0, 0.0, 1.0 }, // third u row
 		{ 1.0, 2.0, 2.0, 1.0 },
		{ 2.0, 2.0, 2.0, 1.0 },
		{ 3.0, 2.0, 2.0, 1.0 },
		{ 0.0, 0.0, 0.0, 1.0 },
		
		{ 0.0, 0.0, 0.0, 1.0 }, // fourth u row
		{ 1.0, 3.0, 1.0, 1.0 },
		{ 2.0, 3.0, 1.0, 1.0 },
		{ 3.0, 3.0, 1.0, 1.0 },
		{ 0.0, 0.0, 0.0, 1.0 },
		
		{ 0.0, 0.0, 0.0, 1.0 }, // first u row repeated
		{ 1.0, 0.0, 0.0, 1.0 },
		{ 2.0, 0.0, 0.0, 1.0 },
		{ 3.0, 0.0, 0.0, 1.0 },
		{ 0.0, 0.0, 0.0, 1.0 },
	};
	
	// Multiplicity arrays are the same size as as the control points.
	const int uMultiplicity[] = { 1, 1, 1, 1, 1 };
	const int vMultiplicity[] = { 1, 1, 1, 1, 1 };
	
	result = surf->create( uDeg, vDeg, uForm, vForm, uNumKnots, vNumKnots,
								uKnotVector, vKnotVector, 
								uNumControlPts, vNumControlPts, (double *) controlPoints,
								uMultiplicity, vMultiplicity );
	if ( result != sSuccess )
		printf("createClosedSurface: failed....\n");
							
	if ( result == sSuccess )
		addSurfaceToDag( surf, "ClosedSurface" );

	if ( surf != NULL )
		delete surf;
}

//
//	Example code to show the creation of surfaces with the
//	::create() method that requires redundant
//	knot values.
//
 
void createPeriodicSurface2()
{ 
	statusCode result = sFailure;
	AlSurface *surf = new AlSurface;
	if ( surf == NULL )
		return;

	int uDeg = 3;
	int vDeg = 3;
	curveFormType uForm = kPeriodic;
	curveFormType vForm = kPeriodic;
	
	// For a periodic surface and the redundant creation method, 
	//	 numKnots = numControlPts + degree + 1
	const int uNumKnots = 13; 
	const int vNumKnots = 13;
	const double uKnotVector[uNumKnots] = { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12 };
	const double vKnotVector[vNumKnots] = { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12 };
	int uNumControlPts = 9;
	int vNumControlPts = 9;

	// Array is of type controlPoints[uNumControlPts][vNumControlPts][4]
	const double controlPoints[][4] = {

		{ 0.0, 0.0, 0.0, 1.0 }, // first u row
		{ 1.0, 0.0, 0.0, 1.0 },
		{ 2.0, 0.0, 0.0, 1.0 },
		{ 3.0, 0.0, 0.0, 1.0 },
		{ 4.0, 0.0, 0.0, 1.0 },
		{ 5.0, 0.0, 0.0, 1.0 },
		{ 0.0, 0.0, 0.0, 1.0 }, // the first 'v degree' number of points repeat
		{ 1.0, 0.0, 0.0, 1.0 },
		{ 2.0, 0.0, 0.0, 1.0 },
		
		{ 0.0, 1.0, 1.0, 1.0 }, // second u row
		{ 1.0, 1.0, 1.0, 1.0 },
		{ 2.0, 1.0, 1.0, 1.0 },
		{ 3.0, 1.0, 1.0, 1.0 },
		{ 4.0, 1.0, 1.0, 1.0 },
		{ 5.0, 1.0, 1.0, 1.0 },
		{ 0.0, 1.0, 1.0, 1.0 }, // the first 'v degree' number of points repeat
		{ 1.0, 1.0, 1.0, 1.0 },
		{ 2.0, 1.0, 1.0, 1.0 },
		
		{ 0.0, 2.0, 2.0, 1.0 }, // third u row
 		{ 1.0, 2.0, 2.0, 1.0 },
		{ 2.0, 2.0, 2.0, 1.0 },
		{ 3.0, 2.0, 2.0, 1.0 },
		{ 4.0, 2.0, 2.0, 1.0 },
		{ 5.0, 2.0, 2.0, 1.0 },
		{ 0.0, 2.0, 2.0, 1.0 }, // the first 'v degree' number of points repeat
 		{ 1.0, 2.0, 2.0, 1.0 },
		{ 2.0, 2.0, 2.0, 1.0 },
		
		{ 0.0, 3.0, 1.0, 1.0 }, // fourth u row
		{ 1.0, 3.0, 1.0, 1.0 },
		{ 2.0, 3.0, 1.0, 1.0 },
		{ 3.0, 3.0, 1.0, 1.0 },
		{ 4.0, 3.0, 1.0, 1.0 },
		{ 5.0, 3.0, 1.0, 1.0 },
		{ 0.0, 3.0, 1.0, 1.0 }, // the first 'v degree' number of points repeat
		{ 1.0, 3.0, 1.0, 1.0 },
		{ 2.0, 3.0, 1.0, 1.0 },
	
		{ 0.0, 0.0, 0.0, 1.0 }, // fifth u row
		{ 1.0, 0.0, 0.0, 1.0 },
		{ 2.0, 0.0, 0.0, 1.0 },
		{ 3.0, 0.0, 0.0, 1.0 },
		{ 4.0, 0.0, 0.0, 1.0 },
		{ 5.0, 0.0, 0.0, 1.0 },
		{ 0.0, 0.0, 0.0, 1.0 }, // the first 'v degree' number of points repeat
		{ 1.0, 0.0, 0.0, 1.0 },
		{ 2.0, 0.0, 0.0, 1.0 },
	
		{ 0.0, 0.0, 0.0, 1.0 }, // sixth u row
		{ 1.0, 0.0, 0.0, 1.0 },
		{ 2.0, 0.0, 0.0, 1.0 },
		{ 3.0, 0.0, 0.0, 1.0 },
		{ 4.0, 0.0, 0.0, 1.0 },
		{ 5.0, 0.0, 0.0, 1.0 },
		{ 0.0, 0.0, 0.0, 1.0 }, // the first 'v degree' number of points repeat
		{ 1.0, 0.0, 0.0, 1.0 },
		{ 2.0, 0.0, 0.0, 1.0 },

		{ 0.0, 0.0, 0.0, 1.0 }, // first u row repeats  
		{ 1.0, 0.0, 0.0, 1.0 },
		{ 2.0, 0.0, 0.0, 1.0 },
		{ 3.0, 0.0, 0.0, 1.0 },
		{ 4.0, 0.0, 0.0, 1.0 },
		{ 5.0, 0.0, 0.0, 1.0 },
		{ 0.0, 0.0, 0.0, 1.0 },  
		{ 1.0, 0.0, 0.0, 1.0 },
		{ 2.0, 0.0, 0.0, 1.0 },
		
		{ 0.0, 1.0, 1.0, 1.0 }, // second u row repeats 
		{ 1.0, 1.0, 1.0, 1.0 },
		{ 2.0, 1.0, 1.0, 1.0 },
		{ 3.0, 1.0, 1.0, 1.0 },
		{ 4.0, 1.0, 1.0, 1.0 },
		{ 5.0, 1.0, 1.0, 1.0 },
		{ 0.0, 1.0, 1.0, 1.0 },  
		{ 1.0, 1.0, 1.0, 1.0 },
		{ 2.0, 1.0, 1.0, 1.0 },
		
		{ 0.0, 2.0, 2.0, 1.0 },	// third u row repeats 
 		{ 1.0, 2.0, 2.0, 1.0 },
		{ 2.0, 2.0, 2.0, 1.0 },
		{ 3.0, 2.0, 2.0, 1.0 },
		{ 4.0, 2.0, 2.0, 1.0 },
		{ 5.0, 2.0, 2.0, 1.0 },
		{ 0.0, 2.0, 2.0, 1.0 },  
 		{ 1.0, 2.0, 2.0, 1.0 },
		{ 2.0, 2.0, 2.0, 1.0 },

	};
	
	result = surf->create( uDeg, vDeg, uForm, vForm, 
								uKnotVector, vKnotVector, 
								uNumControlPts, vNumControlPts, (double *) controlPoints );
	if ( result != sSuccess )
		printf("createPeriodicSurface2: failed....\n");

	if ( result == sSuccess )
		addSurfaceToDag( surf, "PeriodicSurface2" );

	if ( surf != NULL )
		delete surf;
}

void createOpenSurface2()
{ 
	statusCode result = sFailure;
	AlSurface *surf = new AlSurface;
	if ( surf == NULL )
		return;
		
	int uDeg = 3;
	int vDeg = 3;
	curveFormType uForm = kOpen;
	curveFormType vForm = kOpen;
	
	// For a open surface and the redundant creation method, 
	//	 numKnots = numControlPts + degree + 1
	const int uNumKnots = 9; 
	const int vNumKnots = 9;
	const double uKnotVector[uNumKnots] = { 0, 0, 0, 0, 1, 2, 2, 2, 2 };
	const double vKnotVector[vNumKnots] = { 0, 0, 0, 0, 1, 2, 2, 2, 2};
	int uNumControlPts = 5;
	int vNumControlPts = 5;

	// Array is of type controlPoints[uNumControlPts][vNumControlPts][4]
	const double controlPoints[][4] = {
		{ 0.0, 0.0, 0.0, 1.0 }, // first u row
		{ 1.0, 0.0, 0.0, 1.0 },
		{ 2.0, 0.0, 0.0, 1.0 },
		{ 3.0, 0.0, 0.0, 1.0 },
		{ 4.0, 0.0, 0.0, 1.0 },
		
		{ 0.0, 1.0, 1.0, 1.0 }, // second u row
		{ 1.0, 1.0, 1.0, 1.0 },
		{ 2.0, 1.0, 1.0, 1.0 },
		{ 3.0, 1.0, 1.0, 1.0 },
		{ 4.0, 1.0, 1.0, 1.0 },
		
		{ 0.0, 2.0, 2.0, 1.0 }, // third u row
 		{ 1.0, 2.0, 2.0, 1.0 },
		{ 2.0, 2.0, 2.0, 1.0 },
		{ 3.0, 2.0, 2.0, 1.0 },
		{ 4.0, 2.0, 2.0, 1.0 },
		
		{ 0.0, 3.0, 1.0, 1.0 }, // fourth u row
		{ 1.0, 3.0, 1.0, 1.0 },
		{ 2.0, 3.0, 1.0, 1.0 },
		{ 3.0, 3.0, 1.0, 1.0 },
		{ 4.0, 3.0, 1.0, 1.0 },
		
		{ 0.0, 4.0, 0.0, 1.0 }, // fifth u row
		{ 1.0, 4.0, 0.0, 1.0 },
		{ 2.0, 4.0, 0.0, 1.0 },
		{ 3.0, 4.0, 0.0, 1.0 },
		{ 4.0, 4.0, 0.0, 1.0 }
	
	};
	
	result = surf->create( uDeg, vDeg, uForm, vForm,
								uKnotVector, vKnotVector, 
								uNumControlPts, vNumControlPts, (double *) controlPoints );
	if ( result != sSuccess )
		printf("createOpenSurface2: failed....\n");

	if ( result == sSuccess )
		addSurfaceToDag( surf, "OpenSurface2" );

	if ( surf != NULL )
		delete surf;
}

void createClosedSurface2()
{ 
	statusCode result = sFailure;
	AlSurface *surf = new AlSurface;
	if ( surf == NULL )
		return;

	int uDeg = 3;
	int vDeg = 3;
	curveFormType uForm = kClosed;
	curveFormType vForm = kClosed;

	// For a closed surface and the redundant creation method, 
	// numKnots = numControlPts + degree + 1
	const int uNumKnots = 9; 
	const int vNumKnots = 9;
	const double uKnotVector[uNumKnots] = { 0, 0, 0, 0, 1, 2, 2, 2, 2 };
	const double vKnotVector[vNumKnots] = { 0, 0, 0, 0, 1, 2, 2, 2, 2 };
	int uNumControlPts = 5;
	int vNumControlPts = 5;

	// Array is of type controlPoints[uNumControlPts][vNumControlPts][4]
	const double controlPoints[][4] = {
		{ 0.0, 0.0, 0.0, 1.0 }, // first u row
		{ 1.0, 0.0, 0.0, 1.0 },
		{ 2.0, 0.0, 0.0, 1.0 },
		{ 3.0, 0.0, 0.0, 1.0 },
		{ 0.0, 0.0, 0.0, 1.0 },
		
		{ 0.0, 0.0, 0.0, 1.0 }, // second u row
		{ 1.0, 1.0, 1.0, 1.0 },
		{ 2.0, 1.0, 1.0, 1.0 },
		{ 3.0, 1.0, 1.0, 1.0 },
		{ 0.0, 0.0, 0.0, 1.0 },
		
		{ 0.0, 0.0, 0.0, 1.0 }, // third u row
 		{ 1.0, 2.0, 2.0, 1.0 },
		{ 2.0, 2.0, 2.0, 1.0 },
		{ 3.0, 2.0, 2.0, 1.0 },
		{ 0.0, 0.0, 0.0, 1.0 },
		
		{ 0.0, 0.0, 0.0, 1.0 }, // fourth u row
		{ 1.0, 3.0, 1.0, 1.0 },
		{ 2.0, 3.0, 1.0, 1.0 },
		{ 3.0, 3.0, 1.0, 1.0 },
		{ 0.0, 0.0, 0.0, 1.0 },
		
		{ 0.0, 0.0, 0.0, 1.0 }, // first u row repeated
		{ 1.0, 0.0, 0.0, 1.0 },
		{ 2.0, 0.0, 0.0, 1.0 },
		{ 3.0, 0.0, 0.0, 1.0 },
		{ 0.0, 0.0, 0.0, 1.0 },
	};
	
	result = surf->create( uDeg, vDeg, uForm, vForm,
								uKnotVector, vKnotVector, 
								uNumControlPts, vNumControlPts, (double *) controlPoints );
	if ( result != sSuccess )
		printf("createClosedSurface2: failed....\n");
								

	if ( result == sSuccess )
		addSurfaceToDag( surf, "ClosedSurface2" );
		
	if ( surf != NULL )
		delete surf;
}

void createClosedSurface3()
{ 
	statusCode result = sFailure;
	AlSurface *surf = new AlSurface;
	if ( surf == NULL )
		return;

	int uDeg = 3;
	int vDeg = 3;
	curveFormType uForm = kClosed;
	curveFormType vForm = kClosed;
	
	// For a closed surface and the non redundant creation method, 
	//	 numKnots = numControlPts - degree + 1
	int uNumKnots = 3; 
	int vNumKnots = 4;
	const double uKnotVector[] = { 0, 1, 2 };
	const double vKnotVector[] = { 0, 1, 2, 3 };
	int uNumControlPts = 5;
	int vNumControlPts = 6;
	
	// The first v row repeats at the end of the array
	// Array is of type controlPoints[uNumControlPts][vNumControlPts][4]
	const double controlPoints[][4] = {
		{ 0.0, 0.0, 0.0, 1.0 }, // first u row
		{ 1.0, 0.0, 0.0, 1.0 },
		{ 2.0, 0.0, 0.0, 1.0 },
		{ 3.0, 0.0, 0.0, 1.0 },
		{ 4.0, 0.0, 0.0, 1.0 },
		{ 0.0, 0.0, 0.0, 1.0 },
		
		{ 0.0, 0.0, 0.0, 1.0 }, // second u row
		{ 1.0, 1.0, 1.0, 1.0 },
		{ 2.0, 1.0, 1.0, 1.0 },
		{ 3.0, 1.0, 1.0, 1.0 },
		{ 4.0, 1.0, 1.0, 1.0 },
		{ 0.0, 0.0, 0.0, 1.0 },
		
		{ 0.0, 0.0, 0.0, 1.0 }, // third u row
 		{ 1.0, 2.0, 2.0, 1.0 },
		{ 2.0, 2.0, 2.0, 1.0 },
		{ 3.0, 2.0, 2.0, 1.0 },
		{ 4.0, 2.0, 2.0, 1.0 },
		{ 0.0, 0.0, 0.0, 1.0 },
		
		{ 0.0, 0.0, 0.0, 1.0 }, // fourth u row
		{ 1.0, 3.0, 1.0, 1.0 },
		{ 2.0, 3.0, 1.0, 1.0 },
		{ 3.0, 3.0, 1.0, 1.0 },
		{ 4.0, 3.0, 1.0, 1.0 },
		{ 0.0, 0.0, 0.0, 1.0 },
		
		{ 0.0, 0.0, 0.0, 1.0 }, // first u row repeated
		{ 1.0, 0.0, 0.0, 1.0 },
		{ 2.0, 0.0, 0.0, 1.0 },
		{ 3.0, 0.0, 0.0, 1.0 },
		{ 4.0, 0.0, 0.0, 1.0 },
		{ 0.0, 0.0, 0.0, 1.0 },
	};
	
	// Multiplicity arrays are the same size as as the control points.
	const int uMultiplicity[] = { 1, 1, 1, 1, 1 };
	const int vMultiplicity[] = { 1, 1, 1, 1, 1, 1 };
	
	result = surf->create( uDeg, vDeg, uForm, vForm, uNumKnots, vNumKnots,
								uKnotVector, vKnotVector, 
								uNumControlPts, vNumControlPts, (double *) controlPoints,
								uMultiplicity, vMultiplicity );
	if ( result != sSuccess )
		printf("createClosedSurface: failed....\n");
								

	if ( result == sSuccess )
		addSurfaceToDag( surf, "ClosedSurface3" );

	if ( surf != NULL )
		delete surf;
}

//
//	Helper functions to add the curves or surfaces to the dag
//

static statusCode addCurveToDag( AlCurve *curve, const char *nodeName )
{
	static int counter = 0;
	
	//
	//	Create the curve node and add the curve to it
	//
	AlCurveNode *curveNode = new AlCurveNode;
	
	if ( curveNode == NULL )
		return sFailure;
		
	statusCode result = sSuccess;
	if ( curveNode->create( curve ) != sSuccess )
		result = sFailure;
		
	if ( nodeName != NULL )
		curveNode->setName( nodeName );
	
	//
	//	Simple code for making sure the objects are not on top of
	//	each other.
	//	
	double x,y,z;
	if ( curveNode->translation( x, y, z ) == sSuccess )
	{
		const int moveby = counter * 2;
		curveNode->setTranslation( x + moveby, y , z );
	}
	
	counter++;	
	delete curveNode;
	return result;
}

static statusCode addSurfaceToDag( AlSurface *surf, const char *nodeName )
{
	static int counter = 1;

	//
	//	Create the surface node and add the surface to it
	//
	AlSurfaceNode *surfaceNode = new AlSurfaceNode;
	
	if ( surfaceNode == NULL )
		return sFailure;
		
	statusCode result = sSuccess;
	if ( surfaceNode->create( surf ) != sSuccess )
		result = sFailure;

	if ( nodeName != NULL )
		surfaceNode->setName( nodeName );

	//
	//	Simple code to ensure the surfaces are not on top of
	//	each other
	//
	double x,y,z;
	if ( surfaceNode->translation( x, y, z ) == sSuccess )
	{
		const int moveby = counter * 6;
		surfaceNode->setTranslation( x - moveby, y , z );
	}
	
	counter++;	
		
	delete surfaceNode;
	return result;
}


//
//	Main line code.
//

int main( int argc, char** argv )
{
	char		*outfile;

	if( argc != 2 )
	{
		fprintf( stderr, "usage: %s <outfilename>\n", argv[0] );
		fprintf( stderr,
			    "description:\n"
				"  This program creates Open, Closed and Periodic \n"
				"  curves and surfaces and then stores them in the output wire file.\n"
				"  It is intended to be a source example of how to use the\n"
				"  Alias Wire File Interface Library.\n"
				);
		exit( 0 );
	}
	outfile = argv[1];

	AlUniverse::initialize( kYUp );

	// Create periodic, open and closed curves with the
	// 2 different AlCurve::create() methods.
	createPeriodicCurve();
	createOpenCurve();
	createClosedCurve();
	createPeriodicCurve2();
	createOpenCurve2();
	createClosedCurve2();

	// Create periodic, open and closed curves with the
	// 2 different AlSurface::create() methods.
	createPeriodicSurface();
	createOpenSurface();
	createClosedSurface();
	createPeriodicSurface2();
	createOpenSurface2();
	createClosedSurface2();

	// Alternate test	
	createClosedSurface3();

	if( sSuccess == AlUniverse::store( outfile ) )
	{
		fprintf( stderr, "File '%s' stored successfully.\n", outfile );
	}
	else
	{
		fprintf( stderr, "Error storing file '%s'.\n", outfile );
		exit( -1 );
	}

    return 0;
}
