#include <stdio.h>
#include <string.h>

#include <AlUniverse.h>

#include <AlGroupNode.h>
#include <AlCurveCV.h>
#include <AlCurve.h>
#include <AlCurveNode.h>
#include <AlCurveOnSurface.h>
#include <AlFace.h>
#include <AlFaceNode.h>
#include <AlSurfaceCV.h>
#include <AlSurface.h>
#include <AlSurfaceNode.h>

#include <AlCluster.h>
#include <AlClusterNode.h>

#include <AlPerspectiveCamera.h>
#include <AlCameraNode.h>

#include <AlPointLight.h>
#include <AlAmbientLight.h>
#include <AlDirectionLight.h>
#include <AlSpotLight.h>
#include <AlLinearLight.h>
#include <AlAreaLight.h>
#include <AlLightNode.h>

#include <AlSet.h>

#include <AlParamAction.h>
#include <AlMotionAction.h>
#include <AlKeyframe.h>
#include <AlChannel.h>
#include <AlTM.h>
#include <AlBlendCurve.h>
#include <AlBlendPoint.h>
#include <AlProduct.h>

#include <AlPrint.h>

#ifndef FALSE
#define FALSE 0
#define TRUE 1
#endif

// ---------- animation creation routines ----------

AlKeyframe*		createAKeyframe(double x, double y)
//
//	Create a keyframe at location 'x' and value 'y'
//
{
	AlKeyframe *keyframe = new AlKeyframe;
	if (keyframe == NULL)
		return NULL;

	keyframe->setLocation(x);
	keyframe->setValue(y);
	return keyframe;
}

AlParamAction*		createAParamAction()
//
//	Create a parameter curve action with two keyframes
//
{
	AlParamAction *action = new AlParamAction;
	if (action == NULL)
		return NULL;

	if( action->addKeyframe( 10, 10, FALSE, kTangentSmooth, kTangentSmooth )
		!= sSuccess )
	{
		delete action;
		return NULL;
	}

	if( action->addKeyframe( 30, 50, FALSE, kTangentSmooth, kTangentSmooth )
		!= sSuccess )
	{
		delete action;
		return NULL;
	}

	return action;
}

AlCurveNode	*createACurve();

AlMotionAction	*createAMotionAction()
//
//	Create a motion path action
//
{
	AlCurveNode	*curveNode = createACurve();
	if (curveNode == NULL)
		return NULL;

	AlMotionAction	*action = new AlMotionAction;
	if (action == NULL)
	{
		delete curveNode;
		return NULL;
	}
	if (action->create(curveNode) != sSuccess)
	{
		delete action;
		delete curveNode;
		return NULL;
	}

	delete curveNode;
	return action;
}
void apply_timewarp(AlChannel *channel, double x, double y)
//
//	Apply a timewarp curve to the given channel that warps the timing
//	so that the channel is evaluated from 0 - 100 in the time 1 - 30.
//	(This is like putting a timing of 1 to 30 on 0 to 100% of a motion
//	path curve).
//
{
	if( channel == NULL )
		return;

	AlParamAction *action = channel->applyWarpO();
	if (action == NULL)
		return;

	AlKeyframe *keyframe = action->firstKeyframe();
	if (keyframe == NULL)
		return;

	keyframe->setLocation(x);

	if( sSuccess != keyframe->nextD() )
		return;

	keyframe->setLocation(y);

	delete keyframe;
	delete action;
}

void animateACurve(AlCurve *curve, AlAction *action)
//
//	Find the first CV and the last CV of the curve, and animate
//	their Z position using the given action.
//
{
	if( curve == NULL || action == NULL )
		return;

	// Find the first CV on the curve
	//
	AlCurveCV *firstCV = curve->firstCV();
	if (firstCV == NULL)
		return;

	// Find the last CV on the curve
	//
	AlCurveCV *lastCV = (AlCurveCV*)firstCV->copyWrapper();
	statusCode status = sSuccess;
	for( ; status == sSuccess; status = lastCV->nextD() );

	AlChannel *channel = new AlChannel;
	if (channel == NULL)
	{
		delete firstCV;
		delete lastCV;
		return;
	}

	if( channel->create(firstCV, kFLD_CURVECV_ZPOSITION, action, kX_COMPONENT)
		!= sSuccess )
	{
		delete channel;
		delete firstCV;
		delete lastCV;
		return;
	}

	delete( channel );
	channel = new AlChannel;
	if (channel == NULL)
	{
		delete firstCV;
		delete lastCV;
		return;
	}

	if( channel->create(lastCV, kFLD_CURVECV_ZPOSITION, action, kX_COMPONENT)
		!= sSuccess )
	{
		delete firstCV;
		delete lastCV;
		delete channel;
		return;
	}

	delete firstCV;
	delete lastCV;
	delete channel;
	return;
}

void animateASurface(AlSurface *surface, AlMotionAction *action)
//
// Find the first CV in U and V, and the last CV in U and V, and animate
// them along this motion path action, with timing curves between 1 and 30.
//
{
	if( surface == NULL || action == NULL )
		return;

	// Find the first CV on the surface
	//
	AlSurfaceCV *firstCV = surface->firstCV();
	if (firstCV == NULL)
		return;

	// Find the last CV on the surface (on the opposite corner of firstCV)
	//
	AlSurfaceCV *lastCV = (AlSurfaceCV*)firstCV->copyWrapper();
	statusCode status = sSuccess;
	for( ; status == sSuccess; status = lastCV->nextInUD() )
		;
	for( status = sSuccess; status == sSuccess; status = lastCV->nextInVD() )
		;

	// Animate the first CV along the motion path
	//
	AlChannel *channel = new AlChannel;
	if (channel == NULL)
	{
		delete firstCV;
		delete lastCV;
        return;
	}

	if( channel->create(firstCV, kFLD_SURFACECV_XPOSITION,
		action, kX_COMPONENT) != sSuccess )
	{
		delete firstCV;
		delete lastCV;
		delete channel;
		return;
	}

	apply_timewarp(channel, 1, 30);

	delete channel;
	channel = new AlChannel;
	if (channel == NULL)
	{
		delete firstCV;
		delete lastCV;
        return;
	}

	if( channel->create(firstCV, kFLD_SURFACECV_YPOSITION,
		action, kY_COMPONENT) != sSuccess )
	{
		delete firstCV;
		delete lastCV;
		delete channel;
		return;
	}

	apply_timewarp(channel, 1, 30);

	delete channel;
	channel = new AlChannel;
	if (channel == NULL)
	{
		delete firstCV;
		delete lastCV;
        return;
	}

	if( channel->create(firstCV, kFLD_SURFACECV_ZPOSITION,
		action, kZ_COMPONENT) != sSuccess )
	{
		delete firstCV;
		delete lastCV;
		delete channel;
		return;
	}

	apply_timewarp(channel, 1, 30);

	// Animate the last CV along the motion path
	//
	delete channel;
	channel = new AlChannel;
	if (channel == NULL)
	{
		delete firstCV;
		delete lastCV;
        return;
	}

	if( channel->create(lastCV, kFLD_SURFACECV_XPOSITION,
						action, kX_COMPONENT)
		!= sSuccess )
	{
		delete firstCV;
		delete lastCV;
		delete channel;
		return;
	}

	apply_timewarp(channel, 1, 30);

	channel = new AlChannel;
	if (channel == NULL)
	{
		delete firstCV;
		delete lastCV;
        return;
	}

	if( channel->create(lastCV, kFLD_SURFACECV_YPOSITION,
						action, kY_COMPONENT)
		!= sSuccess )
	{
		delete firstCV;
		delete lastCV;
		delete channel;
		return;
	}

	apply_timewarp(channel, 1, 30);

	delete channel;
	channel = new AlChannel;
	if (channel == NULL)
	{
		delete firstCV;
		delete lastCV;
        return;
	}

	if( channel->create(lastCV, kFLD_SURFACECV_ZPOSITION,
						action, kZ_COMPONENT)
		!= sSuccess )
	{
		delete firstCV;
		delete lastCV;
		delete channel;
		return;
	}

	apply_timewarp(channel, 1, 30);

	delete firstCV;
	delete lastCV;
	delete channel;
	return;
}

void animateADagNode(AlDagNode *dagNode, AlAction *action)
//
//	Animate the visibility field of the dag node using the given action
//
{
	if ( NULL == dagNode || NULL == action)
		return;

	AlChannel *channel = new AlChannel;
	if (channel == NULL)
		return;

	channel->create(dagNode, kFLD_DAGNODE_VISIBILITY, action, kX_COMPONENT);
	delete channel;
}

void animateACamera(AlPerspectiveCamera *camera, AlAction *action)
//
//	Animate the FOV of the camera using the given action
//
{
	if ( NULL == camera || NULL == action)
		return;

	AlChannel *channel = new AlChannel;
	if (channel == NULL)
		return;

	channel->create(camera, kFLD_CAMERA_FOV, action, kX_COMPONENT);
	delete channel;
}

void animateALight(AlLight *light, AlMotionAction *action)
//
//	Animate the intensity of the light using the Y-component of the
//	motion path action.
//
{
	if ( NULL == light || NULL == action)
		return;

	AlChannel *channel = new AlChannel;
	if (channel == NULL)
		return;

	channel->create(light, kFLD_LIGHT_INTENSITY, action, kY_COMPONENT);
	delete channel;
}


// ---------- object creation routines ----------

AlCurveNode	*createACurve()
//
//  Create a 2 span cubic b-spline curve.  The curve node is added to
//  the Universe when it is created.
//
{
	static double	curveKnots[] = { 0.0, 1.0, 2.0 };
	static double	curvePoints[][4] =
	{
		{ 0.0,	0.0,  0.0, 1.0 },
		{ 10.0,	0.0,  0.0, 1.0 },
		{ 10.0,	10.0, 0.0, 1.0 },
		{ 0.0,	10.0, 0.0, 1.0 },
		{ 0.0,	20.0, 0.0, 1.0 }
	};
	static int		multiplicity[] = {1, 1, 1, 1, 1};

	int				degree = 3;
	curveFormType	form   = kOpen;
	int				numKnots = 3;
	int				numControlPoints = 5;

	AlCurve *curve = new AlCurve;
	if( curve == NULL )
		return NULL;

	if( sSuccess != curve->create( degree, form, numKnots, curveKnots,
		numControlPoints, curvePoints, multiplicity ) )
	{
		delete curve;
		return NULL;
	}

	AlCurveNode *curveNode = new AlCurveNode;
	if( NULL == curveNode )
	{
		delete curve;
		return NULL;
	}

	if( sSuccess != curveNode->create( curve ))
	{
		delete curve;
		delete curveNode;
		return NULL;
	}
	delete curve;
	curveNode->setName( "curveNode" );
	return curveNode;
}


AlFaceNode*		createAFace()
//
//  Create a face node which contains two face curves, one inside the other.
//  The face node is added to the Universe when it is created.
//
{
	static double	outerFaceKnots[] = { 0.0, 1.0, 2.0, 3.0, 4.0 };
	static double	outerFacePoints[][4] =
	{
		{ 0.0, 0.0, 0.0, 1.0 },
		{ 1.0, 0.0, 0.0, 1.0 },
		{ 1.0, 1.0, 0.0, 1.0 },
		{ 0.0, 1.0, 0.0, 1.0 }
	};

	static double	innerFaceKnots[] = { 0.0, 1.0, 2.0, 3.0, 4.0 };
	static double	innerFacePoints[][4] =
	{
		{ 0.25, 0.25, 0.0, 1.0 },
		{ 0.75, 0.25, 0.0, 1.0 },
		{ 0.75, 0.75, 0.0, 1.0 },
		{ 0.25, 0.75, 0.0, 1.0 }
	};
	static int		multiplicity[] = {1, 1, 1, 1};
	int				degree = 3;
	curveFormType	form   = kPeriodic;
	int				numKnots = 5;
	int				numControlPoints = 4;

	AlFace *face = new AlFace;
	if( face == NULL ) 
		return NULL;

	statusCode status =  face->create( degree, form, numKnots, outerFaceKnots,
		numControlPoints, outerFacePoints, multiplicity );
	if( status != sSuccess )
	{
		delete face;
		return NULL;
	}

	AlFaceNode *faceNode = new AlFaceNode;
	if( NULL == faceNode )
	{
        delete face;
        return NULL;
	}

	if( sSuccess != faceNode->create( face ))
	{
		delete face;
		delete faceNode;
		return NULL;
	}

	delete face;
	face = new AlFace;
	if( face == NULL )
	{
		delete faceNode;
		return NULL;
	}

	if( sSuccess != face->create( degree, form, numKnots, innerFaceKnots,
		numControlPoints, innerFacePoints, multiplicity ) )
	{
		delete face;
		delete faceNode;
		return NULL;
	}

	faceNode->addFace( face );
	faceNode->setName( "faceNode" );

	delete face;
	return faceNode;
}

AlCurveOnSurface*   createACurveOnSurface( AlSurface *surface )
{
    int           degree = 3;
    curveFormType form = kOpen;
    int           numKnots = 2;
    int           numCPs = 4;
    static double controlPoints[][4] =
	{
        { 0.5, 0.0, 0.0, 1.0},
        { 0.5, 0.0, 0.0, 1.0},
        { 0.5, 1.0, 0.0, 1.0},
        { 0.5, 1.0, 0.0, 1.0}
	};
    static double knots[] = { 0.0, 1.0 };

	if ( NULL == surface )
		return NULL;

    AlCurveOnSurface *new_cos = new AlCurveOnSurface;
    if( new_cos == NULL )
		return NULL;

	if( sSuccess != new_cos->create( degree, form, numKnots, knots,
		numCPs, controlPoints ))
	{
		delete new_cos;
		return NULL;
	}

	surface->addCurveOnSurface( new_cos );

	return new_cos;
}


AlSurfaceNode*		createASurface()
//
//  Create a single span bi-cubic b-spline surface.  The surface node is
//  added to the Universe when it is created.
//
{
	static double	surfaceUKnots[] = { 0.0, 1.0 };
	static double	surfaceVKnots[] = { -1.0, 1.0 };
	static double	surfacePoints[][4] = {
		{ 0.0, 0.0, 0.0, 1.0 },	// first v row
		{ 1.0, 0.0, 0.0, 1.0 },
		{ 2.0, 0.0, 0.0, 1.0 },
		{ 3.0, 0.0, 0.0, 1.0 },
		{ 0.0, 1.0, 0.0, 1.0 },	// second v row
		{ 1.0, 1.0, 0.0, 1.0 },
		{ 2.0, 1.0, 0.0, 1.0 },
		{ 3.0, 1.0, 0.0, 1.0 },
		{ 0.0, 2.0, 0.0, 1.0 },	// third v row
		{ 1.0, 2.0, 0.0, 1.0 },
		{ 2.0, 2.0, 0.0, 1.0 },
		{ 3.0, 2.0, 0.0, 1.0 },
		{ 0.0, 3.0, 0.0, 1.0 },	// fourth v row
		{ 1.0, 3.0, 0.0, 1.0 },
		{ 2.0, 3.0, 0.0, 1.0 },
		{ 3.0, 3.0, 0.0, 1.0 }
		};
	static int 		uMultiplicity[] = {
        1, 2, 1, 1,                     // to give multiplicity 2 in the 2nd
        };								// row of CVs in the U direction

	static int 		vMultiplicity[] = {
        1, 3, 1, 1,                     // to give multiplicity 3 in the 2nd
        };								// column of CVs in the V direction

	int				uDegree = 3;
	int				vDegree = 3;
	curveFormType	uForm   = kOpen;
	curveFormType	vForm   = kOpen;
	int				numUKnots = 2;
	int				numVKnots = 2;
	int				uLengthControlP = 4;
	int				vLengthControlP = 4;

	AlSurface *surface = new AlSurface;
	if( NULL == surface )
		return NULL;

	if( sSuccess != surface->create( uDegree, vDegree, uForm, vForm,
		numUKnots, numVKnots, surfaceUKnots, surfaceVKnots,
		uLengthControlP, vLengthControlP, (double*)surfacePoints, 
		uMultiplicity, vMultiplicity ) )
	{
		delete surface;
		return NULL;
	}

	AlSurfaceNode *surfaceNode = new AlSurfaceNode;
	if( surfaceNode == NULL )
	{
		delete surface;
		return NULL;
	}

	if( sSuccess != surfaceNode->create( surface )) 
	{
		delete surface;
		delete surfaceNode;
		return NULL;
	}

	createACurveOnSurface( surface );

	surfaceNode->setName( "surfaceNode" );

	delete surface;
	return surfaceNode;
}


AlPerspectiveCamera* createACamera()
//
//	Description:
//		Creates a new camera.  Creating a camera causes camera eye,
//		view and up dag nodes to be created and added to the DAG.
//
{
	AlPerspectiveCamera *camera = new AlPerspectiveCamera;
	if( NULL == camera )
		return NULL;

	if( sSuccess != camera->create() )
	{
		delete camera;
		return NULL;
	}

	// change the position of the eye point so that the camera
	// looks down (if y is up) at the origin at a 45 degree angle
	//
	camera->setWorldEye( 0.0, 12.0, 12.0 );

	return camera;
}


AlCluster* createACluster()
//
//	Description:
//		Creates a new cluster.  Creating a cluster causes a
//		cluster dag nodes to be created and added to the DAG.
//
{
	AlCluster *cluster = new AlCluster;
	if( NULL == cluster )
		return NULL;

	if( sSuccess != cluster->create() )
	{
		delete cluster;
		return NULL;
	}
	return cluster;
}


AlSet* createASet()
//
//	Description:
//		Creates a new, empty, MULTIPLE set.  
//
{
	AlSet *set = new AlSet;
	if( NULL == set )
		return NULL;

	if( sSuccess != set->create( FALSE ) )
	{
		delete set;
		return NULL;
	}
	return set;
}

statusCode createLights(AlMotionAction *action)
//
//	Description:
//		Create an AlLight of each type.  An AlLightNode will be created
//		for each light and automatically inserted into the dag.
//		Some of the lights will be animated using the given action.
//
{
	statusCode status;

	// Create an ambient light
	//
	AlAmbientLight *ambLight = new AlAmbientLight;
	if( NULL == ambLight )
		return sInsufficientMemory;

	status = ambLight->create();
	if( sSuccess != status )
	{
		delete ambLight;
		return status;
	}
	ambLight->setColor( 255.0, 0.0, 0.0);
	ambLight->setShadeFactor( 0.2);
	animateALight(ambLight, action);
	delete ambLight;

	// Create a point light
	//
	AlPointLight *ptLight = new AlPointLight;
	if( NULL == ptLight )
		return sInsufficientMemory;

	status = ptLight->create();
	if( sSuccess != status )
	{
		delete ptLight;
		return status;
	}
	ptLight->setColor( 0.0, 255.0, 0.0);
	delete ptLight;

	// Create a directional light
	//
	AlDirectionLight *dirLight = new AlDirectionLight;
	if( NULL == dirLight )
		return sInsufficientMemory;
	status = dirLight->create();
	if( sSuccess != status )
	{
		delete dirLight;
		return status;
	}
	dirLight->setColor( 0.0, 0.0, 255.0);
	delete dirLight;

	// Create a spot light
	//
	AlSpotLight *spotLight = new AlSpotLight;
	if( NULL == spotLight )
		return sInsufficientMemory;
	status = spotLight->create();
	if( sSuccess != status )
	{
		delete spotLight;
		return status;
	}

	spotLight->setDropOff( 0.99);
	spotLight->setMaxBias( 99);
	spotLight->setMinBias( 25);
	spotLight->setSpreadAngle( 90.0);
	spotLight->setOffset( 126);
	spotLight->setMultFactor( 2);
	spotLight->setShadowSize( 512);
	spotLight->setPenumbra( 90.0);
	animateALight(spotLight, action);
	delete spotLight;

	// Create a linear light
	//
	AlLinearLight *linLight = new AlLinearLight;
	if( NULL == linLight )
		return sInsufficientMemory;
	status = linLight->create();
	if( sSuccess != status )
	{
		delete linLight;
		return status;
	}
	linLight->setAxis( 12.0, 0.0, 1.0);
	delete linLight;

	// Create an area light
	//
	AlAreaLight *areaLight = new AlAreaLight;
	if( NULL == areaLight )
		return sInsufficientMemory;
	status = areaLight->create();
	if( sSuccess != status )
	{
		delete areaLight;
		return status;
	}
	areaLight->setShortAxis( 0.0, 12.0, 1.0);
	areaLight->setLongAxis( 0.0, -12.0, 1.0);
	delete areaLight;

	return sSuccess;
}


// ------------------------------------------------

void findAndAddCamerasToSet( AlDagNode *dagNode, AlSet *newSet )
//
//	Description:
//		Recursive function that adds cameras to the given set.
//
{
	if( dagNode == NULL )
		return;

	// create a copy of the dagNode passed in and then put the new one
	// into the local pointer .. thus protecting the caller's orig
	// dagNode.
	dagNode = (AlDagNode *)dagNode->copyWrapper();
	AlPerspectiveCamera	*cameraPtr = NULL;
	AlDagNode	*downNode, *tmpDag = NULL;

	for(; AlIsValid( dagNode ); dagNode = tmpDag )
	{
		// If the dag node is a camera, then add it to the set.
		//
		if( dagNode->asCameraNodePtr() != NULL )
		{
			cameraPtr = dagNode->asCameraNodePtr()->camera()->
							asPerspectiveCameraPtr();
			if( cameraPtr != NULL )
			{
				cameraPtr->addToSet( newSet );
			}
		}
		else
		{
			// If the dag node is a group node, then traverse its children
			if( dagNode->asGroupNodePtr() != NULL )
			{
				downNode = dagNode->asGroupNodePtr()->childNode();
				if( downNode != NULL )
				{
					findAndAddCamerasToSet( downNode, newSet );
					delete downNode;
				}
			}
		}
		tmpDag = dagNode->nextNode();
		delete dagNode;
	}
}

AlBlendPoint *lastBlendPoint( AlBlendCurve *blend )
{
	int numPoints = blend->numberOfPoints();
	return blend->getPoint( numPoints -1 );
}

AlBlendCurve* createBlendCurve()
{
	// Create a blend curve that is not attached to any geometry
	double points[3][3] = {
								{ 0, 1, 1 },
								{ 0, 2, 2 },
								{ 0, 3, 3 }
							};
		
	AlBlendCurve *blendCurve = new AlBlendCurve;
	if ( blendCurve == NULL )
		return NULL;
	
	if ( blendCurve->create( 1, AlBlendCurve::kUniform, 3, points ) != sSuccess )
		return NULL;
		
	return blendCurve;
}

void findProductAndVersion()
{
	const char *product = AlProduct::productName();
	const char *version = AlProduct::versionNumber();
	if ( product != NULL )
		fprintf(stdout, "%sName    = '%s'\n",indentSpace(),product);	
	if ( version != NULL )
		fprintf(stdout, "%sVersion = '%s'\n",indentSpace(),version);	
}

// ------------------------------------------------


int main( int argc, char** argv )
{
	char	*outfile;

	if( argc != 2 )
	{
		fprintf(stderr, "usage: %s <outputfilename>\n", argv[0] );
		fprintf(stderr,
			"description:\n"
			"  This program creates a curve, a face (with two curves),\n"
			"  and a surface (with a curve on surface) and\n"
			"  groups these objects under a group node.  It creates a\n"
			"  camera, a cluster, a set and one of each type of light.\n"
			"  It animates some of the curve and surface CVs, as well as\n"
			"  a dag node, a camera, and some of the lights.\n"
			"  It prints the objects created and then stores the objects\n"
			"  as an Alias Wire file.\n"
			"  It is intended to be a source example of how to use OpenModel.\n"
			);
		exit( 0 );
	}
	outfile  = argv[1];

	// This must be called before any other OpenModel method can be called
	//
	AlUniverse::initialize( kYUp );

	// Print the product info
	fprintf(stdout, "-------------------------------------- PRODUCT INFO\n");
	findProductAndVersion();

	AlParamAction *paramAction = createAParamAction();
	if ( NULL == paramAction )
	{
		fprintf(stderr, "error: could not create a parameter curve action\n");
		exit(-1);
	}

	AlMotionAction *motionAction = createAMotionAction();
	if ( NULL == motionAction )
	{
		fprintf(stderr, "error: could not create a motion path action\n");
		exit(-1);
	}

	AlCurveNode *curveNode = createACurve();
	if( NULL == curveNode )
	{
		fprintf(stderr, "error: could not create a curve node\n");
		exit( -1 );
	}
	AlCurve *curve = curveNode->curve();

	animateACurve(curve, paramAction);
	delete curve;

	AlFaceNode *faceNode = createAFace();
	if( NULL == faceNode )
	{
		fprintf(stderr, "error: could not create a face node\n");
		exit( -1 );
	}

	curve = faceNode->firstFace();
	animateACurve(curve, paramAction);
	delete curve;

	AlSurfaceNode *surfaceNode = createASurface();
	if( NULL == surfaceNode )
	{
		fprintf(stderr, "error: could not create a surface node\n");
		exit( -1 );
	}

	AlSurface *surface = surfaceNode->surface();
    if( NULL == surface )
    {
        fprintf(stderr, "error: could not create a surface\n");
        exit( -1 );
    }

	animateASurface(surface, motionAction);
	delete surface;
	
	// the group node is added to the universe upon creation
	AlGroupNode *groupNode = new AlGroupNode;
	if( NULL == groupNode )
	{
		fprintf(stderr, "error: could not instantiate a group node\n");
		exit( -1 );
	}

	if( sSuccess != groupNode->create() )
	{
		fprintf(stderr, "error: could not create a group node\n");
		exit( -1 );
	}
	animateADagNode(groupNode, paramAction);

	groupNode->addChildNode( curveNode );
	groupNode->addChildNode( faceNode );
	delete groupNode;
	delete faceNode;

	// make the surface a child of the group node by making the
	// surface a sibling of the curve
	//
	curveNode->addSiblingNode( surfaceNode );
	delete surfaceNode;
	delete curveNode;
	
	AlPerspectiveCamera *camera = createACamera();
	if( NULL == camera )
	{
		fprintf(stderr, "error: could not create a camera\n");
		exit( -1 );
	}
	animateACamera(camera, paramAction);
	delete camera;
	delete paramAction;

	// create a cluster and add all dag nodes in the universe to the set
	//
	AlCluster *cluster = createACluster();
	if( NULL == cluster )
	{
		fprintf(stderr, "error: could not create a cluster\n");
		exit( -1 );
	}

	AlDagNode *dagNode = AlUniverse::firstDagNode();
	AlDagNode *tmpNode = NULL;
	for( ; dagNode; dagNode = tmpNode )
	{
		dagNode->addToCluster( cluster, 0.5 );
		tmpNode = dagNode->nextNode();
		delete dagNode;
	}
	delete cluster;

	// create a set and add all cameras in the universe to the set
	//
	AlSet *set = createASet();
	if( NULL == set )
	{
		fprintf(stderr, "error: could not create a set\n");
		exit( -1 );
	}

	dagNode = AlUniverse::firstDagNode();
	findAndAddCamerasToSet( dagNode, set );
	delete dagNode;
	delete set;

	// create one of each light
	//
	createLights(motionAction);
	delete motionAction;

	// create blend curves
	AlBlendCurve *blendCurve = createBlendCurve( );
	if ( blendCurve == NULL )
	{
		fprintf(stderr, "error: could not create blend curve\n");
		exit( -1 );
	}

	AlBlendPoint *blendPoint = lastBlendPoint( blendCurve );
	if ( blendPoint == NULL )
	{
		fprintf(stderr, "error: could not find last blend point\n");
		exit( -1 );
	}

	AlSurfaceNode *surfaceNodeForBlend = createASurface();
	if( NULL == surfaceNodeForBlend )
	{
		fprintf(stderr, "error: could not create a surface node for blend operation\n");
		exit( -1 );
	}
	AlSurface *surfaceForBlend = surfaceNodeForBlend->surface();
	
	// Attach the last blend point to the surface
	if ( blendPoint->attachConstraint( surfaceForBlend, 1, 1) != sSuccess )
	{
		fprintf(stderr, "error: could not attach the blend curve to a surface\n");
		exit( -1 );
	}
	delete blendCurve;
	delete blendPoint;
	delete surfaceNodeForBlend;
	delete surfaceForBlend;
	
	// print all the objects in the DAG, and all actions in the universe
	fprintf(stdout, "-------------------------------------- OBJECTS IN DAG\n");
	printAllAlObjects( AlUniverse::firstDagNode(), AlTM::identity() );
	fprintf(stdout, "--------------------------------- ACTIONS IN UNIVERSE\n");
	printAllActions();
		
	// write the DAG out as an Alias Wire File
	if( sSuccess == AlUniverse::store( outfile ) )
		fprintf(stderr, "File '%s' stored successfully.\n", outfile );
	else
		fprintf(stderr, "Error storing file '%s'.\n", outfile );

	// delete all the objects which were created
	AlUniverse::deleteAll();

    return 0;
}

