
#include <stdio.h>
#include <stdlib.h>
#include <AlUniverse.h>
#include <AlCluster.h>
#include <AlClusterNode.h>
#include <AlClusterMember.h>
#include <AlDagNode.h>
#include <AlGroupNode.h>
#include <AlSurface.h>
#include <AlSurfaceNode.h>
#include <AlSurfaceCV.h>
#include <AlFace.h>
#include <AlFaceNode.h>
#include <AlCurve.h>
#include <AlCurveNode.h>
#include <AlCurveCV.h>
#include <AlPolysetVertex.h>

void cluster_test( void );
void print_all_clusters( AlCluster* );
void print_all_cluster_parents( AlDagNode* );
void print_branchs_clusters( AlDagNode* );

void print_dag_nodes_clusters( AlDagNode* );
void print_curve_cvs_clusters( AlCurveCV* );
void print_surface_cvs_clusters( AlSurfaceCV* );

#ifndef FALSE
#define FALSE 0
#define TRUE 1
#endif

//
// This program reads in a file, then prints the cluster information in
// three different formats and in three stages.
//
//	Format 1: Walk all clusters printing cluster member information
//	Format 2: Walk the dag node, printing names of cluster node parents
//	Format 3: Walk the world, printing out which dag nodes and/or CVs
//			  are in clusters and the percentage effects.
//
//	Stage 1: Original file as read in
//	Stage 2: Original file with all odd cluster members removed
//			 and percentage effects set to be equal to the cluster member index.
//	Stage 3: Original file with all cluster members removed
//
//
int
main( int argc, char** argv )
{
	int which;

	if( argc < 2 )
	{
		fprintf( stderr, "usage: %s <filename> {<filename>}* \n",argv[0] );
		fprintf( stderr,
				"Description:\n"
				"  This program reads in wire file(s) and dumps info on\n"
				"  the clusters present\n"
				"  This is an example of how to use the AlCluster class\n"
				"  of the Alias OpenModel API\n" 
				);
		exit( 0 );
	}

	AlUniverse::initialize( kYUp );

	for( which = 1; which < argc; which++ )
	{
		fprintf( stderr, "Processing file %s\n", argv[which] );
		switch( AlUniverse::retrieve( argv[ which ] ) )
		{
		case sSuccess:
			{
			fprintf( stderr, "Retrieve of '%s' succeeded\n", argv[ which ] );
			cluster_test();
			AlUniverse::deleteAll();
			break;
			}

		case sFailure:
			{
			fprintf( stderr, "Retrieve failed\n" );
			break;	// lets go try the next file
			}

		default:
			{
			fprintf( stderr, "Retrieve failed for unknown reason\n" );
			exit( -1 );  // quitting here is a bit arbitrary.  We could
						 //	try the next file .. naahhh
			}
		}

	}

    return 0;
}

void
cluster_test( void )
{
	AlCluster* cluster = AlUniverse::firstCluster();
	if( cluster == NULL )
		{
		printf( "No Clusters found in this Universe\n" );
		return;
		}

	printf( "Clusters as read in\n" );
	printf( "===================\n" );
	printf( "{\n" );
	print_all_clusters( cluster );

	AlDagNode* world = AlUniverse::firstDagNode();
	if( !AlIsValid( world ) ) return;

	printf( "1 Cluster Parents\n" );
	printf( "===============\n" );
	print_all_cluster_parents( world );

	printf( "Branch's Clusters\n" );
	printf( "=================\n" );
	print_branchs_clusters( world );

	printf( "}\n" );

	// Delete every other cluster member
	//
	do {
		double	effect;

		effect = 0.1;
		AlClusterMember* clusterMember = cluster->firstMember();
		if( AlIsValid( clusterMember ) )
			{
			do {
				AlClusterMember* lastMember = clusterMember->copyWrapper()
						->asClusterMemberPtr();
				if( sSuccess == clusterMember->nextClusterMemberD() )
					{
					AlObject *object = clusterMember->object();
					if( ! object->asDagNodePtr() )
						{
						AlClusterable *cable = object->asClusterablePtr();
						cable->setPercentEffect( cluster, effect);
						}
					delete object;
					effect += 0.1;
					}
				// the removal of a member closes the linked list without
				// the need to do "cluster = cluster -> .. "
				lastMember->removeFromCluster( cluster );
				delete lastMember;
				} while( sSuccess == clusterMember->nextClusterMemberD() );

			delete clusterMember;
			}

		// Subtest; try to change the cluster type
		//
		AlClusterRestrict res;
		cluster->clusterRestrict( res );
		if( res == kMultiCluster )
		{
			switch( cluster->setClusterRestrict( kExclusiveCluster ) )
			{
			case sSuccess:
				printf( "Set cluster %p to exclusive\n", cluster );
				break;

			case sFailure:
				printf( "Failed to set cluster %p to exclusive\n",
							cluster );
				break;

			default:
				printf( "UNKNOWN: Set cluster %p to exclusive\n",
							cluster );
				break;
			}
		}
		else
		{
			switch( cluster->setClusterRestrict( kMultiCluster ) ) {
			case sSuccess:
				printf( "Set cluster %p to multi\n",
							cluster );
				break;

			case sFailure:
				printf( "Failed to set cluster %p to multi\n",
						cluster );
				break;

			default:
				printf( "UNKNOWN: Set cluster %p to multi\n",
						cluster );
				break;
			}
		}
		} while( sSuccess == cluster->nextClusterD() );

	printf( "Clusters with odd members removed\n" );
	printf( "=================================\n" );
	printf( "{\n" );
	print_all_clusters( cluster );

	printf( "2 Cluster Parents\n" );
	printf( "===============\n" );
	print_all_cluster_parents( world );

	printf( "Branch's Clusters\n" );
	printf( "=================\n" );
	print_branchs_clusters( world );

	printf( "}\n" );

	// Delete the rest of the cluster members
	//
	cluster = AlUniverse::firstCluster();
	if( cluster != NULL && AlIsValid( cluster ) )
		{
		AlClusterMember *clusterMember;
		do
			{
			clusterMember = cluster->firstMember();
			while ( clusterMember != NULL )
				{
				// removing a member advances the pointer
				clusterMember->removeFromCluster( cluster );
				clusterMember = cluster->firstMember();
				}

			} while( sSuccess == cluster->nextClusterD() );
		}

	printf( "Clusters with no members\n" );
	printf( "========================\n" );
	printf( "{\n" );
	print_all_clusters( cluster );
	delete cluster;

	printf( "3 Cluster Parents\n" );
	printf( "===============\n" );
	print_all_cluster_parents( world );

	printf( "Branch's Clusters\n" );
	printf( "=================\n" );
	print_branchs_clusters( world );
	delete world;

	printf( "}\n" );

	return;
}

void
print_cluster_members( AlCluster* cluster )
//
//	Description:
//		Walk all the members of the cluster.
//
{
if ( !AlIsValid( cluster ) )
	return;

AlClusterMember *member = cluster->firstMember();
if( AlIsValid( member ) )
	{
	do
		{
		AlObject *object = member->object();

		AlClusterable *cable = object->asClusterablePtr();
		printf( "Object CV: effect = %g\n",
					cable->percentEffect( cluster ) );
		delete object;
		} while( sSuccess == member->nextClusterMemberD() );

	delete member;
	}
}

void
print_all_clusters( AlCluster* cluster )
//
//	Description:
//		Walk list of clusters given and print them out
//
{
AlClusterRestrict restCode;

if( ! AlIsValid( cluster ) )
	{
	printf( "invalid cluster - quitting\n" );
	return;
	}

printf( "List of Clusters:\n" );
printf( "=================\n" );

do
	{
	printf( "{\n" );
	printf( "Type: " );

	cluster->clusterRestrict(restCode) ;
	switch( restCode ) {
	case kMultiCluster :
		printf( "MULTI      " );
		break;
	case kExclusiveCluster :
		printf( "EXCLUSIVE  " );
		break;
	default:
		printf( "UNKNOWN    " );
		}
	printf( "IsEmpty: %s  ", cluster->isEmpty() ? "YES" : "NO" );
	printf( "# Members: %d\n", cluster->numberOfMembers() );
	printf( "}\n" );
	} while( sSuccess == cluster->nextClusterD() );

return;
}

void
print_all_cluster_parents( AlDagNode* startingHere )
//
//	Description:
//		Recursive function to print out all the dag node names that have
//		a cluster under them.
//
{
	if( !AlIsValid( startingHere ) )
		{
		printf( "invalid cluster - quitting\n" );
		return;
		}

	// If the given dag node is a cluster node, then print out its name
	//
	if( startingHere->type() == kClusterNodeType )
		{
		const char * s = startingHere->name();
		fprintf(stdout,"Found Cluster under node '%s'\n",s);
		}
	else
		{
		// if the starting node is a group dag node, check its
		// list of children to see if a child is a cluster node
		//
		AlGroupNode *groupNode = startingHere->asGroupNodePtr();
		if( groupNode != NULL )
			{
			// Down to look for children
			AlDagNode *childNode = groupNode->childNode();
			if( childNode != NULL )
				{
				print_all_cluster_parents( childNode );
				delete childNode;
				}
			}
		}
	
	// Continue to traverse the dag.  For the next sibling of 
	// this node, call print_all_cluster_parents().
	//
	AlDagNode *siblingNode = startingHere->nextNode();
	if( siblingNode != NULL )
		{
		print_all_cluster_parents( siblingNode );
		delete siblingNode;
		}
}

void
print_branchs_clusters( AlDagNode* startingHere )
//
//	Description:
//		Recursive function to print out all the clusters that a given
//		dag branch is in
//
{
	if( !AlIsValid( startingHere ) )
		{
		printf( "invalid cluster - quitting\n" );
		return;
		}

	// If the given dag node is a cluster node, then print out its name
	//
	printf( "{\n" );
	print_dag_nodes_clusters( startingHere );

	switch( startingHere->type() ) {
	case kClusterNodeType:
		{
		printf( "This branch ends in a cluster\n" );
		break;
		}

	case kCurveNodeType:
		{
		AlCurveNode *curveNode = startingHere->asCurveNodePtr();
		if( curveNode == NULL ) break;

		AlCurve *curve = curveNode->curve();
		if( curve == NULL )
			{
			delete curve;
			break;
			}

		printf( "CURVE CV LIST:\n" );
		printf( "{\n" );

		AlCurveCV *curveCV = curve->firstCV();
		if( AlIsValid( curveCV ) ) 
			{
			do
				{
				print_curve_cvs_clusters( curveCV );
				} while( sSuccess == curveCV->nextD() );
			delete curveCV;
			}

		printf( "}\n" );
		delete curve;
		break;
		}

	case kFaceNodeType:
		{
		AlFaceNode*	faceNode = startingHere->asFaceNodePtr();
		if( faceNode == NULL ) break;

		AlFace	*face = faceNode->firstFace();
		if( ! AlIsValid( face ) )
			{
			break;
			}

		printf( "FACE CV LIST:\n" );
		printf( "{\n" );

		do
		{
			AlCurveCV *faceCurveCV = face->firstCV();
			if( ! AlIsValid( faceCurveCV ) )
				printf( "No valid CVs for this face\n" );
			else
			{
				do
				{
					print_curve_cvs_clusters( faceCurveCV );
				} while( sSuccess == faceCurveCV->nextD() );
			}
			delete faceCurveCV;
		} while( sSuccess == face->asFacePtr()->nextFaceD() );
		delete face;

		printf( "}\n" );
		break;
		}

	case kSurfaceNodeType:
		{
		AlSurfaceNode*	surfaceNode = startingHere->asSurfaceNodePtr();
		if( surfaceNode == NULL ) break;

		AlSurface	*surface = surfaceNode->surface();
		if( ! AlIsValid( surface ) )
			{
			break;
			}

		printf( "SURFACE CV LIST:\n" );
		printf( "{\n" );

		AlSurfaceCV *lastCV = surface->firstCV();
		if( ! AlIsValid( lastCV ) )
			printf( "No valid CVs for this surface\n" );
		else
			{
			do {
				AlSurfaceCV *surfaceCV =
						lastCV->copyWrapper()->asSurfaceCVPtr();
				do {
					print_surface_cvs_clusters( surfaceCV );
					} while( sSuccess == surfaceCV->nextInVD() );
				delete surfaceCV;

				} while( sSuccess == lastCV->nextInUD() );
			delete lastCV;
			}
		printf( "}\n" );
		delete surface;
		break;
		}

	case kGroupNodeType:
		{
		AlGroupNode *groupNode = startingHere->asGroupNodePtr();
		if( AlIsValid( groupNode ) )
			{
			AlDagNode *childNode = groupNode->childNode();
			if( AlIsValid( childNode ) )
				{
				print_branchs_clusters( childNode );
				delete childNode;
				}
			}
		break;
		}

		} // end of switch( type )
	
	// Take a look for siblings of this node ...
	//
	AlDagNode *siblingNode = startingHere->nextNode();
	if( AlIsValid( siblingNode ) ) 
	{
		print_branchs_clusters( siblingNode );
		delete siblingNode;
	}
	printf( "}\n" );
}

void print_dag_node_effects( AlDagNode* dagNode, AlCluster*	cluster,
							 boolean	doSiblings )
//
//	Description:
//		Print the percentage effects of all CVs underneath the given dag node
//		in the given cluster.
//
//		This is a good example of how to walk the affected CVs under a cluster.
//		The CVs are processed immediately in here, and the dag nodes are
//		recursively called until they reach the leaf level at which time their
//		CVs are processed.
//
{
	if( !AlIsValid( dagNode ) || !AlIsValid( cluster ) ) 
		return;

	// If the given dag node is a cluster node, then print out its name
	//
	switch( dagNode->type() ) {
	case kClusterNodeType:
		{
		AlClusterNode	*clusterNode = dagNode->asClusterNodePtr();
		if( clusterNode == NULL ) break;

		AlCluster	*newCluster = clusterNode->cluster();
		if( newCluster == NULL )
			{
			break;
			}

		AlClusterMember *member = newCluster->firstMember();
		if( ! AlIsValid( member ) )
			{
			printf( "ERROR: No valid members to print\n" );	
			delete cluster;
			break;
			}
		else
			{
			do {
				AlObject *object = member->object();

				if( object->asDagNodePtr() )
					print_dag_node_effects( object->asDagNodePtr(),
							cluster, TRUE );
				else
					{
					AlClusterable * cable = object->asClusterablePtr();
					const char * nm = dagNode->name();
					printf(
						"(CLUSTER %s) MEMBER %p, effect = %g\n",
						nm, object, cable->percentEffect( cluster ) );
					}
				delete object;
				} while( sSuccess == member->nextClusterMemberD() );
			delete member;
			} // else member is not valid
		delete cluster;
		break;
		}

	case kCurveNodeType:
		{
		AlCurveNode *curveNode = dagNode->asCurveNodePtr();
		if( curveNode == NULL ) break;

		AlCurve	*curve = curveNode->curve();
		if( !AlIsValid( curve ) )
			{
			break;
			}

		AlCurveCV *curveCV = curve->firstCV();
		if( ! AlIsValid( curveCV ) )
			{
			printf( "ERROR: No valid curves to print\n" );
			delete curve;
			break;
			}
		else
			do
				{
				const char * nm = dagNode->name();
				printf( "(%s) CV %p, effect = %g\n", 
					nm, curveCV, curveCV->percentEffect( cluster ) );
				} while( sSuccess == curveCV->nextD() );
			delete curveCV;

		delete curve;
		break;
		}

	case kFaceNodeType:
		{
		AlFaceNode*	faceNode = dagNode->asFaceNodePtr();
		if( faceNode == NULL ) break;

		AlCurve*		face = faceNode->curve();
		if( face == NULL )
			{
			break;
			}

		if( AlIsValid( face ) ) {
			do {
				AlCurveCV *curveCV = face->firstCV();
				if( AlIsValid( curveCV ) )
					{
					do {
						const char *nm = dagNode->name();
						printf( "(%s) CV %p, effect = %g\n",
								nm, curveCV,
								curveCV ->percentEffect( cluster ) );
						} while( sSuccess == curveCV->nextD() );
					delete curveCV;
					}
				} while( sSuccess == face->asFacePtr()->nextFaceD() );
			delete face;
			}
		break;
		}

	case kSurfaceNodeType:
		{
		AlSurfaceNode*	surfaceNode = dagNode->asSurfaceNodePtr();
		if( surfaceNode == NULL ) break;
		AlSurface*		surface = surfaceNode->surface();
		if( ! AlIsValid( surface ) )
			{
			break;
			}
		AlSurfaceCV *lastCV = surface->firstCV();
		if( ! AlIsValid( lastCV ) )
			{ 
			delete surface;
			break;
			}

		do {
			AlSurfaceCV *surfaceCV = lastCV->copyWrapper()->asSurfaceCVPtr();
			do {
				printf( "(%s) CV %p, effect = %g\n",
						dagNode->name(), surfaceCV,
						surfaceCV->percentEffect( cluster ) );
				} while( sSuccess == surfaceCV->nextInVD() );
			delete surfaceCV;
			} while( sSuccess == lastCV->nextInUD() );
		delete lastCV;

		delete surface;
		break;
		}

	case kGroupNodeType:
		{
		AlGroupNode *groupNode = dagNode->asGroupNodePtr();
		if( ! AlIsValid( groupNode ) )
			{
			// Traverse the dag.  For the child under this dag, 
			// call print_branchs_clusters().
			//
			AlDagNode *childNode = groupNode->childNode();
			if( AlIsValid( childNode ) )
				{
				print_dag_node_effects( childNode, cluster, TRUE );
				delete childNode;
				}
			}
		break;
		}
		} // end of switch
	
	// Continue to traverse the dag.  For the next sibling of 
	// this node, call print_dag_node_effects().
	//
	if( doSiblings )
		{
		AlDagNode *siblingNode = dagNode->nextNode();
		if( AlIsValid( siblingNode ) )
			{
			print_dag_node_effects( siblingNode, cluster, TRUE );
			delete siblingNode;
			}
		}
}

void print_dag_nodes_clusters( AlDagNode *dagNode )
//
//	Description:
//		Print all clusters that the given dag node is in
//
{
	AlCluster*	cluster;

	if( !AlIsValid( dagNode ) )
		return;

	cluster = dagNode->firstCluster();
	const char * nm = dagNode->name();
	if( cluster == NULL )
		printf( "Dag node '%s' (NO CLUSTER)\n", nm );
	else
		{
		do
			{
			printf( "Dag node '%s' in cluster %p\n", nm, cluster );
			print_dag_node_effects( dagNode, cluster, FALSE );
			} while( sSuccess == dagNode->nextClusterD( cluster ) );

		delete cluster;
		}
}

void print_curve_cvs_clusters( AlCurveCV *curveCV )
//
//	Description:
//		Print all clusters that the given curve CV is in
//
{
	if( !AlIsValid( curveCV ) ) 
		return;

	AlCluster	*cluster = curveCV->firstCluster();
	if( cluster != NULL ) {
		do {
			printf( "Curve CV %p in cluster %p (%g effect)\n",
						curveCV, cluster, curveCV->percentEffect( cluster ) );
		} while( sSuccess == cluster->nextClusterD() );
		delete cluster;
	}
}

void print_surface_cvs_clusters( AlSurfaceCV *surfaceCV )
//
//	Description:
//		Print all clusters that the given surface CV is in
//
{
	AlCluster*	cluster;

	if( !AlIsValid( surfaceCV ) )
		return;

	cluster = surfaceCV->firstCluster();
	if( AlIsValid( cluster ) ) {
		do {
			printf( "Surface CV %p in cluster %p (%g effect)\n",
				surfaceCV, cluster, surfaceCV ->percentEffect( cluster ) );
		} while( sSuccess == cluster->nextClusterD() );
		delete cluster;
	}
}
