

#include <stdio.h>
#include <string.h>

#include <AlUniverse.h>
#include <AlPixFile.h>
#include <AlCloud.h>
#include <AlDebug.h>

#include <math.h>

//	********************************************************************
//	This is a simple example that illustrates the use of the AlCloud and
//	AlPixFile Alias API classes.  This application takes an image, opens
//	it with AlPixFile and then processes the points to create a cloud.
//
//	The point recognizition algorithm could use some work.
//	********************************************************************


//	The dimensions of the cloud box
#define MAX_HEIGHT 48
#define MAX_WIDTH  64

void printPixel( AlPixel *pixel )
{
    printf("pixel: %d %d %d\n",pixel->r,pixel->g,pixel->b);
}

// Extremely fuzzy logic to try to map rgb to black
boolean blackPixel( AlPixel *pixel )
{
    if ( pixel->r < 20 && pixel->g < 20 && pixel->b < 20 )
        return TRUE;
    return FALSE;
}

// Even more fuzzier logic
boolean grayPixel( AlPixel *pixel )
{
    double gray = .3 * pixel->r + .58 * pixel->g + .12 * pixel->b;
    if ( gray < 128.0 )
        return TRUE;
    return FALSE;
}

// Gray conversion of a pixel
int gray( AlPixel *pixel )
{
    return (int)(.3 * pixel->r + .58 * pixel->g + .12 * pixel->b);
}

// Routine to add a point to a float [][3] array 
void addPoint( int width, int height, int x, int y, int z, int &pointCount,float * points)
{
    float *pt = (points + pointCount*3);
    pt[0] = (x/(float) width) * MAX_WIDTH;
    pt[1] = (y/(float) height) * MAX_HEIGHT;
    pt[2] = z;
    pointCount++;
}

// Routine that given a fileName, creates the 2D cloud
void convert_image( char *fileName )
{

	if ( fileName == NULL )
	{
		printf("Conversion routine passed a null pointer\n");
		return;
	}

    // Open the Pix file
    AlPixFile *pix = new AlPixFile;
    statusCode ret = pix->openForRead(fileName );
    if ( ret != sSuccess )
    {
        printf("Failed to open pix file: %s\n",fileName);
        return;
    }

    // Get the dimensions
    int width, height;
    width = pix->width();
    height= pix->height();

    // Create the cloud 
    AlCloud *cloud = new AlCloud;
    if ( cloud != NULL && cloud->create() == sSuccess )
    {
        // Allocate enough storage for the max size cloud
        float *points = (float *) malloc( width*height*sizeof(float)*3 );
        if ( points != NULL )
        {
            AlPixel *pixel = new AlPixel[width];
            int pointCount = 0;
            for ( int i = height - 1; i >= 0; i-- )
            {
                int pixelsRead = pix->read( pixel );
                if ( pixelsRead < 1 )
                    break;

                int gp,prev_gp = 255;
                for ( int j = 0; j < pixelsRead; j++ )
                {
                    gp = gray( &pixel[j] );
                    if ( prev_gp > ( gp + 3 ) )
                    {
						// Only creating point in x,y plane
                        addPoint(width,height,j,i,0,pointCount,points);
                    }
                    prev_gp = gp;
                }
            }
            if ( pointCount != 0 )
            {
                if ( cloud->addPoints(0,0,0,width,height,1,pointCount,points) != sSuccess )
                {
                    printf("Failed to add points to the cloud\n");
                }
            }
			// Remove point structure
            delete points;
			// Free cloud wrapper
			delete cloud;	
        }
    }

    // Close the pix file
    if ( pix->close() != sSuccess )
    {
        printf("Failed to close pix file: %s\n",fileName);
        return;
    }

	delete pix;
}

int main( int argc, char** argv )
{
	char	*imageFile, *outfileName;
	outfileName = NULL;

	if( argc !=3  )
	{
		fprintf(stderr, "usage: %s <imageFilename> <outfileName>\n", argv[0] );
		fprintf(stderr,
				"description:\n"
				"  This example illustrates how to create a cloud of\n"
				"  points from an image file.\n"
				"  It is intended to be a source example of how to use the\n"
				"  Alias Wire File Interface Library.\n"
				);
		exit( 0 );
	}

	imageFile  = argv[1];
	outfileName = argv[2];

	//
	// This must be called before any other Alias Wire File Interface
	// method can be called
	//
	AlUniverse::initialize( kYUp );

	convert_image( imageFile );	

	if ( AlUniverse::store( outfileName ) != sSuccess )
	{
		printf("Failed to store wire file: %s\n",outfileName);
		exit(0);
	}

	if ( AlUniverse::deleteAll() != sSuccess )
	{
		printf("Failed to delete the universe\n");
		exit(0);
	}

#if 0
	printf("--------------------------------------------------------\n");
	printf("Dumping wrapper usage...\n");
	AlDebug::outputWrapperUsage(kStdout);
	printf("--------------------------------------------------------\n");
#endif

    return 0;
}
