/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/

//
//	OpenAlias plug-in that creates a window with a single menu.  Invoking the
//	menu item creates a dialog box.  This example illustrates how to mix 
//	OpenAlias code and Windows(TM) UI functions together.
//

// OpenAlias includes
#include <AlUniverse.h>
#include <AlLiveData.h>
#include <AlFunctionHandle.h>
#include <AlFunction.h>

// 
#include <windows.h>

// Standard C libraries
#include "string.h"
#include <stdlib.h>

// Local includes
#include "winDialogBoxExample.h"

//
//	Static variables
//
static HMENU pluginMenuResource 	= NULL;
static HGLOBAL pluginDialogResource = NULL;
static HRSRC menuResource 			= NULL;
static HWND mainWindow 				= NULL;
static char *pluginName 			= NULL;

// 
//	Plug-in name helper functions
//
static void setPluginPathName( const char *name )
{
	char nameBuf[FILENAME_MAX];

	if ( name != NULL )
	{
		// The plug-in name has to be hard coded for now.
		sprintf(nameBuf,"%s\\%s",name, PLUGIN_NAME );
		pluginName = strdup( nameBuf );
		unsigned int i;
		for ( i = 0; i < strlen(pluginName); i++ )
		{
			if ( pluginName[i] == '/' )
				pluginName[i] = '\\';
		}
	}
	else	
	{
		pluginName = NULL;
	}
}

const char *pluginPathName()
{
	return pluginName;
}

void freePluginNameString()
{
	if ( pluginName != NULL )
		free( pluginName );
}

//
//	Find a handle to the plug-in so that resources can be found
//
HMODULE pluginHandle()
{
	const char *pluginPath = pluginPathName();
	if ( pluginPath == NULL )
		return NULL;

	HMODULE piModule = GetModuleHandle( pluginPath );
	if ( piModule == NULL )
		return NULL;

	return piModule;
}

boolean loadResources()
{
	HMODULE piModule = pluginHandle();
	if ( piModule == NULL )
		return FALSE;

	//
	// The plug-in handle is necessary since the resources we
	// need are located there.
	//
	pluginMenuResource = LoadMenu( piModule, MENU_NAME );
	if ( pluginMenuResource == NULL )
		return FALSE;
	
	HRSRC resource = FindResource( piModule, DIALOG_NAME, RT_DIALOG );
	if ( resource == NULL )
		return FALSE;

	pluginDialogResource = LoadResource( piModule, resource );
	if ( pluginDialogResource == NULL )
		return FALSE;
	
	return TRUE;
}

BOOL CALLBACK DlgProc( HWND hDlg , UINT iMsg, WPARAM wParam, LPARAM lParam )
{
	switch ( iMsg )
	{
		case WM_INITDIALOG:
			return TRUE;
		case WM_COMMAND:
			switch ( LOWORD( wParam ) )
			{
				case IDOK:
				case IDCANCEL:
					const int bufSize=100;
					char buf[bufSize];
					HWND editBox = GetDlgItem( hDlg, DIALOG_EDIT_TEXT_ID );
					if ( editBox != NULL )
					{
						buf[0] = '\0';
						GetWindowText( editBox, buf, bufSize );
						AlPrintf( kPrompt, "User entered: %s", buf );
					}
					EndDialog( hDlg, 0 );
					return TRUE;
			}
			break;
	}
	return FALSE;
}

LRESULT CALLBACK WndProc( HWND hwnd, UINT iMsg, WPARAM wParam, LPARAM lParam )
{
	static PAINTSTRUCT ps;

	switch ( iMsg )
	{
		case WM_CREATE:
			return 0;
		case WM_COMMAND:
			switch( LOWORD( wParam ) )
			{
				case IDM_INFO:
						// Locate the resource file and pass the parameter
						// into the DialogBox() function.
						HMODULE piModule = pluginHandle(); 
						if ( piModule == NULL )
							return 0;
						DialogBox(piModule, DIALOG_NAME, hwnd, DlgProc );
						return 0;
			}
			break;
		case WM_PAINT:
			BeginPaint( hwnd, &ps );
			EndPaint( hwnd, &ps );
		case WM_DESTROY:
			PostQuitMessage( 0 );
			break;
	}
	
	return DefWindowProc( hwnd, iMsg, wParam, lParam );
}

static void createWinDialogBox()
{
	static char szAppName[] 	= APP_NAME;
	static char szMenuName[] 	= MENU_NAME;
	HINSTANCE hInstance 		= GetModuleHandle(NULL); //	Application instance 
	WNDCLASSEX wndClass; 
	HWND hwnd;
	MSG msg;

	// Load resources before the window class is created.
	if (! loadResources() )
	{
		AlPrintf( kPrompt, "Failed to find window resources.");
		return;
	}

	wndClass.cbSize 		= sizeof(wndClass);
	wndClass.style 			= CS_HREDRAW|CS_VREDRAW ;
	wndClass.lpfnWndProc 	= WndProc ;
	wndClass.cbClsExtra 	= 0;
	wndClass.cbWndExtra 	= 0;
	wndClass.hInstance 		= hInstance;
	wndClass.hIcon 			= NULL;
	wndClass.hCursor 		= NULL;
	wndClass.hbrBackground 	= (HBRUSH__ *) GetStockObject( WHITE_BRUSH );
	wndClass.lpszMenuName 	= szMenuName;
	wndClass.lpszClassName 	= szAppName;
	wndClass.hIconSm 		= NULL;

	RegisterClassEx(&wndClass);
	mainWindow = CreateWindow( szAppName, "Simple OpenAlias UI example.",
	 						WS_OVERLAPPEDWINDOW,
							CW_USEDEFAULT, CW_USEDEFAULT,
							MAIN_WINDOW_WIDTH, MAIN_WINDOW_HEIGHT,	
							NULL, pluginMenuResource, hInstance, NULL );

	ShowWindow( mainWindow, SW_SHOW );
	UpdateWindow( mainWindow );
	DrawMenuBar( mainWindow );

	// Message loop for the window
	while ( GetMessage( &msg, NULL, 0, 0 ) )
	{
		TranslateMessage( &msg );
		DispatchMessage( &msg );
	}
}

void destroyWindow()
{
	if ( mainWindow != NULL )
	{
		DestroyWindow( mainWindow );
	}
	freePluginNameString();
}

//
// Standard plug-in handles and functions
//
static AlFunctionHandle h;
static AlMomentaryFunction hFunc;

extern "C"
PLUGINAPI_DECL int plugin_init( const char* path )
{
	AlUniverse::initialize();
	hFunc.create( "WinDialog Plug-in", createWinDialogBox );

	h.create( "WinDialog_Plug", &hFunc );
	h.addToMenu("mp_objtools" );

	AlPrintf( kPrompt, "The Dialog Box plug-in is installed on Palette 'Object Edit'");

	setPluginPathName( path );

	return 0;
}

extern "C"
PLUGINAPI_DECL int plugin_exit( void )
{
	destroyWindow();
	h.deleteObject();
	hFunc.deleteObject();
	return 0;
}
