/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/

#include <AlUniverse.h>
#include <AlViewFrame.h>
#include <AlPickList.h>
#include <AlObject.h>

#include <AlLiveData.h>
#include <AlFunction.h>
#include <AlFunctionHandle.h>
#include <AlIterator.h>

#ifdef _WIN32
#include <string.h>
#else
#include <strings.h>
#endif

// A simple iterator used to pass the frame number
// along when viewing an object.
class viewIterator: public AlIterator
{
	public:
				viewIterator( double frame = 0.0 )
				: frameNum( frame ) {};
		virtual	~viewIterator() {};

		virtual	int func( AlObject* );

	private:
		double frameNum;
};

// The iterator's one and only method. Does a viewframe
// on the passed in object using the frame number in the
// iterator object.
int viewIterator::func( AlObject* obj )
{
	if ( obj == NULL )
		return 0;

	AlAnimatable*	animitem;
	AlCurve*		curve;
	AlSurface*		surface;

	AlObjectType	type = obj->type();

	if( animitem = obj->asAnimatablePtr() )
		AlViewFrame::viewFrame( animitem, frameNum, AlViewFrame::kObjectAndBelow );
	else if( curve = obj->asCurvePtr() )
		AlViewFrame::viewFrame( curve, frameNum );
	else if( surface = obj->asSurfacePtr() )
		AlViewFrame::viewFrame( surface, frameNum );
	else
		AlPrintf( kPrompt, "Sorry, %s is not a viewable object", obj->name() );
		;	// a non viewable object
	return 0;
}

// The actual plugin function.
void my_view()
{
	int		frameNum;

	// Get the frame number from the option box.
	if( sSuccess == AlGetInteger( "frame", frameNum ) )
	{
		// Print the number to the promptline.
		AlPrintf( kPrompt, "Viewing frame %d.", frameNum );

		// Instantiate an iterator and apply it to all picked
		// objects.
		int rc;
		viewIterator*	view = new viewIterator( frameNum );
		if ( sSuccess != AlPickList::applyIteratorToItems( view, rc ) )
			AlPrintf( kPrompt, "Iterator failed." );
		if ( rc != 0 )
			AlPrintf( kPrompt, "Return code is %d.", rc );

		AlPrintf( kPrompt, "Viewing complete." );
	}
}

// This handle may have to be global if you wish to remove the
// plugin from the menu using the h.destroy() method in the
// 'momentary_exit' function.
// The menu entry is automatically removed when Alias exits.
//
static AlFunctionHandle h;
static AlMomentaryFunction hFunc;

extern "C"
PLUGINAPI_DECL int plugin_init( const char *dirName )
//
// This routine initializes the plugins and attaches it to the menu.
// It returns 0 if there is no initialization error.
//
{
	char *dirScm;

	// Initialize the universe. This must be done by all
	// plugins. If the universe is not initialized the plugin
	// will fail.
	AlUniverse::initialize( );

	dirScm = makeAltPath(dirName,"scm");

	// Invoke the scheme file which sets defaults for the scheme
	// variables.
	AlInvokeSchemeFile( "viewFrameExample_init.scm", dirScm );

	// Allocate a function handle. The first argument is the label on
	// the menu and the second is the function to invoke when the
	// menu item is selected.
	hFunc.create( my_view );
	h.create( "View Frame", &hFunc ); 

	// Define the attribute string for the attribute line below
	// the prompt line.
	h.setAttributeString( "viewframe" );

	// Read in the option box. The first argument is the
	// option box in $ALIAS_WORKENV, and
	// the second argument is the name of the option box given
	// in the scheme file. The third argument is a directory
	// path(possibly NULL) where to look for the scheme file.
	// If a call to this method is omitted, there will be no option box.
    if ( sSuccess != h.setOptionBox( "viewFrameExample.scm", "frame.options", dirScm) ) {
        AlPrintf( 
			kPrompt, 
			"Frame plug-in unable to find .scm file for option box"
		);
        return (1);
    }


	// Indicate which menu to add the plugin to. addToMenu()
	// adds the plugin to the bottom of the menu, while
	// appendToMenu() will add it to the top of the menu.
	h.addToMenu( "mp_objtools" );

	h.setIconPath( makeAltPath( dirName, NULL ) );

	// Return a success code.
	// Returning a non zero indicates an error.
	// An error value ( a non-zero ) will be printed on the prompt
	// line in Alias.
	AlPrintf( kPrompt, "Frame attached to Palette 'Object Edit'");
	return 0;
}

extern "C"
PLUGINAPI_DECL int plugin_exit( void )
{
	// Remove the plugin from the menu and free the FunctionHandle.
	// Note that h.destroy() implicitly calls h.removeFromMenu()
	h.deleteObject();
	hFunc.deleteObject();
	return 0;
}

