/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/

/*

toggleObjectInvisibility.plugin ( formerly ) jptInvertVisible.plugin
	
	This plugin makes objects that are visible invisible,
	and vice versa.

    Installed under ObjectDisplay menu.

Options:

    Camera: Ignores cameras.

    Light: Ignores lights.

    Texture Nodes: Ignores texture nodes.

    Construction Entities: Ignores construction entities.

Version History:

    Icon now stays on the shelf after quit.

	
*/

#include <AlUniverse.h>
#include <AlPickList.h>
#include <AlObject.h>

#include <AlLiveData.h>
#include <AlFunction.h>
#include <AlFunctionHandle.h>
#include <AlIterator.h>
#include <AlDagNode.h>
#include <AlGroupNode.h>
#include <AlAccessTypes.h>
#include <AlConstructionEntity.h>

#include <stdio.h>
#include <string.h>

static int gCamera = 0;
static int gLight = 0;
static int gTexture = 0;
static int gConstructionEntity = 0;

/*
*
*/
boolean jIgnoreObject( AlGroupNode *group )
{
	if ( !group)
		return FALSE;

	AlDagNode *child;

	child = group->childNode();

	if ( !child )
		return FALSE;

	if ( !gCamera && child->asCameraNodePtr() )
		return TRUE;

	if ( !gLight && child->asLightNodePtr() )
		return TRUE;

	return FALSE;
}

/*
*
*/
void jProcessObject( AlDagNode *object )
{
	if ( object == NULL )
		return;

	if ( gTexture && object->asTextureNodePtr() )
	{
		object->setDisplayMode( 
			kDisplayModeInvisible, 
			!object->isDisplayModeSet( kDisplayModeInvisible ) );
	}
	else if ( jIgnoreObject( object->asGroupNodePtr() ) )
	{
		;
	}
	else 	// invert visible
	{
		object->setDisplayMode( 
			kDisplayModeInvisible, 
			!object->isDisplayModeSet( kDisplayModeInvisible ) );
	}

	AlDagNode *next = object->nextNode();
	jProcessObject( next );

	delete next;
}

void jProcessConstructionEntities()
{
	// Should we ignore these objects
	if ( gConstructionEntity )
		return;
		
	AlConstructionEntity *ce = AlUniverse::firstConstructionEntity();
	if ( ce != NULL )
	{
		while ( ce != NULL)
		{
			boolean inv = ce->invisible();
			ce->setInvisible( ! inv );
			AlConstructionEntity *cur = ce;
			ce = AlUniverse::nextConstructionEntity(ce);
			delete cur;
		}
		delete ce;
	}
}

/*
*
*/
void do_it( void )
{ 
	if ( sSuccess != AlGetInteger("invertVisible.camera", gCamera) )
		gCamera = 0;
	if ( sSuccess != AlGetInteger("invertVisible.light", gLight) )
		gLight = 0;
	if ( sSuccess != AlGetInteger("invertVisible.texture", gTexture) )
		gTexture = 0;
	if ( sSuccess != AlGetInteger("invertVisible.constructionentity", gConstructionEntity) )
		gConstructionEntity = 0;
		
	AlPickList::clearPickList();
	AlDagNode *dag = AlUniverse::firstDagNode();

	jProcessObject( dag );
	
	jProcessConstructionEntities();
		
	AlUniverse::redrawScreen();

	delete dag;
};

// This handle may have to be global if you wish to remove the
// plugin from the menu using the h.destroy() method in the
// 'momentary_exit' function.
// The menu entry is automatically removed when Alias exits.
//
static AlFunctionHandle h;
static AlMomentaryFunction hFunc;

extern "C"
PLUGINAPI_DECL int plugin_init( const char *dirName )
//
// This routine initializes the plugins and attaches it to the menu.
// It returns 0 if there is no initialization error.
//
{
	// Initialize the universe. This must be done by all
	// plugins. If the universe is not initialized the plugin
	// will fail.
	AlUniverse::initialize( );

	char *dirScm = makeAltPath( dirName, "scm" );
	// Invoke the scheme file which sets defaults for the scheme
	// variables.
	// AlInvokeSchemeFile( "nothing_init.scm" );

	//	Create the function as a AlMomentaryFunction
	//	note that the icon name will be nothing_func.S, nothing_func.M
	hFunc.create( "pl_jptInvertVisible", do_it );

	// Allocate a function handle. The first argument is the label on
	// the menu and the second is the function to invoke when the
	// menu item is selected.

	h.create( "Toggle Object Invisibility", &hFunc ); 

	// Define the attribute string for the attribute line below
	// the prompt line.
	h.setAttributeString( "invert_visible" );

	// Read in the option box. The first argument is the
	// option box in $ALIAS_WORKENV or an absolute path, and
	// the second argument is the name of the option box given
	// in the scheme file. If a call to this method is omitted
	// there will be no option box.
	h.setOptionBox( "toggleObjectInvisibility.scm", "invertVisible.options", dirScm );

	//	Alternatively, you could have used:
	//	h.setOptionBox( "nothing.scm", "nothing.options", dirName )

	h.setIconPath( makeAltPath( dirName, NULL ) );

	//
	//	Then the scheme file and the icon images would be loaded from
	//	the same path as the plugin rather than from the standard locations

	// Indicate which menu to add the plugin to. addToMenu()
	// adds the plugin to the bottom of the menu, while
	// appendToMenu() will add it to the top of the menu.
	h.addToMenu( "mp_objdisplay" );

	// Return a success code.
	// Returning a non zero indicates an error.
	// An error value ( a non-zero ) will be printed on the prompt
	// line in Alias.
	AlPrintf( kPrompt, "Toggle Object Invisibility plug-in installed under menu 'ObjectDisplay'");
	return 0;
}

extern "C"
PLUGINAPI_DECL int plugin_exit( void )
{
	// Remove the plugin from the menu and free the FunctionHandle.
	// Note that h.destroy() implicitly calls h.removeFromMenu()
	h.deleteObject();
	hFunc.deleteObject();
	return 0;
}

