/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/


#include <AlUniverse.h>
#include <AlPickList.h>
#include <AlChannel.h>
#include <AlParamAction.h>
#include <AlViewFrame.h>
#include <AlLinkItem.h>
#include <AlList.h>
#include <AlIterator.h>
#include <AlAnimatable.h>
#include <AlDirectionLight.h>
#include <AlLightNode.h>
#include <AlGroupNode.h>

#include <AlLiveData.h>
#include <AlFunction.h>
#include <AlFunctionHandle.h>

#include <Tsun.h>
#include <math.h>
#include <string.h>

//
//	This plugin creates a directional light which simulates the sun
//	(based on time and location of the viewer).
//

// Our instance of the function handle
//
static AlFunctionHandle sunShineHandle;
static AlMomentaryFunction	sunShineFunc;

// Local helper classes
//

// A structure to hold all the information in the option dialog
//
struct TsunOptionInfo {
	int		iPosition;			// 0: Time/Location  1: Alt/Azimuth
	double	dAltitude;			// Sun altitude
	double	dAzimuth;			// Sun azimuth
	int		iMonth;				// Month
	int		iDay;				// Day
	int		iHour;				// Hour (12-hour format)
	int		iMinute;			// Minute
	int		iAmPm;				// 0: AM  1: PM
	double	dLatitude;			// Latitude (degrees.minutes)
	int		iNS;				// 0: North  1: South
};

// Our actual  control class
//
class TsunShine {
public:
	TsunShine ();
	~TsunShine ();

	statusCode createSunShine (void);
private:
	TsunOptionInfo	fSunOptionInfo;
};

// Local functions
//
void createSunShine (void);

/*
======================================================================
plugin_init() - OpenAlias entry/initialization function
----------------------------------------------------------------------
Parameters:
    None
Returns:
    (int) 0 upon successful completion
======================================================================
*/
extern "C" 
PLUGINAPI_DECL int plugin_init (const char *dirName)
{
	char *dirScm;

	// Initialize OpenAlias
	//
	AlUniverse::initialize (kYUp);

	dirScm = makeAltPath(dirName,"scm");

	// Invoke our initialisation scheme module
	//
	AlInvokeSchemeFile ("sunShine.i.scm", dirScm);

	// Establish ourselves as a OpenAlias plug-in
	//
	sunShineFunc.create ("pl_sunShine", createSunShine);
	sunShineHandle.create ("Sun Shine", &sunShineFunc);
	sunShineHandle.setAttributeString ("sun");

	// Attach our option box
	//
	if ( sSuccess != sunShineHandle.setOptionBox ("sunShine.o.scm", 
												  "sunshine.options", dirScm) ) {
		AlPrintf( kPrompt, "SunShine plug-in unable to find .scm file for option box" );
		return 1;
	}
	sunShineHandle.setIconPath( makeAltPath( dirName, NULL ) );
	sunShineHandle.installOnSubmenu ("rp_light_sub");

	AlPrintf( kPrompt, "Sunshine installed in Render->Create lights Submenu.");
	return (0);
}

/*
======================================================================
plugin_exit() - OpenAlias cleanup function
----------------------------------------------------------------------
Parameters:
    None
Returns:
    Nothing
======================================================================
*/
extern "C" 
PLUGINAPI_DECL int plugin_exit (void)
{
	sunShineHandle.deleteObject ();
	sunShineFunc.deleteObject ();
	return (0);
}


/*
======================================================================
createSunShine() - OpenAlias entry/initialization function
----------------------------------------------------------------------
Parameters:
    None
Returns:
    Nothing
======================================================================
*/
void
createSunShine (void)
{
	TsunShine	sunShine;

	if (sunShine.createSunShine () == sSuccess) {
		AlPrintf (kPrompt, "Sun Shine complete");
	}
}

/*
======================================================================
TsunShine::TsunShine() - Constructor for our
                       use expression results control object
----------------------------------------------------------------------
Parameters:
    None
Notes:
    The constructor will retrieve all the settings in the option dialog
Returns:
    Nothing
======================================================================
*/
TsunShine::TsunShine ()
{
	// Read our option dialog settings
	//
	AlGetInteger ("ao_sunshine_position", fSunOptionInfo.iPosition);
	if (fSunOptionInfo.iPosition == 0) {	// Time/Location
		AlGetInteger ("ao_sunshine_month", fSunOptionInfo.iMonth);
		AlGetInteger ("ao_sunshine_day", fSunOptionInfo.iDay);
		AlGetInteger ("ao_sunshine_hour", fSunOptionInfo.iHour);
		AlGetInteger ("ao_sunshine_minute", fSunOptionInfo.iMinute);
		AlGetInteger ("ao_sunshine_ampm", fSunOptionInfo.iAmPm);
		AlGetDouble ("ao_sunshine_latitude", fSunOptionInfo.dLatitude);
		AlGetInteger ("ao_sunshine_ns", fSunOptionInfo.iNS);
	}
	else {									// Alt/Azimuth
		AlGetDouble ("ao_sunshine_altitude", fSunOptionInfo.dAltitude);
		AlGetDouble ("ao_sunshine_azimuth", fSunOptionInfo.dAzimuth);
	}
}

/*
======================================================================
TsunShine::TsunShine() - Destructor for our
                       use expression results control object
----------------------------------------------------------------------
Parameters:
    None
Notes:
    The destructor is responsible for freeing up the list of actions
    that we created a long the way, as well as delete'ing all the OpenAlias
    channel wrappers contained in the list
Returns:
    Nothing
======================================================================
*/
TsunShine::~TsunShine ()
{
}

/*
======================================================================
TsunShine::createSunShine () - Actual control module to create the sun
----------------------------------------------------------------------
Parameters:
    None
Returns:
    Nothing
======================================================================
*/
statusCode
TsunShine::createSunShine ()
{
	Tsun	*theSun;
	statusCode	status = sSuccess;

	// Get the sun vector
	//
	if (fSunOptionInfo.iPosition == 0) {	// Time/Location
		theSun = new Tsun (
			fSunOptionInfo.iMonth,
			fSunOptionInfo.iDay,
			fSunOptionInfo.iHour + (fSunOptionInfo.iAmPm == 1 ? 12 : 0),
			fSunOptionInfo.iMinute,
			fSunOptionInfo.dLatitude * (fSunOptionInfo.iNS == 1 ? -1.0 : 1.0)
		);
	}
	else {									// Alt/Azimuth
		theSun = new Tsun (fSunOptionInfo.dAltitude, fSunOptionInfo.dAzimuth);
	}

	// Now create the directional light
	//
	AlDirectionLight *sunLight = new AlDirectionLight;
	status = sunLight->create ();
	if (status != sSuccess) {
		return (status);
	}
	sunLight->setShadows (TRUE);
	AlLightNode *sunNode = sunLight->lightNode ();
	if (!sunNode) {
		return (sFailure);
	}
	AlGroupNode *sunGroup = sunNode->parentNode ();
	if (!sunGroup) {
		return (sFailure);
	}
	sunGroup->setName ("Sunlight");

	// Orient the Sunlight properly
	//
	if (AlUniverse::coordinateSystem () == kYUp) {
		sunGroup->setRotation (90.0 - theSun->altitude (), 180.0 - theSun->azimuth (), 0.0);
	}
	else {
		sunGroup->setRotation (90.0 - theSun->altitude (), 0.0, 180.0 - theSun->azimuth ());
	}

	// Redraw the screen, since we have added stuff to the universe
	//
	AlUniverse::redrawScreen ();
	return (status);
}
