/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/

#include <AlCoordinateSystem.h>
#include <AlUniverse.h>

#include <AlLiveData.h>
#include <AlFunction.h>
#include <AlFunctionHandle.h>
#include <AlPolysetNode.h>
#include <AlPolyset.h>
#include <AlPolygon.h>
#include <AlPolysetVertex.h>
#include <AlIterator.h>
#include <AlPickList.h>
#include <AlObject.h>

#define FN_UNFREEZE 0
#define FN_FREEZE   1

//
//  This is an example of an iterator used in conjunction with the picklist.
//  Note that with iterators, your function can have its own private data.
//
//

class modNormal : public AlIterator
{
    private:
	  int function;
      int count;

	  void freezeNormal( AlPolysetVertex *vertex,
						 AlPolysetNode   *polysetnode );

	  void unFreezeNormal( AlPolysetVertex *vertex,
						   AlPolysetNode *polysetnode );

    public:
      modNormal( int request )   { count=0; function=request; }
      ~modNormal()  {}
      virtual int func (AlObject* obj)
      { 
          // Don't include Shaders, TExtures, or Environments
          if ( obj->asPolysetVertexPtr() || obj->asPolysetNodePtr() ) {

            count++;

			switch ( function ) {

			  case FN_FREEZE:
				freezeNormal( obj->asPolysetVertexPtr(),
							  obj->asPolysetNodePtr() );
				break;

			  case FN_UNFREEZE:
				unFreezeNormal( obj->asPolysetVertexPtr(),
					  		    obj->asPolysetNodePtr() );
				break;

			  default:
				break;
			}
		  }
          return 0;
      }
      int getcount()
      {
          return count;
      }
};
void modNormal::freezeNormal( AlPolysetVertex *vertex,
							  AlPolysetNode   *polysetnode )
{
	if ( vertex ) {
		AlPolyset       *pset;

		vertex->setFreezeNormalFlag( TRUE );

		pset = vertex->polyset();
		pset->freezeNormals();
		delete pset;
	}
	else {
		AlPolyset       *pset;
		
		pset = polysetnode->polyset();

		pset->freezeNormals();

		//
		// Note that we simply freeze the polyset, and not all the
		// individual vertices.
		//

	}


}


void modNormal::unFreezeNormal( AlPolysetVertex *vertex,
							    AlPolysetNode *polysetnode )
{
	if ( vertex ) {
		AlPolyset      *pset;

		vertex->setFreezeNormalFlag( FALSE );
		pset = vertex->polyset();
		pset->unFreezeNormals();
		delete pset;
	}
	else {
		AlPolyset       *pset;
		AlPolysetVertex *vert;
		int              i;
		
		pset = polysetnode->polyset();

		pset->unFreezeNormals();

		for ( i = 0; i < pset->numberOfVertices(); i++ ) {
		
			vert = pset->vertex( i );

			vert->setFreezeNormalFlag( FALSE );

			delete vert;
		}
	}

}

void do_modNormal( void )
{
    int returnCode;
	int mode;

	AlGetInteger( "normal.mode", mode );
    modNormal Counter( mode );

    if (sSuccess == AlPickList::applyIteratorToItems( &Counter, returnCode))
        AlPrintf(kPrompt,   "Normals Modified: %d",
                            Counter.getcount());
    else
        AlPrintf(kPrompt, "Iteration Failed.");
}

static AlMomentaryFunction hFunc;
static AlFunctionHandle handle;

extern "C"
PLUGINAPI_DECL int plugin_init( const char *dirName )
{
    char *dirScm;

    AlUniverse::initialize( kZUp );

    hFunc.create( "NormalState", do_modNormal );

    handle.create( "Set Normal Freeze", &hFunc );
    handle.setAttributeString( "Normal State" );

    dirScm = makeAltPath( dirName, "scm" );

	AlInvokeSchemeFile( "setNormalFreeze_init.scm", dirScm );

    if ( sSuccess != handle.setOptionBox( "setNormalFreeze.scm", "normal.options", 
										 dirScm ) ) {

        AlPrintf( 
            kPrompt, 
            "Set Normal Freeze plug-in unable to find .scm file for option box"
        );
        return (1);
    }

    handle.setIconPath( makeAltPath( dirName, NULL ) ); 

    handle.appendToMenu( "al_polyedit" );

    AlPrintf( kPrompt, "Set Normal Freeze installed on Palette 'Polygon Edit'");
    return 0;
}

extern "C"
PLUGINAPI_DECL int plugin_exit( void )
{
    handle.deleteObject();
    hFunc.deleteObject();

    return 0;
}
