/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/
//
//	This is a sample OpenAlias plugin which will reset all the transformations
//	for the picked objects.  This is a very useful capability for skeletons
//	which have been ik dragged.  When resetTransforms is applied, the translate
//	and rotate values are set to 0.0, and the scale values are set to 1.0.
//	This is a handy way of returning a skeleton to its "rest" position, without
//	having to perform the operation manually.
//

#include <AlUniverse.h>
#include <AlPickList.h>
#include <AlChannel.h>
#include <AlParamAction.h>
#include <AlLinkItem.h>
#include <AlList.h>
#include <AlIterator.h>
#include <AlDagNode.h>

#include <AlLiveData.h>
#include <AlFunction.h>
#include <AlFunctionHandle.h>

#include <WalkTree.h>

#include <string.h>

// Our instance of the function handle
//
static AlFunctionHandle resetTransformsHandle;
static AlMomentaryFunction	resetTransformsFunc;

// Local helper classes
//

// A structure to hold all the information in the option dialog
//
struct TresetOptionInfo {
	int		iHierarchy;			// Hierachy settings
	boolean	bTranslate;			// Reset translate
	boolean	bRotate;			// Reset rotate
	boolean	bScale;				// Reset scale
};

// A static entry to hold the current hierarchy setting
//
static enumHier hierSetting;

// Our actual resetTransforms control class
//
class TresetTransforms {
public:
	TresetTransforms ();
	~TresetTransforms ();

	statusCode resetTransforms (void);

	TresetOptionInfo	fResetOptionInfo;
};

// A derived class to walk through the picked items
//
class TresetPickIterator : public AlIterator
{
public:
	TresetPickIterator (TresetTransforms *resetObject) : fResetObject (resetObject) {}
	virtual ~TresetPickIterator () {}
	virtual int func (AlObject *pTheObject);
private:
	TresetTransforms *fResetObject;
};

// A derived class to walk through an object's hierarchy
//
class TresetHierIterator : public AlIterator
{
public:
	TresetHierIterator (TresetTransforms *resetObject) : fResetObject (resetObject) {}
	virtual ~TresetHierIterator () {}
	virtual int func (AlObject *pTheObject);
private:
	TresetTransforms *fResetObject;
};

// Local functions
//
void resetTransforms (void);

/*
======================================================================
plugin_init() - OpenAlias entry/initialization function
----------------------------------------------------------------------
Parameters:
    None
Returns:
    (int) 0 upon successful completion
======================================================================
*/
extern "C" 
PLUGINAPI_DECL int plugin_init (const char *dirName)
{
	char *dirScm;

	// Initialize OpenAlias
	//
	AlUniverse::initialize (kYUp);

	dirScm = makeAltPath(dirName,"scm");

	// Invoke our initialisation scheme module
	//
	AlInvokeSchemeFile ("resetTransforms.i.scm", dirScm);

	// Establish ourselves as a OpenAlias plug-in
	//
	resetTransformsFunc.create ("pl_ResetTransforms", resetTransforms);
	resetTransformsHandle.create ("Reset Transforms", &resetTransformsFunc);
	resetTransformsHandle.setAttributeString ("reset");

	// Attach our option box
	//
	if ( sSuccess != resetTransformsHandle.setOptionBox ("resetTransforms.o.scm", 
														 "reset.options", dirScm) ) {
		AlPrintf( kPrompt, "ResetTransforms plug-in unable to find .scm file for option box" );
		return 1;
	}
	resetTransformsHandle.setIconPath( makeAltPath( dirName, NULL ) );
	resetTransformsHandle.appendToMenu ("mp_objtools");

	AlPrintf( kPrompt, "Reset Transforms added to Palette 'Object Edit'");
	return (0);
}

/*
======================================================================
plugin_exit() - OpenAlias cleanup function
----------------------------------------------------------------------
Parameters:
    None
Returns:
    Nothing
======================================================================
*/
extern "C" 
PLUGINAPI_DECL int plugin_exit (void)
{
	resetTransformsHandle.deleteObject ();
	resetTransformsFunc.deleteObject ();
	return (0);
}


/*
======================================================================
resetTransforms() - OpenAlias entry/initialization function
----------------------------------------------------------------------
Parameters:
    None
Returns:
    Nothing
======================================================================
*/
void
resetTransforms (void)
{
	TresetTransforms	resetTransforms;

	if (resetTransforms.resetTransforms () == sSuccess) {
		AlUniverse::redrawScreen ();
		AlPrintf (kPrompt, "Reset Transforms complete");
	}
}

/*
======================================================================
TresetTransforms::TresetTransforms() - Constructor for our
                       reset transforms control object
----------------------------------------------------------------------
Parameters:
    None
Notes:
    The constructor will retrieve all the settings in the option dialog
Returns:
    Nothing
======================================================================
*/
TresetTransforms::TresetTransforms ()
{
	// Read our option dialog settings
	//
	AlGetInteger ("mo_reset_transforms_hierarchy", fResetOptionInfo.iHierarchy);
	hierSetting = (enumHier)fResetOptionInfo.iHierarchy;
	int i;
	AlGetInteger ("mo_reset_transforms_translate", i );
	fResetOptionInfo.bTranslate = (boolean) i;	
	AlGetInteger ("mo_reset_transforms_rotate", i );
	fResetOptionInfo.bRotate = (boolean) i;
	AlGetInteger ("mo_reset_transforms_scale", i);
	fResetOptionInfo.bScale = (boolean) i;
}

/*
======================================================================
TresetTransforms::TresetTransforms() - Destructor for our
                       reset transforms control object
----------------------------------------------------------------------
Parameters:
    None
Returns:
    Nothing
======================================================================
*/
TresetTransforms::~TresetTransforms ()
{
}

/*
======================================================================
TresetTransforms::resetTransforms() - Control module to reset
                           the specified transformations for picked objects
----------------------------------------------------------------------
Parameters:
    None
Returns:
    Nothing
======================================================================
*/
statusCode
TresetTransforms::resetTransforms ()
{
	TresetPickIterator	*resetWalk;
	int					iResult;

	// Use an iterator to walk through the pick list
	//
	resetWalk = new TresetPickIterator (this);
	AlPickList::applyIteratorToItems (resetWalk, iResult);
	return (sSuccess);
}

/*
======================================================================
TresetPickIterator::func() - PickList iterator function to hierarchicaly
                           walk each picked object
----------------------------------------------------------------------
Parameters:
    None
Returns:
    Nothing
======================================================================
*/
int
TresetPickIterator::func (AlObject *pTheObject)
{
	TresetHierIterator *hierWalk = new TresetHierIterator (fResetObject);

	WalkTree::walkHierarchy (hierSetting, pTheObject, hierWalk);
	return (0);
}

/*
======================================================================
TresetHierIterator::func() - Hierarchy iterator function to actually
                           reset the specified transformations
----------------------------------------------------------------------
Parameters:
    None
Returns:
    Nothing
======================================================================
*/
int
TresetHierIterator::func (AlObject *theObject)
{
	if (!theObject || !fResetObject) {
		return (0);
	}
	AlDagNode	*theDagNode = theObject->asDagNodePtr ();
	if (!AlIsValid (theDagNode)) {
		return (0);
	}
	if (fResetObject->fResetOptionInfo.bTranslate) {
		theDagNode->setTranslation (0.0, 0.0, 0.0);
	}
	if (fResetObject->fResetOptionInfo.bRotate) {
		theDagNode->setRotation (0.0, 0.0, 0.0);
	}
	if (fResetObject->fResetOptionInfo.bScale) {
		theDagNode->setScale (1.0, 1.0, 1.0);
	}
	return (0);
}
