/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/

/*

jptPolyData.plugin

	This plugin shows how many different types of polygons 
	(i.e., tris, quads, pents) you have, and gives you a total
	count of each type.

	If polysets are selected, then the count is done only on
	the selected ones.  Otherwise it goes through the scene
	and counts each polyset.

	Installed under Windows menu.


*/

#include <AlUniverse.h>
#include <AlPickList.h>
#include <AlObject.h>

#include <AlLiveData.h>
#include <AlFunction.h>
#include <AlFunctionHandle.h>
#include <AlIterator.h>
#include <AlPickList.h>
#include <AlDagNode.h>
#include <AlGroupNode.h>

#include <AlPolysetNode.h>
#include <AlPolyset.h>
#include <AlPolygon.h>

#include <stdio.h>
#include <string.h>

#define MAX_VERTS 128

int gPolyData[MAX_VERTS], gPolys, gVerts;
boolean foundPickedPoly = FALSE;

/*
*
*/
void jResetData( int *data )
{
	int i;
	for (i = 0; i < MAX_VERTS; i++)
		data[i] = 0;
}

/*
*
*/
void jPrintData( int *data )
{
	int i;
	for (i = 0; i < MAX_VERTS; i++)
		if ( data[i] )
		{
			fprintf(stderr, "  Polygons with %d vertices: %d\n", i, data[i]);
			AlPrintf(kPrompt, "  Polygons with %d vertices: %d\n", i, data[i]);
		}
}

/*
*
*/
void jCountPolygons( AlPolysetNode *pnode )
{
	if ( pnode == NULL )
		return;

	foundPickedPoly = TRUE;

	fprintf(stderr, "\n------\n");
	AlPrintf(kPrompt, "\n------\n");
	fprintf(stderr, "%s:\n", pnode->name() );
	AlPrintf(kPrompt, "%s:\n", pnode->name() );

	AlPolyset *polyset = pnode->polyset();

	int np = polyset->numberOfPolygons();
	int nv = polyset->numberOfVertices();
	fprintf(stderr, "  polygons: %d\n", np );
	AlPrintf(kPrompt, "  polygons: %d\n", np );
	fprintf(stderr, "  vertices: %d\n", nv );
	AlPrintf(kPrompt, "  vertices: %d\n", nv );

	gPolys += np;
	gVerts += nv;

	AlPolygon polygon;
	int i, data[MAX_VERTS];

	jResetData( data );

	for (i = 0; i < np; i++)
	{
		polyset->polygonD(i, polygon);

		if ( !(polygon.numberOfVertices() < MAX_VERTS) )
			return;
		data[ polygon.numberOfVertices() ]++;
		gPolyData[ polygon.numberOfVertices() ]++;

	}

	jPrintData( data );

	delete polyset;
}

/*
*
*/
void jProcessObject( AlObject *object, boolean picked )
{
	if ( object == NULL )
		return;

//	const char *name = object->name();
//	fprintf(stderr, "\n%s:\n", name);
	jCountPolygons( object->asPolysetNodePtr() );

	if ( object->asGroupNodePtr() )
	{
		AlObject *child = ((AlGroupNode*)object)->childNode();

	//	fprintf(stderr, "-----  child node found...\n");
		jProcessObject( child, FALSE );

		delete child; 
	}

	AlObject *next;
	if ( picked )
	{
		AlPickList::nextPickItem();
		next = AlPickList::getObject();
	}
	else
		next = ((AlDagNode*)object)->nextNode();
	jProcessObject( next, picked );

	delete next;
}

/*
*
*/
void do_it( void )
{
	AlObject *obj = NULL;

	fprintf(stderr, "\n\n*****   Polyset Data Output   *****\n");
	AlPrintf(kPrompt, "\n\n*****   Polyset Data Output   *****\n");
	jResetData( gPolyData );
	gPolys = gVerts = 0;
	foundPickedPoly = FALSE;
	
	if ( AlPickList::firstPickItem() == sSuccess )
	{
		obj = AlPickList::getObject();
		jProcessObject( obj, TRUE );
	}
	
	if ( ! foundPickedPoly )
	{
		//fprintf(stderr, "Nothing picked.  Go through all objects.\n");
		AlPrintf(kPrompt, "No picked polygons.  Go through all objects.\n");
		obj = AlUniverse::firstDagNode();
		jProcessObject( obj, FALSE );
	}

	fprintf(stderr, "\n--------------------\n\n");
	AlPrintf(kPrompt, "\n--------------------\n\n");
	fprintf(stderr, "TOTAL COUNT:\n\n");
	AlPrintf(kPrompt, "TOTAL COUNT:\n\n");
	fprintf(stderr, "  polygons: %d\n", gPolys);
	AlPrintf(kPrompt, "  polygons: %d\n", gPolys);
	fprintf(stderr, "  vertices: %d\n", gVerts);
	AlPrintf(kPrompt, "  vertices: %d\n", gVerts);
	jPrintData( gPolyData );

	delete obj;
};

// This handle may have to be global if you wish to remove the
// plugin from the menu using the h.destroy() method in the
// 'momentary_exit' function.
// The menu entry is automatically removed when Alias exits.
//
static AlFunctionHandle h;
static AlMomentaryFunction hFunc;

extern "C"
PLUGINAPI_DECL int plugin_init( const char *dirName )
//
// This routine initializes the plugins and attaches it to the menu.
// It returns 0 if there is no initialization error.
//
{
	// Initialize the universe. This must be done by all
	// plugins. If the universe is not initialized the plugin
	// will fail.
	AlUniverse::initialize( );

	// Invoke the scheme file which sets defaults for the scheme
	// variables.
	// AlInvokeSchemeFile( "nothing_init.scm" );

	//	Create the function as a AlMomentaryFunction
	//	note that the icon name will be nothing_func.S, nothing_func.M
	hFunc.create( "pl_jptPolyData", do_it );

	// Allocate a function handle. The first argument is the label on
	// the menu and the second is the function to invoke when the
	// menu item is selected.

	h.create( "Polyset Data", &hFunc ); 

	// Define the attribute string for the attribute line below
	// the prompt line.
	h.setAttributeString( "poly_data" );

	// Read in the option box. The first argument is the
	// option box in $ALIAS_WORKENV or an absolute path, and
	// the second argument is the name of the option box given
	// in the scheme file. If a call to this method is omitted
	// there will be no option box.
	// h.setOptionBox( "nothing.scm", "nothing.options", dirName );

	//	Alternatively, you could have used:
	//	h.setOptionBox( "nothing.scm", "nothing.options", dirName )

	h.setIconPath( makeAltPath( dirName, NULL ) );

	//
	//	Then the scheme file and the icon images would be loaded from
	//	the same path as the plugin rather than from the standard locations

	// Indicate which menu to add the plugin to. addToMenu()
	// adds the plugin to the bottom of the menu, while
	// appendToMenu() will add it to the top of the menu.
	h.addToMenu( "mp_windows_menu" );

	// Return a success code.
	// Returning a non zero indicates an error.
	// An error value ( a non-zero ) will be printed on the prompt
	// line in Alias.
	AlPrintf( kPrompt, "PolyData plugin installed under Menu 'Windows'");
	return 0;
}

extern "C"
PLUGINAPI_DECL int plugin_exit( void )
{
	// Remove the plugin from the menu and free the FunctionHandle.
	// Note that h.destroy() implicitly calls h.removeFromMenu()
	h.deleteObject();
	hFunc.deleteObject();
	return 0;
}
