/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/

#include <AlCoordinateSystem.h>
#include <AlUniverse.h>

#include <AlPolysetNode.h>
#include <AlPolyset.h>
#include <AlPolygon.h>
#include <AlPolysetVertex.h>

#include <AlLiveData.h>
#include <AlFunction.h>
#include <AlFunctionHandle.h>

#include <AlCommand.h>
#include <AlUserCommand.h>
#include <AlNotifyDagNode.h>
#include <AlPickList.h>
#include <AlIterator.h>

#include <fstream>
#include <iostream>
#include <math.h>
#include <string.h>

using namespace std;

//
//	This plugin demonstrates the use of construction history to dynamically
//	recreate geometry.
//

#define PGON_CMD_ID	43
const int dataId = 100;

static void computePolygon( AlPolygon * opoly, AlPolyset * npset, double ratio )
{
	double ax=0.0,ay=0.0,az=0.0;
	double wx,wy,wz;
	double wx2,wy2,wz2;
	double dx,dy,dz;
	double fx,fy,fz;
	double l;
	double al = 0.0;
	int ct;
	int idx;
	ct = opoly->numberOfVertices();
	if( !ct ) {
		return;
	}
	AlPolysetVertex * vt1 = new AlPolysetVertex;
	AlPolysetVertex * vt2 = new AlPolysetVertex;
	int * iarr = new int[ ct ];
	for( idx = 0; idx < ct; idx++ ) {
		opoly->vertexD( idx, *vt1 );
		opoly->vertexD( ( idx + 1 ) % ct, *vt2 );
		vt1->unaffectedPosition( wx,wy,wz );
		ax+=wx;ay+=wy;az+=wz;
		vt2->unaffectedPosition( wx2,wy2,wz2 );
		dx = wx2-wx; dy = wy2-wy; dz = wz2 - wz;
		l = sqrt( dx*dx + dy*dy + dz*dz );
		al+=l;
		iarr[ idx ] = npset->newVertex( wx,wy,wz );
	}
	delete vt1; delete vt2;
	al/=(double)ct; ax/=(double)ct; ay/=(double)ct; az/=(double)ct;
	al*=ratio;

	double nx,ny,nz;
	double nl;
	opoly->normal( nx,ny,nz );
	nl = sqrt( nx*nx + ny*ny + nz*nz );
	nx/=nl; ny/=nl; nz/=nl;
	fx = ax + ( nx * al ); 
	fy = ay + ( ny * al ); 
	fz = az + ( nz * al ); 

	int fv = npset->newVertex( fx,fy,fz );
	AlPolygon * pgon = new AlPolygon;
	for( idx = 0; idx < ct; idx++ ) {
		npset->polygonD( npset->newPolygon(), *pgon );
		pgon->addVertex( iarr[ idx ] );
		pgon->addVertex( iarr[ ( idx + 1 ) % ct ] );
		pgon->addVertex( fv );
	}
	delete pgon;
	delete [] iarr;
}

static void computePolyset( AlPolyset *opset, AlPolyset *npset, double ratio )
{
	AlPolygon * pgon = new AlPolygon;
	int ct = opset->numberOfPolygons();
	for( int idx = 0; idx < ct; idx++ ) {
		statusCode code = opset->polygonD( idx, *pgon );
		computePolygon( pgon, npset, ratio );
	}
	delete pgon;
}

static void do_one_polyset( AlPolyset *opset, AlPolyset *npset, double ratio )
{
	int ct = npset->numberOfPolygons();
	int * arr = new int[ ct ];
	int idx;
	for( idx = 0; idx < ct; idx++ ) {
		arr[ idx ] = idx;
	}
	npset->deletePolygons( ct, arr );
	delete [] arr;
	ct = npset->numberOfVertices();
	for( idx = 0; idx < ct; idx++ ) {
		npset->deleteVertex( idx );
	}
	computePolyset( opset, npset, ratio );
}

struct polyCmdData {
	polyCmdData* polyCmdDataPtr() { return this; }
	AlPolysetNode * source_node;
	AlPolysetNode * my_node;
	double ratio;
	AlData *data;
};

class polygonCmd: public AlUserCommand, private polyCmdData
{
public:
	polygonCmd();
	~polygonCmd();

    virtual int         isValid();
    virtual int         execute();
    virtual int         declareReferences();
    virtual int         instanceDag( AlDagNode *oldDag, AlDagNode *newDag );
	virtual int			undo();

    virtual int         geometryModified( AlDagNode *dag );
    virtual int         listModifiedDagNodes( const AlNotifyDagNode *dagMod, AlObject *obj );
    virtual int         debug( const char *prefix );
    virtual void *      asDerivedPtr();

    virtual statusCode  retrieveWire( AlInput *input );
    virtual statusCode  storeWire( AlOutput *output );

    // for your own use
    virtual int         type();

	// we didn't bother to implement these commands since they don't apply
	// to us (they would be similar to instanceDag && geometryModified)
	//
	// virtual int dagModified( AlDagNode *dag );
    // virtual int curveOnSurfaceModified( AlCurveOnSurface *surf );

	// The following command is not yet supported
	// virtual statusCode  storeSDL( ofstream &outputSDL );

public:
	// methods that the UI commands can use to set our fields
	void set( AlPolysetNode* dn1, AlPolysetNode* dn2, double ratio );
};

void polygonCmd::set( AlPolysetNode *dn1, AlPolysetNode* dn2, double r )
{ 
	source_node = NULL; my_node = NULL;
	if( dn1 ) {
		source_node = dn1->copyWrapper()->asPolysetNodePtr(); 
	}
	if( dn2 ) {
		my_node = dn2->copyWrapper()->asPolysetNodePtr(); 
	}
	ratio = r;
}

polygonCmd::polygonCmd()
//
//	Initialize the structure
//
{
	source_node = NULL;
	my_node = NULL;
	ratio = 1.0;
	data = 0;
}

polygonCmd::~polygonCmd()
//
// nothing to clean up
//
{
	if( source_node ) {
		delete source_node;
	}
	if( my_node ) {
		delete my_node;
	}
	if ( data ) {
		delete data;
	}
	source_node = 0;
	my_node = 0;
	data = 0;
}

void *polygonCmd::asDerivedPtr()
//
//	Provide safe down casting.  This isn't really necessary (a simple
//	typecast should work).
//
{
	return this;
}

int polygonCmd::type()
//
//	User defined value so you can determine what your class type is.
//
{
	return 43;
}

int polygonCmd::isValid()
//
//	Figure out if the command is valid
//
{
	if( !my_node->asPolysetNodePtr() || !source_node->asPolysetNodePtr() ) {
		return -1;
	}

	return 0;
}

int polygonCmd::execute()
//
//	Execute the command.  Note that the isValid() command is implicitly
//	called before this routine.
//
{
	AlPolyset * opset = source_node->polyset();
	AlPolyset * npset = my_node->polyset();
	do_one_polyset( opset, npset, ratio );
	delete opset;
	delete npset;
	return 0;
}

int polygonCmd::instanceDag( AlDagNode *oldDag, AlDagNode *newDag )
//
//	Handle a dag node being instanced.  Go through the class and replace
//	any references to oldDag with newDag
//
{
	if( AlAreEqual( source_node, oldDag ) )
		source_node = newDag->copyWrapper()->asPolysetNodePtr();
	if( AlAreEqual( my_node, oldDag ) ) 
		my_node = newDag->copyWrapper()->asPolysetNodePtr();

	return 0;
}

int polygonCmd::declareReferences()
//
//	Declare any reference to constructors
//	Declare any references to dagnodes that may be destructors
//
{
	addConstructorRef( source_node );
	addTargetRef( my_node );
	return 0;
}

int polygonCmd::geometryModified( AlDagNode *)
{
	return 0;
}

int polygonCmd::debug( const char * )
{
	cerr << "Whatever." << endl;
	return 0;
}

int polygonCmd::undo()
//
//	Undo everything the 'execute' did.
//
{
	return 0;
}

int polygonCmd::listModifiedDagNodes( const AlNotifyDagNode *dagMod, AlObject *obj )
//
//	This routine should call dagMod->notify on every dag node that might
//	be affected if obj is modified.  In our example, nothing should change,
//	but for illustrative purposes assume that modifying our CV changes the
//	dag node.
//
{
	if( AlAreEqual( obj, my_node ) )
	{
		 if( sSuccess == dagMod->notify( my_node ))
				return 0;
		return 0;
	}
	return -1;
}

statusCode polygonCmd::retrieveWire( AlInput *input )
//
//	This routine copies the saved data chunk from the wire file.
//	Pointers are dealt with in the resolve routine.
//
{
	// Replace the old pointers with the newly relocated ones.
	// Resolve in the same order as you declare.
	AlObject *sDag = input->resolveObject();
	AlObject *mDag = input->resolveObject();

	// if a pointer was not resolved, then NULL is returned
	//
	// This can happen if the geometry has been deleted on you
	// (the geometry was modified and the plugin was not around)
	// Alternatively, we could have just returned sSuccess.
	// When our command is executed, it will be invalid and so it will be
	// deleted.

	source_node = sDag ? sDag->asPolysetNodePtr() : NULL;
	my_node = mDag ? mDag->asPolysetNodePtr() : NULL;

	if( ! (source_node && my_node) )
		return sFailure;
		
	if ( data )
		delete data;
	data = input->resolveData( AlData::kDataDouble, dataId );
	if ( ! data )
		return sFailure;
	ratio = *(data->asDoublePtr());
	printf("\tRetrieving ratio: %g\n",ratio);
		
	return sSuccess;
}

statusCode polygonCmd::storeWire( AlOutput *output )
//
//	This routine writes out our data to the wire file.  It also translates
//	the pointers into handles.  This routine is to be called ONCE with
//	your final data block.
//
{
	// declare all of our references to data so that we can get them back
	// on retrieval
	output->declareObject( source_node );
	output->declareObject( my_node );
	
	if ( data )
		delete data;
	data = new AlData;
	if ( data->create( dataId, &ratio, 1 ) != sSuccess )
		return sFailure;
	if ( output->declareData( data ) != sSuccess )
		return sFailure;
	printf("\tSaving ratio: %g\n",ratio);
		
	return sSuccess;
}

//
//	End command definition
//

void do_enhanceOnePolyset( AlPolysetNode * psn )
{
	AlPolyset * pset = psn->polyset();
	AlPolyset * npset = new AlPolyset;
	npset->create();
	double ratio;
	AlGetDouble( "polyhist.ratio", ratio );
	do_one_polyset( pset, npset, ratio );

	// Now attach the command to the new polyset.
	AlCommand cmd;
	if( sSuccess == cmd.create( "myPolyCmd" ) )
	{
		polygonCmd * pcmd = (polygonCmd*)cmd.userCommand()->asDerivedPtr();
		AlPolysetNode * npsn = npset->polysetNode();
		pcmd->set( psn, npsn, ratio );
		if( cmd.execute() == 0 ) 
			cmd.install();
		else 
			cmd.deleteObject();
		
		delete npsn;
	}
	delete pset;
	delete npset;
}

void do_enhancePolyset( void )
{
	for(	statusCode stat = AlPickList::firstPickItem();
			stat == sSuccess;
			stat = AlPickList::nextPickItem() )
	{
		AlObject *pickedItem = AlPickList::getObject();

		if( !pickedItem ) {
			continue;
		}

		AlPolysetNode *psn;

		if( psn = pickedItem->asPolysetNodePtr() )
		{
			do_enhanceOnePolyset( psn );
		}
		delete pickedItem;
	}
	AlUniverse::redrawScreen( kRedrawActive | kRedrawInactive );
}

AlUserCommand *allocPolyCmd()
//
//	Allocate & return a new cv command.  This function will be passed
//	to the AlCommand::add routine
//
{
	return new polygonCmd;
}

static AlMomentaryFunction hFunc;
static AlFunctionHandle handle;

extern "C"
PLUGINAPI_DECL int plugin_init( const char *dirName )
{
    char *dirScm;

    AlUniverse::initialize( kZUp );

    dirScm = makeAltPath(dirName,"scm");

    AlCommand::add(  allocPolyCmd, "myPolyCmd" );

    hFunc.create( "enhance_polyset", do_enhancePolyset );
    AlInvokeSchemeFile( "polygonHistoryExample_init.scm", dirScm );

    handle.create( "PolygonHistory", &hFunc );
    handle.setAttributeString( "Polygon History" );
    handle.appendToMenu( "mp_objtools" );
    if ( sSuccess != handle.setOptionBox( "polygonHistoryExample.scm", "polyhist.options", 
										  dirScm ) ) {
		AlPrintf( kPrompt, "polyonHistoryExample plug-in unable to find .scm file for option box" );
		return 1;
	}
	handle.setIconPath( makeAltPath( dirName, NULL ) );

    AlPrintf( kPrompt, "Polyon History plug-in installed on Palette 'Object Edit'");

    return 0;
}

extern "C"
PLUGINAPI_DECL int plugin_exit( void )
{
    AlCommand::remove( "myPolyCmd" );
    handle.deleteObject();
    hFunc.deleteObject();

    // do nothing
    return 0;
}
