/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/

/*
 * pickMultiSpanSurfaces.cpp
 *
 * This plug-in will select all of the
 * multispan surfaces in your scene. The 
 * implementation of this plugin is very simple 
 * since it basically goes through each surface 
 * and checks for multispans and then selects the 
 * object (even if it is in a invisible layer or has
 * visibility turned off.). 
 */

#include <AlCoordinateSystem.h>
#include <AlUniverse.h>

#include <AlLiveData.h>
#include <AlFunction.h>
#include <AlFunctionHandle.h>

#include <AlSurfaceNode.h>
#include <AlSurface.h>

#include <WalkTree.h>

#include <math.h>

static int surfaceCounter = 1;

class PickMultiSpanSurfaceIterator: public AlIterator
{
    public:
        virtual int func( AlObject* );
};

// #define VERBOSE 1

int PickMultiSpanSurfaceIterator::func( AlObject* obj )
{
    if ( obj == NULL )
        return 0;

	AlSurfaceNode *surfaceNode = obj->asSurfaceNodePtr();
    if ( surfaceNode != NULL )
    {
		#ifdef VERBOSE
		AlPrintf( kStdout, "%d) Querying surface '%s' for multi-spans.\n",surfaceCounter,surfaceNode->name());
		AlPrintf( kPrompt, "%d) Querying surface '%s' for multi-spans.\n",surfaceCounter,surfaceNode->name());
		#endif
		surfaceCounter++;
		
		AlSurface *surface = surfaceNode->surface();
		if ( surface != NULL )
		{
			int uSpans = 0;
			int vSpans = 0;	
			
			uSpans = surface->uNumberOfSpans();
			vSpans = surface->vNumberOfSpans();
			
			if ( uSpans > 1 || vSpans > 1 )
			{
				#ifdef VERBOSE
				AlPrintf( kStdout, "\tInformation: surface '%s' has multi-spans.\n",surfaceNode->name());
				AlPrintf( kPrompt, "   Information: surface '%s' has multi-spans.\n",surfaceNode->name());
				#endif
				
				if ( surfaceNode->isPicked() )
				{
					#ifdef VERBOSE
					AlPrintf( kStdout, "\tInformation: multi-span surface '%s' already picked.\n",surfaceNode->name());
					#endif
					AlPrintf( kPrompt, "   Information: multi-span surface '%s' already picked.\n",surfaceNode->name());
				}
				else
				{
					if ( surfaceNode->pick() != sSuccess )
					{
						#ifdef VERBOSE
						AlPrintf( kStdout, "\tInformation: failed to pick multi-span surface '%s'.\n",surfaceNode->name());
						#endif
						AlPrintf( kPrompt, "   Information: failed to pick multi-span surface '%s'.\n",surfaceNode->name());
					}
					else
					{
						#ifdef VERBOSE
						AlPrintf( kStdout, "\tInformation: picked multi-span surface '%s'.\n",surfaceNode->name());
						#endif
						AlPrintf( kPrompt, "   Information: picked multi-span surface '%s'.\n",surfaceNode->name());
					}
				}
			}
			
			delete surface;
		}
    }

	return 0;
}

void pickMultiSpanSurfaces( void )
{
	PickMultiSpanSurfaceIterator* pickMS = new PickMultiSpanSurfaceIterator();

	if ( pickMS != NULL )
	{
			surfaceCounter= 1;
        	AlDagNode *dag = AlUniverse::firstDagNode();
        	while ( dag != NULL )
        	{
			WalkTree::walkHierarchy( kHierBelow, (AlObject *) dag, pickMS );
				AlDagNode *tmpDag = dag;	// Store the wrapper
				dag = dag->nextNode();		// Move to the next code
				delete tmpDag;				// Free the stored wrapper
        	}
		AlUniverse::redrawScreen( kRedrawActive );
	}
	else
	{
		AlPrintf( kStdout, "Error: failed to initialize multi-span pick iterator.\n");
		AlPrintf( kPrompt, "Error: failed to initialize multi-span pick iterator.\n");
	}
}

static AlFunctionHandle h;
static AlMomentaryFunction hFunc;

extern "C"
PLUGINAPI_DECL int plugin_init( char *dirName )
{
	AlUniverse::initialize( kZUp );

	hFunc.create( pickMultiSpanSurfaces );
	h.create( "Pick Multi-Span Surfaces", &hFunc ); 
	h.setAttributeString( "pickmultispansurfaces" );
	h.appendToMenu( "mp_pick" );
	h.setIconPath( makeAltPath( dirName, NULL ) );
	
	AlPrintf( kPrompt, "Pick Multi-Span surfaces installed on Palette 'Pick'." );
	return 0;
}

extern "C"
PLUGINAPI_DECL int plugin_exit( void )
{
    h.deleteObject();
    hFunc.deleteObject();
	return 0;
}
