/********************************************************************
 * (C) Copyright 2016 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/

/* This plugin is an example for using pack reference functionality. */

#include <AlLiveData.h>
#include <AlUniverse.h>
#include <AlDagNode.h>
#include <AlSurface.h>
#include <AlSurfaceNode.h>
#include <AlMesh.h>
#include <AlMeshNode.h>
#include <AlPickExt.h>
#include <AlFunction.h>
#include <AlFunctionHandle.h>
#include <AlIterator.h>
#include <AlPickExt.h>
#include <WalkTree.h>
#include <memory.h>
#include <AlSidCanvasExt.h>
#include <list>

// This handle may have to be global if you wish to remove the
// plugin from the menu using the h.destroy() method in the
// 'momentary_exit' function.
// The menu entry is automatically removed when Alias exits.
//
static AlFunctionHandle h;
static AlMomentaryFunction hFunc;

namespace
{
    struct ObjectCollector : public AlIterator
    {
        ~ObjectCollector();
        virtual int func( AlObject* obj );
        std::list<AlObject*> m_list;
    };

    ObjectCollector::~ObjectCollector()
    {
        for( AlObject* obj : m_list )
        {
            delete obj;
        }
        m_list.clear();
    }

    int
    ObjectCollector::func( AlObject* obj )
    {
        m_list.push_back( obj->copyWrapper() );
        return 1;
    }

    void
    process( AlRayPick& picker )
    {
        for( auto iter = picker.iterator(); iter; ++iter )
        {
            AlRayPick::Hit hit = iter.getHit();
            auto obj = hit.obj;

            if( AlPickable* objPickable = obj->asPickablePtr() )
            {
                objPickable->pick();
            }
        }
    }

    void
    process( AlBoxPick& picker )
    {
        for( AlBoxPick::HitIterator iter = picker.iterator();
             iter.valid(); iter.next() )
        {
            AlBoxPick::Hit hit = iter.getHit();
            auto obj = hit;

            if( AlPickable* objPickable = obj->asPickablePtr() )
            {
                objPickable->pick();
            }
        }
    }

    void
    process( AlSpherePick& picker )
    {
        // Select closest (the first in a pick list) picked object
        auto iter = picker.iterator(); 
        if( iter.valid() )
        {
            AlSpherePick::Hit hit = iter.getHit();
            auto obj = hit.obj;

            if( AlPickable* objPickable = obj->asPickablePtr() )
            {
                objPickable->pick();
            }
        }
    }

    void pick()
    {
        // Get list of all nodes
        ObjectCollector collector;
        WalkTree::walkHierarchy ( kHierBelow, nullptr, &collector );

        // Setup picker objects
        double orig[3] = { 0, 0, 0 };
        double dir[3] = { 0.5, 0.5, 0.5 };
        AlRayPick rayPick( orig, dir );

        double min[3] = { 0, 0, 0 };
        double max[3] = { 10, 10, 10 };
        AlBoxPick boxPick( min, max );

        double pt[3] = { 0.5, 0.5, 0.5 };
        AlSpherePick spherePick( pt );

        // Test objects
        for( AlObject* obj : collector.m_list )
        {
            rayPick.add( obj );
            boxPick.add( obj );
            spherePick.add( obj );
        }

        // Process objects which passed tests
        process( rayPick );
        process( boxPick );
        process( spherePick );
    }
}

// This routine initializes the plugins and attaches it to the menu.
// It returns 0 if there is no initialization error.
extern "C"
PLUGINAPI_DECL int plugin_init( const char *dirName )
{
	// Initialize the universe. This must be done by all
	// plugins. If the universe is not initialized the plugin
	// will fail.
	AlUniverse::initialize( );

	//	Create the function as a AlMomentaryFunction
	//	note that the icon name will be nothing_func.S, nothing_func.M
	hFunc.create( "pl_Pick", pick );

	// Allocate a function handle. The first argument is the label on
	// the menu and the second is the function to invoke when the
	// menu item is selected.
	h.create( "Pick", &hFunc ); 

	// Define the attribute string for the attribute line below
	// the prompt line.
	h.setAttributeString( "pick" );

	// Indicate which menu to add the plugin to. addToMenu()
	// adds the plugin to the bottom of the menu, while
	// appendToMenu() will add it to the top of the menu.
	h.addToMenu( "al_file" );

	// Return a success code.
	// Returning a non zero indicates an error.
	// An error value ( a non-zero ) will be printed on the prompt
	// line in Alias.
	AlPrintf( kPrompt, "Pick plug-in installed under menu 'File'");
	return 0;
}

extern "C"
PLUGINAPI_DECL int plugin_exit( void )
{
	// Remove the plugin from the menu and free the FunctionHandle.
	// Note that h.destroy() implicitly calls h.removeFromMenu()
	h.deleteObject();
	hFunc.deleteObject();
	return 0;
}
